/*
 * device.c
 *
 * Copyright 2002, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 */


#include"config.h"
#include"types.h"
#include"lib.h"
#include"mm.h"
#include"interrupt.h"
#include"proc.h"
#include"fs.h"
#include"lock.h"
#include"device.h"
#include"except.h"
#include"errno.h"
#include"test.h"


/***********************************************************************************
 *
 * Device filesystem
 *
 ***********************************************************************************/

enum{
	MAX_DEVICE_NAME=8,			/* Device name size */
};


/* ǥХȤ */
typedef struct DEV_INODE{
	uchar pt_num;		/* Partition number */
	uchar pt_type;		/* Partition type */
	short ref_count;	/* Reference count */
	DEV_INFO *dev;		/* Device infomation pointer */
	uint begin_blk;		/* Begin block number */
	uint blocks;		/* Number of blocks */
	struct DEV_INODE *next;
}DEV_INODE;


/* PRIVATE */
static DEV_INFO devInfoTop;
static DEV_INODE dinode[MAX_DEVICE_OPEN];
static DEV_INODE dinodeTop;
static DEV_INODE dinodeEmpty;


/*
 * PUBLIC
 * inodeɥ쥹ֹ롣
 * parameters : DEV_INODE address
 * return : ֹ
 */
static inline int getInodeNum(DEV_INODE *dind)
{
	return ((uint)dind-(int)dinode)/sizeof(DEV_INODE);
}


/*
 * PUBLIC
 * ǥХinodeץγǧ
 * return : 0 or -1
 */
static inline int isMaxDevOpen()
{
	return (dinodeEmpty.next==NULL)?0:-1;
}


/*
 * PUBLIC
 * ǥХinodeƥե륷ƥ󥯤Ͽ롣
 * parameters : device name,begin block,blocks,partition type,partition number,FS
 * return : device inode or NULL
 */
static DEV_INODE *makeInode(const char *name,uint beg,uint size,int ptype,int pnum,DEV_INFO *dev)
{
	DEV_INODE *prev_inode,*dinode;


	/* inode󥯤θ */
	prev_inode=&dinodeTop;
	for(;prev_inode->next!=NULL;prev_inode=prev_inode->next)
		switch(strcmp(name,prev_inode->next->dev->name))
		{
			case 0:
				if(pnum==prev_inode->next->pt_num)return NULL;
				else if(pnum<prev_inode->next->pt_num)goto EXIT;
				break;
			case -1:
				goto EXIT;
		}
EXIT:
	dinode=dinodeEmpty.next;
	dinodeEmpty.next=dinode->next;
	dinode->next=prev_inode->next;
	prev_inode->next=dinode;

	/* inodeν */
	dinode->pt_type=ptype;
	dinode->pt_num=pnum;
	dinode->ref_count=0;
	dinode->dev=dev;
	dinode->begin_blk=beg;
	dinode->blocks=size;

	return dinode;
}


/*
 * PUBLIC
 * ̾ǥХõ
 * parameters : device name
 * return : DEV_INFO address or NULL
 */
/*static inline DEV_INFO *searchDevInfo(const char *name)
{
	DEV_INFO *p;


	for(p=devInfoTop.next;p!=NULL;p=p->next)
		switch(strcmp(name,p->name))
		{
			case 0:
				return p;
			case -1:
				return NULL;
		}

	return NULL;
}
*/

/*
 * PUBLIC
 * Search device inode
 * ƱǥХդädevDEV_INFOɥ쥹롣
 * parameters : device name,partition number,FS address pointer
 * return : Device inode address or NULL
 */
static inline DEV_INODE *searchDevInode(const char *name,int pnum,DEV_INFO **dev)
{
	DEV_INODE *p;


	for(p=dinodeTop.next;p!=NULL;p=p->next)
		switch(strcmp(name,p->dev->name))
		{
			case 0:
				*dev=p->dev;
				if(pnum==p->pt_num)return p;
				else if(pnum<p->pt_num)return NULL;
			case -1:
				return NULL;
		}

	return NULL;
}


/*
 * PUBLIC
 * Init device filesystem.
 */
static void initDevFs()
{
	DEV_INODE *p;
	int i;


	for(i=0,p=&dinodeEmpty;i<MAX_DEVICE_OPEN;++i)
	{
		dinode[i].next=p->next;
		p->next=&dinode[i];
	}
}


/*
 * PUBLIC
 * Ϳ줿inodeɥ쥹μΥ󥯤inode֤
 * Ϳ줿inodeNULLʤȥåץ󥯤inode֤
 * parameters : device inode
 * return : device inode or NULL
 */
static DEV_INODE *getNextInode(DEV_INODE *dind)
{
	if(dind==NULL)return dinodeTop.next;
	else return dind->next;
}


/*
 * GLOBAL
 * ǥХμ
 * parameters : device inode,stat store buffer
 * return : 0 or error=-1
 */
int get_dev_stat(int din,DEV_STAT *buf)
{
	if(dinode[din].ref_count==0)return -1;

	buf->prt_type=dinode[din].pt_type;
	buf->sect_size=dinode[din].dev->sector_size;
	buf->all_sect=dinode[din].blocks;

	return 0;
}


/*
 * GLOBAL
 * Regist device.
 * parameters : Device infomation struct address
 * return : device inode or Error number
 */
int regist_device(DEV_INFO *info)
{
	DEV_INFO *p;


	/* Name sizeγǧ  */
	if(strlen(info->name)>MAX_DEVICE_NAME)return -ENAMETOOLONG;

	/* 󥯤³ */
	p=&devInfoTop;
	for(;p->next!=NULL;p=p->next)
		switch(strcmp(info->name,p->next->name))
		{
			case 0:
				return -EEXIST;
			case -1:
				goto EXIT;
		}
EXIT:
	info->next=p->next;
	p->next=info;

	/* Device inodeκ */
	if(makeInode(info->name,info->begin_blk,info->last_blk-info->begin_blk+1,0,0,info)==NULL)
		return -EEXIST;

	return 0;
}


/*
 * GLOBAL
 * Delete from regist table
 * parameters : name
 * return : 0,error -1
 */
int delete_device(const char *name)
{
	return 0;
}


/************************************************************************************************
 *
 * ľžؿ
 *
 ************************************************************************************************/

/*
 * GLOBAL
 * parameters : device inode number,buffer address,transfer sectors,begin sector
 * return : transfer sectors or error=-1
 */
int read_direct(int din,void *buf,size_t sectors,size_t begin)
{
	/* ǥ¤Υå */
	if(begin>dinode[din].blocks)return -1;
	if(begin+sectors>dinode[din].blocks)sectors=dinode[din].blocks-begin;

	return dinode[din].dev->read(buf,sectors,begin+dinode[din].begin_blk);
}


/*
 * GLOBAL
 * parameters : device inode number,buffer address,transfer sectors,begin sector
 * return : transfer sectors or error=-1
 */
int write_direct(int din,void *buf,size_t sectors,size_t begin)
{
	/* ǥ¤Υå */
	if(begin>dinode[din].blocks)return -1;
	if(begin+sectors>dinode[din].blocks)sectors=dinode[din].blocks-begin;

	return dinode[din].dev->write(buf,sectors,begin+dinode[din].begin_blk);
}


/************************************************************************************************
 *
 * ǥХå
 *
 * ա
 *  ǥХϥ512ХȤˤƤΤǡ¾Υ
 *  ǥХоݤˤˤϡɬס
 *
 * todoꥹ
 *  ָŤå˽񤭹ߤäν
 *  ν
 *
 ************************************************************************************************/

enum{
	CACHE_NUM=8,				/* IOå֥å */
	CACHE_BLOCK_SIZE=0x1000,	/* IOå֥å */
	HASH_NUM=CACHE_NUM,			/* Number of cache hash table */
	CACHE_UNIT_SIZE=512,		/* å֥åΥӥåȥޥåñ̥Хȥ */
};


typedef struct CACHE_TABLE{
	struct CACHE_TABLE *hnext;		/* Hash next */
	struct CACHE_TABLE *hprev;		/* Hash previous */
	struct CACHE_TABLE *next;		/* Cache chain next */
	struct CACHE_TABLE *prev;		/* Cache chain previous */
	struct CACHE_TABLE *write_next;	/* ǥХؽ񤭹߹ꥹ */
	struct CACHE_TABLE *write_prev; /* ǥХؽ񤭹߹ꥹ */
	void *addr;						/* Cache address */
	uint block;						/* Device block index */
	uchar din;						/* Device inode number */
	uchar bitmap;					/* ǡ¸ߥӥåȥޥå */
	uchar dirty;					/* ߤե饰 */
	uchar volatile writing;			/* ǥХؽ񤭹߹ե饰 */
	int gate;						/* ԥå */
}CACHE_TABLE;

typedef struct{
	uchar start;		/* ߳ϥ˥åȡ */
	uchar sectors;		/* ߥ˥åȿ */
}DELAY_WRITE_STATE;


static CACHE_TABLE cache_tbl[CACHE_NUM];			/* Cache table */
static CACHE_TABLE *cache_hash[HASH_NUM];			/* Hash table for searching cache table */
static CACHE_TABLE *cache_old=&cache_tbl[0];		/* old chache block.nextֿ֥å */
static PROC *write_task=NULL;						/* ٱ񤭹ѥ */
static CACHE_TABLE delay_list=						/* ٱ񤭹ߥꥹȤƬ */
	{NULL,NULL,NULL,NULL,&delay_list,&delay_list,NULL,0,0,0,0,0,0};
static int cacheLock=0;								/* å󥯹ѥåȡ */


/*
 * PRIVATE
 * ǥХ饭åɤ߹ࡣ
 * parameters : CACHE_TABLE *p
 * return : 0 or error number
 */
static int readDevice(CACHE_TABLE *p)
{
	int i,k;


	if(p->dirty==0)
	{
		if(dinode[p->din].dev->read(p->addr,CACHE_BLOCK_SIZE/dinode[p->din].dev->sector_size,p->block+dinode[p->din].begin_blk)<0)
			return  -EIO;
	}
	else
	{
		/* ɤ߹ޤƤʤս򥭥åɤ߹ࡣ */
		for(i=0;i<CACHE_BLOCK_SIZE/CACHE_UNIT_SIZE;++i)
			if((p->bitmap&(1<<i))==0)
			{
				for(k=i+1;k<CACHE_BLOCK_SIZE/CACHE_UNIT_SIZE;++k)
					if((p->bitmap&(1<<k))!=0)break;
				if(dinode[p->din].dev->read((char*)p->addr+i*CACHE_UNIT_SIZE,k-i,p->block+i+dinode[p->din].begin_blk)<0)
					return -EIO;
				i=k;
			}
	}

	p->bitmap=0xff;

	return 0;
}


/*
 * PRIVATE
 * å夫ǥХ˽᤹
 * parameters : CACHE_TABLE struct
 * return : 0 or error number
 */
static int writeDevice(CACHE_TABLE *p)
{
	int i,k;


	if(p->bitmap==0xff)
	{
		if(dinode[p->din].dev->write((char*)p->addr,CACHE_BLOCK_SIZE/dinode[p->din].dev->sector_size,p->block+dinode[p->din].begin_blk)<0)
			return -EIO;
	}
	else
	{
		/* սǥХ˽񤭹ࡣ */
		for(i=0;i<CACHE_BLOCK_SIZE/CACHE_UNIT_SIZE;++i)
			if((p->bitmap&(1<<i))!=0)
			{
				for(k=i+1;k<CACHE_BLOCK_SIZE/CACHE_UNIT_SIZE;++k)
					if((p->bitmap&(1<<k))==0)break;
				if(dinode[p->din].dev->write((char*)p->addr+i*CACHE_UNIT_SIZE,k-i,p->block+i+dinode[p->din].begin_blk)<0)
					return -EIO;
				i=k;
			}
	}

	p->dirty=0;

	return 0;
}


/*
 * PRIVATE
 * ǥХؤٱ񤭹ߥؿ
 */
static void delay_write()
{
	CACHE_TABLE *p;


	while(write_task==NULL);

	del_from_schedule(TASK_WAIT);
	wait_task();

	for(;;)
	{
		for(;;)
		{
			enter_spinlock(&delay_list.gate);

			if(delay_list.write_next!=&delay_list)
			{
				p=delay_list.write_prev;
				p->write_prev->write_next=&delay_list;
				delay_list.write_prev=p->write_prev;
				p->write_prev=p->write_next=p;

				exit_spinlock(&delay_list.gate);
				break;
			}
			else
			{
				exit_spinlock(&delay_list.gate);

				/* ꡼ס */
				del_from_schedule(TASK_WAIT);
				wait_task();
			}
		}

		do
		{
			p->writing>>=1;

			if(writeDevice(p)<0)printk("Device error in delay write function!\n");
		}while((p->writing&=0x2)!=0);
	}
}


/*
 * PRIVATE
 * ٱ񤭹ߥꥹȤ˲ä롣
 * parameters : device inode number,񤭹ߥ֥å,֥åλϤޤ
 */
static void add_delay_list(CACHE_TABLE *p)
{
	if(p->dirty==0)return;

	p->writing|=0x2;

	enter_spinlock(&delay_list.gate);
	{
		p->write_next->write_prev=p->write_prev;
		p->write_prev->write_next=p->write_next;
		p->write_next=delay_list.write_next;
		p->write_prev=&delay_list;
		p->write_next->write_prev=p;
		delay_list.write_next=p;
	}
	exit_spinlock(&delay_list.gate);

	/* ٱ񤭹ߥ򵯤 */
	add_to_schedule(write_task,TASK_WAIT);
}


/*
 * PRIVATE
 * ϥåͤ׻롣
 * return : hash value
 */
static inline int calc_hash(int value)
{
	return (value/8)%HASH_NUM;
}


static inline int cache_rounddown(int value)
{
	return ~(CACHE_BLOCK_SIZE-1)/value;
}


/*
 * PRIVATE
 * åơ֥뤫鳺ǥХ֥åõ
 * parameters : device inode,begin block
 * return : block addres or Failed=NULL
 */
static CACHE_TABLE *search_cache(int din,size_t begin)
{
	int hash=calc_hash(begin);
	CACHE_TABLE *p;


	/* å򸡺롣 */
	for(p=cache_hash[hash];p!=NULL;p=p->hnext)
	{
		/* åˤ */
		if((p->block==begin)&&(p->din==din))
		{
			/* ָŤåξ硣 */
			if(p==cache_old)
			{
				cache_old=p->prev;
				if(cache_old->dirty)add_delay_list(cache_old);
			}

			/* 󥯤ֿ֤ˤ롣 */
			p->next->prev=p->prev;
			p->prev->next=p->next;
			p->next=cache_old->next;
			p->prev=cache_old;
			cache_old->next=p;
			p->next->prev=p;

			return p;
		}
	}

	return NULL;
}


/*
 * PRIVATE
 * ˥å֥åƤ롣
 * parameters : device inode,device block
 * return : cache table
 */
static CACHE_TABLE *getEmptyCache(int din,size_t begin)
{
	int hash=calc_hash(begin);
	CACHE_TABLE *p;


	/* ָŤå򲣼ꡣ */
	p=cache_old;
	cache_old=p->prev;

	/* μθŤå᤹ */
	if(cache_old->prev->dirty)add_delay_list(cache_old->prev);

	/* ϥå󥯤 */
	p->hprev->hnext=p->hnext;
	if(p->hnext!=NULL)p->hnext->hprev=p->hprev;

	/* ϥå󥯤˲ä롣 */
	p->hnext=cache_hash[hash];
	p->hprev=(CACHE_TABLE*)&cache_hash[hash];
	if(cache_hash[hash]!=NULL)cache_hash[hash]->hprev=p;
	cache_hash[hash]=p;

	/* ٱ񤭹Ԥ */
	while(p->writing!=0);

	p->din=din;
	p->block=begin;
	p->bitmap=0;

	return p;
}


/*
 * PUBLIC
 * ٱ񤭹ߥεư
 */
static void start_delay_write_task()
{
	int proc;


	switch(proc=sys_fork())
    {
		case -1:
			printk("Faild start_delay_write_task\n");
			break;
		case 0:
			delay_write();
		default:
			write_task=(PROC*)proc;
	}
}


/*
 * PUBLIC
 * ߤꥭå֥å򤹤٤ٱߤ롣
 */
static void writeBackAllCache()
{
	CACHE_TABLE *p=cache_old;


	do
	{
		if((p->dirty)&&(p->writing==0))add_delay_list(p);
	}while((p=p->next)!=cache_old);

	while(delay_list.write_next!=&delay_list)
		wait_task();
}


/*
 * GLOBAL
 * Read from cache
 * parameters : device inode number,buffer address,transfer sectors,begin sector
 * return : transfer sectors or error number
 */
int read_cache(int din,void *buf,size_t sector_num,size_t begin)
{
	int sector_size=dinode[din].dev->sector_size;	/* ǥХ */
	int sectors=CACHE_BLOCK_SIZE/sector_size;		/* å֥åˤϤ륻 */
	int cache_mask=sectors-1;						/* åƬ򻻽Ф뤿Υӥåȥޥ */
	int rest_num=sector_num;						/* Ĥž */
	int start,sect_num;
	CACHE_TABLE *p;


	/* ǥ¤Υå */
	if(begin>dinode[din].blocks)return -EIO;
	if(begin+sectors>dinode[din].blocks)return -EIO;

	start=begin&cache_mask;
	begin&=~cache_mask;
	sect_num=(rest_num<sectors-start)?rest_num:sectors-start;
	for(;;)
	{
		/* å򸡺 */
		enter_spinlock(&cacheLock);
		{
			if((p=search_cache(din,begin))==NULL)p=getEmptyCache(din,begin);
		}
		exit_spinlock(&cacheLock);

		/* å׵᥻ʤХǥХɤ߹ࡣ */
		if((p->bitmap|(((1<<sect_num)-1)<<start))!=p->bitmap)
			if(readDevice(p)<0)return -EIO;

		/* Хåե˥ԡ롣 */
		memcpy(buf,(char*)p->addr+start*sector_size,sect_num*sector_size);

		if((rest_num-=sect_num)==0)break;
		(char*)buf+=sect_num*sector_size;
		begin+=sectors;
		sect_num=(rest_num<sectors)?rest_num:sectors;
		start=0;
	}

	return sector_num;
}


/*
 * GLOBAL
 * Write to cache
 * parameters : device inode number,buffer address,transfer sectors,begin sector
 * return : transfer sectors or error number
 */
int write_cache(int din,void *buf,size_t sector_num,size_t begin)
{
	int sector_size=dinode[din].dev->sector_size;	/* ǥХ */
	int sectors=CACHE_BLOCK_SIZE/sector_size;		/* å֥åˤϤ륻 */
	int cache_mask=sectors-1;						/* åƬ򻻽Ф뤿Υӥåȥޥ */
	int rest_num=sector_num;						/* Ĥž */
	int start,sect_num;
	CACHE_TABLE *p;


	/* ǥ¤Υå */
	if(begin>dinode[din].blocks)return -EIO;
	if(begin+sectors>dinode[din].blocks)return -EIO;

	start=begin&cache_mask;
	begin&=~cache_mask;
	sect_num=(rest_num<sectors-start)?rest_num:sectors-start;
	for(;;)
	{
		/* å򸡺 */
		enter_spinlock(&cacheLock);
		{
			if((p=search_cache(din,begin))==NULL)p=getEmptyCache(din,begin);
		}
		exit_spinlock(&cacheLock);

		/* å˥ԡ롣 */
		memcpy((char*)p->addr+start*sector_size,buf,sect_num*sector_size);

		p->bitmap|=((1<<sect_num)-1)<<start;
		p->dirty=1;

		if((rest_num-=sect_num)==0)break;
		(char*)buf+=sect_num*sector_size;
		begin+=sectors;
		sect_num=(rest_num<sectors)?rest_num:sectors;
		start=0;
	}

	return sector_num;
}


/*
 * GLOBAL
 * Init cache system
 * return : 0 or Error number
 */
static int init_cache()
{
	int i;


	/* Init cache address */
	for(i=0;i<CACHE_NUM;++i)
	{
		memset(&cache_tbl[i],0,sizeof(CACHE_TABLE));
		if((cache_tbl[i].addr=kmalloc(CACHE_BLOCK_SIZE))==NULL)return -ENOMEM;
		cache_tbl[i].next=&cache_tbl[i+1];
		cache_tbl[i].prev=&cache_tbl[i-1];
		cache_tbl[i].hprev=&cache_tbl[i];
		cache_tbl[i].write_prev=&cache_tbl[i];
		cache_tbl[i].write_next=&cache_tbl[i];
	}
	cache_tbl[CACHE_NUM-1].next=&cache_tbl[0];
	cache_tbl[0].prev=&cache_tbl[CACHE_NUM-1];

	/* Init hash table */
	for(i=0;i<HASH_NUM;++i)cache_hash[i]=NULL;

	return 0;
}


/************************************************************************************************
 *
 * System call interface
 *
 ************************************************************************************************/

/*
 * PRIVATE
 * ǥХ̾ѡƥֹФ롣
 * parameters : device name,name buffer
 * return : pertition number or error number
 */
static int getPartitionNum(const char *name,char *buf)
{
	int num=0;
	int exp=1;
	int i;


	i=strlen(name)-1;
	for(;i>=0;--i)
	{
		if((name[i]>='0')&&(name[i]<='9'))
		{
			num+=(name[i]-'0')*exp;
			exp*=10;
		}
		else break;
	}

	if(++i>MAX_DEVICE_NAME)return -ENAMETOOLONG;
	memcpy(buf,name,i);
	buf[i]='\0';

	return num;
}


/*
 * parameters : Device name,device inode=0,directory block=0
 * return : device inode number or error number
 */
static int open(const char *path,int dev,uint dir,int flag)
{
	enum{
		/* Partition table */
		PT_NUM=4,				/* 1Υѡƥơ֥ */
		PT_BEGIN=0x1be,			/* ѡƥơ֥볫ϥեå */
		PT_MAGIC_ADDR=0x1fe,	/* ѡƥơ֥ޥåʥСեå */
		PT_MAGIC_NUMBER=0xaa55,	/* ѡƥơ֥ޥåʥС */

		/* Partition type */
		PT_TYPE_NULL=0,			/* NULL partition */
		PT_TYPE_EXT=0x5,		/* ext partition */
		PT_TYPE_LEXT=0xf,		/* ext(LBA) partition */
	};

	/* Partition table */
	typedef struct{
		uchar active;			/* Boot active flag=0x80 */
		uchar chs_begin[3];		/* CHS begin block */
		uchar type;				/* Partition type */
		uchar chs_last[3];		/* CHS last block */
		uint lba_begin;			/* LBA begin block */
		uint size;				/* Partition size by sector */
	}PARTITON_TBL;

	const char *next;
	char name[MAX_DEVICE_NAME+1];
	char *buf;
	int pt_num;
	int rest;
	uint beg_blk,next_blk;
	DEV_INFO *dinf;
	DEV_INODE *dind;
	PARTITON_TBL *pt;
	int i;



	/* ץγǧ */
	if(isMaxDevOpen()==0)return -EMDEV;

	/* "."ǥ쥯ȥν */
	for(;(next=cmpPathNull(".",path))!=NULL;path=next);

	/* Ǥ˥ץ󤵤Ƥ뤫ǧ */
	pt_num=0;
	if((dind=searchDevInode(path,pt_num,&dinf))!=NULL)
	{
		++dind->ref_count;
		return getInodeNum(dind);
	}

	/* ̾˥ѡƥֹ椬СѴ롣 */
	if((pt_num=getPartitionNum(path,name))<0)return pt_num;

	/* ǥХϿƤ뤫ǧ */
	dinf=NULL;
	searchDevInode(name,pt_num,&dinf);
	if(dinf==NULL)return -ENODEV;

	/* Open device. */
	if(dinf->open()<0)return -EIO;

	/*
	 * ѡƥơ֥򸡺롣
	 */
	if((buf=(char*)kmalloc(dinf->sector_size))==NULL)return -ENOMEM;
	if((rest=dinf->read(buf,1,0))<0)goto ERR;
		
	/* ѡƥơ֥ǧ */
	if(*(ushort*)(buf+PT_MAGIC_ADDR)!=PT_MAGIC_NUMBER)
	{
		rest=-ENODEV;
		goto ERR;
	}
	pt=(PARTITON_TBL*)(buf+PT_BEGIN);
	if(pt_num<=PT_NUM)
	{
		if(pt[--pt_num].type==PT_TYPE_EXT)		/* ΰγĥѡƥϳʤ */
		{
			rest=-ENODEV;
			goto ERR;
		}
		dind=makeInode(name,pt[pt_num].lba_begin,pt[pt_num].size,pt[pt_num].type,pt_num+1,dinf);
	}
	else
	{
		/* ĥѡƥǧ */
		for(i=0;pt[i].type!=PT_TYPE_EXT;++i)
			if(i>=PT_NUM)
			{
				rest=-ENODEV;
				goto ERR;
			}

		/* ĥѡƥ󥯤򤿤ɤ롣 */
		beg_blk=next_blk=pt[i].lba_begin;
		for(i=PT_NUM;;)
		{
			if((rest=dinf->read(buf,1,next_blk))<0)goto ERR;
			if(++i==pt_num)break;
			if(pt[1].type!=PT_TYPE_EXT)
			{
				rest=-ENODEV;
				goto ERR;
			}
			next_blk=beg_blk+pt[1].lba_begin;
		}
		dind=makeInode(name,pt[0].lba_begin+next_blk,pt[0].size,pt[0].type,pt_num,dinf);
	}
	kfree(buf);
	dind->ref_count=1;

	return getInodeNum(dind);
ERR:
	kfree(buf);

	return rest;
}

/*
 * parameters : device inode,inode block
 */
static int close(int din,uint inode)
{
	if(--dinode[inode].ref_count<0)dinode[inode].ref_count=0;

	return 0;
}

/*
 * parameters : device inode,inode,Buffer,Transfer bytes,begin byte
 */
static int read(int din,uint inode,void *buf,size_t size,size_t begin)
{
	char *tmp=NULL;
	int sect_siz;		/*  */
	int rest=size;		/* ĤžХȿ */
	int beg_sect;		/* žϤΥֹ */
	int s,t;


	if((sect_siz=dinode[inode].dev->sector_size)==0)
	{
		/* 饯ǥХ */
		return dinode[inode].dev->read(buf,size,0);
	}

	beg_sect=begin/sect_siz;

	/* ǽΥ̤ž */
	if((s=begin%sect_siz)!=0)
	{
		if((tmp=(char*)kmalloc(sect_siz))==NULL)return -1;
		if(read_cache(din,tmp,1,beg_sect)!=1)return 0;
		t=sect_siz-s;
		if(t>size)
		{
			memcpy(buf,tmp+s,size);
			kfree(tmp);
			return size;
		}
		memcpy(buf,tmp+s,t);
		(char*)buf+=t;
		beg_sect+=1;
		rest-=t;
	}

	s=rest/sect_siz;
	if((t=read_cache(din,buf,s,beg_sect))<s)
	{
		kfree(tmp);
		return size-rest+t*sect_siz;
	}

	/* Ĥ꤬̤ž */
	if(s*sect_siz<rest)
	{
		t=s*sect_siz;
		if((tmp==NULL)&&((tmp=(char*)kmalloc(sect_siz))==NULL))return -1;
		if(read_cache(din,tmp,1,beg_sect+s)!=1)
		{
			kfree(tmp);
			return size-rest+t;
		}
		memcpy((char*)buf+t,tmp,rest-t);
	}

	kfree(tmp);

	return size;
}

/*
 * parameters : device inode,inode,Buffer,Transfer bytes,begin byte
 */
static int write(int din,uint inode,void *buf,size_t size,size_t begin)
{
	char *tmp=NULL;
	int sect_siz;		/*  */
	int rest=size;		/* ĤžХȿ */
	int beg_sect;		/* žϤΥֹ */
	int s,t;


	if((sect_siz=dinode[inode].dev->sector_size)==0)
	{
		/* 饯ǥХ */
		return dinode[inode].dev->write(buf,size,0);
	}

	beg_sect=begin/sect_siz;

	/* ǽΥ̤ž */
	if((s=begin%sect_siz)!=0)
	{
		if((tmp=(char*)kmalloc(sect_siz))==NULL)return -1;
		if(read_cache(din,tmp,1,beg_sect)!=1)return 0;
		if((t=sect_siz-s)>size)
		{
			memcpy(tmp+s,buf,size);
			if(write_cache(din,tmp,1,beg_sect)!=1)return 0;
			kfree(tmp);
			return size;
		}
		memcpy(tmp+s,buf,t);
		if(write_cache(din,tmp,1,beg_sect)!=1)return 0;
		(char*)buf+=t;
		beg_sect+=1;
		rest-=t;
	}

	s=rest/sect_siz;
	if((t=write_cache(din,buf,s,beg_sect))<s)
	{
		kfree(tmp);
		return size-rest+t*sect_siz;
	}

	/* Ĥ꤬̤ž */
	if(s*sect_siz<rest)
	{
		t=s*sect_siz;
		if((tmp==NULL)&&((tmp=(char*)kmalloc(sect_siz))==NULL))return -1;
		if(read_cache(din,tmp,1,beg_sect+s)!=1)
		{
			kfree(tmp);
			return size-rest+t;
		}
		memcpy(tmp,(char*)buf+t,rest-t);
		if(write_cache(din,tmp,1,beg_sect+s)!=1)
		{
			kfree(tmp);
			return size-rest+t;
		}
	}

	kfree(tmp);

	return size;
}


static int opendir(const char *path,int din,uint dir_inode,DIR *dir)
{
	const char *next;


	for(;(next=cmpPathNull(".",path))!=NULL;path=next);
	if(*path!='\0')return -ENOENT;

	dir->index=0;

	return 0;
}


static inline void itoa(int v,char *buf)
{
	char tmp[4];
	int i;


	for(i=0;;++i)
	{
		tmp[i]=v%10+'0';
		if((v/=10)==0)break;
	}
	buf[i+1]='\0';
	while(i>=0)*buf++=tmp[i--];
}

static int readdir(int din,uint inode,DIR *dir,char *name)
{
	char tmp[4];
	DEV_INODE *dind;


	if((dind=getNextInode((DEV_INODE*)dir->index))==NULL)
		*name='\0';
	else
	{
		strcpy(name,dind->dev->name);
		if(dind->pt_num>0)
		{
			itoa(dind->pt_num,tmp);
			strcat(name,tmp);
		}
		dir->index=(uint)dind;
	}

	return 0;
}


static int stat(uint inode,const char *path,int din,uint dir_inode,FSTAT *fstat)
{
	return 0;
}


/*
 * parameters : device inode,inode,Command,Parameter struct pointer
 */
static int ioctl(int din,uint ionode,int com,void *prm)
{
	return 0;
}


static uint mount(int din)
{
	return -EDNOFUNC;
}


static int umount(int din)
{
	return -EDNOFUNC;
}


static int rename(int din,uint old_dinode,const char *old_path,uint new_dinode,const char *new_path)
{
	return -EDNOFUNC;
}


static int creat(const char *path,int din,uint dir_inode,int mode)
{
	return -EDNOFUNC;
}


static int mkdir(const char *path,int din,uint dir_inode,int mode)
{
	return -EDNOFUNC;
}


static int delete(const char *path,int din,uint dir_inode,int type)
{
	return -EDNOFUNC;
}


static FS dev_fs={
	"dev_fs",mount,umount,open,close,read,write,ioctl,rename,creat,opendir,mkdir,stat,delete,readdir
};


/*
 * å夫ǥХ˥ǡ᤹
 */
int sys_sync()
{
	writeBackAllCache();

	return 0;
}


/************************************************************************************************
 *
 * 
 *
 ************************************************************************************************/


/*
 * Init device system
 * return : 0 or error=-1
 */
int init_device()
{
	int rest;


	/* ǥХե륷ƥν */
	initDevFs();

	/* IOå */
	init_cache();

	/* ٱ񤭹ѥư롣 */
	start_delay_write_task();

	/* ե륷ƥϿ롣 */
	if((rest=regist_fs(&dev_fs))<0)return rest;

	/* ǥХե륷ƥޥȤ롣 */
	if((rest=mount_dev_fs(&dev_fs))<0)return rest;

	return 0;
}
/****************************************************************************/
void test_device()
{
	printk("buf=%x\n",*(uint*)0x158f58);
}
/*****************************************************************************/
