/*
 * console.c
 *
 * Copyright 2002, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 */


#include"types.h"
#include"lib.h"
#include"lock.h"
#include"device.h"
#include"proc.h"
#include"interrupt.h"
#include"signal.h"
#include"keymap.h"
#include"term.h"
#include"errno.h"
#include"console.h"


/***************************************************************************************
 *
 * ü饹
 *
 ***************************************************************************************/

/* PRIVATE */
static PROC *topProc=NULL;	/* üοƥץ */


/*
 * PRIVATE
 * ҥץƱGIDΥץSIGINT롣
 * parameters : process
 */
static void sendSigint(PROC *p,int gid)
{
	if((p->child!=NULL)&&(p->child->gid)==gid)sendSigint(p->child,gid);
	if(p->gid==gid)sendSignal(p->signal_struct,SIGINT);
	if(p->brother!=NULL)sendSigint(p->brother,gid);
}


/*
 * PUBLIC
 * Console termios struct.
 */
static struct termios termio={
	ICRNL|IXOFF|IXON,	/* c_iflag. */
	OPOST,				/* c_oflag. */
	CS8|CREAD,			/* c_cflag. */
	ISIG|ICANON|ECHO,	/* c_lflag. */
};


/*
 * PUBLIC
 * Init terminal.
 */
static inline void initTerm()
{
	/* termiosꡣ */
	termio.c_cc[VINTR]=	'c'&0x1f;
	termio.c_cc[VQUIT]=	'/'&0x1f;
	termio.c_cc[VSTART]='q'&0x1f;
	termio.c_cc[VSTOP]=	's'&0x1f;
	termio.c_cc[VSUSP]=	'z'&0x1f;
	termio.c_cc[VEOF]=	'd'&0x1f;
	termio.c_cc[VEOL]=	0;
	termio.c_cc[VERASE]='h'&0x1f;
	termio.c_cc[VKILL]=	'u'&0x1f;
}


/*
 * GLOBAL
 * Set top process.
 * Note!
 *
 * return : 0 or error number
 */
int cnsoleSetTopProc()
{
	if(topProc==NULL)
	{
		topProc=get_current_task();
		return 0;
	}
	else return -EBUSY;
}


/***************************************************************************************
 *
 * ǥץ쥤饹
 *
 * ǽ
 * ʸΰ
 * ΰư
 * 륢åס
 *
 ***************************************************************************************/

enum{
	/* VGA Video ram. */
	VRAM_BASE_ADDR=0xb8000,		/* Start address of video memory */
	VRAM_SIZE=0x8000,		/* Size of video memory */

	/* CRT IO registers. */
	CTL_ADDR=0x3d4,			/* Addres register */
	CTL_DATA=0x3d5,			/* Data IO register */
	START_ADDR_HIGH=0xc,	/* High byte of print start address */
	START_ADDR_LOW=0xd,		/* Low  byte of print start address  */
	CURSOR_ADDR_HIGH=0xe,	/* High byte of cursor address */
	CURSOR_ADDR_LOW=0xf,	/* Low  byte of cursor address */

	SCR_LINES=25,			/* Number of lines. */
	SCR_COLUMNS=80,			/* Number of columns. */
	SCR_LAST=				/* Video ram last address. */
		VRAM_SIZE/2-SCR_LINES*SCR_COLUMNS-SCR_COLUMNS*3,
};


typedef struct{
	int viewTop;	/* ӥǥ꡼β̤γϰ֡ */
	int start;		/* ϰ֡ */
	int cursor;		/* ֡ */
}SCREEN;


/* PRIVATE */
static char *vram=(char*)VRAM_BASE_ADDR;
static SCREEN screen={0,0,0};


/*
 * PRIVATE
 * Update cursor position.
 */
static inline void updateCursor()
{
	outb(CTL_ADDR,CURSOR_ADDR_HIGH);
	outb(CTL_DATA,screen.cursor>>8);
	outb(CTL_ADDR,CURSOR_ADDR_LOW);
	outb(CTL_DATA,screen.cursor);
}


/*****************************************************
static void testPrint(int pos,int uvalue)
{
	char tmp_buf[12];
	int i;


	for(i=0;;++i)
	{
		tmp_buf[i]=uvalue%10+'0';
		if(!(uvalue/=10))break;
	}
	for(;i>=0;--i,++pos)vram[pos*2]=tmp_buf[i];
}
*******************************************************/


/*
 * PRIVATE
 * Hard scroll up.
 */
static void scrollup()
{
	char *p;
	int i,last;


	screen.viewTop=ROUNDUP(screen.cursor-SCR_COLUMNS*SCR_LINES+1,SCR_COLUMNS);

	if(screen.viewTop>=SCR_LAST)
	{
		/* Video ramƬ˰ư */
		p=&vram[screen.viewTop*2];
		last=(screen.cursor-screen.viewTop)*2;
		for(i=0;i<last;i+=2)vram[i]=p[i];

		/* Video ram0ꥢ */
		for(;i<(SCR_LAST+SCR_COLUMNS*SCR_LINES)*2;i+=2)vram[i]=0;

		screen.start=screen.cursor-=screen.viewTop;
		screen.viewTop=0;
	}

	/* Hard scroll up */
	outb(CTL_ADDR,START_ADDR_HIGH);
	outb(CTL_DATA,screen.viewTop>>8);
	outb(CTL_ADDR,START_ADDR_LOW);
	outb(CTL_DATA,screen.viewTop);
}


/*
 * PUBLIC
 * 1ʸ
 * parameters : buffer,size
 */
static inline void putIntoScreen(char c)
{
	vram[screen.start*2]=c;
	++screen.start;
}


/*
 * PUBLIC
 * ʣʸ
 * parameters : buffer,size
 */
static inline void putNumIntoScreen(char *buf,int size)
{
	int i,last;


	for(i=screen.start*2,last=i+size*2;i<last;i+=2)vram[i]=*buf++;
}


/*
 * PUBLIC
 * 
 */
/* Left cursor. */
static inline void leftCursor(int n)
{
	screen.start=screen.cursor-=n;
	updateCursor();
}

/* Right cursor. */
static inline void rightCursor(int n)
{
	screen.start=screen.cursor+=n;
	if(screen.cursor-screen.viewTop>=SCR_COLUMNS*SCR_LINES)
		scrollup();
	updateCursor();
}

/* Carriage retun. */
static inline void carriageRetun()
{
	screen.start=screen.cursor=ROUNDDOWN(screen.cursor,SCR_COLUMNS);
	updateCursor();
}

/* New line. */
static inline void newLine()
{
	screen.start=screen.cursor=ROUNDUP(screen.cursor+1,SCR_COLUMNS);
	if(screen.cursor-screen.viewTop>=SCR_COLUMNS*SCR_LINES)
		scrollup();
	updateCursor();
}


/*
 * PUBLIC
 * 
 */
static void initScreen()
{
	short *p;


	/* Start position 0 */
	outb(CTL_ADDR,START_ADDR_HIGH);
	outb(CTL_DATA,0);
	outb(CTL_ADDR,START_ADDR_LOW);
	outb(CTL_DATA,0);

	/* Clear screen */
	for(p=(short*)VRAM_BASE_ADDR;p<(short*)(VRAM_BASE_ADDR+VRAM_SIZE);++p)*p=0x700;
}


/***************************************************************************************
 *
 * 󥽡Хåե饹
 *
 ***************************************************************************************/

 enum{
	CONSOLE_BUF_SIZE=90,	/* Line buffer size. */
};


typedef struct{
	int crt;					/* ȥݥ */
	int last;					/* 饹ȥݥ */
	int lineTop;				/* 饤ȥåץݥ */
	char buf[CONSOLE_BUF_SIZE];	/* 󥽡Хåե */
}CONSOL_BUF;


/* PUBLIC */
static CONSOL_BUF lbuf;


/*
 * PRIVATE
 * ꡼Ǹ夫饳ԡ
 * parameters : destination buffer,source buffer
 */
static inline void memcpyBack(char *dst,char *src,int n)
{
	while(--n>=0)dst[n]=src[n];
}


/*
 * PUBLIC
 * 󥽡Хåե
 */
/* 1ʸɲá */
static inline void putToBuf(char c)
{
	if(lbuf.crt==CONSOLE_BUF_SIZE)return;

	/* Ȱʹߤʸ˰ư */
	if(lbuf.last==CONSOLE_BUF_SIZE)--lbuf.last;
	memcpyBack(lbuf.buf+lbuf.crt+1,lbuf.buf+lbuf.crt,lbuf.last-lbuf.crt);
	++lbuf.last;

	/* ʸ */
	lbuf.buf[lbuf.crt]=c;
	putNumIntoScreen(lbuf.buf+lbuf.crt,lbuf.last-lbuf.crt);
	rightCursor(1);
	++lbuf.crt;
}

/* Delete. */
static inline void deleteBuf()
{
	/* +1ʹߤʸ˰ưǸ˶ */
	if(lbuf.last>lbuf.crt)
	{
		memcpy(lbuf.buf+lbuf.crt,lbuf.buf+lbuf.crt+1,lbuf.last-lbuf.crt);
		lbuf.buf[lbuf.last-1]='\0';
		putNumIntoScreen(lbuf.buf+lbuf.crt,lbuf.last-lbuf.crt);
		--lbuf.last;
	}

}

/* Back space. */
static inline void backSpaceBuf()
{
	/* Ȱʹߤʸ˰ưǸ˶ */
	if(lbuf.crt>lbuf.lineTop)
	{
		memcpy(lbuf.buf+lbuf.crt-1,lbuf.buf+lbuf.crt,lbuf.last-lbuf.crt);
		lbuf.buf[lbuf.last-1]='\0';
		--lbuf.crt;
		leftCursor(1);
		putNumIntoScreen(lbuf.buf+lbuf.crt,lbuf.last-lbuf.crt);
		--lbuf.last;
	}
}

/* Left cursor. */
static inline void leftCursorBuf()
{
	if(lbuf.crt>lbuf.lineTop)
	{
		leftCursor(1);
		--lbuf.crt;
	}
}

/* Right cursor. */
static inline void rightCursorBuf()
{
	if(lbuf.crt<CONSOLE_BUF_SIZE)
	{
		rightCursor(1);
		++lbuf.crt;
		if(lbuf.last<lbuf.crt)lbuf.last=lbuf.crt;
	}
}

/* Erase. */
static inline void eraseLine()
{
	memset(lbuf.buf+lbuf.lineTop,0,lbuf.last-lbuf.lineTop);
	leftCursor(lbuf.crt-lbuf.lineTop);
	putNumIntoScreen(lbuf.buf+lbuf.lineTop,lbuf.last-lbuf.lineTop);
	lbuf.crt=lbuf.last=lbuf.lineTop;
}

/* New line. */
static inline void newLineBuf()
{
	newLine();
	if(lbuf.last<CONSOLE_BUF_SIZE)
		lbuf.buf[lbuf.last++]='\n';
	lbuf.lineTop=lbuf.crt=lbuf.last;
}


/*
 * PUBLIC
 * Check console buffer.
 * parameters : user buffer,max copy size
 * return : ϥХȿ
 */
static int readConsoleBuf(uchar *user_buf,size_t size)
{
	int i;


	if(termio.c_lflag&ICANON)	/* Canonical mode. */
	{
		if(lbuf.lineTop==0)return 0;

		i=0;
		for(;lbuf.buf[i]!='\n';++i);
		size=(size>i)?i:size;
		memcpy(user_buf,lbuf.buf,size);
		++i;
		memcpy(lbuf.buf,lbuf.buf+i,lbuf.last-i);
		lbuf.crt-=i;
		lbuf.last-=i;
		lbuf.lineTop-=i;
	}
	else	/* Non-canonical mode. */
	{
/************************ NON CANONICAL MODE ̤ ***************************/
	}

	return size;
}


/***************************************************************************************
 *
 * ܡɥ饹
 *
 ***************************************************************************************/

enum{
	IRQ_NO=1,		/* IRQ number */
};


/* GLOBAL */
static PROC_LINK waitLink={&waitLink,&waitLink};	/* Ԥץ󥯡 */


/*
 * PRIVATE
 * Do control command in canonical mode.
 * parameters : character
 * return : 1=ƤӽФȤϽλ or 0
 */
static inline int doCanonTermCtr(uchar ch)
{
	uchar *cc;


	if(termio.c_lflag&ICANON)
	{
		cc=termio.c_cc;
		if(ch==cc[VINTR])			/* Send SIGINT. */
		{
			/* ҥץƱGIDSIGINT롣 */
			if(topProc->child!=NULL)
				sendSigint(topProc->child,topProc->gid);
			return 1;
		}
		else if(ch==cc[VQUIT])		/* Send SIGQUIT. */
		{
			return 0;
		}
		else if(ch==cc[VSTART])		/* Send SIGCONT.  */
		{
			return 0;
		}
		else if(ch==cc[VSTOP])		/* Send SIGSTOP. */
		{
			return 0;
		}
		else if(ch==cc[VSUSP])		/* Send SIGTSTP. */
		{
			return 0;
		}
		else if(ch==cc[VEOF])		/* եνλ */
		{
			return 0;
		}
		else if(ch==cc[VEOL])		/* Ԥνλ */
		{
			newLineBuf();
			return 1;
		}
		else if(ch==cc[VERASE])		/* ʸõ */
		{
			backSpaceBuf();
			return 1;
		}
		else if(ch==cc[VKILL])		/* á */
		{
			eraseLine();
			return 1;
		}
	}

	return 0;
}


/*
 * PRIVATE
 * Turn on/off LED.
 */
static inline void switchLed(uchar value)
{
	outb(KEYBOARD_OUTPUT_REG,0xed);		/* Command turn on/of LED. */
	while(inb(KEYBOARD_STAT_REG)&2);
	outb(KEYBOARD_OUTPUT_REG,value);	/* LED on/off. */
}


/*
 * PRIVATE
 * keyboard main handler.
 * parameters : scan key coad
 * return : task switch on=1 or off=0
 */
static int keyboardHnadler(uchar key_coad)
{
	enum{
		KEY_BREAK=0x80,	/* key break bit. */

		/* Add key bit. */
		SHT_BIT=1<<0,
		CTR_BIT=1<<1,

		/* LED bit. */
		LED_SCRL=1<<0,	/* ScrollLock LED. */
		LED_NMLC=1<<1,	/* NumLock LED. */
		LED_CPSL=1<<2,	/* CapsLock LED. */
	};

	static int add_key=0;				/* normal=0,add shift=1,add ctrl=2. */
	static int func_key=0;				/* ǽե饰(󥳡0xe0) */
	static int shift_key=0;				/* Shift key flag. */
	static int led=0;					/* LED flag, */
	static int ctr_key=0;				/* Ctrl key. */
	static uchar add_map[]={0,1,2,2};	/* ղåѴޥåס */

	uchar map_coad;


	/* ǽν */
	if(func_key)
	{
		if((key_coad>=VMAP_START_COAD)&&(key_coad<=VMAP_END_COAD))
			key_coad+=VMAP_ADD;
		func_key=0;
	}
	else if(key_coad==KEY_FUNC)
	{
		func_key=1;
		return 0;
	}

	/* ޥåץɼ */
	if((map_coad=keymap[key_coad&~KEY_BREAK][add_key])==0)return 0;

	/* ֥졼ɤθCtrl ShiftʳϼΤƤ롣 */
	if(key_coad&KEY_BREAK)
	{
		switch(map_coad)
		{
			case MAP_SHT:
				shift_key=0;
				break;
			case MAP_CTR:
				ctr_key=0;
				break;
			default:
				return 0;
		}
		add_key=add_map[(shift_key|ctr_key)^(led>>2)];

		return 0;
	}
	
	if(termio.c_lflag&ICANON)		/* Canonical mode. */
	{
		/* ȥ륳ޥɤθ */
		if(map_coad<' ')
		{
			if(doCanonTermCtr(map_coad))return 0;
			switch(map_coad)
			{
				case '\r':
					if(termio.c_iflag&IGNCR)break;
				case '\n':
					newLineBuf();
					return 1;
			}
		}
		else
		{
			if(map_coad<MIN_MAPKEY)putToBuf(map_coad);
			else
				switch(map_coad)
				{
					case MAP_BS:
						backSpaceBuf();
						break;
					case MAP_DEL:
						deleteBuf();
						break;
					case MAP_LFT:
						leftCursorBuf();
						break;
					case MAP_RHT:
						rightCursorBuf();
						break;			
					case MAP_SHT:
						shift_key=SHT_BIT;
						add_key=add_map[(shift_key|ctr_key)^(led>>2)];
						break;
					case MAP_CPSL:
						switchLed(led^=LED_CPSL);
						add_key=add_map[(shift_key|ctr_key)^(led>>2)];
						break;
					case MAP_CTR:
						ctr_key=CTR_BIT;
						add_key=add_map[(shift_key|ctr_key)^(led>>2)];
						break;
				}
		}	
	}
	else						/* Non-canonical mode. */
	{
		/************************ ̤ ***************************/
	}

	return 0;
}


/*
 * GLOBAL
 * keyboard interrupt handler.
 * return : task switch on off
 */
static int keyboardIntrHandler()
{
	if(keyboardHnadler(inb(KEYBOARD_OUTPUT_REG)))
	{
		wakeProc(&waitLink);
		return 1;
	}
	return 0;
}


/***************************************************************************************
 *
 * ƥॳ륯饹
 *
 ***************************************************************************************/

static int open()
{
	return 0;
}

/* size=žǽХȿ */
static int read(void *buf,size_t size,size_t start)
{
	int len;


	/* Ϥ뤫󥽡Хåեǧ */
	if((len=readConsoleBuf(buf,size))==0)
	{
		waitProc(&waitLink);			/* Ԥ */
		len=readConsoleBuf(buf,size);
	}

	return len;
}


int writeConsole(void *buf,size_t size,size_t start)
{
	char *str=(char*)buf;
	int cursol;
	int i;


	cursol=0;
	i=0;
	for(;i<size;++i)
	{
		if(str[i]>=' ')
		{
			putIntoScreen(str[i]);
			++cursol;
		}
    	else if(str[i]=='\n')
    	{
    		newLine();
    		cursol=0;
    	}
    	else if(str[i]=='\r')
    	{
    		carriageRetun();
    		cursol=0;
    	}
    }
    rightCursor(cursol);

	return i;
}


/***************************************************************************************
 *
 * 饹
 *
 ***************************************************************************************/

/* PRIVATE,Device interface */
static DEV_INFO devInfo={
	"console",0,0,0,open,read,writeConsole,NULL
};


/*
 * CLOBAL
 * ǥХϿ򤹤롣
 */
int registConsole()
{
	return regist_device(&devInfo);
}


/*
 * GLOBAL
 * Init console
 */
int initConsole()
{
	/* ꡼ν */
	initScreen();

	/* üǽν */
	initTerm();

	return 0;
}


/*
 * GLOBAL
 * init keyboard driver
 */
void initKeyboard()
{
	/* Turn off LED. */
	switchLed(0);

	/* ߤ */
	irq_entry[IRQ1]=keyboardIntrHandler;

	release_irq_mask(IRQ_NO);

	/* clear data register */
	inb(KEYBOARD_OUTPUT_REG);
}
