/*
 * orig_fs.c
 *
 * Copyright 2002, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 *
 * ꥸʥե륷ƥ
 */


#include"config.h"
#include"types.h"
#include"lib.h"
#include"errno.h"
#include"mm.h"
#include"fs.h"
#include"device.h"
#include"orig_fs.h"


enum{
	PT_TYPE_ORIG=0x9a,			/* Partition type numberʲ */
	SBLOCK_MAGIC=0xdddddddd,	/* superblock magic numberʲˡ */
	SBLOCK_SECTER=2,			/* ѡ֥åϥ */
	BLOCK_SIZE=4096,			/* ֥å */
	MAX_NAME_SIZE=127,			/* ե̾ */
	MAX_DIR_NUM=4,				/* ǥ쥯ȥ֥å */
};


/* Super block entry */
typedef struct{
	uint magic_number;		/* ޥȻγǧѥޥåʥС */
	int secters;			/* block size/secter size */
	size_t all_blk;			/* All blocks */
	size_t useable_blk;		/* Useable blocks */
	uint root_blk;			/* root block number */
	uint empty_blk;			/* ߤζ֥åǥå֥å */
}SUPER_BLOCK;

/* ֥å */
typedef struct{
	int current;		/* ֥åθ */
	uint empty[0];		/* ֥å */
}ENPTY_BLOCK;

/* Inode */
typedef struct{
	ushort owner_id;			/* Owner UID */
	ushort group_id;			/* Group ID */
	size_t size;				/* File size by byte */
	uint64 access_time;			/* Last access time */
	uint64 create_time;			/* Create time */
	uint64 modificat_time;		/* Last modification time */
	uint my_secter;				/* my block number */
	ushort file_type;			/* File type */
	ushort indirect;			/* ܻȥΡɿ */
	uint secter[0];				/* file secter number */
}INODE;

/*
 * Directory entry
 * ؿdel_entry()Ǥ֤˰¸Ƥ롣
 */
typedef struct DIR_ENT{
	uint secter;		/* secter number */
	uchar type;			/* File or directory */
	uchar size;			/* entry size */
	uchar low_dir;		/* low entry directory */
	uchar high_dir;		/* high entry directory */
	ushort low;			/* low entry offset */
	ushort high;		/* high entry offset */
	ushort low_num;		/* lowȥ꡼ */
	ushort high_num;	/* highȥ꡼ */
	char name[0];		/* Name string */
}DIR_ENT;

/* Directory empty entry */
typedef struct{
	ushort prev;	/* prev empty entry */
	ushort next;	/* next empty entry */
	int size;		/* entry size */
}EMPTY_ENT;

/* Directory top */
typedef struct{
	uint my_secter;			/* my block number */
	uint next_dir;			/* next directory link */
	ushort empty_prev;		/* empty entry offset */
	ushort empty_next;		/* empty entry offset */
	uchar entry_dir;		/* Top entry direcctory */
	ushort entry_top;		/* Top entry offset */
	ushort size;			/* directory size */
	ushort owner_id;		/* Owner UID */
	ushort group_id;		/* Group ID */
	uint64 access_time;		/* Last access time */
	uint64 create_time;		/* Create time */
	uint64 modificat_time;	/* Last modification time */
	DIR_ENT entry[0];		/* begin entry */
}DIR;


#define ENTRY(entry,buf) ((DIR_ENT*)(entry+(uint)buf))
#define EMPTY(empty,buf) ((EMPTY_ENT*)(empty+(uint)buf))


static SUPER_BLOCK *super_blk[MAX_DEVICE_OPEN];
static ENPTY_BLOCK *empty_blk[MAX_DEVICE_OPEN];


static uint mount(int din);
static int umount(int din);
static int open(const char *path,int din,uint dir_blk);
static int close(int din,uint inode);
static int read(int din,uint ind,void *buf,size_t size,size_t begin);
static int write(int din,uint ind,void *buf,size_t size,size_t begin);
static int mkdir(const char *path,int din,uint dir_blk);
static int creat(const char *path,int din,uint dir_blk);
static int opendir(const char *path,int din,uint dir_blk);
static int ioctr(int din,uint inode,int cmd,void *param);
static int rename(int din,uint inode,const char *name);

static FS orig_fs={
	"orig_fs",mount,umount,open,close,read,write,ioctr,rename,creat,opendir,mkdir
};


/*
 * cmpare path strings
 * parameters : Destination string(end='\0'),Sorce string(end='/' or '\0')
 * return : =Υѥݥ or =1 or =-1;
 */
extern inline int cmp_path(const char *path1,const char *path2)
{
	while(*path1==*path2++)
		if(*path1++=='\0')return (int)path2-1;

	if((*(path2-1)=='/')&&(*path1=='\0'))return (int)path2;
	if(*path1<*(path2-1))return 1;

	return -1;
}


/*
 * ǸΥѥʸ󤫤ɤȽꤹ롣
 * parameters : path
 * return : Ǹ 1,ǸǤʤ 0
 */
extern inline int is_last_path(const char *path)
{
	if(*path=='\0')return 1;

	while(*path++!='/')
		if(*path=='\0')return 1;

	return 0;
}


/************************************************************************************
 *
 * block  functions
 *
 ************************************************************************************/

/*
 * ֥å롣
 * parameters : device inode
 * return : empty block secter or error=0
 */
uint get_empty_blk(int din)
{
	uint secter;


	if((secter=empty_blk[din]->empty[empty_blk[din]->current++])==0)return 0;

	if(empty_blk[din]->current>(BLOCK_SIZE-sizeof(ENPTY_BLOCK))/sizeof(uint))
		if(read_cache(din,empty_blk[din],super_blk[din]->secters,secter)!=super_blk[din]->secters)
			return 0;

	return secter;
}


/*
 * ֥åơ֥åǥå˲ä롣
 * parameters :
 * return : 0 or error=-1
 */
int release_blk(int din,uint blk)
{
	if(--empty_blk[din]->current<0)
	{
		if((write_direct(din,empty_blk[din],super_blk[din]->secters,blk))!=super_blk[din]->secters)
			return -1;
		empty_blk[din]->current=(BLOCK_SIZE-sizeof(ENPTY_BLOCK))/sizeof(uint);
	}

	empty_blk[din]->empty[empty_blk[din]->current]=blk;

	return 0;
}


/***********************************************************************************
 *
 * directory functions
 * ȥ꡼ʬڹ¤ˤ롣
 *
 ************************************************************************************/

/*
 * ȥ꡼low¦˲ž롣
 * parameters : directory buffer number pointer,entry offset pointer,directory buffer
 * return : new top entry
 */
static void round_low(uchar *dn,ushort *entry,DIR **dir)
{
	uchar a;
	ushort b;
	DIR_ENT *top,*high;


	top=ENTRY(*entry,dir[*dn]);
	if((high=ENTRY(top->high,dir[top->high_dir]))==(DIR_ENT*)dir[0])return;

	/* topȥ꡼¸ */
	a=*dn;
	b=*entry;

	/* highȥ꡼topοƥȥ꡼λҥΡɤ˰ư */
	*entry=top->high;
	*dn=top->high_dir;

	/* highȥ꡼lowȥ꡼tophighȥ꡼˰ư */
	top->high_dir=high->low_dir;
	top->high=high->low;
	top->high_num=high->low_num;

	/* tophighȥ꡼lowȥ꡼˰ư */
	high->low_dir=a;
	high->low=b;
	high->low_num=top->low_num+top->high_num;
}


/*
 * ȥ꡼high¦˲ž롣
 * parameters : directory buffer number pointer,entry offset pointer,directory buffer
 * return : new top entry
 */
static void round_high(uchar *dn,ushort *entry,DIR **dir)
{
	uchar a;
	ushort b;
	DIR_ENT *top,*low;


	top=ENTRY(*entry,dir[*dn]);
	if((low=ENTRY(top->low,dir[top->low_dir]))==(DIR_ENT*)dir[0])return;

	/* topȥ꡼¸ */
	a=*dn;
	b=*entry;

	/* highȥ꡼topοƥȥ꡼λҥΡɤ˰ư */
	*entry=top->low;
	*dn=top->low_dir;

	/* highlowȥ꡼tophighȥ꡼˰ư */
	top->low_dir=low->high_dir;
	top->low=low->high;
	top->low_num=low->high_num;

	/* tophighlowȥ꡼˰ư */
	low->high_dir=a;
	low->high=b;
	low->high_num=top->low_num+top->high_num;
}


/*
 * ʬ٨ʿؿ
 * ¦Υȥ꡼ȿ¦Υȥ꡼2ܤ¿
 * ¦¦Υȥ꡼¦Υȥ꡼1/2꾯ʤ硢ȿ¦زž롣
 * parameters : directory buffer number pointer,entry offset pointer,directory buffer
 */
static void splay(uchar *dn,ushort *entry,DIR **dir)
{
	DIR_ENT *top;


	top=ENTRY(*entry,dir[*dn]);
	if((top->high_num>top->low_num*2)&&(ENTRY(top->high,dir[top->high_dir])->low_num<top->high_num/2))
		round_low(dn,entry,dir);
	else if((top->low_num>top->high_num*2)&&(ENTRY(top->low,dir[top->low_dir])->high_num<top->low_num/2))
		round_high(dn,entry,dir);
}


/*
 * ǥ쥯ȥƱ̾Υȥ꡼õ
 * ҥåȤ硢name˼ΥѥƬݥ󥿤֤
 * parameters : name address pointer,directory buffer
 * return : directory entry or error=NULL
 */
DIR_ENT *search_same_entry(const char **name,DIR **dir)
{
	int r;
	DIR_ENT *p;


	for(p=ENTRY(dir[0]->entry_top,dir[dir[0]->entry_dir]);p!=(DIR_ENT*)dir[0];)
	{
		r=cmp_path(p->name,*name);
		if(r==-1)p=ENTRY(p->low,dir[p->low_dir]);
		else if(r==1)p=ENTRY(p->high,dir[p->high_dir]);
		else
		{
			*name=(const char*)r;
			return p;
		}
	}

	return NULL;
}


/*
 * ǥ쥯ȥǰ̾οƥȥ꡼롣
 * ǥ쥯ȥκǽΥȥ꡼¸ߤ뤳ȤȤ롣
 * parameters : name,directory buffer pointer,entry directory buffer pointer,entry offset pointer
 * return : 0 or error=-1
 */
int search_prev_entry(const char *name,DIR **dir,uchar **dn,ushort **ent)
{
	uchar *ent_dir;
	ushort *entry;
	int r;
	DIR_ENT *p=NULL;


	for(ent_dir=&dir[0]->entry_dir,entry=&dir[0]->entry_top;*entry!=0;)
	{
		splay(ent_dir,entry,dir);
		p=ENTRY(*entry,dir[*ent_dir]);
		r=cmp_path(p->name,name);
		if(r==-1)
		{
			++p->low_num;
			ent_dir=&p->low_dir;
			entry=&p->low;
		}
		else if(r==1)
		{
			++p->high_num;
			ent_dir=&p->high_dir;
			entry=&p->high;
		}
		else return -1;
	}

	*dn=ent_dir;
	*ent=entry;

	return 0;
}


/*
 * ǥ쥯ȥζȥ꡼ơΥݥ󥿡롣
 * parameters : name size,directory pointer,directory buffer number pointer,entry offset pointer
 * return : directory entry pointer or error=NULL
 */
DIR_ENT *get_empty_entry(int path_len,DIR **dir,uchar *dn,ushort *entry)
{
	enum{MIM_EMPTY_SIZE=sizeof(DIR_ENT)+sizeof(int)};		/* ȥ꡼ǺǾλĤ */

	int entry_len;
	uchar i;
	ushort k,last;
	EMPTY_ENT *p,*q;


	/* ̾Υǧ */
	if(path_len>MAX_NAME_SIZE)return NULL;

	entry_len=ROUNDUP(path_len+1,sizeof(int))+sizeof(DIR_ENT);

	/* ȥ꡼򸡺 */
	last=(uint)dir[0]->empty_prev-(uint)dir[0];
	for(i=0;dir[i]!=NULL;++i)
	{
		for(k=dir[i]->empty_next;k!=last;k=p->next)
		{
			p=EMPTY(k,dir[i]);
			if(p->size>entry_len)
			{
				if(p->size>entry_len+MIM_EMPTY_SIZE)
				{
					q=EMPTY(k+entry_len,dir[i]);
					q->prev=p->prev;
					q->next=p->next;
					EMPTY(q->prev,dir[i])->next+=entry_len;
					EMPTY(q->next,dir[i])->prev+=entry_len;
					q->size=p->size-entry_len;
					memset(p,0,sizeof(DIR_ENT));
					((DIR_ENT*)p)->size=(uchar)entry_len;
				}
				else
				{
					EMPTY(p->prev,dir[i])->next=p->next;
					EMPTY(p->next,dir[i])->prev=p->prev;
					memset(p,0,sizeof(DIR_ENT));
					((DIR_ENT*)p)->size=p->size;
				}

				*dn=i;
				*entry=k;

				return (DIR_ENT*)p;
			}
		}
	}

	return NULL;
}


/*
 * ȥ꡼ɲä롣
 * parameters :
 */
void add_empty_entry(ushort entry,DIR *dir)
{
	ushort i,last;
	EMPTY_ENT *p,*q;


	i=dir->empty_next;
	p=(EMPTY_ENT*)&dir->empty_prev;

	/* եåȽ˶ȥ꡼Υ󥯤õ */
	for(last=(uint)&dir->empty_prev-(uint)dir;i!=last;i=p->next)
	{
		p=EMPTY(i,dir);
		if(i>entry)
		{
			/* ϰϤ³Ƥз礹롣 */
			if(i+p->size==entry)
			{
				p->size+=ENTRY(entry,dir)->size;
				return;
			}
			else break;
		}
	}

	/* ȥ꡼󥯤˲ä롣 */
	q=EMPTY(entry,dir);
	q->size=((DIR_ENT*)q)->size;
	q->next=p->next;
	q->prev=(uint)p-(uint)dir;
	p->next=entry;
	EMPTY(q->next,dir)->prev=entry;
}


/*
 * ȥ꡼ξʤزžʬǥȥ꡼򲼹ߤơ󥯤롣
 * parameters : path name,directory
 * return : 0 or error=-1
 */
int del_entry(const char *name,DIR **dir)
{
	uchar *ent_dir,i;
	ushort *entry,k;
	int r;
	DIR_ENT *p,*q;


	ent_dir=&dir[0]->entry_dir;
	entry=&dir[0]->entry_top;

	/* ȥ꡼򸡺롣 */
	for(;;)
	{
		if(*entry==0)return -1;

		p=ENTRY(*entry,dir[*ent_dir]);
		r=cmp_path(p->name,name);
		if(r==-1)
		{
			--p->low_num;
			ent_dir=&p->low_dir;
			entry=&p->low;
		}
		else if(r==1)
		{
			--p->high_num;
			ent_dir=&p->high_dir;
			entry=&p->high;
		}
		else break;
	}

	/* λޤ0ˤʤޤǲߤ롣 */
	while(p->low_num*p->high_num)
	{
		/* ʤޤ˲ž롣 */
		if(p->low_num<p->high_num)
		{
			round_low(ent_dir,entry,dir);
			q=ENTRY(*entry,dir[*ent_dir]);
			ent_dir=&q->low_dir;
			entry=&q->low;
		}
		else
		{
			round_high(ent_dir,entry,dir);
			q=ENTRY(*entry,dir[*ent_dir]);
			ent_dir=&q->high_dir;
			entry=&q->high;
		}
	}

	/*  */
	i=*ent_dir;
	k=*entry;
	*ent_dir=p->low_dir+p->high_dir;
	*entry=p->low+p->high;
	if(*entry!=dir[0]->entry_top)*(entry+2)=p->low_num+p->high_num;	/* DIR_ENT¤ΤǤ֤˰¸롣 */
	add_empty_entry(k,dir[i]);

	return 0;
}


/*
 * ֥åХåե˿Υǥ쥯ȥ롣
 * parameters : directory buffer,secter
 */
void make_dir(void *dir,uint secter)
{
	EMPTY_ENT *empty;


	((DIR*)dir)->my_secter=secter;
	((DIR*)dir)->next_dir=0;
	((DIR*)dir)->empty_prev=sizeof(DIR);
	((DIR*)dir)->empty_next=sizeof(DIR);
	((DIR*)dir)->entry_dir=0;
	((DIR*)dir)->entry_top=0;
	((DIR*)dir)->size=BLOCK_SIZE;
	((DIR*)dir)->owner_id=0;		/* ̤ */
	((DIR*)dir)->group_id=0;		/* ̤ */
	((DIR*)dir)->access_time=0;		/* ̤ */
	((DIR*)dir)->create_time=0;		/* ̤ */
	((DIR*)dir)->modificat_time=0;	/* ̤ */

	/* ȥ꡼ꡣ */
	empty=(EMPTY_ENT*)((DIR*)dir)->entry;
	empty->prev=(uint)&((DIR*)dir)->empty_prev-(uint)dir;
	empty->next=empty->prev;
	empty->size=BLOCK_SIZE-sizeof(DIR);
}


/*
 * ѥΥǥ쥯ȥꥨȥ꡼õ
 * parameters : path,device inode,directory block,directory pointer
 * return : directory entry or error=NULL
 */
DIR_ENT *search_dir_entry(const char *orig_path,int din,uint dir_blk,DIR **dir)
{
	const char *path=orig_path;
	int secters=super_blk[din]->secters;
	DIR_ENT *entry;
	int i;
	uint next_dir;


	if(*orig_path=='\0')return NULL;

	if((dir[0]=(DIR*)kmalloc(BLOCK_SIZE))==NULL)return NULL;
	for(;;)
	{
		/* ǥ쥯ȥХåեɤ߹ࡣ */
		if(read_cache(din,dir[0],super_blk[din]->secters,dir_blk)!=super_blk[din]->secters)return NULL;
		for(i=1,next_dir=dir[0]->next_dir;next_dir!=0;next_dir=dir[i++]->next_dir)
		{
			if(dir[i]==NULL)
				if((dir[i-1]=(DIR*)kmalloc(BLOCK_SIZE))==NULL)return NULL;
			if(read_cache(din,dir[0],secters,next_dir)!=secters)return NULL;
		}

		/* ǥ쥯ȥ򸡺 */
		if((entry=search_same_entry(&path,dir))==NULL)return NULL;
		if(*path=='\0')break;
		if(entry->type!=DIRECTORY)return NULL;
		dir_blk=entry->secter;
	}

	return entry;
}


/*
 * ѥ̾Υǥ쥯ȥꥨȥ꡼롣ȥ꡼ˤϥѥ̾롣
 * parameters : path,device inode,directory block,directory pointer
 * return : directory entry or error=NULL
 */
DIR_ENT *get_new_dir_entry(const char *orig_path,int din,uint dir_blk,DIR **dir)
{
	const char *path;
	const char *top;
	uchar *dn;
	ushort *ent;
	int secters=super_blk[din]->secters;
	DIR_ENT *entry;
	uint next_dir;
	int i,s;


	if(*orig_path=='\0')return NULL;

	/* ǸΥѥ̾Ƭ */
	top=orig_path;
	for(path=orig_path;*path!='\0';++path)
		if(*path=='/')top=path+1;

	if((dir[0]=(DIR*)kmalloc(BLOCK_SIZE))==NULL)return NULL;
	path=orig_path;
	for(;;)
	{
		/* ǥ쥯ȥХåեɤ߹ࡣ */
		if(read_cache(din,dir[0],super_blk[din]->secters,dir_blk)!=super_blk[din]->secters)return NULL;
		for(i=1,next_dir=dir[0]->next_dir;next_dir!=0;next_dir=dir[i++]->next_dir)
		{
			if(dir[i]==NULL)
				if((dir[i-1]=(DIR*)kmalloc(BLOCK_SIZE))==NULL)return NULL;
			if(read_cache(din,dir[0],secters,next_dir)!=secters)return NULL;
		}

		if(path==top)break;

		/* ǥ쥯ȥ򸡺 */
		if((entry=search_same_entry(&path,dir))==NULL)return NULL;
		if(entry->type!=DIRECTORY)return NULL;
		dir_blk=entry->secter;
	}

	/* ȥ꡼ɲä롣 */
	if(search_prev_entry(path,dir,&dn,&ent)==-1)return NULL;
	s=strlen(path);
	if((entry=get_empty_entry(s,dir,dn,ent))==NULL)return NULL;
	memcpy(entry->name,path,s+1);

	return entry;
}


/************************************************************************************
 *
 * inode functions
 *
 ************************************************************************************/

static int indirect_size[3]={
	BLOCK_SIZE,
	BLOCK_SIZE*(BLOCK_SIZE/sizeof(uint)),
	BLOCK_SIZE*(BLOCK_SIZE/sizeof(uint))*(BLOCK_SIZE/sizeof(uint))
};


/*
 * inodeХåե˥֥åǡɤ߹ࡣ
 * parameters : device inode,žХåե,ž(byte),ϥեå,ܻȥΡ,
 *              inodeХåեݥ(ǥåɬ0)
 * return : ɤ߹ߥХȿ or error=-1
 */
int read_indirect_inode(int din,void *buf,size_t size,int begin,int node,uint *inode)
{
	void *tmp_buf=NULL;
	int secters=super_blk[din]->secters;
	uint j,s,last;
	int i;


	i=begin/indirect_size[node];
	j=begin;
	last=size+begin;

	if(node==0)
	{
		/* ǽΥ֥å̤ž */
		if(j%BLOCK_SIZE)
		{
			if((tmp_buf=kmalloc(BLOCK_SIZE))==NULL)return -1;
			if(read_cache(din,tmp_buf,secters,inode[i])!=secters)goto ERR;
			s=ROUNDUP(j,indirect_size[node]);
			s=(last<s)?last-j:s-j;
			if(read_cache(din,tmp_buf,secters,inode[i++])!=secters)goto ERR;
			memcpy(buf,(char*)tmp_buf+(j%BLOCK_SIZE),s);
			(char*)buf+=s;
			j+=s;
		}

		if(last%BLOCK_SIZE)
		{
			for(s=last-BLOCK_SIZE;j<s;begin+=BLOCK_SIZE)
			{
				if(read_cache(din,buf,secters,inode[i++])!=secters)goto ERR;
				(char*)buf+=BLOCK_SIZE;
			}

			/* Ĥ꤬֥å̤ž */
			if(tmp_buf==NULL)
				if((tmp_buf=kmalloc(BLOCK_SIZE))==NULL)return -1;
			if(read_cache(din,tmp_buf,secters,inode[i])!=secters)goto ERR;
			if(read_cache(din,tmp_buf,secters,inode[i])!=secters)goto ERR;
			memcpy(buf,tmp_buf,last-j);
		}
		else
			for(;j<last;j+=BLOCK_SIZE)
			{
				if(read_cache(din,buf,secters,inode[i++])!=secters)goto ERR;
				(char*)buf+=BLOCK_SIZE;
			}
	}
	else
	{
		if((tmp_buf=kmalloc(BLOCK_SIZE))==NULL)return -1;

		/* ǽΥ֥å̤ž */
		if(read_cache(din,tmp_buf,secters,inode[i++])!=secters)goto ERR;
		s=ROUNDUP(begin,indirect_size[node]);
		s=(last<s)?last-begin:s-begin;
		if(read_indirect_inode(din,buf,s,begin%indirect_size[node],node-1,(uint*)tmp_buf)==-1)goto ERR;
		(char*)buf+=indirect_size[node];
		j+=s;

		for(s=last-indirect_size[node];j<s;j+=indirect_size[node])
		{
			if(read_cache(din,tmp_buf,secters,inode[i++])!=secters)goto ERR;
			if(read_indirect_inode(din,buf,indirect_size[node],0,node-1,(uint*)tmp_buf)==-1)goto ERR;
			(char*)buf+=indirect_size[node];
		}

		/* ǸΥ֥å̤ž */
		if(read_cache(din,tmp_buf,secters,inode[i++])!=secters)goto ERR;
		if(read_indirect_inode(din,buf,last-j,0,node-1,(uint*)tmp_buf)==-1)goto ERR;
	}

	kfree(tmp_buf);

	return 0;

ERR:
	kfree(tmp_buf);

	return -1;
}


/*
 * ܻȤ񤭹ɲä롣
 * parameters : device inode,žХåե,ž(byte),ϥեå,ܻȥΡ,
 *              inodeХåեݥ(ǥåɬ0),ߤΥե륵
 * return : 0 or error=-1
 */
int write_indirect_inode(int din,void *buf,size_t size,int begin,int node,uint *inode,size_t fsize)
{
	void *tmp_buf=NULL;
	int secters=super_blk[din]->secters;
	uint j,s,last;
	int i;


	i=begin/indirect_size[node];
	j=begin;

	if(node==0)
	{
		/*
		 * ¸֥å˽񤭹ࡣ
		 */
		last=(size+begin<fsize)?size+begin:ROUNDUP(fsize,BLOCK_SIZE);

		/* ǽΥ֥å̤ž */
		if((j<last)&&(j%BLOCK_SIZE))
		{
			if((tmp_buf=kmalloc(BLOCK_SIZE))==NULL)return -1;
			if(read_cache(din,tmp_buf,secters,inode[i])!=secters)goto ERR;
			s=ROUNDUP(j,indirect_size[node]);
			s=(last<s)?last-j:s-j;
			memcpy((char*)tmp_buf+(j%BLOCK_SIZE),buf,s);
			if(write_cache(din,tmp_buf,secters,inode[i++])!=secters)goto ERR;
			(char*)buf+=s;
			j+=s;
		}

		if(last%BLOCK_SIZE)
		{
			for(s=last-BLOCK_SIZE;j<s;begin+=BLOCK_SIZE)
			{
				if(write_cache(din,buf,secters,inode[i++])!=secters)goto ERR;
				(char*)buf+=BLOCK_SIZE;
			}

			/* Ĥ꤬֥å̤ž */
			if(tmp_buf==NULL)
				if((tmp_buf=kmalloc(BLOCK_SIZE))==NULL)return -1;
			if(read_cache(din,tmp_buf,secters,inode[i])!=secters)goto ERR;
			memcpy(tmp_buf,buf,last-j);
			if(write_cache(din,tmp_buf,secters,inode[i])!=secters)goto ERR;
		}
		else
			for(;j<last;j+=BLOCK_SIZE)
			{
				if(write_cache(din,buf,secters,inode[i++])!=secters)goto ERR;
				(char*)buf+=BLOCK_SIZE;
			}

		/*
		 * ֥åɲäƽ񤭹ࡣ
		 */
		last=size+begin;

		if(last%BLOCK_SIZE)
		{
			for(;j<last;j+=BLOCK_SIZE)
			{
				if((inode[i]=get_empty_blk(din))==0)goto ERR;
				if(write_cache(din,buf,secters,inode[i++])!=secters)goto ERR;
				(char*)buf+=BLOCK_SIZE;
			}
		}
		else
		{
			for(s=last-BLOCK_SIZE;j<s;j+=BLOCK_SIZE)
			{
				if((inode[i]=get_empty_blk(din))==0)goto ERR;
				if(write_cache(din,buf,secters,inode[i++])!=secters)goto ERR;
				(char*)buf+=BLOCK_SIZE;
			}
			if(tmp_buf==NULL)
				if((tmp_buf=kmalloc(BLOCK_SIZE))==NULL)return -1;
			memcpy(tmp_buf,buf,last-j);
			if(write_cache(din,tmp_buf,secters,inode[i])!=secters)goto ERR;
		}
	}
	else
	{
		if((tmp_buf=kmalloc(BLOCK_SIZE))==NULL)return -1;

		/* ¸֥å˽񤭹ࡣ */
		last=(size+begin<fsize)?size+begin:ROUNDUP(fsize,indirect_size[node]);

		if(j<last)
		{
			if(read_cache(din,tmp_buf,secters,inode[i++])!=secters)goto ERR;
			s=ROUNDUP(j,indirect_size[node]);
			s=(last<s)?last-j:s-j;
			if(write_indirect_inode(din,buf,s,j%indirect_size[node],node-1,(uint*)tmp_buf,fsize)==-1)goto ERR;
			(char*)buf+=s;
			fsize-=s;
			j+=s;
		}
		for(;j<last;j+=indirect_size[node])
		{
			if(read_cache(din,tmp_buf,secters,inode[i++])!=secters)goto ERR;
			if(write_indirect_inode(din,buf,indirect_size[node],0,node-1,(uint*)tmp_buf,fsize)==-1)goto ERR;
			(char*)buf+=indirect_size[node];
			fsize-=indirect_size[node];
		}

		/* ֥åɲäƽ񤭹ࡣ */
		last=size+begin;
		for(;j<last;j+=indirect_size[node])
		{
			if((inode[i]=get_empty_blk(din))==0)goto ERR;
			memset(tmp_buf,0,BLOCK_SIZE);
			if(write_indirect_inode(din,buf,indirect_size[node],0,node-1,(uint*)tmp_buf,0)==-1)goto ERR;
			(char*)buf+=indirect_size[node];
			if(write_cache(din,tmp_buf,secters,inode[i++])!=secters)goto ERR;
		}
	}

	kfree(tmp_buf);

	return 0;

ERR:
	kfree(tmp_buf);

	return -1;
}


/*
 * ֥åХåե˿inode롣
 * parameters : buffer,secter,fiel type
 */
void make_inode(INODE *inode,uint secter,int type)
{
	memset(inode,0,BLOCK_SIZE);
	inode->my_secter=secter;
	inode->file_type=type;
}


/************************************************************************************
 *
 * Init functions
 *
 ************************************************************************************/

/*
 * ǥ˥ե륷ƥۤ롣
 * parameters : device path
 * return : 0 or error=-1
 */
int mk_orig_fs(const char *dev_path)
{
	void *dbuf;
	int fd;
	int din;
	SUPER_BLOCK *sbuf;
	ENPTY_BLOCK *ebuf=NULL;
	DEV_STAT dev_stat;
	int i,j,last,emp_num,index;


	/* Device open */
	if((fd=sys_open(dev_path,0))==-1)return -1;
	if((din=get_inode(fd))==-1)return -1;

	/* ѡƥ󥿥פγǧ */
	if(get_dev_stat(din,&dev_stat)==-1)return -1;
	if((dev_stat.pt_pres==1)&&(dev_stat.prt_type!=PT_TYPE_ORIG))return -1;

	/* Ǥ˥ե륷ƥबۤƤ뤫ǧ */
	if((sbuf=(SUPER_BLOCK*)kmalloc(ROUNDUP(sizeof(SUPER_BLOCK),dev_stat.sect_size)))==NULL)return -1;
	i=ROUNDUP(sizeof(SUPER_BLOCK),dev_stat.sect_size)/dev_stat.sect_size;
	if(read_direct(din,sbuf,i,SBLOCK_SECTER)!=i)goto ERR;
	if(sbuf->magic_number==SBLOCK_MAGIC)
	{
		kfree(sbuf);
		return sys_close(fd);
	}
	
	printk("Make orig filesystem on %s\n",dev_path);
	
	/* ѡ֥åꡣ */
	sbuf->magic_number=SBLOCK_MAGIC;
	sbuf->secters=BLOCK_SIZE/dev_stat.sect_size;
	sbuf->all_blk=dev_stat.all_sect/sbuf->secters;
	sbuf->useable_blk=(dev_stat.all_sect-SBLOCK_SECTER)/sbuf->secters-1;

	/* root directory  */
	sbuf->root_blk=SBLOCK_SECTER+sbuf->secters;
	if((dbuf=kmalloc(BLOCK_SIZE))==NULL)goto ERR;
	make_dir(dbuf,sbuf->root_blk);
	if(write_direct(din,dbuf,sbuf->secters,sbuf->root_blk)!=sbuf->secters)goto ERR;
	kfree(dbuf);

	/* ֥åǥåꡣ */
	if((ebuf=(ENPTY_BLOCK*)kmalloc(BLOCK_SIZE))==NULL)goto ERR;
	emp_num=(BLOCK_SIZE-sizeof(ENPTY_BLOCK))/sizeof(uint);
	last=dev_stat.all_sect-sbuf->secters;
	index=sbuf->root_blk+sbuf->secters;
	j=0;
	for(i=index;i<last;i+=sbuf->secters)
	{
		ebuf->empty[j++]=i;

		if(j>=emp_num)
		{
			ebuf->current=0;
			if(write_direct(din,ebuf,sbuf->secters,index)!=sbuf->secters)goto ERR;
			index=i;
			j=0;
			printk("Init empty index %d%%\r",i/(dev_stat.all_sect/100));	/* ʹԾɽ */
		}
	}
	ebuf->empty[j]=0;
	if(write_direct(din,ebuf,sbuf->secters,index)!=sbuf->secters)goto ERR;
	printk("Init empty index 100%%\n");										/* ʹԾɽ */
	kfree(ebuf);

	sbuf->empty_blk=sbuf->root_blk+sbuf->secters;

	/* ѡ֥å񤭹ࡣ */
	write_direct(din,sbuf,(sizeof(SUPER_BLOCK)+dev_stat.sect_size-1)/dev_stat.sect_size,SBLOCK_SECTER);
	kfree(sbuf);

	/* device close */
	return sys_close(fd);

ERR:
	kfree(ebuf);
	kfree(sbuf);
	sys_close(fd);

	return -1;
}

/*
 * 롼ȥե륷ƥν
 * parameters : device path
 * return : 0 or erroe=-1
 */
int init_orig_fs(const char *device)
{
	int i;


	for(i=0;i<MAX_DEVICE_OPEN;++i)
	{
		super_blk[i]=NULL;
		empty_blk[i]=NULL;
	}

	if(mk_orig_fs(device)==-1)return -1;

	return regist_fs(&orig_fs);
}


/************************************************************************************
 *
 * System call interface
 *
 ************************************************************************************/

static uint mount(int din)
{
	void *buf;
	int secters;			/* BLOCK_SIZE/secter */
	DEV_STAT dev_stat;


	/* Ǥ˥ޥȤƤ뤫ǧ */
	if(super_blk[din]!=NULL)return 0;

	/* Read super block */
	if(get_dev_stat(din,&dev_stat)==-1)return 0;
	secters=BLOCK_SIZE/dev_stat.sect_size;
	if((buf=kmalloc(BLOCK_SIZE))==NULL)return 0;
	if(read_direct(din,buf,secters,SBLOCK_SECTER)!=secters)return 0;
	if(((SUPER_BLOCK*)buf)->magic_number!=SBLOCK_MAGIC)return 0;
	super_blk[din]=(SUPER_BLOCK*)buf;

	/* Read empty block */
	if((buf=kmalloc(BLOCK_SIZE))==NULL)goto ERR;
	if(read_direct(din,buf,secters,super_blk[din]->empty_blk)!=secters)goto ERR;
	empty_blk[din]=(ENPTY_BLOCK*)buf;

	return super_blk[din]->root_blk;

ERR:
	kfree(super_blk[din]);
	super_blk[din]=NULL;
	kfree(empty_blk[din]);
	empty_blk[din]=NULL;

	return 0;
}


static int umount(int din)
{
	int secters=super_blk[din]->secters;


	/* Write empty block */
	if((super_blk[din]->empty_blk=empty_blk[din]->empty[empty_blk[din]->current])!=0)
		if(write_cache(din,empty_blk[din],secters,super_blk[din]->empty_blk)!=secters)return -1;
	kfree(empty_blk[din]);
	empty_blk[din]=NULL;

	/* Write super block */
	if(write_direct(din,super_blk[din],secters,SBLOCK_SECTER)!=secters)return -1;
	kfree(super_blk[din]);
	super_blk[din]=NULL;

	return 0;
}


static int open(const char *path,int din,uint dir_blk)
{
	int inode;
	DIR *dir[MAX_DIR_NUM+1];
	DIR_ENT *entry;
	int i;


	memset(dir,0,sizeof(DIR*)*(MAX_DIR_NUM+1));

	/* ѥ򸡺 */
	if((entry=search_dir_entry(path,din,dir_blk,dir))==NULL)inode=-1;
	else
	{
		if(entry->type==NORMAL_FILE)inode=entry->secter;
		else inode=-1;
	}

	/* ǥ쥯ȥХåե롣 */
	for(i=0;dir[i]!=NULL;++i)kfree(dir[i]);

	return inode;
}


static int close(int din,uint inode)
{
	return 0;
}


static int read(int din,uint ind,void *buf,size_t size,size_t begin)
{
	INODE *inode;	/* inodeХåե */


	if((inode=(INODE*)kmalloc(BLOCK_SIZE))==NULL)return -1;

	if(read_cache(din,inode,super_blk[din]->secters,ind)!=super_blk[din]->secters)goto ERR;

	/* γǧ */
	if(begin>inode->size)goto ERR;
	if((size+begin)>inode->size)size=inode->size-begin;

	/* եɤ߹ߡ */
	if(read_indirect_inode
		(din,buf,size,begin+sizeof(INODE)/sizeof(uint)*indirect_size[inode->indirect],inode->indirect,(uint*)inode)==-1)goto ERR;

	kfree(inode);

	return 0;

ERR:
	kfree(inode);

	return -1;
}


static int write(int din,uint ind,void *buf,size_t size,size_t begin)
{
	uint *_inode=NULL;
	int secters=super_blk[din]->secters;
	INODE *inode;
	int i;


	if(size==0)return 0;

	if((inode=(INODE*)kmalloc(BLOCK_SIZE))==NULL)return -1;
	if(read_cache(din,inode,secters,ind)!=secters)goto ERR;

	if(begin>inode->size)goto ERR;

	/* ɬפʤ鼡δܻȥΡɤꤹ롣 */
	i=inode->indirect;
	if((begin+size)>(indirect_size[i]*((BLOCK_SIZE-sizeof(INODE))/sizeof(uint))))
	{
		if((_inode=(uint*)kmalloc(BLOCK_SIZE))==NULL)goto ERR;
		do
		{
			memcpy(_inode,inode->secter,BLOCK_SIZE-sizeof(INODE));
			memset((char*)_inode+(BLOCK_SIZE-sizeof(INODE)),0,sizeof(INODE));
			if((inode->secter[0]=get_empty_blk(din))==-1)goto ERR;
			if(write_cache(din,_inode,secters,inode->secter[0])!=secters)goto ERR;
			memset(&inode->secter[1],0,BLOCK_SIZE-sizeof(INODE)-sizeof(uint));
		}while((begin+size)>(indirect_size[++i]*((BLOCK_SIZE-sizeof(INODE))/sizeof(uint))));
		inode->indirect=i;
	}

	/* 񤭹ߡ */
	if(write_indirect_inode(din,buf,size,begin+sizeof(INODE)/sizeof(uint)*indirect_size[i],i,(uint*)inode,inode->size)==-1)
		goto ERR;
	if(begin+size>inode->size)inode->size=begin+size;
	if(write_cache(din,inode,secters,ind)!=secters)goto ERR;

	kfree(_inode);
	kfree(inode);

	return 0;
ERR:
	kfree(_inode);
	kfree(inode);

	return -1;
}


static int mkdir(const char *path,int din,uint dir_blk)
{
	uint secter;
	int secters=super_blk[din]->secters;
	DIR *dir[MAX_DIR_NUM+1];
	DIR *newdir;
	DIR_ENT *entry;
	int i;


	memset(dir,0,(MAX_DIR_NUM+1)*sizeof(DIR*));

	/* ǥ쥯ȥꥨȥ꡼ */
	if((entry=get_new_dir_entry(path,din,dir_blk,dir))==NULL)goto ERR1;

	/* ǥ쥯ȥꥨȥ꡼롣 */
	if((secter=get_empty_blk(din))==0)goto ERR1;
	if((newdir=(DIR*)kmalloc(BLOCK_SIZE))==NULL)goto ERR2;
	make_dir(newdir,secter);
	if(write_cache(din,newdir,secters,secter)!=secters)
	{
		kfree(newdir);
		goto ERR2;
	}
	kfree(newdir);
	entry->secter=secter;
	entry->type=(uchar)DIRECTORY;

	/* ߤΥǥ쥯ȥ᤹ */
	for(i=0;dir[i]!=NULL;++i)
		if(write_cache(din,dir[i],secters,dir[i]->my_secter)!=secters)goto ERR2;

	for(i=0;dir[i]!=NULL;++i)kfree(dir[i]);

	return 0;

ERR2:
	release_blk(din,secter);
ERR1:
	for(i=0;dir[i]!=NULL;++i)kfree(dir[i]);

	return -1;
}


static int creat(const char *path,int din,uint dir_blk)
{
	int secter;
	int secters=super_blk[din]->secters;
	DIR *dir[MAX_DIR_NUM+1];
	INODE *newinode;
	DIR_ENT *entry;
	int i;


	memset(dir,0,(MAX_DIR_NUM+1)*sizeof(DIR*));

	/* ǥ쥯ȥꥨȥ꡼ */
	if((entry=get_new_dir_entry(path,din,dir_blk,dir))==NULL)goto ERR1;

	/* ǥ쥯ȥꥨȥ꡼롣 */
	if((secter=get_empty_blk(din))==0)goto ERR1;
	if((newinode=(INODE*)kmalloc(BLOCK_SIZE))==NULL)goto ERR2;
	make_inode(newinode,secter,NORMAL_FILE);
	if(write_cache(din,newinode,secters,secter)!=secters)
	{
		kfree(newinode);
		goto ERR2;
	}
	kfree(newinode);
	entry->secter=secter;
	entry->type=(uchar)NORMAL_FILE;

	/* ߤΥǥ쥯ȥ᤹ */
	for(i=0;dir[i]!=NULL;++i)
		if(write_cache(din,dir[i],secters,dir[i]->my_secter)!=secters)goto ERR2;

	for(i=0;dir[i]!=NULL;++i)kfree(dir[i]);

	return 0;

ERR2:
	release_blk(din,secter);
ERR1:
	for(i=0;dir[i]!=NULL;++i)kfree(dir[i]);

	return -1;
}


static int opendir(const char *path,int din,uint dir_blk)
{
	int newdir;
	DIR *dir[MAX_DIR_NUM+1];
	DIR_ENT *entry;
	int i;


	/* ǥ쥯ȥХåեν */
	memset(dir,0,sizeof(DIR*)*(MAX_DIR_NUM+1));

	/* ѥ򸡺 */
	if((entry=search_dir_entry(path,din,dir_blk,dir))==NULL)newdir=-1;
	else
	{
		if(entry->type==DIRECTORY)newdir=entry->secter;
		else newdir=-1;
	}

	/* ǥ쥯ȥХåե롣 */
	for(i=0;dir[i]!=NULL;++i)kfree(dir[i]);

	return newdir;
}


/*********************************** ̤ ***************************************/
static int ioctr(int din,uint inode,int cmd,void *param)
{
	return 0;
}


static int rename(int din,uint inode,const char *name)
{
	return 0;
}


/************************************** ƥ *******************************************/
void test_orig_fs()
{
	enum{
		BEG=0,
		SIZE=256,
		END=SIZE/4-1,
	};
	int fd,din,inode;
	int *buf1=(int*)0x90000;
	int *buf2=(int*)0x80000;
	DIR *dir[MAX_DIR_NUM+1];


	memset(dir,0,(MAX_DIR_NUM+1)*sizeof(DIR*));

	if(sys_mount("/dev/hdb7","orig_fs","/")==-1)
	{
		printk("Failed sys_mount\n");
		return;
	}

	if((fd=sys_open("/dev/hdb7",0))==-1)return;
	if((din=get_inode(fd))==-1)return;

/*	printk("creat()=%d\n",creat("text",din,super_blk[din]->root_blk));*/

	printk("open()=%d\n",inode=open("text",din,super_blk[din]->root_blk));

	buf1[BEG]=0xafafafaf;
	buf1[END]=0xbdbdbdbd;

/*	printk("write()=%d\n",write(din,inode,buf1,SIZE,0));*/
	printk("read()=%d\n",read(din,inode,buf2,SIZE,0));
	printk("buf2[BEG]=%x,buf2[END]=%x\n",buf2[BEG],buf2[END]);

	printk("umount()=%d\n",umount(din));
}
