/*
 * device.c
 *
 * Copyright 2002, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 */


#include"config.h"
#include"types.h"
#include"lib.h"
#include"mm.h"
#include"interrupt.h"
#include"proc.h"
#include"fs.h"
#include"errno.h"
#include"lock.h"
#include"device.h"


enum{
	MAX_NAME=16,			/* Device name size */
	HASH_NUM=CACHE_NUM,		/* Number of cache hash table */
	SECTER_SIZE=512
};


/* Partition table */
typedef struct{
	uchar active;			/* Boot active flag=0x80 */
	uchar chs_begin[3];		/* CHS begin block */
	uchar type;				/* Partition type */
	uchar chs_last[3];		/* CHS last block */
	uint lba_begin;			/* LBA begin block */
	uint size;				/* Partition size by secter */
}PARTITON_TBL;

typedef struct CACHE_TABLE{
	struct CACHE_TABLE *hnext;		/* Hash next */
	struct CACHE_TABLE *hprev;		/* Hash previous */
	struct CACHE_TABLE *next;		/* Cache chain next */
	struct CACHE_TABLE *prev;		/* Cache chain previous */
	struct CACHE_TABLE *write_next;	/* ǥХؽ񤭹߹ꥹ */
	struct CACHE_TABLE *write_prev; /* ǥХؽ񤭹߹ꥹ */
	void *addr;						/* Cache address */
	uint block;						/* Device block index */
	uchar din;						/* Device inode number */
	uchar ref_count;				/* ȥ */
	uchar update;					/* ե饰 */
	uchar writing;					/* ǥХؽ񤭹߹ե饰 */
	int gate;						/* ԥå */
}CACHE_TABLE;


static DEV_INFO *device[MAX_DEVICES];
DEV_INODE dinode[MAX_DEVICE_OPEN];


static uint open(const char*,void*,int);
static int read(uint,void*,size_t,size_t);
static int write(uint,void*,size_t,size_t);
static int ioctl(uint,int,void*);
static int close(uint);

/* Cache */
static CACHE_TABLE cache_tbl[CACHE_NUM];			/* Cache table */
static CACHE_TABLE *cache_hash[HASH_NUM];			/* Hash table for searching cache table */
static CACHE_TABLE *cache_tbl_old=&cache_tbl[0];	/* Cache table old point */
static int cache_gate=0;							/* åơ֥ѥԥåե饰 */
static int cache_refcnt=0;							/* åơ֥뻲ȥ */
static int cache_update=0;							/* åơ֥빹ե饰 */

static PROC *write_task=NULL;						/* ٱ񤭹ѥ */
static CACHE_TABLE delay_list=						/* ٱ񤭹ߥꥹȤƬ */
	{NULL,NULL,NULL,NULL,&delay_list,&delay_list,NULL,0,0,0,0,0,0};


static int init_cache();
static void start_delay_write_task();
static void add_delay_list(CACHE_TABLE*);
static void delay_write();


static FS dev_fs={
	"dev_fs",NULL,NULL,open,read,write,ioctl,NULL,NULL,NULL,NULL,close
};


 /*
  * ǥХѥ̾Ӥ롣
  * parameters : destination path,sorce path
  * return :  פΥݥ,԰ NULL
  */
extern inline const char *cmp_devpath(const char *s1,const char *s2)
{
	while(*s1!='\0')
		if(*s1++!=*s2++)return NULL;

	return s2;
}


/************************************************************************************************
 *
 * ؿ
 *
 ************************************************************************************************/


/*
 * Init device system
 * return : 0 or error=-1
 */
int init_device()
{
	int i;


	for(i=0;i<MAX_DEVICES;++i)device[i]=NULL;
	for(i=0;i<MAX_DEVICES;++i)dinode[i].ref_count=0;

	/* IOå */
	init_cache();
	
	/* ٱ񤭹ѥư롣 */
	start_delay_write_task();

	return regist_fs(&dev_fs);
}


/*
 * Register device
 * parameters : Device infomation struct address
 * return : 0 or Error number
 */
int regist_device(DEV_INFO *info)
{
	int i;


	for(i=0;i<MAX_DEVICES;++i)
		if(device[i]==NULL)
		{
			device[i]=info;
			return 0;
		}

	return -EMDEV;
}


/*
 * Delete from regist table
 * parameters : name
 * return : 0,error -1
 */
int delete_device(const char *name)
{
	return 0;
}


 /***********************************************************************************
 *
 * ¾δؿ
 *
 ***********************************************************************************/

/*
 * ǥХΥ롣
 * parameters : device inode number
 * return : secter size(byte)
 */
int get_secter_size(int din)
{
	return dinode[din].dev->secter_size;
}


/************************************************************************************************
 *
 * ľžؿ
 *
 ************************************************************************************************/

/*
 * parameters : device inode number,buffer address,transfer secters,begin secter
 * return : transfer secters or error=-1
 */
int read_direct(int din,void *buf,size_t secters,size_t begin)
{
	/* ǥ¤Υå */
	if(begin>dinode[din].blocks)return -1;
	if((secters>dinode[din].blocks)||(begin+secters>dinode[din].blocks))
		secters=dinode[din].blocks-begin;

	return dinode[din].dev->read(buf,secters,begin+dinode[din].begin_blk);
}


/*
 * parameters : device inode number,buffer address,transfer secters,begin secter
 * return : transfer secters or error=-1
 */
int write_direct(int din,void *buf,size_t secters,size_t begin)
{
	/* ǥ¤Υå */
	if(begin>dinode[din].blocks)return -1;
	if((secters>dinode[din].blocks)||(begin+secters>dinode[din].blocks))
		secters=dinode[din].blocks-begin;

	return dinode[din].dev->write(buf,secters,begin+dinode[din].begin_blk);
}


/***********************************************************************************
 *
 * IOåؿ
 *  å֥åñ̤ɤ߽񤭤롣
 *
 ***********************************************************************************/

/*
 * ϥåͤ׻롣
 * return : hash value
 */
extern inline int calc_hash(int value)
{
	return (value/8)%HASH_NUM;
}


extern inline int cache_rounddown(int value)
{
	return ~(CACHE_BLOCK_SIZE-1)/value;
}


/*
 * ɤ߼궦ͭåûξѡ
 * parameters : ɤ߼껲ȥ,ե饰ݥ
 */
extern inline void enter_read_lock(uchar *refcnt,uchar *update,int *gate)
{
	enter_spinlock(gate);
	{
		while(*update==1);
		++*refcnt;
	}
	exit_spinlock(gate);
}


/*
 * ɤ߼궦ͭåФ롣
 * parameters : ɤ߼껲ȥ
 */
extern inline void exit_read_lock(uchar *refcnt)
{
	--*refcnt;
}

extern inline void set_update(uchar *refcnt,uchar *update,int *gate)
{
	/* ʣãƤɤ褦ˡȥȤö餹 */
	--*refcnt;
	enter_spinlock(gate);
	{
		while(*refcnt!=0);
		*update=1;
		++*refcnt;
	}
	exit_spinlock(gate);
}

extern inline void reset_update(uchar *update)
{
	*update=0;
}


/*
 * Init cache
 * return : 0 or Error number
 */
int init_cache()
{
	int i;


	/* Init cache address */
	for(i=0;i<CACHE_NUM;++i)
	{
		memset(&cache_tbl[i],0,sizeof(CACHE_TABLE));
		if((cache_tbl[i].addr=kmalloc(CACHE_BLOCK_SIZE))==NULL)return -ENOMEM;
		cache_tbl[i].next=&cache_tbl[i+1];
		cache_tbl[i].prev=&cache_tbl[i-1];
		cache_tbl[i].hprev=&cache_tbl[i];
	}
	cache_tbl[CACHE_NUM-1].next=&cache_tbl[0];
	cache_tbl[0].prev=&cache_tbl[CACHE_NUM-1];

	/* Init hash table */
	for(i=0;i<HASH_NUM;++i)cache_hash[i]=NULL;

	return 0;
}


/*
 * Read from cache
 * parameters : device inode number,buffer address,transfer secters,begin secter
 * return : transfer secters or error=-1
 */
int read_cache(int din,void *buf,size_t secters,size_t begin)
{
	int hash;
	int mask;
	int secter_size;
	int cashe_sect;
	uint all;
	CACHE_TABLE *p;
	int num;


	/* ǥ¤Υå */
	if(begin>dinode[din].blocks)return -1;
	if((secters>dinode[din].blocks)||(begin+secters>dinode[din].blocks))
		secters=dinode[din].blocks-begin;

	secter_size=dinode[din].dev->secter_size;
	mask=cache_rounddown(secter_size);
	all=secters+begin;
	cashe_sect=~mask+1;
	hash=calc_hash(begin&mask);
	for(num=(begin&mask)+cashe_sect-begin;all!=begin;begin+=num,num=cashe_sect,hash=calc_hash(begin))
	{
		if(all-begin<cashe_sect)num=all-begin;

		enter_read_lock((uchar*)&cache_refcnt,(uchar*)&cache_update,&cache_gate);

		/* å򸡺롣 */
		for(p=cache_hash[hash];;p=p->hnext)
		{
			/* åˤʤ */
			if(p==NULL)
			{
				set_update((uchar*)&cache_refcnt,(uchar*)&cache_update,&cache_gate);

				/* ָŤå򲣼 */
				p=cache_tbl_old;
				cache_tbl_old=p->next;

				/* Delete old hash link */
				p->hprev->hnext=p->hnext;
				if(p->hnext!=NULL)p->hnext->hprev=p->hprev;

				/* Add new hash link */
				p->hnext=cache_hash[hash];
				p->hprev=(CACHE_TABLE*)&cache_hash[hash];
				if(cache_hash[hash]!=NULL)cache_hash[hash]->hprev=p;
				cache_hash[hash]=p;

				p->din=din;
				p->block=begin&mask;

				/* updateå */
				enter_spinlock(&p->gate);
				{
					/* ٱ񤭹Ԥ */
					while(p->writing!=0);

					reset_update((uchar*)&cache_update);
					--cache_refcnt;
					while(cache_refcnt!=0);
					p->update=1;
					++cache_refcnt;
				}
				exit_spinlock(&p->gate);

				/* ǥХ֥åɤ߼롣*/
				if(dinode[din].dev->read(p->addr,num,begin+dinode[din].begin_blk)<num)
					{exit_read_lock((uchar*)&cache_refcnt);return -1;}

				reset_update(&p->update);

				break;
			}

			/* åˤ */
			if((p->block==(begin&mask))&&(p->din==din))break;
		}

		exit_read_lock((uchar*)&cache_refcnt);

		/* Хåեž롣 */
		enter_read_lock(&p->ref_count,&p->update,&p->gate);
		{
			memcpy(buf,p->addr,num*secter_size);
			(uint)buf+=num*secter_size;
		}
		exit_read_lock(&p->ref_count);
	}

	return secters;
}


/*
 * Write to cache
 * parameters : device inode number,buffer address,transfer secters,begin secter
 * return : transfer secters or error=-1
 */
int write_cache(int din,void *buf,size_t secters,size_t begin)
{
	int hash;
	int mask;
	int secter_size;
	int cashe_sect;
	uint all;
	CACHE_TABLE *p;
	int num;


	/* ǥ¤Υå */
	if(begin>dinode[din].blocks)return -1;
	if((secters>dinode[din].blocks)||(begin+secters>dinode[din].blocks))
		secters=dinode[din].blocks-begin;

	secter_size=dinode[din].dev->secter_size;
	mask=cache_rounddown(secter_size);
	all=secters+begin;
	cashe_sect=~mask+1;
	hash=calc_hash(begin&mask);
	for(num=(begin&mask)+cashe_sect-begin;all!=begin;begin+=num,num=cashe_sect,hash=calc_hash(begin))
	{
		if(all-begin<cashe_sect)num=all-begin;

		enter_read_lock((uchar*)&cache_refcnt,(uchar*)&cache_update,&cache_gate);

		/* Search in hash table */
		for(p=cache_hash[hash];;p=p->hnext)
		{
			/* åˤʤ */
			if(p==NULL)
			{
				exit_read_lock((uchar*)&cache_refcnt);

				/* ǥľܽ񤭹ࡣ */
				dinode[din].dev->write(buf,secters,begin+dinode[din].begin_blk);
				(uint)buf+=num*secter_size;

				break;
			}

			/* åˤ */
			if((p->block==(begin&mask))&&(p->din==din))
			{
				exit_read_lock((uchar*)&cache_refcnt);

				/* å˽񤭹ࡣ */
				enter_read_lock(&p->ref_count,&p->update,&p->gate);
				{
					memcpy(p->addr,buf,num*secter_size);
					(uint)buf+=num*secter_size;
				}
				exit_read_lock(&p->ref_count);
				
				/* ٱ񤭹ߡ */
				add_delay_list(p);

				break;
			}
		}
	}

	return secters;
}


/************************************************************************************************
 *
 * ǥХٱ񤭹
 *
 ************************************************************************************************/

/*
 * ٱ񤭹ߥεư
 */
void start_delay_write_task()
{
	int proc;


	switch(proc=sys_fork())
    {
		case -1:
			printk("Faild start_delay_write_task\n");
			break;
		case 0:
			delay_write();
		default:	
			write_task=(PROC*)proc;
	}
}


/*
 * ٱ񤭹ߥꥹȤ˲ä롣
 * parameters : device inode number,񤭹ߥ֥å,֥åλϤޤ
 */
void add_delay_list(CACHE_TABLE *p)
{
	enter_spinlock(&p->gate);
	{
		p->writing=1;
	}
	exit_spinlock(&p->gate);
	
	enter_spinlock(&delay_list.gate);
	{
		p->write_next=delay_list.write_next;
		p->write_prev=&delay_list;
		p->write_next->write_prev=p;
		delay_list.write_next=p;
	}
	exit_spinlock(&delay_list.gate);

	/* ٱ񤭹ߥ򵯤 */
	if(p->write_next==&delay_list)add_to_schedule(write_task);
}


/*
 * ǥХؤٱ񤭹ߥѴؿ
 */
void delay_write()
{
	CACHE_TABLE *p;


	while(write_task==NULL);
	del_from_schedule(write_task);
	wait_task();
	
	for(;;)
	{
		enter_spinlock(&delay_list.gate);

		if(delay_list.write_next!=&delay_list)
		{
			p=delay_list.write_prev;
			p->write_prev->write_next=&delay_list;
			delay_list.write_prev=p->write_prev;
			
			exit_spinlock(&delay_list.gate);

			/* ǥХ˽񤭹ࡣ */
			dinode[p->din].dev->write(
				p->addr,CACHE_BLOCK_SIZE/dinode[p->din].dev->secter_size,p->block+dinode[p->din].begin_blk);
			
			p->writing=0;
		}
		else
		{
			exit_spinlock(&delay_list.gate);

			/* ꡼פ롣 */
			del_from_schedule(write_task);
			wait_task();
		}
	}
}


/************************************************************************************************
 *
 * System call interface
 *
 ************************************************************************************************/

/*
 * Open device
 * parameters : Device name,inode,DEV_INFO struct(for copy)
 * return : device inode number or error=MAX_DEVICES
 */
uint open(const char *dev_name,void *dir,int dev)
{
	enum{
		/* Partition table */
		PT_NUM=4,				/* 1Υѡƥơ֥ */
		PT_BEGIN=0x1be,			/* ѡƥơ֥볫ϥեå */
		PT_MAGIC_ADDR=0x1fe,	/* ѡƥơ֥ޥåʥСեå */
		PT_MAGIC_NUMBER=0xaa55,	/* ѡƥơ֥ޥåʥС */

		/* Partition type */
		PT_TYPE_NULL=0,			/* NULL partition */
		PT_TYPE_EXT=0x5,		/* ext partition */
		PT_TYPE_LEXT=0xf,		/* ext(LBA) partition */
	};

	const char *str;
	char *buf;
	int inode_num;
	int pt_num;
	uint beg_blk,next_blk;
	PARTITON_TBL *pt;
	int i;


	/* ϿǥХ̾Ǹ */
	for(i=0;(str=cmp_devpath(device[i]->name,dev_name))==NULL;++i)
		if(i>=MAX_DEVICES)return MAX_DEVICES;
	dev=i;

	/* ̾˥ѡƥֹ椬СѴ */
	pt_num=0;
	if(*str!='\0')
		if((pt_num=atoi(str))<=0)return MAX_DEVICES;		/* ѡƥֹ1ʾ */

	/* Ǥ˥ץ󤵤Ƥ뤫ǧ */
	for(i=0;i<MAX_DEVICE_OPEN;++i)
		if((dinode[i].dev==device[dev])&&(dinode[i].pt_num==pt_num))
		{
			++dinode[i].ref_count;
			return MAX_DEVICES;
		}

	/* ǥХinode򤹤롣 */
	for(i=0;dinode[i].ref_count!=0;++i)
		if(i>=MAX_DEVICE_OPEN)return MAX_DEVICES;
	inode_num=i;

	/* Open device */
	if(device[dev]->open()<0)return MAX_DEVICES;

	/* ѡƥơ֥򸡺 */
	if((buf=(char*)kmalloc(device[dev]->secter_size))==NULL)return MAX_DEVICES;
	if(device[dev]->read(buf,1,0)!=1)goto ERR;
	if(pt_num==0)
	{
		if(*(ushort*)(buf+PT_MAGIC_ADDR)!=PT_MAGIC_NUMBER)dinode[inode_num].pt_prn=1;
		else dinode[inode_num].pt_prn=0;

		dinode[inode_num].pt_prn=0;
		dinode[inode_num].pt_num=0;
		dinode[inode_num].pt_type=0;
		dinode[inode_num].begin_blk=device[dev]->begin_blk;
		dinode[inode_num].blocks=device[dev]->last_blk-device[dev]->begin_blk+1;
	}
	else
	{
		if(*(ushort*)(buf+PT_MAGIC_ADDR)!=PT_MAGIC_NUMBER)goto ERR;

		pt=(PARTITON_TBL*)(buf+PT_BEGIN);
		if(pt_num<=PT_NUM)
		{
			dinode[inode_num].pt_num=pt_num;
			--pt_num;
			dinode[inode_num].pt_type=pt[pt_num].type;
			dinode[inode_num].begin_blk=pt[pt_num].lba_begin;
			dinode[inode_num].blocks=pt[pt_num].size;
		}
		else
		{
			for(i=0;pt[i].type!=PT_TYPE_EXT;++i)
				if(i>=PT_NUM)goto ERR;			/* ĥѡƥ󤬤ʤ */

			/* ĥѡƥ󥯤򤿤ɤ롣 */
			beg_blk=next_blk=pt[i].lba_begin;
			for(i=PT_NUM;;)
			{
				if(device[dev]->read(buf,1,next_blk)!=1)goto ERR;
				if(++i==pt_num)break;
				if(pt[1].type!=PT_TYPE_EXT)goto ERR;
				next_blk=beg_blk+pt[1].lba_begin;
			}

			dinode[inode_num].pt_num=pt_num;
			dinode[inode_num].pt_type=pt[0].type;
			dinode[inode_num].begin_blk=pt[0].lba_begin+next_blk;
			dinode[inode_num].blocks=pt[0].size;
		}

		dinode[inode_num].pt_prn=1;
	}

	kfree(buf);

	dinode[inode_num].dev=device[dev];
	dinode[inode_num].ref_count=1;

	return inode_num;

ERR:
	kfree(buf);

	return MAX_DEVICES;
}


/*
 * parameters : Device inode number,Buffer,Transfer size(byte),Begin bytes
 */
int read(uint inode,void *buf,size_t size,size_t begin)
{
	return 0;

}

/*
 * parameters : Device regist number,Buffer,Transfer size(byte),Begin block number
 */
int write(uint inode,void *buf,size_t size,size_t begin)
{
	return 0;
}

/*
 * parameters : Device regist number,Command,Parameter struct pointer
 */
int ioctl(uint inode,int command,void *param)
{
	return 0;
}

/*
 * parameters : device inode index
 */
int close(uint din)
{
	return 0;
}
/****************************************************************************/
void test_device()
{
	uint *buf1=(uint*)0x70000;
	uint *buf2=(uint*)0x80000;
	int i;


	printk("open=%d\n",i=(int)open("hdb1",NULL,0));
	printk("dinode.pt_num=%d\n",dinode[i].pt_num);
	printk("dinode.pt_prn=%d\n",dinode[i].pt_prn);
	printk("dinode.pt_type=%x\n",dinode[i].pt_type);
	printk("dinode.ref_count=%d\n",dinode[i].ref_count);
	printk("dinode.dev=%x\n",dinode[i].dev);
	printk("dinode.begin_blk=%x\n",dinode[i].begin_blk);
	printk("dinode.blocks=%x\n",dinode[i].blocks);

	printk("read blocks=%d\n",read_cache(i,buf1,16,8003));
	printk("buf1[0]=%x,buf1[0x400-1]=%x\n",buf1[0],buf1[0x800-1]);
	buf2[0]=0xa5a5a5a5,buf2[0x800-1]=0x5a5a5a5a;
	write_cache(i,buf2,16,8003);
	
	printk("read blocks=%d\n",read_cache(i,buf1,16,6003));
	printk("buf1[0]=%x,buf1[0x400-1]=%x\n",buf1[0],buf1[0x800-1]);
	buf2[0]=0xa5a5a5a5,buf2[0x800-1]=0x5a5a5a5a;
	write_cache(i,buf2,16,6003);
	
	mili_timer(2000);
	
	printk("read blocks=%d\n",read_direct(i,buf1,16,8003));
	printk("buf1[0]=%x,buf1[0x400-1]=%x\n",buf1[0],buf1[0x800-1]);
	printk("read blocks=%d\n",read_direct(i,buf1,16,6003));
	printk("buf1[0]=%x,buf1[0x400-1]=%x\n",buf1[0],buf1[0x800-1]);
}
/*****************************************************************************/
