/* readelf.c -- display contents of an ELF format file
   Copyright 1998, 1999, 2000, 2001, 2002, 2003, 2004, 2005
   Free Software Foundation, Inc.
   Originally developed by Eric Youngdale <eric@andante.jic.com>
   Modifications by Nick Clifton <nickc@redhat.com>
   This file is part of GNU Binutils.
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA.  */


#define RELOC_MACROS_GEN_FUNC
#define BFD64

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <assert.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include "bfd.h"
#include "elf/common.h"
#include "elf/external.h"
#include "elf/internal.h"
#include "elf/arm.h"
#include "elf/i386.h"
#include "elf/ia64.h"
#include "elf/m68k.h"
#include "elf/m68hc11.h"
#include "elf/mips.h"
#include "elf/ppc.h"
#include "elf/ppc64.h"
#include "elf/sh.h"
#include "elf/sparc.h"
#include "elf/x86-64.h"


#define _(String) (String)
#define UNKNOWN -1
#define SECTION_NAME(X)	((X) == NULL ? "<none>" : 								\
		 				((X)->sh_name >= string_table_length ? "<corrupt>" :	\
		 				string_table + (X)->sh_name))
/* Given st_shndx I, map to section_headers index.  */
#define SECTION_HEADER_INDEX(I)	((I) < SHN_LORESERVE ? (I) : 					\
								((I) <= SHN_HIRESERVE ? 0 :						\
								(I) - (SHN_HIRESERVE + 1 - SHN_LORESERVE)))
/* Reverse of the above.  */
#define SECTION_HEADER_NUM(N)	((N) < SHN_LORESERVE ?								\
								(N) : (N) + (SHN_HIRESERVE + 1 - SHN_LORESERVE))
#define SECTION_HEADER(I)	(section_headers + SECTION_HEADER_INDEX (I))
#define BYTE_GET(field)		byte_get(field, sizeof (field))
#define NUM_ELEM(array)		(sizeof (array) / sizeof ((array)[0]))
/* This is just a bit of syntatic sugar.  */
#define streq(a,b)	(strcmp ((a), (b)) == 0)


char *program_name = "readelf";
long archive_file_offset;
unsigned long archive_file_size;
char *string_table;
unsigned long string_table_length;
Elf_Internal_Ehdr elf_header;
Elf_Internal_Shdr *section_headers;
Elf_Internal_Phdr *program_headers;
Elf_Internal_Shdr *symtab_shndx_hdr;
int show_name;
int do_syms;
int do_reloc;
int is_32bit_elf;
bfd_vma eh_addr_size;

/* How to rpint a vma value.  */
typedef enum print_mode {
  HEX,
  DEC,
  DEC_5,
  UNSIGNED,
  PREFIX_HEX,
  FULL_HEX,
  LONG_HEX
} print_mode;


static bfd_vma (*byte_get)(unsigned char *, int);
static void (*byte_put)(unsigned char *, bfd_vma, int);


static void error(const char *message, ...)
{
	va_list args;

	va_start (args, message);
	fprintf(stderr, "%s: Error: ", program_name);
	vfprintf(stderr, message, args);
	va_end (args);
}

static void warn(const char *message, ...)
{
	va_list args;

	va_start (args, message);
	fprintf(stderr, "%s: Warning: ", program_name);
	vfprintf(stderr, message, args);
	va_end (args);
}

static void *get_data(void *var, FILE *file, long offset, size_t size, const char *reason)
{
	void *mvar;

	if (size == 0) {
		return NULL;
	}

	if (fseek (file, archive_file_offset + offset, SEEK_SET)) {
		error("Unable to seek to 0x%x for %s\n", archive_file_offset + offset, reason);
		return NULL;
	}

	mvar = var;
	if (mvar == NULL) {
		mvar = malloc (size);

		if (mvar == NULL) {
			error("Out of memory allocating 0x%x bytes for %s\n", size, reason);
			return NULL;
		}
	}

	if (fread (mvar, size, 1, file) != 1) {
		error("Unable to read in 0x%x bytes of %s\n", size, reason);
		if (mvar != var) {
			free(mvar);
		}
		return NULL;
	}

	return mvar;
}

static bfd_vma byte_get_little_endian(unsigned char *field, int size)
{
	switch (size) {
	case 1:
		return *field;
	case 2:
		return ((unsigned int) (field[0])) | (((unsigned int) (field[1])) << 8);
	case 4:
		return  ((unsigned long) (field[0]))
			| (((unsigned long) (field[1])) << 8)
			| (((unsigned long) (field[2])) << 16)
			| (((unsigned long) (field[3])) << 24);
	case 8:
		return  ((bfd_vma) (field[0]))
			| (((bfd_vma) (field[1])) << 8)
			| (((bfd_vma) (field[2])) << 16)
			| (((bfd_vma) (field[3])) << 24)
			| (((bfd_vma) (field[4])) << 32)
			| (((bfd_vma) (field[5])) << 40)
			| (((bfd_vma) (field[6])) << 48)
			| (((bfd_vma) (field[7])) << 56);
	default:
		error("Unhandled data length: %d\n", size);
		abort();
		return 0;
	}
}

static void byte_put_little_endian(unsigned char *field, bfd_vma value, int size)
{
	switch (size) {
	case 8:
		field[7] = (((value >> 24) >> 24) >> 8) & 0xff;
		field[6] = ((value >> 24) >> 24) & 0xff;
		field[5] = ((value >> 24) >> 16) & 0xff;
		field[4] = ((value >> 24) >> 8) & 0xff;
		/* Fall through.  */
	case 4:
		field[3] = (value >> 24) & 0xff;
		field[2] = (value >> 16) & 0xff;
		/* Fall through.  */
	case 2:
		field[1] = (value >> 8) & 0xff;
		/* Fall through.  */
	case 1:
		field[0] = value & 0xff;
		break;
	default:
		error("Unhandled data length: %d\n", size);
		abort();
	}
}

static int print_dec_vma(bfd_vma vma, int is_signed)
{
	char buf[40];
	char *bufp = buf;
	int nc = 0;

	if (is_signed && (bfd_signed_vma) vma < 0) {
		vma = -vma;
		putchar('-');
		nc = 1;
	}

	do {
		*bufp++ = '0' + vma % 10;
		vma /= 10;
	}
	while (vma != 0);
	nc += bufp - buf;
	while (bufp > buf) {
		putchar(*--bufp);
	}

	return nc;
}

static int print_hex_vma(bfd_vma vma)
{
	char buf[32];
	char *bufp = buf;
	int nc;

	do {
		char digit = '0' + (vma & 0x0f);
		if (digit > '9')
			digit += 'a' - '0' - 10;
		*bufp++ = digit;
		vma >>= 4;
	}
	while (vma != 0);
	nc = bufp - buf;

	while (bufp > buf)
		putchar(*--bufp);
	return nc;
}

/* Print a VMA value.  */
static int print_vma(bfd_vma vma, print_mode mode)
{
	if (is_32bit_elf) {
		switch (mode) {
		case FULL_HEX:
			return printf("0x%8.8lx", (unsigned long) vma);
		case LONG_HEX:
			return printf("%8.8lx", (unsigned long) vma);
		case DEC_5:
			if (vma <= 99999)
			return printf("%5ld", (long) vma);
			/* Drop through.  */
		case PREFIX_HEX:
			return printf("0x%lx", (unsigned long) vma);
		case HEX:
			return printf("%lx", (unsigned long) vma);
		case DEC:
			return printf("%ld", (unsigned long) vma);
		case UNSIGNED:
			return printf("%lu", (unsigned long) vma);
		}
	}
	else {
		int nc = 0;

		switch (mode) {
		case FULL_HEX:
			nc = printf("0x");
			/* Drop through.  */
		case LONG_HEX:
			printf_vma (vma);
			return nc + 16;
		case PREFIX_HEX:
			nc = printf("0x");
			/* Drop through.  */
		case HEX:
			return nc + print_hex_vma(vma);
		case DEC:
			return print_dec_vma(vma, 1);
		case DEC_5:
			if (vma <= 99999)
				return printf("%5ld", _bfd_int64_low(vma));
			else
				return print_hex_vma(vma);
		case UNSIGNED:
			return print_dec_vma(vma, 0);
		}
	}

	return 0;
}

/* Display a symbol on stdout.  If do_wide is not true then
   format the symbol to be at most WIDTH characters,
   truncating as necessary.  If WIDTH is negative then
   format the string to be exactly - WIDTH characters,
   truncating or padding as necessary.  */
static void print_symbol(int width, const char *symbol)
{
	if (width < 0) {
		printf("%-*.*s", width, width, symbol);
	}
	else {
		printf("%-.*s", width, symbol);
	}
}

static bfd_vma byte_get_big_endian(unsigned char *field, int size)
{
	switch (size) {
	case 1:
		return *field;
	case 2:
		return ((unsigned int) (field[1])) | (((int) (field[0])) << 8);
	case 4:
		return ((unsigned long) (field[3]))
			| (((unsigned long) (field[2])) << 8)
			| (((unsigned long) (field[1])) << 16)
			| (((unsigned long) (field[0])) << 24);
	case 8:
		return ((bfd_vma) (field[7]))
			| (((bfd_vma) (field[6])) << 8)
			| (((bfd_vma) (field[5])) << 16)
			| (((bfd_vma) (field[4])) << 24)
			| (((bfd_vma) (field[3])) << 32)
			| (((bfd_vma) (field[2])) << 40)
			| (((bfd_vma) (field[1])) << 48)
			| (((bfd_vma) (field[0])) << 56);
	default:
		error("Unhandled data length: %d\n", size);
		abort();
		return 0;
	}
}

static void byte_put_big_endian(unsigned char *field, bfd_vma value, int size)
{
	switch (size) {
	case 8:
		field[7] = value & 0xff;
		field[6] = (value >> 8) & 0xff;
		field[5] = (value >> 16) & 0xff;
		field[4] = (value >> 24) & 0xff;
		value >>= 16;
		value >>= 16;
		/* Fall through.  */
	case 4:
		field[3] = value & 0xff;
		field[2] = (value >> 8) & 0xff;
		value >>= 16;
		/* Fall through.  */
	case 2:
		field[1] = value & 0xff;
		value >>= 8;
		/* Fall through.  */
	case 1:
		field[0] = value & 0xff;
		break;

	default:
		error("Unhandled data length: %d\n", size);
		abort();
	}
}

/* Return a pointer to section NAME, or NULL if no such section exists.  */
static Elf_Internal_Shdr *find_section(const char *name)
{
	unsigned int i;

	for (i = 0; i < elf_header.e_shnum; i++) {
		if (streq(SECTION_NAME (section_headers + i), name)) {
			return section_headers + i;
		}
	}

	return NULL;
}

/* Guess the relocation size commonly used by the specific machines.  */
static int guess_is_rela(unsigned long e_machine)
{
	switch (e_machine) {
	/* Targets that use REL relocations.  */
	case EM_ARM:
	case EM_386:
	case EM_486:
	case EM_960:
	case EM_DLX:
	case EM_OPENRISC:
	case EM_OR32:
	case EM_CYGNUS_M32R:
	case EM_D10V:
	case EM_CYGNUS_D10V:
	case EM_MIPS:
	case EM_MIPS_RS3_LE:
		return FALSE;

	/* Targets that use RELA relocations.  */
	case EM_68K:
	case EM_H8_300:
	case EM_H8_300H:
	case EM_H8S:
	case EM_SPARC32PLUS:
	case EM_SPARCV9:
	case EM_SPARC:
	case EM_PPC:
	case EM_PPC64:
	case EM_V850:
	case EM_CYGNUS_V850:
	case EM_D30V:
	case EM_CYGNUS_D30V:
	case EM_MN10200:
	case EM_CYGNUS_MN10200:
	case EM_MN10300:
	case EM_CYGNUS_MN10300:
	case EM_FR30:
	case EM_CYGNUS_FR30:
	case EM_CYGNUS_FRV:
	case EM_SH:
	case EM_ALPHA:
	case EM_MCORE:
	case EM_IA_64:
	case EM_AVR:
	case EM_AVR_OLD:
	case EM_CRIS:
	case EM_860:
	case EM_X86_64:
	case EM_S390:
	case EM_S390_OLD:
	case EM_MMIX:
	case EM_MSP430:
	case EM_MSP430_OLD:
	case EM_XSTORMY16:
	case EM_CRX:
	case EM_VAX:
	case EM_IP2K:
	case EM_IP2K_OLD:
	case EM_IQ2000:
	case EM_XTENSA:
	case EM_XTENSA_OLD:
	case EM_M32R:
		return TRUE;
	case EM_MMA:
	case EM_PCP:
	case EM_NCPU:
	case EM_NDR1:
	case EM_STARCORE:
	case EM_ME16:
	case EM_ST100:
	case EM_TINYJ:
	case EM_FX66:
	case EM_ST9PLUS:
	case EM_ST7:
	case EM_68HC16:
	case EM_68HC11:
	case EM_68HC08:
	case EM_68HC05:
	case EM_SVX:
	case EM_ST19:
	default:
		warn("Don't know about relocations on this machine architecture\n");
		return FALSE;
	}
}

static int slurp_rela_relocs(
	FILE *file,
	unsigned long rel_offset,
	unsigned long rel_size,
	Elf_Internal_Rela **relasp,
	unsigned long *nrelasp)
{
	Elf_Internal_Rela *relas;
	unsigned long nrelas;
	unsigned int i;

	if (is_32bit_elf) {
		Elf32_External_Rela *erelas;

		erelas = get_data(NULL, file, rel_offset, rel_size, "relocs");
		if (!erelas) {
			return 0;
		}

		nrelas = rel_size / sizeof (Elf32_External_Rela);
		relas = malloc (nrelas * sizeof (Elf_Internal_Rela));

		if (relas == NULL) {
			error("out of memory parsing relocs");
			return 0;
		}

		for (i = 0; i < nrelas; i++) {
			relas[i].r_offset = BYTE_GET(erelas[i].r_offset);
			relas[i].r_info	 = BYTE_GET(erelas[i].r_info);
			relas[i].r_addend = BYTE_GET(erelas[i].r_addend);
		}

		free(erelas);
	}
	else {
		Elf64_External_Rela *erelas;

		erelas = get_data(NULL, file, rel_offset, rel_size, "relocs");
		if (!erelas)
			return 0;

		nrelas = rel_size / sizeof (Elf64_External_Rela);
		relas = malloc (nrelas * sizeof (Elf_Internal_Rela));

		if (relas == NULL) {
			error("out of memory parsing relocs");
			return 0;
		}

		for (i = 0; i < nrelas; i++) {
			relas[i].r_offset = BYTE_GET(erelas[i].r_offset);
			relas[i].r_info	 = BYTE_GET(erelas[i].r_info);
			relas[i].r_addend = BYTE_GET(erelas[i].r_addend);
		}

		free(erelas);
	}
	*relasp = relas;
	*nrelasp = nrelas;

	return 1;
}

static int slurp_rel_relocs(
	FILE *file,
	unsigned long rel_offset,
	unsigned long rel_size,
	Elf_Internal_Rela **relsp,
	unsigned long *nrelsp)
{
	Elf_Internal_Rela *rels;
	unsigned long nrels;
	unsigned int i;

	if (is_32bit_elf) {
		Elf32_External_Rel *erels;

		erels = get_data(NULL, file, rel_offset, rel_size, "relocs");
		if (!erels) {
			return 0;
		}

		nrels = rel_size / sizeof (Elf32_External_Rel);
		rels = malloc(nrels * sizeof (Elf_Internal_Rela));

		if (rels == NULL) {
			error("out of memory parsing relocs");
			return 0;
		}

		for (i = 0; i < nrels; i++) {
			rels[i].r_offset = BYTE_GET(erels[i].r_offset);
			rels[i].r_info	 = BYTE_GET(erels[i].r_info);
			rels[i].r_addend = 0;
		}

		free(erels);
	}
	else {
		Elf64_External_Rel *erels;

		erels = get_data(NULL, file, rel_offset, rel_size, "relocs");
		if (!erels) {
			return 0;
		}

		nrels = rel_size / sizeof (Elf64_External_Rel);
		rels = malloc (nrels * sizeof (Elf_Internal_Rela));

		if (rels == NULL) {
			error("out of memory parsing relocs");
			return 0;
		}

		for (i = 0; i < nrels; i++) {
			rels[i].r_offset = BYTE_GET(erels[i].r_offset);
			rels[i].r_info	 = BYTE_GET(erels[i].r_info);
			rels[i].r_addend = 0;
		}

		free(erels);
	}
	*relsp = rels;
	*nrelsp = nrels;

	return 1;
}

/* Decode the data held in 'elf_header'.  */
static int process_file_header(void)
{
	if (   elf_header.e_ident[EI_MAG0] != ELFMAG0
		|| elf_header.e_ident[EI_MAG1] != ELFMAG1
		|| elf_header.e_ident[EI_MAG2] != ELFMAG2
		|| elf_header.e_ident[EI_MAG3] != ELFMAG3) {
		error("Not an ELF file - it has the wrong magic bytes at the start\n");
		return 0;
	}

	return 1;
}

static int get_32bit_program_headers(FILE *file, Elf_Internal_Phdr *program_headers)
{
	Elf32_External_Phdr *phdrs;
	Elf32_External_Phdr *external;
	Elf_Internal_Phdr *internal;
	unsigned int i;

	phdrs = get_data(NULL, file, elf_header.e_phoff, elf_header.e_phentsize * elf_header.e_phnum, "program headers");
	if (!phdrs) {
		return 0;
	}

	for (i = 0, internal = program_headers, external = phdrs; i < elf_header.e_phnum; i++, internal++, external++) {
		internal->p_type	= BYTE_GET(external->p_type);
		internal->p_offset	= BYTE_GET(external->p_offset);
		internal->p_vaddr	= BYTE_GET(external->p_vaddr);
		internal->p_paddr	= BYTE_GET(external->p_paddr);
		internal->p_filesz	= BYTE_GET(external->p_filesz);
		internal->p_memsz	= BYTE_GET(external->p_memsz);
		internal->p_flags	= BYTE_GET(external->p_flags);
		internal->p_align	= BYTE_GET(external->p_align);
	}

	free(phdrs);

	return 1;
}

static int get_64bit_program_headers(FILE *file, Elf_Internal_Phdr *program_headers)
{
	Elf64_External_Phdr *phdrs;
	Elf64_External_Phdr *external;
	Elf_Internal_Phdr *internal;
	unsigned int i;

	phdrs = get_data(NULL, file, elf_header.e_phoff, elf_header.e_phentsize * elf_header.e_phnum, "program headers");
	if (!phdrs) {
		return 0;
	}

	for (i = 0, internal = program_headers, external = phdrs; i < elf_header.e_phnum; i++, internal++, external++) {
		internal->p_type	= BYTE_GET(external->p_type);
		internal->p_flags	= BYTE_GET(external->p_flags);
		internal->p_offset	= BYTE_GET(external->p_offset);
		internal->p_vaddr	= BYTE_GET(external->p_vaddr);
		internal->p_paddr	= BYTE_GET(external->p_paddr);
		internal->p_filesz	= BYTE_GET(external->p_filesz);
		internal->p_memsz	= BYTE_GET(external->p_memsz);
		internal->p_align	= BYTE_GET(external->p_align);
	}

	free(phdrs);

	return 1;
}

/* Returns 1 if the program headers were read into `program_headers'.  */
static int get_program_headers(FILE *file)
{
	Elf_Internal_Phdr *phdrs;

	/* Check cache of prior read.  */
	if (program_headers != NULL) {
		return 1;
	}

	phdrs = malloc(elf_header.e_phnum * sizeof (Elf_Internal_Phdr));

	if (phdrs == NULL) {
		error("Out of memory\n");
		return 0;
	}

	if (is_32bit_elf ? get_32bit_program_headers (file, phdrs) : get_64bit_program_headers (file, phdrs)) {
		program_headers = phdrs;
		return 1;
	}

	free(phdrs);

	return 0;
}

static int get_32bit_section_headers(FILE *file, unsigned int num)
{
	Elf32_External_Shdr *shdrs;
	Elf_Internal_Shdr *internal;
	unsigned int i;

	shdrs = get_data(NULL, file, elf_header.e_shoff, elf_header.e_shentsize * num, "section headers");
	if (!shdrs) {
		return 0;
	}

	section_headers = malloc (num * sizeof (Elf_Internal_Shdr));

	if (section_headers == NULL) {
		error("Out of memory\n");
		return 0;
	}

	for (i = 0, internal = section_headers; i < num; i++, internal++) {
		internal->sh_name		= BYTE_GET(shdrs[i].sh_name);
		internal->sh_type		= BYTE_GET(shdrs[i].sh_type);
		internal->sh_flags		= BYTE_GET(shdrs[i].sh_flags);
		internal->sh_addr		= BYTE_GET(shdrs[i].sh_addr);
		internal->sh_offset		= BYTE_GET(shdrs[i].sh_offset);
		internal->sh_size		= BYTE_GET(shdrs[i].sh_size);
		internal->sh_link		= BYTE_GET(shdrs[i].sh_link);
		internal->sh_info		= BYTE_GET(shdrs[i].sh_info);
		internal->sh_addralign	= BYTE_GET(shdrs[i].sh_addralign);
		internal->sh_entsize	= BYTE_GET(shdrs[i].sh_entsize);
	}

	free(shdrs);

	return 1;
}

static int get_64bit_section_headers(FILE *file, unsigned int num)
{
	Elf64_External_Shdr *shdrs;
	Elf_Internal_Shdr *internal;
	unsigned int i;

	shdrs = get_data(NULL, file, elf_header.e_shoff, elf_header.e_shentsize * num, "section headers");
	if (!shdrs) {
		return 0;
	}

	section_headers = malloc(num * sizeof (Elf_Internal_Shdr));

	if (section_headers == NULL) {
		error("Out of memory\n");
		return 0;
	}

	for (i = 0, internal = section_headers; i < num; i++, internal++) {
		internal->sh_name		= BYTE_GET(shdrs[i].sh_name);
		internal->sh_type		= BYTE_GET(shdrs[i].sh_type);
		internal->sh_flags		= BYTE_GET(shdrs[i].sh_flags);
		internal->sh_addr		= BYTE_GET(shdrs[i].sh_addr);
		internal->sh_size		= BYTE_GET(shdrs[i].sh_size);
		internal->sh_entsize	= BYTE_GET(shdrs[i].sh_entsize);
		internal->sh_link		= BYTE_GET(shdrs[i].sh_link);
		internal->sh_info		= BYTE_GET(shdrs[i].sh_info);
		internal->sh_offset		= BYTE_GET(shdrs[i].sh_offset);
		internal->sh_addralign	= BYTE_GET(shdrs[i].sh_addralign);
	}

	free(shdrs);

	return 1;
}

#define GET_ELF_SYMBOLS(file, section)						\
	(is_32bit_elf ? get_32bit_elf_symbols (file, section)	\
	 : get_64bit_elf_symbols (file, section))

static Elf_Internal_Sym *get_32bit_elf_symbols(FILE *file, Elf_Internal_Shdr *section)
{
	unsigned long number;
	Elf32_External_Sym *esyms;
	Elf_External_Sym_Shndx *shndx;
	Elf_Internal_Sym *isyms;
	Elf_Internal_Sym *psym;
	unsigned int j;

	esyms = get_data(NULL, file, section->sh_offset, section->sh_size, "symbols");
	if (!esyms) {
		return NULL;
	}

	shndx = NULL;
	if (symtab_shndx_hdr != NULL && (symtab_shndx_hdr->sh_link
		== (unsigned long) SECTION_HEADER_NUM (section - section_headers))) {
		shndx = get_data(NULL, file, symtab_shndx_hdr->sh_offset, symtab_shndx_hdr->sh_size, "symtab shndx");
		if (!shndx) {
			free(esyms);
			return NULL;
		}
	}

	number = section->sh_size / section->sh_entsize;
	isyms = malloc (number * sizeof (Elf_Internal_Sym));

	if (isyms == NULL) {
		error("Out of memory\n");
		if (shndx)
			free(shndx);
		free(esyms);
		return NULL;
	}

	for (j = 0, psym = isyms; j < number; j++, psym++) {
		psym->st_name  = BYTE_GET(esyms[j].st_name);
		psym->st_value = BYTE_GET(esyms[j].st_value);
		psym->st_size  = BYTE_GET(esyms[j].st_size);
		psym->st_shndx = BYTE_GET(esyms[j].st_shndx);
		if (psym->st_shndx == SHN_XINDEX && shndx != NULL)
			psym->st_shndx = byte_get((unsigned char *) &shndx[j], sizeof (shndx[j]));
		psym->st_info  = BYTE_GET(esyms[j].st_info);
		psym->st_other = BYTE_GET(esyms[j].st_other);
	}

	if (shndx)
		free(shndx);
	free(esyms);

	return isyms;
}

static Elf_Internal_Sym *get_64bit_elf_symbols(FILE *file, Elf_Internal_Shdr *section)
{
	unsigned long number;
	Elf64_External_Sym *esyms;
	Elf_External_Sym_Shndx *shndx;
	Elf_Internal_Sym *isyms;
	Elf_Internal_Sym *psym;
	unsigned int j;

	esyms = get_data(NULL, file, section->sh_offset, section->sh_size, "symbols");
	if (!esyms) {
		return NULL;
	}

	shndx = NULL;
	if (symtab_shndx_hdr != NULL && (symtab_shndx_hdr->sh_link
		== (unsigned long) SECTION_HEADER_NUM (section - section_headers))) {
		shndx = get_data(NULL, file, symtab_shndx_hdr->sh_offset, symtab_shndx_hdr->sh_size, "symtab shndx");
		if (!shndx) {
			free(esyms);
			return NULL;
		}
	}

	number = section->sh_size / section->sh_entsize;
	isyms = malloc (number * sizeof (Elf_Internal_Sym));

	if (isyms == NULL) {
		error("Out of memory\n");
		if (shndx)
		free(shndx);
		free(esyms);
		return NULL;
	}

	for (j = 0, psym = isyms; j < number; j++, psym++) {
		psym->st_name  = BYTE_GET(esyms[j].st_name);
		psym->st_info  = BYTE_GET(esyms[j].st_info);
		psym->st_other = BYTE_GET(esyms[j].st_other);
		psym->st_shndx = BYTE_GET(esyms[j].st_shndx);
		if (psym->st_shndx == SHN_XINDEX && shndx != NULL)
			psym->st_shndx = byte_get((unsigned char *) &shndx[j], sizeof (shndx[j]));
		psym->st_value = BYTE_GET(esyms[j].st_value);
		psym->st_size  = BYTE_GET(esyms[j].st_size);
	}

	if (shndx)
		free(shndx);
	free(esyms);

	return isyms;
}

/* Display the contents of the relocation data found at the specified offset.  */
static int dump_relocations(
	FILE *file,
	unsigned long rel_offset,
	unsigned long rel_size,
	Elf_Internal_Sym *symtab,
	unsigned long nsyms,
	char *strtab,
	unsigned long strtablen,
	int is_rela)
{
	unsigned int i;
	Elf_Internal_Rela *rels;

	if (is_rela == UNKNOWN) {
		is_rela = guess_is_rela(elf_header.e_machine);
	}

	if (is_rela) {
		if (!slurp_rela_relocs(file, rel_offset, rel_size, &rels, &rel_size)) {
			return 0;
		}
	}
	else {
		if (!slurp_rel_relocs(file, rel_offset, rel_size, &rels, &rel_size)) {
			return 0;
		}
	}

	if (is_32bit_elf) {
		if (is_rela) {
			printf(" Offset     Info    Type            Sym.Value  Sym. Name + Addend\n");
		}
		else {
			printf(" Offset     Info    Type            Sym.Value  Sym. Name\n");
		}
	}
	else {
		if (is_rela) {
			printf("  Offset          Info           Type           Sym. Value    Sym. Name + Addend\n");
		}
		else {
			printf("  Offset          Info           Type           Sym. Value    Sym. Name\n");
		}
	}

	for (i = 0; i < rel_size; i++) {
		const char *rtype;
		const char *rtype2 = NULL;
		const char *rtype3 = NULL;
		bfd_vma offset;
		bfd_vma info;
		bfd_vma symtab_index;
		bfd_vma type;
		bfd_vma type2 = 0;
		bfd_vma type3 = 0;

		offset = rels[i].r_offset;
		info = rels[i].r_info;

		if (is_32bit_elf) {
			type		 = ELF32_R_TYPE (info);
			symtab_index = ELF32_R_SYM  (info);
		}
		else {
			if (elf_header.e_machine == EM_MIPS) {
				/* In little-endian objects, r_info isn't really a 64-bit
				 little-endian value: it has a 32-bit little-endian
				 symbol index followed by four individual byte fields.
				 Reorder INFO accordingly.  */
				if (elf_header.e_ident[EI_DATA] != ELFDATA2MSB) {
					info = (((info & 0xffffffff) << 32)
						| ((info >> 56) & 0xff)
						| ((info >> 40) & 0xff00)
						| ((info >> 24) & 0xff0000)
						| ((info >> 8) & 0xff000000));
				}
				type  = ELF64_MIPS_R_TYPE (info);
				type2 = ELF64_MIPS_R_TYPE2 (info);
				type3 = ELF64_MIPS_R_TYPE3 (info);
			}
			else if (elf_header.e_machine == EM_SPARCV9) {
				type = ELF64_R_TYPE_ID (info);
			}
			else {
				type = ELF64_R_TYPE (info);
			}
			symtab_index = ELF64_R_SYM  (info);
		}

		if (is_32bit_elf) {
			printf("%8.8lx  %8.8lx ", _bfd_int64_low(offset), _bfd_int64_low(info));
		}
		else {
			printf("%4.4lx%8.8lx  %4.4lx%8.8lx ",
				_bfd_int64_high(offset),
				_bfd_int64_low(offset),
				_bfd_int64_high(info),
				_bfd_int64_low(info));
		}

		switch (elf_header.e_machine) {
		case EM_386:
		case EM_486:
			rtype = elf_i386_reloc_type (type);
			break;
		case EM_68HC11:
		case EM_68HC12:
			rtype = elf_m68hc11_reloc_type (type);
			break;
		case EM_68K:
			rtype = elf_m68k_reloc_type (type);
			break;
		case EM_SH:
			rtype = elf_sh_reloc_type (type);
			break;
		case EM_PPC:
			rtype = elf_ppc_reloc_type (type);
			break;
		case EM_PPC64:
			rtype = elf_ppc64_reloc_type (type);
			break;
		case EM_MIPS:
		case EM_MIPS_RS3_LE:
			rtype = elf_mips_reloc_type (type);
			if (!is_32bit_elf) {
				rtype2 = elf_mips_reloc_type (type2);
				rtype3 = elf_mips_reloc_type (type3);
			}
			break;
		case EM_ARM:
			rtype = elf_arm_reloc_type (type);
			break;
		case EM_IA_64:
			rtype = elf_ia64_reloc_type (type);
			break;
		case EM_X86_64:
			rtype = elf_x86_64_reloc_type (type);
			break;
		default:
			rtype = NULL;
			break;
		}

		if (rtype == NULL) {
			printf("unrecognized: %-7lx"), _bfd_int64_low(type);
		}
		else {
			printf("%-17.17s", rtype);
		}

		if (symtab_index) {
			if (symtab == NULL || symtab_index >= nsyms) {
				printf(" bad symbol index: %08lx", (unsigned long) symtab_index);
			}
			else {
				Elf_Internal_Sym *psym;

				psym = symtab + symtab_index;

				printf(" ");
				print_vma(psym->st_value, LONG_HEX);
				printf(is_32bit_elf ? "   " : " ");

				if (psym->st_name == 0) {
					const char *sec_name = "<null>";
					char name_buf[40];

					if (ELF_ST_TYPE (psym->st_info) == STT_SECTION) {
						bfd_vma sec_index = (bfd_vma) -1;

						if (psym->st_shndx < SHN_LORESERVE) {
							sec_index = psym->st_shndx;
						}
						else if (psym->st_shndx > SHN_HIRESERVE) {
							sec_index = psym->st_shndx - (SHN_HIRESERVE + 1 - SHN_LORESERVE);
						}

						if (sec_index != (bfd_vma) -1) {
							sec_name = SECTION_NAME (section_headers + sec_index);
						}
						else if (psym->st_shndx == SHN_ABS) {
							sec_name = "ABS";
						}
						else if (psym->st_shndx == SHN_COMMON) {
							sec_name = "COMMON";
						}
						else if (elf_header.e_machine == EM_IA_64
							 && elf_header.e_ident[EI_OSABI] == ELFOSABI_HPUX
							 && psym->st_shndx == SHN_IA_64_ANSI_COMMON) {
							sec_name = "ANSI_COM";
						}
						else {
							sprintf(name_buf, "<section 0x%x>", (unsigned int) psym->st_shndx);
							sec_name = name_buf;
						}
					}
					print_symbol(22, sec_name);
				}
				else if (strtab == NULL) {
					printf("<string table index: %3ld>", psym->st_name);
				}
				else if (psym->st_name > strtablen) {
					printf("<corrupt string table index: %3ld>", psym->st_name);
				}
				else {
					print_symbol(22, strtab + psym->st_name);
				}

				if (is_rela) {
					printf(" + %lx", (unsigned long) rels[i].r_addend);
				}
			}
		}
		else if (is_rela) {
			printf("%*c", is_32bit_elf ? 28 : 20, ' ');
			print_vma(rels[i].r_addend, LONG_HEX);
		}

		if (elf_header.e_machine == EM_SPARCV9 && streq(rtype, "R_SPARC_OLO10")) {
			printf(" + %lx", (unsigned long) ELF64_R_TYPE_DATA (info));
		}

		putchar('\n');

		if (! is_32bit_elf && elf_header.e_machine == EM_MIPS) {
			printf("                    Type2: ");

			if (rtype2 == NULL) {
				printf("unrecognized: %-7lx", _bfd_int64_low(type2));
			}
			else {
				printf("%-17.17s", rtype2);
			}

			printf("\n                    Type3: ");

			if (rtype3 == NULL) {
				printf("unrecognized: %-7lx", _bfd_int64_low(type3));
			}
			else {
				printf("%-17.17s", rtype3);
			}

			putchar('\n');
		}
	}

	free(rels);

	return 1;
}

/* Process the reloc section.  */
static void process_relocs(FILE *file)
{
	unsigned long rel_size;
	unsigned long rel_offset;
	Elf_Internal_Shdr *section;
	unsigned long i;
	int found = 0;

	for (i = 0, section = section_headers; i < elf_header.e_shnum; i++, section++) {
		if (section->sh_type != SHT_RELA && section->sh_type != SHT_REL) {
			continue;
		}

		rel_offset = section->sh_offset;
		rel_size = section->sh_size;

		if (rel_size) {
			Elf_Internal_Shdr *strsec;
			int is_rela;

			printf("\nRelocation section ");

			if (string_table == NULL) {
				printf("%d", section->sh_name);
			}
			else {
				printf("'%s'", SECTION_NAME (section));
			}

			printf(" at offset 0x%lx contains %lu entries:\n", rel_offset, (unsigned long) (rel_size / section->sh_entsize));

			is_rela = section->sh_type == SHT_RELA;

			if (section->sh_link) {
				Elf_Internal_Shdr *symsec;
				Elf_Internal_Sym *symtab;
				unsigned long nsyms;
				unsigned long strtablen;
				char *strtab = NULL;

				symsec = SECTION_HEADER (section->sh_link);
				nsyms = symsec->sh_size / symsec->sh_entsize;
				symtab = GET_ELF_SYMBOLS(file, symsec);

				if (symtab == NULL) {
					continue;
				}

				strsec = SECTION_HEADER(symsec->sh_link);

				strtab = get_data(NULL, file, strsec->sh_offset, strsec->sh_size, "string table");
				strtablen = strtab == NULL ? 0 : strsec->sh_size;

				dump_relocations(file, rel_offset, rel_size, symtab, nsyms, strtab, strtablen, is_rela);
				if (strtab) {
					free(strtab);
				}
				free(symtab);
			}
			else {
				dump_relocations(file, rel_offset, rel_size, NULL, 0, NULL, 0, is_rela);
			}

			found = 1;
		}
	}

	if (!found) {
		printf("\nThere are no relocations in this file.\n");
	}
}

static const char *get_symbol_binding(unsigned int binding)
{
	static char buff[32];

	switch (binding) {
	case STB_LOCAL:
		return "LOCAL";
	case STB_GLOBAL:
		return "GLOBAL";
	case STB_WEAK:
		return "WEAK";
	default:
		if (binding >= STB_LOPROC && binding <= STB_HIPROC) {
			snprintf(buff, sizeof (buff), "<processor specific>: %d", binding);
		}
		else if (binding >= STB_LOOS && binding <= STB_HIOS) {
			snprintf(buff, sizeof (buff), "<OS specific>: %d", binding);
		}
		else {
			snprintf(buff, sizeof (buff), "<unknown>: %d", binding);
		}
		return buff;
	}
}

static const char *get_symbol_type(unsigned int type)
{
	static char buff[32];

	switch (type) {
	case STT_NOTYPE:
		return "NOTYPE";
	case STT_OBJECT:
		return "OBJECT";
	case STT_FUNC:
		return "FUNC";
	case STT_SECTION:
		return "SECTION";
	case STT_FILE:
		return "FILE";
	case STT_COMMON:
		return "COMMON";
	case STT_TLS:
		return "TLS";
	default:
		if (type >= STT_LOPROC && type <= STT_HIPROC) {
			if (elf_header.e_machine == EM_ARM && type == STT_ARM_TFUNC) {
				return "THUMB_FUNC";
			}
			snprintf(buff, sizeof (buff), "<processor specific>: %d", type);
		}
		else if (type >= STT_LOOS && type <= STT_HIOS) {
			snprintf(buff, sizeof (buff), "<OS specific>: %d", type);
		}
		else{
			snprintf(buff, sizeof (buff), "<unknown>: %d", type);
		}
		return buff;
	}
}

static const char *get_symbol_visibility(unsigned int visibility)
{
	switch (visibility) {
	case STV_DEFAULT:
		return "DEFAULT";
	case STV_INTERNAL:
		return "INTERNAL";
	case STV_HIDDEN:
		return "HIDDEN";
	case STV_PROTECTED:
		return "PROTECTED";
	default:
		abort();
		return 0;
	}
}

static const char *get_symbol_index_type(unsigned int type)
{
	static char buff[32];

	switch (type) {
	case SHN_UNDEF:	return "UND";
	case SHN_ABS:	return "ABS";
	case SHN_COMMON:	return "COM";
	default:
		if (type == SHN_IA_64_ANSI_COMMON
			&& elf_header.e_machine == EM_IA_64
			&& elf_header.e_ident[EI_OSABI] == ELFOSABI_HPUX) {
			return "ANSI_COM";
		}
		else if (type >= SHN_LOPROC && type <= SHN_HIPROC) {
			sprintf(buff, "PRC[0x%04x]", type);
		}
		else if (type >= SHN_LOOS && type <= SHN_HIOS) {
			sprintf(buff, "OS [0x%04x]", type);
		}
		else if (type >= SHN_LORESERVE && type <= SHN_HIRESERVE) {
			sprintf(buff, "RSV[0x%04x]", type);
		}
		else {
			sprintf(buff, "%3d", type);
		}
		break;
	}

	return buff;
}

/* Dump the symbol table.  */
static void process_symbol_table(FILE *file)
{
	Elf_Internal_Shdr *section;
	unsigned int i;

	for (i = 0, section = section_headers; i < elf_header.e_shnum; i++, section++) {
		unsigned int si;
		char *strtab;
		Elf_Internal_Sym *symtab;
		Elf_Internal_Sym *psym;

		if (section->sh_type != SHT_SYMTAB && section->sh_type != SHT_DYNSYM) {
			continue;
		}

		printf("\nSymbol table '%s' contains %lu entries:\n", SECTION_NAME (section),
			(unsigned long) (section->sh_size / section->sh_entsize));
		if (is_32bit_elf) {
			printf("   Num:    Value  Size Type    Bind   Vis      Ndx Name\n");
		}
		else {
			printf("   Num:    Value          Size Type    Bind   Vis      Ndx Name\n");
		}

		symtab = GET_ELF_SYMBOLS(file, section);
		if (symtab == NULL) {
			continue;
		}

		if (section->sh_link == elf_header.e_shstrndx) {
			strtab = string_table;
		}
		else {
			Elf_Internal_Shdr *string_sec;
			string_sec = SECTION_HEADER(section->sh_link);
			strtab = get_data(NULL, file, string_sec->sh_offset, string_sec->sh_size, "string table");
		}

		for (si = 0, psym = symtab; si < section->sh_size / section->sh_entsize; si++, psym++) {
			printf("%6d: ", si);
			print_vma(psym->st_value, LONG_HEX);
			putchar(' ');
			print_vma(psym->st_size, DEC_5);
			printf(" %-7s", get_symbol_type(ELF_ST_TYPE(psym->st_info)));
			printf(" %-6s", get_symbol_binding(ELF_ST_BIND(psym->st_info)));
			printf(" %-3s", get_symbol_visibility(ELF_ST_VISIBILITY(psym->st_other)));
			printf(" %4s ", get_symbol_index_type(psym->st_shndx));
			print_symbol(25, strtab + psym->st_name);
			putchar('\n');
		}

		free(symtab);
		if (strtab != string_table) {
			free(strtab);
		}
	}
}

static int get_file_header(FILE *file)
{
	/* Read in the identity array.  */
	if (fread(elf_header.e_ident, EI_NIDENT, 1, file) != 1) {
		return 0;
	}

	/* Determine how to read the rest of the header.  */
	switch (elf_header.e_ident[EI_DATA]) {
	case ELFDATANONE: /* fall through */
	case ELFDATA2LSB:
		byte_get = byte_get_little_endian;
		byte_put = byte_put_little_endian;
		break;
	case ELFDATA2MSB:
		byte_get = byte_get_big_endian;
		byte_put = byte_put_big_endian;
		break;
	}

	/* For now we only support 32 bit and 64 bit ELF files.  */
	is_32bit_elf = (elf_header.e_ident[EI_CLASS] != ELFCLASS64);

	/* Read in the rest of the header.  */
	if (is_32bit_elf) {
		Elf32_External_Ehdr ehdr32;

		if (fread(ehdr32.e_type, sizeof (ehdr32) - EI_NIDENT, 1, file) != 1) {
			return 0;
		}

		elf_header.e_type      = BYTE_GET(ehdr32.e_type);
		elf_header.e_machine   = BYTE_GET(ehdr32.e_machine);
		elf_header.e_version   = BYTE_GET(ehdr32.e_version);
		elf_header.e_entry     = BYTE_GET(ehdr32.e_entry);
		elf_header.e_phoff     = BYTE_GET(ehdr32.e_phoff);
		elf_header.e_shoff     = BYTE_GET(ehdr32.e_shoff);
		elf_header.e_flags     = BYTE_GET(ehdr32.e_flags);
		elf_header.e_ehsize    = BYTE_GET(ehdr32.e_ehsize);
		elf_header.e_phentsize = BYTE_GET(ehdr32.e_phentsize);
		elf_header.e_phnum     = BYTE_GET(ehdr32.e_phnum);
		elf_header.e_shentsize = BYTE_GET(ehdr32.e_shentsize);
		elf_header.e_shnum     = BYTE_GET(ehdr32.e_shnum);
		elf_header.e_shstrndx  = BYTE_GET(ehdr32.e_shstrndx);
	}
	else {
		Elf64_External_Ehdr ehdr64;

		/* If we have been compiled with sizeof (bfd_vma) == 4, then
		 we will not be able to cope with the 64bit data found in
		 64 ELF files.  Detect this now and abort before we start
		 overwriting things.  */
		if (sizeof (bfd_vma) < 8) {
			error("This instance of readelf has been built without support for a\n\
				64 bit data type and so it cannot read 64 bit ELF files.\n");
			return 0;
		}

		if (fread (ehdr64.e_type, sizeof (ehdr64) - EI_NIDENT, 1, file) != 1) {
			return 0;
		}

		elf_header.e_type      = BYTE_GET(ehdr64.e_type);
		elf_header.e_machine   = BYTE_GET(ehdr64.e_machine);
		elf_header.e_version   = BYTE_GET(ehdr64.e_version);
		elf_header.e_entry     = BYTE_GET(ehdr64.e_entry);
		elf_header.e_phoff     = BYTE_GET(ehdr64.e_phoff);
		elf_header.e_shoff     = BYTE_GET(ehdr64.e_shoff);
		elf_header.e_flags     = BYTE_GET(ehdr64.e_flags);
		elf_header.e_ehsize    = BYTE_GET(ehdr64.e_ehsize);
		elf_header.e_phentsize = BYTE_GET(ehdr64.e_phentsize);
		elf_header.e_phnum     = BYTE_GET(ehdr64.e_phnum);
		elf_header.e_shentsize = BYTE_GET(ehdr64.e_shentsize);
		elf_header.e_shnum     = BYTE_GET(ehdr64.e_shnum);
		elf_header.e_shstrndx  = BYTE_GET(ehdr64.e_shstrndx);
	}

	return 1;
}

static int process_section_headers(FILE *file)
{
	Elf_Internal_Shdr *section;
	unsigned int i;

	section_headers = NULL;

	if (is_32bit_elf) {
		if (! get_32bit_section_headers (file, elf_header.e_shnum)) {
			return 0;
		}
	}
	else if (! get_64bit_section_headers (file, elf_header.e_shnum)) {
		return 0;
	}

	/* Read in the string table, so that we have names to display.  */
	section = SECTION_HEADER(elf_header.e_shstrndx);

	if (section->sh_size != 0) {
		string_table = get_data(NULL, file, section->sh_offset, section->sh_size, "string table");
		if (string_table == NULL) {
			return 0;
		}
		string_table_length = section->sh_size;
	}

	symtab_shndx_hdr = NULL;

	eh_addr_size = is_32bit_elf ? 4 : 8;
	switch (elf_header.e_machine) {
	case EM_MIPS:
	case EM_MIPS_RS3_LE:
		/* The 64-bit MIPS EABI uses a combination of 32-bit ELF and 64-bit
		 FDE addresses.  However, the ABI also has a semi-official ILP32
		 variant for which the normal FDE address size rules apply.
		 GCC 4.0 marks EABI64 objects with a dummy .gcc_compiled_longXX
		 section, where XX is the size of longs in bits.  Unfortunately,
		 earlier compilers provided no way of distinguishing ILP32 objects
		 from LP64 objects, so if there's any doubt, we should assume that
		 the official LP64 form is being used.  */
		if ((elf_header.e_flags & EF_MIPS_ABI) == E_MIPS_ABI_EABI64 && find_section(".gcc_compiled_long32") == NULL) {
			eh_addr_size = 8;
		}
		break;
	}

	for (i = 0, section = section_headers; i < elf_header.e_shnum; i++, section++) {
		char *name = SECTION_NAME (section);

		if (section->sh_type == SHT_SYMTAB_SHNDX) {
			if (symtab_shndx_hdr != NULL) {
			    error("File contains multiple symtab shndx tables\n");
				continue;
			}
			symtab_shndx_hdr = section;
		}
	}

	return 1;
}

/* Process one ELF object file according to the command line options.
   This file may actually be stored in an archive.  The file is
   positioned at the start of the ELF object.  */
static int process_object(char *file_name, FILE *file)
{
	unsigned int i;

	if (!get_file_header(file)) {
		error("%s: Failed to read file header\n", file_name);
		return 1;
	}
	/* Process the file.  */
	if (show_name) {
		printf("\nFile: %s\n", file_name);
	}
	if (!process_file_header()) {
		return 1;
	}
	if (!process_section_headers(file)) {
		do_syms = do_reloc = 0;
	}

	if (do_reloc) {
		process_relocs(file);
	}
	if (do_syms) {
		process_symbol_table(file);
	}

	if (program_headers) {
		free(program_headers);
		program_headers = NULL;
	}
	if (section_headers) {
		free(section_headers);
		section_headers = NULL;
	}
	if (string_table) {
		free(string_table);
		string_table = NULL;
		string_table_length = 0;
	}

	return 0;
}

static int process_file(char *file_name)
{
	FILE *file;
	struct stat statbuf;
	int ret;

	if (stat(file_name, &statbuf) < 0) {
		if (errno == ENOENT)
			error("'%s': No such file\n", file_name);
		else
			error("Could not locate '%s'.  System error message: %s\n", file_name, strerror(errno));
		return 1;
	}

	if (!S_ISREG(statbuf.st_mode)) {
		error("'%s' is not an ordinary file\n", file_name);
		return 1;
	}

	file = fopen(file_name, "rb");
	if (file == NULL) {
		error("Input file '%s' is not readable.\n", file_name);
		return 1;
	}

	rewind(file);
	archive_file_size = archive_file_offset = 0;
	ret = process_object(file_name, file);

	fclose(file);

	return ret;
}

static void usage(void)
{
	fprintf(stdout, "Usage: readelf <option(s)> elf-file(s)\n");
	fprintf(stdout, " Display information about the contents of ELF format files\n");
	fprintf(stdout, " Options are:\n\
	-s --syms              Display the symbol table\n\
	    --symbols          An alias for --syms\n\
	-r --relocs            Display the relocations (if present)\n");

	exit (0);
}

int main(int argc, char **argv)
{
	int err;
	int c;

	if (argc < 2){
		usage();
	}
	while ((c = getopt_long(argc, argv, "rs", NULL, NULL)) != EOF) {
		switch (c) {
		case 0:
			/* Long options.  */
			break;
		case 'r':
			do_reloc++;
			break;
		case 's':
			do_syms++;
			break;
		default:
			usage();
		}
	}

	if (optind < (argc - 1)){
		show_name = 1;
	}

	err = 0;
	while (optind < argc){
		err |= process_file(argv[optind++]);
	}

	return err;
}
