/*
 * tcp.c
 *
 * Copyright 2004, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 *
 * Transmission control Protocl.
 */


#include<sys/config.h>
#include<sys/types.h>
#include<sys/param.h>
#include<kern/lib.h>
#include<kern/errno.h>
#include<kern/mm.h>
#include<kern/proc.h>
#include<kern/time.h>
#include<kern/lock.h>
#include<kern/interrupt.h>
#include<kern/signal.h>
#include<kern/fs.h>
#include<dev/console/console.h>
#include<net/net.h>
#include<net/ethernet.h>
#include<net/ip.h>
#include<net/tcp.h>
#include<net/netlib.h>
#include<kern/debug.h>
#include<kern/test.h>


//#define DEBUG_TCP 1
#ifdef DEBUG_TCP
	#define STATIC
	#define INLINE
#else
	#define STATIC static
	#define INLINE	inline
#endif


/**************************************************************************
 *
 * 
 *
 **************************************************************************/


//================================== PROTECTED ==========================================


enum{
	/* TCP flag. */
	TCP_FLG_FIN = 1<<0,
	TCP_FLG_SYN = 1<<1,
	TCP_FLG_RST = 1<<2,
	TCP_FLG_PSH = 1<<3,
	TCP_FLG_ACK = 1<<4,
	TCP_FLG_URG = 1<<5,

	TCP_WIN_SIZE=0xffff,	/* Window size. */
};


typedef struct{
	uint8_t  kind;
	uint8_t  len;
	uint16_t mss;
}MSS;


/*
 * إåꡣ
 * parameters : dectination IP,pseudo header address,data size
 */
STATIC INLINE void setPseudoHead(in_addr_t dstip,in_addr_t srcip,uint16_t size,PSEUDO_HEADER *pshead)
{
	pshead->dstip=dstip;
	pshead->srcip=srcip;
	pshead->tmp=0;
	pshead->prot=IPPROTO_TCP;
	pshead->len=swapWord(size);
}


/*
 * Get sequence number.
 * return : sequence number
 */
STATIC INLINE uint32_t getSeqNumber()
{
	/* 4msǣ䤹RFC793ˡ */
	return calcTimeFromClock(rdtsc()) * 4;
}


/*
 * TCPХȿ֤
 */
STATIC INLINE int getTcpSize(IP_HEADER *ip)
{
	return swapWord(ip->len) - (ip->verhead & 0xf) * 4;
}


/*
 * TCPǡХȿ֤
 */
STATIC int getTcpDataSize(IP_HEADER *ip)
{
	TCP_HEADER *tcp = (TCP_HEADER*)ip->data;

	return getTcpSize(ip) - (tcp->headlen >> 4) * 4;
}


/**************************************************************************
 *
 * إåץ
 *
 **************************************************************************/


enum{
	/* Header option. */
	TCP_KIND_END=  0,
	TCP_KIND_NOP=  1,
	TCP_KIND_MSS=  2,
	TCP_KIND_WSC=  3,
	TCP_KIND_TSTMP=8,

	TCP_OPT_MAXSIZE=32,		/* TCP ץκ祵 */
};


typedef struct{
	uint32_t timeStamp;
	uint32_t timeEcho;
	uint16_t mss;
	uint8_t  winscal;
}OPTIONS;


//================================== PROTECTED ==========================================


/*
 * Get TCP options.
 * parameters : option address
 */
STATIC void getTcpOptions(uchar *c, int size, OPTIONS *optst)
{
	uchar *last;

	last = c + size;
	while (c < last){
		switch (*c){
		case TCP_KIND_MSS:
			optst->mss = *(uint16_t*)(c + 2);
			c += 4;
			break;
		case TCP_KIND_WSC:
			optst->winscal = *(uint8_t*)(c + 2);
			c += 3;
			break;
		case TCP_KIND_TSTMP:
			optst->timeStamp = *(uint32_t*)(c + 2);
			optst->timeEcho = *(uint32_t*)(c + 6);
			c += 10;
			break;
		case TCP_KIND_NOP:
			c += 1;
			break;
		case TCP_KIND_END:
			break;
		default:
			c += *(c + 1);
		}
	}
}


/**************************************************************************
 *
 * 
 *
 **************************************************************************/


enum{
	SEND_BUF_SIZE = 0x1000,		// Хåե
};


//================================== PROTECTED ==========================================


// Transmit a tcp segment.
// աեץǡɬintǳڤ륵ˤ뤳ȡ
STATIC void transSegment(
	SOCKET *sock,
	uint8_t flag,
	void *opt,
	int opt_size,
	void *data,
	int data_size,
	uint16_t wndsize)
{
	char head[sizeof(PSEUDO_HEADER) + sizeof(TCP_HEADER) + TCP_OPT_MAXSIZE];
	TRANS_BUF_INFO tbi;
	TCP_HEADER *send_tcp = (TCP_HEADER*)(head + sizeof(PSEUDO_HEADER));

	// إå
	setPseudoHead(sock->dstip, getEtherIp(getEtherNum(sock->dstip)), sizeof(TCP_HEADER) + opt_size+data_size, (PSEUDO_HEADER*)head);

	// Хåեơ֥
	tbi.data[2] = data;
	tbi.size[2] = data_size;
	tbi.data[1] = (char*)send_tcp;
	tbi.size[1] = sizeof(TCP_HEADER) + opt_size;
	tbi.ipType = IPPROTO_TCP;

	// Make TCP header.
	send_tcp->srcport = sock->srcport;
	send_tcp->dstport = sock->dstport;
	send_tcp->seqnum =  swapInt32(sock->seqnum);
	send_tcp->acknum =  swapInt32(sock->acknum);
	send_tcp->headlen = ((sizeof(TCP_HEADER) + opt_size) / sizeof(uint32_t)) << 4;
	send_tcp->flag = flag;
	send_tcp->wndsize = swapWord(wndsize);
	send_tcp->chksum = 0;
	send_tcp->urgpoint = 0;
	memcpy(send_tcp->option, opt, opt_size);
	send_tcp->chksum = calcSumM((uint*)head, data, sizeof(PSEUDO_HEADER) + sizeof(TCP_HEADER) + opt_size, data_size);
/************************************************************************************************************************
{
	union{
		uint ui;
		uchar uc[4];
	}src_ip;

	src_ip.ui = sock->dstip;
	printk("send srcport=%x,dstport=%x,seqnum=%x,acknum=%x,flag=%x,ip=%u.%u.%u.%u\n",
		swapWord(send_tcp->srcport),swapWord(send_tcp->dstport),swapInt32(send_tcp->seqnum),swapInt32(send_tcp->acknum),
		send_tcp->flag,src_ip.uc[0],src_ip.uc[1],src_ip.uc[2],src_ip.uc[3]);
}
************************************************************************************************************************/
	// 
	transIp(sock, &tbi, sock->dstip, 0);
}


// Send FIN ack.
STATIC void sendFinAck(SOCKET *sock)
{
	sock->stat |= SOCK_CNCT_RECV_FIN;
	sock->acknum += 1;
	transSegment(sock, TCP_FLG_ACK, NULL, 0, NULL, 0, TCP_WIN_SIZE);
}


// ꥻåȥȤ
STATIC INLINE void transReset(SOCKET *sock, TCP_HEADER *tcp)
{
	sock->seqnum= 0;
	sock->acknum= swapInt32(tcp->seqnum) + 1;
	transSegment(sock, TCP_FLG_RST, NULL, 0, NULL, 0, TCP_WIN_SIZE);
}


// ͥ󤬤ʤΥꥻåȥȤ
STATIC void sendReset(IP_HEADER *ip)
{
	SOCKET sock;
	TCP_HEADER *tcp = (TCP_HEADER*)ip->data;

	sock.srcport = tcp->dstport;
	sock.dstport = tcp->srcport;
	sock.dstip = ip->srcip;
	transReset(&sock, tcp);
}


// åȤ롣
STATIC void ackFromCloseSock(SOCKET *sock, IP_HEADER *ip)
{
	TCP_HEADER *tcp = (TCP_HEADER*)ip->data;

	sock->acknum += getTcpDataSize(ip);

	/* FINФ */
	if (swapInt32(tcp->acknum) == sock->seqnum){
		sock->stat |= SOCK_CNCT_RECV_FINACK;
	}

	/* FINα */
	if (tcp->flag & TCP_FLG_FIN){
		if (sock->stat & SOCK_CNCT_RECV_FINACK){
			sendFinAck(sock);
		}
		else{
			if (swapInt32(tcp->acknum) == sock->seqnum){
				sock->stat |= SOCK_CNCT_RECV_FINACK;
				sendFinAck(sock);
			}
		}
	}

	sock->waitTime = sys_time();
}


// ǡ¸Хåե¸줿ǡ򡢰ΥȰ֤
// ХåեκǸΰ֤ޤǡ祵Ȥ롣
// return : ̤ʤ
STATIC int sendFromBuf(SOCKET *sock, const size_t i_sendStart, const uint8_t transFlag)
{
	size_t sendLast = getLastRingBuf(sock->ringBuf);	// 롼׽ǸΥǡ
	size_t sendStart = i_sendStart;						// 롼ȤΥǡϰ
	int sendBufLen;										// 롼Ȥɥ
	
	// ΥȰ֤ϡɬǡ¸Хåեγϰ֤ȺǸΰ֤δ֤ˤ뤳ȡ
	ASSERT((getStartRingBuf(sock->ringBuf) <= i_sendStart) && (i_sendStart <= getLastRingBuf(sock->ringBuf)));
	
	// ΥȰ֤ǡ¸ХåեΥȰ֤ʤ顢Ϣ³Ȥߤʤ
	// ǡ¸ХåեХåեθ֤߰˥ԡ롣
	// ΥȰ֤ǡ¸ХåեΥȰ֤Ʊʤ顢ľΤȤߤʤ
	// ǡ¸ХåեХåեκǽ˥ԡ롣
	if (i_sendStart == getStartRingBuf(sock->ringBuf)){
		sendBufLen = SEND_BUF_SIZE;
		sock->sendBuf.curLen = 0;
		sock->seqnum = i_sendStart;
	}
	else{
		sendBufLen = SEND_BUF_SIZE - (i_sendStart - getStartRingBuf(sock->ringBuf));
	}
	while (sendStart < sendLast){
		int sendLen = (sock->mss <= (sendLast - sendStart))? sock->mss : (sendLast - sendStart);
		sendLen = (sendLen <= (sock->mss - sock->sendBuf.curLen))? sendLen : (sock->mss - sock->sendBuf.curLen);
		readRingBuf(sock->ringBuf, sendStart, sendLen, sock->sendBuf.buf + sock->sendBuf.curLen);
		sock->sendBuf.curLen += sendLen;
		if (sock->sendBuf.curLen == sock->mss){
			sendBufLen -= sock->mss;
			transSegment(sock, transFlag, NULL, 0, sock->sendBuf.buf, sock->mss, sendBufLen);
			sock->seqnum += sock->mss;
			sock->sendBuf.curLen = 0;
		}
		sendStart += sendLen;
	}
	
	return sendLast - i_sendStart - sock->sendBuf.curLen;
}


/**************************************************************************
 *
 * 
 *
 **************************************************************************/


/* PUBLIC */
static int waitCnctLock;	/* Ԥѥåȡ */


//================================== PRIVATE ============================================


static SOCKET waitCnctSc;	/* Ԥ󥯡 */
static SOCKET waitLstnSc;	/* å󥯡 */


//Գ
// ԤåȤõ
// return : 0 = ̾ｪλ 1 = 򵯤 -1 = 顼
STATIC int searchSocket(struct mbuf *mbuf)
{
	SOCKET *sock;
	IP_HEADER *ip = getMbufDataPointer(mbuf);
	TCP_HEADER *tcp = (TCP_HEADER*)ip->data;
/***************************************************************************
enum{COUNT = 7,};
static int cnt = 0;
***************************************************************************/

	/* ͥ󥯤õ */
	for (sock = waitCnctSc.next; sock != &waitCnctSc; sock = sock->next){
		if ((sock->srcport == tcp->dstport) && (sock->dstport == tcp->srcport) && (sock->dstip == ip->srcip)){
/****************************************************************************************
{
	union{
		uint ui;
		uchar uc[4];
	}srcIp;

	if ((ip->srcip == 0x40019ac) && (swapWord(tcp->srcport) == 20) && (++cnt <= COUNT)){
		srcIp.ui = ip->srcip;
		printk(
			"searchSocket() "
//			"%u.%u.%u.%u "
//			"srcport=%u "
//			"dstport=%u "
			"seq=%x "
			"sockAck=%x\n",
//			srcIp.uc[0],srcIp.uc[1],srcIp.uc[2],srcIp.uc[3],
//			swapWord(tcp->srcport),
//			swapWord(tcp->dstport),
			swapInt32(tcp->seqnum),
			sock->acknum);
	}
}
****************************************************************************************/
			/* åȤϴǤƤ롣 */
			if (sock->stat & SOCK_DISCON_RECV){
				setIpTask(sock, mbuf);
				return 1;
			}

			// ǡ򥽥åȤϤ
			attachRecvBufTcp(mbuf, &sock->recvBuf, &sock->lockGate);
			if (sock->waitProc != NULL){
				sigWakeSoon(sock->waitProc);
			}
			return 1;
		}
	}

	/* å󥯤õ */
	for (sock = waitLstnSc.next; sock != &waitLstnSc; sock = sock->next){
		if ((sock->srcport == tcp->dstport) && ((sock->srcip == INADDR_ANY) || (sock->srcip == ip->dstip))){
			attachRecvBufTcp(mbuf, &sock->recvBuf, &sock->lockGate);
			sys_wake(sock->listenThread);
			return 1;
		}
	}
	
	// Ĥ餺
	setIpTask(NULL, mbuf);

	return 0;
}


//================================== PROTECTED ==========================================


// ǡ
// return : error number
STATIC int waitReceive(SOCKET *sock, uint timeout)
{
	struct mbuf *mbuf;
	int restTime;

	for (mbuf = refRecvBuf(sock->recvBuf); (mbuf == NULL) || (mbuf->mh_access == YES); mbuf = refRecvBuf(sock->recvBuf)){
		mbuf->mh_access = NO;
		sock->waitProc = get_current_task();
		restTime = sys_sleep(timeout);
		if (restTime == 0){			/* Time out. */
			if (refRecvBuf(sock->recvBuf) == NULL){
				return -ETIMEDOUT;
			}
		}

		/* ʥߤ */
		doSignalRestart();
		if (isSigint() == TRUE){
			return -EINTR;
		}
	}

	return NOERR;
}


/*
 * Add to connection link.
 * parameters : socket
 */
STATIC void addWaitCnctLink(SOCKET *sc)
{
	int eflag;

	eflag = enterCli();
	enter_spinlock(&waitCnctLock);
	{
		sc->next = waitCnctSc.next;
		sc->prev = &waitCnctSc;
		waitCnctSc.next->prev = sc;
		waitCnctSc.next = sc;
	}
	exit_spinlock(&waitCnctLock);
	exitCli(eflag);
}


/*
 * Add to connection link.
 * parameters : socket
 */
STATIC void addWaitLstnLink(SOCKET *sc)
{
	int eflag;

	eflag = enterCli();
	enter_spinlock(&waitCnctLock);
	{
		sc->next=waitLstnSc.next;
		sc->prev=&waitLstnSc;
		waitLstnSc.next->prev=sc;
		waitLstnSc.next=sc;
	}
	exit_spinlock(&waitCnctLock);
	exitCli(eflag);
}


/*
 * Delete from connection link.
 * parameters : socket
 */
STATIC void delWaitLink(SOCKET *sc)
{
	int eflag;

	eflag = enterCli();
	enter_spinlock(&waitCnctLock);
	{
		sc->next->prev = sc->prev;
		sc->prev->next = sc->next;
		sc->next = sc->prev = sc;
	}
	exit_spinlock(&waitCnctLock);
	exitCli(eflag);
}


/*
 * å󥽥åȤݡȤõ
 * parameters : source port
 * return : socket address or NULL
 */
STATIC SOCKET *searchListenSocket(in_port_t srcport)
{
	SOCKET *sc,*search_sc = NULL;
	int eflag;

	eflag = enterCli();
	enter_spinlock(&waitCnctLock);
	{
		for (sc = waitLstnSc.next; sc != &waitLstnSc; sc = sc->next)
			if (sc->srcport == srcport)
			{
				search_sc = sc;
				break;
			}
	}
	exit_spinlock(&waitCnctLock);
	exitCli(eflag);
	
	return search_sc;
}


// Ȥγǧ
// parameters : TCP HEADER, ֹ, ֹ
enum{
	ACK_EQ_SEQ_EQ = 1,	// ֹ = ֹ桢ֹ = ֹ
	ACK_LW_SEQ_EQ,		// ֹ < ֹ桢ֹ = ֹ
	ACK_EQ_SEQ_LW,		// ֹ = ֹ桢ֹ < ֹ
	ACK_EQ_SEQ_GR,		// ֹ = ֹ桢ֹ > ֹ
};
STATIC int isAck(TCP_HEADER *tcp, uint32_t sent, uint32_t received)
{
	static uchar returnValue[3][3] = {
		{ACK_EQ_SEQ_EQ,	ACK_EQ_SEQ_LW,	ACK_EQ_SEQ_GR},
		{ACK_LW_SEQ_EQ,	ERR,			ERR},
		{ERR,			ERR,			ERR},
	};
	int isAck, isSeq;
	uint32_t tcpAck = swapInt32(tcp->acknum);
	uint32_t tcpSeq = swapInt32(tcp->seqnum);


	/* ե饰ӡ */
	if ((tcp->flag & TCP_FLG_ACK) == 0)
		return ERR;

	/* ACKͤSEQͤ */
	if (tcpAck == sent){
		isAck = 0;
	}
	else if (tcpAck < sent){
		isAck = 1;
	}
	else{
		isAck = 2;
	}
	if (tcpSeq == received){
		isSeq = 0;
	}
	else if (tcpSeq < received){
		isSeq = 1;
	}
	else{
		isSeq = 2;
	}

	return returnValue[isAck][isSeq];
}


/*
 * ǡ
 * 쥷֥Хåեϥ󥯤ڤΥʤ
 * parameters : socket,쥷֥Хåեݥ,send flag
 * return : error number
 */
STATIC int recvData(SOCKET *sock, const uint timeout)
{
	int error;
	
	for (;;){
		struct mbuf *mbuf;
		IP_HEADER *ip;
		TCP_HEADER *tcp;

		// Ԥ
		error = waitReceive(sock, timeout);
		if (error != NOERR){
			return error;
		}
		mbuf = refRecvBuf(sock->recvBuf);
		ip = getMbufDataPointer(mbuf);
		tcp = (TCP_HEADER*)ip->data;

		// ꥻåȥե饰ʤ饳ͥ
		if (tcp->flag & TCP_FLG_RST){
			detachRecvBuf(mbuf, &sock->recvBuf, &sock->lockGate);
			m_freem(mbuf);
			return -ECONNRESET;
		}

		// γǧ
		switch (isAck(tcp, sock->seqnum, sock->acknum))
		{
			case ACK_EQ_SEQ_EQ:
				return 0;
			case ACK_EQ_SEQ_GR:
				// ΥȤ줿ǽΤǱƤ
				mbuf->mh_access = YES;
				if (tcp->flag & TCP_FLG_FIN){
					sendFinAck(sock);
				}
				else{
					transSegment(sock, TCP_FLG_ACK, NULL, 0, NULL, 0, TCP_WIN_SIZE);
					if (isNoRecvBuf() == YES){
						// IPХåե̵ϥåȤαIPХåեĳ롣
						struct mbuf *end = refRecvEndBuf(sock->recvBuf);
						if (end != NULL){
							detachRecvBuf(end, sock->recvBuf, &sock->lockGate);
							m_freem(end);
						}
					}
				}
				break;
			case ACK_EQ_SEQ_LW:
				//¦ΤǱƤ
				if (tcp->flag & TCP_FLG_FIN){
					sendFinAck(sock);
				}
				else{
					transSegment(sock, TCP_FLG_ACK, NULL, 0, NULL, 0, TCP_WIN_SIZE);
				}
				//
			default:
				detachRecvBuf(mbuf, &sock->recvBuf, &sock->lockGate);
				m_freem(mbuf);
	    }
    }

	return NOERR;
}


/*
 * ǧ롣
 * ˥ǡȤϥǡ˽λ
 * parameters : å,Ԥ֡á,ֹֹˡ
 * return : 0 or RECV_SEND_GET_DATA : ǡȤ or error number
 */
enum{RECV_SEND_GET_DATA = 1,};
STATIC int recvSend(SOCKET *sock, const uint timeout, size_t *o_tcpAck)
{
	int error;

	if (0 < timeout){
		error = waitReceive(sock, timeout);
		if (error != NOERR){
			return error;
		}
	}

	for (;;){
		struct mbuf *mbuf;
		IP_HEADER *ip;
		TCP_HEADER *tcp;

		mbuf = refRecvBuf(sock->recvBuf);
		if (mbuf == NULL){
			break;
		}
		ip = getMbufDataPointer(mbuf);
		tcp = (TCP_HEADER*)ip->data;

		// ꥻåȥե饰ʤ饳ͥ
		if (tcp->flag & TCP_FLG_RST){
			detachRecvBuf(mbuf, &sock->recvBuf, &sock->lockGate);
			m_freem(mbuf);
			return -ECONNRESET;
		}

		switch (isAck(tcp, sock->seqnum, sock->acknum)){
			case ACK_EQ_SEQ_EQ:
				// 
			case ACK_LW_SEQ_EQ:
				*o_tcpAck = swapInt32(tcp->acknum);
				sock->window = swapWord(tcp->wndsize);
				if (0 < getTcpDataSize(ip)){
					// ǡ硢쥷ִؿϤѥåȤĤƤ
					// ϤѥåȤĤäƤ롣
					if (sock->prevAck != tcp->seqnum){
						// ֹ򵭲
						sock->prevAck = tcp->seqnum;
						return RECV_SEND_GET_DATA;
					}
				}
				if (tcp->flag & TCP_FLG_FIN){
					sendFinAck(sock);
				}
				// 
			default:
				detachRecvBuf(mbuf, &sock->recvBuf, &sock->lockGate);
				m_freem(mbuf);
	    }
    }

	return 0;
}


/*
 * Init.
 */
STATIC void initReceive()
{
	waitCnctSc.next=waitCnctSc.prev=&waitCnctSc;
	waitLstnSc.next=waitLstnSc.prev=&waitLstnSc;
}


//================================== PUBLIC =============================================


//Գ
// return : 0 or task switch=1
int receiveTcp(struct mbuf *mbuf)
{
	IP_HEADER *ip = getMbufDataPointer(mbuf);
	int tcpSize;
	PSEUDO_HEADER pseudoHead;

	/* åγǧ */
	tcpSize = getTcpSize(ip);
	setPseudoHead(ip->dstip, ip->srcip, tcpSize, &pseudoHead);
	if (calcSumM((uint*)&pseudoHead, (uint*)ip->data, sizeof(PSEUDO_HEADER), tcpSize)){
		return 0;
	}
	
	/* γǧ */
	if (TCP_MSS_SIZE < getTcpDataSize(ip)){
		return 0;
	}

	return searchSocket(mbuf);
}


/**************************************************************************
 *
 * å󥹥å
 *
 **************************************************************************/

enum{
	LISTEN_WAIT_MAX = 5,	/* listenԤॢȻ֡á */
};


//================================== PRIVATE ============================================


/*
 * åȤå󥽥åȤ³
 * ֿॢȤΥåȤ
 * parameters : å󥽥å,å
 */
STATIC void addNewSocket(SOCKET *sc, SOCKET *newsc)
{
	enum{DEL_NUM = 3};
	SOCKET **p;
	SOCKET *delSock[DEL_NUM];
	int delNum = 0;
	uint currentTime = sys_time(NULL);
	int eflag;
	int i;
	
	// åȤǸ³
	newsc->waitTime = currentTime;
	newsc->newsc_next = NULL;
	eflag = enterCli();
	enter_spinlock(&sc->lockGate);
	{
		for (p = &sc->newsc_next;; p = &(*p)->newsc_next){
			if (*p == NULL){
				*p = newsc;
				break;
			}

			// ֿॢȤΥåȤ
			if ((LISTEN_WAIT_MAX < currentTime - (*p)->waitTime) && (refRecvBuf(sc->recvBuf) != NULL)){
				if (delNum < DEL_NUM){
					delSock[delNum] = *p;
					*p = (*p)->newsc_next;
					++delNum;
					if (*p == NULL){
						*p = newsc;
						break;
					}
				}
			}
		}
	}
	exit_spinlock(&sc->lockGate);
	exitCli(eflag);
	
	// ֿॢȥåȤ򥳥ͥ󥯤
	for (i = 0; i < delNum; ++i){
		delWaitLink(delSock[i]);
		kfree(delSock[i]);
	}
	
}


/*
 * åȤ
 * return :  error number
 */
STATIC int makeNewSocket(SOCKET *sock, struct mbuf *mbuf, SOCKET **o_newSock)
{
	SOCKET *newSock;
	IP_HEADER *ip;
	TCP_HEADER *tcp;
	OPTIONS optst;
	int error;

	error = makeSocket(sock->sockType, IPPROTO_TCP, &newSock);
	if (error < 0){
		return error;
	}

	ip = getMbufDataPointer(mbuf);;
	tcp = (TCP_HEADER*)ip->data;

	/* ץ(MSSΤ߼) */
	optst.mss = TCP_MSS_SIZE;
	getTcpOptions(tcp->option, getTcpDataSize(ip), &optst);

	/*
	 * åȤꡣ
	 */
	memcpy(&newSock->fs, &sock->fs, sizeof(SOCKET) - OFFSETOF(SOCKET, fs));
	newSock->dstip			= ip->srcip;
	newSock->dstport		= tcp->srcport;
	newSock->srcport		= tcp->dstport;
	newSock->sc				= newSock;
	newSock->acknum			= swapInt32(tcp->seqnum) + 1;
	newSock->seqnum			= getSeqNumber();
	newSock->prevAck		= 0;
	newSock->stat			= SOCK_LINK_ON;
	newSock->mss			= optst.mss;
	newSock->window			= swapWord(tcp->wndsize);
	newSock->sendBuf.buf	= (char*)0xa5a5a5a5;	// ١
	newSock->ringBuf		= (void*)0xa5a5a5a5;	// ١

	*o_newSock = newSock;
	
	return NOERR;
}


/*
 * listenԤ³
 */
STATIC void listenConnect(SOCKET *sock)
{
	struct mbuf *mbuf = getDetachRecvBuf(&sock->recvBuf, &sock->lockGate);
	IP_HEADER *ip;
	TCP_HEADER *tcp;
	MSS mss;
	SOCKET *newSock;
	
	ASSERT(mbuf != NULL);
	ip = getMbufDataPointer(mbuf);
	tcp = (TCP_HEADER*)ip->data;

	/* ³׵᥻Ȥγǧ */
	if ((tcp->flag & TCP_FLG_SYN) && (tcp->acknum == 0)){
		;
	}
	else{
		sendReset(ip);
		m_freem(mbuf);
		return;
	}

	/* åȤκ */
	if (makeNewSocket(sock, mbuf, &newSock) != NOERR){
		sendReset(ip);
		m_freem(mbuf);
		return;
	}
	m_freem(mbuf);
	
	/* ͥ󥯤³ */
	addWaitCnctLink(newSock);

	/* إåץꡣ */
	mss.kind=TCP_KIND_MSS;
	mss.len= 4;
	mss.mss= swapWord(TCP_MSS_SIZE);

	/* SYNȤ */
	transSegment(newSock, TCP_FLG_SYN|TCP_FLG_ACK, &mss, sizeof(mss), NULL, 0, TCP_WIN_SIZE);

	/* å󥽥åȤ³ */
	addNewSocket(sock, newSock);
}


// åɽλ롼
STATIC void listenRoutin(SOCKET *sock)
{
	sock->listenThread = get_current_task();
	
	/* ͥγ */
	addWaitLstnLink(sock);

	for (;;){
		if (refRecvBuf(sock->recvBuf) == NULL){
			/* ꡼ */
			sigWait();

			/* å֤ǤʤХåɽλ */
			if ((sock->stat & SOCK_LISTEN) == 0){
				break;
			}
		}
		else{
			listenConnect(sock);
		}
	}

	/*  */
	kfree(sock);
	sys_exit(0);
}


//================================== PROTECTED ==========================================


/*
 * ³ѿåȤƥåȤå󥽥åȤ
 * ǽԤϻ֤
 * return : ³ѿå or NULL
 */
STATIC SOCKET *getFromListenSocket(SOCKET *sc)
{
	SOCKET **p,*rest = NULL;
	int eflag;
	
	eflag = enterCli();
	enter_spinlock(&sc->lockGate);
	{
		for (p = &sc->newsc_next; *p != NULL; p = &(*p)->newsc_next){
			if (refRecvBuf((*p)->recvBuf) != NULL){
				rest = *p;
				*p = rest->newsc_next;
				rest->waitTime = 0;
				break;
			}
		}
	}
	exit_spinlock(&sc->lockGate);
	exitCli(eflag);
	
	return rest;
}


/*
 * å󥽥åȥ󥯤餹٤ƤΥåȤ
 */
STATIC void releaseListenSocket(SOCKET *sc)
{
	SOCKET *p,*q;
	
	for (p = sc->newsc_next; p != NULL; p = q){
		q = p->newsc_next;
		releaseRecvBuf(p->recvBuf);
		kfree(p);
	}
}


/*
 * å󥹥åɤκ
 * parameters : socket,file discripter
 * return     : 0 or error number
 */
STATIC int makeListenThread(SOCKET *sock, int fd)
{
	/* listenѥץư */
	switch(forkKernelThread())
    {
		case -1:
			return -ENOBUFS;
		case 0:
			// СåȤǤ褦
			// եǥץȥȤ򸺤餹
			sys_close(fd);

			listenRoutin(sock);
			// Not reached.
	}
	
	/* listenץưޤԤ */
	while (sock->next == sock){
		wait_task();
	}

	return 0;
}


/*
 * å󥹥åɤνλ
 * parameters : socket
 */
STATIC void endListenThread(SOCKET *sock)
{
	sock->stat &= ~SOCK_LISTEN;
	sys_wake(sock->listenThread);
}


/**************************************************************************
 *
 * å״ؿ
 *
 **************************************************************************/


//================================== PROTECTED ==========================================


// Хåե˻ĤäƤǡ롣
// return : 0 or error number
STATIC int sendRestData(SOCKET *sock, const uint8_t transFlag)
{
	enum{
		WAIT_TIME = 3,	// Ԥ
		RESEND_MAX = 3,	// 
	};
	size_t tcpAck;
	int sendBufLen;
	int c_resend;
	int endLoop;
	int rest;

	if (sock->sendBuf.curLen <= 0){
		return 0;
	}

	sendBufLen = SEND_BUF_SIZE - (getLastRingBuf(sock->ringBuf) - getStartRingBuf(sock->ringBuf));
	c_resend = 0;
	for(endLoop = 0; endLoop == 0;){
		transSegment(sock, transFlag, NULL, 0, sock->sendBuf.buf, sock->sendBuf.curLen, sendBufLen);
		sock->seqnum += sock->sendBuf.curLen + ((transFlag & TCP_FLG_FIN) == TCP_FLG_FIN);
		
		rest = recvSend(sock, WAIT_TIME, &tcpAck);
		switch (rest){
			case 0:
				// 
			case RECV_SEND_GET_DATA:
				if (sock->seqnum == tcpAck){
					sock->sendBuf.curLen = 0;
					return 0;
				}
				// 
			case -ETIMEDOUT:
				if (RESEND_MAX <= ++c_resend){
					endLoop = 1;
					rest = -ENOTCONN;
				}
				break;
			default:
				endLoop = 1;
		}

		sock->seqnum -= sock->sendBuf.curLen + ((transFlag & TCP_FLG_FIN) == TCP_FLG_FIN);
	}
	
	return rest;
}


/**************************************************************************
 *
 * ͥ
 *
 **************************************************************************/


enum{
	CNCT_CLOSE_WAIT_TIME = 5000,	/* ǽॢȻms */
	REPLY_TIMEOUT = 7,				/* ॢȥߥá */
};


//================================== PRIVATE ============================================


static SOCKET *finSocket;		/* 楽åȥ󥯡 */


/*
 * ǳϥåȤ󥯤³롣
 * parameters : socket
 */
STATIC void addCloseSocket(SOCKET *sock)
{
	sock->waitNext = finSocket;
	finSocket = sock;
	sock->waitTime = sys_time();
}


/*
 * ǳϥåȤǡॢȤõ
 */
STATIC void delCloseTimeout()
{
	uint time;
	SOCKET *p;


	time = sys_time();
	while (finSocket != NULL)
	{
		if (CNCT_CLOSE_WAIT_TIME <= time - finSocket->waitTime)
		{
			p = finSocket->waitNext;
			delWaitLink(finSocket);
			releaseRecvBuf(finSocket->recvBuf);
			kfree(finSocket);
			finSocket = p;
		}
		else
			break;
	}
}


//================================== PROTECTED ==========================================


/*
 * ͥ
 * parameters :
 */
STATIC void disconnect(SOCKET *sock)
{
	/* FINȤ */
	transSegment(sock, TCP_FLG_FIN | TCP_FLG_ACK, NULL, 0, NULL, 0, SEND_BUF_SIZE);
	sock->seqnum += 1;
	sock->stat |= SOCK_CNCT_TRANS_FIN;
	
	if (sock->stat | SOCK_DISCON_SEND){
		freeRingBuf(sock->ringBuf);
		kfree(sock->sendBuf.buf);
	}

	delCloseTimeout();		/* ॢȥåȤ롣 */
	addCloseSocket(sock);	/* åȥ󥯤³롣 */
}


/*
 * ͥΩ
 * parameters : socket
 * return : 0 or error number
 */
STATIC int connectTo(SOCKET *sock)
{
	struct mbuf *mbuf = NULL;
	IP_HEADER *ip;
	TCP_HEADER *tcp;
	OPTIONS optst;
	MSS mss;
	int error;

	/* ͥ󥯤³ */
	addWaitCnctLink(sock);

	/* إåץꡣ */
	mss.kind=TCP_KIND_MSS;
	mss.len= 4;
	mss.mss= swapWord(TCP_MSS_SIZE);

	/* SYNȤ */
	transSegment(sock, TCP_FLG_SYN, &mss, sizeof(mss), NULL, 0, TCP_WIN_SIZE);
	sock->seqnum += 1;

	/* Ȥμ */
	error = waitReceive(sock, REPLY_TIMEOUT);
	if (error != NOERR){
		goto ERR;
	}
	mbuf = getDetachRecvBuf(&sock->recvBuf, &sock->lockGate);
	ASSERT(mbuf != NULL);
	ip = getMbufDataPointer(mbuf);
	tcp = (TCP_HEADER*)ip->data;
	if (((tcp->flag & TCP_FLG_SYN) == 0) || (isAck(tcp, sock->seqnum, 0) != ACK_EQ_SEQ_GR)){
		if (tcp->flag & TCP_FLG_RST){
			return -ECONNREFUSED;
		}
		transReset(sock, tcp);
		error = -EAGAIN;
		goto ERR;
	}

	/* ץγǧ(MSSΤ߼) */
	optst.mss = TCP_MSS_SIZE;
	getTcpOptions(tcp->option, getTcpDataSize(ip), &optst);
	sock->mss = optst.mss;

	/* ǧ */
	sock->window = swapWord(tcp->wndsize);
	sock->acknum = swapInt32(tcp->seqnum) + 1;
	sock->stat |= SOCK_CNCT_ON;
	transSegment(sock, TCP_FLG_ACK, NULL, 0, NULL, 0, TCP_WIN_SIZE);

	sock->sendBuf.buf = kmalloc(sock->mss);
	if (sock->sendBuf.buf == NULL){
		error = -ENOMEM;
		goto ERR;
	}
	sock->sendBuf.curLen = 0;
	error = allocRingBuf(SEND_BUF_SIZE, sock->seqnum, &sock->ringBuf);
	if (error < 0){
		goto ERR;
	}

	m_freem(mbuf);

	return 0;
ERR:
	m_freem(mbuf);

	/* ͥ󥯤 */
	delWaitLink(sock);
	releaseRecvBuf(sock->recvBuf);

	return error;
}


/**************************************************************************
 *
 * TCPӥؿ
 *
 **************************************************************************/


//================================== PUBLIC =============================================


/*
 * IPåɤƤӽФ
 */
void doTcpService(SOCKET *sock, IP_HEADER *ip)
{
	if (sock == NULL){
		sendReset(ip);
	}
	else{
		ackFromCloseSock(sock, ip);
	}
}


/*
 * åȤ򥳥ͥ󤫤鳫롣
 * parameters : socket
 * return ; 0 or 1(åȤƤϤʤ)
 */
int releaseSockFromConect(SOCKET *sock)
{
	if (sock->stat & SOCK_CNCT_ON){
		int isFin = 0;

		// ĤäƤǡ
		sendRestData(sock, TCP_FLG_PSH | TCP_FLG_ACK);

		sock->stat |= SOCK_DISCON_RECV | SOCK_DISCON_SEND;

		// ѥåȤ˱
		while (refRecvBuf(sock->recvBuf) != NULL){
			struct mbuf *mbuf = getDetachRecvBuf(&sock->recvBuf, &sock->lockGate);
			IP_HEADER *ip;
			TCP_HEADER *tcp;

			ASSERT(mbuf != NULL);
			ip = getMbufDataPointer(mbuf);
			tcp = (TCP_HEADER*)ip->data;
			sock->acknum += getTcpDataSize(ip);
			isFin = tcp->flag & TCP_FLG_FIN;
			m_freem(mbuf);
		}
		if (isFin && ((sock->stat & SOCK_CNCT_RECV_FIN) == 0)){
			sendFinAck(sock);
		}
		
		if ((sock->stat & SOCK_CNCT_TRANS_FIN) == 0){
			disconnect(sock);
		}

		return 1;
	}

	if (sock->stat & SOCK_LISTEN){
		delWaitLink(sock);
		releaseRecvBuf(sock->recvBuf);
		releaseListenSocket(sock);
		endListenThread(sock);

		return 1;
	}

	if (sock->stat & SOCK_LINK_ON){
		delWaitLink(sock);
		releaseRecvBuf(sock->recvBuf);
	}

	return 0;
}


/**************************************************************************
 *
 * åȥӥؿ
 *
 **************************************************************************/


STATIC int send(SOCKET *sock, void *msg, const size_t i_size, int flags, struct sockaddr *to)
{
	enum{
		RESEND_MAX = 3,			// 
	};
	
	// 롼ǡ
	size_t unsentSize = i_size;	// 롼׽Ȥ̤
	char *sendBuf = msg;		// ǡ
	int window = sock->window;	// 롼׽Ȥɥ
	int sendLen;				// 롼׽Ȥ
	int c_resend = 0;			// 
	int timeout = 0;			// Ԥ֡á

	// ХåեϥͥȻ˳ƤƤϤ
	ASSERT(sock->ringBuf != NULL);
	
	if ((sock->stat & SOCK_CNCT_ON) == 0){
		return -ENOTCONN;
	}
	
	for (; 0 < unsentSize; unsentSize -= sendLen){
		int rest;			// ؿ
		size_t tcpAck = 0;	// ֹ
		int sendBufLen;		// ¸Хåեζ
		size_t sendStart;	// ¸Хåեϰ

		// ǡ¸Хåե¸롣
		// Хåեޤɥ0ˤʤä
		// 򥿥ॢԤäơʤϥХåեκǽ餫롣

		// γǧ
		rest = recvSend(sock, timeout, &tcpAck);
		switch (rest){
			case 0:
				if (getStartRingBuf(sock->ringBuf) < tcpAck){
					setStartRingBuf(sock->ringBuf, tcpAck);
					sendStart = getCurrentRingBuf(sock->ringBuf);
					c_resend = 0;
				}
				else if (tcpAck == getStartRingBuf(sock->ringBuf)){
					// ѥåȥȤ򤷤Ƥ
					window = sock->window;
					sendStart = tcpAck;
					c_resend += 1;
				}
				else{
					sendStart = getCurrentRingBuf(sock->ringBuf);
				}
				break;
			case -ETIMEDOUT:
			{
				// ХåեλϤᤫ
				window = sock->window;
				sendStart = getStartRingBuf(sock->ringBuf);
				c_resend += 1;
			}
				break;
			case RECV_SEND_GET_DATA:
				return i_size - unsentSize;
			default:
				return rest;
		}

		// ɥʲ¸Хåեζʲ
		sendBufLen = getAvailableSizeRingBuf(sock->ringBuf);
		sendLen = (unsentSize < window)? unsentSize : window;
		sendLen = (sendLen < sendBufLen)? sendLen : sendBufLen;
		if (0 < sendLen){
			// ¸Хåե˳Ǽ
			writeRingBuf(sock->ringBuf, sendLen, sendBuf);
			sendBuf += sendLen;
			timeout = 0;
		}
		else{
			if (RESEND_MAX <= c_resend){
				return -ENOTCONN;
			}
			timeout = 1;	// Ԥॢ
		}

		// ¸Хåե
		window -= sendFromBuf(sock, sendStart, TCP_FLG_PSH | TCP_FLG_ACK);
	}
	
	return i_size;
}


/*
 * recv()Υǥơ֥
 * FINե饰ͭǥǡͭ			   0 0   0 0     0 0   0 0       1 1   1 1     1 1   1 1       2 2   2 2     2 2   2 2
 * 		 桼ХåեäѤ			   0 0   0 0     1 1   1 1       0 0   0 0     1 1   1 1       0 0   0 0     1 1   1 1
 * 		 ǡ桼Ϥ			   0 0   1 1     0 0   1 1       0 0   1 1     0 0   1 1       0 0   1 1     0 0   1 1
 * 		 ΥǡƤ뤫		   0 1   0 1     0 1   0 1       0 1   0 1     0 1   0 1       0 1   0 1     0 1   0 1
 * 
 * ̡							   0     1       0     1         0     0       0     0         0     1       1     1  
 *      ʺƱѥåȤɬפȤʤ
 * 		 λ								   0 0   1 0     1 1   1 1       0 0   1 1     1 1   1 1       0 0   1 1     1 1   1 1
 *      ʼλ桼Хåե
 */
static const char isSend[3][2][2]		 = {{{ 0,    1   },{ 0,    1   }},{{ 0,    0   },{ 0,     0  }},{{ 0,    1   },{ 1,    1   }}};
static const char isReturn[3][2][2][2]	 = {{{{0,0},{1,0}},{{1,1},{1,1}}},{{{0,0},{1,1}},{{1,1},{1,1}}},{{{0,0},{1,1}},{{1,1},{1,1}}}};

STATIC int recv(SOCKET *sock, struct msghdr *msg)
{
	enum{
		RECEIVE_TIMEOUT = 5,	// ॢ
	};
	char *data;
	int allSize, tcpDataSize, dataSize, cpSize, iovArrey, iovOffset;
	int rest;

	// ͥγǧ
	if ((sock->stat & SOCK_CNCT_ON) == 0){
		return -ENOTCONN;
	}

	// ǤFINȤƤ뤫
	if (sock->stat & SOCK_CNCT_RECV_FIN){
		return 0;
	}
	
	// ХåեλĤ
	rest = sendRestData(sock, TCP_FLG_PSH | TCP_FLG_ACK);
	if (rest < 0){
		return rest;
	}

	allSize = iovArrey = iovOffset = 0;
	for (;;){
		struct mbuf *mbuf;
		IP_HEADER *ip;
		TCP_HEADER *tcp;
		int isFin, isFullUserBuf, isAllDataCopy;

		// 
		rest = recvData(sock, RECEIVE_TIMEOUT);
		if (rest != NOERR){
			return rest;
		}
		mbuf = refRecvBuf(sock->recvBuf);
		ASSERT(mbuf != NULL);
		ip = getMbufDataPointer(mbuf);
		tcp = (TCP_HEADER*)ip->data;

		// 桼Хåե˥ԡ
		tcpDataSize = getTcpDataSize(ip);
		dataSize = tcpDataSize - mbuf->mh_offset;
		data = (char*)tcp + getTcpSize(ip) - dataSize;
		cpSize = copyIovec(data, dataSize, msg->msg_iov, msg->msg_iovlen, &iovArrey, &iovOffset);
		allSize += cpSize;
		mbuf->mh_offset += cpSize;

		//FINե饰Ƥ뤫
		if (tcp->flag & TCP_FLG_FIN){
			isFin = (0 < dataSize)? 1 : 2;
		}
		else{
			isFin = 0;
		}
		// 桼ХåեäѤ
		isFullUserBuf = (iovArrey == msg->msg_iovlen);
		// ǡ桼Ϥ
		isAllDataCopy = (mbuf->mh_offset == tcpDataSize);

		ASSERT((isFullUserBuf == 1) || (isAllDataCopy == 1));

		if (isSend[isFin][isFullUserBuf][isAllDataCopy]){
			uint8_t  flag = tcp->flag;
			
			// IPХåեγ
			detachRecvBuf(mbuf, &sock->recvBuf, &sock->lockGate);
			m_freem(mbuf);

			// 
			sock->acknum += tcpDataSize;
			if ((flag & (TCP_FLG_PSH | TCP_FLG_FIN)) || (refRecvBuf(sock->recvBuf) == NULL)){
				if (flag & TCP_FLG_FIN){
					sendFinAck(sock);
				}
				else{
					transSegment(sock, TCP_FLG_ACK, NULL, 0, NULL, 0, TCP_WIN_SIZE);
				}
			}
		}
		if (isReturn[isFin][isFullUserBuf][isAllDataCopy][refRecvBuf(sock->recvBuf) != NULL]){
			// λ
			break;
		}
	}

	return allSize;
}


STATIC int shutdown(SOCKET *sock,int flag)
{
	if ((sock->stat & SOCK_CNCT_ON) == 0)
		return -ENOTCONN;

	switch(flag){
		case SHUT_RD:
			sock->stat |= SOCK_DISCON_RECV;
			break;
		case SHUT_WR:
			sock->stat |= SOCK_DISCON_SEND;
			break;
		case SHUT_RDWR:
			sock->stat |= SOCK_DISCON_RECV | SOCK_DISCON_SEND;
			break;
	}

	if ((sock->stat & (SOCK_DISCON_RECV | SOCK_DISCON_SEND)) && (sock->stat & SOCK_CNCT_TRANS_FIN) == 0){
		disconnect(sock);
	}

	return 0;
}


STATIC int connect(SOCKET *sock)
{
	if (sock->stat & SOCK_CNCT_ON)
		return -EISCONN;

	/* Set connect structure. */
	sock->next = sock->prev = sock;
	sock->stat = SOCK_LINK_ON;
	sock->seqnum = getSeqNumber();
	sock->acknum = 0;

	/* ͥγΩ */
	return connectTo(sock);
}


STATIC int listen(SOCKET *sock,int fd)
{
	if (sock->stat & SOCK_LISTEN)
		return -EADDRINUSE;

	if (searchListenSocket(sock->srcport) != NULL)
		return -EADDRINUSE;

	sock->next = sock->prev = sock;
	sock->stat = SOCK_LINK_ON | SOCK_LISTEN;
	sock->waitProc = get_current_task();
	
	/* å󥹥åɤγ */
	makeListenThread(sock,fd);
	
	return 0;
}


STATIC int accept(SOCKET *sock,SOCKET **o_newSock,uint32_t *o_srcIp)
{
	SOCKET *newSock;
	struct mbuf *mbuf;
	int rest;
	
	for (;;){
		for (;;){
			/* ʥߤ */
			doSignalRestart();
			if (isSigint() == TRUE){
				return -EINTR;
			}

			newSock = getFromListenSocket(sock);
			if (newSock != NULL){
				break;
			}

			/* Ԥ */
			sigWait();
			newSock = getFromListenSocket(sock);
			if (newSock != NULL){
				break;
			}
		}
		mbuf = refRecvBuf(newSock->recvBuf);

		for (;;){
			IP_HEADER *ip = getMbufDataPointer(mbuf);
			TCP_HEADER *tcp = (TCP_HEADER*)ip->data;

			/* γǧ */
			switch (isAck(tcp, newSock->seqnum, newSock->acknum)){
				case -1:
					/*  */
				case ACK_LW_SEQ_EQ:
					detachRecvBuf(mbuf, &newSock->recvBuf, &newSock->lockGate);
					m_freem(mbuf);
					mbuf = refRecvBuf(newSock->recvBuf);
					if (mbuf == NULL){
						kfree(newSock);
						goto NEXT;
					}
					break;
				case ACK_EQ_SEQ_GR:
					// ȤΥֹ椬礭ȤϡΥȤ夷ǽ
					goto NEXT;
					break;
				default:
					/* ׵ */
					if (tcp->flag & TCP_FLG_RST){
						releaseRecvBuf(newSock->recvBuf);
						kfree(newSock);
						goto NEXT;
					}
					else{
						goto OUT;
					}
			}
		}
NEXT:	;
	}
OUT:
	detachRecvBuf(mbuf, &newSock->recvBuf, &newSock->lockGate);
	m_freem(mbuf);

	newSock->stat |= SOCK_CNCT_ON;
	newSock->seqnum += 1;

	newSock->sendBuf.buf = kmalloc(newSock->mss);
	if (newSock->sendBuf.buf == NULL){
		return -ENOMEM;
	}
	newSock->sendBuf.curLen = 0;
	rest = allocRingBuf(SEND_BUF_SIZE, newSock->seqnum, &newSock->ringBuf);
	if (rest < 0){
		return rest;
	}

	*o_srcIp = newSock->dstip;
	*o_newSock = newSock;

	return 0;
}


STATIC int poll(SOCKET *sock)
{
	// ͥγǧ 
	if (((sock->stat & SOCK_CNCT_ON) == 0) && ((sock->stat & SOCK_LISTEN) == 0)){
		return -ENOTCONN;
	}

	// ХåեλĤ
	sendRestData(sock, TCP_FLG_PSH | TCP_FLG_ACK);

	if (sock->stat & SOCK_LISTEN){
		if (sock->newsc_next != NULL){
			return POLLIN | POLLRDNORM | POLLOUT | POLLWRNORM;
		}
	}
	else{
		if (refRecvBuf(sock->recvBuf) != NULL){
			return POLLIN | POLLRDNORM | POLLOUT | POLLWRNORM;
		}
	}

	sock->waitProc = get_current_task();

	return POLLOUT | POLLWRNORM;
}


/**************************************************************************
 *
 * 
 *
 **************************************************************************/


static SOCKET_INFO sockInfo = {send, recv, connect, shutdown, listen, accept, poll};


//================================== PUBLIC =============================================


// Init TCP.
int initTcp()
{
	registSocket(&sockInfo,IPPROTO_TCP);
	initReceive();

	return 0;
}

/***************************************************************************/
void test_tcp()
{
	printk("waitNext=%x\n",*(uint*)0x16be64);
}
/****************************************************************************/
