/*
 * ether.c
 *
 * Copyright 2004, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 *
 * Ethernet controller manager.
 */


#include <sys/config.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/mbuf.h>
#include <sys/sockio.h>
#include <sys/errno.h>
#include <net/net.h>
#include <net/if_var.h>
#include <net/if_types.h>
#include <net/ethernet.h>
#include <net/ip.h>
#include <net/arp.h>
#include <kern/lib.h>
#include <kern/device.h>
#include <kern/buf.h>
#include <dev/console/console.h>


//#define DEBUG_NET_ETHER 1
#ifdef DEBUG_NET_ETHER
	#define STATIC
	#define INLINE
#else
	#define STATIC static
	#define INLINE	inline
#endif


//================================== PRIVATE ============================================


enum{
	ETHER_DEVICE_MAX = 2,		// ETHER_DEVICE¤γǼ¿
};

/* Ethernet device infomation. */
typedef struct{
	struct ifnet	*ifnet;
	void			*sc;
	uint32_t		ip;			/* IP address. */
	uint32_t		netmask;	/* ֥ͥåȥޥ */
	uint32_t		subnet;		/* ֥ͥåȡ */
	int				mtu;		/* MTU size. */
	char			mac[6];		/* MAC address. */
}ETHER_DEVICE;


static ETHER_DEVICE ethDev[ETHER_DEVICE_MAX];
static int etherCount;					// ͥåȥǥХ


//================================== PUBLIC =============================================


// ͥåȥǥХֹ
// return : ͥåȥǥХֹ
int getEtherNum(
	const uint32_t dstIp)		// ɣ
{
	int i;

	for (i = 0; i < etherCount; ++i){
		if ((dstIp & ethDev[i].netmask) == ethDev[i].subnet){
			return i;
		}
	}
	/* ǥեȥȥء */
	for (i = 0; i < etherCount; ++i){
		if ((IP_DEFAULT_GATEWAY & ethDev[i].netmask) == ethDev[i].subnet){
			return i;
		}
	}

	printk("Do not match net mask of default gateway!\n");
	return 0;
}


// macɥ쥹μ
void getEtherMac(
	const int etheNum,			// ͥåȥǥХֹ
	char *m_mac)				// macɥ쥹ԡХåե
{
	ASSERT(etheNum < etherCount);

	memcpy(m_mac, ethDev[etheNum].mac, 6);
}


// ipɥ쥹μ
// return : IP
uint32_t getEtherIp(
	const int etheNum)			// ͥåȥǥХֹ
{
	ASSERT(etheNum < etherCount);

	return ethDev[etheNum].ip;
}


// ifnet¤Υɥ쥹μ
// return : ifnet¤Υɥ쥹
struct ifnet *getEtherIfnet(
	const int etheNum)			// ͥåȥǥХֹ
{
	ASSERT(etheNum < etherCount);

	return ethDev[etheNum].ifnet;
}


// MTUμ
// return : MTU
int getEtherMtu(
	const int etheNum)			// ͥåȥǥХֹ
{
	ASSERT(etheNum < etherCount);

	return ethDev[etheNum].mtu;
}


// IPɥ쥹μ
// return : IP
uint32_t getTransIp(
	const uint32_t dstIp,		// IP
	const int etheNum)			// ͥåȥǥХֹ
{
	ASSERT(etheNum < etherCount);

	if ((dstIp & ethDev[etheNum].netmask) == ethDev[etheNum].subnet){
		return dstIp;
	}
	if ((IP_DEFAULT_GATEWAY & ethDev[etheNum].netmask) == ethDev[etheNum].subnet){
		return IP_DEFAULT_GATEWAY;
	}
	
	// IPΥ֥ͥåȥޥȰפ륤ͥåȥǥХֹƤϤ
	ASSERT(0);
	return 0;
}


// 
// IPե졼˥ͥåȥإåդѥåȤämbuf¸ɥ饤СϤ
void transEther(
	struct ifnet *ifnet,			// ifnet¤
	TRANS_BUF_INFO *transBuf,		// Хåե¤
	const char *dstMac)				// MACɥ쥹
{
	int packetSize;
	int allocSize;
	struct ether_header *etheHead;
	struct mbuf *mbuf;

	// ͥåȥѥåȥ
	packetSize = ETHER_HDR_LEN + transBuf->size[0] + transBuf->size[1] + transBuf->size[2];
		if (ETHER_MIN_LEN - ETHER_CRC_LEN <= packetSize){
		allocSize = packetSize + ETHER_CRC_LEN;
	}
	else{
		allocSize = ETHER_MIN_LEN;
	}

	// mbufγ
	mbuf = mallocTrans(allocSize);
	if (mbuf == NULL){
		printk("transEther() error! : mbuf allocate error\n");
		return;
	}

	// ͥåȥإå
	etheHead = (struct ether_header*)mbuf->mh_data;
	getEtherMac(ifnet->if_unit, etheHead->ether_shost);
	memcpy(etheHead->ether_dhost, dstMac, ETHER_ADDR_LEN);
	etheHead->ether_type = swapWord(transBuf->type);

	// ե졼Υԡ
	memcpy(mbuf->mh_data + ETHER_HDR_LEN, transBuf->data[0], transBuf->size[0]);
	memcpy(mbuf->mh_data + ETHER_HDR_LEN + transBuf->size[0], transBuf->data[1], transBuf->size[1]);
	memcpy(mbuf->mh_data + ETHER_HDR_LEN + transBuf->size[0] + transBuf->size[1], transBuf->data[2], transBuf->size[2]);
	mbuf->mh_pkthdr.len = packetSize;

	// ɥ饤СϤ
	IF_PREPEND(&ifnet->if_snd, mbuf);
	ifnet->if_start(ifnet);

	// mbufγߥϥɥλŻ
}


/*
 * Ethernet output routine.
 * Encapsulate a packet of type family for the local net.
 * Use trailer local net encapsulation if enough data in first
 * packet leaves a multiple of 512 bytes of data in remainder.
 * Assumes that ifp is actually pointer to arpcom structure.
 */
int ether_output(
	struct ifnet *ifp,
	struct mbuf *m,
	struct sockaddr *dst,
	struct rtentry *rt0)
{
	return 0;
}


// return : error number
int ether_ioctl(ifp, command, data)
	struct ifnet *ifp;
	int command;
	caddr_t data;
{
	struct ifaddr *ifa = (struct ifaddr *) data;
	int error = NOERR;

	switch (command) {
		case SIOCSIFADDR:	// 󥿡եɥ쥹
			switch (ifa->ifa_addr->sa_family) {
				case AF_INET:
					break;
				default:
					break;
			}
			break;
		case SIOCGIFADDR:	// 󥿡ե̾ץȥ륢ɥ쥹
			break;
		case SIOCSIFMTU:
			break;
	}
	
	return error;
}


#ifdef DEVICE_POLLING

/*
 * Try to register routine for polling. Returns 1 if successful
 * (and polling should be enabled), 0 otherwise.
 * A device is not supposed to register itself multiple times.
 *
 * This is called from within the *_intr() functions, so we do not need
 * further locking.
 */
int ether_poll_register(poll_handler_t *h, struct ifnet *ifp)
{
	return 0;
}


/*
 * Remove interface from the polling list. Normally called by *_stop().
 * It is not an error to call it with IFF_POLLING clear, the call is
 * sufficiently rare to be preferable to save the space for the extra
 * test in each driver in exchange of one additional function call.
 */
int ether_poll_deregister(struct ifnet *ifp)
{
	return 0;
}

#endif


/*
 * Process a received Ethernet packet. We have two different interfaces:
 * one (conventional) assumes the packet in the mbuf, with the ethernet
 * header provided separately in *eh. The second one (new) has everything
 * in the mbuf, and we can tell it because eh == NULL.
 * The caller MUST MAKE SURE that there are at least
 * sizeof(struct ether_header) bytes in the first mbuf.
 */
void ether_input(
	struct ifnet *ifp, 
	struct ether_header *eh, 
	struct mbuf *mbuf)
{
	if (eh == NULL) {
		if (mbuf->mh_len < sizeof(struct ether_header)) {
			return;
		}
		eh = mtod(mbuf, struct ether_header *);
		m_adj(mbuf, sizeof(*eh));
	}

	switch (ntohs(eh->ether_type)) {
	case ETHERTYPE_IP:
		receiveIp(mbuf);
		break;
	case ETHERTYPE_ARP:
		receiveArp(ifp->if_unit, mtod(mbuf, ARP_HEADER*));
		break;
	default:
		break;
	}
}


// ͥåȥǥХ
// ǥХɥ饤Сattach᥽åɤƤФ
// return : error number.
int setEtherDevice(struct ifnet *ifnet, const char *mac)
{
	if (ETHER_DEVICE_MAX <= etherCount){
		return -E2BIG;
	}

	ifnet->if_unit = etherCount;
	memset(&ifnet->if_snd, 0, sizeof(ifnet->if_snd));
	ethDev[etherCount].ifnet = ifnet;
	memcpy(ethDev[etherCount].mac, mac, 6);

	// IPɥ쥹ꡣ
	//ԸƤվѹɬ
	switch(etherCount){
	case 0:
		ethDev[0].ip = IP_ADDRESS0;
		ethDev[0].netmask = IP_SUBNETMASK0;
		ethDev[0].subnet = IP_ADDRESS0 & IP_SUBNETMASK0;
		ethDev[0].mtu = 1500;
		break;
	case 1:
		ethDev[1].ip = IP_ADDRESS1;
		ethDev[1].netmask = IP_SUBNETMASK1;
		ethDev[1].subnet = IP_ADDRESS1 & IP_SUBNETMASK1;
		ethDev[1].mtu = 1500;
		break;
	}
	
	++etherCount;

	return NOERR;
}


/*
 * Perform common duties while detaching an Ethernet interface
 */
void ether_ifdetach(struct ifnet *ifp, int bpf)
{
	// ̤
}


// ͥåȤν
// return : error number
int initEther()
{
	extern int if_rl_pciInitDevice();
	extern int miibus_rlInitDevice();
	extern int if_fxp_pciInitDevice();
	extern int miibus_fxpInitDevice();
	int (*nic[])() = {
		if_rl_pciInitDevice,
		if_fxp_pciInitDevice
	};
	int (*mii[])() = {
		miibus_rlInitDevice,
		miibus_fxpInitDevice
	};
	int i;

	// ǥХɥ饤СϿ
	for (i = 0; i < ARRAY_LENGTH(nic); i++){
		if (nic[i]() == 0){
			mii[i]();
		}
	}

	// ǥХɥ饤Сν
	for (i = 0; i < etherCount; ++i){
		struct ifnet *ifnet = ethDev[i].ifnet;
		ifnet->if_init(ifnet->if_softc);
	}

	return NOERR;
}
