/*
 * time.c
 *
 * Copyright 2002, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 */


#include <sys/types.h>
#include <sys/param.h>
#include <sys/time.h>
#include <sys/limits.h>
#include <sys/systm.h>
#include <sys/buf.h>
#include <kern/lib.h>
#include <kern/errno.h>
#include <kern/interrupt.h>
#include <kern/proc.h>
#include <kern/sp.h>
#include <kern/mp.h>
#include <kern/lock.h>
#include <kern/mm.h>
#include <kern/signal.h>
#include <kern/list.h>
#include <kern/time.h>
#include <kern/debug.h>
#include <kern/test.h>
#include <dev/8254.h>
#include <dev/MC146818.h>


//#define DEBUG_TIME 1
#ifdef DEBUG_TIME
	#define STATIC
	#define INLINE
#else
	#define STATIC static
	#define INLINE inline
#endif


/***************************************************************************************************
 *
 * ִ
 *
 ***************************************************************************************************/

//================================== PRIVATE ============================================

static uint unixTime;		// UNIX TIME.(number of seconds since 00:00:00.1.1.1970).
static uint64 crntRdtsc;	// ߤRDTSC
static uint clock_1micro;	// cpu clock/s
static uint clock_1ms;		// cpu clock/ms

//FreeBSD
STATIC void timevalfix(struct timeval *t1)
{
	if (t1->tv_usec < 0) {
		t1->tv_sec--;
		t1->tv_usec += 1000000;
	}
	if (t1->tv_usec >= 1000000) {
		t1->tv_sec++;
		t1->tv_usec -= 1000000;
	}
}

/*
 * ֤ǤΥå¬
 */
STATIC void calcTimeClock(void)
{
	uint clock = getCpuClock();

	clock_1micro = clock / 1000000 + 1;		/* 1micros */
	clock_1ms = clock / 1000 + 1;			/* 1ms */
}

//================================== PUBLIC =============================================

/*
 * UNIX TIME롣
 */
uint getUnixTime(void)
{
	return unixTime;
}

/*
 * UNIX TIMEꤹ롣
 */
void setUnixTime(
	uint seconds)
{
	if (seconds != unixTime) {
		unixTime = seconds;
		crntRdtsc = rdtsc();
	}
}

void gettimeofday(struct timeval *m_tp)
{
	m_tp->tv_sec = unixTime;
	m_tp->tv_usec = (rdtsc() - crntRdtsc) / clock_1micro;
	if (1000000 <= m_tp->tv_usec){
		m_tp->tv_sec += 1;
		m_tp->tv_usec -= 1000000;
	}
}

void getmicrouptime(struct timeval *m_tp)
{
	uint64 current = rdtsc();
	
	m_tp->tv_sec = current / (clock_1micro * 1000000);
	m_tp->tv_usec = (current % (clock_1micro * 1000000)) / clock_1micro;
}

void microtime(struct timeval *tv)
{
	gettimeofday(tv);
}

/*
 *FreeBSD
 * Add and subtract routines for timevals.
 * N.B.: subtract routine doesn't deal with
 * results which are before the beginning,
 * it just gets very confused in this case.
 * Caveat emptor.
 */
void timevaladd(struct timeval *t1, struct timeval *t2)
{
	t1->tv_sec += t2->tv_sec;
	t1->tv_usec += t2->tv_usec;
	timevalfix(t1);
}

/*
 *FreeBSD
 */
void timevalsub(struct timeval *t1, struct timeval *t2)
{
	t1->tv_sec -= t2->tv_sec;
  	t1->tv_usec -= t2->tv_usec;
	timevalfix(t1);
}

/*
 * ưηвä
 */
time_t time_second()
{
	return rdtsc() / clock_1ms / 1000;
}

/*
 * cpuå¬
 * ֤ : å(clock/s)
 */
uint getCpuClock(void)
{
	uint clock;
	long long r_beg,r_end;
	double time;

 	setClockSecond();

 	r_beg = rdtsc();	/* clock counter */
	for (;;) {
		if (0x400000 < rdtsc() - r_beg){	/* ٱ */
			break;
		}
	}
	r_end = rdtsc();

	time = getClockSecond();

	clock = (r_end - r_beg) / time;

	return clock;
}

/*
 * micro seccond ޡ
 */
void micro_timer(
	uint count)		// ¬(s)
{
	uint64 end;


	end = rdtsc();
	end += count * clock_1micro;
	while (rdtsc() < end);
}

/*
 * mili seccond timer
 */
void mili_timer(
	uint count)		// ¬(ms)
{
	uint64 end;


	end = rdtsc();
	end += count * clock_1ms;
	while (rdtsc() < end);
}

/*
 * Υåͤвᤷ֤
 * return : в(ms)
 */
uint getTimeFromClock(
	uint64 begin)		// clock
{
	return (rdtsc() - begin) / clock_1ms;
}

/*
 * å֤ͤ׻
 * return : ׻(ms)
 */
uint calcTimeFromClock(
	uint64 clock)		// clock
{
	return clock / clock_1ms;
}

/***************************************************************************************************
 *
 * ޡ
 *
 ***************************************************************************************************/

//================================== PROTECTED ==========================================

typedef struct TIMER{
	OBJ_LIST	list;
	OBJ_LIST	parentList;		// ƤΥ
	int			isDelete;		// Ѥ YES or NO
	uint		init;			// ޡ
	uint		count;			// Ĥ
	struct TIMER *link;			// ³
	OBJ_LIST **parent;			// ƥإå
	void (*handle)(void*);		// ϥɥ
	void		*arg;			// ϥɥ
} TIMER;

enum{
	TIMER_MAX = 10,			// ޡ
};

static int timerCount;
static OBJ_LIST *systemTimerHead;	// ƥݻޡإå
static TIMER intervalTimer;			// Interval timer link. */
static TIMER realTimer;				// Real timer link queue */
static int timerLock;				// ޡԥå

/*
 *Գ
 * 顼ϥɥ
 */
STATIC void handleAlarm(void *proc)
{
	sendSignal(proc, SIGALRM);
}

/*
 *Գ
 * ϥɥ
 */
STATIC void handleWake(void *proc)
{
	forceSendSignal(proc, SIGWAKE, TASK_SIGNAL_WAIT);
}

/*
 * ϥɥ
 */
STATIC void handleDevWake(void *proc)
{
	forceSendSignal(proc, SIGWAKE, TASK_SIGNAL_WAIT | TASK_DEVICE_WAIT);
}

/*
 *Գ
 * 󥰥ϥɥ
 */
STATIC void handleTasking(
	void *dummy)		// ̤
{
	// åԤ
	setIntervalTasking(YES);
/*******************************************/
	{
		static char value = '0';

		if ('9' < value){
			value = '0';
		}
		displayTop(60, &value, 1);
		++value;
	}
/********************************************/
}

/*
 * ޡƤ
 * return : error number
 */
STATIC int allocTimer(TIMER **o_timer)
{
	TIMER *timer;

	if (TIMER_MAX <= timerCount){
		// ɬƤ褦Ĵɬ
		WARNING(0);
		return -ENOMEM;
	}
	++timerCount;
/****************************************************************************************************************/
printDebug(80 * 1+ 70, "     %d", timerCount);
/****************************************************************************************************************/
	timer = kmalloc(sizeof(*timer));
	if (timer == NULL){
		return -ENOMEM;
	}
	memset(timer, 0, sizeof(*timer));
	*o_timer = timer;
	
	return NOERR;
}

/*
 * ޡ
 */
STATIC void freeTimer(TIMER *timer)
{
	--timerCount;
/****************************************************************************************************************/
printDebug(80 * 1+ 70, "     %d", timerCount);
/****************************************************************************************************************/
	ASSERT(0 <= timerCount);

	kfree(timer);
}

/*
 *Թ
 * ޡֽ³롣
 */
STATIC void linkTimer(
	const uint i_time,		// 
	TIMER *head,			// ³إå
	TIMER *m_timer)			// ³륿ޡ
{
	uint time = i_time;
	TIMER *nextTimer;
	
	ASSERT(head != NULL);
	ASSERT(m_timer != NULL);

	nextTimer = getNextList(OFFSETOF(TIMER, list), &head->list);
	for (; nextTimer != head; nextTimer = getNextList(OFFSETOF(TIMER, list), &nextTimer->list)){
		if (time < nextTimer->count){
			nextTimer->count -= time;
			break;
		}
		else{
			time -= nextTimer->count;
		}
	}
	addPrevList(&nextTimer->list, &m_timer->list);	// Фꤹ
	m_timer->count = time;
}

/*
 *Թ
 * ޡإå󥯤ڤΥ
 */
STATIC void unlinkTimer(TIMER *m_timer)
{
	TIMER *nextTimer;

	ASSERT(m_timer != NULL);

	nextTimer = getNextList(OFFSETOF(TIMER, list), &m_timer->list);
	nextTimer->count += m_timer->count;
	removeList(&m_timer->list);
}

/*
 * ơʽͭ˥إå˥󥯤롣
 */
STATIC void linkParent(OBJ_LIST **m_head, TIMER *src)
{
	addHeadPrevList(m_head, &src->parentList);
}

/*
 * ơʽͭ˥إå󥯤Ϥ
 */
STATIC void unlinkParent(OBJ_LIST **m_head, TIMER *src)
{
	removeFromHeadList(m_head, &src->parentList);
}

/*
 * ơʽͭ˥إå󥯤ϥɥõ
 * return : error number
 */
STATIC int findTimerByHandle(const OBJ_LIST *head, void (*handle)(void*), TIMER **o_timer)
{
	TIMER *headTimer = getHeadObj(OFFSETOF(TIMER, parentList), head);

	if (headTimer != NULL) {
		TIMER *timer = headTimer;;
		
		do {
			if (timer->handle == handle) {
				*o_timer = timer;
				return NOERR;
			}
			timer = getNextList(OFFSETOF(TIMER, parentList), &timer->parentList);
		} while (timer != headTimer);
	}
	return -ENOENT;
}

/*
 * ơʽͭ˥إå󥯤ѥޡõ
 * return : error number
 */
STATIC int findDelTimer(const OBJ_LIST *head, TIMER **o_timer)
{
	TIMER *headTimer = getHeadObj(OFFSETOF(TIMER, parentList), head);

	if (headTimer != NULL) {
		TIMER *timer = headTimer;
		
		do {
			if (timer->isDelete == YES) {
				*o_timer = timer;
				return NOERR;
			}
			timer = getNextList(OFFSETOF(TIMER, parentList), &timer->parentList);
		} while (timer != headTimer);
	}
	return -ENOENT;
}

/*
 *ադδؿɬ delFromSchedule() θǻȤȡ⤷˥ॢȤȥץʤ롣
 *Թ
 * return : error number
 */
STATIC int setProcessTimer(
	const uint i_time,
	const uint initTime,
	void (*handle)(void*),
	void *arg,
	TIMER **o_timer)
{
	TIMER  *timer;
	int error;

/****************************************************************************************************************/
//displayTop(80 * 5 + 70, "1         ", 10);
/****************************************************************************************************************/
	ASSERT(0 < i_time);
	ASSERT(handle != NULL);

	// ץͭޡʤƱϥɥ˾񤭤롣
	if (findTimerByHandle(get_current_task()->procTimerHead, handle, &timer) == NOERR){
		unlinkTimer(timer);
	}
	else{
		error = allocTimer(&timer);
		if (error != NOERR) {
			return error;
		}
		linkParent(&get_current_task()->procTimerHead, timer);
	}

	// ³󥯸
	if (i_time < REAL_TIMER_RATE){
		linkTimer(setIntervalTimer(i_time), &intervalTimer, timer);
		timer->link = &intervalTimer;
	}
	else{
		linkTimer(i_time, &realTimer, timer);
		timer->link = &realTimer;
	}
	timer->isDelete = NO;
	timer->init = initTime;
	timer->parent = &get_current_task()->procTimerHead;
	timer->handle = handle;
	timer->arg = arg;
	*o_timer = timer;
/****************************************************************************************************************/
//displayTop(80 * 5 + 70, "2         ", 10);
/****************************************************************************************************************/

	return NOERR;
}

/*
 *ադδؿɬ delFromSchedule() θǻȤȡ⤷˥ॢȤȥץʤ롣
 *Թ
 * return : error number
 */
STATIC int setSystemTimer(
	const uint i_time,
	const uint initTime,
	void (*handle)(),
	void *arg,
	TIMER **o_timer)
{
	TIMER  *timer;
	int error;

/****************************************************************************************************************/
//displayTop(80 * 5 + 70, "3         ", 10);
/****************************************************************************************************************/
	ASSERT(0 < i_time);
	ASSERT(handle != NULL);

	if (findDelTimer(systemTimerHead, &timer) == NOERR) {
		unlinkTimer(timer);
	}
	else {
		error = allocTimer(&timer);
		if (error != NOERR) {
			return error;
		}
		linkParent(&systemTimerHead, timer);
	}

	// ³󥯸
	if (i_time < REAL_TIMER_RATE){
		linkTimer(setIntervalTimer(i_time), &intervalTimer, timer);
		timer->link = &intervalTimer;
	}
	else{
		linkTimer(i_time, &realTimer, timer);
		timer->link = &realTimer;
	}
	timer->isDelete = NO;
	timer->init = initTime;
	timer->handle = handle;
	timer->arg = arg;
	timer->parent = &systemTimerHead;
	*o_timer = timer;
/****************************************************************************************************************/
//displayTop(80 * 5 + 70, "4         ", 10);
/****************************************************************************************************************/

	return NOERR;
}

/*
 *Ի
 * Ĥ֤
 */
STATIC void getRestTime(TIMER *i_timer, struct itimerval *value)
{
	uint sec, mili, time = 0;
	TIMER *nextTimer;
	
/****************************************************************************************************************/
//displayTop(80 * 5 + 70, "5         ", 10);
/****************************************************************************************************************/
	ASSERT(i_timer != NULL);
	ASSERT(value != NULL);

	nextTimer = getNextList(OFFSETOF(TIMER, list), &i_timer->list);
	if (nextTimer != i_timer){
		TIMER *timer;

		// Ĥ֤û
		for (timer = i_timer; timer != i_timer->link; timer = getPrevList(OFFSETOF(TIMER, list), &timer->list)){
			time += timer->count;
		}
	}

	sec = i_timer->init / 1000;
	mili = i_timer->init - sec * 1000;
	value->it_interval.tv_sec = sec;
	value->it_interval.tv_usec = mili * 1000;

	sec = time / 1000;
	mili = time - sec * 1000;
	value->it_value.tv_sec = sec;
	value->it_value.tv_usec = mili * 1000;
/****************************************************************************************************************/
//displayTop(80 * 5 + 70, "6         ", 10);
/****************************************************************************************************************/
}

/*
 *աtimerLock¾椵줿椫ƤӽФ뤳
 *Գ
 *Թ
 * ֤вᤷޡϥɥ¹Ԥ
 */
STATIC void execTimer(
	TIMER *head,				// 󥯸
	const uint i_time)		// в
{
	static struct handler{
		void (*handle)(void*);
		void *arg;
	}exec[TIMER_MAX];
	int c_exec = 0;
	OBJ_LIST *waitHead = NULL;
	TIMER *timer;
	TIMER *save;
	uint time = i_time;
	int eflags;
	int i;

/****************************************************************************************************************/
//displayTop(80 * 5 + 70, "7         ", 10);
/****************************************************************************************************************/
	eflags = enterCli();
	enter_spinlock(&timerLock);
	{
		for (timer = getNextList(OFFSETOF(TIMER, list), &head->list); timer != head;){
			if (timer->count <= time){
				save = getNextList(OFFSETOF(TIMER, list), &timer->list);
				removeList(&timer->list);
				time -= timer->count;
				timer->count = 0;
				if (timer->isDelete == NO){
					// ޡƤƤΤ̵ʤ뤳ȤϤʤϤ
					ASSERT(c_exec < TIMER_MAX);
					exec[c_exec].handle = timer->handle;
					exec[c_exec].arg = timer->arg;
					++c_exec;

					// ͤȤԤ󥯤ˤĤʤǸǥåȤ
					if (timer->init != 0){
						addHeadList(&waitHead, &timer->list);
					}
					else {
						// ˺ʤsys_sleepǥॢȸĤ֤뤿ᡣ
						timer->isDelete = YES;
					}
				}
				else{
					unlinkParent(timer->parent, timer);
					freeTimer(timer);
				}

				timer = save;
			}
			else{
				timer->count -= time;
				break;
			}
		}

		// ͤΤ󥯤ľ
		for (timer = getHeadObj(OFFSETOF(TIMER, list), waitHead); timer != NULL; timer = save){
			save = getNextList(OFFSETOF(TIMER, list), &timer->list);
			linkTimer(timer->init, head, timer);
		}
	}
	exit_spinlock(&timerLock);
	exitCli(eflags);

	// ޡϥɥ¹Ԥ
	for (i = 0; i < c_exec; ++i){
		exec[i].handle(exec[i].arg);
	}
/****************************************************************************************************************/
//displayTop(80 * 5 + 70, "8         ", 10);
/****************************************************************************************************************/
}

/*
 * ޡ˺դ
 */
STATIC void delTimer(TIMER *timer)
{
//	TIMER *nextTimer;

	ASSERT(timer != NULL);
	
	// 󥯤ڤΥƤ
//	nextTimer = getNextList(OFFSETOF(TIMER, list), &timer->list);
//	if (nextTimer == timer){
//		freeTimer(timer);
//	}
//	else{
		timer->isDelete = YES;
//	}
}

/*
 *Թ
 * إå˥󥯤Ƥޡ
 */
STATIC void releaseProcessTimer(OBJ_LIST *head)
{
	int eflags;

	eflags = enterCli();
	enter_spinlock(&timerLock);
	{
		TIMER *headTimer = getHeadObj(OFFSETOF(TIMER, parentList), head);
		TIMER *timer;
		
		if (headTimer != NULL) {
			timer = headTimer;
			do {
				TIMER *save = getNextList(OFFSETOF(TIMER, parentList), &timer->parentList);

				unlinkTimer(timer);
				freeTimer(timer);
				timer = save;
			} while (timer != headTimer);
		}
    }
	exit_spinlock(&timerLock);
	exitCli(eflags);
}

/*
 * ץޡν
 */
STATIC void initProcessTimer(PROC *proc)
{
	proc->procTimerHead = NULL;
}

/*
 * 
 */
STATIC void initSysTimer()
{
	initCirculList(&intervalTimer.list);
	initCirculList(&realTimer.list);
}

//================================== PUBLIC =============================================

/*
 * 󥿡Х륿ޡμ¹
 */
void execIntervalTimer(
	const uint time,		// вᤷ(ms)
	uint *o_nextTime)		// Υޡηвͽ(ms)
{
	TIMER *nextTimer;

	execTimer(&intervalTimer, time);

	nextTimer = getNextList(OFFSETOF(TIMER, list), &intervalTimer.list);
	if (nextTimer != &intervalTimer) {
		*o_nextTime = nextTimer->count;
	}
	else {
		*o_nextTime = 0;
	}
}

/*
 * ꥢ륯åޡμ¹
 */
void execRealTimer(
	const uint time)		// вᤷ(ms)
{
	execTimer(&realTimer, time);
}

/*
 * 󥰤
 * return : error number
 */
int setTasking(
	const uint i_time)		// 󥰻֡ms
{
	TIMER *timer;
	int eflags;
	int error;
	
	ASSERT(0 < i_time);

	// ޡκ
	eflags = enterCli();
	enter_spinlock(&timerLock);
	{
		error = setSystemTimer(i_time, i_time, handleTasking, NULL, &timer);
	}
	exit_spinlock(&timerLock);
	exitCli(eflags);
	if (error != NOERR){
		return error;
	}

	return NOERR;
}

/*
 * Wake timer from device waiting.
 * return : error number
 */
int setDevWakeTimer(
	const uint	i_time,		// ޡ֡ms
	void		**o_timer)	// ޡ
{
	TIMER *timer;
	int eflags;
	int error;
	
	ASSERT(0 < i_time);

	// ޡκ
	eflags = enterCli();
	enter_spinlock(&timerLock);
	{
		error = setProcessTimer(i_time, 0, handleDevWake, get_current_task(), &timer);
	}
	exit_spinlock(&timerLock);
	exitCli(eflags);
	if (error != NOERR){
		return error;
	}
	*o_timer = timer;

	return NOERR;
}


// Delete timer.
void delWakeTimer(
	void *i_timer)		// ͭʥޡ
{
	delTimer(i_timer);
}

/*
 * timeout --
 *	Execute a function after a specified length of time.
 *
 * untimeout --
 *	Cancel previous timeout function call.
 *
 * callout_handle_init --
 *	Initialize a handle so that using it with untimeout is benign.
 *
 *	See AT&T BCI Driver Reference Manual for specification.  This
 *	implementation differs from that one in that although an 
 *	identification value is returned from timeout, the original
 *	arguments to timeout as well as the identifier are used to
 *	identify entries for untimeout.
 *
 *Գ
 *ա¾ץƤФ礢
 */
struct callout_handle timeout(timeout_t *ftn, void *arg, int to_ticks)
{
	TIMER *timer;
	struct callout_handle callout;
	int eflags;
	int error;

	if (to_ticks == 0) {
		callout.callout = NULL;
		return callout;
	}
	eflags = enterCli();
	enter_spinlock(&timerLock);
	{
		error = setSystemTimer(to_ticks, 0, ftn, arg, &timer);
	}
	exit_spinlock(&timerLock);
	exitCli(eflags);
	if (error == NOERR) {
		callout.callout = timer;
	}
	else {
		printk("Failed timeout()!\n");
		callout.callout = NULL;
	}

	return callout;
}

/*
 *Գ
 */
void untimeout(timeout_t *ftn, void *arg, struct callout_handle callout)
{
	int eflags;

	if (callout.callout == NULL){
		return;
	}
	eflags = enterCli();
	enter_spinlock(&timerLock);
	{
		delTimer(callout.callout);
    }
	exit_spinlock(&timerLock);
	exitCli(eflags);
}

/*
 *	Initialize a handle so that using it with untimeout is benign.
 */
void callout_handle_init(struct callout_handle *callout)
{
	callout->callout = NULL;
}

/*
 *
 */
void callout_init(void *c)
{
// Ȥꤢ⤷ʤ
}

/*
 *
 */
void callout_reset(void *c, int to_ticks, void (*ftn)(void *), void *arg)
{
// Ȥꤢ⤷ʤ
}

/*
 * return : 0 : ȥåcallout̵ or 1 : ȥåcalloutͭ
 */
int callout_stop(void *c)
{
// Ȥꤢ⤷ʤ
	return 0;
}

/*
 *FreeBSD
 * General sleep call.  Suspends the current process until a wakeup is
 * performed on the specified identifier.  The process will then be made
 * runnable with the specified priority.  Sleeps at most timo/hz seconds
 * (0 means no timeout).  If pri includes PCATCH flag, signals are checked
 * before and after sleeping, else signals are not checked.  Returns 0 if
 * awakened, EWOULDBLOCK if the timeout expires.  If PCATCH is set and a
 * signal needs to be delivered, ERESTART is returned if the current system
 * call should be restarted if possible, and EINTR is returned if the system
 * call should be interrupted by the signal (return EINTR).
 */
int	tsleep(void *chan, int pri, const char *wmesg, int timo)
{
	struct buf *bp = chan;
	TIMER *timer;
	int eflags;
	int error;

	bp->proc = get_current_task();

	if (0 < timo){
		eflags = enterCli();
		enter_spinlock(&timerLock);
		{
			error = setProcessTimer(timo, 0, handleWake, get_current_task(), &timer);
		}
		exit_spinlock(&timerLock);
		exitCli(eflags);
		if (error != NOERR){
			return 0;
		}
	}
	else if (timo == 0) {
		// ॢȤʤ
		eflags = enterCli();
		enter_spinlock(&timerLock);
		{
			error = setProcessTimer(UINT_MAX, UINT_MAX, handleWake, get_current_task(), &timer);
		}
		exit_spinlock(&timerLock);
		exitCli(eflags);
		if (error != NOERR){
			return 0;
		}
	}
	else {
		return 0;
	}

	sigWait();

	return 0;
}

/***************************************************************************************************
 *
 * ƥ।󥿡Х륿ޡ
 *
 ***************************************************************************************************/

//================================== PRIVATE ============================================

static ITIMER itimerReal[MAX_CPU];			// ꥢ륤󥿡Х륿ޡ󥯸
static int itimerRock[MAX_CPU];				// ꥢ륤󥿡Х륿ޡѥå
static uint64 beforeRealRdtsc[MAX_CPU];		// ľΥꥢ륤󥿡Х륿ޡrdtsc

/*
 * 󥿡Х륿ޡ¹Ԥ
 */
STATIC void doProcItimer(
	const int64 time,	// в֡rdtsc͡
	const int signal,	// ޡв륷ʥ
	ITIMER *m_itimer)
{
	if (m_itimer->cur <= 0){
		return;
	}
	m_itimer->cur -= time;
	if (m_itimer->cur <= 0){
		sendSignal(m_itimer->proc, signal);
		m_itimer->cur = m_itimer->init;
	}
}

//================================== PROTECTED ==========================================

STATIC void getItimer(int which, struct itimerval *m_value)
{
	time_t sec;
	suseconds_t micro;
	ITIMER *itimer = &get_current_task()->itimer[which];
	
	micro = itimer->init / clock_1micro;
	sec = micro / 1000000;
	m_value->it_interval.tv_sec = sec;
	m_value->it_interval.tv_usec = micro % 1000000;

	micro = itimer->cur / clock_1micro;
	sec = micro / 1000000;
	m_value->it_value.tv_sec = sec;
	m_value->it_value.tv_usec = micro % 1000000;
}

STATIC void setItimer(const int which, const struct itimerval *value, struct itimerval *m_value)
{
	uint64 micro;
	ITIMER *itimer = &get_current_task()->itimer[which];
	int cpu = get_current_cpu();
	
	/* ޡͤ */
	if (m_value != NULL){
		getItimer(which, m_value);
	}
	
	/* ޡͤ */
	micro = value->it_interval.tv_sec * 1000000 + value->it_interval.tv_usec;
	itimer->init = micro * clock_1micro;
	micro = value->it_value.tv_sec * 1000000 + value->it_value.tv_usec;
	itimer->cur = micro * clock_1micro;

	// ꥢ륿ޡ󥯤³
	if (which == ITIMER_REAL){
		int eflags = enterCli();
		enter_spinlock(&itimerRock[cpu]);
		{
			if (0 < itimer->cur){
				addPrevList(&itimerReal[get_current_cpu()].list, &itimer->list);
			}
			else{
				removeList(&itimer->list);
			}
		}
		exit_spinlock(&itimerRock[cpu]);
		exitCli(eflags);
	}
}

/*
 * ץ󥿡Х륿ޡν
 */
STATIC void initProcItimer(PROC *proc)
{
	ITIMER *itimer = proc->itimer;

	memset(itimer, 0, sizeof(proc->itimer));
	initCirculList(&itimer[ITIMER_REAL].list);
	itimer[ITIMER_REAL].proc = proc;
}

/*
 * ץ󥿡Х륿ޡγ
 */
STATIC void releaseProcItimer(ITIMER *itimer)
{
	int cpu = get_current_cpu();
	int eflags;
	
	eflags = enterCli();
	enter_spinlock(&itimerRock[cpu]);
	{
		removeList(&itimer[ITIMER_REAL].list);
	}
	exit_spinlock(&itimerRock[cpu]);
	exitCli(eflags);
}

/*
 * 󥿡Х륿ޡν
 */
STATIC void initItimer()
{
	int i;

	for (i = 0; i < MAX_CPU; i++){
		initCirculList(&itimerReal[i].list);
	}
}

//================================== PUBLIC =============================================

/*
 * ꥢ륤󥿡Х륿ޡ¹Ԥ
 */
void doRealItimer()
{
	int cpu = get_current_cpu();
	const int64 curTime = rdtsc();
	const int64 time = curTime - beforeRealRdtsc[cpu];
	int eflags;
	ITIMER *itimer;
	
	eflags = enterCli();
	enter_spinlock(&itimerRock[cpu]);
	{
		for (itimer = getNextList(OFFSETOF(TIMER, list), &itimerReal[cpu].list); itimer != &itimerReal[cpu];){
			ITIMER *next = getNextList(OFFSETOF(TIMER, list), &itimer->list);

			itimer->cur -= time;
			if (itimer->cur <= 0){
				sendSignal(itimer->proc, SIGALRM);
				itimer->cur = itimer->init;
			}
			if (itimer->cur <= 0){
				removeList(&itimer->list);
			}
			itimer = next;
		}
	}
	exit_spinlock(&itimerRock[cpu]);
	exitCli(eflags);
	
	beforeRealRdtsc[cpu] = curTime;
}

/*
 * ƥв֤ˤ륤󥿡Х륿ޡ
 */
void doSystemItimer(
	int64 time,
	ITIMER *itimer)		// в֡rdtsc͡
{
	// ꥢ륤󥿡Х륿ޡμ¹
	doProcItimer(time, SIGPROF, &itimer[ITIMER_PROF]);
}

/*
 * 桼в֤ˤ륤󥿡Х륿ޡ
 */
void doUserItimer(
	int64 time,
	ITIMER *itimer)		// в֡rdtsc͡
{
	// ꥢ륤󥿡Х륿ޡμ¹
	doProcItimer(time, SIGVTALRM, &itimer[ITIMER_VIRTUAL]);
	doProcItimer(time, SIGPROF, &itimer[ITIMER_PROF]);
}

/***************************************************************************************************
 *
 * ץ
 *
 ***************************************************************************************************/

// ץޡطν
void initProcTimer(void *i_proc)
{
	initProcessTimer(i_proc);
	initProcItimer(i_proc);
}

// ץޡطγ
void releaseProcTimer()
{
	releaseProcessTimer(get_current_task()->procTimerHead);
	releaseProcItimer(get_current_task()->itimer);
}

/***************************************************************************************************
 *
 * ƥॳ
 *
 ***************************************************************************************************/

int sys_time(int *tp)
{
	if(tp != NULL)
	{
		if (checkMem(tp,sizeof(int)) == -1)
			return -EFAULT;
		*tp = unixTime;
	}

	return (int)unixTime;
}

int sys_sleep(uint seconds)
{
	TIMER *timer;
	struct itimerval value;
	int time;
	int eflags;
	int error;

	if (seconds == 0){
		return 0;
	}

	eflags = enterCli();
	enter_spinlock(&timerLock);
	{
		error = setProcessTimer(seconds * 1000, 0, handleWake, get_current_task(), &timer);
	}
	exit_spinlock(&timerLock);
	exitCli(eflags);
	if (error != NOERR){
		return seconds;
	}
	sigWait();

	// Ĥ֤
	eflags = enterCli();
	enter_spinlock(&timerLock);
	{
		getRestTime(timer, &value);
	}
	exit_spinlock(&timerLock);
	exitCli(eflags);

	// 
	delTimer(timer);
	time = value.it_value.tv_sec;

	return time;
}

//Գ
void sys_wake(void *proc)
{
	forceSendSignal(proc, SIGWAKE, TASK_SIGNAL_WAIT);
}

int sys_alarm(uint seconds)
{
	TIMER *timer;
	int eflags;
	int error;

	if (0 < seconds){
		eflags = enterCli();
		enter_spinlock(&timerLock);
		{
			setProcessTimer(seconds*1000, 0, handleAlarm, get_current_task(), &timer);
		}
		exit_spinlock(&timerLock);
		exitCli(eflags);
	}
	else{
		int eflags;
		
		eflags = enterCli();
		enter_spinlock(&timerLock);
		{
			 error = findTimerByHandle(get_current_task()->procTimerHead, handleAlarm, &timer);
	    }
		exit_spinlock(&timerLock);
		exitCli(eflags);
		if (error == NOERR){
			delTimer(timer);
		}
	}
	
	return 0;
}

int sys_settime(uint seconds)
{
	if (get_current_task()->uid!=0){
		return -EPERM;
	}
	setUnixTime(seconds);

	return 0;
}

int sys_gettimeofday(struct timeval *tp, void *tzp)
{
	if (checkMem(tp, sizeof(struct timeval)) == -1){
		return 0;
	}
	gettimeofday(tp);

	return 0;	
}

int sys_getitimer(int which, struct itimerval *m_value)
{
	if (checkMem(m_value, sizeof(*m_value)) == -1){
		return -EFAULT;
	}
	if (ITIMER_PROF < which){
		return -EFAULT;
	}

	getItimer(which, m_value);

	return 0;
}

int sys_setitimer(int which, struct itimerval *value, struct itimerval *m_value)
{
	if (checkMem(value, sizeof(*value)) == -1){
		return -EFAULT;
	}
	if (m_value != NULL){
		if (checkMem(m_value, sizeof(*m_value)) == -1){
			return -EFAULT;
		}
	}
	if (ITIMER_PROF < which){
		return -EFAULT;
	}

	setItimer(which, value, m_value);

	return 0;
}

/***************************************************************************************************
 *
 * 
 *
 ***************************************************************************************************/

void initTimer(void)
{
	initSysTimer();
	initItimer();
	calcTimeClock();
}

/*************************************************************************************/
void test_time(int cnt)
{
	static int go = NO;
	
	if (go == YES){
		printk("test_time() %d\n", cnt);
	}
	if (cnt == 0){
		go = YES;
	}
}
