/*
 * signal.c
 *
 * Copyright 2004, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 */


#include <sys/types.h>
#include <sys/param.h>
#include "lib.h"
#include "proc.h"
#include "interrupt.h"
#include "mm.h"
#include "lock.h"
#include "errno.h"
#include "term.h"
#include "signal.h"
#include "test.h"
#include "debug.h"


//#define DEBUG_SIGNAL 1
#ifdef DEBUG_SIGNAL
	#define STATIC
	#define INLINE
#else
	#define STATIC static
	#define INLINE	inline
#endif


/****************************************************************************************
 *
 * ʥ
 *
 *****************************************************************************************/


//================================== PRIVATE ============================================


static SIG_ACTION defaultAction[NSIG];


/*
 * Default aactinos.
 */
/* terminate process. */
STATIC void terminate(int signum)
{
	exit(0,signum);
}

/* abnormal terminate process. */
STATIC void abnormalTerminate(int signum)
{
	exit(0,signum);
}

/* stop process. */
STATIC void stopProcess(int signum)
{
	PROC *proc=get_current_task();

	proc->signum = signum;

	/* ƥץwaitƤ鵯 */
	if ((isForeground(proc) == YES) && (proc->parent->state & TASK_CHILD_WAIT)){
		addToSchedule(proc->parent, TASK_SIGNAL_WAIT);
	}
	delFromSchedule(TASK_SEND_SIGNAL_WAIT);
	wait_task();
}


/*
 * ʥϥɥμ¹
 */
STATIC void doSigHandler(SIGNAL *sig,int sigNum)
{
	if (USER_BEG <= (uint)sig->action[sigNum])
	{
		uint beforeMask;

		/* ʥޥ촹 */
		beforeMask = sig->mask;
		sig->mask |= sig->actMask[sigNum] | (1 << sigNum);

		/* 桼ϥɥ¹ԡ */
		doUserHandler(sigNum,sig->action[sigNum]);

		/* ʥޥ᤹ */
		sig->mask = beforeMask;
	}
	else if ((sig->action[sigNum] != SIG_IGN) && (sig->action[sigNum] != SIG_DFL)){
		/* ǥեȥϥɥ¹ԡ */
		sig->action[sigNum](sigNum);
	}
}

/* Хñ̤Υʥ뤴ȤΥʥֹ */
static uchar signalNum[] = {
/*	0 1 2 3 4 5 6 7 8 9 a b c d e f */
	0,1,2,1,3,1,2,1,4,1,2,1,3,1,2,1,	/* 0 */
	5,1,2,1,3,1,2,1,4,1,2,1,3,1,2,1,	/* 1 */
	6,1,2,1,3,1,2,1,4,1,2,1,3,1,2,1,	/* 2 */
	5,1,2,1,3,1,2,1,4,1,2,1,3,1,2,1,	/* 3 */
	7,1,2,1,3,1,2,1,4,1,2,1,3,1,2,1,	/* 4 */
	5,1,2,1,3,1,2,1,4,1,2,1,3,1,2,1,	/* 5 */
	6,1,2,1,3,1,2,1,4,1,2,1,3,1,2,1,	/* 6 */
	5,1,2,1,3,1,2,1,4,1,2,1,3,1,2,1,	/* 7 */
	8,1,2,1,3,1,2,1,4,1,2,1,3,1,2,1,	/* 8 */
	5,1,2,1,3,1,2,1,4,1,2,1,3,1,2,1,	/* 9 */
	6,1,2,1,3,1,2,1,4,1,2,1,3,1,2,1,	/* a */
	5,1,2,1,3,1,2,1,4,1,2,1,3,1,2,1,	/* b */
	7,1,2,1,3,1,2,1,4,1,2,1,3,1,2,1,	/* c */
	5,1,2,1,3,1,2,1,4,1,2,1,3,1,2,1,	/* d */
	6,1,2,1,3,1,2,1,4,1,2,1,3,1,2,1,	/* e */
	5,1,2,1,3,1,2,1,4,1,2,1,3,1,2,1,	/* f */
};

/*
 * ʥֹμ
 */
STATIC int getSigNum(uint signal)
{
	union{
		uint ui;
		uchar uc[4];
	}sigValue;
	int sigNum;
	
	sigValue.ui = signal;
	if (sigValue.uc[0] != 0)
		sigNum = signalNum[sigValue.uc[0]] - 1;
	else if (sigValue.uc[1] != 0)
		sigNum = signalNum[sigValue.uc[1]] + 7;
	else if (sigValue.uc[2] != 0)
		sigNum = signalNum[sigValue.uc[2]] + 15;
	else
		sigNum = signalNum[sigValue.uc[3]] + 23;
	
	return sigNum;
}


//================================== PROTECTED ==========================================


static void initSigAction()
{
	// ǥեȥꤹ롣
	defaultAction[0]			= NULL;
	defaultAction[SIGABRT]		= abnormalTerminate;
	defaultAction[SIGALRM]		= terminate;
	defaultAction[SIGBUS]		= abnormalTerminate;
	defaultAction[SIGCHLD]		= SIG_IGN;
	defaultAction[SIGCONT]		= SIG_DFL;
	defaultAction[SIGFPE]		= abnormalTerminate;
	defaultAction[SIGHUP]		= terminate;
	defaultAction[SIGILL]		= abnormalTerminate;
	defaultAction[SIGINT]		= terminate;
	defaultAction[SIGKILL]		= terminate;
	defaultAction[SIGPIPE]		= terminate;
	defaultAction[SIGPOLL]		= terminate;
	defaultAction[SIGPROF]		= terminate;
	defaultAction[SIGQUIT]		= abnormalTerminate;
	defaultAction[SIGSEGV]		= abnormalTerminate;
	defaultAction[SIGSTOP]		= stopProcess;
	defaultAction[SIGSYS]		= abnormalTerminate;
	defaultAction[SIGTERM]		= terminate;
	defaultAction[SIGTRAP]		= abnormalTerminate;
	defaultAction[SIGTSTP]		= stopProcess;
	defaultAction[SIGTTIN]		= stopProcess;
	defaultAction[SIGTTOU]		= stopProcess;
	defaultAction[SIGURG]		= SIG_IGN;
	defaultAction[SIGUSR1]		= terminate;
	defaultAction[SIGUSR2]		= terminate;
	defaultAction[SIGVTALRM]	= terminate;
	defaultAction[SIGXCPU]		= abnormalTerminate;
	defaultAction[SIGXFSZ]		= abnormalTerminate;
	defaultAction[SIGWINCH]		= SIG_IGN;
	defaultAction[SIGINFO]		= SIG_IGN;
	defaultAction[SIGWAKE]		= SIG_IGN;
}


//================================== PUBLIC =============================================


/*
 * ʥ¹
 * parameters : user esp address
 */
void doSignal()
{
	SIGNAL *sig;

	/* ͥǥͥȤƤ뤫 */
	if (isNest() == TRUE)
		return;

	sig = get_current_task()->signal_struct;
	while (sig->signal != 0)
	{
		/* ʥֹμ */
		int sigNum = getSigNum(sig->signal);

		/* ʥ륯ꥢ */
		sig->signal ^= 1 << sigNum;

		/* ʥϥɥμ¹ */
		doSigHandler(sig,sigNum);
	}
}

// ƥȳѥʥ¹
// return : FALSE or TRUE : SIGINTȯԤ줿
void doSignalRestart()
{
	uint signal;
	uint saveSignal;
	SIGNAL *sig;
	
	/* ͥǥͥȿϣ */
	if (isSingleNest() == FALSE){
		return;
	}

	sig = get_current_task()->signal_struct;
	signal = saveSignal = sig->signal;
	while (signal != 0){
		/* ʥֹμ */
		int sigNum = getSigNum(signal);

		/* ʥ륯ꥢ */
		signal ^= 1 << sigNum;
		if (sig->actFlag[sigNum] & SA_RESTART){
			/* ʥ륯ꥢ */
			saveSignal ^= 1 << sigNum;
			
			/* ʥϥɥμ¹ */
			doSigHandler(sig, sigNum);
		}
	}
	sig->signal = saveSignal;
}


//Գ
// ʥ
// ǥեȥΡcontinue processפϡʥץư롣
// return : 0 or error number
int sendSignal(PROC *proc, int signal)
{
	int sig;
	SIGNAL *sig_st;

	sig_st = proc->signal_struct;

	// ʥ뤬̵ˤƤʤ
	if (sig_st->action[signal] == SIG_IGN){
		return -EPERM;
	}
	
	// ޥ줿ʥα
	sig_st->reserve |= (1 << signal) & sig_st->mask;

	sig = ((1 << signal) | sig_st->mask) ^ sig_st->mask;
	if (sig != 0){
		sig_st->signal |= sig;
		addToSchedule(proc, TASK_SIGNAL_WAIT);
	}

	return 0;
}


//Գ
// ʥ
void forceSendSignal(PROC *proc, const int signal, const int procStat)
{
	SIGNAL *sig = proc->signal_struct;
	int eflag;

	eflag = enterCli();
	enter_spinlock(&sig->lock);
	{
		sig->signal |= 1 << signal;
		addToSchedule(proc, procStat);
	}
	exit_spinlock(&sig->lock);
	exitCli(eflag);
}


// ʥ
void psignal(struct proc *proc, int sig)
{
	sendSignal(proc, sig);
}


// ʥԤ
void sigWait()
{
	SIGNAL *sig = get_current_task()->signal_struct;
	int eflag;

	eflag = enterCli();
	enter_spinlock(&sig->lock);
	if(sig->signal == 0){
		delFromSchedule(TASK_SIGNAL_WAIT);
		exit_spinlock(&sig->lock);
		exitCli(eflag);

		wait_task();
	}
	else{
		exit_spinlock(&sig->lock);
		exitCli(eflag);
	}

	// SIGWAKEϥꥢ
	sig->signal &= ~(1 << SIGWAKE);
}


// ˥ץ򵯤
void sigWakeSoon(PROC *proc)
{
	SIGNAL *sig = proc->signal_struct;
	int eflag;

	eflag = enterCli();
	enter_spinlock(&sig->lock);
	{
		sig->signal |= 1 << SIGWAKE;
		addToScheduleSoon(proc, TASK_SIGNAL_WAIT);
	}
	exit_spinlock(&sig->lock);
	exitCli(eflag);
}


// ٤ƤΥץ򵯤
void wakeup(void *ident)
{
	sendSignalAllProc(SIGWAKE);
}


/*
 * fork()Υʥ뵡ꡣ
 * parameters : ƥץΥʥ빽¤Ρ
 * return : 0 or error number
 */
SIGNAL *setSignal(SIGNAL *parent_sig)
{
	SIGNAL *sig_st;

	sig_st = kmalloc(sizeof(SIGNAL));
	if (sig_st == NULL){
		return NULL;
	}

	memcpy(sig_st, parent_sig, sizeof(SIGNAL));
	sig_st->signal = 0;
	sig_st->reserve = 0;

	return sig_st;
}


/*
 * exit()Υʥ뵡γ
 * parameters : ʥ빽¤
 */
void releaseSignal(SIGNAL *sig_st)
{
	kfree(sig_st);
}


/*
 * ʥ뵡Υꥻåȡ
 * parameters : ʥ빽¤
 */
void resetSignal(SIGNAL *sig_st)
{
	memset(sig_st,0,sizeof(SIGNAL));

	/* ǥեȥ򥳥ԡ */
	memcpy(sig_st->action, defaultAction, sizeof(defaultAction));
}


/*
 * SIGINTȯ
 * return : FALSE or TRUE
 */
int isSigint()
{
	uint signal = ((SIGNAL*)get_current_task()->signal_struct)->signal;

	// SIGWAKEϥƥѤʤΤǾʤ
	return ((signal & ~(1 << SIGWAKE)) == 0)? FALSE : TRUE;
}


/*
 * 󥫡ͥ륹åѡλʥ򤹤٤ƥޥ롣
 */
void setSigAllMask()
{
	SIGNAL *sigStruct = get_current_task()->signal_struct;
	
	sigStruct->mask = SIGABRT | SIGBUS | SIGFPE | SIGILL | SIGQUIT | SIGSEGV | SIGSYS | SIGTRAP | SIGXCPU | SIGXFSZ;
}


/*
 * ʥ뵡ν
 * return : SIGNAL struct or NULL
 */
SIGNAL *setFirstSignal()
{
	SIGNAL *sig_st;

	if((sig_st=kmalloc(sizeof(SIGNAL)))==NULL)return NULL;

	/*  */
	resetSignal(sig_st);	
	sig_st->mask=0;
	sig_st->lock=0;

	return sig_st;
}


/****************************************************************************************
 *
 * < ƥॳ >
 *
 *****************************************************************************************/


int sys_sigaction(int signum, const struct sigaction *act, struct sigaction *oldact)
{
	SIGNAL *signal = get_current_task()->signal_struct;

	if (signum == 0){
		return 0;
	}
	if ((NSIG <= signum) || (signum < 0)){
		return -EINVAL;
	}

	if (act != NULL){
		if ((signum == SIGKILL) || (signum == SIGSTOP)){
			return -EINVAL;		/* SIGKILLSIGSTOPѹԲġ */
		}

		if (act->sa_handler == SIG_DFL){
			signal->action[signum] = defaultAction[signum];
		}
		else if (act->sa_handler == SIG_IGN){
			signal->action[signum] = SIG_IGN;
		}
		else if (USER_BEG <= (uint)act->sa_handler){
			signal->action[signum] = act->sa_handler;
		}
		else{
			return -EINVAL;		/* ϥɥ饢ɥ쥹 */
		}

		signal->actMask[signum] = act->sa_mask;
		signal->actFlag[signum] = act->sa_flags;
	}
	if (oldact != NULL){
		if (checkMem(oldact,sizeof(*oldact)) == ERR){
			return -EFAULT;
		}

		if (signal->action[signum] == SIG_IGN){
			oldact->sa_handler = SIG_IGN;
		}
		else if (signal->action[signum] == defaultAction[signum]){
			oldact->sa_handler = SIG_DFL;
		}
		else{
			oldact->sa_handler = signal->action[signum];
		}

		oldact->sa_mask = signal->actMask[signum];
	}

	return 0;
}


int sys_sigprocmask(int how,uint *set,uint *oldset)
{
	uint tmp_mask=*set;
	SIGNAL *sig_st;


	sig_st=get_current_task()->signal_struct;

	if (oldset != NULL){
		*oldset = sig_st->mask;
	}

	switch (how){
		case SIG_BLOCK:
			if (tmp_mask & ((1 << SIGKILL) | (1 << SIGSTOP))){
				return -EINVAL;			// SIGKILLSIGSTOPϥ֥åԲġ
			}
			sig_st->mask |= tmp_mask;
			break;
		case SIG_UNBLOCK:
			sig_st->mask &= ~tmp_mask;
			break;
		case SIG_SETMASK:
			if (tmp_mask & ((1 << SIGKILL) | (1 << SIGSTOP))){
				return -EINVAL;			// SIGKILLSIGSTOPϥ֥åԲġ
			}
			sig_st->mask = tmp_mask;
			break;
	}

	return 0;
}


int sys_kill(pid_t pid,int sig)
{
	int rest;
	pid_t pgid;
	PROC *proc;


	if ((uint)sig >= NSIG)
		return -EPERM;

	if (pid > 0)
	{
		if ((proc = searchProcList(pid)) == NULL)
			return -ESRCH;
		rest = sendSignal(proc,sig);
	}
	else if (pid == 0)		/* callץƱ롼ס */
	{
		proc = get_current_task();
		pgid = proc->pgid;
		while ((proc = getNextProc(proc,pgid)) != NULL)
			sendSignal(proc,sig);
	}
	else if (pid == -1)	/* ץ */
	{
		sendSignalAllProc(sig);
		sendSignal(get_current_task(),sig);
	}
	else				/* pid<-1 -pidΥ롼ס */
	{
		pgid = -pid;
		for (proc = NULL; (proc = getNextProc(proc,pgid)) != NULL;)
			sendSignal(proc,sig);
	}

	return 0;
}


int sys_sigpending(sigset_t *set,int flag)
{
	SIGNAL *sig_st;


	/* ݥ󥿤å */
	if (checkMem(set,sizeof(sigset_t)) == -1)
		return -EINVAL;

	sig_st = get_current_task()->signal_struct;

	switch(flag)
	{
		case SIG_PENDING_SOTORE:
			*set = sig_st->reserve;
			break;
		case SIG_PENDING_SET:
			sig_st->reserve = *set;
			break;
	}

	return 0;
}


int sys_sigsuspend(const sigset_t *mask)
{
	uint beforeMask;
	SIGNAL *sig_st;


	/* SIGKILLSIGSTOPϥ֥åԲġ */
	if(*mask&((1<<SIGKILL)|(1<<SIGSTOP)))return -EINVAL;

	sig_st=get_current_task()->signal_struct;
	beforeMask=sig_st->mask;
	sig_st->mask=*mask;

	/* suspend. */
	delFromSchedule(TASK_SIGNAL_WAIT);
	wait_task();

	sig_st->mask=beforeMask;

	return -EINTR;
}


/****************************************************************************************
 *
 * 
 *
 *****************************************************************************************/


void initSignal()
{
	initSigAction();
}

//****************************************************************************************
void testSignal()
{
}
