/*
 * proc.c
 *
 * Copyright 2002, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 *
 * ץط
 */


#include <sys/config.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/times.h>
#include <machine/except_i386.h>
#include <machine/segment.h>
#include <net/net.h>
#include <kern/mm.h>
#include <kern/fs.h>
#include <kern/errno.h>
#include <kern/interrupt.h>
#include <kern/lock.h>
#include <kern/time.h>
#include <kern/elf.h>
#include <kern/proc.h>
#include <kern/signal.h>
#include <dev/console/console.h>
#include <kern/term.h>
#include <kern/list.h>
#include <kern/debug.h>
#include <kern/test.h>


//#define DEBUG_PROC 1
#ifdef DEBUG_PROC
	#define STATIC
	#define INLINE
#else
	#define STATIC static
	#define INLINE inline
#endif


/* tssǼ¤(104Х) */
typedef struct{
	uint link,esp0,ss0,esp1,ss1,esp2,ss2,cr3,eip,eflags,eax,ecx,
		edx,ebx,esp,ebp,esi,edi,es,cs,ss,ds,fs,gs,ldts,tflag_iomap;
}TSS;


CPU cputask[MAX_CPU];		/* Task infomations for each cpu */

/* TSS󥰻Υååɬ */
static TSS kernel_tss={
	0,
	KERNEL_ESP_BEG - FPU_SAVE_SIZE,		/* esp0 */
	KERNEL_DATA_DES,					/* ss0 */
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
};
static PROC *idle_proc[MAX_CPU];	/* Idle process */
static PROC *proc1;					/* pid1ץ¤Ρ */


/******************************************************************************************************
 *
 * 塼
 *
 *******************************************************************************************************/


//================================== PRIVATE ============================================


// 򥿥塼³
// return : YES = ³NO = ³Ƥʤ
STATIC int addTask(PROC *proc, const int cpu, const int flag, int *isIntr)
{
	// ˥ǥХߤεߤʤ褦ˤ뤿Υե饰ȥߥåꤹ
	*isIntr = YES;

	/* ȥե饰γǧ */
	if(proc->state & flag){
		proc->state = TASK_RUNNING;

		if(cputask[cpu].current_task->next == idle_proc[cpu]){
			proc->next = proc->prev = proc;
			cputask[cpu].current_task->next = proc;
		}
		else{
			PROC *current = cputask[cpu].current_task->next->prev;

			if (current != proc){
				proc->prev = current;
				proc->next = current->next;
				current->next->prev = proc;
				current->next = proc;
			}
		}
		return YES;
	}
	else{
		return NO;
	}
}


// 򥿥塼뤫ڤΥ
// return : YES = ڤΥNO = ڤΥƤʤ
STATIC int delTask(const int flag, int *isIntr)
{
	int cpu = get_current_cpu();
	PROC *proc = cputask[cpu].current_task;

	if (*isIntr == NO){
		if(proc->next == proc){
			idle_proc[cpu]->next = idle_proc[cpu]->prev = idle_proc[cpu];
			proc->next = idle_proc[cpu];
		}
		else{
			proc->next->prev = proc->prev;
			proc->prev->next = proc->next;
		}
		proc->state = flag;

		return YES;
	}
	else{
		return NO;
	}
}

//================================== PUBLIC =============================================


//Գ
//ա¾ץƤФ롣
// ¹ԥ塼³롣
// parameters : "TASK_RUNNING"ʤ"o_isDel""PROC_SCHEDULE_NOT_DEL"
void addToSchedule(PROC *proc, const int flag)
{
	int cpu = proc->cpu;
	int eflags;

	eflags = enterCli();
	enter_spinlock(&cputask[cpu].gate);
	{
		int dummy;
		addTask(proc, cpu, flag, &dummy);
	}
	exit_spinlock(&cputask[cpu].gate);
	exitCli(eflags);
}


//Գ
//ա¾ץƤФ롣
// ľ˼¹Ԥ롣
void addToScheduleSoon(PROC *proc, const int flag)
{
	int cpu = get_current_cpu();
	int eflags;
	int isAdd;

	eflags = enterCli();
	enter_spinlock(&cputask[cpu].gate);
	{
		int dummy;
		isAdd = addTask(proc, cpu, flag, &dummy);
	}
	exit_spinlock(&cputask[cpu].gate);
	exitCli(eflags);
	if (isAdd == YES){
		--cputask[proc->cpu].proc_num;
		++cputask[cpu].proc_num;
		proc->cpu = cpu;
	}
}


//առץƤ֡
// ¹ԥ塼뤫ڤΥ
void delFromSchedule(const int flag)
{
	int cpu = get_current_cpu();
	int eflags;

	eflags = enterCli();
	enter_spinlock(&cputask[cpu].gate);
	{
		int isIntr = NO;
		delTask(flag, &isIntr);
	}
	exit_spinlock(&cputask[cpu].gate);
	exitCli(eflags);
}


//աեץߤ򵯤˸ƤӽФɬפ
// ưե饰
void initAddScheduleFlag(int *m_isIntr)
{
	*m_isIntr = NO;
}


//Գ
//ա¾ץƤФ롣
// ưե饰ꤷƻư
// return : YES = ư or NO = ưƤʤ
int addToScheduleSet(PROC *proc, const int flag, int *m_isIntr)
{
	int cpu = proc->cpu;
	int eflags;
	int isAdd;

	eflags = enterCli();
	enter_spinlock(&cputask[cpu].gate);
	{
		isAdd = addTask(proc, cpu, flag, m_isIntr);
	}
	exit_spinlock(&cputask[cpu].gate);
	exitCli(eflags);
	
	return isAdd;
}


// ưե饰ˤ꥿¹ԥ塼뤫ڤΥ
//առץƤ֡
// return : YES = ߤNO = ߤƤʤ
int delFromScheduleRef(const int flag, int *isIntr)
{
	int cpu = get_current_cpu();
	int eflags;
	int isDel;
	
	eflags = enterCli();
	enter_spinlock(&cputask[cpu].gate);
	{
		isDel = delTask(flag, isIntr);
	}
	exit_spinlock(&cputask[cpu].gate);
	exitCli(eflags);
	
	return isDel;
}


// 򥦥ȥ塼˰ܤ
//առץƤ֡
void waitProcSignal(void **waitHead)
{
	int cpu = get_current_cpu();
	PROC *proc = get_current_task();
	int eflags;

	eflags = enterCli();
	enter_spinlock(&cputask[cpu].gate);
	{
		int isIntr = NO;

		delTask(TASK_SIGNAL_WAIT, &isIntr);
		addHeadPrevList((OBJ_LIST**) waitHead, &proc->waitList);
	}
	exit_spinlock(&cputask[cpu].gate);
	exitCli(eflags);
}


// 򥦥ȥ塼¹ԥ塼³
void runProcSignal(void **waitHead)
{
	PROC *proc = getHeadObj(OFFSETOF(PROC, waitList), *waitHead);
	int cpu = proc->cpu;
	int eflags;

	ASSERT(*waitHead != NULL);

	eflags = enterCli();
	enter_spinlock(&cputask[cpu].gate);
	{
		int dummy;

		removeFromHeadList((OBJ_LIST**) waitHead, &proc->waitList);
		addTask(proc, cpu, TASK_SIGNAL_WAIT, &dummy);
	}
	exit_spinlock(&cputask[cpu].gate);
	exitCli(eflags);
}


// Process switch
// parameters : γ߻Υ֥åȥåesp,γ߻espͤ¸륹å
// returns : Page directory
uint *switch_task(uint esp, volatile uint beforeEsp)
{
	uint volatile *p_beforeEsp = &beforeEsp;
	uint volatile *p_esp = &esp;
	int cpu;
	PROC *current;

	cpu = get_current_cpu();
	current = cputask[cpu].current_task;
	*p_beforeEsp = current->esp;
	current->esp = esp;

	current = current->next;
	*p_esp = (uint)&current->esp;
	cputask[cpu].current_task = current;

	// TSե饰򥻥åȤFPU㳰褦ˤ
	setTs();

	return getPageDir();
}


/******************************************************************************************************
 *
 * ץꥹ
 *
 *******************************************************************************************************/


//================================== PRIVATE ============================================


enum{
	PROC_HASH_NUM=29,	/* ץϥåꥹȺ */
};


static PROC_LIST procHash[PROC_HASH_NUM];		/* ץϥåꥹȡ */


/*
 * ץꥹͤץɥ쥹롣
 * parameters : process link address
 * return : process address
 */
STATIC INLINE PROC *procLinkToProc(PROC_LIST *p)
{
	return (PROC*)((uint)p-(uint)&((PROC*)0)->proc_next);
}


/*
 * ϥåͤ롣
 * parameters : pid
 * return : hash value
 */
STATIC INLINE int getProcHash(int pid)
{
	return pid % PROC_HASH_NUM;
}


//================================== PROTECTED ==========================================


/*
 * Add to process list.
 * parameters : process address
 */
STATIC INLINE void addProcList(PROC *proc)
{
	PROC_LIST *hash = &procHash[getProcHash(proc->pid)];
	PROC_LIST *p = (PROC_LIST*)&proc->proc_next;


	enter_spinlock(&hash->lockGate);
	{
		p->prev = hash;
		p->next = hash->next;
		hash->next->prev = p;
		hash->next = p;
	}
	exit_spinlock(&hash->lockGate);
}


/*
 * Delete from process list.
 * parameters : process address
 */
STATIC INLINE void delProcList(PROC *proc)
{
	PROC_LIST *hash=&procHash[getProcHash(proc->pid)];
	PROC_LIST *p=(PROC_LIST*)&proc->proc_next;


	enter_spinlock(&hash->lockGate);
	{
		p->next->prev=p->prev;
		p->prev->next=p->next;
	}
	exit_spinlock(&hash->lockGate);
}


/*
 * Init process list;
 */
STATIC void initProclist()
{
	int i;


	/* ץϥåꥹȽ */
	for(i=0;i<PROC_HASH_NUM;++i)
	{
		procHash[i].next=procHash[i].prev=&procHash[i];
		procHash[i].lockGate=0;
	}
}


//================================== PUBLIC =============================================


/*
 * Search process from process list.
 * parameters : pid
 * return : process address or failed=NULL
 */
PROC *searchProcList(int pid)
{
	PROC_LIST *hash=&procHash[getProcHash(pid)];
	PROC_LIST *p;


	for(p=hash->next;p!=hash;p=p->next)
		if(procLinkToProc(p)->pid==pid)return procLinkToProc(p);

	return NULL;
}


/******************************************************************************************************
 *
 * ץID롼ID
 *
 *******************************************************************************************************/


//================================== PRIVATE ============================================


/*
 * ץ롼פθ
 * parameters : search start process,pgid
 * return : process or Failed=NULL
 */
STATIC PROC *searchPg(PROC *proc,pid_t pgid)
{
	PROC *rest;


	do
	{
		if(proc->state!=TASK_EXIT)
		{
			if(proc->pgid==pgid)return proc;
			if(proc->child!=NULL)
				if((rest=searchPg(proc->child,pgid))!=NULL)return rest;
		}
	}while((proc=proc->brother)!=NULL);

	return NULL;
}


/*
 * ֲλҥץ饷ʥ
 * parameters : search start process or NULL=proc1
 * return : prosecc or Failed=NULL
 */
STATIC void sendToProc(PROC *proc,int signal)
{
	/* μν֤Υץ */
	if (proc->child != NULL)
		sendToProc(proc->child,signal);
	if (proc->brother != NULL)
		sendToProc(proc->brother,signal);

	if (proc != get_current_task())
		sendSignal(proc,signal);
}


//================================== PROTECTED ==========================================


/*
 * Get process ID.
 * ա
 *  ĹϢ³Ư硢Сեˤʤǽꡣ
 * return : process ID.
 */
STATIC INLINE int getPid(PROC *proc)
{
	static int pid=1;

	return pid++;
}


//================================== PUBLIC =============================================


/*
 * Ϳ줿ץ롼פΥץ롣
 * parameters : search start process or NULL=proc1,groupe ID
 * return : prosecc or Failed=NULL
 */
PROC *getNextProc(PROC *proc,pid_t pgid)
{
	/* proc1õ */
	if (proc == NULL)
		proc = proc1;

	/* μν֤Υץõ */
	if (proc->child != NULL)
		return searchPg(proc->child,pgid);
	else
		do
		{
			if (proc->brother != NULL)
				return searchPg(proc->brother,pgid);
			if ((proc = proc->parent) == NULL)
				break;
		}while (proc->pgid == pgid);

	return NULL;
}


/*
 * ץʳץ˥ʥ
 */
void sendSignalAllProc(int signal)
{
	if (proc1->child != NULL)
		sendToProc(proc1->child, signal);
}


// ѡ桼
// return : error number
int suser(PROC *proc)
{
	return (proc->uid == 0)? NOERR : -EPERM;
}


/******************************************************************************************************
 *
 * ץ
 *
 *******************************************************************************************************/


extern int *active_cpu;		/* ư CPU */

/* PUBLIC */
static int procLockGate;		/* ¾Υץ¤λȻΥåȡ */


//================================== PRIVATE ============================================


// ֥ξʤCPU롣
// retuuns : cpu number
STATIC int GetFewTaskCpu()
{
	int cpu = 0;
	uint num;
	int i;

	num = cputask[0].proc_num;
	for(i = 1; i < *active_cpu; ++i)
		if(cputask[i].proc_num<num){
			num = cputask[i].proc_num;
			cpu = i;
		}

	return cpu;
}


/*
 * λҥץξ֤𤹤롣
 * ӥåȥޥåװ֤ΰ̣"sys/wait.h"򻲾ȤΤ
 * parameters : process,state address
 * return : 0 or not exit or stop=-1
 */
STATIC void setState(PROC *proc,int *stat_loc)
{
	union{
		int i;
		char c[4];
	}stat;
	
	
	if (stat_loc == NULL)
		return;

	stat.i = 0;
	stat.c[0] = proc->signum;
	stat.c[1] = proc->exit_state;
	if (proc->state == TASK_SEND_SIGNAL_WAIT)
	{
		stat.c[0] = 0x7F;
		stat.c[1] = proc->signum;
	}
	*stat_loc = stat.i;
}


/*
 * exitҥץ¤Τ롣
 * parameters : current process
 * return : 󥯤κǸλҥץ
 */
STATIC PROC *releaseChild(PROC *proc)
{
	PROC **p,*q;


	for (p = &proc->child; *p != NULL;)
	{
		if ((*p)->state == TASK_EXIT)
		{
			q = *p;
			*p = (*p)->brother;
			kfree(q);
		}
		else
			p = &(*p)->brother;
	}

	return *p;
}


//================================== PROTECTED ==========================================


/*
 * ץ¹Բǽˤ
 * parameters : process
 */
STATIC void addNewProc(PROC *proc)
{
	proc->cpu=GetFewTaskCpu();
	proc->state=TASK_WAIT;
	addToSchedule(proc, TASK_WAIT);
	++cputask[proc->cpu].proc_num;
}


/*
 * ƥץ¹ԡ
 * 1 exitҥץγ
 * 2 Ĥäҥץpid0ץϤ
 * parameters : current process
 */
STATIC void releaseChildProc(PROC *proc)
{
	PROC *p,**q;


	for (p = proc->child; p != NULL;p = p->brother){
		p->parent = proc1;
	}
	enter_spinlock(&proc->gate);
	{
		p = releaseChild(proc);
	}
	exit_spinlock(&proc->gate);
	enter_spinlock(&proc1->gate);
	{
		for (q = &proc1->child; *q != NULL;q = &(*q)->brother);
		*q = p;
	}
	exit_spinlock(&proc1->gate);
}


/*
 * 1 exitҥץ롣
 * 2 λҥץξ֤𤹤롣
 * parameters : process structure,search pid,search pgid,task state,state address
 * return : pid or 0 or error number
 */
STATIC int searchExitState(PROC *proc, pid_t pid, pid_t pgid, int state, int *stat_loc)
{
	int rest = -ECHILD;
	PROC *p;


	if (stat_loc != NULL){
		*stat_loc = 0;
	}

	if (pid != 0){
		for (p = proc->child; p != NULL; p = p->brother){
			if (p->pid == pid){
				rest = 0;
				if (p->state & state){
					setState(p,stat_loc);
					rest = pid;
				}
				break;
			}
		}
	}
	else if (pgid != 0){
		for (p = proc->child; p != NULL; p = p->brother){
			if (p->pgid == pgid){
				rest = 0;
				if (p->state & state){
					setState(p,stat_loc);
					rest = p->pid;
					break;
				}
			}
		}
	}
	else if (proc->child != NULL){
		rest = 0;
		for (p = proc->child; p != NULL; p = p->brother){
			if (p->state & state){
				setState(p,stat_loc);
				rest = p->pid;
				break;
			}
		}
	}

	/* exitҥץγ */
	enter_spinlock(&proc->gate);
	{
		releaseChild(proc);
	}
	exit_spinlock(&proc->gate);

	return rest;
}


// parameters : isKernelThreadˤϥͥ륹åɤȤ"0"ʳꤹ
STATIC int fork(int isKernelThread)
{
	void *stack;
	uint esp;
	PROC *parent = get_current_task();
	PROC *child;
	int rest;

	if (parent->count_child >= MAX_CHILD){
		return -EAGAIN;
	}

	/* PROC¤ΤƤ */
	if ((child = kmalloc(sizeof(PROC))) == NULL){
		return -ENOMEM;
	}

	/*  */
	memset(&child->PROC_ZERO_TOP, 0, (uint) &child->PROC_ZERO_END - (uint) &child->PROC_ZERO_TOP);
	child->parent = parent;
	enter_spinlock(&parent->gate);
	{
		child->brother = parent->child;
		parent->child = child;
	}
	exit_spinlock(&parent->gate);
	child->nestCount = parent->nestCount;

	/* ID. */
	child->pid = getPid(child);
	child->pgid = parent->pgid;
	child->uid = 0;
	child->gid = parent->gid;
	child->sid = parent->sid;

	/* å󥢥ɥ쥹(ü¤Υɥ쥹)򥳥ԡ */
	child->ctlterm = parent->ctlterm;
	
	child->gate = 0;

	/* եط¤Τγơ */
	child->file_struct = copyFileStruct(parent->file_struct);
	if ((child->file_struct) == NULL){
		goto ERR;
	}

	/* Page tableƤ */
	rest = forkPage(parent->mm_struct, isKernelThread, &child->mm_struct, &stack);
	if (rest < 0){
		goto ERR;
	}

	/* ʥꤹ롣 */
	child->signal_struct = setSignal(parent->signal_struct);
	if (child->signal_struct == NULL){
		goto ERR;
	}

	/* ޡν */
	initProcTimer(child);

	/* ҥץΥͥ륹å˥ƥȤꤹ */
	esp = setContext((uint)stack);
	if (esp == 0){
		return 0;		// ҥץ
	}

	child->esp=esp;

	/* ץꥹȤ˲ä롣 */
	addProcList(child);

	/* Add child process to schedule queue */
	addNewProc(child);

	++parent->count_child;

	return child->pid;
ERR:
	kfree(child);

	return -ENOMEM;
}


//================================== PUBLIC =============================================


/*
 * proc1wait
 */
void waitProc1()
{
	for(;;)
	{
		/* Ԥ */
		delFromSchedule(TASK_SIGNAL_WAIT);
		wait_task();

		enter_spinlock(&proc1->gate);
		{
			releaseChild(proc1);
		}
		exit_spinlock(&proc1->gate);
	}
	idle();
}


/*
 * ͥͥȥȤ䤹
 */
void incrementNest()
{
	get_current_task()->nestCount += 1;
}


/*
 * ͥͥȥȤ򸺤餹
 */
void decrementNest()
{
	get_current_task()->nestCount -= 1;

	ASSERT(0 <= get_current_task()->nestCount);
}


/*
 * ͥǥͥȤƤ뤫
 */
int isNest()
{
	return (0 < get_current_task()->nestCount)? TRUE : FALSE;
}


int isSingleNest()
{
	return (get_current_task()->nestCount == 1)? TRUE : FALSE;
}


// ͥ륹åɤfork
int forkKernelThread()
{
	return fork(1);
}


// return : error number or 0
int kthread_create(void (*func)(void *), void *arg, struct proc **newpp, const char *fmt, ...)
{
	int pid = forkKernelThread();

	switch(pid){
	case 0:
		func(arg);
		exit(0, 0);
	case -1:
		return -ENOBUFS;
	default:
		*newpp = searchProcList(pid);
		return 0;
	}
}


void kthread_exit(int ecode)
{
	exit(ecode, 0);
}


/***************************************************************************************************
 *
 * ƥ
 *
 ***************************************************************************************************/


//================================== PRIVATE ============================================


static uint64 beforeTime[MAX_CPU];	// ľrdtsc


//================================== PROTECTED ==========================================


// CPU֤ƥץγΰ­
STATIC void addCpuTimeToParent()
{
	PROC *proc = get_current_task();
	PROC *parent = proc->parent;

	parent->ctime[0] += calcTimeFromClock(proc->sysTime) / TASK_TIME;
	parent->ctime[1] += calcTimeFromClock(proc->userTime) / TASK_TIME;
}


//================================== PUBLIC =============================================


// ͥΥƥ֤׻
//աեͥͥȤ򥫥Ȥ˸ƤӽФ
void calcCpuTimeIn()
{
	int cpu = get_current_cpu();
	PROC *proc = get_current_task();
	uint64 curTime = rdtsc();

	if (proc->nestCount == 1){
		proc->userTime += curTime - beforeTime[cpu];
	}
	else{
		proc->sysTime += curTime - beforeTime[cpu];
	}

	// 󥿡Х륿ޡμ¹
	doRealItimer();
	if (1 < proc->nestCount){
		doSystemItimer(curTime - beforeTime[cpu], proc->itimer);
	}
	else{
		doUserItimer(curTime - beforeTime[cpu], proc->itimer);
	}

	beforeTime[cpu] = curTime;
}


// ͥ˽ФΥƥ֤׻
//աեͥͥȤ򥫥Ȥ˸ƤӽФ
void calcCpuTimeOut()
{
	int cpu = get_current_cpu();
	PROC *proc = get_current_task();
	uint64 curTime = rdtsc();

	proc->sysTime += curTime - beforeTime[cpu];

	// 󥿡Х륿ޡμ¹
	doSystemItimer(curTime - beforeTime[cpu], proc->itimer);

	beforeTime[cpu] = curTime;
}


/******************************************************************************************************
 *
 * ƥॳ
 *
 *******************************************************************************************************/

/*
 * return : ƥץʤҥץID,ҥץʤ0,error=error number
 * todoꥹ
 *  ǥ쥯ȥꥹȥ꡼򥳥ԡ롣
 */
int sys_fork()
{
	return fork(0);
}


int sys_exec(const char *path, char **argv, char **envp, volatile SYSCALL4_FRAME frame)
{
	enum{
		PARAM = sizeof(frame.para)	/* ƥॳΰΥ */
	};
	uint entry_addr,stack_esp;
	int rest;
	EXEC_FD fd;
	SYS_INFO *sys_info;

	if ((rest = exec_open(path,&fd)) < 0){
		return rest;
	}

	/* ȴĶѿ桼å˥ԡ롣 */
	if ((stack_esp = setUserStack(argv,envp)) == 0)
		return -1;

	/* ELFХʥΥɡ */
	if ((entry_addr = loadElf(&fd)) == 0)
		return -1;

	/* եΥ */
	exec_close(&fd);
	
	/* å˥ƥꡣ */
	sys_info = (SYS_INFO*)(USER_ESP_BEG - sizeof(SYS_INFO));
	sys_info->lastAddr = getLastLinearAddr();

	/* ʥ륢ν */
	resetSignal(get_current_task()->signal_struct);

	/* ե빽¤Τν */
	initExecFs();

	/* ƥॳե졼ꡣ */
	frame.es = USER_DATA_DES;
	frame.ds = USER_DATA_DES;
	frame.eip = entry_addr;
	frame.cs = USER_CODE_DES;
	frame.user_esp = stack_esp - PARAM;
	frame.ss = USER_DATA_DES;

	return 0;
}


/*
 * ڡǥ쥯ȥγ̥ץԤ
 * parameters : ƥץϤ
 */
void exit(int state, int signum)
{
	void *mm_struct;
	PROC *proc=get_current_task();
	PROC *parent;

	/* üץγ */
	releaseTerm(proc);

	/* ץꥹȤ롣 */
	delProcList(proc);

	/* ҥץγ */
	releaseChildProc(proc);

	parent = proc->parent;

	--parent->count_child;

	/* եǥץγ */
	releaseFileStruct(proc->file_struct);

	/* ޡγ */
	releaseProcTimer();

	/* ʥγ */
	releaseSignal(proc->signal_struct);

	mm_struct = proc->mm_struct;
	proc->exit_state = state & 0xff;
	proc->signum = signum & 0xff;

	// ǥХԲ㳰ץΥꥢ
	clearFpuBeforeProc();

	/* CPUƥץ­ */
	addCpuTimeToParent();

	/* 塼뤫waitƤƤƤӽФ */
	cli();
	enter_spinlock(&procLockGate);
	{
		{
//			static PROC tmp_proc[MAX_CPU];
			int cpu=get_current_cpu();

			delFromSchedule(TASK_EXIT);
//			tmp_proc[cpu].next=proc->next;
//			tmp_proc[cpu].prev=proc->prev;
//			cputask[cpu].current_task=&tmp_proc[cpu];
			--cputask[cpu].proc_num;
		}

		sendSignal(parent,SIGCHLD);
//		if ((isForeground(proc) == YES) && (parent->state & TASK_CHILD_WAIT))
		if (parent->state & TASK_CHILD_WAIT){
			addToSchedule(parent, TASK_SIGNAL_WAIT | TASK_SEND_SIGNAL_WAIT);
		}
	}
	exit_spinlock(&procLockGate);

	/* Υ˥åơڡǥ쥯ȥ롣 */
	returnExit(mm_struct);
}


int sys_exit(int state)
{
	exit(state,0);

	return 0;
}


int sys_wait(int pid,int *stat_loc,int options)
{
	int rest;
	int state;
	PROC *proc = get_current_task();

	/* Ĵ٤ץ֤ꡣ */
	if (options & WUNTRACED)
		state = TASK_WAIT | TASK_SEND_SIGNAL_WAIT | TASK_EXIT;
	else
		state = TASK_TRACE | TASK_EXIT;

	/* ǽλҥץ */
	if (pid > 0){
		rest = searchExitState(proc,pid,0,state,stat_loc);
	}
	else if (pid == 0){
		rest = searchExitState(proc,0,proc->pgid,state,stat_loc);
	}
	else if (pid == -1){
		rest = searchExitState(proc,0,0,state,stat_loc);
	}
	else{
		rest = searchExitState(proc,0,-pid,state,stat_loc);
	}

	if (0 < rest){
		return rest;
	}

	if ((rest < 0) && (options & WNOHANG)){
		return rest;
	}

	/* Ԥ */
	delFromSchedule(TASK_SIGNAL_WAIT|TASK_CHILD_WAIT);
	wait_task();

	/* ҥץ */
	if (pid > 0)
		rest = searchExitState(proc,pid,0,state,stat_loc);
	else if (pid == 0)
		rest = searchExitState(proc,0,proc->pgid,state,stat_loc);
	else if (pid == -1)
		rest = searchExitState(proc,0,0,state,stat_loc);
	else
		rest = searchExitState(proc,0,-pid,state,stat_loc);

	return rest;
}


int sys_getpid()
{
	return get_current_task()->pid;
}


int sys_getgid()
{
	return get_current_task()->gid;
}


int sys_getpgid(pid_t pid)
{
	if (pid == 0)
		return get_current_task()->pgid;
	else
	{
		PROC *proc = searchProcList(pid);
		
		if (proc == NULL)
			return -ESRCH;
		if (proc->sid != get_current_task()->sid)
			return -EPERM;
		return proc->pgid;
	}
}


int sys_getppid()
{
	return get_current_task()->parent->pid;
}


int sys_getuid()
{
	return get_current_task()->uid;
}


int sys_getsid(pid_t pid)
{
	if (pid == 0)
		return get_current_task()->sid;
	else
	{
		PROC *proc = searchProcList(pid);
		
		if (proc == NULL)
			return -ESRCH;
		if (proc->sid != get_current_task()->sid)
			return -EPERM;
		return proc->sid;
	}
}


int sys_setgid(gid_t gid)
{
	PROC *proc = get_current_task();


	if (proc->uid != 0)
		return -EPERM;
	proc->gid = gid;
	return 0;
}


int sys_setpgid(pid_t pid, pid_t pgid)
{
	PROC *proc,*pgproc;
	
	
	if (pid == 0)
		proc = get_current_task();
	else
	{
		proc = searchProcList(pid);
		if (proc == NULL)
			return -ESRCH;
		if ((proc != get_current_task()) && (proc->parent != get_current_task()))
			return -ESRCH;
	}
	if (pgid < 0)
		return -EINVAL;
	if (pgid == 0)
		pgid = proc->pid;
	
	/* Session leader */
	if (proc->sid == proc->pid)
		return -EPERM;
	if (proc->sid != get_current_task()->sid)
		return -EPERM;
	
	/* ץۤʤ륻åΥץ롼פ˰ư褦Ȥ */
	pgproc = searchProcList(pgid);
	if (pgproc == NULL)
		return -EPERM;
	if (pgproc->sid != proc->sid)
		return -EPERM;
	proc->pgid = pgid;

	return 0;
}


int sys_setuid(uid_t uid)
{
	PROC *proc = get_current_task();


	if (proc->uid != 0)
		return -EPERM;
	proc->uid = uid;
	return 0;
}


int sys_setsid()
{
	PROC *proc = get_current_task();
	
	
	if (proc->pgid != proc->pid)
	{
		proc->sid = proc->pid;
		proc->pgid = proc->pid;
		proc->ctlterm = NULL;
		
		return proc->sid;
	}
	else
		return -EPERM;
}


int sys_times(struct tms *tms)
{
	PROC *proc;

	if (checkMem(tms,sizeof(struct tms)) == -1){
		return -EFAULT;
	}

	proc=get_current_task();
	tms->tms_stime = calcTimeFromClock(proc->sysTime) / TASK_TIME;
	tms->tms_utime = calcTimeFromClock(proc->userTime) / TASK_TIME;
	tms->tms_cstime = proc->ctime[0];
	tms->tms_cutime = proc->ctime[1];

	return 0;
}


/******************************************************************************************************
 *
 * 
 *
 *******************************************************************************************************/


//================================== PUBLIC =============================================


/*
 * Init taskcpu
 * cpu˥ꥢꤹɬפ
 * parameters : number of cpu
 * returns : 0
 */
int init_cputask(int cpu)
{
	cputask[cpu].cpu=cpu;
	cputask[cpu].current_task=NULL;
	cputask[cpu].proc_num=0;
	cputask[cpu].gate=0;

	return 0;
}


/*
 * TSSǥץꤹ
 *  : ǥץ͡TSSɥ쥹
 */
void set_tss(int des)
{
	set_gdt(des,(uint)&kernel_tss,104,TYPE_TSS);

	asm volatile(
		"movl	%0,%%eax\n"
		"ltr	%%ax"
		::"m"(des)
	);
}


/*
 * Set idle process table
 * parameters : Page directory
 * returns : 0 or Error number
 */
int set_idle_proc(int cpu)
{
	/* Set process table */
	idle_proc[cpu] = kmalloc(sizeof(PROC));
	if (idle_proc[cpu] == NULL){
		return -ENOMEM;
	}
	memset(idle_proc[cpu], 0, sizeof(PROC));
	idle_proc[cpu]->next = idle_proc[cpu]->prev=idle_proc[cpu];
	idle_proc[cpu]->cpu = cpu;
	idle_proc[cpu]->state = TASK_RUNNING;

	/* init paging */
	if((idle_proc[cpu]->mm_struct=initPaging())==NULL)return -1;

	/* init file descriptor. */
	if(init_file_struct(idle_proc[cpu])==-1)return -ENOMEM;

	/* init signal. */
	if ((idle_proc[cpu]->signal_struct = setFirstSignal()) == NULL){
		return -ENOMEM;
	}

	/* add to schedule queue */
	cputask[cpu].current_task=idle_proc[cpu];

	/* ץꥹȤν */
	initProclist();

	return 0;
}


/*
 * ǽΥץꡣ
 * return : 0 or error number
 */
int initProc1()
{
	proc1 = get_current_task();
	proc1->parent = NULL;

	return initProc1Fs();
}


/*************************************************************************************************/
void test_proc(int isLs)
{
	static int access = NO;
	
	if (isLs == YES) {
		access = YES;
	}
}
/**************************************************************************************************/
