/*
 * fs.c
 *
 * Copyright 2002, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 *
 * ۥե륷ƥࡣ
 * ե륷ƥ
 */


#include <sys/config.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/limits.h>
#include <sys/utime.h>
#include <sys/resource.h>
#include <net/net.h>
#include <kern/lib.h>
#include <kern/lib_path.h>
#include <kern/mm.h>
#include <kern/proc.h>
#include <kern/lock.h>
#include <kern/errno.h>
#include <kern/interrupt.h>
#include <kern/time.h>
#include <kern/signal.h>
#include <kern/device.h>
#include <kern/fs.h>
#include <kern/debug.h>
#include <kern/test.h>


//#define DEBUG_FS 1
#ifdef DEBUG_FS
	#define STATIC
	#define INLINE
#else
	#define STATIC	static
	#define INLINE	inline
#endif


/***************************************************************************************
 * ۥե륷ƥ
 *
 *  ޥȻȥǥ쥯ȥѹե륪ץե¹Իɲä롣
 *  ȥ꡼󥯤κǸϺǽΥȥ꡼Ϣ뤹롣
 *  ǥ쥯ȥϥޥȤ줿ΰʳϥǥХ֥åǤ϶
 *  ƥॳǸüȥ꡼ϡΥƥॳ˺ʤ褦
 *  󥿤䤷ΥƥॳκǸǥ󥿤򸺤餹
 ***************************************************************************************/

//================================== PRIVATE ============================================

typedef struct{
	const char *path;
	VENTRY *vent;
}SEARCH_PARAM;

static char _rootEnt[sizeof(VENTRY)+1];
static VENTRY *rootEnt = (VENTRY*)&_rootEnt;	// 롼ȥȥ꡼
static char _devEnt[sizeof(VENTRY)+1];
static VENTRY *devEnt=(VENTRY*)&_devEnt;		// devǥ쥯ȥꥨȥ꡼
static  MODIFY_WAIT_LOCK fsModifyLock;			// ۥե륷ƥࡦե륷ƥॳå

/*
 * ۥե륷ƥλȳ
 */
STATIC INLINE void refVfs()
{
	modifyWaitRef(&fsModifyLock);
}

/*
 * ۥե륷ƥλȽλ
 */
STATIC INLINE void refEndVfs()
{
	modifyWaitRefEnd(&fsModifyLock);
}

/*
 * ۥե륷ƥι
 */
STATIC INLINE void modifyVfs()
{
	modifyWaitLock(&fsModifyLock);
}

/*
 * ۥե륷ƥιλ
 */
STATIC INLINE void modifyEndVfs()
{
	modifyWaitUnlock(&fsModifyLock);
}

/*
 *ԻաԺƵ
 * ѥ롼ȤޤäƥХåե˥ԡ롣
 * return : write size or ERR
 */
STATIC int copyCurrentPath(const VENTRY *vent, const size_t size, char *m_path)
{
	int len;

	if (vent == rootEnt){
		m_path[0] = '/';
		return 1;
	}
	else{
		if ((len = copyCurrentPath(vent->parent, size, m_path)) == ERR){
			return ERR;
		}
		if (len + vent->len + 1 > size) {
			return ERR;
		}
		memcpy(&m_path[len], vent->name, vent->len);
		m_path[len + vent->len] = '/';

		return len + vent->len + 1;
	}
}

/*
 *ԻաԼ
 * ҥΡɥ󥯤饨ȥ꡼õ
 * return : error number
 */
STATIC int searchEntry(
	const char *path,
	VENTRY *parent,			// ƥȥ꡼
	VENTRY **o_entry)		// Ĥäȥ꡼
{
	VENTRY *entry;

	if (cmpPath(".", path, 1) == 0){
		*o_entry = parent;
		return NOERR;
	}
	if (cmpPath("..", path, 2) == 0){
		*o_entry = parent->parent;
		return NOERR;
	}

	// ƱΡɥ󥯤õ
	if (parent->child == NULL){
		return -ENOENT;
	}
	entry = parent->child;
	while (cmpPath(entry->name, path, entry->len) != 0) {
		entry = entry->next;
		if (entry == parent->child){
			return -ENOENT;
		}
	}
	*o_entry = entry;

	return NOERR;
}

/*
 *ԹաԼ
 * ѥƬ̾Υȥ꡼Ρɲä롣
 * ˥ȥ꡼ϤΥȥ꡼ݥ󥿤ꤷ-EEXIST֤
 * return : error number
 */
STATIC int addEntry(
	const char *path, 
	VENTRY *parent, 
	VENTRY **o_entry)
{
	VENTRY *newEntry;

	// Ʊ̾ȥ꡼뤫
	if (searchEntry(path, parent, o_entry) == NOERR){
		return -EEXIST;
	}

	// ȥ꡼κ
	newEntry = kmalloc(sizeof(VENTRY) + pathLen(path));
	if (newEntry == NULL){
		return -ENOMEM;
	}
	memset(newEntry, 0, sizeof(VENTRY));

	// 󥯤ɲä
	if (parent->child == NULL){
		newEntry->prev = newEntry->next = newEntry;
		parent->child = newEntry;
	}
	else{
		VENTRY *child = parent->child;
		newEntry->prev = child;
		newEntry->next = child->next;
		child->next->prev = newEntry;
		child->next = newEntry;
	}
	newEntry->parent = parent;
	newEntry->child = NULL;
	
	*o_entry = newEntry;

	return NOERR;
}

/*
 *Թ
 * ƱΡɥ󥯤饨ȥ꡼롣
 * return : error number
 */
STATIC int delEntry(
	VENTRY *entry)		// ȥ꡼
{
	char name[PATH_MAX];
	VENTRY *parent;

	if (0 < entry->openCount){
		return -EBUSY;
	}
	if (entry->child != NULL){
		return -EBUSY;
	}
	if (entry->mount != NULL){
		return -EBUSY;
	}

	parent = entry->parent;

	if (entry->next == entry){
		parent->child = NULL;
	}
	else{
		if (parent->child == entry){
			parent->child = entry->next;
		}
		entry->prev->next = entry->next;
		entry->next->prev = entry->prev;
	}

	// ޥȤ
	if (parent->mount == entry){
		entry->fs->umount(entry->din, entry->inodeObj);
		parent->mount = NULL;
	}

	// 
	if (entry->del == YES){
		char *pname;

		name[PATH_MAX - 1] = '\0';
		pname = &name[PATH_MAX - 1];
		memcpy(pname -= entry->len, entry->name, entry->len);
		for (parent = entry->parent; parent->inodeObj == NULL; parent = parent->parent){
			if (pname + parent->len + 1 < name){
				goto EXIT;
			}
			*--pname = '/';
			memcpy(pname -= parent->len, parent->name, parent->len);
		}
		entry->fs->delete(pname, parent->din, parent->inodeObj, S_IFREG);
	}
EXIT:
	kfree(entry);

	return NOERR;
}

/*
 *ԻաԼ
 * ѥ̾inode֥Ȥ륨ȥ꡼򸡺롣
 * ϰˤinode֥ȤäƤǽΥȥ꡼ȥѥ롣
 * return : error number
 */
STATIC int searchPath(
	const char *i_path, 
	VENTRY *i_entry, 
	const char **o_path, 
	VENTRY **o_vent)
{
	const char *path;
	VENTRY *entry, *nextEntry;
	int error;

	*o_path = i_path;
	*o_vent = i_entry;
	entry = i_entry;
	for (path = i_path; *path != '\0';){
		if (entry->type != S_IFDIR){
			return -ENOTDIR;
		}
		error = searchEntry(path, entry, &nextEntry);
		if (error != NOERR){
			return error;
		}
		entry = nextEntry;
		path = getNextPath(path);
		if (entry->mount != NULL){
			*o_path = path;
			*o_vent = entry->mount;
			entry = entry->mount;
		}
		else if (entry->inodeObj != NULL){
			*o_path = path;
			*o_vent = entry;
		}
	}

	if (entry->inodeObj != NULL){
		return NOERR;
	}
	else{
		return -ENOENT;
	}
}

/*
 *Թ
 * ȥ꡼̤äƺ롣
 */
STATIC void delUpEntry(VENTRY *entry)
{
	VENTRY *parent;

	for(;;){
		parent = entry->parent;
		if (delEntry(entry) != NOERR){
			break;
		}
		entry = parent;
	}
}

/*
 *Ի
 * ۥȥ꡼򥪡ץ󤹤
 */
STATIC INLINE void openVent(VENTRY *m_vent)
{
	ASSERT(m_vent->openCount < _POSIX_OPEN_MAX);

	++m_vent->openCount;
}

/*
 *Թ
 * ۥȥ꡼򥯥
 * return : YES = ,NO = ̤
 */
STATIC int closeVent(VENTRY *m_vent)
{
	ASSERT(0 < m_vent->openCount);

	--m_vent->openCount;
	if (m_vent->openCount == 0){
		delUpEntry(m_vent);
		return YES;
	}
	else{
		return NO;
	}
}

/*
 *Ի
 * ȥ꡼
 */
STATIC INLINE void unlinkVent(VENTRY *m_vent)
{
	m_vent->del = YES;
}

/*
 *ԹաԼ
 * ȥ꡼ѥɲä롣
 * return : error number
 */
STATIC int creatEntry(
	const char *i_path, 
	VENTRY *parent, 
	VENTRY **o_entry)
{
	const char *path = i_path;
	VENTRY *entry;
	VENTRY *newEntry;
	int error;

	ASSERT(*path != NULL);

	for(entry = parent;;){
		error = addEntry(path, entry, &newEntry);
		switch (error){
		case NOERR:
			// ȥ꡼ν
			newEntry->fs = entry->fs;
			newEntry->din = entry->din;
			newEntry->type = S_IFDIR;
			newEntry->parent = entry;
			newEntry->len = pathLen(path);
			newEntry->del = NO;
			memcpy(newEntry->name, path, newEntry->len);
		case -EEXIST:
			break;
		default:
			return error;
		}

		path += newEntry->len;
		if (*path == '\0'){
			break;
		}
		++path;
		entry = newEntry;
	}
	newEntry->openCount = 1;
	*o_entry = newEntry;

	return NOERR;
}

/*
 * struct statfs˳Ǽ롣
 * return : 0 or error number
 */
STATIC int getFsstat(VENTRY *vent, struct statfs *statfs)
{
	int error;
	DEV_STAT devstat;

	if ((error = vent->fs->statfs(vent->din, statfs)) < 0){
		return error;
	}
	get_devstat(vent->din,&devstat);

	if (devstat.prt_num > 0){
		sprintk(statfs->f_mntfromname, "/dev/%s%d", devstat.name, devstat.prt_num);
	}
	else{
		sprintk(statfs->f_mntfromname, "/dev/%s", devstat.name);
	}

	copyCurrentPath(vent, MNAMELEN, statfs->f_mntonname);

	return 0;
}

/*
 *Թ
 * ȥ꡼˿ȥ꡼ޥȤ롣
 * return : error number
 */
STATIC int mountEntry(VENTRY *dstEnt, FS *fs, int din, void *rootInodeObj)
{
	VENTRY *mountEnt;

	// Ǥ˥ޥȤƤ뤫
	if (dstEnt->mount != NULL){
		return -EBUSY;
	}

	// ޥȥȥ꡼κ
	mountEnt = kmalloc(sizeof(VENTRY) + dstEnt->len);
	if (mountEnt == NULL){
		return -ENOMEM;
	}
	memset(mountEnt, 0, sizeof(VENTRY));
	mountEnt->next		= dstEnt->next;
	mountEnt->prev		= dstEnt->prev;
	mountEnt->inodeObj	= rootInodeObj;
	mountEnt->parent	= dstEnt->parent;
	mountEnt->fs		= fs;
	mountEnt->din		= din;
	mountEnt->type		= S_IFDIR;
	mountEnt->len		= dstEnt->len;
	memcpy(mountEnt->name, dstEnt->name, dstEnt->len);

	// ޥȤ
	dstEnt->mount = mountEnt;

	openVent(dstEnt);

	return NOERR;
}

/*
 *Թ
 * ޥȤ
 * return : error number
 */
STATIC int umountEntry(VENTRY *entry)
{
	VENTRY *mountVent = entry->mount;
	FS *fs;
	void *inodeObj;
	int din;
	int error;

	if (mountVent == NULL){
		return -ENOENT;
	}
	if (mountVent->child != mountVent->child->next->next){
		return -EBUSY;
	}

	/* ޥȥȥ꡼κ */
	fs = mountVent->fs;
	inodeObj = mountVent->inodeObj;
	din = mountVent->din;
	if (closeVent(mountVent) == YES){
		error = fs->umount(mountVent->din, mountVent->inodeObj);
		if (error != NOERR){
			return error;
		}
	}

	return NOERR;
}

/*
 *ԻաԺƵ
 * ǥ쥯ȥޥȥȥ꡼θ
 * return ޥȿ
 */
STATIC int searchMountDir(VENTRY *vent, struct statfs *m_statfs, int lenStatfs)
{
	int rest, count;
	VENTRY *vent_begin;

	if (lenStatfs <= 0){
		return 0;
	}
	vent_begin = vent;
	count = lenStatfs;

	do{
		// ޥȤ
		if (vent->mount != NULL){
			while (vent->mount != NULL){
				vent = vent->mount;
			}
			if (m_statfs != NULL){
				if (getFsstat(vent, m_statfs) == 0){
					m_statfs += 1;
					count -= 1;
				}
			}
		}

		if (vent->child != NULL){
			rest = searchMountDir(vent->child, m_statfs, count);
			m_statfs += rest;
			count -= rest;
		}
	}while ((vent = vent->next) != vent_begin);

	return lenStatfs - count;
}

/*
 *Ի
 * ޥȥȥ꡼򸡺롣
 * return : struct statfsسǼ or ޥȿstruct statfs=NULL
 */
STATIC int searchMount(struct statfs *m_statfs, int lenStatfs)
{
	int rest;

	if (lenStatfs <= 0){
		return 0;
	}

	/* rootե륷ƥࡣ */
	if (m_statfs != NULL){
		getFsstat(rootEnt, m_statfs);
	}

	rest = searchMountDir(rootEnt->child, m_statfs + 1, lenStatfs - 1);

	return rest + 1;
}


/************************************************************************************
 * ե륷ƥ
 ************************************************************************************/

// ե륷ƥϿ󥯥ȥå
static FS fsInfoTop;

/*
 * Ͽ줿ե륷ƥõ
 * return : FS address or NULL
 */
STATIC FS *search_fs(const char *name)
{
	FS *p;

	for (p = fsInfoTop.next; p != NULL; p = p->next){
		switch (strcmp(name, p->name)){
		case 0:
			return p;
		case -1:
			return NULL;
		}
	}

	return NULL;
}

//================================== PUBLIC =============================================

/*
 * ե륷ƥϿ롣
 * return : 0 or Error=-1
 */
int regist_fs(FS *info)
{
	FS *p;

	p = &fsInfoTop;
	for(; p->next != NULL; p = p->next){
		switch (strcmp(info->name, p->next->name)){
		case 0:
			return -EEXIST;
		case -1:
			goto EXIT;
		}
	}
EXIT:
	info->next = p->next;
	p->next = info;

	return 0;
}


/************************************************************************************
 * ץΥե
 ************************************************************************************/

//================================== PRIVATE ============================================

/*
 * եǥץֹ
 * return : fd number or error number
 */
STATIC int getEmptyFdNum(FILE_STRUCT *fstruct, const uint min)
{
	int num;

	// ֹϺ祪ץ̤
	if (_POSIX_OPEN_MAX <= min){
		return -EINVAL;
	}

	for (num = min; fstruct->fd[num] != NULL;){
		if (_POSIX_OPEN_MAX == ++num){
			num = 0;
		}
		if (num == min){
			return -ENFILE;
		}
	}

	return num;
}

/*
 * եǥץ鳺ե롣
 */
STATIC void delFromFd(F_DSC *fd)
{
	char path[PATH_MAX];
	int len;

	len = copyCurrentPath(fd->vent, PATH_MAX, path);
	if (len >= PATH_MAX){
		return;
	}

	/*  */
	rootEnt->fs->delete(path, rootEnt->din, rootEnt->inodeObj, S_IFREG);
}

/*
 * Get current directory entry.
 * return : current directory entry
 */
STATIC INLINE VENTRY *getCurrentDir()
{
	return ((FILE_STRUCT*)get_current_task()->file_struct)->current_dir;
}

/*
 * Set current directory.
 */
STATIC void setCurrentDir(VENTRY *vent)
{
	FILE_STRUCT *fstruct;

	fstruct = get_current_task()->file_struct;
	closeVent(fstruct->current_dir);
	fstruct->current_dir = vent;
}

/*
 * Get file descriptor flag.
 * return : flag or error number
 */
STATIC int getFdFlag(uint num)
{
	FILE_STRUCT *fstruct;

	// ֹϺ祪ץ̤
	if (_POSIX_OPEN_MAX <= num){
		return -EBADF;
	}
	fstruct=get_current_task()->file_struct;
	if(fstruct->fd[num]==NULL)return -EBADF;

	return fstruct->flag[num];
}

/*
 * Set file descriptor flag.
 * return : 0 or error number
 */
STATIC int setFdFlag(uint num,int flag)
{
	FILE_STRUCT *fstruct;

	// ֹϺ祪ץ̤
	if (_POSIX_OPEN_MAX <= num){
		return -EBADF;
	}
	fstruct = get_current_task()->file_struct;
	if(fstruct->fd[num] == NULL){
		return -EBADF;
	}
	fstruct->flag[num] = flag;

	return 0;
}

/*
 * Get file open mode flag.
 * returm : open mode flag
 */
STATIC int getOpenFlag(uint num)
{
	FILE_STRUCT *fstruct;

	// ֹϺ祪ץ̤
	if (_POSIX_OPEN_MAX <= num){
		return -EINVAL;
	}
	fstruct = get_current_task()->file_struct;
	if (fstruct->fd[num] == NULL){
		return -EBADF;
	}

	return fstruct->fd[num]->accessMode;
}

/*
 * Set file open mode flag.
 * returm : open mode flag
 */
STATIC int setOpenFlag(uint num,int flag)
{
	FILE_STRUCT *fstruct;

	// ֹϺ祪ץ̤
	if (_POSIX_OPEN_MAX <= num){
		return -EBADF;
	}
	fstruct = get_current_task()->file_struct;
	if (fstruct->fd[num] == NULL){
		return -EBADF;
	}

	fstruct->fd[num]->accessMode &= ~(O_APPEND | O_NONBLOCK);
	fstruct->fd[num]->accessMode |= flag & (O_APPEND | O_NONBLOCK);

	return 0;
}

/*
 * Copy file descriptor.
 * return : file descriptor number or error number
 */
STATIC int copyFd(int num1,int num2)
{
	FILE_STRUCT *fstruct = get_current_task()->file_struct;

	// ǥץֹ
	if (_POSIX_OPEN_MAX <= (uint)num2){
		num2 = 0;
	}
	num2 = getEmptyFdNum(fstruct,num2);

	// եǥץ򥳥ԡ
	fstruct->fd[num2] = fstruct->fd[num1];
	++fstruct->fd_count;
	++fstruct->fd[num2]->refCount;
	fstruct->flag[num2] = 0;			// Clear close-on-exec flag.
	openVent(fstruct->fd[num2]->vent);

	return num2;
}

/*
 * file descriptorλȥ󥿤򸺤餹
 * return : error number
 */
STATIC int unlinkFd(FILE_STRUCT *fstruct, int fdNum)
{
	int din;
	void *inodeObj;
	F_DSC *fd;
	FS *fs;
	int rest;

	fd = fstruct->fd[fdNum];
	--fd->refCount;
	if (fd->form == FD_FORM_NETSOKET){
		if (fd->refCount <= 0){
			closeSocket((SOCKET*)fd);
		}
	}
	else{
		fs = fd->vent->fs;
		din = fd->vent->din;
		inodeObj = fd->vent->inodeObj;

		rest = closeVent(fd->vent);
		if (rest == YES){
			rest = fs->close(din, inodeObj);
			WARNING(rest == 0);
		}

		if (fd->refCount <= 0){
			// õե饰ǧ
			if (fd->accessMode & O_DEL){
				delFromFd(fd);
			}
			kfree(fd);
		}
	}

	fstruct->fd[fdNum] = NULL;
	fstruct->next_fd = fdNum;
	--fstruct->fd_count;

	return NOERR;
}

//================================== PUBLIC =============================================

/*
 * fork˥եطι¤Τƥץ饳ԡ롣
 * return : child file structure or error=NULL
 */
FILE_STRUCT *copyFileStruct(FILE_STRUCT *fstruct)
{
	FILE_STRUCT *cp_struct;
	int i, cnt;

	cp_struct = kmalloc(sizeof(FILE_STRUCT));
	if (cp_struct == NULL){
		return NULL;
	}
	memcpy(cp_struct, fstruct, sizeof(FILE_STRUCT));

	// եǥץλȥ󥿤1䤹
	for (cnt = fstruct->fd_count, i = 0; 0 < cnt; ++i){
		if (cp_struct->fd[i] != NULL){
			++cp_struct->fd[i]->refCount;
			if (cp_struct->fd[i]->form == FD_FORM_LOCALFILE){
				openVent(cp_struct->fd[i]->vent);
			}

			--cnt;
		}
	}

	return cp_struct;
}

/*
 * եطι¤Τ롣
 */
void releaseFileStruct(FILE_STRUCT *fstruct)
{
	int i;

	for (i = 0; i < _POSIX_OPEN_MAX; ++i){
		if (fstruct->fd[i] != NULL){
			// ۥե륷ƥι
			modifyVfs();
			{
				unlinkFd(fstruct, i);
			}
			modifyEndVfs();
			if (fstruct->fd_count <= 0){
				break;
			}
		}
	}

	kfree(fstruct);
}

/*
 * ե빽¤Τν
 */
void initExecFs()
{
	FILE_STRUCT *fstruct = get_current_task()->file_struct;
	int i, cnt;

	// Close file descriptor.
	for (cnt = fstruct->fd_count, i = 0; 0 < cnt; ++i){
		if (fstruct->fd[i] != NULL){
			if (fstruct->flag[i] & FD_CLOEXEC){
				// ۥե륷ƥι
				modifyVfs();
				{
					unlinkFd(fstruct,i);
				}
				modifyEndVfs();
			}
			--cnt;
		}
	}
}


/*
 * ɥץѥե빽¤Τν
 * parameters : process
 * return : 0 or error number
 */
int init_file_struct(PROC *proc)
{
	FILE_STRUCT *fstruct;

	if((fstruct=kmalloc(sizeof(FILE_STRUCT)))==NULL)return -ENOMEM;
	memset(fstruct,0,sizeof(FILE_STRUCT));
	fstruct->current_dir=rootEnt;
	proc->file_struct=fstruct;

	return 0;
}

/*
 * եǥץν
 */
void initFd(VENTRY *vent, const int form, const int amode, F_DSC *m_fd)
{
	m_fd->form = form;
	m_fd->accessMode = amode;
	m_fd->f_pos = 0;
	m_fd->refCount = 1;
	m_fd->vent = vent;
}

/*
 * եǥץϿ롣
 * return : number of file descriptor
 */
int setFd(F_DSC *fd, const int flag)
{
	int num;
	FILE_STRUCT *fstruct = get_current_task()->file_struct;

	num = getEmptyFdNum(fstruct, fstruct->next_fd);
	fstruct->next_fd = (num + 1 < _POSIX_OPEN_MAX)? num + 1 : 0;
	++fstruct->fd_count;
	fstruct->fd[num] = fd;
	fstruct->flag[num] = flag;

	return num;
}

/*
 * File descriptor롣
 * return : file descriptor or NULL
 */
F_DSC *getFd(int num)
{
	if (_POSIX_OPEN_MAX <= (uint)num){
		return NULL;
	}
	return ((FILE_STRUCT*)get_current_task()->file_struct)->fd[num];
}

/*
 * ץե礫
 * return : FALSE or TRUE
 */
int isFileOpenMax()
{
	FILE_STRUCT *fstruct = get_current_task()->file_struct;

	return  (_POSIX_OPEN_MAX <= fstruct->fd_count)? TRUE : FALSE;
}


/************************************************************************************
 *
 * ƥॳ
 *
 ************************************************************************************/

//================================== PROTECTED ==========================================

/*
 * pathȥåץǥ쥯ȥ롣
 * parameters : ѥɥ쥹ݥ
 * return : Ȳۥǥ쥯ȥꥨȥ꡼
 */
STATIC INLINE VENTRY *getTopVdir(const char *path, const char **o_path)
{
	char *p = (char*)path;

	if(*p == '/'){
		while(*++p == '/');
		*o_path = p;
		return rootEnt;
	}
	else{
		*o_path = p;
		return getCurrentDir();
	}
}

/*
 * ѥ򸡺
 * return : error number
 */
STATIC int lookup(
	const char *i_path, 	// ѥ
	uint *o_inodeNum)		// inodeֹ
{
	char linkPath[PATH_MAX + 1];
	const char *path;
	uint inodeNum;
	SEARCH_PARAM start;
	SEARCH_PARAM end;
	int error;

	/* ѥĹθ */
	if (PATH_MAX <= strlen(i_path)){
		return -ENAMETOOLONG;
	}
	if(*i_path=='\0'){
		return -ENOENT;
	}

	// ۥեõ
	start.vent = getTopVdir(i_path, &start.path);
	error = searchPath(start.path, start.vent, &end.path, &end.vent);
	for (;;) {
		switch (error){
		case NOERR:		// ۥե˺ߤ
			if ((end.vent->type == S_IFDIR) && (end.vent->openCount == 0)) {		// inode֥Ȥ̵
				error = end.vent->fs->lookup(end.path, end.vent->din, end.vent->inodeObj, &inodeNum, &path);
			}
			break;
		case -ENOENT:	// ۥե̵
			error = end.vent->fs->lookup(end.path, end.vent->din, end.vent->inodeObj, &inodeNum, &path);
			break;
		default:
			return error;
		}

		if (error == NOERR) {
			*o_inodeNum = inodeNum;
			break;
		}
		else if (error == ESYMBOLIC) {
			void *inodeObj;
			int len;
			int type;

			// ܥåեɤ
			error = end.vent->fs->open(end.vent->din, inodeNum, O_RDONLY, &inodeObj, &type);
			if (error != NOERR) {
				break;
			}
			len = end.vent->fs->read(end.vent->din, inodeObj, linkPath, PATH_MAX, 0);
			if (len < 0) {
				error = len;
				break;
			}
			linkPath[len] = '\0';
			
			// ѥꤹ
			
			
			// ۥեõ
			start.vent = getTopVdir(linkPath, &start.path);
			error = searchPath(start.path, start.vent, &end.path, &end.vent);
		}
		else {
			break;
		}
	}

	return error;
}

/*
 * ǥХ򥪡ץ󤹤롣
 * return : error number
 */
STATIC int openDevice(const char *devPath, void **o_devInode)
{
	enum{DEV_NAME_SIZE = 5};		// "/dev/"Υ
	uint inodeNum;
	int type;
	int rest;

	// ǥХץ
	if (cmpPathNull("/dev", devPath) == NULL){
		return -EINVAL;
	}
	rest = devEnt->mount->fs->lookup(devPath + DEV_NAME_SIZE, 0, 0, &inodeNum, NULL);
	if (rest < 0){
		return rest;
	}
	rest = devEnt->mount->fs->open(0, inodeNum, O_RDWR, o_devInode, &type);
	if (rest < 0){
		return rest;
	}

	return NOERR;
}

/*
 * ۥե륷ƥ򸡺ƥޥȤƤե륷ƥ򥢥ޥȤ롣
 */
STATIC void umountAllFs(VENTRY *ventry)
{
	VENTRY *vent = ventry;

	do{
		if (vent->child != NULL){
			umountAllFs(vent->child);
		}
		/* umouont file system. */
		if (vent->mount != NULL){
			vent->mount->fs->umount(vent->mount->din, vent->mount->inodeObj);
		}
	}while ((vent = vent->next) != ventry);
}

STATIC int chattr(FS *fs, int din, void *inodeObj, int cmd, uint attr)
{
	int error;
	PROC *proc;
	struct stat stat;

	/* Change attribute. */
	proc=get_current_task();
	if ((error = fs->stat(din, inodeObj, &stat)) < 0){
		return error;
	}
	switch(cmd){
		case CHATTR_MODE:
			if ((proc->uid == 0) || (proc->uid == stat.st_uid)){
				error = fs->chattr(din, inodeObj, attr, stat.st_uid, stat.st_gid, stat.st_atime, stat.st_mtime);
			}
			else{
				error=-EPERM;
			}
			break;
		case CHATTR_UID:
			if(proc->uid == 0){
				error = fs->chattr(din, inodeObj, stat.st_mode, attr, stat.st_gid, stat.st_atime, stat.st_mtime);
			}
			else{
				error=-EPERM;
			}
			break;
		case CHATTR_GID:
			if (((proc->uid == 0) || (proc->uid == stat.st_uid)) && (attr == proc->gid)){
				error = fs->chattr(din, inodeObj, stat.st_mode, stat.st_gid, attr, stat.st_atime, stat.st_mtime);
			}
			else{
				error=-EPERM;
			}
			break;
		case CHATTR_UTIME:
			if ((proc->uid == 0) || (proc->uid == stat.st_uid)){
				struct utimbuf *ut = (struct utimbuf*)attr;
				uint time;

				if (ut == NULL){
					time = sys_time(NULL);
					error = fs->chattr(din, inodeObj, stat.st_mode, stat.st_gid, attr, time, time);
				}
				else{
					error = fs->chattr(din, inodeObj, stat.st_mode, stat.st_gid, attr, ut->actime, ut->modtime);
				}
			}
			else{
				error=-EPERM;
			}
			break;
	}

	return error;
}

/*
 * fd_set¤ƱΤOR黻
 */
STATIC void orFdset(const int lastFdNum, fd_set *dstFdSet, fd_set *srcFdSet)
{
	int last = (lastFdNum < FD_SETSIZE)? lastFdNum / CHAR_BIT + 1 : FD_BITMAP_LEN;
	int i;
	
	for (i = 0; i < last; ++i){
		dstFdSet->bitmap[i] |= srcFdSet->bitmap[i];
	}
}

/*
 * fd_set¤ΤΥǥץֹʾꤵƤǥץֹ֤
 * return : number of discriptor or -1 = ʤ
 */
STATIC int getFdFromFdset(const int lastFdNum, const int firstFd, const fd_set *fdSet)
{
	int fdNum;
	int array;
	int last = (lastFdNum < FD_SETSIZE)? lastFdNum / CHAR_BIT + 1 : FD_BITMAP_LEN;
	
	for (array = firstFd / CHAR_BIT, fdNum = firstFd % CHAR_BIT; array < last; ++array){
		if (fdSet->bitmap[array] != 0){
			for (; fdNum < CHAR_BIT; ++fdNum){
				if (fdSet->bitmap[array] & (1 << fdNum)){
					return array * CHAR_BIT + fdNum;
				}
			}
		}
		fdNum = 0;
	}
	return -1;
}

//================================== PUBLIC =============================================

/*
 * ¹ԥե륪ץ̾Υץδάؿ
 * return : 0 or error number
 */
int exec_open(const char *path, EXEC_FD *fd)
{
	uint inodeNum;
	int fileType;
	SEARCH_PARAM start, end;
	int error;

	if (*path == '\0'){
		return -ENOENT;
	}
	/* ѥĹθ */
	if (PATH_MAX <= strlen(path)){
		return -ENAMETOOLONG;
	}

	// ۥե륷ƥλȳ
	refVfs();

	start.vent = getTopVdir(path, &start.path);
	error = searchPath(start.path, start.vent, &end.path, &end.vent);
	switch (error){
	case NOERR:		// ۥեˤ
		if (end.vent->type != S_IFREG){
			error = -EACCES;
			goto END;
		}
		fd->din = end.vent->din;
		fd->inodeObj = end.vent->inodeObj;
		fd->fs = end.vent->fs;
		break;
	case -ENOENT:{	// ۥե̵
		void *inodeObj;

		error = start.vent->fs->lookup(end.path, end.vent->din, end.vent->inodeObj, &inodeNum, NULL);
		if (error < 0){
			goto END;
		}
		error = end.vent->fs->open(end.vent->din, inodeNum, O_RDONLY, &inodeObj, &fileType);
		if (error < 0){
			goto END;
		}
		if (fileType != S_IFREG){
			error = -EACCES;
			goto END;
		}	
		fd->din = end.vent->din;
		fd->inodeObj = inodeObj;
		fd->fs = end.vent->fs;
		break;
	}
	default:
		goto END;
	}
	error = NOERR;
END:
	// ۥե륷ƥλȽλ
	refEndVfs();

	return error;
}

/*
 * ¹ԥեɤ߹ߡ"sys_read"δάؿ
 * parameters : open file struct,buffer,read size
 * return : read byte or error=-1
 */
int exec_read(EXEC_FD *fd, void *buf, size_t size)
{
	return fd->fs->read(fd->din, fd->inodeObj, buf, size, fd->offset);
}

/*
 * ¹ԥեΥ
 */
int exec_close(EXEC_FD *fd)
{
	int error;
	
	// ۥե륷ƥλȳ
	refVfs();
	{
		error = fd->fs->close(fd->din, fd->inodeObj);
	}
	refEndVfs();
	
	return error;
}

int sys_mount(const char *devPath, const char *fsName, const char *path)
{
	void *mountRootObj;
	int mountDin;
	FS *fs;
	SEARCH_PARAM start, end;
	VENTRY *vent;
	int error;

	/* ѥĹθ */
	if (PATH_MAX <= strlen(devPath)){
		return -ENAMETOOLONG;
	}
	if (PATH_MAX <= strlen(path)){
		return -ENAMETOOLONG;
	}

	/* Search file system. */
	fs = search_fs(fsName);
	if (fs == NULL){
		return -ENODEV;
	}

	/* ǥХץ */
	error = openDevice(devPath, (void**)&mountDin);
	if (error < 0){
		return error;
	}

	// ޥȴؿƤӽФ
	error = fs->mount(mountDin, &mountRootObj);
	if (error < 0){
		goto ERR;
	}

	// ۥե륷ƥι
	modifyVfs();

	start.vent = getTopVdir(path,&start.path);
	error = searchPath(start.path, start.vent, &end.path, &end.vent);
	switch (error){
	case NOERR:		// ۥեˤ
		vent = end.vent;
		break;
	case -ENOENT:{	// ۥե̵
		uint inodeNum;
		void *rootInodeObj;
		int type;

		// ǥ쥯ȥ꤬¸ߤ뤫
		error = start.vent->fs->lookup(start.path, start.vent->din, start.vent->inodeObj, &inodeNum, NULL);
		if (error  < 0){
			goto ERR2;
		}

		// ե륿פγǧ
		error = start.vent->fs->open(start.vent->din, inodeNum, O_RDWR, &rootInodeObj, &type);
		if (error  < 0){
			goto ERR2;
		}
		start.vent->fs->close(start.vent->din, rootInodeObj);
		if ((type & S_IFMT) != S_IFDIR){
			error = -ENOTDIR;
			goto ERR2;
		}

		// ۥǥ쥯ȥκ
		error = creatEntry(end.path, end.vent, &vent);
		if (error != NOERR){
			goto ERR2;
		}
		vent->type = type;

		break;
	}
	default:
		goto ERR2;
	}

	/* ޥȤ벾ۥǥ쥯ȥ롣 */
	error = mountEntry(vent, fs, mountDin, mountRootObj);
	if (error != NOERR){
		goto ERR2;
	}

	// ۥե륷ƥιλ
	modifyEndVfs();

	return NOERR;
ERR2:
	// ۥե륷ƥιλ
	modifyEndVfs();
ERR:
	fs->umount(mountDin, mountRootObj);
	return error;
}

int sys_umount(const char *path)
{
	SEARCH_PARAM start, end;
	int error;

	/* ѥĹθ */
	if (PATH_MAX <= strlen(path)){
		return -ENAMETOOLONG;
	}

	// ۥե륷ƥι
	modifyVfs();

	start.vent = getTopVdir(path,&start.path);
	error = searchPath(start.path, start.vent, &end.path, &end.vent);
	switch (error){
	case NOERR:		// ۥեˤ
		if (end.vent->type == S_IFDIR){
			break;
		}
		//
	case -ENOENT:	// ۥե̵
		error = -ENOENT;
		goto END;
	default:
		goto END;
	}

	/* ޥȡ */
	error = umountEntry(end.vent);
END:
	// ۥե륷ƥιλ
	modifyEndVfs();

	return error;
}

int sys_lseek(int fdNum, int offset, int whence)
{
	int rest;
	struct stat fstat;
	F_DSC *fd;
	VENTRY *vent;

	fd = getFd(fdNum);
	if (fd == NULL){
		return -EBADF;
	}

	vent = fd->vent;
	vent->fs->stat(vent->din, vent->inodeObj, &fstat);
	switch(whence){
		case SEEK_CUR:
			rest = fd->f_pos + offset;
			break;
		case SEEK_SET:
			rest = offset;
			break;
		case SEEK_END:
			rest = fstat.st_size - offset;
			break;
		default:
			return -EINVAL;
	}
	if (rest < 0){
		return -EINVAL;
	}

	return fd->f_pos = rest;
}

/*
 *ԸƤFIFOեǤΥץ
 * return : file descriptor number or error number
 */
int sys_open(const char *path, const int flag, const mode_t i_mode)
{
	uint inodeNum = UINT_MAX;	//UINT_MAXʤlookup()ꤵƤʤΤopen()ʤ
	void *inodeObj;
	int type;
	SEARCH_PARAM start, end;
	VENTRY *vent;
	F_DSC *fd;
	int fdNum;
	int error;

	// ѥĹθ
	if (PATH_MAX <= strlen(path)){
		return -ENAMETOOLONG;
	}
	if(*path=='\0'){
		return -ENOENT;
	}

	// ץΥå
	if (isFileOpenMax() == TRUE){
		return -EMFILE;
	}

	// ۥե륷ƥι
	modifyVfs();

	start.vent = getTopVdir(path, &start.path);
	error = searchPath(start.path, start.vent, &end.path, &end.vent);
	switch (error){
	case NOERR:		// ۥեˤ
		if ((flag & (O_CREAT | O_EXCL)) == (O_CREAT | O_EXCL)) {
			error = -EEXIST;
			goto ERR;
		}

		if (end.vent->type == S_IFDIR){
			if (end.vent->openCount == 0){
				error = end.vent->fs->lookup(end.path, end.vent->din, end.vent->inodeObj, &inodeNum, NULL);
				if (error < 0){
					goto ERR;
				}
			}
		}

		// ۥեλȿ䤹
		vent = end.vent;
		openVent(vent);
		break;
	case -ENOENT:	// ۥե̵
		error = end.vent->fs->lookup(end.path, end.vent->din, end.vent->inodeObj, &inodeNum, NULL);
		switch (error) {
		case NOERR:
			if ((flag & (O_CREAT | O_EXCL)) == (O_CREAT | O_EXCL)) {
				error = -EEXIST;
			}
			break;
		case -ENOENT:
			if (flag & O_CREAT) {
				mode_t mode = i_mode;

				mode |= (flag & O_FIFO) ? S_IFIFO : S_IFREG | S_IRUSR | S_IWUSR;
				error = end.vent->fs->creat(end.path, end.vent->din, end.vent->inodeObj, mode, &inodeNum);
			}
			break;
		default:
			break;
		}
		if (error != NOERR) {
			goto ERR;
		}

		// ۥեκ
		error = creatEntry(end.path, end.vent, &vent);
		if (error != NOERR) {
			goto ERR;
		}
		break;
	default:
		goto ERR;
	}

	// եǥץν
	fd = kmalloc(sizeof(F_DSC));
 	if (fd == NULL){
		error = -ENOMEM;
		goto ERR2;
	}
	initFd(vent, FD_FORM_LOCALFILE, flag, fd);
	
	// ե륪ץ
	if (inodeNum != UINT_MAX){
		error = end.vent->fs->open(vent->din, inodeNum, flag, &inodeObj, &type);
		if (error < 0){
			goto ERR3;
		}
		vent->inodeObj = inodeObj;
		vent->type = type;
	}

	/* ǥ쥯ȥɤ߹ߤΤ */
	if (vent->type == S_IFDIR){
		if (flag != O_RDONLY){
			error = -EISDIR;
			goto ERR4;
		}
	}
	// TRUNCATEǥץ
	if (flag & O_TRUNC){
		if ((vent->type & S_IFMT) == S_IFREG){
			error = vent->fs->ioctl(vent->din, vent->inodeObj, I_TRUNC, NULL, 0);
			if (error < 0){
				goto ERR4;
			}
		}
	}
	// FIFOեǤΥץ
	if (flag & O_FIFO){
		vent->type = S_IFIFO;
	}

	// եǥץϿ
	fdNum = setFd(fd, FD_CLOEXEC);

	// APPENDǥץ
	if (flag & O_APPEND){
		sys_lseek(fdNum, 0, SEEK_END);
	}

	// ۥե륷ƥιλ
	modifyEndVfs();

	return fdNum;

ERR4:
	vent->fs->close(vent->din, vent->inodeObj);
ERR3:
	kfree(fd);
ERR2:
	closeVent(vent);
ERR:
	// ۥե륷ƥιλ
	modifyEndVfs();

	return error;
}

int sys_close(int fdNum)
{
	F_DSC *fd;
	FILE_STRUCT *fstruct;
	int error;

	if ((fd = getFd(fdNum)) == NULL){
		return -EBADF;
	}
	fstruct = (FILE_STRUCT*)get_current_task()->file_struct;

	modifyVfs();
	{
		error = unlinkFd(fstruct, fdNum);
	}
	modifyEndVfs();

	return error;
}

int sys_read(int fdNum, void *buf, size_t size)
{
	F_DSC *fd;
	VENTRY *vent;
	int rest;

	/* Хåեå */
	if (checkMem(buf, size) == ERR) {
		return -EFAULT;
	}

	fd = getFd(fdNum);
	if (fd == NULL) {
		return -EBADF;
	}

	/* ⡼ɤΥå */
	if ((fd->accessMode & (O_RDONLY | O_RDWR)) == 0) {
		return -EBADF;
	}

	vent = fd->vent;
	if (vent->type == S_IFIFO) {
		rest = vent->fs->read(vent->din, vent->inodeObj, buf, size, 0);
	}
	else{
		rest = vent->fs->read(vent->din, vent->inodeObj, buf, size, fd->f_pos);
		if (0 < rest) {
			fd->f_pos += rest;
		}
	}

	return rest;
}

int sys_write(int fdNum, void *buf, size_t size)
{
	F_DSC *fd;
	VENTRY *vent;
	int rest;

	/* Хåեå */
	if (checkMem(buf, size) == -1){
		return -EFAULT;
	}

	fd = getFd(fdNum);
	if (fd == NULL){
		return -EBADF;
	}

	/* ⡼ɤΥå */
	if ((fd->accessMode & (O_WRONLY | O_RDWR)) == 0){
		return -EBADF;
	}

	/* writeؿƤӽФ */
	vent = fd->vent;
	if (vent->type == S_IFIFO){
		rest=vent->fs->write(vent->din, vent->inodeObj, buf, size, 0);
	}
	else{
		rest = vent->fs->write(vent->din, vent->inodeObj, buf, size, fd->f_pos);
		if (0 < rest){
			fd->f_pos += rest;
		}
	}

	return rest;
}

int sys_ioctl(int fdNum, int cmd, caddr_t parm, int fflag)
{
	F_DSC *fd;
	VENTRY *vent;

	if ((fd = getFd(fdNum)) == NULL){
		return -EBADF;
	}
	vent = fd->vent;

	return vent->fs->ioctl(vent->din, vent->inodeObj, cmd, parm, fflag);
}

int sys_mkdir(const char *path, int mode)
{
	SEARCH_PARAM start,end;
	int error;

	if (*path == '\0'){
		return -ENOENT;
	}
	/* ѥĹθ */
	if (PATH_MAX <= strlen(path)){
		return -ENAMETOOLONG;
	}

	// ۥե륷ƥι
	modifyVfs();

	start.vent = getTopVdir(path,&start.path);
	error = searchPath(start.path, start.vent, &end.path, &end.vent);
	switch (error){
	case NOERR:		// ۥեˤ
		error = -EEXIST;
		goto END;
	case -ENOENT:
		error = end.vent->fs->mkdir(end.path, end.vent->din, end.vent->inodeObj, mode);
	default:
		goto END;
	}

END:
	// ۥե륷ƥιλ
	modifyEndVfs();

	return error;
}

int sys_unlink(const char *path)
{
	SEARCH_PARAM start, end;
	int error;

	if (*path == '\0')
		return -ENOENT;
	/* ѥĹθ */
	if (PATH_MAX <= strlen(path)){
		return -ENAMETOOLONG;
	}

	// ۥե륷ƥι
	modifyVfs();

	start.vent = getTopVdir(path,&start.path);
	error = searchPath(start.path, start.vent, &end.path, &end.vent);
	switch (error){
	case NOERR:		// ۥեˤ
		unlinkVent(end.vent);
		error = -EBUSY;
		break;
	case -ENOENT:	// ۥե̵
		error = end.vent->fs->delete(end.path, end.vent->din, end.vent->inodeObj, S_IFREG | S_IFLNK);
		break;
	default:
		break;
	}

	// ۥե륷ƥιλ
	modifyEndVfs();

	return error;
}

int sys_rmdir(const char *path)
{
	SEARCH_PARAM start, end;
	int error;

	if (*path == '\0')
		return -ENOENT;
	/* ѥĹθ */
	if (PATH_MAX <= strlen(path)){
		return -ENAMETOOLONG;
	}

	// ۥե륷ƥι
	modifyVfs();

	start.vent = getTopVdir(path,&start.path);
	error = searchPath(start.path, start.vent, &end.path, &end.vent);
	switch (error){
	case NOERR:		// ۥեˤ
		unlinkVent(end.vent);
		error = -EBUSY;
		break;
	case -ENOENT:	// ۥե̵
		error = end.vent->fs->delete(end.path, end.vent->din, end.vent->inodeObj, S_IFDIR);
		break;
	default:
		break;
	}

	// ۥե륷ƥιλ
	modifyEndVfs();

	return error;
}

int sys_opendir(const char *path)
{
	uint block;
	void *inodeObj;
	int index;
	SEARCH_PARAM start, end;
	VENTRY *vent;
	DIR_STRAEM *dir;
	int error;

	if (*path == '\0'){
		return -ENOENT;
	}
	/* ѥĹθ */
	if (PATH_MAX <= strlen(path)){
		return -ENAMETOOLONG;
	}

	// ۥե륷ƥι
	modifyVfs();

	start.vent = getTopVdir(path, &start.path);
	error = searchPath(start.path, start.vent, &end.path, &end.vent);
	switch (error){
	case NOERR:		// ۥեˤ
		if(end.vent->type != S_IFDIR){
			error = -ENOTDIR;
			goto END;
		}
		openVent(end.vent);
		vent = end.vent;
		error = end.vent->fs->opendir(end.path, vent->din, vent->inodeObj, &inodeObj, &block, &index);
		if (error < 0){
			goto END;
		}
		break;
	case -ENOENT:	// ۥե̵
		if(end.vent->type == S_IFDIR){
			error = end.vent->fs->opendir(end.path, end.vent->din, end.vent->inodeObj, &inodeObj, &block, &index);
			if (error < 0){
				goto END;
			}

			/* ۥȥ꡼κ */
			error = creatEntry(end.path, end.vent, &vent);
			if(error != NOERR){
				end.vent->fs->close(end. vent->din, end.vent->inodeObj);
				goto END;
			}
			vent->inodeObj = inodeObj;
			vent->type = S_IFDIR;
		}
		else{
			error = -ENOENT;
			goto END;
		}
		break;
	default:
		goto END;
	}

	/* ǥ쥯ȥꥹȥ꡼κ */
	dir = kmalloc(sizeof(DIR_STRAEM));
	if (dir == NULL){
		error = -ENOMEM;
		goto END;
	}
	initFd(vent, FD_FORM_DIRSTREAM, 0, (F_DSC*)dir);
	error = setFd((F_DSC*)dir, FD_CLOEXEC);
	dir->block = block;
	dir->index = index;
END:
	// ۥե륷ƥιλ
	modifyEndVfs();

	return error;
}

int sys_readdir(int dirNum, char *name)
{
	DIR_STRAEM *dir;
	VENTRY *vent;

	dir = (DIR_STRAEM*)getFd(dirNum);
	if (dir == NULL){
		return -EBADF;
	}
	if (dir->fd.form != FD_FORM_DIRSTREAM){
		return -EBADF;
	}

	vent = dir->fd.vent;

	return vent->fs->readdir(vent->din, vent->inodeObj, &dir->block, &dir->index, name);
}

int sys_rewinddir(int dirNum)
{
	DIR_STRAEM *dir;

	dir = (DIR_STRAEM*)getFd(dirNum);
	if(dir == NULL){
		return -EBADF;
	}
	if (dir->fd.form != FD_FORM_DIRSTREAM){
		return -EBADF;
	}
	dir->index = 0;

	return 0;
}

int sys_rename(const char *old_path,const char *new_path)
{
	SEARCH_PARAM old_start, old_end;
	SEARCH_PARAM new_start, new_end;
	int error;

	if (strcmp(old_path, new_path) == 0){
		return 0;
	}

	// ۥե륷ƥι
	modifyVfs();

	old_start.vent = getTopVdir(old_path, &old_start.path);
	error = searchPath(old_start.path, old_start.vent, &old_end.path, &old_end.vent);
	switch (error){
	default:
		goto END;
	}

	/* եõ */
	new_start.vent = getTopVdir(new_path, &new_start.path);
	error = searchPath(new_start.path, new_start.vent, &new_start.path, &new_start.vent);
	switch (error){
	default:
		goto END;
	}

	if (old_end.vent->din == new_end.vent->din){
		if ((*old_end.path != '\0') && (*new_end.path != '\0')){
			error = old_end.vent->fs->
				rename(old_end.vent->din, old_end.vent->inodeObj, old_end.path, new_end.vent->inodeObj, new_end.path);
		}
		else{
			error = -EBUSY;
		}
	}
	else{
		error = -EXDEV;
	}

END:
	// ۥե륷ƥιλ
	modifyEndVfs();

	return error;
}

int sys_chdir(const char *path)
{
	void *inodeObj;
	int error = 0;
	SEARCH_PARAM start,end;
	VENTRY *vent;
	int dummy1;
	uint dummy2;

	if (*path == '\0'){
		return -ENOENT;
	}
	/* ѥĹθ */
	if (PATH_MAX <= strlen(path)){
		return -ENAMETOOLONG;
	}

	// ۥե륷ƥι
	modifyVfs();

	/* ǥ쥯ȥõ */
	start.vent = getTopVdir(path,&start.path);
	error = searchPath(start.path, start.vent, &end.path, &end.vent);
	switch (error){
	case NOERR:		// ۥեˤ
		if (end.vent->type == S_IFDIR){
			if (end.vent->openCount == 1){
				error = end.vent->fs->opendir(end.path, end.vent->din, end.vent->inodeObj, &inodeObj, &dummy1, &dummy2);
				if (error < 0){
					goto ERR;
				}
				end.vent->inodeObj = inodeObj;
			}
			openVent(end.vent);
			vent = end.vent;
		}
		else{
			error = -ENOTDIR;
			goto ERR;
		}
		break;
	case -ENOENT:	// ۥե̵
		if (end.vent->type == S_IFDIR){
			error = end.vent->fs->opendir(end.path,end.vent->din,end.vent->inodeObj,&inodeObj,&dummy1,&dummy2);
			if (error >= 0){
				error = creatEntry(end.path, end.vent, &vent);
				if (error != NOERR){
					end.vent->fs->close(end.vent->din,end.vent->inodeObj);
					goto ERR;
				}
				vent->inodeObj = inodeObj;
				vent->type = S_IFDIR;
			}
			else{
				goto ERR;
			}
		}
		else{
			error = -ENOTDIR;
			goto ERR;
		}
		break;
	default:
		goto ERR;
	}

	/* ե빽¤Τ˿ȥǥ쥯ȥϿ */
	setCurrentDir(vent);

	// ۥե륷ƥιλ
	modifyEndVfs();

	return 0;
ERR:
	// ۥե륷ƥιλ
	modifyEndVfs();

	return error;
}

int sys_fchdir(int fdNum)
{
	F_DSC *fd;

	fd = getFd(fdNum);
	if (fd == NULL){
		return -EBADF;
	}
	if (fd->vent->type != S_IFDIR){
		return -ENOTDIR;
	}

	openVent(fd->vent);
	
	/* ե빽¤Τ˿ȥǥ쥯ȥϿ */
	setCurrentDir(fd->vent);

	return 0;
}

int sys_getcwd(char *buf, size_t size)
{
	int len;
	VENTRY *vent;

	if (size == 0){
		return -EINVAL;
	}

	/* Хåեå */
	if (checkMem(buf, size) == ERR){
		return -EFAULT;
	}

	vent = getCurrentDir();
	// ۥե륷ƥλȳ
	refVfs();
	{
		len = copyCurrentPath(vent, size, buf);
	}
	refEndVfs();
	if (len == ERR){
		return -ERANGE;
	}
	if (len == 1){
		buf[1] = '\0';
	}
	else{
		buf[len-1] = '\0';
	}

	return 0;
}

/*
 * pathNULLʤfile descriptorõ
 */
int sys_stat(int fdNum, const char *path, struct stat *m_stat)
{
	F_DSC *fd;
	SEARCH_PARAM start, end;
	int error;

	if (checkMem(m_stat,sizeof(struct stat)) == ERR){
		return -EFAULT;
	}
	if (path == NULL){
		if ((fd = getFd(fdNum)) == NULL){
			return -EBADF;
		}
		m_stat->st_crtpos = fd->f_pos;

		return fd->vent->fs->stat(fd->vent->din, fd->vent->inodeObj, m_stat);
	}
	else{
		/* ѥĹθ */
		if (PATH_MAX <= strlen(path)){
			return -ENAMETOOLONG;
		}
		m_stat->st_crtpos = 0;

		// ۥե륷ƥλȳ
		refVfs();

		start.vent = getTopVdir(path, &start.path);
		error = searchPath(start.path, start.vent, &end.path, &end.vent);
		switch (error){
		case NOERR:		// ۥեˤ
			error = end.vent->fs->stat(end.vent->din, end.vent->inodeObj, m_stat);
			break;
		case -ENOENT:{	// ۥե̵
			uint inodeNum;
			void *inodeObj;
			int type;

			error = end.vent->fs->lookup(end.path, end.vent->din, end.vent->inodeObj, &inodeNum, NULL);
			if (error < 0){
				goto END;
			}
			error = end.vent->fs->open(end.vent->din, inodeNum, O_RDONLY, &inodeObj ,&type);
			if (error < 0){
				goto END;
			}
			error = end.vent->fs->stat(end.vent->din, inodeObj, m_stat);
			end.vent->fs->close(end.vent->din, inodeObj);
			break;
		}
		default:
			break;
		}
END:
		// ۥե륷ƥλȽλ
		refEndVfs();
	}
/************** ̤ **************************/
	m_stat->st_flags = 0;
/*****************************************************/

	return error;
}

/*
 * 롼ȥե륷ƥ򥢥ޥȤ롣
 */
int sys_umount_root()
{
	umountAllFs(rootEnt);
	rootEnt->fs->umount(rootEnt->din, rootEnt->inodeObj);

	return 0;
}


int sys_fcntl(int fdNum, int cmd, uint arg)
{
	switch(cmd){
		case F_DUPFD:{
			if (isFileOpenMax() == TRUE){
				return -EMFILE;
			}
			if (getFd(fdNum) == NULL){			/* File descriptorΥå */
				return -EBADF;
			}
			return copyFd(fdNum,arg);			/* Copy File descriptor. */
		}
		case F_GETFD:
			return getFdFlag(fdNum);
		case F_SETFD:
			return setFdFlag(fdNum,arg);
		case F_GETFL:
			return getOpenFlag(fdNum);
		case F_SETFL:
			return setOpenFlag(fdNum, arg);
		case F_GETLK:
		case F_SETLK:
		case F_SETLKW:
		case F_GETOWN:
		case F_SETOWN:
		default:
			return -ENOTSUP;
	}
}

int sys_chattr(const char *path, int cmd, uint attr)
{
	SEARCH_PARAM start,end;
	int error;

	/* ѥĹθ */
	if (PATH_MAX <= strlen(path)){
		return -ENAMETOOLONG;
	}

	// ۥե륷ƥι
	modifyVfs();

	start.vent = getTopVdir(path, &start.path);
	error = searchPath(start.path, start.vent, &end.path, &end.vent);
	switch (error){
	case NOERR:		// ۥեˤ
		error = chattr(end.vent->fs, end.vent->din, end.vent->inodeObj, cmd, attr);
		break;
	case -ENOENT:{	// ۥե̵
		uint inodeNum;
		void *inodeObj;
		int type;

		error = end.vent->fs->lookup(end.path, end.vent->din, end.vent->inodeObj, &inodeNum, NULL);
		if (error < 0){
			goto ERR;
		}
		error = end.vent->fs->open(end.vent->din, inodeNum, O_RDWR, &inodeObj, &type);
		if (error < 0){
			goto ERR;
		}
		error = chattr(end.vent->fs, end.vent->din, inodeObj, cmd, attr);
		end.vent->fs->close(end.vent->din, inodeObj);
		break;
	}
	default:
		break;
	}
ERR:
	// ۥե륷ƥιλ
	modifyEndVfs();

	return error;
}

int sys_fchattr(int fdNum, int cmd, uint attr)
{
	F_DSC *fd;
	int error;

	if ((fd = getFd(fdNum)) == NULL){
		return -EBADF;
	}
	if (fd->vent->type != S_IFREG){
		return -EBADF;
	}

	// ۥե륷ƥλȳ
	refVfs();

	error = chattr(fd->vent->fs, fd->vent->din, fd->vent->inodeObj, cmd, attr);

	// ۥե륷ƥλȽλ
	refEndVfs();

	return error;
}

int sys_link(const char *dst_path, const char *src_path)
{
	void *dstInodeObj;
	int din;
	SEARCH_PARAM dst_start, src_start, dst_end, src_end;
	int error;

	/* ѥĹθ */
	if (PATH_MAX <= strlen(dst_path)){
		return -ENAMETOOLONG;
	}
	if (PATH_MAX <= strlen(src_path)){
		return -ENAMETOOLONG;
	}

	// ۥե륷ƥλȳ
	refVfs();

	/* dst_pathΥե򳫤 */
	dst_start.vent = getTopVdir(dst_path, &dst_start.path);
	error = searchPath(dst_start.path, dst_start.vent, &dst_end.path, &dst_end.vent);
	switch (error){
	case NOERR:		// ۥեˤ
		if (dst_end.vent->type == S_IFDIR){
			error = -EISDIR;
			goto END;
		}
		din = dst_end.vent->din;
		dstInodeObj = dst_end.vent->inodeObj;
		break;
	case -ENOENT:{	// ۥե̵
		uint inodeNum;
		void *inodeObj;
		int type;

		din = dst_end.vent->din;
		error = dst_end.vent->fs->lookup(dst_end.path, din, dst_end.vent->inodeObj, &inodeNum, NULL);
		if (error < 0){
			goto END;
		}
		error = dst_end.vent->fs->open(din, inodeNum, O_RDONLY, &inodeObj, &type);
		if (error < 0){
			goto END;
		}
		dstInodeObj = inodeObj;
		break;
	}
	default:
		goto END;
	}

	/* src_pathΥե򸡺롣 */
	src_start.vent = getTopVdir(src_path, &src_start.path);
	error = searchPath(src_start.path, src_start.vent, &src_end.path, &src_end.vent);
	switch (error){
	case NOERR:		// ۥեˤ
		error = -EEXIST;
		break;
	case -ENOENT:	// ۥե̵
		if (src_end.vent->din != din){
			error=-EXDEV;
		}
		else{
			error = src_end.vent->fs->link(din, dstInodeObj, src_end.vent->inodeObj, src_end.path);
		}
		break;
	default:
		break;
	}

	if (*dst_end.path != '\0'){
		dst_end.vent->fs->close(din, dstInodeObj);
	}
END:
	// ۥե륷ƥλȽλ
	refEndVfs();

	return error;
}

/*
 * Хѥꤹ
 */
int sys_symlink(char *linkPath, const char *srcPath)
{
	SEARCH_PARAM start, end;
	int error;
	
	/* ѥĹθ */
	if (PATH_MAX <= strlen(srcPath)){
		return -ENAMETOOLONG;
	}
	if (PATH_MAX <= strlen(linkPath)){
		return -ENAMETOOLONG;
	}

	if (*srcPath == '\0'){
		return -ENOENT;
	}

	// ۥե륷ƥι
	modifyVfs();

	/* srcPath򳫤 */
	start.vent = getTopVdir(srcPath, &start.path);
	error = searchPath(start.path, start.vent, &end.path, &end.vent);
	switch (error){
	case NOERR:		// ۥեˤ
		error = -EEXIST;
		goto END;
	case -ENOENT:	// ۥե̵
	{
		char targetPath[PATH_MAX + 1];
		const char *dummy;
		int len;

		// åȥѥХѥꤹ
		len = copyCurrentPath(getTopVdir(linkPath, &dummy), PATH_MAX, targetPath);
		if (len < 0) {
			error = len;
			goto END;
		}

		// ܥ󥯥ե

		break;
	}
	default:
		goto END;
	}

	error = 0;
END:
	// ۥե륷ƥιλ
	modifyEndVfs();

	return error;
}

int sys_readlink(const char *path, char *buf, size_t bufSize)
{
	void *inodeObj;
	SEARCH_PARAM start, end;
	struct stat stat;
	int size;
	int error = 0;

	/* ѥĹθ */
	if (PATH_MAX <= strlen(path)){
		return -ENAMETOOLONG;
	}
	/* ݥγǧ */
	if (checkMem(buf,bufSize) == ERR){
		return -EFAULT;
	}

	// ۥե륷ƥλȳ
	refVfs();

	/* ե򸡺 */
	start.vent = getTopVdir(path,&start.path);
	error = searchPath(start.path, start.vent, &end.path, &end.vent);
	switch (error){
	case NOERR:		// ۥեˤ
		inodeObj = end.vent->inodeObj;
		break;
	case -ENOENT:{	// ۥե̵
		uint inodeNum;
		int type;

		error = end.vent->fs->lookup(end.path, end.vent->din, end.vent->inodeObj, &inodeNum, NULL);
		if (error < 0){
			goto END;
		}
		error = end.vent->fs->open(end.vent->din, inodeNum, O_RDONLY, &inodeObj, &type);
		if (error < 0){
			goto END;
		}
		break;
	}
	default:
		goto END;
	}

	/* ե륿פǧ */
	error = end.vent->fs->stat(end.vent->din, inodeObj, &stat);
	if (error < 0){
		goto END2;
	}
	if ((stat.st_mode & S_IFLNK) == 0){
		error = -EINVAL;
		goto END2;
	}
	
	/* 󥯥ѥɤ */
	size = (stat.st_size <= bufSize)? stat.st_size : bufSize;
	error = end.vent->fs->read(end.vent->din, inodeObj, buf, size, 0);

END2:	
	if (*end.path != '\0'){
		end.vent->fs->close(end.vent->din, inodeObj);
	}
END:
	// ۥե륷ƥλȽλ
	refEndVfs();

	return error;
}

int sys_select(const int lastFdNum, fd_set *m_fdsRead, fd_set *m_fdsWrite, fd_set *m_fdsError, struct timeval *timeout)
{
	enum{
		FDS_LAST = 3,
	};
	int seconds;
	fd_set fdSetAll;
	fd_set *fdSet[] = {m_fdsRead, m_fdsWrite, m_fdsError};
	int pollEvents[] = {POLLIN, POLLOUT, POLLERR};
	int isFds;
	int rest;
	int i;

	if (lastFdNum < 0){
		return -EINVAL;
	}

	// ॢȻ֤
	if (timeout == NULL){
		// ʥȯޤǥ֥å
		seconds = -1;
	}
	else{
		// ̤ͼθ
		seconds = timeout->tv_sec + (500000 <= timeout->tv_usec);
	}

	// FD
	FD_ZERO(&fdSetAll);
	isFds = NO;
	for (i = 0; i < FDS_LAST; ++i){
		if (fdSet[i] != NULL){
			if (checkMem(fdSet[i], sizeof(fd_set)) == ERR){
				return -EFAULT;
			}
			isFds = YES;
			orFdset(lastFdNum, &fdSetAll, fdSet[i]);
		}
	}

	if (isFds == YES){
		// ٥Ȥθ
		for (;;){
			int fdNum;

			for (fdNum = 0; (fdNum = getFdFromFdset(lastFdNum, fdNum, &fdSetAll)) != -1;){
				int c_update = 0;
				int events;
				F_DSC *fd = getFd(fdNum);

				if (fd == NULL){
					return -EBADF;
				}
				// Ϥ٥Ȥ
				events = 0;
				for (i = 0; i < FDS_LAST; ++i){
					if (FD_ISSET(fdNum, fdSet[i])){
						events |= pollEvents[i];
					}
				}
				rest = fd->vent->fs->poll(fd->vent->inodeObj, events);
				
				// ǽFDΥ
				for (i = 0; i < FDS_LAST; ++i){
					if (rest & pollEvents[i]){
						if (fdSet[i] != NULL){
							if (FD_ISSET(fdNum, fdSet[i])){
								c_update += 1;
							}
						}
					}
				}
				// FD_SET˥٥Ȥ
				if (0 < c_update){
					for (i = 0; i < FDS_LAST; ++i){
						if (rest & pollEvents[i]){
							if (fdSet[i] != NULL){
								if (FD_ISSET(fdNum, fdSet[i])){
									FD_ZERO(fdSet[i]);
									FD_SET(fdNum, fdSet[i]);
								}
								else{
									FD_ZERO(fdSet[i]);
								}
							}
						}
					}
					return c_update;
				}
				fdNum += 1;
			}
			
			if (seconds == 0){
				// ǥץ̵
				for (i = 0; i < FDS_LAST; ++i){
					if (fdSet[i] != NULL){
						FD_ZERO(fdSet[i]);
					}
				}
				return 0;
			}
			
			// Ԥ
			sys_sleep(seconds);

			// ʥߤ
			doSignalRestart();
			if(isSigint() == TRUE){
				return -EINTR;
			}

			// ܤԤ֤򥯥ꥢ
			seconds = 0;
		}
	}
	else{
		// ʥԤ
		sys_sleep(seconds);

		return 0;
	}
}

int sys_poll(struct pollfd *ufds, const nfds_t nfds, const int miliSeconds)
{
	int seconds;
	int i, rest;

	if (RLIMIT_NOFILE < nfds){
		return -EINVAL;
	}

	// ꡼å
	if (checkMem(ufds, sizeof(ufds) * nfds) == -1){
		return -EFAULT;
	}
	for (i = 0; i < nfds; ++i){
		if (checkMem(&ufds[i], sizeof(struct pollfd)) == ERR){
			return -EFAULT;
		}
	}
	
	// ॢȻ֤
	if (miliSeconds < 0){
		seconds = -1;
	}
	else{
		seconds = miliSeconds / 1000 + (0 < miliSeconds % 1000);
	}

	// Init revents.
	for (i = 0; i < nfds; ++i){
		ufds[i].revents = 0;
	}

	// ٥Ȥθ
	for (;;){
		for (i = 0; i < nfds; ++i){
			F_DSC *fd  = getFd(ufds[i].fd);

			if (fd == NULL){
				ufds[i].revents |= POLLNVAL;
				return 1;
			}

			rest = fd->vent->fs->poll(fd->vent->inodeObj, ufds[i].events);
			if (rest < 0){
				ufds[i].revents |= POLLNVAL;
				return 1;
			}
			if (rest & (POLLIN | POLLRDNORM)){
				if (ufds[i].events & (POLLIN | POLLRDNORM)){
					ufds[i].revents |= rest;
				}
			}
			if (rest & (POLLOUT | POLLWRNORM)){
				if (ufds[i].events & (POLLOUT | POLLWRNORM)){
					ufds[i].revents |= rest;
				}
			}
			if (rest & (POLLERR | POLLHUP)){
				ufds[i].revents |= (rest & (POLLERR | POLLHUP));
			}
			if (ufds[i].revents != 0){
				return 1;
			}
		}

		if (seconds == 0){
			break;
		}

		/* sleep */
		sys_sleep(seconds);
		
		/* ʥߤ */
		doSignalRestart();
		if(isSigint() == TRUE){
			return -EINTR;
		}

		seconds = 0;
	}

	return 0;
}

int sys_getstatfs(struct statfs *buf, long bufsize, int flags)
{
	int cnt;
	int rest;

	if (buf == NULL){
		cnt = INT_MAX;
	}
	else{
		cnt = bufsize / sizeof(struct statfs);
	}
	
	refVfs();
	{
		rest = searchMount(buf, cnt);
	}
	refEndVfs();
	
	return rest;
}

int sys_statfs(const char *path, struct statfs *m_statfs)
{
	SEARCH_PARAM start, end;
	VENTRY *vent, *p;
	int error;

	if (*path == '\0'){
		return -ENOENT;
	}
	/* ѥĹθ */
	if (PATH_MAX <= strlen(path)){
		return -ENAMETOOLONG;
	}
	/* Хåեå */
	if (checkMem(m_statfs,sizeof(struct statfs)) == ERR){
		return -EFAULT;
	}

	// ۥե륷ƥλȳ
	refVfs();

	/* ǥ쥯ȥõ */
	start.vent = getTopVdir(path, &start.path);
	error = searchPath(start.path, start.vent, &end.path, &end.vent);
	switch (error){
	case NOERR:		// ۥեˤ
		if (end.vent->mount == NULL){
			vent = end.vent;
			for (p = vent; p != rootEnt; p = p->parent){
				if (p->din == vent->din){
					break;
				}
			}
			memcpy(m_statfs->f_mntonname, p->name, p->len);	/* directory on which mounted */
			m_statfs->f_mntonname[p->len] = '\0';
		}
		else{
			vent = end.vent->mount;
			memcpy(m_statfs->f_mntonname, vent->name, vent->len);	/* directory on which mounted */
			m_statfs->f_mntonname[vent->len] = '\0';
		}
		break;
	case -ENOENT:	// ۥե̵
		error = -ENOENT;
		goto ERR;
	default:
		goto ERR;
	}
	
	error = getFsstat(vent, m_statfs);

	// ۥե륷ƥλȽλ
	refEndVfs();
ERR:
	return error;
}

/************************************* ̤ ***************************************/
int sys_lock()
{
	return 0;
}


/************************************************************************************
 *
 * 
 *
 ************************************************************************************/

//================================== PRIVATE ============================================

/*
 * devǥ쥯ȥ
 * return : error number
 */
STATIC int setDevDir(const int devInode, FS *fs, void *rootInode)
{
	uint inodeNum;
	void *inodeObj;
	int type;
	int error;

	error = fs->lookup("dev", devInode, rootInode, &inodeNum, NULL);
	if (error < 0){
		return error;
	}
	error = fs->open(devInode, inodeNum, O_RDONLY, &inodeObj, &type);
	if (error < 0){
		return error;
	}
	
	devEnt->din			= devInode;
	devEnt->fs			= fs;
	devEnt->inodeObj	= inodeObj;
	devEnt->openCount	+= 1;
	
//	fs->close(devInode, inodeObj);
	
	return NOERR;
}

//================================== PUBLIC =============================================

/*
 * Init filesystem
 */
void init_fs()
{
	/* rootȥ꡼ν */
	rootEnt->fs			= NULL;
	rootEnt->inodeObj	= 0;
	rootEnt->next		= rootEnt;
	rootEnt->prev		= rootEnt;
	rootEnt->parent		= rootEnt;
	rootEnt->child		= NULL;
	rootEnt->mount		= NULL;
	rootEnt->openCount	= 2;		// ǥեȥȥǥ쥯ȥʤΤǣꤹ롣
	rootEnt->din		= 0;
	rootEnt->type		= S_IFDIR;
	rootEnt->len		= 1;
	rootEnt->name[0]	= '/';

	/* devȥ꡼ν */
	devEnt->fs			= NULL;
	devEnt->inodeObj	= 0;
	devEnt->next		= devEnt;
	devEnt->prev		= devEnt;
	devEnt->parent		= devEnt;
	devEnt->child		= NULL;
	devEnt->mount		= NULL;
	devEnt->openCount	= 1;
	devEnt->din			= 0;
	devEnt->type		= S_IFDIR;
	devEnt->len			= 3;
	devEnt->name[0]		= 'd';
	devEnt->name[1]		= 'e';
	devEnt->name[2]		= 'v';

	/* Device filesystemȥ꡼ꡣ */
	rootEnt->child = devEnt;
}

/*
 * Mount device filesystem.
 * return : error number
 */
int mount_dev_fs(FS *fs)
{
	return mountEntry(devEnt, fs, 0, 0);
}

/*
 * mount filesystem
 * return : error number
 */
int mountRoot(const char *devPath, const char *fsName)
{
	int devInode;
	void *rootInodeObj;
	FS *fs;
	int error;

	/* Search file system number */
	if ((fs = search_fs(fsName)) == NULL){
		return -ENODEV;
	}

	/* ǥХץ */
	error = openDevice(devPath, (void**)&devInode);
	if (error < 0){
		return error;
	}

	/* ޥȴؿƤӽФ */
	error = fs->mount(devInode, &rootInodeObj);
	if (error < 0){
		goto ERR;
	}

	/* 롼Ȳۥǥ쥯ȥꡣ */
	rootEnt->din = devInode;
	rootEnt->fs = fs;
	rootEnt->inodeObj = rootInodeObj;

	/* "dev"ǥ쥯ȥ */
	return setDevDir(devInode, fs, rootInodeObj);

ERR:
	fs->close(0, (void*)devInode);
	return error;
}

/*
 * ޥȥڡХååץǥХ
 * return : device inode or error number
 */
int mountPageBackupDev(const char *devPath, void **o_devInode)
{
	return openDevice(devPath, o_devInode);
}

/*
 * Init process0 file struct.
 * return 0 or error number
 */
int initProc1Fs()
{
	int error;

	/*
	 * ɸϤ򳫤
	 */
	error = sys_open("/dev/console", O_RDONLY, 0);	/* ɸϡ */
	if (error < 0){
		return error;
	}
	error = setFdFlag(error, 0);						/* No exec fd close. */
	if (error < 0){
		return error;
	}
	error = sys_open("/dev/console", O_WRONLY, 0);	/* ɸϡ */
	if (error < 0){
		return error;
	}
	error = setFdFlag(error, 0);
	if (error < 0){
		return error;
	}
	error = sys_open("/dev/console", O_WRONLY, 0);	/* ɸ२顼 */
	if (error < 0){
		return error;
	}
	error = setFdFlag(error, 0);
	if (error < 0){
		return error;
	}

	return 0;
}

/*************************************************************************************************/

void testFs(int num, int nest)
{
	printk("testFs() %d %d\n", num, nest);
}
