/*
 * freebsd_driver_func.c
 *
 * Copyright 2006, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 *
 * FreeBSDɥ饤СܿѴؿ
 */


#include <sys/types.h>
#include <sys/param.h>
#include <sys/errno.h>
#include <sys/conf.h>
#include <sys/bus.h>
#include <sys/bus_private.h>
#include <sys/libkern.h>
#include <sys/systm.h>
#include <sys/malloc.h>
#include <sys/resource.h>
#include <sys/devicestat.h>
#include <sys/time.h>
#include <sys/ctype.h>
#include <sys/sysctl.h>
#include <isa/isareg.h>
#include "lib.h"
#include "mm.h"
#include "interrupt.h"
#include "device.h"
#include "dev_method.h"
#include "debug.h"
#include "test.h"


//#define DEBUG_FREEBSD 1
#ifdef DEBUG_FREEBSD
	#define STATIC
	#define INLINE
#else
	#define STATIC static
	#define INLINE	inline
#endif


/****************************************************************************************
* sysctl
*****************************************************************************************/


/* Initialize a new context to keep track of dynamically added sysctls. */
int sysctl_ctx_init(struct sysctl_ctx_list *c)
{
	if (c == NULL) {
		return -EINVAL;
	}
	TAILQ_INIT(c);
	return (0);
}


/* Free the context, and destroy all dynamic oids registered in this context */
int sysctl_ctx_free(struct sysctl_ctx_list *clist)
{
	/// ̤
	return NOERR;
}


/*
 * Handle an int, signed or unsigned.
 * Two cases:
 *     a variable:  point arg1 at it.
 *     a constant:  pass it in arg2.
 */
int sysctl_handle_int(SYSCTL_HANDLER_ARGS)
{
	/// ̤
	return NOERR;
}


/****************************************************************************************
* ǥХ꥽
*****************************************************************************************/


//================================== PRIVATE ============================================


// ǥХ꥽ϡɥǥ
static struct config_resource isaResource[] = {
	{"at", RES_STRING, {(int)"isa0"}},
	{"ioport", RES_INT, {IO_FD1}},
	{"irq", RES_INT, {IRQ6}},
	{"drq", RES_INT, {2}},
};
static struct config_resource fdcResource[] = {
	{"at", RES_STRING, {(int)"fdc0"}},
};
static struct config_device devtab[] = {
	{"fdc", 0, 4, isaResource},
	{"fd", 0, 1, fdcResource},
};
static int devtab_count = 2;


//FreeBSD
STATIC int resource_match_string(int i, const char *resname, const char *value)
{
	int j;
	struct config_resource *res;

	for (j = 0, res = devtab[i].resources; j < devtab[i].resource_count; j++, res++){
		if (!strcmp(res->name, resname)
			&& res->type == RES_STRING
			&& !strcmp(res->u.stringval, value)){
			return j;
		}
	}
	
	return -1;
}


//FreeBSD
STATIC int resource_find(const char *name, int unit, const char *resname, struct config_resource **result)
{
	int i, j;
	struct config_resource *res;

	/*
	 * First check specific instances, then generic.
	 */
	for (i = 0; i < devtab_count; i++) {
		if (devtab[i].unit < 0)
			continue;
		if (!strcmp(devtab[i].name, name) && devtab[i].unit == unit) {
			res = devtab[i].resources;
			for (j = 0; j < devtab[i].resource_count; j++, res++)
				if (!strcmp(res->name, resname)) {
					*result = res;
					return 0;
				}
		}
	}
	for (i = 0; i < devtab_count; i++) {
		if (devtab[i].unit >= 0)
			continue;
		/* XXX should this `&& devtab[i].unit == unit' be here? */
		/* XXX if so, then the generic match does nothing */
		if (!strcmp(devtab[i].name, name) && devtab[i].unit == unit) {
			res = devtab[i].resources;
			for (j = 0; j < devtab[i].resource_count; j++, res++)
				if (!strcmp(res->name, resname)) {
					*result = res;
					return 0;
				}
		}
	}
	return -ENOENT;
}


//================================== PUBLIC =============================================


//FreeBSD
// return : struct resource_list_entry or NULL
struct resource_list_entry *resource_list_find(
	struct resource_list *rl,
	int type, 
	int rid)
{
	struct resource_list_entry *rle;

	for (rle = rl->slh_first; rle != NULL; rle = rle->link.sle_next){
		if (rle->type == type && rle->rid == rid){
			return rle;
		}
	}

	return NULL;
}


int bus_deactivate_resource(device_t dev, int type, int rid, struct resource *r)
{
	// Ȥꤢ⤷ʤ
	return NOERR;
}


u_long bus_get_resource_count(device_t dev, int type, int rid)
{
	// Ȥꤢ֤
	return 1;
}


int bus_set_resource(device_t dev, int type, int rid, u_long start, u_long count)
{
	// Ȥꤢ⤷ʤ
	return 0;
}


//FreeBSD
int resource_query_string(int i, const char *resname, const char *value)
{
	if (i < 0){
		i = 0;
	}
	else{
		i = i + 1;
	}
	for (; i < devtab_count; i++){
		if (0 <= resource_match_string(i, resname, value)){
			return i;
		}
	}

	return -1;
}


//FreeBSD
char *resource_query_name(int i)
{
	return devtab[i].name;
}


//FreeBSD
int resource_query_unit(int i)
{
	return devtab[i].unit;
}


//FreeBSD
int resource_int_value(const char *name, int unit, const char *resname, int *result)
{
	int error;
	struct config_resource *res;

	if ((error = resource_find(name, unit, resname, &res)) != 0){
		return error;
	}
	if (res->type != RES_INT){
		return -EFTYPE;
	}
	*result = res->u.intval;

	return 0;
}


//FreeBSD
struct resource *bus_alloc_resource(device_t dev, int type, int *rid, u_long start, u_long end, u_long count, u_int flags)
{
	if (dev->parent == 0){
		return (0);
	}
	return (BUS_ALLOC_RESOURCE(dev->parent, dev, type, rid, start, end,count, flags));
}


int bus_release_resource(device_t dev, int type, int rid, struct resource *r)
{
	kfree(r);
	return NOERR;
}


//FreeBSD
// Add a resource entry or modify an existing entry if one exists with the same type and rid.
void resource_list_add(
	struct resource_list *rl,
	int type, 
	int rid,
	u_long start, 
	u_long end, 
	u_long count)
{
	struct resource_list_entry *rle;

	rle = resource_list_find(rl, type, rid);
	if (rle == NULL) {
		rle = malloc(sizeof(struct resource_list_entry), M_BUS, M_NOWAIT);
		if (rle == NULL){
			panic("resource_list_add: can't record entry");
		}
		SLIST_INSERT_HEAD(rl, rle, link);
		rle->type = type;
		rle->rid = rid;
		rle->res = NULL;
	}

	if (rle->res){
		panic("resource_list_add: resource entry is busy");
	}

	rle->start = start;
	rle->end = end;
	rle->count = count;
}


// ꥽ꥹȥȥ꡼õ
// return : resource_list_entry or NULL
struct resource_list_entry *resource_list_entry_find(
	struct resource_list *rl,
	int type, 
	int *rid)
{
	struct resource_list_entry *rle;

	rle = resource_list_find(rl, type, *rid);
	if (rle == NULL){
		return NULL;
	}
	if (rle->res != NULL){
		panic("resource_list_alloc: resource entry is busy");
	}

	return rle;
}


//FreeBSD
int resource_list_release(
	struct resource_list *rl,
	device_t bus, 
	device_t child,
	int type, 
	int rid, 
	struct resource *res)
{
	struct resource_list_entry *rle = 0;
	int passthrough = (device_get_parent(child) != bus);
	int error;

	if (passthrough) {
		return BUS_RELEASE_RESOURCE(device_get_parent(bus), child, type, rid, res);
	}

	rle = resource_list_find(rl, type, rid);

	if (rle == NULL){
		panic("resource_list_release: can't find resource");
	}
	if (!rle->res){
		panic("resource_list_release: resource entry is not busy");
	}

	error = BUS_RELEASE_RESOURCE(device_get_parent(bus), child, type, rid, res);
	if (error){
		return error;
	}

	rle->res = NULL;
	return 0;
}


/****************************************************************************************
* ǥХ饹
*****************************************************************************************/


//================================== PRIVATE ============================================


static devclass_list_t devclasses = TAILQ_HEAD_INITIALIZER(devclasses);


//================================== PROTECTED ==========================================


//FreeBSD
// 饹õ̵к
STATIC devclass_t devclass_find_internal(const char *classname, int create)
{
	devclass_t dc;

	if (classname == NULL){
		return NULL;
	}

	for (dc = TAILQ_FIRST(&devclasses); dc; dc = TAILQ_NEXT(dc, link)){
		if (strcmp(dc->name, classname) == 0){
			return dc;
		}
	}

	if (create) {
		dc = kmalloc(sizeof(struct devclass) + strlen(classname) + 1);
		if (!dc){
			return NULL;
		}
		bzero(dc, sizeof(struct devclass) + strlen(classname) + 1);
		dc->name = (char*) (dc + 1);
		strcpy((char*)dc->name, classname);
		dc->maxunit = DEVCLASS_MAX_UNIT;
		TAILQ_INIT(&dc->drivers);
		TAILQ_INSERT_TAIL(&devclasses, dc, link);
	}

	return dc;
}


//FreeBSD
// ǥХ饹ɲäǥХ᥽åɤϿ
// return : error number
STATIC int devclass_add_driver(
	devclass_t dc, 		// ƥǥХ饹
	driver_t *driver)
{
	driverlink_t dl;
	int i;

	dl = malloc(sizeof *dl, M_BUS, M_NOWAIT);
	if (dl == NULL){
		return -ENOMEM;
	}
	bzero(dl, sizeof *dl);

//	if (driver->ops == NULL){
//		compile_methods(driver);
//	}

	// Make sure the devclass which the driver is implementing exists.
	devclass_find_internal(driver->name, TRUE);

	dl->driver = driver;
    TAILQ_INSERT_TAIL(&dc->drivers, dl, link);
    driver->refs++;

	/*
	 * Call BUS_DRIVER_ADDED for any existing busses in this class.
	 */
	for (i = 0; i < dc->maxunit; i++){
		if (dc->devices[i]){
			BUS_DRIVER_ADDED(dc->devices[i], driver);
		}
	}

	return NOERR;
}


STATIC int devclass_add_device(devclass_t dc, device_t dev)
{
	int buflen;
	int unit;

	buflen = strlen(dc->name) + 5;
	dev->nameunit = kmalloc(buflen);
	if (!dev->nameunit){
		return -ENOMEM;
	}
	memset(dev->nameunit, 0, buflen);

	// ˥åȥڡõ
	for (unit = 0; dc->devices[unit] != NULL; unit++){
		if (dc->maxunit <= unit){
			return -ENOBUFS;
		}
	}
	dc->devices[unit] = dev;
	dev->devclass = dc;
	dev->unit = unit;
	snprintf(dev->nameunit, buflen, "%s%d", dc->name, dev->unit);

	return NOERR;
}


STATIC int devclass_delete_device(devclass_t dc, device_t dev)
{
	if (!dc || !dev){
		return 0;
	}

	if (dev->devclass != dc || dc->devices[dev->unit] != dev){
		panic("devclass_delete_device: inconsistent device class");
	}
	dc->devices[dev->unit] = NULL;
	if (dev->flags & DF_WILDCARD){
		dev->unit = -1;
	}
	dev->devclass = NULL;
	free(dev->nameunit, M_BUS);
	dev->nameunit = NULL;

	return 0;
}


STATIC driverlink_t devclass_find_driver_internal(devclass_t dc, const char *classname)
{
	driverlink_t dl;

	for (dl = TAILQ_FIRST(&dc->drivers); dl; dl = TAILQ_NEXT(dl, link)) {
		if (!strcmp(dl->driver->name, classname)){
			return dl;
		}
	}

	return NULL;
}


//================================== PUBLIC =============================================


//FreeBSD
const char *devclass_get_name(devclass_t dc)
{
	return dc->name;
}


//FreeBSD
const char *device_get_name(device_t dev)
{
	if (dev->devclass){
		return devclass_get_name(dev->devclass);
	}
	return NULL;
}


//FreeBSD
device_t devclass_get_device(devclass_t dc, int unit)
{
	if (dc == NULL || unit < 0 || unit >= dc->maxunit){
		return NULL;
	}
	return dc->devices[unit];
}


//FreeBSD
void *devclass_get_softc(devclass_t dc, int unit)
{
	device_t dev;

	dev = devclass_get_device(dc, unit);
	if (!dev){
		return (NULL);
	}

	return (device_get_softc(dev));
}


int device_set_devclass(device_t dev, const char *classname)
{
	devclass_t dc;

	if (!classname) {
		if (dev->devclass){
			devclass_delete_device(dev->devclass, dev);
		}
		return 0;
	}

	if (dev->devclass) {
		printf("device_set_devclass: device class already set\n");
		return -EINVAL;
	}

	dc = devclass_find_internal(classname, TRUE);
	if (!dc){
		return -ENOMEM;
	}

	return devclass_add_device(dc, dev);
}


/****************************************************************************************
* ǥХơ
*****************************************************************************************/


//================================== PRIVATE ============================================


static int devstat_current_devnumber;
static int devstat_num_devs;
static long devstat_generation;
static STAILQ_HEAD(devstatlist, devstat) device_statq;


//================================== PUBLIC =============================================


//FreeBSD
device_state_t device_get_state(device_t dev)
{
    return dev->state;
}


//FreeBSD
// Take a malloced and zeroed devstat structure given to us, fill it in
// and add it to the queue of devices.
void devstat_add_entry(
	struct devstat			*ds,
	const char				*dev_name,
	int						unit_number,
	u_int32_t				block_size,
	devstat_support_flags	flags,
	devstat_type_flags		device_type,
	devstat_priority		priority
					  )
{
	struct devstatlist *devstat_head;
	struct devstat *ds_tmp;

	if (ds == NULL){
		return;
	}

	if (devstat_num_devs == 0){
		STAILQ_INIT(&device_statq);
	}

	devstat_generation++;
	devstat_num_devs++;

	devstat_head = &device_statq;

	/*
	* Priority sort.  Each driver passes in its priority when it adds
	* its devstat entry.  Drivers are sorted first by priority, and
	* then by probe order.
	*
	* For the first device, we just insert it, since the priority
	* doesn't really matter yet.  Subsequent devices are inserted into
	* the list using the order outlined above.
	*/
	if (devstat_num_devs == 1){
		STAILQ_INSERT_TAIL(devstat_head, ds, dev_links);
	}
	else {
		for (ds_tmp = STAILQ_FIRST(devstat_head); ds_tmp != NULL; ds_tmp = STAILQ_NEXT(ds_tmp, dev_links)) {
			struct devstat *ds_next;

			ds_next = STAILQ_NEXT(ds_tmp, dev_links);

			// If we find a break between higher and lower
			// priority items, and if this item fits in the
			// break, insert it.  This also applies if the
			// "lower priority item" is the end of the list.
			if ((priority <= ds_tmp->priority) && ((ds_next == NULL) || (priority > ds_next->priority))) {
				STAILQ_INSERT_AFTER(devstat_head, ds_tmp, ds, dev_links);
				break;
			}
			else if (priority > ds_tmp->priority) {
				// If this is the case, we should be able
				// to insert ourselves at the head of the
				// list.  If we can't, something is wrong.
				if (ds_tmp == STAILQ_FIRST(devstat_head)) {
					STAILQ_INSERT_HEAD(devstat_head, ds, dev_links);
					break;
				}
				else {
					STAILQ_INSERT_TAIL(devstat_head, ds, dev_links);
					printf("devstat_add_entry: HELP! sorting problem detected for %s%d\n", dev_name, unit_number);
					break;
				}
			}
		}
	}

	ds->device_number = devstat_current_devnumber++;
	ds->unit_number = unit_number;
	strncpy(ds->device_name, dev_name, DEVSTAT_NAME_LEN);
	ds->device_name[DEVSTAT_NAME_LEN - 1] = '\0';
	ds->block_size = block_size;
	ds->flags = flags;
	ds->device_type = device_type;
	ds->priority = priority;
	getmicrotime(&ds->dev_creation_time);
}


//FreeBSD
// Record a transaction start.
void devstat_start_transaction(struct devstat *ds)
{
	/* sanity check */
	if (ds == NULL){
		return;
	}

	/*
	* We only want to set the start time when we are going from idle
	* to busy.  The start time is really the start of the latest busy
	* period.
	*/
	if (ds->busy_count == 0){
		getmicrouptime(&ds->start_time);
	}
	ds->busy_count++;
}


//FreeBSD
// Record the ending of a transaction, and incrment the various counters.
void devstat_end_transaction(
		struct devstat		*ds,
	u_int32_t			bytes,
	devstat_tag_type	tag_type,
	devstat_trans_flags	flags)
{
	struct timeval busy_time;

	/* sanity check */
	if (ds == NULL){
		return;
	}

	getmicrouptime(&ds->last_comp_time);
	ds->busy_count--;

	/*
	* There might be some transactions (DEVSTAT_NO_DATA) that don't
	* transfer any data.
	*/
	if (flags == DEVSTAT_READ) {
		ds->bytes_read += bytes;
		ds->num_reads++;
	}
	else if (flags == DEVSTAT_WRITE) {
		ds->bytes_written += bytes;
		ds->num_writes++;
	}
	else if (flags == DEVSTAT_FREE) {
		ds->bytes_freed += bytes;
		ds->num_frees++;
	}
	else{
		ds->num_other++;
	}

	/*
	* Keep a count of the various tag types sent.
	*/
	if ((ds->flags & DEVSTAT_NO_ORDERED_TAGS) == 0 && tag_type != DEVSTAT_TAG_NONE){
		ds->tag_types[tag_type]++;
	}

	/*
	* We only update the busy time when we go idle.  Otherwise, this
	* calculation would require many more clock cycles.
	*/
	if (ds->busy_count == 0) {
		/* Calculate how long we were busy */
		busy_time = ds->last_comp_time;
		timevalsub(&busy_time, &ds->start_time);

		/* Add our busy time to the total busy time. */
		timevaladd(&ds->busy_time, &busy_time);
	}
	else if (ds->busy_count < 0){
		printf("devstat_end_transaction: HELP!! busy_count for %s%d is < 0 (%d)!\n", ds->device_name, ds->unit_number, ds->busy_count);
	}
}


//FreeBSD
void devstat_end_transaction_buf(struct devstat *ds, struct buf *bp)
{
	devstat_trans_flags flg;

	if (bp->b_flags & B_FREEBUF){
		flg = DEVSTAT_FREE;
	}
	else if (bp->b_flags & B_READ){
		flg = DEVSTAT_READ;
	}
	else{
		flg = DEVSTAT_WRITE;
	}

	devstat_end_transaction(
			ds,
	bp->b_bcount - bp->b_resid,
	(bp->b_flags & B_ORDERED) ? DEVSTAT_TAG_ORDERED : DEVSTAT_TAG_SIMPLE, flg);
}


/****************************************************************************************
* ǥХ
*****************************************************************************************/


//================================== PRIVATE ============================================


//FreeBSD
STATIC int error_method(void)
{
	return -ENXIO;
}

static struct device_ops null_ops = {
	1, 
	{ error_method }
};


//FreeBSD
int device_is_quiet(device_t dev)
{
	return (dev->flags & DF_QUIET) != 0;
}


//FreeBSD
STATIC driverlink_t first_matching_driver(devclass_t dc, device_t dev)
{
	if (dev->devclass){
		return devclass_find_driver_internal(dc, dev->devclass->name);
	}
	else{
		return TAILQ_FIRST(&dc->drivers);
	}
}


//FreeBSD
STATIC driverlink_t next_matching_driver(devclass_t dc, device_t dev, driverlink_t last)
{
	if (dev->devclass) {
		driverlink_t dl;
		for (dl = TAILQ_NEXT(last, link); dl; dl = TAILQ_NEXT(dl, link)){
			if (!strcmp(dev->devclass->name, dl->driver->name)){
				return dl;
			}
		}
		return NULL;
	}
	else{
		return TAILQ_NEXT(last, link);
	}
}


//FreeBSD
STATIC void device_set_desc_internal(device_t dev, const char* desc, int copy)
{
	if (dev->desc && (dev->flags & DF_DESCMALLOCED)) {
		kfree(dev->desc);
		dev->flags &= ~DF_DESCMALLOCED;
		dev->desc = NULL;
	}

	if (copy && desc) {
		dev->desc = kmalloc(strlen(desc) + 1);
		if (dev->desc) {
			strcpy(dev->desc, desc);
			dev->flags |= DF_DESCMALLOCED;
		}
	}
	else{
		/* Avoid a -Wcast-qual warning */
		dev->desc = (char *)(uintptr_t) desc;
	}
}


// ǥХ饹¤ΤȥǥХ¤Τκ
// return : device_t or NULL
STATIC device_t make_device(
	device_t parent,	// ƥǥХ
	const char *name,	// ǥХ̾
	int unit)
{
	device_t dev;
	devclass_t dc;

	if (name) {
		// 饹θ
		dc = devclass_find_internal(name, TRUE);
		if (!dc) {
			printf("make_device: can't find device class %s\n", name);
			return NULL;
		}
	}
	else{
		dc = NULL;
	}

	dev = kmalloc(sizeof(struct device));
	if (!dev){
		return 0;
	}
	memset(dev, 0, sizeof(struct device));

	dev->parent = parent;
	TAILQ_INIT(&dev->children);
	dev->ops = &null_ops;
	dev->driver = NULL;
	dev->devclass = NULL;
	dev->unit = unit;
	dev->nameunit = NULL;
	dev->desc = NULL;
	dev->busy = 0;
	dev->devflags = 0;
	dev->flags = DF_ENABLED;
	dev->order = 0;
	if (unit == -1){
		dev->flags |= DF_WILDCARD;
	}
	if (name) {
		dev->flags |= DF_FIXEDCLASS;
		devclass_add_device(dc, dev);
	}
	dev->ivars = NULL;
	dev->softc = NULL;
	dev->state = DS_NOTPRESENT;

	return dev;
}


STATIC int device_print_child(device_t dev, device_t child)
{
	int retval = 0;

	if (device_is_alive(child)) {
		retval += BUS_PRINT_CHILD(dev, child);
	}
	else{
		retval += device_printf(child, " not found\n");
	}

	return (retval);
}


STATIC int device_probe_child(
	device_t dev, 		// ƥǥХ
	device_t child)
{
	devclass_t dc;
	driverlink_t best = 0;
	driverlink_t dl;
	int result, pri = 0;
	int hasclass = (child->devclass != 0);

	dc = dev->devclass;
	if (!dc){
		panic("device_probe_child: parent device has no devclass");
	}

	if (child->state == DS_ALIVE){
		return 0;
	}

	for (dl = first_matching_driver(dc, child); dl; dl = next_matching_driver(dc, child, dl)) {
		device_set_driver(child, dl->driver);
		if (!hasclass){
			device_set_devclass(child, dl->driver->name);
		}
		result = DEVICE_PROBE(child);
		if (!hasclass){
			device_set_devclass(child, 0);
		}
		/*
		 * If the driver returns SUCCESS, there can be no higher match
		 * for this device.
		 */
		if (result == 0) {
			best = dl;
			pri = 0;
			break;
		}
		/*
		 * The driver returned an error so it certainly doesn't match.
		 */
		if (result != 0) {
			device_set_driver(child, 0);
			continue;
		}
		/*
		 * A priority lower than SUCCESS, remember the best matching
		 * driver. Initialise the value of pri for the first match.
		 */
		if (best == 0 || result > pri) {
			best = dl;
			pri = result;
			continue;
		}
	}

	/*
	 * If we found a driver, change state and initialise the devclass.
	 */
	if (best) {
		if (!child->devclass){
			device_set_devclass(child, best->driver->name);
		}
		device_set_driver(child, best->driver);
		if (pri < 0) {
			/*
			 * A bit bogus. Call the probe method again to make sure
			 * that we have the right description.
			 */
			DEVICE_PROBE(child);
		}
		child->state = DS_ALIVE;
		return 0;
	}

	return -ENXIO;
}


//================================== PUBLIC =============================================


//FreeBSD
int device_print_prettyname(device_t dev)
{
	const char *name = device_get_name(dev);

	if (name == 0){
		return printf("unknown: ");
	}
	else{
		return printf("%s%d: ", name, device_get_unit(dev));
	}
}


//FreeBSD
int device_is_enabled(device_t dev)
{
    return (dev->flags & DF_ENABLED) != 0;
}


// ǥХ饹¤ΤθȥǥХ¤Τκ
device_t device_add_child_ordered(
	device_t dev,		// ƥǥХ
	int order,
	const char *name,	// ǥХ̾
	int unit)			// ˥åֹ
{
	device_t child;
	device_t place;

	// ǥХ饹¤ΤθȥǥХ¤Τκ
	child = make_device(dev, name, unit);
	if (child == NULL){
		return child;
	}
	child->order = order;

	// ƤλҥǥХꥹȤ³
	TAILQ_FOREACH(place, &dev->children, link){
		if (place->order > order){
			break;
		}
	}
	if (place) {
		/*
		 * The device 'place' is the first device whose order is
		 * greater than the new child.
		 */
		TAILQ_INSERT_BEFORE(place, child, link);
	}
	else {
		/*
		 * The new child's order is greater or equal to the order of
		 * any existing device. Add the child to the tail of the list.
		 */
		TAILQ_INSERT_TAIL(&dev->children, child, link);
	}

	return child;
}


//FreeBSD
// ƥǥХƤӽФ
device_t device_add_child(device_t dev, const char *name, int unit)
{
	// ҥǥХ¤Τκ
	return device_add_child_ordered(dev, 0, name, unit);
}


//FreeBSD
int device_delete_child(device_t dev, device_t child)
{
	int error;
	device_t grandchild;

	/* remove children first */
	while ( (grandchild = TAILQ_FIRST(&child->children)) ) {
		error = device_delete_child(child, grandchild);
		if (error){
			return error;
		}
	}

	if ((error = device_detach(child)) != 0){
		return error;
	}
	if (child->devclass){
		devclass_delete_device(child->devclass, child);
	}
	TAILQ_REMOVE(&dev->children, child, link);
	device_set_desc(child, NULL);
	free(child, M_BUS);

	return 0;
}


//FreeBSD
void device_quiet(device_t dev)
{
    dev->flags |= DF_QUIET;
}


//FreeBSD
void *device_get_softc(device_t dev)
{
	return dev->softc;
}


//FreeBSD
device_t device_get_parent(device_t dev)
{
	return dev->parent;
}


//FreeBSD
int device_get_unit(device_t dev)
{
	return dev->unit;
}


//FreeBSD
void *device_get_ivars(device_t dev)
{
	return dev->ivars;
}


//FreeBSD
void device_set_ivars(device_t dev, void * ivars)
{
	if (!dev){
		return;
	}
	dev->ivars = ivars;

	return;
}


//FreeBSD
void device_enable(device_t dev)
{
	dev->flags |= DF_ENABLED;
}


//FreeBSD
void device_disable(device_t dev)
{
	dev->flags &= ~DF_ENABLED;
}


//FreeBSD
u_int32_t device_get_flags(device_t dev)
{
	return dev->devflags;
}


//FreeBSD
const char *device_get_nameunit(device_t dev)
{
	return dev->nameunit;
}


int bus_print_child_header(device_t dev, device_t child)
{
	int	retval = 0;

	if (device_get_desc(child)) { 
		retval += device_printf(child, "<%s>", device_get_desc(child));
	}
	else {
		retval += printf("%s", device_get_nameunit(child));
	}

	return (retval);
}


int bus_print_child_footer(device_t dev, device_t child)
{
	return(printf(" on %s\n", device_get_nameunit(dev)));
}


//FreeBSD
int device_printf(device_t dev, const char * fmt, ...)
{
	va_list ap;
	int retval;

	retval = device_print_prettyname(dev);
	va_start(ap, fmt);
	retval += vprintf(fmt, ap);
	va_end(ap);
	return retval;
}


//FreeBSD
const char *device_get_desc(device_t dev)
{
	return dev->desc;
}


//FreeBSD
void device_set_desc(device_t dev, const char* desc)
{
	device_set_desc_internal(dev, desc, FALSE);
}


//FreeBSD
void device_set_desc_copy(device_t dev, const char* desc)
{
	device_set_desc_internal(dev, desc, TRUE);
}


//FreeBSD
void device_busy(device_t dev)
{
	if (dev->state < DS_ATTACHED){
		panic("device_busy: called for unattached device");
	}
	if (dev->busy == 0 && dev->parent){
		device_busy(dev->parent);
	}
	dev->busy++;
	dev->state = DS_BUSY;
}


//FreeBSD
void device_unbusy(device_t dev)
{
	if (dev->state != DS_BUSY){
		panic("device_unbusy: called for non-busy device");
	}
	dev->busy--;
	if (dev->busy == 0) {
		if (dev->parent){
			device_unbusy(dev->parent);
		}
		dev->state = DS_ATTACHED;
	}
}


//FreeBSD
int device_detach(device_t dev)
{
	int error;

	if (dev->state == DS_BUSY){
		return -EBUSY;
	}
	if (dev->state != DS_ATTACHED){
		return 0;
	}

	if ((error = DEVICE_DETACH(dev)) != 0){
		return error;
	}
	device_printf(dev, "detached\n");
	if (dev->parent){
		BUS_CHILD_DETACHED(dev->parent, dev);
	}

	if (!(dev->flags & DF_FIXEDCLASS)){
		devclass_delete_device(dev->devclass, dev);
	}

	dev->state = DS_NOTPRESENT;
	device_set_driver(dev, NULL);

	return 0;
}


// device_t¤Τ
// return : error number
int bus_generic_shutdown(device_t device)
{
	kfree(device->softc);
	kfree(device->ivars);

	return NOERR;
}


// return : error number
int	bus_generic_suspend(device_t dev)
{
	// Ȥꤢ⤷ʤ
	return NOERR;
}


// return : error number
int	bus_generic_resume(device_t dev)
{
	// Ȥꤢ⤷ʤ
	return NOERR;
}


//FreeBSD
int bus_generic_print_child(device_t dev, device_t child)
{
	int	retval = 0;

	retval += bus_print_child_header(dev, child);
	retval += bus_print_child_footer(dev, child);

	return (retval);
}


//FreeBSD
int bus_generic_attach(device_t dev)
{
	device_t child;

	for (child = TAILQ_FIRST(&dev->children); child; child = TAILQ_NEXT(child, link)){
		device_probe_and_attach(child);
	}

	return 0;
}


// return : error number
int	bus_generic_detach(device_t dev)
{
	// Ȥꤢ⤷ʤ
	return NOERR;
}


//FreeBSD
void bus_generic_driver_added(device_t dev, driver_t *driver)
{
	device_t child;

	DEVICE_IDENTIFY(driver, dev);

	for (child = TAILQ_FIRST(&dev->children); child; child = TAILQ_NEXT(child, link)){
		if (child->state == DS_NOTPRESENT){
			device_probe_and_attach(child);
		}
	}
}


//FreeBSD
int bus_teardown_intr(device_t dev, struct resource *r, void *cookie)
{
	if (dev->parent == 0){
		return -EINVAL;
	}
	return (BUS_TEARDOWN_INTR(dev->parent, dev, r, cookie));
}


//FreeBSD
int device_set_driver(device_t dev, driver_t *driver)
{
	if (DS_ATTACHED <= dev->state){
		return -EBUSY;
	}

	if (dev->driver == driver){
		return 0;
	}

	if (dev->softc && !(dev->flags & DF_EXTERNALSOFTC)) {
		free(dev->softc, M_BUS);
		dev->softc = NULL;
	}
	dev->ops = &null_ops;
	dev->driver = driver;
	if (driver) {
		dev->ops = driver->ops;
		if (!(dev->flags & DF_EXTERNALSOFTC)) {
			dev->softc = malloc(driver->softc, M_BUS, M_NOWAIT);
			if (!dev->softc) {
				dev->ops = &null_ops;
				dev->driver = NULL;
				return -ENOMEM;
			}
			bzero(dev->softc, driver->softc);
		}
	}
	return 0;
}


//FreeBSD
int device_is_alive(device_t dev)
{
	return dev->state >= DS_ALIVE;
}


//FreeBSD
int bus_setup_intr(device_t dev, struct resource *r, int flags, driver_intr_t handler, void *arg, void **cookiep)
{
	if (dev->parent == 0){
		return -EINVAL;
	}
	return (BUS_SETUP_INTR(dev->parent, dev, r, flags, handler, arg, cookiep));
}


//FreeBSD
int device_probe_and_attach(device_t dev)
{
	device_t bus = dev->parent;
	int error = 0;
	int hasclass = (dev->devclass != 0);

	if (dev->state >= DS_ALIVE){
		return 0;
	}

	if (dev->flags & DF_ENABLED) {
		error = device_probe_child(bus, dev);
		if (!error) {
			if (!device_is_quiet(dev)){
				device_print_child(bus, dev);
			}
			error = DEVICE_ATTACH(dev);
			if (!error){
				dev->state = DS_ATTACHED;
			}
			else {
				printf("device_probe_and_attach: %s%d attach returned %d\n", dev->driver->name, dev->unit, error);
				/* Unset the class that was set in device_probe_child */
				if (!hasclass){
					device_set_devclass(dev, 0);
				}
				device_set_driver(dev, NULL);
				dev->state = DS_NOTPRESENT;
			}
		} 
		else {
			if (!(dev->flags & DF_DONENOMATCH)) {
//				BUS_PROBE_NOMATCH(bus, dev);
				dev->flags |= DF_DONENOMATCH;
			}
		}
	} 
	else {
		if (bootverbose) {
			device_print_prettyname(dev);
			printf("not probed (disabled)\n");
		}
	}

	return error;
}


// ХǥХ
// return : device_t or NULL
device_t make_bus_device(
	const char *name,		// ǥХ̾
	int unit)				// ˥åֹ
{
	return make_device(NULL, name, unit);
}


//FreeBSD
int driver_module_handler(module_t mod, int what, void *arg)
{
	struct driver_module_data *dmd = (struct driver_module_data *)arg;
	devclass_t bus_devclass;
	int error = 0;
	int i;

	bus_devclass = devclass_find_internal(dmd->dmd_busname, TRUE);

	switch (what) {
	case MOD_LOAD:
		if (dmd->dmd_chainevh){
			error = dmd->dmd_chainevh(mod, what, dmd->dmd_chainarg);
		}

		for (i = 0; !error && i < dmd->dmd_ndrivers; i++) {
			error = devclass_add_driver(bus_devclass, dmd->dmd_drivers[i]);
		}
		if (error){
			break;
		}

		/*
		 * The drivers loaded in this way are assumed to all implement the same devclass.
		 */
		*dmd->dmd_devclass = devclass_find_internal(dmd->dmd_drivers[0]->name, TRUE);
		break;
	case MOD_UNLOAD:
/*		for (i = 0; !error && i < dmd->dmd_ndrivers; i++) {
			error = devclass_delete_driver(bus_devclass, dmd->dmd_drivers[i]);
		}

		if (!error && dmd->dmd_chainevh){
			error = dmd->dmd_chainevh(mod, what, dmd->dmd_chainarg);
		}
*/		break;
	}

	return (error);
}


// ǥХν
// return 0 or error number
int initDev(struct driver_module_data *arg)
{
	return driver_module_handler(NULL, MOD_LOAD, arg);
}


//******************************************************************************
void test_freebsd()
{
	printk("test_freebsd() %d ", *(int*)0x155bb0);
}
