/*
 * ext2.c
 *
 * Copyright 2003, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 *
 * EXT2ե륷ƥࡣ
 */


#include <sys/types.h>
#include <sys/param.h>
#include <sys/libkern.h>
#include <kern/lib.h>
#include <kern/lib_path.h>
#include <kern/mm.h>
#include <kern/fs.h>
#include <kern/device.h>
#include <kern/proc.h>
#include <kern/time.h>
#include <kern/errno.h>
#include <kern/lock.h>
#include <kern/test.h>
#include <kern/ext2.h>
#include <kern/debug.h>


//#define DEBUG_EXT2_FS 1
#ifdef DEBUG_EXT2_FS
	#define STATIC
	#define INLINE
#else
	#define STATIC static
	#define INLINE	inline
#endif


/************************************************************************************
 *
 * 
 *
 *************************************************************************************/


enum{
	INT_LOG = 2,					/* intΥХȿΣ򺬤Ȥп */
	UINT_LOG = INT_LOG,

	EXT2_BLOCK_BASE=1024,			/* ֥å׻뤿Υ١Хȿ */
	EXT2_BLOCK_BASE_LOG=10,			/* ١֥åΣ򺬤Ȥп */

	EXT2_SUPER_BLK_POSITION=1024,	/* ѡ֥åΰ֡ƬΥХȿˡ */

	EXT2_GROUP_DESC_BLK=1,			/* group descriptor block number. */

	/* File system states at ss_state". */
	EXT2_VALID_FS=0x0001,			/* Unmounted cleanly */
	EXT2_ERROR_FS=0x0002,			/* Errors detected */

	/* Maximal mount counts. */
	EXT2_DFL_MAX_MNT_COUNT=20,
	EXT2_DFL_CHECKINTERVAL=0,

	/* Magic number at "s_magic". */
	EXT2_SUPER_MAGIC=0xEF53,

	/* Behaviour when detecting errors. */
	EXT2_ERRORS_CONTINUE=1,			/* continue as if nothing happened. */
	EXT2_ERRORS_RO=2,				/* remount read-only. */
	EXT2_ERRORS_PANIC=3,			/* cause a kernel panic. */
	EXT2_ERRORS_DEFAULT=EXT2_ERRORS_CONTINUE,

	/* revision level value. */
	EXT2_GOOD_OLD_REV=0,			/* original format. */
	EXT2_DYNAMIC_REV=1,				/* V2 format with dynamic inode sizes. */

	/* EXT2_*_INO values at "s_first_ino". */
	EXT2_BAD_INO=0x01, 				/* bad blocks inode */
	EXT2_ROOT_INO=0x02, 			/* root directory inode */
	EXT2_ACL_IDX_INO=0x03, 			/* ACL index inode (deprecated?) */
	EXT2_ACL_DATA_INO=0x04, 		/* ACL data inode (deprecated?) */
	EXT2_BOOT_LOADER_INO=0x05,		/* boot loader inode */
	EXT2_UNDEL_DIR_INO=0x06,		/* undelete directory inode */
};

/* super block. */
typedef struct{
	uint s_inodes_count;			/* 00,total number of inodes. */
	uint s_blocks_count;			/* 04,total number of blocks. */
	uint s_r_blocks_count;			/* 08,total number of blocks reserved for the usage of the super user. */
	uint s_free_blocks_count;		/* 0c,total number of free blocks. */
	uint s_free_inodes_count;		/* 10,total number of free inodes. */
	uint s_first_data_block;		/* 14,the block containing the superblock structure.block size larger than 1KB=0,block size of 1KB=1. */
	uint s_log_block_size;			/* 18,number of bits to shift left the value 1024 for computing block size. */
	int s_log_frag_size;			/* 1c,number of bits to shift the value 1024 for computing fragment size. */
	uint s_blocks_per_group;		/* 20,total number of blocks per group. */
	uint s_frags_per_group;			/* 24,total number of fragments per group. */
	uint s_inodes_per_group;		/* 28,total number of inodes per group. */
	uint s_mtime;					/* 2c,Unix time,last time the file system was mounted. */
	uint s_wtime;					/* 30,Unix time,last write access to the file system. */
	ushort s_mnt_count;				/* 34,how many time the file system was mounted. */
	short s_max_mnt_count;			/* 36,the maximum number of mount times. */
	ushort s_magic;					/* 38,magic number 0xEF53. */
	ushort s_state;					/* 3a,file system state. */
	ushort s_errors;				/* 3c,behaviour when an error is detected. */
	ushort s_minor_rev_level;		/* 3e,minor revision level. */
	uint s_lastcheck;				/* 40,Unix time,last file system check. */
	uint s_checkinterval;			/* 44,between file system checks. */
	uint s_creator_os;				/* 48,os that created the file system. */
	uint s_rev_level;				/* 4c,revision level value. */
	ushort s_def_resuid;			/* 4e,default user id for reserved blocks. */
	ushort s_def_resgid;			/* 50,default group id for reserved blocks. */

	/* EXT2_DYNAMIC_REV Specific */
	uint s_first_ino;				/* 54,index to the first inode useable for standard files. */
	ushort s_inode_size;			/* 58,size of the inode structure. */
	ushort s_block_group_nr;		/* 5a,block group number hosting this superblock. */
	uint s_feature_compat;			/* 5c,bitmask of compatible features. */
	uint s_feature_incompat;		/* 60,bitmask of incompatible features. */
	uint s_feature_ro_compat;		/* 64,bitmask of read-only features. */
	uchar s_uuid[16];				/* 68,volume id. */
	char s_volume_name[16];			/* 78,volume name,mostly unusued. */
	char s_last_mounted[64];		/* 88,last mounted directory path,not normally used. */
	uint s_algo_bitmap;				/* compression algorithms. */

	/* Performance Hints */
	uchar s_prealloc_blocks;		/* nmber of blocks to try to preallocate. */
	uchar s_prealloc_dir_blocks;	/* number to preallocate for dirs. */
	ushort s_padding1;

	/* Journaling Support */
	uchar s_journal_uuid[16];		/*  */
	uint s_journal_inum;			/*  */
	uint s_journal_dev;				/*  */
	uint s_last_orphan;				/*  */

	/* ꡼¸ȼ */
	int blockSize;					/* ֥å */
	int sectorSize;					/*  */
	int sectors;					/* 1֥åΥ */
}SUPER_BLOCK;

/* groupe descriptor. */
typedef struct{
	uint bg_block_bitmap;			/* 0,the first block of the "block bitmap". */
	uint bg_inode_bitmap;			/* 4,the first block of the "inode bitmap". */
	uint bg_inode_table;			/* 8,the first block of the "inode table". */
	ushort bg_free_blocks_count;	/* c,total number of free blocks. */
	ushort bg_free_inodes_count;	/* e,total number of free inodes. */
	ushort bg_used_dirs_count;		/* 10,number of inodes allocated to directories. */
	ushort bg_pad;
	uint bg_reserved[3];
}GROUP_DESC;


/*================================== ѥ֥å ========================================*/


/* ӥåȥޥåפǥХ֥ͤåΰ֤롣 */
static const uchar emptyBit[]={
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,4,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,5,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,4,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,6,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,4,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,5,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,4,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,7,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,4,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,5,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,4,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,6,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,4,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,5,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,4,
	0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,0,
};


// ޥȻ
static SUPER_BLOCK *superBlock[MAX_DEVICE_OPEN];
static GROUP_DESC *grpDesc[MAX_DEVICE_OPEN];
static int grpDescNum[MAX_DEVICE_OPEN];			// 롼ץǥץ
static int grpDescBlocks[MAX_DEVICE_OPEN];		// 롼ץǥץ֥å


/*
 * ֥å饻򻻽Ф롣
 * parameters : device inode,block
 * return : secter number
 */
STATIC INLINE uint blkToSect(int din,uint blk)
{
	return superBlock[din]->blockSize * blk / superBlock[din]->sectorSize;
}


/*
 * ѡ֥åȥ롼ץǥץǥ˽᤹
 * return : 0 or error=-1
 */
STATIC int writeBackSbGd(int din)
{
	SUPER_BLOCK *sb = superBlock[din];
	int sectors = sb->sectors;
	void *buf;
	size_t ioSize;

	/* 롼ץǥץǥ˽᤹ */
	write_cache(din, grpDesc[din], grpDescBlocks[din] * sectors, blkToSect(din, EXT2_GROUP_DESC_BLK + sb->s_first_data_block), &ioSize);
	ASSERT(ioSize == grpDescBlocks[din] * sectors);
	write_cache(din, grpDesc[din], grpDescBlocks[din] * sectors,
		blkToSect(din, EXT2_GROUP_DESC_BLK + sb->s_first_data_block + sb->s_blocks_per_group), &ioSize);
	ASSERT(ioSize == grpDescBlocks[din] * sectors);

	/* ѡ֥åǥ˽᤹ */
	buf = kmalloc(sb->blockSize);
	if (buf==NULL){
		return -1;
	}
	/*read_cache(din,buf,sectors,blkToSect(din,sb->s_first_data_block));*/
	memset(buf, 0, sb->blockSize);
	if (sb->s_first_data_block == 1){
		memcpy(buf, sb, sizeof(SUPER_BLOCK));
	}
	else{
		memcpy((char*)buf + EXT2_SUPER_BLK_POSITION, sb, sizeof(SUPER_BLOCK));
	}
	write_cache(din, buf, sectors, blkToSect(din, sb->s_first_data_block), &ioSize);
	if (sb->s_first_data_block == 0){
		memset((char*)buf + EXT2_SUPER_BLK_POSITION, 0,sizeof(SUPER_BLOCK));
		memcpy(buf, sb, sizeof(SUPER_BLOCK));
	}
	ASSERT(ioSize == sectors);
	write_cache(din, buf, sectors, blkToSect(din, sb->s_first_data_block + sb->s_blocks_per_group), &ioSize);
	kfree(buf);
	ASSERT(ioSize == sectors);

	return 0;
}


/***********************************************************************
 *
 * Block
 *
 ***********************************************************************/


/* ӥåȥޥå׹¤Ρ */
typedef struct{
	uchar *bitmap;			/* EXT2롼ץ֥åӥåȥޥåץХåե */
	ushort currentGd;		/* ߤΥ롼 */
	ushort currentByte;		/* ߤΥӥåȥޥåפΥХȥեåȡ */
}BITMAP;


/*================================== ץ饤١ ========================================*/


static BITMAP blockBm[MAX_DEVICE_OPEN];
static int blkLock[MAX_DEVICE_OPEN];			// ֥åԥå


/*================================== ѥ֥å ========================================*/


/*
 * ֥å롣
 * parameters : device inode
 * return : empty block or error number;
 */
STATIC int getBlock(int din)
{
	GROUP_DESC *grpDsc = grpDesc[din];
	SUPER_BLOCK *superBlk = superBlock[din];
	size_t ioSize;
	int block;
	int grp, byte;
	int last;
	int rest;

	enter_spinlock(&blkLock[din]);
	{
		// ߤΥ롼פ˶֥åʤ
		if (grpDsc[blockBm[din].currentGd].bg_free_blocks_count == 0){
			int sectors = superBlk->sectors;

			// ߤΥӥåȥޥåפ᤹
			rest = write_cache(din, blockBm[din].bitmap, sectors, blkToSect(din, grpDsc[blockBm[din].currentGd].bg_block_bitmap), &ioSize);
			if (rest != NOERR){
				return rest;
			}
			ASSERT(ioSize == sectors);

			// ֥åΤ륰롼פõ
			for (grp = blockBm[din].currentGd + 1;; ++grp){
				if (grp == grpDescNum[din]){
					grp = 0;
				}
				if (grp == blockBm[din].currentGd){
					return -ENOSPC;
				}
				if (0 < grpDsc[grp].bg_free_blocks_count){
					break;
				}
			}
			rest = read_cache(din, blockBm[din].bitmap, sectors, blkToSect(din, grpDsc[grp].bg_block_bitmap), &ioSize);
			if (rest != NOERR){
				return rest;
			}
			// ֥åɤʤХ֥å֤
			ASSERT(sectors == ioSize);
			blockBm[din].currentGd = grp;
			blockBm[din].currentByte = 0;
		}

		/* ӥåȥޥåפθ */
		last = ROUNDUP_DIV(superBlk->s_blocks_per_group , BYTE_BIT);
		for (byte = blockBm[din].currentByte; blockBm[din].bitmap[byte] == 0xff;){
			if (last == ++byte){
				byte = 0;
			}
			// ӥåȥޥåפϤ
			ASSERT(blockBm[din].currentByte != byte)
		}
		blockBm[din].currentByte = byte;
		block = byte * BYTE_BIT + emptyBit[blockBm[din].bitmap[byte]];
		blockBm[din].bitmap[byte] |= 1 << emptyBit[blockBm[din].bitmap[byte]];
		
		/* ֥åȤ򸺤餹 */
		--grpDsc[blockBm[din].currentGd].bg_free_blocks_count;
		--superBlk->s_free_blocks_count;
	}
	exit_spinlock(&blkLock[din]);

	return block + blockBm[din].currentGd * superBlk->s_blocks_per_group + superBlk->s_first_data_block;
}


/*
 * ֥å롣
 * parameters : device inode,block number
 */
STATIC void releaseBlock(int din,uint block)
{
	SUPER_BLOCK *sb = superBlock[din];
	size_t ioSize;
	int sectors = sb->sectors;
	int grp;
	int a;

	/* ֥åӥåȥޥåפθ */
	grp = (block - sb->s_first_data_block) / sb->s_blocks_per_group;
	enter_spinlock(&blkLock[din]);
	{
		if (grp != blockBm[din].currentGd){
			if (write_cache(din, blockBm[din].bitmap, sectors,
				blkToSect(din, grpDesc[din][blockBm[din].currentGd].bg_block_bitmap), &ioSize) != NOERR){
				return;
				ASSERT(sectors == ioSize);
			}
			if (read_cache(din, blockBm[din].bitmap, sectors, blkToSect(din, grpDesc[din][grp].bg_block_bitmap), &ioSize) != NOERR){
				return;
				// ֥åɤʤХ֥å֤
				ASSERT(sectors == ioSize);
			}
			blockBm[din].currentGd = grp;
		}

		/* ӥåȥޥåפ˽񤭹ࡣ */
		a = block - grp * sb->s_blocks_per_group - sb->s_first_data_block;
		blockBm[din].bitmap[a / BYTE_BIT] &= ~(1 << a % BYTE_BIT);

		/* ֥åȤ䤹 */
		++grpDesc[din][grp].bg_free_blocks_count;
		++sb->s_free_blocks_count;
	}
	exit_spinlock(&blkLock[din]);
}


/*
 * ޥȻν
 * parameters : device inode
 * return : groupe descriptor number or error number
 */
STATIC int initBlock(int din)
{
	SUPER_BLOCK *sb = superBlock[din];
	GROUP_DESC *gd = grpDesc[din];
	size_t ioSize;
	int grp, rest;

	blockBm[din].bitmap = kmalloc(sb->blockSize);
	if (blockBm[din].bitmap == NULL){
		return -ENOMEM;
	}
	blockBm[din].currentGd = 0;
	blockBm[din].currentByte = 0;

	// ֥å˶Τ륰롼פΥӥåȥޥåפɤ߽Ф
	for (grp = 0; gd[grp].bg_free_blocks_count == 0; ++grp){
		if(grp == grpDescNum[din]){
			return -ENOSPC;
		}
	}
	rest = read_direct(din, blockBm[din].bitmap, sb->sectors, blkToSect(din, gd[grp].bg_block_bitmap), &ioSize);
	if (rest != NOERR){
		return rest;
	}
	blockBm[din].currentGd = grp;
	blockBm[din].currentByte = 0;

	return blockBm[din].currentGd;
}


/*
 * ޥȻν
 * parameters : device inode
 */
STATIC void umountBlock(int din)
{
	int sectors = superBlock[din]->sectors;
	size_t ioSize;

	write_cache(din, blockBm[din].bitmap, sectors, blkToSect(din, grpDesc[din][blockBm[din].currentGd].bg_block_bitmap), &ioSize);
	kfree(blockBm[din].bitmap);
	ASSERT(ioSize == sectors);
}


/***********************************************************************
 *
 * Inode
 *
 ***********************************************************************/


enum{
	/* at i_mode */
	EXT2_S_IFMT		= 0xF000,	/* format mask */
	EXT2_S_IFSOCK	= 0xC000,	/* socket */
	EXT2_S_IFLNK	= 0xA000,	/* symbolic link */
	EXT2_S_IFREG	= 0x8000,	/* regular file */
	EXT2_S_IFBLK	= 0x6000,	/* block device */
	EXT2_S_IFDIR	= 0x4000,	/* directory */
	EXT2_S_IFCHR	= 0x2000,	/* character device */
	EXT2_S_IFIFO	= 0x1000,	/* fifo */
	EXT2_S_ISUID	= 0x0800,	/* SUID */
	EXT2_S_ISGID	= 0x0400,	/* SGID */
	EXT2_S_ISVTX	= 0x0200,	/* sticky bit */
	EXT2_S_IRWXU	= 0x01C0,	/* user access rights mask */
	EXT2_S_IRUSR	= 0x0100,	/* read */
	EXT2_S_IWUSR	= 0x0080,	/* write */
	EXT2_S_IXUSR	= 0x0040,	/* execute */
	EXT2_S_IRWXG	= 0x0038,	/* group access rights mask */
	EXT2_S_IRGRP	= 0x0020,	/* read */
	EXT2_S_IWGRP	= 0x0010,	/* write */
	EXT2_S_IXGRP	= 0x0008,	/* execute */
	EXT2_S_IRWXO	= 0x0007,	/* others access rights mask */
	EXT2_S_IROTH	= 0x0004,	/* read */
	EXT2_S_IWOTH	= 0x0002,	/* write */
	EXT2_S_IXOTH	= 0x0001,	/* execute */

	/* behaviour control flags at "i_flags". */
	EXT2_SECRM_FL		= 0x00000001,	/* secure deletion */
	EXT2_UNRM_FL		= 0x00000002,	/* record for undelete */
	EXT2_COMPR_FL		= 0x00000004,	/* compressed file */
	EXT2_SYNC_FL		= 0x00000008,	/* synchronous updates */
	EXT2_IMMUTABLE_FL	= 0x00000010,	/* immutable file */
	EXT2_APPEND_FL		= 0x00000020,	/* append only */
	EXT2_NODUMP_FL		= 0x00000040,	/* do not dump/delete file */
	EXT2_NOATIME_FL		= 0x00000080,	/* do not update .i_atime */
	EXT2_DIRTY_FL		= 0x00000100,	/* dirty (file is in use?) */
	EXT2_COMPRBLK_FL	= 0x00000200,	/* compressed blocks */
	EXT2_NOCOMPR_FL		= 0x00000400,	/* access raw compressed data */
	EXT2_ECOMPR_FL		= 0x00000800,	/* compression error */
	EXT2_BTREE_FL		= 0x00010000,	/* b-tree format directory */
	EXT2_INDEX_FL		= 0x00010000,	/* Hash indexed directory */

	EXT2_ROOT_INODE_INDEX = 2,		/* 롼inodeΥǥå */

	/* indirect data block index. */
	INODE_DATA_INDIRECT1	= 12,		/* ܻȥ֥å͡ */
	INODE_DATA_INDIRECT2	= 13,		/* Ŵܻȥ֥å͡ */
	INODE_DATA_INDIRECT3	= 14,		/* Ŵܻȥ֥å͡ */

	INODE_DATA_BLOCK_NUM	= 15,		/* ǡ֥å */
	INODE_DIRECT_BLOCK_NUM	= INODE_DATA_INDIRECT1,	/* ľܥ֥å */

	INODE_BLOCK_SIZE	= 512,			/* inodei_blocksΥ֥åñ̥Хȿ */
};


typedef struct{
	ushort i_mode;			/* 00,format of the described file and the access rights. */
	ushort i_uid;			/* 02,user id. */
	uint i_size;			/* 04,size of the file in bytes. */
	uint i_atime;			/* 08,Unix time,last time this file was accessed. */
	uint i_ctime;			/* 0c,Unix time,when file was created. */
	uint i_mtime;			/* 10,Unix time,last time this file was modified. */
	uint i_dtime;			/* 14,Unix time,when file was deleted. */
	ushort i_gid;			/* 18,group id. */
	ushort i_links_count;	/* 1a,how many times this inode is linked. */
	uint i_blocks;			/* 1c,both currently in used and currently reserved blocks.number of 512 bytes block. */
	uint i_flags;			/* 20,behave when accessing the data for this inode. */
	uint i_number;			/* 24,OS dependent 1,inode number(open) */
	/*
	 * data block number.
	 * 112 direct block number.
	 * 13 indirect block number.
	 * 14 bi-indirect block number.
	 * 15 tri-indirect block number.
	 * Each indirect block array contains as many entries of 32bit block numbers as possible
	 * (to fill one entire block).
	 */
	uint i_block[INODE_DATA_BLOCK_NUM];		/* 28 */
	uint i_generation;		/* 64,file version (used by NFS). */
	uint i_file_acl;		/* 68,block number containing the extended attributes, */
	uint i_dir_acl;			/* 6c,"high size" of the file. */
	uint i_faddr;			/* 70,location of the last file fragment. */
	uint i_osd2[3];			/* 74,OS dependent 2 */
}INODE;

typedef struct{
	const char *path;
	INODE *inodeData;
	INODE *inodeBuf;
}INODE_PARAM;


/*====================================================================================================*
 *                                       read data                                                    *
 *====================================================================================================*/


typedef int (*READ_DATA)(int,void*,uint*,size_t,size_t,int);


//================================== PRIVATE ============================================


static int readDirectBlock(int,void*,uint*,size_t,size_t,int);
static int readIndirectBlock(int,void*,uint*,size_t,size_t,int);

static READ_DATA readIndirect[]={
	readDirectBlock,
	readIndirectBlock,
	readIndirectBlock
};


// ǡɹߡ
// parameters : device inode,buffer,block table,֥åơ֥νߥ,֥åơ֥γϥХȿ,node
// return : ɹߥ or error number
STATIC int readDirectBlock(int din, void *buf, uint *m_blkTbl, size_t size, size_t begin, int node)
{
	char *blkBuf = NULL;
	int blkSize = superBlock[din]->blockSize;
	int sectors = superBlock[din]->sectors;
	int restSize = size;
	size_t ioSize;
	int rest;

	m_blkTbl += begin / blkSize;

	/* ϥХ֥ͤåǤʤ硣 */
	if (0 < (begin % blkSize)){
		int forward = begin % blkSize;
		int back = blkSize - forward;

		if (restSize < back){
			back = restSize;
		}
		blkBuf = kmalloc(blkSize);
		if (blkBuf == NULL){
			return -ENOMEM;
		}
		rest = read_cache(din, blkBuf, sectors, blkToSect(din, *m_blkTbl), &ioSize);
		if (rest != NOERR){
			goto ERR;
		}
		ASSERT(ioSize == sectors);
		memcpy(buf, blkBuf + forward, back);
		buf = (char*)buf + back;
		restSize -= back;
		++m_blkTbl;
	}

	for (;blkSize <= restSize; restSize -= blkSize){
		rest = read_cache(din, buf, sectors, blkToSect(din, *m_blkTbl), &ioSize);
		if (rest != NOERR){
			goto ERR;
		}
		ASSERT(ioSize == sectors);
		buf = (char*)buf + blkSize;
		++m_blkTbl;
	}

	/* ǸΥХ֥ͤåǤʤ硣 */
	if (0 < restSize){
		if (blkBuf == NULL){
			blkBuf = kmalloc(blkSize);
			if (blkBuf == NULL){
				return -ENOMEM;
			}
		}
		rest = read_cache(din, blkBuf, sectors, blkToSect(din, *m_blkTbl), &ioSize);
		if (rest != NOERR){
			goto ERR;
		}
		ASSERT(ioSize == sectors);
		memcpy(buf, blkBuf, restSize);
		restSize = 0;
	}

	kfree(blkBuf);
	return size - restSize;
ERR:
	kfree(blkBuf);
	return rest;
}


// ɹߡ
// parameters : ǥХinode,Хåե,֥åơ֥,֥åơ֥ɹߥ,֥åơ֥γϥХȿ,node
// return : ɹߥ or error number
STATIC int readIndirectBlock(int din, void *buf, uint *m_blkTbl, size_t size, size_t begin, int node)
{
	uint *blkBuf;
	int sectors = superBlock[din]->sectors;
	int blkSize = superBlock[din]->blockSize;
	int idrct_block_log = EXT2_BLOCK_BASE_LOG + superBlock[din]->s_log_block_size - UINT_LOG;
	int idrct_size = blkSize << (idrct_block_log * node);
	size_t restSize = size;
	size_t ioSize;
	int rest;

	blkBuf = kmalloc(blkSize);
	if (blkBuf == NULL){
		return -ENOMEM;
	}
	m_blkTbl += begin / idrct_size;

	/* ϥХ֥ͤåǤʤ硣 */
	if (0 < (begin % idrct_size)){
		int forward = begin % idrct_size;
		int back = idrct_size - forward;
		
		if (restSize < back){
			back = restSize;
		}
		rest = read_cache(din, blkBuf, sectors, blkToSect(din, *m_blkTbl), &ioSize);
		if (rest != NOERR){
			goto ERR;
		}
		// ֥åɤʤХ֥å֤
		ASSERT(ioSize == sectors)
		rest = readIndirect[node - 1](din, buf, blkBuf, back, forward, node - 1);
		if (rest < 0){
			goto ERR;
		}
		buf = (char*)buf + back;
		restSize -= back;
		++m_blkTbl;
	}

	for (;idrct_size <= restSize; restSize -= idrct_size){
		rest = read_cache(din, blkBuf, sectors, blkToSect(din, *m_blkTbl), &ioSize);
		if (rest != NOERR){
			goto ERR;
		}
		ASSERT(ioSize == sectors);

		rest = readIndirect[node - 1](din, buf, blkBuf, idrct_size, 0, node - 1);
		if (rest < 0){
			goto ERR;
		}
		buf = (char*)buf + idrct_size;
		++m_blkTbl;
	}

	if (0 < restSize){
		rest = read_cache(din, blkBuf, sectors, blkToSect(din, *m_blkTbl), &ioSize);
		if(rest != NOERR){
			goto ERR;
		}
		ASSERT(ioSize == sectors)

		rest = readIndirect[node - 1](din, buf, blkBuf, restSize, 0, node - 1);
		if (rest < 0){
			goto ERR;
		}
	}

	kfree(blkBuf);
	return size;
ERR:
	kfree(blkBuf);
	return rest;
}


//================================== PROTECTED ==========================================


// եǡɤ߹ࡣ
// parameters : device inode,inode,buffer,read size,begin byte
// return : read size or error number
STATIC int readData(int din, INODE *inode, void *buf, size_t size, size_t begin)
{
	uint64 nsize;
	uint *table;
	size_t restSize = size;
	size_t ioSize;
	int sectors = superBlock[din]->sectors;
	int blkSize = superBlock[din]->blockSize;
	int nblock = blkSize / sizeof(uint);		// ֥åơ֥Υ֥å
	int read_size;
	int rest;
	int i;

	nsize = INODE_DATA_INDIRECT1 * blkSize;	// ľܥ֥å
	if (begin < nsize){
		goto DIRECT;
	}
	begin -= nsize;
	nsize = blkSize * nblock;				// ܥ֥å
	for (i = INODE_DATA_INDIRECT1; ; ++i){
		if (begin < nsize){
			goto INDIRECT;
		}
		begin -= nsize;
		nsize *= nblock;
	}

DIRECT:
	/* ľܥ֥å */
	read_size = nsize - begin;
	if (restSize < read_size){
		read_size = restSize;
	}
	rest = readDirectBlock(din, buf, inode->i_block, read_size, begin, 0);
	if (rest < 0){
		return rest;
	}
	restSize -= read_size;
	if (restSize == 0){
		return size;
	}
	buf = (char*)buf + read_size;
	begin = 0;

	/* ܥ֥å */
	nsize = blkSize*nblock;		/* ܥ֥å */
	i = INODE_DATA_INDIRECT1;

INDIRECT:
	table = kmalloc(blkSize);
	if (table == NULL){
		return -ENOMEM;
	}
	for (; ; ++i){
		rest = read_cache(din, table, sectors, blkToSect(din, inode->i_block[i]), &ioSize);
		if (rest != NOERR){
			goto ERR;
		}
		ASSERT(ioSize == sectors);
		read_size = nsize - begin;
		if (restSize < read_size){
			read_size = restSize;
		}
		rest = readIndirect[i - INODE_DATA_INDIRECT1](din, buf, table, read_size, begin, i - INODE_DATA_INDIRECT1);
		if (rest < 0){
			goto ERR;
		}
		restSize -= read_size;
		if (restSize == 0){
			break;
		}
		buf = (char*)buf + read_size;
		begin = 0;
		nsize *= nblock;
	}

	kfree(table);
	return size;
ERR:
	kfree(table);
	return rest;
}


/*====================================================================================================*
 *                                      write data                                                    *
 *====================================================================================================*/


typedef struct{
	uint *buf[3];	/* 0:ܥ֥å 1:Ŵܥ֥å 2:Ŵܥ֥å */
	uint block[3];	/* ֥åʥС */
	uint begin;		/* ܥ֥åν񤭹ϤΥǥå */
	uint last;		/* ܥ֥åΥǥå */
}WRITE_BLOCK;


//================================== PRIVATE ============================================


// ֥åơ֥뤢Υ֥å
// 0:1024֥å 1:4096֥å
static int blkNum[2][3] = {{256, 256  * 256, 256  * 256  * 256},
   	                       {1024,1024 * 1024,1024 * 1024 * 1024}};


// KХȥ֥å
STATIC INLINE int is4Kbyte(int din)
{
	return superBlock[din]->blockSize / 4096;
}

STATIC INLINE int divBlk(int din,int node,size_t startblk)
{
	return startblk / blkNum[is4Kbyte(din)][node];
}

STATIC INLINE int surpBlk(int din,int node,size_t startblk)
{
	return startblk % blkNum[is4Kbyte(din)][node];
}


// WRITE_BLOCK¤Τȴܥ֥å˥֥åƤ
// parameters : device inode, ХȰ, ܥΡ, ֥åֹ
// return : 0 ro error number
STATIC int getIndirectBlock(int din, size_t startBlk, int node, uint *m_blk, WRITE_BLOCK *o_blkStruct)
{
	size_t ioSize;
	int rest;

	// ֥åХåեγ
	if (o_blkStruct->buf[node] == NULL){
		if ((o_blkStruct->buf[node] = kmalloc(superBlock[din]->blockSize)) == NULL){
			return -ENOMEM;
		}
	}

	// ֥åγ̵äƤ롣
	if (*m_blk == 0){
		/* ߤΥ֥åХåե᤹ */
		if (o_blkStruct->block[node] != 0){
			rest = write_cache(din, o_blkStruct->buf[node], superBlock[din]->sectors, blkToSect(din, o_blkStruct->block[node]), &ioSize);
			if (rest != NOERR){
				return rest;
			}
			ASSERT(ioSize == superBlock[din]->sectors);
		}

		/* ֥åƤ */
		rest = getBlock(din);
		if (rest < 0){
			return rest;
		}
		o_blkStruct->block[node] = *m_blk = rest;
		memset(o_blkStruct->buf[node], 0, superBlock[din]->blockSize);
		rest = write_cache(din, o_blkStruct->buf[node], superBlock[din]->sectors, blkToSect(din, *m_blk), &ioSize);
		if (rest != NOERR){
			return rest;
		}
		ASSERT(ioSize == superBlock[din]->sectors);
	}
	else if (*m_blk != o_blkStruct->block[node]){
		/* ߤΥ֥åХåե᤹ */
		if (o_blkStruct->block[node] != 0){
			rest = write_cache(din, o_blkStruct->buf[node], superBlock[din]->sectors, blkToSect(din, o_blkStruct->block[node]), &ioSize);
			if (rest != NOERR){
				return rest;
			}
			ASSERT(ioSize == superBlock[din]->sectors);
		}

		/* ֥åɤ߹ */
		rest = read_cache(din, o_blkStruct->buf[node], superBlock[din]->sectors, blkToSect(din, *m_blk), &ioSize);
		if (rest != NOERR){
			return rest;
		}
		ASSERT(ioSize == superBlock[din]->sectors);
		o_blkStruct->block[node] = *m_blk;
	}
	
	if (0 < node){
		return getIndirectBlock(
			din, surpBlk(din, node - 1, startBlk), node - 1, &o_blkStruct->buf[node][divBlk(din, node - 1, startBlk)], o_blkStruct);
	}
	
	return 0;
}


// ֥åơ֥WRITE_BLOCK¤Τ˥åȤ
// WRITE_BLOCK¤ΤͰʳϽƤ뤳Ȥ
STATIC int getBlockTable(int din, INODE *inode, size_t start, WRITE_BLOCK *o_blkStruct)
{
	size_t startBlk = start;
	int rest;
	
	// Ǥ˳ƤƤơ֥뤬inodeʤöꥢ
	if (o_blkStruct->buf[0] == (uint*)inode){
		o_blkStruct->buf[0] = NULL;
	}

	if (startBlk < INODE_DIRECT_BLOCK_NUM){
		if (o_blkStruct->buf[0] != NULL){
			kfree(o_blkStruct->buf[0]);
		}
		o_blkStruct->buf[0] = (uint*)inode;
		o_blkStruct->block[0] = 0;
		o_blkStruct->begin = startBlk + OFFSETOF(INODE, i_block[0]) / sizeof(uint);
		o_blkStruct->last = OFFSETOF(INODE,i_block[INODE_DATA_INDIRECT1]) / sizeof(uint);
	}
	else if ((startBlk -= INODE_DIRECT_BLOCK_NUM)  <  blkNum[is4Kbyte(din)][0]){
		rest = getIndirectBlock(din, startBlk, 0, &inode->i_block[INODE_DATA_INDIRECT1], o_blkStruct);
		if (rest < 0){
			return rest;
		}
		o_blkStruct->begin = startBlk;
		o_blkStruct->last = blkNum[is4Kbyte(din)][0];
	}
	else if ((startBlk -= blkNum[is4Kbyte(din)][0]) < blkNum[is4Kbyte(din)][1]){
		rest = getIndirectBlock(din, startBlk, 1, &inode->i_block[INODE_DATA_INDIRECT2], o_blkStruct);
		if (rest < 0){
			return rest;
		}
		o_blkStruct->begin = startBlk % blkNum[is4Kbyte(din)][0];
		o_blkStruct->last = blkNum[is4Kbyte(din)][0];
	}
	else if ((startBlk -= blkNum[is4Kbyte(din)][1]) < blkNum[is4Kbyte(din)][2]){
		rest = getIndirectBlock(din, startBlk, 2, &inode->i_block[INODE_DATA_INDIRECT3], o_blkStruct);
		if (rest < 0){
			return rest;
		}
		o_blkStruct->begin = startBlk % blkNum[is4Kbyte(din)][0];
		o_blkStruct->last = blkNum[is4Kbyte(din)][0];
	}
	else{
		return -EFBIG;
	}
	
	return 0;
}


//================================== PROTECTED ==========================================


// ǡ֥å˽񤭹ࡣ
// parameters : device inode,inode,buffer,read size,begin byte
// return : 0 or error number
STATIC int writeData(int din, INODE *inode, void *buf, size_t lenByte, size_t beginByte)
{
	char *data = buf;
	char *dataBuf = NULL;
	int ext2BlkSize = superBlock[din]->blockSize;
	int blkSects = superBlock[din]->sectors;
	size_t startByte;
	size_t allLastByte;
	size_t startBlk;
	size_t lastBlk;
	size_t allLastBlk;
	size_t ioSize;
	WRITE_BLOCK blkStruct;
	int rest, c_blk;

	// 
	memset(&blkStruct, 0, sizeof(WRITE_BLOCK));

	// ߤκǸΥ֥å顢񤭹ǽޤǤ̤ƥ֥åƤ
	if (inode->i_size < beginByte){
		startBlk = ROUNDUP_DIV(inode->i_size, ext2BlkSize);
		allLastBlk = ROUNDUP_DIV(beginByte, ext2BlkSize);
		for (; startBlk < allLastBlk;){
			rest = getBlockTable(din, inode, startBlk, &blkStruct);
			if (rest < 0){
				goto END;
			}
			lastBlk = min(allLastBlk - startBlk + blkStruct.begin, blkStruct.last);
			for (c_blk = blkStruct.begin; c_blk < lastBlk; ++c_blk){
				ASSERT(blkStruct.buf[0][c_blk] == 0);

				rest = getBlock(din);
				if (rest < 0){
					goto END;
				}
				blkStruct.buf[0][c_blk] = rest;
			}
			startBlk += lastBlk - blkStruct.begin;
		}
	}

	// ֥åơ֥뤴Ȥ˥ǡ񤭹
	for (startByte = beginByte, allLastByte = lenByte + beginByte; startByte < allLastByte;){
		rest = getBlockTable(din, inode, startByte / ext2BlkSize, &blkStruct);
		if (rest < 0){
			goto END;
		}
		c_blk = blkStruct.begin;

		// ǽΥǡ֥å椫ϤޤȤϡö֥åɤ߹Ǥ񤭹
		if (startByte % ext2BlkSize){
			int blkOffset, copyLen;
	
			ASSERT(blkStruct.buf[0][c_blk] != 0);

			dataBuf = kmalloc(ext2BlkSize);
			if (dataBuf == NULL){
				return -ENOMEM;
			}
			rest = read_cache(din, dataBuf, blkSects, blkToSect(din, blkStruct.buf[0][c_blk]), &ioSize);
			if (rest != NOERR){
				goto END;
			}
			ASSERT(ioSize == blkSects);
			blkOffset = startByte % ext2BlkSize;
			copyLen = min(allLastByte - startByte, ext2BlkSize - blkOffset);
			memcpy(dataBuf + blkOffset, data, copyLen);
			data += copyLen;
			rest = write_cache(din, dataBuf, blkSects, blkToSect(din, blkStruct.buf[0][c_blk]), &ioSize);
			if (rest != NOERR){
				goto END;
			}
			ASSERT(ioSize == blkSects);
			++c_blk;
			
			// ǡκǸޤǽ񤭹Ǥ齪λ
			if (copyLen == allLastByte - startByte){
				break;
			}
		}
	
		lastBlk = min((allLastByte / ext2BlkSize) - (startByte / ext2BlkSize) + blkStruct.begin, blkStruct.last);
		for (; c_blk < lastBlk; ++c_blk){
			if (blkStruct.buf[0][c_blk] == 0){
				rest = getBlock(din);
				if (rest < 0){
					goto END;
				}
				blkStruct.buf[0][c_blk] = rest;
			}
			rest = write_cache(din, data, blkSects, blkToSect(din, blkStruct.buf[0][c_blk]), &ioSize);
			if (rest != NOERR){
				goto END;
			}
			ASSERT(ioSize == blkSects);
			data += ext2BlkSize;
		}
		
		// ǸΥǡ֥åǽ꤫ĥ֥åƤƤȤϡ
		// ֥åɤ߹Ǥǡ񤭹
		if ((lastBlk < blkStruct.last) && (allLastByte % ext2BlkSize)){
			if (dataBuf == NULL){
				if ((dataBuf = kmalloc(ext2BlkSize)) == NULL){
					return -ENOMEM;
				}
			}
			if (blkStruct.buf[0][lastBlk] == 0){
				if ((rest = getBlock(din)) < 0){
					goto END;
				}
				blkStruct.buf[0][lastBlk] = rest;
			}
			else{
				rest = read_cache(din, dataBuf, blkSects, blkToSect(din, blkStruct.buf[0][lastBlk]), &ioSize);
				if (rest != NOERR){
					goto END;
				}
				ASSERT(ioSize == blkSects);
			}
			memcpy(dataBuf, data, allLastByte % ext2BlkSize);
			rest = write_cache(din, dataBuf, blkSects, blkToSect(din, blkStruct.buf[0][lastBlk]), &ioSize);
			if (rest != NOERR){
				goto END;
			}
			ASSERT(ioSize == blkSects);

			break;
		}
		
		startByte = ROUNDDOWN(startByte, ext2BlkSize) + (lastBlk - blkStruct.begin) * ext2BlkSize;
	}
	rest = 0;
END:
	for (c_blk = 0; c_blk < 3; ++c_blk){
		int rest2;
		
		if (blkStruct.block[c_blk] != 0){
			rest2 = write_cache(din, blkStruct.buf[c_blk], blkSects, blkToSect(din, blkStruct.block[c_blk]), &ioSize);
			if (rest2 != NOERR){
				rest = rest2;
			}
			ASSERT(ioSize == blkSects);
			kfree(blkStruct.buf[c_blk]);
		}
	}
	kfree(dataBuf);

	return rest;
}


/*====================================================================================================*
 *                                       release                                                      *
 *====================================================================================================*/


typedef int (*RELEASE_DATA)(int,uint*,int,int);


static int releaseDirect(int,uint*,int,int);
static int releaseIndirect(int,uint*,int,int);
RELEASE_DATA releaseData[]={releaseDirect,releaseIndirect,releaseIndirect,releaseIndirect};


//================================== PRIVATE ============================================


// ľܥ֥åơ֥Υ֥å롣
// parameters : device inode,block table,number of blocks,node=0
// return : 0
STATIC int releaseDirect(int din, uint *table, int blk_num, int node)
{
	int i;

	for (i = 0; i < blk_num; ++i){
		releaseBlock(din, table[i]);
	}

	return 0;
}


// ܥ֥åơ֥Υ֥å롣
// parameters : device inode,block table,number of blocks,node
// return : 0 or error number
STATIC int releaseIndirect(int din, uint *table, int blk_num, int node)
{
	uint *blkBuf;
	int sectors = superBlock[din]->sectors;
	int nblk = 1 << ((EXT2_BLOCK_BASE_LOG + superBlock[din]->s_log_block_size - INT_LOG) * node);
	size_t ioSize;
	int rest;
	int i;

	blkBuf = kmalloc(superBlock[din]->blockSize);
	if (blkBuf == NULL){
		return -ENOMEM;
	}

	i = 0;
	for(; nblk < blk_num; blk_num -= nblk)
	{
		rest = read_cache(din, blkBuf, sectors, blkToSect(din, table[i]), &ioSize);
		if (rest != NOERR){
			goto END;
		}
		ASSERT(ioSize == sectors);
		rest = releaseData[node - 1](din, blkBuf, nblk, node - 1);
		if (rest < 0){
			goto END;
		}
		releaseBlock(din, table[i++]);
	}
	rest = read_cache(din, blkBuf, sectors, blkToSect(din, table[i]), &ioSize);
	if (rest != NOERR){
		goto END;
	}
	ASSERT(ioSize == sectors);
	rest = releaseData[node - 1](din, blkBuf, blk_num, node - 1);
	if (rest < 0){
		goto END;
	}
	releaseBlock(din, table[i]);

	rest = 0;
END:
	kfree(blkBuf);

	return rest;
}


//================================== PROTECTED ==========================================


// ǡ֥å롣
// return : 0 or error number
STATIC int releaseDataBlock(int din, INODE *inode)
{
	uint *blkBuf;
	int sectors = superBlock[din]->sectors;
	int blkSize = superBlock[din]->blockSize;
	int blk_num = ROUNDUP_DIV(inode->i_size, blkSize);
	int nblk;
	size_t ioSize;
	int rest;
	int blk, i;

	// ľܥ֥å
	blk = (INODE_DATA_INDIRECT1 <= blk_num)? INODE_DATA_INDIRECT1 : blk_num;
	releaseDirect(din, inode->i_block, blk, 0);
	for (i = 0; i < blk; ++i){
		inode->i_block[i] = 0;
	}
	if ((blk_num -= blk) == 0){
		goto END;
	}

	// ܥ֥å
	blkBuf = kmalloc(blkSize);
	if (blkBuf == NULL){
		return -ENOMEM;
	}
	blk = nblk = blkSize / sizeof(uint);
	for (i = INODE_DATA_INDIRECT1;; ++i)
	{
		rest = read_cache(din, blkBuf, sectors, blkToSect(din,inode->i_block[i]), &ioSize);
		if (rest != NOERR){
			goto ERR;
		}
		ASSERT(ioSize == sectors);
		rest = releaseData[i - INODE_DATA_INDIRECT1](din, blkBuf, (blk < blk_num)? blk : blk_num, i - INODE_DATA_INDIRECT1);
		if (rest < 0){
			goto ERR;
		}
		releaseBlock(din, inode->i_block[i]);
		inode->i_block[i] = 0;

		blk_num -= blk;
		if (blk_num <= 0){
			break;
		}
		blk *= nblk;
	}
	kfree(blkBuf);
END:
	inode->i_size = 0;

	return 0;
ERR:
	kfree(blkBuf);

	return rest;
}


/*====================================================================================================*
 *                                          misc                                                      *
 *====================================================================================================*/


//================================== PRIVATE ============================================


static BITMAP inodeBm[MAX_DEVICE_OPEN];


//================================== PROTECTED ==========================================


// inodeֹ椫顢inodeΥ֥åХåեɤ߹ࡣ
// parameters : device inode,inode number,block buffer
// return : inode addres or error number;
STATIC INLINE int readInode(int din,uint inumber,INODE *sector_buf)
{
	int gdesc;
	int index;
	uint sector;
	size_t ioSize;
	int error;

	gdesc = (inumber - 1) / superBlock[din]->s_inodes_per_group;
	index = (inumber - 1) % superBlock[din]->s_inodes_per_group;
	sector = blkToSect(din,grpDesc[din][gdesc].bg_inode_table) + index * sizeof(INODE) / superBlock[din]->sectorSize;
	error = read_cache(din, sector_buf, 1, sector, &ioSize);
	if (error != NOERR){
		return error;
	}
	ASSERT(ioSize == 1);

	return (int)sector_buf + index * sizeof(INODE) % superBlock[din]->sectorSize;
}


// inodeǥ˽񤭹ࡣ
// parameters : device inode,inode number,block buffer
// return : 0 or error number
STATIC INLINE int writeInode(int din,uint inumber,INODE *sector_buf)
{
	int gdesc;
	int index;
	uint sector;
	size_t ioSize;
	int rest;

	gdesc = (inumber - 1) / superBlock[din]->s_inodes_per_group;
	index = (inumber - 1) % superBlock[din]->s_inodes_per_group;
	sector = blkToSect(din,grpDesc[din][gdesc].bg_inode_table) + index * sizeof(INODE) / superBlock[din]->sectorSize;
	rest = write_cache(din, sector_buf, 1, sector, &ioSize);
	if (rest != NOERR){
		return rest;
	}
	ASSERT(ioSize == 1);

	return 0;
}


// inodeֹ롣
// parameters : device inode,file type
// return : inode number or error = 0
STATIC uint getNewInode(int din,int type)
{
	GROUP_DESC *gd = grpDesc[din];
	SUPER_BLOCK *sb = superBlock[din];
	uint inode;
	size_t ioSize;
	int i,last;

	/* ߤΥ롼פ˶inodeʤ */
	if (gd[inodeBm[din].currentGd].bg_free_inodes_count == 0){
		int sectors=sb->sectors;

		/* ߤΥӥåȥޥåפ᤹ */
		if (write_cache(din, inodeBm[din].bitmap, sectors, blkToSect(din, gd[inodeBm[din].currentGd].bg_inode_bitmap), &ioSize) != NOERR){
			return 0;
		}
		ASSERT(ioSize == sectors);

		for (i = inodeBm[din].currentGd + 1;; ++i)
		{
			if (i == grpDescNum[din]){
				i = 0;
			}
			if (i == inodeBm[din].currentGd){
				return 0;
			}
			if (gd[i].bg_free_inodes_count != 0){
				break;
			}
		}
		if (read_cache(din, inodeBm[din].bitmap, sectors, blkToSect(din, gd[i].bg_inode_bitmap), &ioSize) != NOERR){
			return 0;
		}
		ASSERT(ioSize == sectors);
		inodeBm[din].currentGd = i;
	}

	/* ӥåȥޥåפθ */
	i=inodeBm[din].currentByte;
	last=sb->s_inodes_per_group/BYTE_BIT;
	while(inodeBm[din].bitmap[i]==0xff)
		if(++i>=last)i=0;
	inodeBm[din].currentByte=i;
	inode=i*BYTE_BIT+emptyBit[inodeBm[din].bitmap[i]];
	inodeBm[din].bitmap[i]|=1<<emptyBit[inodeBm[din].bitmap[i]];

	/* 롼ץǥץȥѡ֥å򹹿롣 */
	--gd[inodeBm[din].currentGd].bg_free_inodes_count;
	--sb->s_free_inodes_count;
	
	/* 롼ץǥץΥǥ쥯ȥꥫȤ䤹 */
	if ((type & EXT2_S_IFMT) == EXT2_S_IFDIR)
		grpDesc[din][inodeBm[din].currentGd].bg_used_dirs_count += 1;

	return inode+inodeBm[din].currentGd*sb->s_inodes_per_group+1;
}


// inodeӥåȥޥåפγ롼ץǥץѹ
// parameters : device inode,inode number
// return : 0 or error number
STATIC int releaseInodeBitAndGrp(int din, int inumber, int type)
{
	uchar *bitmap;
	int grp;
	SUPER_BLOCK *sb = superBlock[din];
	int sectors = sb->sectors;
	size_t ioSize;
	int a, rest;

	/* ֥åӥåȥޥåפθ */
	grp = (inumber - 1) / (sb->s_inodes_per_group);
	if (grp != inodeBm[din].currentGd){
		if ((bitmap = kmalloc(sb->blockSize)) == NULL){
			return -ENOMEM;
		}
		rest = read_cache(din, bitmap, sectors, blkToSect(din, grpDesc[din][grp].bg_inode_bitmap), &ioSize);
		if (rest != NOERR){
			return rest;
		}
		ASSERT(ioSize == sectors);
	}
	else{
		bitmap = inodeBm[din].bitmap;
	}

	/* ӥåȥޥåפ˽񤭹ࡣ */
	a = inumber - grp * sb->s_inodes_per_group - 1;
	bitmap[a / BYTE_BIT] &= ~(1 << a % BYTE_BIT);
	if (grp != inodeBm[din].currentGd)
	{
		/* ߤΥ֥å롼פȰפ뤫 */
		if (grp == blockBm[din].currentGd)
		{
			uchar *buf;

			buf = inodeBm[din].bitmap;
			inodeBm[din].bitmap = bitmap;
			bitmap = buf;
			inodeBm[din].currentGd = grp;
			inodeBm[din].currentByte = a / BYTE_BIT;
		}
		rest = write_cache(din, bitmap, sectors, blkToSect(din, grpDesc[din][grp].bg_inode_bitmap), &ioSize);
		if (rest != NOERR){
			return rest;
		}
		ASSERT(ioSize == sectors);
		kfree(bitmap);
	}

	/* ֥åȤ䤹 */
	++grpDesc[din][grp].bg_free_inodes_count;
	++sb->s_free_inodes_count;

	/* ǥ쥯ȥʤǥ쥯ȥꥫȤ򸺤餹 */
	if ((type & EXT2_S_IFMT) == EXT2_S_IFDIR)
		--grpDesc[din][grp].bg_used_dirs_count;

	return 0;
}


// ̾եʤinode󥯥Ȥ򸺤餷ƥ󥯣ʤ
// ǥ쥯ȥʤ󥯥ȣΤȤ
// parameters : device inode,inode
// return : number of links or errnor number
STATIC int unlinkInode(int din,INODE *inode)
{
	INODE *sect_buf;
	int rest;
	
	/* ̾ե */
	if ((inode->i_mode & EXT2_S_IFMT) == EXT2_S_IFREG){
		/* 󥯥Ȥ򸺤餹 */
		inode->i_links_count -= 1;

		ASSERT(0 <= (short)inode->i_links_count);
	}
	/* ǥ쥯ȥ */
	else if ((inode->i_mode & EXT2_S_IFMT) == EXT2_S_IFDIR){
		ASSERT(2 <= inode->i_links_count);

		/* 󥯥2礭н */
		if (2 < inode->i_links_count){
			return inode->i_links_count;
		}

		inode->i_links_count = 0;
	}
	/* ¾Υե륿פ̤б */
	else{
		return -EOPNOTSUPP;
	}

	/* 󥯥Ȥ0ʤ */
	if (inode->i_links_count == 0){
		/* ǡ֥å롣 */
		rest = releaseDataBlock(din,inode);
		if (rest < 0){
			return rest;
		}

		/* inodeΰ */
		rest = releaseInodeBitAndGrp(din,inode->i_number,inode->i_mode);
		if (rest < 0){
			return rest;
		}

		/* delete timeꤹ */
		inode->i_dtime = (uint)sys_time(NULL);
	}

	/* inode᤹ */
	sect_buf = kmalloc(superBlock[din]->sectorSize);
	if (sect_buf == NULL){
		return -ENOMEM;
	}
	rest = readInode(din,inode->i_number,sect_buf);
	if (rest < 0){
		return rest;
	}
	memcpy((void*)rest,inode,sizeof(INODE));
	rest = writeInode(din,inode->i_number,sect_buf);
	if (rest < 0){
		return rest;
	}
	kfree(sect_buf);

	return inode->i_links_count;
}


// ޥȻν
// parameters : device inode,򳫻Ϥ֥å롼ץʥС
// return : 0 or error nomber
STATIC int initInode(int din,int begin_gr)
{
	GROUP_DESC *gd=grpDesc[din];
	SUPER_BLOCK *sb=superBlock[din];
	INODE *inode,*sectBuf;
	size_t ioSize;
	int i,rest;


	/* inodeӥåȥޥåפɤ߹ࡣ */
	inodeBm[din].bitmap = kmalloc(sb->blockSize);
	if (inodeBm[din].bitmap == NULL)
		return -ENOMEM;
	inodeBm[din].currentGd = begin_gr;
	inodeBm[din].currentByte = 0;

	i = begin_gr;
	do
	{
		if (gd[i].bg_free_inodes_count > 0)
		{
			rest = read_direct(din,inodeBm[din].bitmap,sb->sectors,blkToSect(din,gd[i].bg_inode_bitmap), &ioSize);
			if (rest != NOERR){
				return rest;
			}
			inodeBm[din].currentGd = i;
			for (i = 0; inodeBm[din].bitmap[i] == 0xff; ++i);
			inodeBm[din].currentByte = i;
			break;
		}
		if (++i == grpDescNum[din]){
			i = 0;
		}
	}while (i != begin_gr);

	/* 롼inodeν */
	sectBuf = kmalloc(superBlock[din]->sectorSize);
	if (sectBuf == NULL)
		return -ENOMEM;
	rest = readInode(din,EXT2_ROOT_INODE_INDEX,sectBuf);
	if (rest < 0)
		goto ERR;
	inode = (INODE*)rest;
	inode->i_number = EXT2_ROOT_INODE_INDEX;
	rest = writeInode(din,EXT2_ROOT_INODE_INDEX,sectBuf);
	if (rest < 0)
		goto ERR;
	kfree(sectBuf);

	return 0;

ERR:
	kfree(sectBuf);

	return rest;
}


// ޥȻν
// parameters : device inode
STATIC int umountInode(int din)
{
	size_t ioSize;

	/* inodeӥåȥޥåפǥ˽᤹ */
	write_cache(
		din, 
		inodeBm[din].bitmap, 
		superBlock[din]->sectors, 
		blkToSect(din, grpDesc[din][inodeBm[din].currentGd].bg_inode_bitmap),
		&ioSize);
	kfree(inodeBm[din].bitmap);
	ASSERT(ioSize == superBlock[din]->sectors);

	return 0;
}


/********************************************************************************************************
 *
 * Directory
 *
 ********************************************************************************************************/


enum{
	EXT2_NAME_LEN = 255,

	/* directory file types. */
	EXT2_DIR_UNKNOWN = 0,
	EXT2_DIR_REG_FILE,
	EXT2_DIR_DIR,
	EXT2_DIR_CHRDEV,
	EXT2_DIR_BLKDEV,
	EXT2_DIR_FIFO,
	EXT2_DIR_SOCK,
	EXT2_DIR_SYMLINK,
	EXT2_DIR_MAX,
};


typedef struct{
	uint inode;				/* inode number. */
	ushort rec_len;			/* directory entry length. */
	uchar name_len;			/* name length. */
	uchar file_type;		/* file type. */
	char name[0];			/* File name. */
}DIR_ENTRY;

typedef struct{
	const char *path;
	DIR_ENTRY *dirBuf;
	uint blkNumber;
}DIR_PARAM;

typedef int (*READ_DIR)(int,uint*,int,DIR_PARAM*,int,int (**READ_DIR)());


#define ROUNDUP4(a) (((a+3)>>2)<<2)		/* 4Ƿ夲 */


//================================== PRIVATE ============================================

/*
 * search entry.
 * return : directory entry address or failed=NULL
 */
STATIC DIR_ENTRY *searchEntry(
	int din,				// device inode
	const char *path,
	DIR_ENTRY *entry)		// entry buffer
{
	DIR_ENTRY *last;

	last = (DIR_ENTRY*)((char*)entry + superBlock[din]->blockSize);
	if (entry->inode == 0) {
		entry = (DIR_ENTRY*)((char*)entry + entry->rec_len);
	}

	for (;entry < last; entry = (DIR_ENTRY*)((char*)entry + entry->rec_len)) {
		if (cmpPath(entry->name, path, entry->name_len) == 0) {
			return entry;
		}
	}

	return NULL;
}


/*
 * ȥ꡼õ
 * parameters : device inode,directory block,directory parameters struct
 * return : directory entry address or failed=NULL
 */
STATIC DIR_ENTRY *searchEntrySpace(int din,DIR_PARAM *dparam)
{
	int add_len,ent_len;
	DIR_ENTRY *entry,*last,*new;


	add_len=strlen(dparam->path)+sizeof(DIR_ENTRY);
	entry=dparam->dirBuf;
	last=(DIR_ENTRY*)((char*)dparam->dirBuf+superBlock[din]->blockSize);

	if(entry->inode == 0)
	{
		if(entry->rec_len>=add_len)
		{
			entry->name_len=add_len-sizeof(DIR_ENTRY);
			return entry;
		}
		else entry=(DIR_ENTRY*)((char*)entry + entry->rec_len);
	}

	for(;entry<last;entry=(DIR_ENTRY*)((char*)entry + entry->rec_len))
	{
		ent_len=ROUNDUP4(entry->name_len+sizeof(DIR_ENTRY));
		if((entry->rec_len-ent_len)>=add_len)
		{
			new=(DIR_ENTRY*)((char*)entry+ent_len);
			new->rec_len=entry->rec_len-ent_len;		/* ȥ꡼ */
			new->name_len=add_len-sizeof(DIR_ENTRY);	/* ͡ॵ */
			entry->rec_len=ent_len;
			return new;
		}
	}

	return NULL;
}


/*
 * ȥ꡼뤫å롣
 * parameters : device inode,directory block,directory parameters struct
 * return : 0 or ȥ꡼=-1
 */
STATIC int checkEntry(int din,DIR_PARAM *dparam)
{
	DIR_ENTRY *entry;


	entry=dparam->dirBuf;

	if(entry->inode == 0)
	{
		if(entry->rec_len==superBlock[din]->blockSize)return 0;
	}
	else
	{
		if (cmpPath(entry->name,".",entry->name_len)==0)
		{
			entry=(DIR_ENTRY*)((char*)entry + entry->rec_len);
			if(entry->rec_len==superBlock[din]->blockSize-(sizeof(DIR_ENTRY)+sizeof(int)))return 0;
		}
	}

	return -1;
}


/*
* ȥ꡼롣
 * parameters : delete entry,directory buffer
 */
STATIC INLINE void delEntry(DIR_ENTRY *entry,DIR_ENTRY *dirBuf)
{
	DIR_ENTRY *p;

	if (entry == dirBuf) {
		entry->inode = 0;
	}
	else {
		for (p = dirBuf; (char*) p + p->rec_len != (char*) entry; p = (DIR_ENTRY*) ((char*) p + p->rec_len));
		p->rec_len += entry->rec_len;
	}
}


/*
 * ľܥ֥åɹߡ
 * ͤͿ줿ХåեΥǥ쥯ȥꥢɥ쥹
 * parameters : device inode,block table,number of block table indexs,directory parameters struct,node=0,READ_DIR struct
 * return : directory entry address or error number
 */
STATIC int readDirectDir(int din,uint *table,int blk_num,DIR_PARAM *dparam,int node,READ_DIR *readDirBlock)
{
	int sectors = superBlock[din]->sectors;
	DIR_ENTRY *entry;
	size_t ioSize;
	int i;
	int error;

	for (i = 0; i < blk_num; ++i){
		error = read_cache(din, dparam->dirBuf, sectors, blkToSect(din, table[i]), &ioSize);
		if (error != NOERR){
			return -EIO;
		}
		ASSERT(ioSize == sectors);
		if ((entry = searchEntry(din, dparam->path, dparam->dirBuf)) != NULL){
			dparam->blkNumber = table[i];
			return (int)entry;
		}
	}

	return -ENOENT;
}


/*
 * ľܥ֥åɹߡ
 * ͤͿ줿ХåեΥǥ쥯ȥꥢɥ쥹
 * parameters : device inode,block table,number of block table indexs,directory parameters struct,node=0,READ_DIR struct
 * return : directory entry address or error number
 */
STATIC int searchDirectDir(int din,uint *table,int blk_num,DIR_PARAM *dparam,int node,READ_DIR *readDirBlock)
{
	int sectors=superBlock[din]->sectors;
	DIR_ENTRY *entry;
	size_t ioSize;
	int i;
	int error;

	for (i = 0; i < blk_num; ++i){
		error = read_cache(din, dparam->dirBuf, sectors, blkToSect(din, table[i]), &ioSize);
		if (error != NOERR){
			return error;
		}
		ASSERT(ioSize == sectors);
		if ((entry = searchEntrySpace(din, dparam)) != NULL){
			dparam->blkNumber = table[i];
			return (int)entry;
		}
	}

	return -ENOENT;
}


/*
 * ľܥ֥åɹߡ
 * parameters : device inode,block table,number of block table indexs,directory parameters struct,node=0,READ_DIR struct
 * return : ʤ=-ENOENT or error number
 */
STATIC int checkDirectDir(int din,uint *table,int blk_num,DIR_PARAM *dparam,int node,READ_DIR *readDirBlock)
{
	int sectors = superBlock[din]->sectors;
	size_t ioSize;
	int i;
	int error;

	for (i = 0; i < blk_num; ++i){
		error = read_cache(din, dparam->dirBuf, sectors, blkToSect(din, table[i]), &ioSize);
		if (error != NOERR){
			return error;
		}
		ASSERT(ioSize == sectors);
		if (checkEntry(din, dparam) == -1){
			return -EEXIST;
		}
	}

	return -ENOENT;
}


/*
 * Υǥ쥯ȥ֥åõ
 * DIR_PARAMblkNumber0ʤǽΥ֥å0ʾʤblkNumberμΥ֥å֤
 * parameters : device inode,block table,number of block table indexs,directory parameters struct,node=0,READ_DIR struct
 * return : next block or error number
 */
STATIC int searchNextBlock(int din,uint *table,int blk_num,DIR_PARAM *dparam,int node,READ_DIR *readDirBlock)
{
	int i;


	if(dparam->blkNumber==0)return *table;
	else
		for(i=0;i<blk_num;++i)
			if(table[i]==dparam->blkNumber)
			{
				if(++i<blk_num)return table[i];
				dparam->blkNumber=0;
				break;
			}

	return -ENOENT;
}


/*
 * ɹߡ
 * ͤͿ줿ХåեΥǥ쥯ȥꥢɥ쥹
 * parameters : device inode,block table,number of block table indexs,directory parameters struct,node,READ_DIR struct
 * return : directory entry address of error number
 */
STATIC int readIndirectDir(int din,uint *table,int blk_num,DIR_PARAM *dparam,int node,READ_DIR *readDirBlock)
{
	uint *buf;
	int sectors = superBlock[din]->sectors;
	int nblk = 1 << ((EXT2_BLOCK_BASE_LOG + superBlock[din]->s_log_block_size - INT_LOG) * node);
	size_t ioSize;
	int rest;
	int i;

	buf = kmalloc(superBlock[din]->blockSize);
	if (buf == NULL){
		return -ENOMEM;
	}

	for (i = 0; nblk < blk_num; blk_num -= nblk, ++i){
		rest = read_cache(din, buf, sectors, blkToSect(din, table[i]), &ioSize);
		if (rest != NOERR){
			goto END;
		}
		ASSERT(ioSize == sectors);
		rest = readDirBlock[node - 1](din, buf, nblk, dparam, node - 1, readDirBlock);
		if (rest != -ENOENT){
			goto END;
		}
	}
	rest = read_cache(din, buf, sectors, blkToSect(din, table[i]), &ioSize);
	if (rest != NOERR){
		goto END;
	}
	ASSERT(ioSize == sectors);
	rest = readDirBlock[node - 1](din, buf, blk_num, dparam, node - 1, readDirBlock);

END:
	kfree(buf);

	return rest;
}


/*
 * read directory.
 * ͤͿ줿ХåեΥǥ쥯ȥꥢɥ쥹
 * parameters : path,device inode,inode,directory parameters struct,READ_DIR struct
 * return : directory address or error number
 */
STATIC int readDir(int din,INODE *inode,DIR_PARAM *dparam,READ_DIR *readDirBlock)
{
	uint *buf;
	int sectors = superBlock[din]->sectors;
	int blkSize = superBlock[din]->blockSize;
	int blk_num = ROUNDUP_DIV(inode->i_size,blkSize);
	int nblk;
	size_t ioSize;
	int rest;
	int blk, i;

	/* ľܥ֥å */
	blk = (blk_num >= INODE_DATA_INDIRECT1)? INODE_DATA_INDIRECT1 : blk_num;
	rest = readDirBlock[0](din, inode->i_block, blk, dparam, 0, readDirBlock);
	if (rest != -ENOENT){
		return rest;
	}
	if ((blk_num -= blk) == 0){
		return -ENOENT;
	}

	/* ܥ֥å */
	buf = kmalloc(blkSize);
	if (buf == NULL){
		return -ENOMEM;
	}
	nblk = blkSize / sizeof(uint);
	blk = nblk;
	for (i = INODE_DATA_INDIRECT1; ; ++i){
		rest = read_cache(din, buf, sectors, blkToSect(din, inode->i_block[i]), &ioSize);
		if (rest != NOERR){
			goto END;
		}
		ASSERT(ioSize == sectors);
		rest = readDirBlock[i - INODE_DATA_INDIRECT1](din, buf, (blk_num > blk)? blk : blk_num, dparam, i - INODE_DATA_INDIRECT1, readDirBlock);
		if (rest != -ENOENT){
			goto END;
		}
		if ((blk_num -= blk) <= 0){
			break;
		}
		blk *= nblk;
	}
END:
	kfree(buf);

	return rest;
}


static READ_DIR readDirBlock[] = {readDirectDir, readIndirectDir, readIndirectDir, readIndirectDir};
static READ_DIR addDirBlock[] = {searchDirectDir, readIndirectDir, readIndirectDir, readIndirectDir};
static READ_DIR checkDirBlock[] = {checkDirectDir, readIndirectDir, readIndirectDir, readIndirectDir};
static READ_DIR searchDirBlock[] = {searchNextBlock, readIndirectDir, readIndirectDir, readIndirectDir};


/*
 * ǥ쥯ȥ˥ȥ꡼ĤäƤ뤫ǧ롣
 * parameters : device inode,directory inode
 * return : ȥ꡼ʤ=-ENOENT or error number
 */
STATIC int INLINE checkDir(int din,INODE *inode)
{
	int rest;
	DIR_PARAM param;


	if((param.dirBuf=(DIR_ENTRY*)kmalloc(superBlock[din]->blockSize))==NULL)return -ENOMEM;
	rest=readDir(din,inode,&param,checkDirBlock);
	kfree(param.dirBuf);

	return rest;
}


//================================== PROTECTED ==========================================


// inodeե륿פǥ쥯ȥե륿פѴ
STATIC uchar getDirFileTypeFromFileType(const ushort type)
{
	switch (type & EXT2_S_IFMT)
	{
		case EXT2_S_IFSOCK:
			return EXT2_DIR_SOCK;
		case EXT2_S_IFLNK:
			return EXT2_DIR_SYMLINK;
		case EXT2_S_IFREG:
			return EXT2_DIR_REG_FILE;
		case EXT2_S_IFBLK:
			return EXT2_DIR_BLKDEV;
		case EXT2_S_IFDIR:
			return EXT2_DIR_DIR;
		case EXT2_S_IFCHR:
			return EXT2_DIR_CHRDEV;
		case EXT2_S_IFIFO:
			return EXT2_DIR_FIFO;
		default:
			return EXT2_DIR_UNKNOWN;
	}
}


// ѥindoe
// return : error number
STATIC INLINE int getInodeFromPath(const int din, INODE_PARAM *m_param, DIR_PARAM *m_dparam)
{
	DIR_ENTRY *entry;
	int rest;

	/* ǥ쥯ȥꥨȥ꡼õ */
	if ((rest = readDir(din, m_param->inodeData, m_dparam, readDirBlock)) < 0){
		return rest;
	}
	entry = (DIR_ENTRY*)rest;
	if (entry->file_type != EXT2_DIR_DIR){
		return -ENOTDIR;
	}

	/* inode롣 */
	if ((rest = readInode(din, entry->inode, m_param->inodeBuf)) < 0){
		return rest;
	}
	m_param->inodeData = (INODE*)rest;
	m_param->inodeData->i_number = entry->inode;

	/* '/'μ˥ѥʤ롣 */
	for (m_dparam->path += entry->name_len + 1; *m_dparam->path == '/'; ++m_dparam->path){
	}

	return NOERR;
}


STATIC int searchDirOrSymbolic(int din, INODE_PARAM *param, DIR_ENTRY *dirBuf)
{
	DIR_PARAM dparam;
	int error;

	dparam.dirBuf = dirBuf;

	for (dparam.path = param->path; (isLastPath(dparam.path) < 0) && ((param->inodeData->i_mode & EXT2_S_IFLNK) == 0);){
		error = getInodeFromPath(din, param, &dparam);
		if (error != NOERR){
			return error;
		}
	}
	param->path = dparam.path;

	return 0;
}


/*
 * ƥǥ쥯ȥޤǸơparam˼Υѥȥǥ쥯ȥinode롣
 * return : 0 or error number
 */
STATIC int searchParentDir(int din, INODE_PARAM *param, DIR_ENTRY *dirBuf)
{
	int rest;
	DIR_ENTRY *entry;
	DIR_PARAM dparam;

	dparam.dirBuf = dirBuf;

	for (dparam.path = param->path; isLastPath(dparam.path) < 0;){
		/* ǥ쥯ȥꥨȥ꡼õ */
		if ((rest = readDir(din, param->inodeData, &dparam, readDirBlock)) < 0){
			return rest;
		}
		entry = (DIR_ENTRY*)rest;
		if (entry->file_type != EXT2_DIR_DIR){
			return -ENOTDIR;
		}

		/* inode롣 */
		if ((rest = readInode(din, entry->inode, param->inodeBuf)) < 0){
			return rest;
		}
		param->inodeData = (INODE*)rest;
		param->inodeData->i_number = entry->inode;
		
		/* '/'μ˥ѥʤ롣 */
		for (dparam.path += entry->name_len + 1; *dparam.path == '/'; ++dparam.path){
		}
	}
	param->path = dparam.path;

	return 0;
}


// ƥǥ쥯ȥ꤫ѥinode롣
// return : error number
STATIC int getInodeNumber(const int din, INODE_PARAM *param, DIR_ENTRY *dirBuf, uint *o_inode)
{
	int rest;
	DIR_ENTRY *entry;
	DIR_PARAM dparam;

	dparam.dirBuf = dirBuf;
	dparam.path = param->path;

	// ǥ쥯ȥꥨȥ꡼õ
	rest = readDir(din, param->inodeData, &dparam, readDirBlock);
	if (rest < 0){
		return rest;
	}
	entry = (DIR_ENTRY*)rest;

	*o_inode = entry->inode;

	return NOERR;
}


// return : error number
STATIC int getInodeData(const int din, const uint inodeNum, INODE_PARAM *param)
{
	int rest;

    // inodeǡ롣
    rest = readInode(din, inodeNum, param->inodeBuf);
	if (rest < 0){
		return rest;
	}
	param->inodeData = (INODE*)rest;
	param->inodeData->i_number = inodeNum;

	return NOERR;
}


/*
 * ƥǥ쥯ȥ꤫ѥΥȥ꡼롣
 * parameters : device inode,INODE_PARAM struct,directory block buffer
 * return : 0 or error number
 */
STATIC int getInode(int din, INODE_PARAM *param, DIR_ENTRY *dirBuf) 
{
	int rest;
	DIR_ENTRY *entry;
	DIR_PARAM dparam;

	dparam.dirBuf = dirBuf;
	dparam.path = param->path;

	/* ǥ쥯ȥꥨȥ꡼õ */
	if ((rest = readDir(din, param->inodeData, &dparam, readDirBlock)) < 0){
		return rest;
	}
	entry = (DIR_ENTRY*)rest;

    /* inode롣 */
	if ((rest = readInode(din, entry->inode, param->inodeBuf)) < 0){
		return rest;
	}
	param->inodeData = (INODE*)rest;
	param->inodeData->i_number = entry->inode;

	return 0;
}


/*
 * ƥǥ쥯ȥ˥ȥ꡼ä롣
 * parameters : device inode,file type,inode number,INODE_PARAM struct,directory block buffer
 * return : 0 or error number
 */
STATIC int addDirEnt(int din,uchar type,uint inumber,INODE_PARAM *param,DIR_ENTRY *dirBuf)
{
	int blkSize = superBlock[din]->blockSize;
	int sectors = superBlock[din]->sectors;
	DIR_ENTRY *new_ent;
	DIR_PARAM dparam;
	size_t ioSize;
	int rest;

	dparam.dirBuf = dirBuf;
	dparam.path = param->path;

	/* ѥ¸ߤ뤫ǧ */
	if (readDir(din,param->inodeData,&dparam,readDirBlock) > 0){
		return -EEXIST;
	}

	/* ǥ쥯ȥ꤫ȥ꡼õ */
	rest = readDir(din,param->inodeData,&dparam,addDirBlock);
	if (rest == -ENOENT){
		/* ˥ǥ쥯ȥ˥֥åƤ롣 */
		if(superBlock[din]->s_free_blocks_count>0){
			new_ent=dparam.dirBuf;
			new_ent->rec_len=blkSize;
			new_ent->name_len=strlen(dparam.path);
			memcpy(new_ent->name,param->path,new_ent->name_len);
			new_ent->inode = inumber;
			new_ent->file_type = type;

			rest = writeData(din, param->inodeData, new_ent,blkSize, param->inodeData->i_size);
			if (0 < rest){
				param->inodeData->i_size += blkSize;
				param->inodeData->i_blocks = ROUNDUP_DIV(param->inodeData->i_size,INODE_BLOCK_SIZE);
			}
			else{
				return rest;
			}
		}
		else{
			return -ENOSPC;
		}
	}
	else if (0 < rest){
		/* ȥ꡼롣 */
		new_ent = (DIR_ENTRY*)rest;
		memcpy(new_ent->name,param->path,new_ent->name_len);
		new_ent->inode = inumber;
		new_ent->file_type = type;
		rest = write_cache(din, dparam.dirBuf, sectors, blkToSect(din, dparam.blkNumber), &ioSize);
		if (rest != NOERR){
			return rest;
		}
		ASSERT(ioSize == sectors);
	}
	else{
		return rest;
	}

	/* ƥǥ쥯ȥinode򹹿 */
	param->inodeData->i_atime = param->inodeData->i_mtime = (uint)sys_time(NULL);
	if (type == EXT2_DIR_DIR){
		param->inodeData->i_links_count += 1;
	}
	rest = writeInode(din,param->inodeData->i_number,param->inodeBuf);
	if (rest < 0){
		return rest;
	}
	
	return 0;
}


/*
 * ƥǥ쥯ȥ꤫ե륨ȥ꡼롣
 * ȥ꡼inodeϰparam롣
 * parameters : device inode,INODE_PARAM struct,directory block buffer
 * return error number
 */
STATIC int delFileEntry(int din, INODE_PARAM *param, DIR_ENTRY *dirBuf)
{
	int rest;
	DIR_ENTRY *entry;
	DIR_PARAM dparam;
	size_t ioSize;

	dparam.dirBuf = dirBuf;
	dparam.path = param->path;

	/* ǥ쥯ȥꥨȥ꡼õ */
	if ((rest = readDir(din,param->inodeData,&dparam,readDirBlock)) < 0)
		return rest;
	entry = (DIR_ENTRY*)rest;

	if (entry->file_type == EXT2_DIR_REG_FILE)
	{
		if ((rest = readInode(din,entry->inode,param->inodeBuf)) < 0)
			return rest;
		param->inodeData = (INODE*)rest;
		param->inodeData->i_number = entry->inode;
		delEntry(entry,dparam.dirBuf);
		rest = write_cache(din, dparam.dirBuf, superBlock[din]->sectors, blkToSect(din, dparam.blkNumber), &ioSize);
		if (rest != NOERR){
			return rest;
		}
		ASSERT(ioSize == superBlock[din]->sectors);
	}
	else
		return -ENOENT;

	return NOERR;
}


/*
 * ƥǥ쥯ȥ꤫ǥ쥯ȥꥨȥ꡼롣
 * ȥ꡼inodeϰparam롣
 * parameters : name,device inode,INODE_PARAM struct,directory block buffer
 * return error number
 */
STATIC int delDirEntry(int din,INODE_PARAM *param,DIR_ENTRY *dirBuf)
{
	DIR_ENTRY *entry;
	DIR_PARAM dparam;
	INODE *parent_inode;
	INODE *parent_ibuf;
	size_t ioSize;
	int rest;

	dparam.dirBuf = dirBuf;
	dparam.path = param->path;

	/* ǥ쥯ȥꥨȥ꡼õ */
	rest = readDir(din, param->inodeData, &dparam, readDirBlock);
	if (rest < 0){
		return rest;
	}
	entry = (DIR_ENTRY*)rest;

	if (entry->file_type == EXT2_DIR_DIR){
		parent_inode = param->inodeData;
		parent_ibuf = param->inodeBuf;

		param->inodeBuf = kmalloc(superBlock[din]->sectorSize);
		if (param->inodeBuf == NULL){
			return -ENOMEM;
		}

		rest = readInode(din, entry->inode, param->inodeBuf);
		if (rest < 0){
			goto ERR;
		}
		param->inodeData = (INODE*)rest;
		param->inodeData->i_number = entry->inode;
		rest = checkDir(din, param->inodeData);
		if (rest != -ENOENT){
			goto ERR;
		}
		delEntry(entry, dparam.dirBuf);
		rest = write_cache(din, dparam.dirBuf, superBlock[din]->sectors, blkToSect(din, dparam.blkNumber), &ioSize);
		if (rest != NOERR){
			goto ERR;
		}
		ASSERT(ioSize == superBlock[din]->sectors);
		--parent_inode->i_links_count;
		rest = writeInode(din, parent_inode->i_number, parent_ibuf);
		if (rest < 0){
			goto ERR;
		}

		kfree(parent_ibuf);
	}
	else{
		return -ENOENT;
	}

	return NOERR;
ERR:
	kfree(parent_ibuf);

	return rest;
}


/*
 * ȥ꡼inodeoldnew촹롣
 * parameters : device inode,old inode parameter,new inode parameter,old block buffer,new block buffer
 *              old path,new path
 * return : 0 or error number
 */
STATIC int changeInode(
	int din,
	INODE_PARAM *old_param,
	INODE_PARAM *new_param,
	DIR_ENTRY *old_dbuf,
	DIR_ENTRY *new_dbuf,
	const char *old_path,
	const char *new_path)
{
	DIR_ENTRY *old_ent,*new_ent;
	DIR_PARAM old_dparam,new_dparam;
	size_t ioSize;
	int rest;

	/* oldΥȥ꡼õ */
	old_dparam.dirBuf=old_dbuf;
	old_dparam.path=old_param->path;
	if((rest=readDir(din,old_param->inodeData,&old_dparam,readDirBlock))<0)return rest;
	old_ent=(DIR_ENTRY*)rest;

	/* newΥȥ꡼õ */
	new_dparam.dirBuf=new_dbuf;
	new_dparam.path=new_param->path;
	rest=readDir(din,new_param->inodeData,&new_dparam,readDirBlock);

	if(rest>0)				/* newΥȥ꡼¸ߤ롣 */
	{
		new_ent=(DIR_ENTRY*)rest;
		if (old_ent->file_type != new_ent->file_type)
			return (old_ent->file_type == EXT2_DIR_REG_FILE)? -EISDIR : -ENOTDIR;

		/* pathͭγǧ */
		if (old_ent->file_type == EXT2_DIR_DIR)
			if(old_path[cmpStrNum(old_path,new_path)]=='\0')
				return -EINVAL;

		/* newΥǡinode롣 */
		if ((rest = readInode(din,new_ent->inode,new_param->inodeBuf)) < 0)
			return rest;
		new_param->inodeData = (INODE*)rest;
		if (new_ent->file_type == EXT2_DIR_DIR)
			if ((rest = checkDir(din,new_param->inodeData)) != -ENOENT)
				return rest;
//		new_param->inodeData->i_number = new_ent->inode;
		rest = unlinkInode(din,new_param->inodeData);
		if (rest < 0)
			return rest;
		if (0 < rest)
			return 

		/* ȥ꡼inodeʥС촹롣 */
		new_ent->inode=old_ent->inode;
		rest = write_cache(din, new_dparam.dirBuf, superBlock[din]->sectors, blkToSect(din, new_dparam.blkNumber), &ioSize);
		if (rest != NOERR){
			return rest;
		}
		ASSERT(ioSize == superBlock[din]->sectors);
	}
	else if(rest == -ENOENT)	/* newΥȥ꡼¸ߤʤ */
	{
		/* ե̾γǧ */
		if (EXT2_NAME_LEN < strlen(new_param->path)){
			return -ENAMETOOLONG;
		}
		/* newȥ꡼ */
		if ((rest = addDirEnt(din, old_ent->file_type, old_ent->inode, new_param, new_dbuf)) < 0){
			return rest;
		}
	}
	else{
		return rest;
	}

	/* oldȥ꡼κ */
	if ((rest = readDir(din, old_param->inodeData, &old_dparam, readDirBlock)) < 0){
		return rest;	/* ɤ߹ߡ */
	}
	delEntry(old_ent, old_dparam.dirBuf);
	rest = write_cache(din, old_dparam.dirBuf, superBlock[din]->sectors, blkToSect(din, old_dparam.blkNumber), &ioSize);
	if (rest != NOERR){
		return rest;
	}
	ASSERT(ioSize == superBlock[din]->sectors);

	return 0;
}


// ̾μΥȥ꡼̾ơΥȥ꡼ΥեåȤ֤
// Υȥ꡼̵0֤
// parameters : device inode,directory block,entry index,name
// return : next index or 0 or error number
STATIC int getNextEntry(int din, uint block, int index, char *o_name)
{
	int blkSize = superBlock[din]->blockSize;
	DIR_ENTRY *buf, *entry;
	size_t ioSize;
	int rest;

	if (index == blkSize){
		return 0;
	}

	buf = kmalloc(blkSize);
	if (buf == NULL){
		return -ENOMEM;
	}
	rest = read_cache(din, buf, superBlock[din]->sectors, blkToSect(din, block), &ioSize);
	if (rest != NOERR){
		goto END;
	}
	ASSERT(ioSize == superBlock[din]->sectors);

	entry = (DIR_ENTRY*)((char*)buf + index);
	if (entry->inode == 0){
		entry=(DIR_ENTRY*)((char*)entry+entry->rec_len);
	}

	if ((char*)entry == (char*)buf + blkSize){
		rest = 0;
	}
	else{
		memcpy(o_name, entry->name, entry->name_len);
		o_name[entry->name_len] = '\0';
		rest = (uint)entry - (uint)buf + entry->rec_len;
	}
END:
	kfree(buf);

	return rest;
}


/*
 * Υǥ쥯ȥ֥å롣Υ֥åʤ-ENOENT֤
 * parameters : device inode,inode,directory block
 * retur : next block or error number
 */
STATIC int INLINE getNextBlock(int din,INODE *inode,uint block)
{
	DIR_PARAM dparam;


	dparam.blkNumber=block;
	return readDir(din,inode,&dparam,searchDirBlock);
}


/*
 * ǥ쥯ȥΥǥ쥯ȥ֥åν
 * parameters : device inode,block number,inode number,parent inode number
 * return : 0 or error number
 */
STATIC int initDirBlock(int din,uint block,uint inumber,uint parent_inumber)
{
	DIR_ENTRY *dent,*dent2;
	size_t ioSize;
	int error=0;
	int rest;


	if((dent=(DIR_ENTRY*)kmalloc(superBlock[din]->sectorSize))==NULL)return -ENOMEM;
	memset(dent,0,superBlock[din]->sectorSize);			/* ꥢʤȥǥ쥯ȥǷٹθˤʤ롣 */

	/* "."ǥ쥯ȥɲá */
	dent->inode=inumber;
	dent->rec_len=sizeof(DIR_ENTRY)+sizeof(int);
	dent->name_len=1;
	dent->file_type = EXT2_DIR_DIR;
	dent->name[0]='.';

	dent2=(DIR_ENTRY*)((char*)dent+dent->rec_len);

	/* ".."ǥ쥯ȥɲá */
	dent2->inode=parent_inumber;
	dent2->rec_len=superBlock[din]->blockSize-dent->rec_len;
	dent2->name_len=2;
	dent2->file_type = EXT2_DIR_DIR;
	dent2->name[0]='.';
	dent2->name[1]='.';

	rest = write_cache(din, dent, 1, blkToSect(din, block), &ioSize);
	if (rest != NOERR){
		error = rest;
	}
	ASSERT(ioSize == 1);

	kfree(dent);

	return error;
}


/***********************************************************************
 *
 * System call
 *
 ***********************************************************************/


/*
 * ե
 * return : error number
 */
STATIC int creatFile(const char *path, int din, uint inumber, ushort mode, uint *o_number)
{
	uint parentInumber;
	uint block = 0;
	DIR_ENTRY *dirBuf;
	INODE *inode;
	INODE_PARAM param;
	PROC *proc;
	int rest,err_rest;

	/* ѥθ */
	if ((param.inodeBuf = kmalloc(superBlock[din]->sectorSize)) == NULL){
		return -ENOMEM;
	}
	if ((dirBuf = kmalloc(superBlock[din]->blockSize)) == NULL){
		kfree(param.inodeBuf);
		return -ENOMEM;
	}
	param.path = path;
	if ((rest = readInode(din,inumber,param.inodeBuf)) < 0){
		goto ERR1;
	}
	param.inodeData = (INODE*)rest;
	if ((rest = searchParentDir(din,&param,dirBuf)) < 0){
		goto ERR1;
	}
	parentInumber = param.inodeData->i_number;

	/* inode롣 */
	if ((inumber = getNewInode(din, mode)) == 0){
		rest = -ENOSPC;
		goto ERR1;
	}

	/* ǥ쥯ȥꥨȥ꡼ƥǥ쥯ȥɲä롣 */
	if ((rest = addDirEnt(din, getDirFileTypeFromFileType(mode), inumber, &param, dirBuf)) < 0){
		goto ERR2;
	}

	if ((mode & EXT2_S_IFMT) == EXT2_S_IFDIR){
		/* ǥ쥯ȥ롣 */
		block = getBlock(din);
		if (block == 0){
			rest = -ENOSPC;
			goto ERR3;
		}
		rest = initDirBlock(din,block,inumber,param.inodeData->i_number);
		if (rest < 0){
			goto ERR4;
		}
	}

	/* inode롣 */
	if ((rest = readInode(din,inumber,param.inodeBuf)) < 0){
		goto ERR4;
	}
	inode = (INODE*)rest;
	memset(inode,0,sizeof(INODE));
	proc = get_current_task();
	inode->i_mode = mode;
	inode->i_uid = proc->uid;
	inode->i_gid = proc->gid;
	inode->i_atime = inode->i_ctime = inode->i_mtime = (uint)sys_time(NULL);
	inode->i_number = inumber;
	if ((mode & EXT2_S_IFMT) == EXT2_S_IFDIR){
		inode->i_links_count = 2;
		inode->i_size = superBlock[din]->blockSize;
		inode->i_blocks = inode->i_size/INODE_BLOCK_SIZE;
		inode->i_block[0] = block;
	}
	else{
		inode->i_links_count = 1;
	}
	if ((rest = writeInode(din,inumber,param.inodeBuf)) < 0){
		goto ERR4;
	}

	kfree(dirBuf);
	kfree(param.inodeBuf);
	
	*o_number = inumber;

	return NOERR;

ERR4:
	if ((mode & EXT2_S_IFMT) == EXT2_S_IFDIR)
		releaseBlock(din,block);
ERR3:
	if ((mode & EXT2_S_IFMT) == EXT2_S_IFDIR)
		if((err_rest = readInode(din,parentInumber,param.inodeBuf)) > 0)
		{
			((INODE*)err_rest)->i_links_count -= 1;
			writeInode(din,parentInumber,param.inodeBuf);
		}
ERR2:
	releaseInodeBitAndGrp(din,inumber,EXT2_S_IFLNK);
ERR1:
	kfree(dirBuf);
	kfree(param.inodeBuf);

	return rest;
}


STATIC int mount(const int din, void **o_rootInodeObj)
{
	void *buf;
	DEV_STAT devStat;
	INODE *inode, *inodeBuf;
	size_t ioSize;
	int sectors;
	int gd;
	int rest;

	// Ǥ˥ޥȤƤ뤫ǧ
	if (superBlock[din] != NULL){
		return -EBUSY;
	}

	// ѡ֥åɤ߹ࡣ
	rest = get_devstat(din, &devStat);
	if (rest < 0){
		return rest;
	}
	buf = kmalloc(PAGE_SIZE);
	if (buf == NULL){
		return -ENOMEM;
	}
	rest = read_direct(din, buf, PAGE_SIZE / devStat.sect_size, 0, &ioSize);
	if (rest != NOERR){
		goto ERR;
	}
	superBlock[din] = kmalloc(sizeof(SUPER_BLOCK));
	if (superBlock[din] == NULL){
		rest = -ENOMEM;
		goto ERR;
	}
	memcpy(superBlock[din], (char*)buf + EXT2_SUPER_BLK_POSITION, sizeof(SUPER_BLOCK));

	/* ޥåʥСγǧ */
	if (superBlock[din]->s_magic != EXT2_SUPER_MAGIC){
		rest = -ENOTBLK;
		goto ERR;
	}

	/* ȼѡ֥å */
	superBlock[din]->blockSize = EXT2_BLOCK_BASE << superBlock[din]->s_log_block_size;
	superBlock[din]->sectorSize = devStat.sect_size;
	superBlock[din]->sectors = superBlock[din]->blockSize / superBlock[din]->sectorSize;

	sectors = superBlock[din]->sectors;

	// 롼ץǥץɤ߹ࡣ
	grpDescNum[din] = ROUNDUP_DIV(superBlock[din]->s_blocks_count, superBlock[din]->s_blocks_per_group);
	grpDescBlocks[din] = ROUNDUP_DIV(grpDescNum[din] * sizeof(GROUP_DESC), superBlock[din]->blockSize);
	grpDesc[din] = kmalloc(superBlock[din]->blockSize * grpDescBlocks[din]);
	if (grpDesc[din] == NULL){
		rest = -ENOMEM;
		goto ERR2;
	}
	rest = read_direct(din,grpDesc[din],sectors * grpDescBlocks[din],
		blkToSect(din,EXT2_GROUP_DESC_BLK + superBlock[din]->s_first_data_block), &ioSize);
	if (rest != NOERR){
		goto ERR3;
	}

	/* ֥åطν */
	rest = initBlock(din);
	if (rest < 0){
		goto ERR3;
	}
	gd = rest;

	/* inodeطν */
	rest = initInode(din, gd);
	if (rest < 0){
		goto ERR3;
	}

	/* root inodeɤ߽Ф */
	rest = readInode(din, EXT2_ROOT_INODE_INDEX, buf);
	if (rest < 0){
		goto ERR3;
	}
	inode = (INODE*)rest;
	inodeBuf = kmalloc(sizeof(INODE));
	if (rest < 0){
		rest = -ENOMEM;
		goto ERR3;
	}
	memcpy(inodeBuf, inode, sizeof(INODE));
	*o_rootInodeObj = inodeBuf;
	
	kfree(buf);

	return 0;

ERR3:
	kfree(grpDesc[din]);
	grpDesc[din] = NULL;
ERR2:
	kfree(superBlock[din]);
	superBlock[din] = NULL;
ERR:
	kfree(buf);

	return rest;
}


STATIC int umount(const int din, void *rootInodeObj)
{
	umountInode(din);
	umountBlock(din);
	writeBackSbGd(din);

	kfree(rootInodeObj);

	kfree(grpDesc[din]);
	grpDesc[din] = NULL;

	kfree(superBlock[din]);
	superBlock[din] = NULL;

	return NOERR;
}


STATIC int lookup(const char *path, const int din, void *inodeObj, uint *o_inodeNum, const char **o_path)
{
	int rest;
	INODE *dirInode = inodeObj;
	DIR_ENTRY *dirBuf;
	INODE_PARAM param;

	ASSERT(path != NULL);

	if ((param.inodeBuf = kmalloc(superBlock[din]->sectorSize)) == NULL){
		return -ENOMEM;
	}
	if ((dirBuf = kmalloc(superBlock[din]->blockSize)) == NULL){
		kfree(param.inodeBuf);
		return -ENOMEM;
	}
	param.path = path;
	if ((rest = readInode(din, dirInode->i_number, param.inodeBuf)) < 0){
		goto END;
	}
	param.inodeData = (INODE*)rest;
	if ((rest = searchParentDir(din, &param, dirBuf)) < 0){
		goto END;
	}
	rest = getInodeNumber(din, &param, dirBuf, o_inodeNum);
	if (rest < 0){
		goto END;
	}

	rest = NOERR;
END:
	kfree(dirBuf);
	kfree(param.inodeBuf);

	return rest;
}


STATIC int creat(const char *path, const int din, void *inodeObj, const int mode, uint *o_inodeNum)
{
	INODE *dirInode = inodeObj;

	ASSERT(path != NULL);

	// ե
	return creatFile(path, din, dirInode->i_number, mode, o_inodeNum);
}


STATIC int open(const int din, const uint inodeNum, int oflag, void **o_inodeObj, int *o_type)
{
	INODE *inodeData;
	INODE_PARAM inodeParam;
	int rest;

	inodeParam.inodeBuf = kmalloc(superBlock[din]->sectorSize);
	if (inodeParam.inodeBuf == NULL){
		return -ENOMEM;
	}
	rest = getInodeData(din, inodeNum, &inodeParam);
	if (rest < 0){
		goto END;
	}

	inodeData = kmalloc(sizeof(INODE));
	if (inodeData == NULL){
		rest = -ENOMEM;
		goto END;
	}
	memcpy(inodeData, inodeParam.inodeData, sizeof(INODE));

	*o_inodeObj = inodeData;
	*o_type = inodeData->i_mode & EXT2_S_IFMT;
	
	rest = NOERR;
END:
	kfree(inodeParam.inodeBuf);

	return rest;
}


STATIC int close(const int din, void *inodeObj)
{
	INODE *sectBuf;
	INODE *diskInode;
	INODE *inode = inodeObj;
	int rest;

	// inode֥Ȥ񤭤Ƴ
	sectBuf = kmalloc(superBlock[din]->sectorSize);
	if (sectBuf == NULL){
		return -ENOMEM;
	}
	rest = readInode(din, inode->i_number, sectBuf);
	if (rest < 0){
		goto END;
	}
	diskInode = (INODE*)rest;
	if (diskInode->i_dtime == 0){
		memcpy(diskInode, inode, sizeof(INODE));
		rest = writeInode(din, inode->i_number, sectBuf);
		if (rest < 0){
			goto END;
		}
	}

	rest = NOERR;
END:
	kfree(sectBuf);
	kfree(inode);

	return rest;
}


STATIC int read(const int din, void *inodeObj, void *buf, size_t size, size_t begin)
{
	INODE *inode = inodeObj;
	int rest;

	/* γǧ */
	if (inode->i_size <= begin){
		return 0;
	}
	if (inode->i_size <= begin + size){
		size = inode->i_size - begin;
	}

	/* ե뤫ɤ߹ࡣ */
	rest = readData(din, inode, buf, size, begin);
	if (rest < 0){
		return rest;
	}

	/* դι */
	inode->i_atime = (uint)sys_time(NULL);

	return size;
}


STATIC int write(const int din, void *inodeObj, void *buf, size_t size, size_t begin)
{
	INODE *inode = inodeObj;
	int rest;

	/* γǧ */
	if (inode->i_size < begin + size){
		if(superBlock[din]->s_free_blocks_count <= (begin + size - inode->i_size) / superBlock[din]->blockSize){
			return -ENOSPC;
		}
	}

	/* եؽ񤭹ߡ */
	if ((rest = writeData(din, inode, buf, size, begin)) < 0){
		return rest;
	}

	/* ι */
	if (inode->i_size < begin + size){
		int ext2BlkSize = superBlock[din]->blockSize;
		int inodeNum = ext2BlkSize / sizeof(uint32_t);
		int blkNum = ROUNDUP_DIV(begin + size, ext2BlkSize);

		inode->i_size = begin + size;

		if (INODE_DATA_INDIRECT1 < blkNum){
			if (INODE_DATA_INDIRECT1 + inodeNum < blkNum){
				if (INODE_DATA_INDIRECT1 + (inodeNum + 1) * inodeNum < blkNum){
					blkNum += ROUNDUP_DIV(blkNum - INODE_DATA_INDIRECT1 - inodeNum, inodeNum * inodeNum);
				}
				blkNum += ROUNDUP_DIV(blkNum - INODE_DATA_INDIRECT1, inodeNum);
			}
			blkNum += 1;
		}
		inode->i_blocks = blkNum * (ext2BlkSize / INODE_BLOCK_SIZE);
	}

	/* դι */
	inode->i_atime = inode->i_mtime = sys_time(NULL);

	return size;
}


//աNULLѥԲġ
STATIC int mkdir(const char *path, const int din, void *dirInodeObj, int mode)
{
	INODE *dirInode = dirInodeObj;
	uint inumber;

	return creatFile(path, din, dirInode->i_number, mode | S_IFDIR, &inumber);
}


//աNULLѥԲġ
STATIC int delete(const char *path, const int din, void *dirInodeObj, int type)
{
	INODE *dirInode = dirInodeObj;
	DIR_ENTRY *dirBuf;
	INODE_PARAM param;
	int rest;

	/* ƥǥ쥯ȥθ */
	param.inodeBuf = kmalloc(superBlock[din]->sectorSize);
	if (param.inodeBuf == NULL){
		return -ENOMEM;
	}
	dirBuf = kmalloc(superBlock[din]->blockSize);
	if (dirBuf == NULL){
		kfree(param.inodeBuf);
		return -ENOMEM;
	}
	param.path = path;
	rest = readInode(din, dirInode->i_number, param.inodeBuf);
	if (rest < 0){
		goto END;
	}
	param.inodeData = (INODE*)rest;
	rest = searchParentDir(din,&param, dirBuf);
	if (rest < 0){
		goto END;
	}

	/* ƥǥ쥯ȥ꤫ǥ쥯ȥꥨȥ꡼롣 */
	else if (type & S_IFREG) {
		rest = delFileEntry(din, &param, dirBuf);
	}
	else if (type & S_IFLNK) {
		rest = delFileEntry(din, &param, dirBuf);
	}
	else if (type & S_IFDIR) {
		rest = delDirEntry(din, &param, dirBuf);
	}
	else {
		rest = -EOPNOTSUPP;
	}

	/* inode */
	if (rest == NOERR){
		rest = unlinkInode(din, param.inodeData);
	}
END:
	kfree(dirBuf);
	kfree(param.inodeBuf);

	return rest;
}


STATIC int opendir(const char *path, const int din, void *dirInodeObj, void **o_inodeObj, uint *o_block, int *o_index)
{
	INODE *dirInode = dirInodeObj;
	INODE *inodeBuf;
	DIR_ENTRY *dirBuf;
	INODE_PARAM param;
	int rest;

	/* ƥХåեΥ */
	param.inodeBuf = kmalloc(superBlock[din]->sectorSize);
	if (param.inodeBuf == NULL){
		return -ENOMEM;
	}
	dirBuf = kmalloc(superBlock[din]->blockSize);
	if (dirBuf == NULL){
		rest = -ENOMEM;
		goto END;
	}

	/* inode򸡺 */
	rest = readInode(din, dirInode->i_number, param.inodeBuf);
	if (rest < 0){
		goto END2;
	}
	if (*path == '\0')
	{
		*o_block = ((INODE*)rest)->i_block[0];
		*o_inodeObj = dirInode;
	}
	else
	{
		param.path = path;
		param.inodeData = (INODE*)rest;
		rest = searchParentDir(din,&param, dirBuf);
		if (rest < 0){
			goto END2;
		}
		rest = getInode(din,&param,dirBuf);
		if (rest < 0){
			goto END2;
		}
		if ((param.inodeData->i_mode & EXT2_S_IFMT) != EXT2_S_IFDIR){
			rest = -ENOTDIR;
			goto END2;
		}
		
		/* inode */
		inodeBuf = kmalloc(sizeof(INODE));
		if (inodeBuf == NULL){
			rest = -ENOMEM;
			goto END2;
		}
		memcpy(inodeBuf, param.inodeData, sizeof(INODE));

		*o_block = param.inodeData->i_block[0];
		*o_inodeObj = inodeBuf;
	}
	*o_index = 0;
	rest = NOERR;
END2:
	kfree(dirBuf);
END:
	kfree(param.inodeBuf);

	return rest;
}


STATIC int readdir(int din, void *inodeObj, uint *m_block, int *m_index, char *o_name)
{
	INODE *dirInode = inodeObj;
	INODE *inode;
	INODE_PARAM param;
	int rest;

	for(;;){
		rest = getNextEntry(din, *m_block, *m_index, o_name);
		if(0 < rest){
			*m_index = rest;
			rest = 0;
			break;
		}
		else if(rest < 0){
			break;
		}

		if((param.inodeBuf = kmalloc(superBlock[din]->sectorSize)) == NULL){
			return -ENOMEM;
		}
		if((rest = readInode(din, dirInode->i_number, param.inodeBuf)) < 0){
			kfree(param.inodeBuf);
			break;
		}
		inode = (INODE*)rest;
		rest = getNextBlock(din, inode, *m_block);
		kfree(param.inodeBuf);
		if(rest == -ENOENT){		/* Υ֥å̵ */
			*o_name = '\0';
			rest = 0;
			break;
		}
		else if(rest < 0){
			break;
		}

		*m_block = rest;
		*m_index = 0;
	}

	return rest;
}


STATIC int stat(int din, void *inodeObj, struct stat *m_stat)
{
	INODE *inode = inodeObj;

	m_stat->st_dev		= din;
	m_stat->st_mode		= inode->i_mode;
	m_stat->st_nlink	= inode->i_links_count;
	m_stat->st_uid		= inode->i_uid;
	m_stat->st_gid		= inode->i_gid;
	m_stat->st_size		= inode->i_size;
	m_stat->st_atime	= inode->i_atime;
	m_stat->st_mtime	= inode->i_mtime;
	m_stat->st_ctime	= inode->i_ctime;
	m_stat->st_blksize	= EXT2_BLOCK_BASE << superBlock[din]->s_log_block_size;
	m_stat->st_ino		= inode->i_number;										/* Ȥꤢinode롣 */
	m_stat->st_blocks	= ROUNDUP(m_stat->st_size,m_stat->st_blksize) / 512;	/* ѤƤ֥å(512Хñ) */
	m_stat->st_rdev		= 0;

	return 0;
}


STATIC int rename(int din, void *oldDirInodeObj, const char *oldPath, void *newDirInodeObj, const char *newPath)
{
	INODE *oldDirInode = oldDirInodeObj;
	INODE *newDirInode = newDirInodeObj;
	DIR_ENTRY *dirBuf1, *dirBuf2;
	INODE_PARAM old_param, new_param;
	int rest;

	new_param.inodeBuf = NULL;
	dirBuf2 = NULL;

	/* old pathõ */
	dirBuf1 = kmalloc(superBlock[din]->blockSize);
	if (dirBuf1 == NULL){
		return -ENOMEM;
	}
	old_param.inodeBuf = kmalloc(superBlock[din]->sectorSize);
	if (old_param.inodeBuf == NULL){
		rest = -ENOMEM;
		goto END;
	}
	old_param.path = oldPath;
	rest = readInode(din, oldDirInode->i_number, old_param.inodeBuf);
	if (rest < 0){
		goto END2;
	}
	old_param.inodeData = (INODE*)rest;
	rest=searchParentDir(din, &old_param, dirBuf1);
	if (rest < 0){
		goto END2;
	}

	/* new pathõ */
	new_param.inodeBuf = kmalloc(superBlock[din]->sectorSize);
	if (new_param.inodeBuf == NULL){
		rest = -ENOMEM;
		goto END2;
	}
	new_param.path = newPath;
	rest = readInode(din,newDirInode->i_number, new_param.inodeBuf);
	if (rest < 0){
		goto END3;
	}
	new_param.inodeData = (INODE*)rest;
	rest = searchParentDir(din,&new_param, dirBuf1);
	if (rest < 0){
		goto END3;
	}

	/* old inodenew inode촹롣 */
	dirBuf2 = kmalloc(superBlock[din]->blockSize);
	if (dirBuf2 == NULL){
		rest = -ENOMEM;
		goto END3;
	}
	rest = changeInode(din, &old_param, &new_param, dirBuf1, dirBuf2, oldPath, newPath);
	if (rest < 0){
		goto END4;
	}

	rest = NOERR;
END4:
	kfree(dirBuf2);
END3:
	kfree(new_param.inodeBuf);
END2:
	kfree(old_param.inodeBuf);
END:
	kfree(dirBuf1);

	return rest;
}


STATIC int chattr(int din, void *inodeObj, uint mode, uint uid, uint gid, uint atime, uint mtime)
{
	INODE *inode = inodeObj;

	inode->i_mode 	= (inode->i_mode & ~0x1ff) | (mode & 0x1ff);
	inode->i_uid 	= uid;
	inode->i_gid 	= gid;
	inode->i_atime 	= atime;
	inode->i_mtime 	= mtime;

	return NOERR;
}


STATIC int link(const int din, void *dstInodeObj, void *srcInodeObj, const char *srcPath)
{
	INODE *dstInode = dstInodeObj;
	INODE *srcInode = srcInodeObj;
	DIR_ENTRY *dirBuf;
	INODE_PARAM param;
	int rest;

	if ((param.inodeBuf = kmalloc(superBlock[din]->sectorSize)) == NULL){
		return -ENOMEM;
	}
	if ((dirBuf = kmalloc(superBlock[din]->blockSize)) == NULL){
		kfree(param.inodeBuf);
		return -ENOMEM;
	}

	/* Υǥ쥯ȥinodeΰõ */
	param.path = srcPath;
	if ((rest = readInode(din,srcInode->i_number,param.inodeBuf)) < 0){
		goto END;
	}
	param.inodeData = (INODE*)rest;
	if ((rest = searchParentDir(din,&param,dirBuf)) < 0){
		goto END;
	}

	/* Υե̾Υå */
	if (strlen(param.path) > EXT2_NAME_LEN){
		rest = -ENAMETOOLONG;
		goto END;
	}

	/* ǥ쥯ȥꥨȥ꡼ɲä롣 */
	if ((rest = addDirEnt(din, getDirFileTypeFromFileType(EXT2_S_IFREG), dstInode->i_number, &param, dirBuf)) < 0){
		goto END;
	}

	/*
	 * inodeϽᤵ֤
	 */
//	if ((rest = readInode(din,dstInode->i_number,param.inodeBuf)) < 0)
//		goto END;
	/* 󥯤䤹 */
	dstInode->i_links_count += 1;
//	memcpy((void*)rest,(void*)dstInode,sizeof(INODE));
//	if ((rest = writeInode(din,dstInode->i_number,param.inodeBuf)) < 0)
//		goto END;
	
	rest = NOERR;
END:
	kfree(dirBuf);
	kfree(param.inodeBuf);

	return rest;
}


STATIC int symlink(const char *filePath, const int din, void *dirInodeObj, char *linkPath)
{
	uint inumber;
	INODE *inode;
	INODE *sectBuf;
	INODE *dirInode = dirInodeObj;
	int rest;

	rest = creatFile(filePath, din, dirInode->i_number, S_IFLNK | S_IRWXU | S_IRWXG | S_IRWXO, &inumber);
	if (rest < 0){
		return rest;
	}

	/* inodeɤ߽Ф */
	sectBuf = kmalloc(superBlock[din]->sectorSize);
	if (sectBuf == NULL){
		return -ENOMEM;
	}
	rest = readInode(din, inumber, sectBuf);
	if (rest < 0){
		goto END;
	}
	inode = (INODE*)rest;

	/* 󥯥ѥ񤭹 */
	rest = write(din, inode, linkPath, strlen(linkPath), 0);
	if (rest < 0){
		goto END;
	}
	
	/* inode᤹ */
	rest = writeInode(din, inumber, sectBuf);
	if (rest < 0){
		goto END;
	}

	rest = NOERR;
END:
	kfree(sectBuf);

	return rest;
}


STATIC int ioctl(int din, void *inodeObj, int cmd, caddr_t param, int fflag)
{
	int rest;

	switch(cmd){
		case I_TRUNC:
			/* truncate. */
			rest = releaseDataBlock(din, inodeObj);
			if(rest < 0){
				return rest;
			}
	}

	return NOERR;
}


STATIC int poll(void *inodeObj, int events)
{
	return POLLIN | POLLRDNORM | POLLOUT | POLLWRNORM;
}


STATIC int statfs(const int din, struct statfs *m_statfs)
{
	SUPER_BLOCK *superBlk = superBlock[din];

	m_statfs->f_bsize		= superBlk->blockSize;				/* File system block size. */
	m_statfs->f_iosize		= superBlk->blockSize;				/* optimal transfer block size */
	m_statfs->f_blocks		= superBlk->s_blocks_count;			/* total data blocks in file system */
	m_statfs->f_bfree		= superBlk->s_free_blocks_count;	/* free blocks in fs */
	m_statfs->f_bavail		= superBlk->s_free_blocks_count;	/* free blocks avail to non-superuser */
	m_statfs->f_files		= superBlk->s_inodes_count;			/* total file nodes in file system */
	m_statfs->f_ffree		= superBlk->s_free_inodes_count;	/* free file nodes in fs */
	m_statfs->f_fsid.val[0] = 0;								/* file system id */
	m_statfs->f_fsid.val[1] = 0;
	m_statfs->f_owner		= 0;								/* user that mounted the filesystem */
	m_statfs->f_type		= MOUNT_EXT2FS;						/* type of filesystem */
	m_statfs->f_flags		= 0;								/* Bit mask of f_flag values */
	memcpy(m_statfs->f_mntfromname, "ext2fs", 7);					/* mounted filesystem */

	return NOERR;
}


/***********************************************************************
 *
 * Init
 *
 ***********************************************************************/

static FS ext2Fs={
	"ext2",
	mount,
	umount,
	lookup,
	creat,
	open,
	close,
	read,
	write,
	ioctl,
	poll,
	opendir,
	readdir,
	rename,
	mkdir,
	stat,
	delete,
	chattr,
	link,
	symlink,
	statfs,
};


/*
 * GLOBAL
 */
int initExt2tFs()
{
	int i;


	for(i=0;i<MAX_DEVICE_OPEN;++i)
	{
		grpDesc[i]=NULL;
		superBlock[i]=NULL;
	}

	return regist_fs(&ext2Fs);
}
/**************************************************************************************/
int test_ext2(int din)
{
	printk("test_ext2(() currentGd=0x%x\n", blockBm[din].currentGd);

	return 0;
}
/**************************************************************************************/

