/*
 * device.c
 *
 * Copyright 2002, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 */


#include <sys/config.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/conf.h>
#include <sys/systm.h>
#include <sys/disklabel.h>
#include <sys/diskslice.h>
#include <sys/bus.h>
#include <sys/buf.h>
#include <machine/except_i386.h>
#include <kern/lib.h>
#include <kern/lib_path.h>
#include <kern/mm.h>
#include <kern/interrupt.h>
#include <kern/proc.h>
#include <kern/fs.h>
#include <kern/lock.h>
#include <kern/device.h>
#include <kern/errno.h>
#include <kern/lock.h>
#include <kern/time.h>
#include <kern/signal.h>
#include <kern/test.h>
#include <kern/debug.h>


//#define DEBUG_DEVICE 1
#ifdef DEBUG_DEVICE
	#define STATIC
	#define INLINE
#else
	#define STATIC static
	#define INLINE inline
#endif


/******************************************************************************************************
 *
 * ǥХԤ
 *
 *******************************************************************************************************/


//================================== PUBLIC =============================================


//աɬȯθȤʤ˸ƤӽФ
void devInitWait(WAIT_INTR *wait)
{
	initAddScheduleFlag(&wait->isIntr);
	wait->proc = get_current_task();
}


void devWaitIntr(WAIT_INTR *wait, uint time)
{
	void *timer;
	int error = setDevWakeTimer(time, &timer);

	if (delFromScheduleRef(TASK_DEVICE_WAIT, &wait->isIntr) == YES){
		wait_task();
	}
	if (error == NOERR){
		delWakeTimer(timer);
	}
}


//Գ
// return : 1 = å or 0
int devWakeIntr(WAIT_INTR *wait)
{
	if (addToScheduleSet(wait->proc, TASK_DEVICE_WAIT, &wait->isIntr) == YES){
		return 1;
	}
	else{
		return 0;
	}
}


/***********************************************************************************
 *
 * ǥХե륷ƥ
 *
 * ¤
 *	inode¤Τǳ
 *		root inodeϣΩ
 *
 * ========== PRIVATE
 * void setRootInode()
 * void partNumToStr(int value, char *m_str)
 * int getMajorNumber()
 * dev_t allocSpecinfo()
 * void freeSpecinfo(dev_t dev)
 * int getInodeNum(DEV_INODE *devInode)
 * void linkInode(int inodeNum)
 * int getFreeInode()
 * int setInode(struct cdevsw *cdevsw, uint blockBeg, uint size, int partType, int major, int partNum, int fflag)
 * ========== PROTECTED
 * void unlinkInode(int inodeNum)
 * int lookupDevInode(const char *path)
 * int getNextInode(int inodeNum)
 * int getStat(int inodeNum, struct stat *m_stat)
 * dev_t getDevInfo(int inodeNum)
 * uint getBlocks(int inodeNum)
 * uint getBegBlock(int inodeNum)
 * uint getMode(int inodeNum)
 * const char *getName(int inodeNum)
 * void initDevFs()
 * ========== PUBLIC
 * dev_t make_dev(struct cdevsw *devsw, int minor, uid_t uid, gid_t gid, int perms, const char *fmt, ...)
 * int get_devstat(int inodeNum, DEV_STAT *m_devStat)
 * int delete_device(const char *name)
 * int registBlockPartition(const char *name)
 ***********************************************************************************/


enum{
	DEVICE_INODE_NUM	= MAX_DEVICE_OPEN,	// ǥХinode
	MAX_PART_NUM_DIGIT	= 4,				// ֥åѡƥֹ
};


/* ǥХinode */
typedef struct DEV_INODE{
	dev_t				dev;					// struct specinfo*
	ushort				partType;				// ѡƥ󥿥
	ushort				major;					// ᥸㡼ֹ
	ushort				partNum;				// ѡƥֹޥʡֹ
	short				mode;					/* ե⡼ */
	int 				refCount;				/* ȥ */
	size_t				blockBeg;				// ǥѡƥХ֥åƬ
	size_t				blocks;					// ǥѡƥХ֥å
	uint				creatTime;				/*  */
	char				name[MAX_DEVICE_NAME];	/* ǥХ̾ */
}DEV_INODE;


//================================== PRIVATE ============================================


/* ǥХե륷ƥinodeơ֥ */
static DEV_INODE devInodeTbl[DEVICE_INODE_NUM];	/* inodeơ֥ */

// dev_tŪΰ
struct devData{
	struct specinfo specinfo[DEVICE_INODE_NUM];
	char busy[DEVICE_INODE_NUM];		// ʤ飱
}devData;


// root inode
STATIC void setRootInode()
{
	/* inode */
	devInodeTbl[DEVICE_ROOT_INODE].partType = 0;
	devInodeTbl[DEVICE_ROOT_INODE].partNum = 0;
	devInodeTbl[DEVICE_ROOT_INODE].refCount = 2;
	devInodeTbl[DEVICE_ROOT_INODE].dev = NULL;
	devInodeTbl[DEVICE_ROOT_INODE].blockBeg = 0;
	devInodeTbl[DEVICE_ROOT_INODE].blocks = 0;
	devInodeTbl[DEVICE_ROOT_INODE].creatTime = sys_time(NULL);
	devInodeTbl[DEVICE_ROOT_INODE].mode = S_IFDIR | S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH;
}


// ᥸㡼ֹ롣
STATIC int getMajorNumber()
{
	static int majorNum = 0;

	return majorNum++;
}


// dev_tƤ
// return : dev_t ro NULL
STATIC dev_t allocSpecinfo()
{
	dev_t dev = NULL;
	int i;

	for (i = 0; i < DEVICE_INODE_NUM; ++i){
		if (devData.busy[i] == NO){
			devData.busy[i] = YES;
			dev = &devData.specinfo[i];
			break;
		}
	}

	return dev;
}


// dev_t
STATIC void freeSpecinfo(dev_t dev)
{
	int i;

	for (i = 0; i < DEVICE_INODE_NUM; ++i){
		if (dev == &devData.specinfo[i]){
			devData.busy[i] = NO;
			break;
		}
	}
}


// ǥХinodeinodeֹ
STATIC INLINE int getInodeNum(DEV_INODE *devInode)
{
	return ((uint)devInode - (uint)devInodeTbl) / sizeof(*devInode);
}


// dev_tinodeֹ򸡺롣
STATIC int searchInodeNum(dev_t dev)
{
	int i;

	for (i = 0; i < DEVICE_INODE_NUM; ++i){
		if (0 < devInodeTbl[i].refCount){
			if (devInodeTbl[i].dev == dev){
				return i;
			}
		}
	}

	return -ENOENT;
}


// inodeλȥ󥿤䤹
STATIC void linkInode(int inodeNum)
{
	ASSERT(0 <= devInodeTbl[inodeNum].refCount);
	devInodeTbl[inodeNum].refCount += 1;
}


// inode
// return : inode number or error number
STATIC int getFreeInode()
{
	int i;

	for (i = 0; i < DEVICE_INODE_NUM; ++i){
		if (devInodeTbl[i].refCount == 0){
			return i;
		}
	}

	return -EMFILE;
}


static int lookupDevInode(const char *);

// ǥХinodeꤹ롣
// return : device inode or error number
STATIC int setInode(
	struct cdevsw *cdevsw,
	size_t blockBeg,
	size_t size,
	int partType,
	int major,
	int partNum,
	int fflag)
{
	char numStr[MAX_PART_NUM_DIGIT + 1] = {'\0'};
	char name[MAX_DEVICE_NAME + MAX_PART_NUM_DIGIT];
	int inodeNum;

	/* ѡƥֹ礷ǥХ͡ʸγǧ */
	if (0 < partNum){
		iToDecimal(partNum, numStr);
	}
	strcpy(name, cdevsw->d_name);
	strcat(name, numStr);
	if (MAX_DEVICE_NAME <= strlen(name)){
		return -ENAMETOOLONG;
	}

	/* ǤϿ뤫 */
	inodeNum = lookupDevInode(name);
	if (inodeNum < 0){
		// Ͽʤ
		/* inode */
		inodeNum = getFreeInode();
		if (inodeNum < 0){
			return inodeNum;
		}

		/* inode */
		devInodeTbl[inodeNum].dev = allocSpecinfo();
		if (devInodeTbl[inodeNum].dev == NULL){
			return -ENFILE;
		}
		devInodeTbl[inodeNum].partType = partType;
		devInodeTbl[inodeNum].major = major;
		devInodeTbl[inodeNum].partNum = partNum;
		devInodeTbl[inodeNum].blockBeg = blockBeg;
		devInodeTbl[inodeNum].blocks = size;
		devInodeTbl[inodeNum].creatTime = sys_time(NULL);
		if (cdevsw->d_flags & (D_TAPE | D_TTY | D_MEM)){
			// 饯ǥХ
			devInodeTbl[inodeNum].mode = S_IFCHR | fflag;
		}
		else if (cdevsw->d_flags & D_DISK){
			// ֥åǥХ
			devInodeTbl[inodeNum].mode = S_IFBLK | fflag;
		}
		else{
			// Ȥꤢ饯ǥХˤƤ
			devInodeTbl[inodeNum].mode = S_IFCHR | fflag;
		}
		/* ǥХ̾ */
		strcpy(devInodeTbl[inodeNum].name, name);
		linkInode(inodeNum);

		// dev
		memset(devInodeTbl[inodeNum].dev, 0, sizeof(*devInodeTbl[inodeNum].dev));
		devInodeTbl[inodeNum].dev->si_udev = makeudev(major, partNum);
		devInodeTbl[inodeNum].dev->si_devsw = cdevsw;
	}
	else{
		return -EEXIST;
	}

	return inodeNum;
}


// cdevsw¤ΤNULLؿѴ롣
STATIC void compile_devsw(struct cdevsw *devsw)
{
	if (devsw->d_open == NULL)
		devsw->d_open = noopen;
	if (devsw->d_close == NULL)
		devsw->d_close = noclose;
	if (devsw->d_read == NULL)
		devsw->d_read = noread;
	if (devsw->d_write == NULL)
		devsw->d_write = nowrite;
	if (devsw->d_ioctl == NULL)
		devsw->d_ioctl = noioctl;
	if (devsw->d_poll == NULL)
		devsw->d_poll = nopoll;
	if (devsw->d_mmap == NULL)
		devsw->d_mmap = nommap;
	if (devsw->d_strategy == NULL)
		devsw->d_strategy = nostrategy;
	if (devsw->d_dump == NULL)
		devsw->d_dump = nodump;
	if (devsw->d_psize == NULL)
		devsw->d_psize = nopsize;
	if (devsw->d_kqfilter == NULL)
		devsw->d_kqfilter = nokqfilter;
}


//================================== PROTECTED ==========================================


// inodeλȥ󥿤򣱸餹
// return : ȿ
STATIC void unlinkInode(int inodeNum)
{
	ASSERT(0 <= devInodeTbl[inodeNum].refCount);
	// ȥȤ򸺤餹
	devInodeTbl[inodeNum].refCount -= 1;
	if (devInodeTbl[inodeNum].refCount == 0){
		// dev_t
		freeSpecinfo(devInodeTbl[inodeNum].dev);
	}
}


// ǥХθ
// return : inode number or error number
STATIC int lookupDevInode(const char *path)
{
	const char *devName;
	int i;

	// ѥθ
	// out : devName = "path"ФǥХ̾
	for (devName = path;;){
		/* "."".."ν */
		if (cmpPath(".",devName,1) == 0){
			if (path[1] == '/'){
				devName = &path[2];
			}
			else{
				devName = &path[1];
			}
		}
		/* ".."̤б */
		else if (cmpPath("..",devName,2) == 0){
			return -ENOENT;
		}
		/* root inode? */
		else if (cmpPath("\0",devName,0) == 0){
			if (path[1] == '/'){
				devName = &path[1];
			}
			else{
				return DEVICE_ROOT_INODE;
			}
		}
		else{
			break;
		}
	}

	// ǥХinodeơ֥򸡺
	for (i = 0; i < DEVICE_INODE_NUM; ++i){
		if (strcmp(devInodeTbl[i].name, devName) == 0){
			if (0 < devInodeTbl[i].refCount){
				return i;
			}
		}
	}

	return -ENOENT;
}


// Ϳ줿inodeμΥ󥯤inode֤
// Ϳ줿inode-1ʤȥåץ󥯤inode֤
// return : next device inode or ERR
STATIC int getNextInode(int inodeNum)
{
	int i;

	if (inodeNum == -1){
		i = DEVICE_ROOT_INODE;
	}
	else{
		i = inodeNum;
	}
	for (++i; i < DEVICE_INODE_NUM; ++i){
		if (0 < devInodeTbl[i].refCount){
			return i;
		}
	}

	return ERR;
}


// dev_t֤
STATIC INLINE dev_t getDevInfo(int inodeNum)
{
	ASSERT((0 <= inodeNum) && (inodeNum < DEVICE_INODE_NUM));
	return devInodeTbl[inodeNum].dev;
}


// ֥å֤
STATIC INLINE uint getBlocks(int inodeNum)
{
	ASSERT((0 <= inodeNum) && (inodeNum < DEVICE_INODE_NUM));
	return devInodeTbl[inodeNum].blocks;
}


// ֥åλϤޤ֤
STATIC INLINE size_t getBegBlock(int inodeNum)
{
	ASSERT((0 <= inodeNum) && (inodeNum < DEVICE_INODE_NUM));
	return devInodeTbl[inodeNum].blockBeg;
}


// ե⡼ɤ֤
STATIC INLINE uint getMode(int inodeNum)
{
	ASSERT((0 <= inodeNum) && (inodeNum < DEVICE_INODE_NUM));
	return devInodeTbl[inodeNum].mode;
}


// ǥХ̾
STATIC const char *getName(int inodeNum)
{
	ASSERT((0 <= inodeNum) && (inodeNum < DEVICE_INODE_NUM));
	return devInodeTbl[inodeNum].name;
}


// statꤹ
STATIC int getStat(int inodeNum, struct stat *m_stat)
{
	ASSERT((0 <= inodeNum) && (inodeNum < DEVICE_INODE_NUM));
	if (devInodeTbl[inodeNum].refCount == 0){
		return -ENOENT;
	}

	m_stat->st_dev			= getDevInfo(inodeNum)->si_udev;
	m_stat->st_ino			= inodeNum;
	m_stat->st_mode			= devInodeTbl[inodeNum].mode;
	m_stat->st_nlink		= devInodeTbl[inodeNum].refCount;
	m_stat->st_uid			= 0;
	m_stat->st_gid			= 0;
	m_stat->st_size			= 0;
	m_stat->st_atime		= devInodeTbl[inodeNum].creatTime;
	m_stat->st_mtime		= devInodeTbl[inodeNum].creatTime;
	m_stat->st_ctime		= devInodeTbl[inodeNum].creatTime;
	m_stat->st_crtpos		= 0;
	if (devInodeTbl[inodeNum].dev == NULL){
		m_stat->st_blksize	= 0;
	}
	else{
		m_stat->st_blksize	= getDevInfo(inodeNum)->disk.sectSize;
	}
	m_stat->st_rdev			= getDevInfo(inodeNum)->si_udev;
	m_stat->st_flags		= 0;
	m_stat->st_blocks		= devInodeTbl[inodeNum].blocks;

	return NOERR;
}


// Init device filesystem.
STATIC void initDevFs()
{
	// root inode
	setRootInode();
}


//================================== PUBLIC =============================================


// ǥХinodeμ
// return : error number
int get_devstat(int inodeNum, DEV_STAT *m_devStat)
{
	if (devInodeTbl[inodeNum].refCount == 0){
		return -ENODEV;
	}

	m_devStat->prt_type = devInodeTbl[inodeNum].partType;
	m_devStat->major = devInodeTbl[inodeNum].major;
	m_devStat->prt_num = devInodeTbl[inodeNum].partNum;
	m_devStat->sect_size = getDevInfo(inodeNum)->disk.sectSize;
	m_devStat->all_sect = devInodeTbl[inodeNum].blocks;
	m_devStat->name = getDevInfo(inodeNum)->si_name;

	return NOERR;
}


int major(dev_t x)
{
	if (x == NODEV){
		return NOUDEV;
	}

	return((x->si_udev >> 8) & 0xff);
}


int minor(dev_t x)
{
	if (x == NODEV){
		return NOUDEV;
	}

	return(x->si_udev & 0xffff00ff);
}


udev_t makeudev(int x, int y)
{
	return ((x << 8) | y);
}


// Regist device.
// parameters : cdevsw¤, ޥʡֹ, uid, gid, ե⡼, Ͽ̾
// return : dev_t or NODEV
dev_t make_dev(struct cdevsw *devsw, int minor, uid_t uid, gid_t gid, int perms, const char *fmt, ...)
{
	int inodeNum, nextInodeNum;
	int i;
	dev_t dev;
	va_list ap;

	/* Name sizeγǧ  */
	if (MAX_DEVICE_NAME <= strlen(devsw->d_name)){
		return NODEV;
	}

	// ƱǥХ̾ʤ
	for (nextInodeNum = -1; (nextInodeNum = getNextInode(nextInodeNum)) != ERR;){
		if (strcmp(devInodeTbl[nextInodeNum].dev->si_name, devsw->d_name) == 0){
			return NODEV;
		}
	}

	// cdevsw¤ΤNULLؿ
	compile_devsw(devsw);

	// ǥХե륷ƥϿ롣
	inodeNum = setInode(
		devsw,
		0,
		0,
		0,
		getMajorNumber(),
		minor,
		perms
	);
	if (inodeNum < 0){
		return NODEV;
	}
	dev = devInodeTbl[inodeNum].dev;

	va_start(ap, fmt);
	i = kvprintf(fmt, NULL, dev->si_name, 32, ap);
	dev->si_name[i] = '\0';
	va_end(ap);

	return dev;
}


// Delete from regist table
// parameters : name
// return : error number
void destroy_dev(dev_t dev)
{
	int inodeNum;

	inodeNum = searchInodeNum(dev);
	if (inodeNum < 0){
		return;
	}

	unlinkInode(inodeNum);
}


/************************************************************************************************
 *
 * ǥХ
 *
 ************************************************************************************************/


//================================== PRIVATE ============================================


struct cdevsw 	*cdevsw[NUMCDEVSW];
static int		bmaj2cmaj[NUMCDEVSW];


//================================== PROTECTED ==========================================


// ǥХڥ졼˥СȤ롣
// return : error number
STATIC INLINE int convReadDev(
	dev_t dev,
	void *buf,
	size_t size,	// READ֥å
	size_t begin,	// Ƭ֥å
	int ioflag,
	size_t *o_size)	// READѥ֥å
{
	struct iovec iov;
	struct uio uio;
	int error;

	ASSERT(0 <= dev->disk.sectSize);

	iov.iov_base = buf;
	iov.iov_len = size * dev->disk.sectSize;
	uio.uio_iov = &iov;
	uio.uio_iovcnt = 1;
	uio.uio_offset = begin * dev->disk.sectSize;
	uio.uio_resid = iov.iov_len;
	uio.uio_segflg = UIO_USERSPACE;
	uio.uio_rw = UIO_READ;
	uio.uio_procp = NULL;

	error = dev->si_devsw->d_read(dev, &uio, ioflag);
	*o_size = size - (uio.uio_resid / dev->disk.sectSize);
	
	return error;
}


// ǥХڥ졼˥СȤ롣
// return : error number
STATIC INLINE int convWriteDev(
	dev_t dev,
	void *buf,
	size_t size,	// WRITE֥å
	size_t begin,	// Ƭ֥å
	int ioflag,
	size_t *o_size)	// WRITEѥ֥å
{
	struct iovec iov;
	struct uio uio;
	int error;

	ASSERT(0 <= dev->disk.sectSize);

	iov.iov_base = buf;
	iov.iov_len = size * dev->disk.sectSize;
	uio.uio_iov = &iov;
	uio.uio_iovcnt = 1;
	uio.uio_offset = begin * dev->disk.sectSize;
	uio.uio_resid = iov.iov_len;
	uio.uio_segflg = UIO_USERSPACE;
	uio.uio_rw = UIO_WRITE;
	uio.uio_procp = NULL;

	error = dev->si_devsw->d_write(dev, &uio, ioflag);
	*o_size = size - (uio.uio_resid / dev->disk.sectSize);
	
	return error;

}


//================================== PUBLIC =============================================


int noopen(dev_t dev, int flags, int fmt, struct proc *p)
{
	return (-ENODEV);
}

int noclose(dev_t dev, int flags, int fmt, struct proc *p)
{
	return (-ENODEV);
}

int noread(dev_t dev, struct uio *uio, int ioflag)
{
	return (-ENODEV);
}

int nowrite(dev_t dev, struct uio *uio, int ioflag)
{
	return (-ENODEV);
}

int noioctl(dev_t dev, u_long cmd, caddr_t data, int flags, struct proc *p)
{
	return (-ENODEV);
}

int nokqfilter(dev_t dev, struct knote *kn)
{
	return (-ENODEV);
}

int nommap(dev_t dev, vm_offset_t offset, int nprot)
{
	/* Don't return ENODEV.  That would allow mapping address ENODEV! */
	return (-1);
}

int nodump(dev_t dev)
{
	return (-ENODEV);
}

// nopoll
int seltrue(dev_t dev, int events, struct proc *p)
{
	return (events & (POLLIN | POLLOUT | POLLRDNORM | POLLWRNORM));
}


//FreeBSD
// cdevsw¤ΤåϿ롣
int cdevsw_add(struct cdevsw *newentry)
{
	int i;
	static int setup;

	if (!setup) {
		for (i = 0; i < NUMCDEVSW; i++){
			if (!bmaj2cmaj[i]){
				bmaj2cmaj[i] = 254;
			}
		}
		setup++;
	}

	compile_devsw(newentry);
	if (newentry->d_maj < 0 || newentry->d_maj >= NUMCDEVSW) {
		printf("%s: ERROR: driver has bogus cdevsw->d_maj = %d\n", newentry->d_name, newentry->d_maj);
		return -EINVAL;
	}
	if (newentry->d_bmaj >= NUMCDEVSW) {
		printf("%s: ERROR: driver has bogus cdevsw->d_bmaj = %d\n",newentry->d_name, newentry->d_bmaj);
		return -EINVAL;
	}
	if (newentry->d_bmaj >= 0 && (newentry->d_flags & D_DISK) == 0) {
		printf("ERROR: \"%s\" bmaj but is not a disk\n", newentry->d_name);
		return -EINVAL;
	}
	if (cdevsw[newentry->d_maj]) {
		printf("WARNING: \"%s\" is usurping \"%s\"'s cdevsw[]\n",newentry->d_name, cdevsw[newentry->d_maj]->d_name);
	}

	cdevsw[newentry->d_maj] = newentry;

	if (newentry->d_bmaj < 0){
		return (0);
	}

	if (bmaj2cmaj[newentry->d_bmaj] != 254) {
		printf("WARNING: \"%s\" is usurping \"%s\"'s bmaj\n", newentry->d_name, cdevsw[bmaj2cmaj[newentry->d_bmaj]]->d_name);
	}
	bmaj2cmaj[newentry->d_bmaj] = newentry->d_maj;

	return (0);
}


//FreeBSD
struct cdevsw *devsw(dev_t dev)
{
	if (dev->si_devsw != NULL){
		return (dev->si_devsw);
	}
	return(cdevsw[major(dev)]);
}


/***********************************************************************************
 *
 * ֥åǥХ
 *
 ***********************************************************************************/


enum{
	/* Partition table */
	PARTITION_TBL_MAX		= 4,		/* 1κѡƥơ֥ */
	PARTITION_BEGIN			= 0x1be,	/* ѡƥơ֥볫ϥեå */
	PARTITION_MAGIC_ADDR	= 0x1fe,	/* ѡƥơ֥ޥåʥСեå */
	PARTITION_MAGIC_NUMBER	= 0xaa55,	/* ѡƥơ֥ޥåʥС */

	/* Partition type */
	PARTITION_TYPE_NULL		= 0,	/* NULL partition */
	PARTITION_TYPE_EXT		= 0x5,	/* ĥ */
};


/* Partition table */
typedef struct{
	uint8_t		active;		/* Boot active flag=0x80 */
	uint8_t		chsBeg[3];	/* CHS begin block */
	uint8_t		type;		/* Partition type */
	uint8_t		chsEnd[3];	/* CHS last block */
	uint32_t	lbaBeg;		/* LBA begin block */
	uint32_t	size;		/* Partition size by sector */
}PARTITON_TBL;


//================================== PROTECTED ==========================================


// ѡƥơ֥򸡺
// return : NOERR
//          ENODEV=Ѥ or ĥѡƥ
//          ENOENT=ĥѡƥ󥯤κǸã
STATIC INLINE dev_t getDevInfo(int inodeNum);
STATIC int readPartition(const int inodeNum, const int partNum, PARTITON_TBL *o_partTbl)
{
	char *sectBuf;
	dev_t dev;
	PARTITON_TBL *partTbl;
	uint extBeg, extNext;
	size_t size;
	int c_part;
	int i, rest;

	// ǥХץ
	dev = getDevInfo(inodeNum);
	if (dev == NULL){
		return -ENOENT;
	}
	if (dev->si_devsw->d_open(dev, O_RDONLY, 0, NULL) < 0){
		return -EIO;
	}

	/* MBRɤ */
	sectBuf = kmalloc(dev->disk.sectSize);
	if (sectBuf == NULL){
		return -ENOMEM;
	}
	rest = convReadDev(dev, sectBuf, 1, 0, 0, &size);
	if (rest < 0){
		goto END;
	}

	/* ѡƥơ֥ǧ */
	if (*(uint16_t*)(sectBuf + PARTITION_MAGIC_ADDR) != PARTITION_MAGIC_NUMBER){
		rest = -ENODEV;
		goto END;
	}
	partTbl = (PARTITON_TBL*)(sectBuf + PARTITION_BEGIN);

	if (partNum <= PARTITION_TBL_MAX){
		/* ܥѡƥ򸡺 */
		c_part = partNum - 1;

		switch (partTbl[c_part].type){
			/* ΰγĥѡƥϳʤ */
			case PARTITION_TYPE_EXT:
				// 
			case PARTITION_TYPE_NULL:
				rest = -ENODEV;
				goto END;
			default:
				/* ѥѡƥơ֥ */
				o_partTbl->type		= partTbl[c_part].type;
				o_partTbl->lbaBeg	= partTbl[c_part].lbaBeg;
				o_partTbl->size		= partTbl[c_part].size;
		}
	}
	else{
		/* ĥѡƥ򸡺 */
		/*
		* MBRγĥѡƥ򸡺
		* out : c_part = ĥѡƥֹ
		*/
		for (i = 0;; ++i){
			if (PARTITION_TBL_MAX <= i){
				rest = -ENOENT;
				goto END;
			}
			if (partTbl[i].type == PARTITION_TYPE_EXT){
				c_part = i;
				break;
			}
		}

		/*
		* ĥѡƥ󥯤򤿤ɤ
		* out : extNext = ĥΰϤޤ꤫Υ֥å sectBuf = ĥΰΥѡƥ󥻥
		*/
		extBeg = extNext = partTbl[c_part].lbaBeg;
		for (i = PARTITION_TBL_MAX; ;){
			rest = convReadDev(dev, sectBuf, 1, extNext, 0, &size);
			if (rest < 0){
				goto END;
			}

			/* ѡƥȯ */
			if (++i == partNum){
				break;
			}

			/* ĥѡƥ󥯤νλ */
			if (partTbl[1].type != PARTITION_TYPE_EXT){
				rest = -ENOENT;
				goto END;
			}

			extNext = extBeg + partTbl[1].lbaBeg;
		}

		/* Ѥߥѡƥ */
		if (partTbl[0].type == PARTITION_TYPE_NULL){
			rest = -ENODEV;
			goto END;
		}
		else{
			/* ѥѡƥơ֥ */
			o_partTbl->type		= partTbl[0].type;
			o_partTbl->lbaBeg	= partTbl[0].lbaBeg + extNext;
			o_partTbl->size		= partTbl[0].size;
		}
	}

	rest = NOERR;
END:
		kfree(sectBuf);

	return rest;
}


// Attempt to read a disk label from a device using the indicated strategy
// routine.  The label must be partly set up before this: secpercyl, secsize
// and anything required in the strategy routine (e.g., dummy bounds for the
// partition containing the label) must be filled in before calling us.
// Returns NULL on success and an error string on failure.
char *readdisklabel(dev_t dev, struct disklabel *lp)
{
	char *buf;
	struct disklabel *dlp;
	char *msg = NULL;
	size_t size;

	buf = kmalloc(lp->d_secsize);
	if (buf == NULL){
		return "No memory";
	}

	if (convReadDev(dev, buf, 1, LABELSECTOR, 0, &size) != NOERR){
		msg = "I/O error";
	}
	else{
		for (dlp = (struct disklabel *)buf;
			dlp <= (struct disklabel *)(buf + lp->d_secsize - sizeof(*dlp));
			dlp = (struct disklabel *)((char *)dlp + sizeof(long))) {
			if (dlp->d_magic != DISKMAGIC || dlp->d_magic2 != DISKMAGIC) {
				if (msg == NULL){
					msg = "no disk label";
			   }
			}
			else if (MAXPARTITIONS < dlp->d_npartitions || dkcksum(dlp) != 0){
				msg = "disk label corrupted";
			}
			else{
				*lp = *dlp;
				msg = NULL;
				break;
			}
		}
	}
	kfree(buf);

	return (msg);
}


//FreeBSD
// Check new disk label for sensibility before setting it.
int setdisklabel(struct disklabel *olp, struct disklabel *nlp, u_long openmask)
{
	int i;
	struct partition *opp, *npp;

	/*
	* Check it is actually a disklabel we are looking at.
	*/
	if (nlp->d_magic != DISKMAGIC || nlp->d_magic2 != DISKMAGIC || dkcksum(nlp) != 0){
		return -EINVAL;
	}
	/*
	* For each partition that we think is open,
	*/
	while ((i = ffs((long)openmask)) != 0) {
		i--;
		/*
		* Check it is not changing....
		*/
		openmask &= ~(1 << i);
		if (nlp->d_npartitions <= i){
			return (EBUSY);
		}
		opp = &olp->d_partitions[i];
		npp = &nlp->d_partitions[i];
		if (npp->p_offset != opp->p_offset || npp->p_size < opp->p_size){
			return (EBUSY);
		}
		/*
		* Copy internally-set partition information
		* if new label doesn't include it.		XXX
		* (If we are using it then we had better stay the same type)
		* This is possibly dubious, as someone else noted (XXX)
		*/
		if (npp->p_fstype == FS_UNUSED && opp->p_fstype != FS_UNUSED) {
			npp->p_fstype = opp->p_fstype;
			npp->p_fsize = opp->p_fsize;
			npp->p_frag = opp->p_frag;
			npp->p_cpg = opp->p_cpg;
		}
	}
	nlp->d_checksum = 0;
	nlp->d_checksum = dkcksum(nlp);
	*olp = *nlp;

	return (0);
}


// Write disk label back to device after modification.
// return : error number
int writedisklabel(dev_t dev, struct disklabel *lp)
{
	char *buf;
	struct disklabel *dlp;
	size_t size;
	int error;

	if (lp->d_partitions[RAW_PART].p_offset != 0){
		return -EXDEV;			/* not quite right */
	}

	buf = kmalloc(lp->d_secsize);
	if (buf == NULL){
		return -ENOMEM;
	}
	memset(buf, 0, lp->d_secsize);

	dlp = (struct disklabel*)buf;
	*dlp = *lp;
	error = convWriteDev(dev, dlp, 1, LABELSECTOR, 0, &size);
	if (error != NOERR){
		return error;
	}
	kfree(buf);

	return NOERR;
}


//FreeBSD
char *dsname(
	dev_t	dev,
	int		unit,
	int		slice,
	int		part,
	char	*partname)
{
	static char name[32];
	const char *dname;

	dname = devsw(dev)->d_name;
	if (strlen(dname) > 16){
		dname = "nametoolong";
	}
	snprintf(name, sizeof(name), "%s%d", dname, unit);
	partname[0] = '\0';
	if (slice != WHOLE_DISK_SLICE || part != RAW_PART) {
		partname[0] = 'a' + part;
		partname[1] = '\0';
		if (slice != COMPATIBILITY_SLICE){
			snprintf(name + strlen(name), sizeof(name) - strlen(name), "s%d", slice - 1);
		}
	}
	return (name);
}


//FreeBSD
// Disk error is the preface to plaintive error messages
// about failing disk transfers.  It prints messages of the form
// hp0g: hard error reading fsbn 12345 of 12344-12347 (hp0 bn %d cn %d tn %d sn %d)
// if the offset of the error in the transfer and a disk label
// are both available.  blkdone should be -1 if the position of the error
// is unknown; the disklabel pointer may be null from drivers that have not
// been converted to use them.  The message is printed with printf
// if pri is LOG_PRINTF, otherwise it uses log at the specified priority.
// The message should be completed (with at least a newline) with printf
// or addlog, respectively.  There is no trailing space.
void diskerr(
	struct buf *bp,
	char *what,
	int pri,
	int blkdone,
	struct disklabel *lp)
{
	int unit = dkunit(bp->b_dev);
	int slice = dkslice(bp->b_dev);
	int part = dkpart(bp->b_dev);
	char partname[2];
	char *sname;
	daddr_t sn;

	sname = dsname(bp->b_dev, unit, slice, part, partname);
	printf("%s%s: %s %sing fsbn ", sname, partname, what, bp->b_flags & B_READ ? "read" : "writ");
	sn = bp->b_blkno;
	if (bp->b_bcount <= DEV_BSIZE){
		printf("%ld", (long)sn);
	}
	else {
		if (blkdone >= 0) {
			sn += blkdone;
			printf("%ld of ", (long)sn);
		}
		printf("%ld-%ld", (long)bp->b_blkno, (long)(bp->b_blkno + (bp->b_bcount - 1) / DEV_BSIZE));
	}
	if (lp && (blkdone >= 0 || bp->b_bcount <= lp->d_secsize)) {
		sn += lp->d_partitions[part].p_offset;
		/*
		* XXX should add slice offset and not print the slice,
		* but we don't know the slice pointer.
		* XXX should print bp->b_pblkno so that this will work
		* independent of slices, labels and bad sector remapping,
		* but some drivers don't set bp->b_pblkno.
		*/
		printf(" (%s bn %ld; cn %ld", sname, (long)sn, (long)(sn / lp->d_secpercyl));
		sn %= (long)lp->d_secpercyl;
		printf(" tn %ld sn %ld)", (long)(sn / lp->d_nsectors), (long)(sn % lp->d_nsectors));
	}
}


// ֥åǥХѡƥǥХե륷ƥϿ
// parmeters : ǥХ̵ֹ̾
int registBlockPartition(const char *name)
{
	int inodeNum;
	dev_t dev;
	PARTITON_TBL partTbl;
	int partNum;

	/* inode */
	inodeNum = lookupDevInode(name);
	if (inodeNum < 0) {
		return inodeNum;
	}

	/* struct cdevsw¤Τ */
	dev = getDevInfo(inodeNum);
	if (dev == NULL) {
		return -ENODEV;
	}

	/* ΰθ */
	for (partNum = 1; partNum <= PARTITION_TBL_MAX; ++partNum) {
		if (readPartition(inodeNum, partNum, &partTbl) == NOERR) {
			setInode(
				dev->si_devsw,
				partTbl.lbaBeg,
				partTbl.size,
				partTbl.type,
				devInodeTbl[inodeNum].major,
				partNum,
				S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP);
		}
	}

	/* ĥΰθ */
	for (partNum = PARTITION_TBL_MAX + 1;; ++partNum){
		int rest;

		switch (rest = readPartition(inodeNum, partNum, &partTbl)) {
			case NOERR:
				setInode(
					dev->si_devsw,
					partTbl.lbaBeg,
					partTbl.size,
					partTbl.type,
					devInodeTbl[inodeNum].major,
					partNum,
					S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP);
				break;
			case -ENODEV:
				continue;
			case -ENOENT:
				goto END;
			default:
				printk("%s registBlockPartition() unkown return value!:%d", __FILE__, rest);
		}
	}
END:
		return NOERR;
}


/************************************************************************************************
 *
 * ľžؿ
 *
 ************************************************************************************************/


//================================== PUBLIC =============================================


// return : error number
int read_direct(
	int inodeNum,
	void *buf,
	size_t i_sectors,
	size_t begin,
	size_t *o_sectors)	// READѥ֥å
{
	dev_t dev = getDevInfo(inodeNum);
	size_t sectors;

	/* ǥ¤Υå */
	if (dev->disk.sectors < begin){
		return -EIO;
	}
	if ((begin + i_sectors) <= dev->disk.sectors){
		sectors = i_sectors;
	}
	else{
		sectors = dev->disk.sectors - begin;
	}

	return convReadDev(dev, buf, sectors, begin + dev->disk.beginSector, 0, o_sectors);
}


// return : error number
int write_direct(
	int inodeNum,
	void *buf,
	size_t i_sectors,
	size_t begin,
	size_t *o_sectors)	// WRITEѥ֥å
{
	dev_t dev = getDevInfo(inodeNum);
	size_t sectors;

	/* ǥ¤Υå */
	if (dev->disk.sectors < begin){
		return -EIO;
	}
	if ((begin + i_sectors) <= dev->disk.sectors){
		sectors = i_sectors;
	}
	else{
		sectors = dev->disk.sectors - begin;
	}

	return convWriteDev(dev, buf, sectors, begin + dev->disk.beginSector, 0, o_sectors);
}


/************************************************************************************************
 *
 * ǥХå
 *
 * ա
 *  ǥХϥ512ХȤˤƤΤǡ¾Υ
 *  ǥХоݤˤˤϡɬס
 *
 ************************************************************************************************/


enum{
	CACHE_NUM = 8,				/* IOå֥å */
	CACHE_BLOCK_SIZE = 0x1000,	/* IOå֥å */
	HASH_NUM = CACHE_NUM,		/* Number of cache hash table */
	CACHE_UNIT_SIZE = 512,		/* å֥åΥӥåȥޥåñ̥Хȥ */
};

typedef struct CACHE_TABLE{
	struct CACHE_TABLE			*hashNext;	// Hash next
	struct CACHE_TABLE			*hashPrev;
	struct CACHE_TABLE			*cacheNext;	// Cache chain next
	struct CACHE_TABLE			*cachePrev;
	struct CACHE_TABLE volatile	*writeNext;	// ٱ񤭹ߥꥹ
	struct CACHE_TABLE		 	*writePrev;
	char						*cache;		// Cache address
	uint						block;		// Device block index
	uchar						inodeNum;	// Device inode number
	uchar volatile				bitmap;		// ǡ¸ߥӥåȥޥå
	uchar volatile				writing;	// ٱԤե饰
	uchar volatile				busy;		// ե饰
	int							gate;		// ԥå
}CACHE_TABLE;


//================================== PRIVATE ============================================


static CACHE_TABLE cache_tbl[CACHE_NUM];			// Cache table
static CACHE_TABLE *cache_hash[HASH_NUM];			// Hash table for searching cache table
static CACHE_TABLE *cacheOldest = &cache_tbl[0];	// old chache block.nextֿ֥å
static int cacheLock;								// å󥯹ѥå
static PROC *writeTask;								// ٱ񤭹ѥ
static CACHE_TABLE delayTop;						// ٱ񤭹ߥꥹȤƬ
static int delayLock;								// å󥯹ѥå
static uchar bitSects[] = {							// ȤΥӥåȥޥå
	0,0x1,0x3,0x7,0xf,0x1f,0x3f,0x7f,0xff
};


// ǥХ饭åɤ߹ࡣ
// return : error number
STATIC int readDevice(CACHE_TABLE *cacheTbl, uchar bitmap)
{
	//ɤ߹ɬפɤ߹ޤƤʤӥåȤΤߣˤ
	uchar map = ~bitmap | cacheTbl->bitmap;
	dev_t dev = getDevInfo(cacheTbl->inodeNum);
	int start, last, allLast;
	size_t ioSize;
	int rest;

	for (start = 0, allLast = CACHE_BLOCK_SIZE / dev->disk.sectSize; start < allLast; ++start){
		if ((map & (1 << start)) == 0){
			for (last = start + 1; last < allLast; ++last){
				if (map & (1 << last)){
					break;
				}
			}
			rest = convReadDev(
				dev, 
				cacheTbl->cache + start * CACHE_UNIT_SIZE,
				last - start, 
				start + cacheTbl->block + dev->disk.beginSector, 
				0,
				&ioSize);
			if (rest < 0){
				return rest;
			}
			ASSERT(ioSize == last - start);

			start = last;
		}
	}

	cacheTbl->bitmap |= bitmap;

	return NOERR;
}


// å夫ǥХ˽᤹
// return : error number
STATIC int writeDevice(CACHE_TABLE *cacheTbl)
{
	dev_t dev = getDevInfo(cacheTbl->inodeNum);
	int start, last;
	size_t size;
	int rest;

	if (cacheTbl->bitmap == 0xff){
		rest = convWriteDev(
			dev, 
			cacheTbl->cache, 
			CACHE_BLOCK_SIZE / CACHE_UNIT_SIZE, 
			cacheTbl->block + dev->disk.beginSector, 
			0,
			&size);
		if (rest < 0){
			return rest;
		}
		ASSERT(size == CACHE_BLOCK_SIZE / CACHE_UNIT_SIZE);
	}
	else{
		// սǥХ˽񤭹ࡣ
		for (start = 0; start < CACHE_BLOCK_SIZE / CACHE_UNIT_SIZE; ++start){
			if (cacheTbl->bitmap & (1 << start)){
				for (last = start + 1; last < CACHE_BLOCK_SIZE / CACHE_UNIT_SIZE; ++last){
					if ((cacheTbl->bitmap & (1 << last)) == 0){
						break;
					}
				}
				rest = convWriteDev(
					dev, 
					cacheTbl->cache + start * CACHE_UNIT_SIZE, 
					last - start, 
					cacheTbl->block + start + dev->disk.beginSector, 
					0,
					&size);
				if (rest < 0){
					return rest;
				}
				ASSERT(size == last - start);
				start = last;
			}
		}
	}

	return NOERR;
}


// ǥХؤٱ񤭹ߥ
STATIC void delayWriteTask()
{
	PROC **q;
	CACHE_TABLE *cacheTbl = NULL;

	// ƥץڤΥ
	for (q = &writeTask->parent->child; *q != writeTask; q = &(*q)->brother){;
	}
	*q = writeTask->brother;

	// λ륷ʥ򤹤٤ƥޥ
	setSigAllMask();

	// å
	enter_spinlock(&delayLock);
	for (;;){
		// 񤭹ԤФ
		if (delayTop.writeNext == &delayTop){
			if (cacheTbl != NULL){
				cacheTbl->writing = NO;
			}
			delFromSchedule(TASK_WAIT);

			// å
			exit_spinlock(&delayLock);

			wait_task();

			// å
			enter_spinlock(&delayLock);
		}
		else{
			cacheTbl = delayTop.writePrev;
			cacheTbl->writeNext->writePrev = cacheTbl->writePrev;
			cacheTbl->writePrev->writeNext = cacheTbl->writeNext;
			cacheTbl->writeNext = cacheTbl->writePrev = cacheTbl;

			// å
			exit_spinlock(&delayLock);

			if (writeDevice(cacheTbl) < 0){
				printk("Device error in delay write function!\n");
			}

			// å
			enter_spinlock(&delayLock);

			if (cacheTbl->writeNext == cacheTbl){
				cacheTbl->writing = NO;
			}
		}
	}
}


//ադδؿ"cacheLock"Ƕ礷ʤؿƤӽФ뤳
// ٱ񤭹ߥꥹȤ˲ä롣
STATIC void addWriteList(CACHE_TABLE *cacheTbl)
{
	enter_spinlock(&delayLock);
	{
		// Ǥ˥󥯤³Ƥ⤢Τǰöǽ򤹤
		cacheTbl->writeNext->writePrev = cacheTbl->writePrev;
		cacheTbl->writePrev->writeNext = cacheTbl->writeNext;

		cacheTbl->writeNext = delayTop.writeNext;
		cacheTbl->writePrev = &delayTop;
		delayTop.writeNext->writePrev = cacheTbl;
		delayTop.writeNext = cacheTbl;
		cacheTbl->writing = YES;

		// ٱ񤭹ߥ򵯤
		addToSchedule(writeTask, TASK_WAIT);
	}
	exit_spinlock(&delayLock);

}


// ϥåͤ׻롣
STATIC INLINE int calcHash(int value)
{
	return (value / 8) % HASH_NUM;
}


// åơ֥뤫鳺ǥХ֥åõ
// return : block addres or NULL
STATIC CACHE_TABLE *searchCache(int inodeNum, size_t begin)
{
	int hash = calcHash(begin);
	CACHE_TABLE *cacheFind;

	for (cacheFind = cache_hash[hash]; cacheFind != NULL; cacheFind = cacheFind->hashNext){
		// åˤ
		if ((cacheFind->block == begin) && (cacheFind->inodeNum == inodeNum)){
			// 󥯤ֿ֤ˤ
			if (cacheFind == cacheOldest){
				cacheOldest = cacheOldest->cachePrev;
			}
			else{
				cacheFind->cacheNext->cachePrev = cacheFind->cachePrev;
				cacheFind->cachePrev->cacheNext = cacheFind->cacheNext;
				cacheFind->cacheNext = cacheOldest->cacheNext;
				cacheFind->cachePrev = cacheOldest;
				cacheOldest->cacheNext->cachePrev = cacheFind;
				cacheOldest->cacheNext = cacheFind;
			}

			return cacheFind;
		}
	}

	return NULL;
}


// ˥å֥åƤ롣
// return : cache table
STATIC CACHE_TABLE *getNewCache(int inodeNum, size_t begin)
{
	int hash = calcHash(begin);
	CACHE_TABLE *newCache;

	// ָŤå򲣼
	newCache = cacheOldest;
	cacheOldest = cacheOldest->cachePrev;

	// ϥå󥯤
	newCache->hashPrev->hashNext = newCache->hashNext;
	if (newCache->hashNext != NULL){
		newCache->hashNext->hashPrev = newCache->hashPrev;
	}

	// ϥå󥯤˲ä
	newCache->hashNext = cache_hash[hash];
	newCache->hashPrev = (CACHE_TABLE*)&cache_hash[hash];
	if (cache_hash[hash] != NULL){
		cache_hash[hash]->hashPrev = newCache;
	}
	cache_hash[hash] = newCache;

	return newCache;
}


// ٱ񤭹ߥεư
STATIC void startDelayWriteTask()
{
	switch (sys_fork()){
		case -1:
			printk("Faild startDelayWriteTask\n");
			break;
		case 0:
			writeTask = get_current_task();
			delayWriteTask();
			// äƤʤ
	}
}


//================================== PROTECTED ==========================================


// ߤꥭå֥å򤹤٤ٱߤ롣
STATIC void writeBackAllCache()
{
	// ٱ񤭹Ԥ
	for (;;){
		enter_spinlock(&delayLock);
		if (delayTop.writeNext != &delayTop){
			exit_spinlock(&delayLock);
			wait_task();
		}
		else{
			exit_spinlock(&delayLock);
			break;
		}
	}
}


// return : 0 or Error number
STATIC int initCache()
{
	int i;

	// åơ֥ν
	for (i = 0; i < CACHE_NUM; ++i){
		memset(&cache_tbl[i], 0, sizeof(CACHE_TABLE));
		cache_tbl[i].cache = kmalloc(CACHE_BLOCK_SIZE);
		if (cache_tbl[i].cache == NULL){
			return -ENOMEM;
		}
		cache_tbl[i].cacheNext = &cache_tbl[i + 1];
		cache_tbl[i].cachePrev = &cache_tbl[i - 1];
		cache_tbl[i].hashPrev = &cache_tbl[i];
		cache_tbl[i].writeNext = cache_tbl[i].writePrev = &cache_tbl[i];
	}
	cache_tbl[CACHE_NUM - 1].cacheNext = &cache_tbl[0];
	cache_tbl[0].cachePrev = &cache_tbl[CACHE_NUM - 1];

	// ϥå󥯤ν
	for (i = 0; i < HASH_NUM; ++i){
		cache_hash[i] = NULL;
	}

	// ٱ񤭹ߥ󥯤ν
	delayTop.writeNext = delayTop.writePrev = &delayTop;

	return 0;
}


//================================== PUBLIC =============================================


//աճߥϥɥ餫θƤӽФԲ
// return : error number
int read_cache(int inodeNum, void *buf, size_t i_sectors, size_t begin, size_t *o_sectors)
{
	dev_t dev = getDevInfo(inodeNum);
	size_t sectors;
	size_t restSects;								/* Ĥž */
	char *readBuf = (char*)buf;
	int sectorSize = dev->disk.sectSize;			/* ǥХ */
	int cacheSects = CACHE_BLOCK_SIZE / sectorSize;	/* å֥åˤϤ륻 */
	int start, cacheStart, copySects;
	CACHE_TABLE *cacheTbl;

	/* ǥ¤Υå */
	if (dev->disk.sectors < begin){
		return -EIO;
	}
	if ((begin + i_sectors) <= dev->disk.sectors){
		restSects = sectors = i_sectors;
	}
	else{
		restSects = sectors = dev->disk.sectors - begin;
	}

	start = begin % cacheSects;
	cacheStart = ROUNDDOWN(begin, cacheSects);
	copySects = (restSects <= cacheSects - start)? restSects : cacheSects - start;
	for(;;){
		enter_spinlock(&cacheLock);
		{
			// å򸡺
			if ((cacheTbl = searchCache(inodeNum, cacheStart)) != NULL){
				// åԤ
				while (cacheTbl->busy == YES);
			}
			else{
				cacheTbl = getNewCache(inodeNum, cacheStart);

				// åԤ
				while (cacheTbl->busy == YES);
				// ٱ񤭹Ԥ
				while (cacheTbl->writing == YES);

				cacheTbl->inodeNum = inodeNum;
				cacheTbl->block = cacheStart;
				cacheTbl->bitmap = 0;
			}

			// å
			cacheTbl->busy = YES;
		}
		exit_spinlock(&cacheLock);

		// å׵᥻ʤХǥХɤ߹
		if ((cacheTbl->bitmap & (bitSects[copySects] << start)) != (bitSects[copySects] << start)){
			if (readDevice(cacheTbl, bitSects[copySects] << start) < 0){
				// ---> åλ
				cacheTbl->busy = NO;
				return -EIO;
			}
		}

		// Хåե˥ԡ
		memcpy(readBuf, cacheTbl->cache + start * sectorSize, copySects * sectorSize);

		// åλ
		cacheTbl->busy = NO;

		if ((restSects -= copySects) == 0){
			break;
		}
		readBuf += copySects * sectorSize;
		cacheStart += cacheSects;
		copySects = (restSects < cacheSects)? restSects : cacheSects;
		start = 0;
	}
	*o_sectors = sectors;
	
	return NOERR;
}


//ա ߥϥɥ餫θƤӽФԲ
// return : error number
int write_cache(int inodeNum, void *wbuf, size_t i_sectors, size_t begin, size_t *o_sectors)
{
	dev_t dev = getDevInfo(inodeNum);
	size_t sectors;
	int restSects;									/* Ĥž */
	char *writeBuf = (char*)wbuf;
	int sectorSize = dev->disk.sectSize;			/* ǥХ */
	int cacheSects = CACHE_BLOCK_SIZE / sectorSize;	/* å֥åˤϤ륻 */
	int start, cacheStart, copySects;
	CACHE_TABLE *cacheTbl;

	/* ǥ¤Υå */
	if (dev->disk.sectors < begin){
		return -EIO;
	}
	if ((begin + i_sectors) <= dev->disk.sectors){
		restSects = sectors = i_sectors;
	}
	else{
		restSects = sectors = dev->disk.sectors - begin;
	}

	start = begin % cacheSects;
	cacheStart = ROUNDDOWN(begin, cacheSects);
	copySects = (restSects <= cacheSects - start)? restSects : cacheSects - start;
	for(;;){
		enter_spinlock(&cacheLock);
		{
			// å򸡺
			if ((cacheTbl = searchCache(inodeNum, cacheStart)) != NULL){
				// åԤ
				while (cacheTbl->busy == YES){
				}
			}
			else{
				cacheTbl = getNewCache(inodeNum, cacheStart);

				// åԤ
				while (cacheTbl->busy == YES){
				}
				// ٱ񤭹Ԥ
				while (cacheTbl->writing == YES){
				}

				cacheTbl->inodeNum = inodeNum;
				cacheTbl->block = cacheStart;
				cacheTbl->bitmap = 0;
			}

			// å
			cacheTbl->busy = YES;
		}
		exit_spinlock(&cacheLock);

		// å˥ԡ
		memcpy(cacheTbl->cache + start * sectorSize, writeBuf, copySects * sectorSize);
		writeBuf += copySects * sectorSize;
		cacheTbl->bitmap |= bitSects[copySects] << start;

		// ٱ񤭹
		addWriteList(cacheTbl);

		// åλ
		cacheTbl->busy = NO;

		if ((restSects -= copySects) == 0){
			break;
		}
		cacheStart += cacheSects;
		copySects = (restSects < cacheSects)? restSects : cacheSects;
		start = 0;
	}
	*o_sectors = sectors;

	return NOERR;
}


int initBlockCache()
{
	/* IOå */
	initCache();

	/* ٱ񤭹ѥư롣 */
	startDelayWriteTask();

	return 0;
}


/****************************************************************************************
* ǥХ
*****************************************************************************************/


//================================== PRIVATE ============================================


static void (*intrHandler[IRQ_ENTRY])(void *);
static void *intrArg[IRQ_ENTRY];

STATIC int deviceIntr(int irq)
{
/****************************************************************************************************************/
//printk("irq=%d ", irq);
/****************************************************************************************************************/
	intrHandler[irq](intrArg[irq]);
/****************************************************************************************************************/
//printk("irq end\n");
/****************************************************************************************************************/
	return 1;
}


//================================== PUBLIC =============================================


// ǥХߤ
// return : error number
void setupIntr(
	struct resource *irq,
	void (*handler)(void *),
	void *softc)
{
	irq_entry[irq->r_start] = deviceIntr;
	intrHandler[irq->r_start] = handler;
	intrArg[irq->r_start] = softc;
}


/****************************************************************************************
* ǥХХåե
*****************************************************************************************/


//================================== PUBLIC =============================================


//FreeBSD
// Seek sort for disks.
// The buf_queue keep two queues, sorted in ascending block order.  The first
// queue holds those requests which are positioned after the current block
// (in the first request); the second, which starts at queue->switch_point,
// holds requests which came in after their block number was passed.  Thus
// we implement a one way scan, retracting after reaching the end of the drive
// to the first request on the second queue, at which time it becomes the
// first queue.
// A one-way scan is natural because of the way UNIX read-ahead blocks are
// allocated.
void bufqdisksort(struct buf_queue_head *bufq, struct buf *bp)
{
	struct buf *bq;
	struct buf *bn;
	struct buf *be;
	
	be = TAILQ_LAST(&bufq->queue, buf_queue);
	/*
	 * If the queue is empty or we are an
	 * ordered transaction, then it's easy.
	 */
	if ((bq = bufq_first(bufq)) == NULL || (bp->b_flags & B_ORDERED) != 0) {
		bufq_insert_tail(bufq, bp);
		return;
	}
	else if (bufq->insert_point != NULL) {
		/*
		 * A certain portion of the list is
		 * "locked" to preserve ordering, so
		 * we can only insert after the insert
		 * point.
		 */
		bq = bufq->insert_point;
	}
	else {
		/*
		 * If we lie before the last removed (currently active)
		 * request, and are not inserting ourselves into the
		 * "locked" portion of the list, then we must add ourselves
		 * to the second request list.
		 */
		if (bp->b_pblkno < bufq->last_pblkno) {
			bq = bufq->switch_point;
			/*
			 * If we are starting a new secondary list,
			 * then it's easy.
			 */
			if (bq == NULL) {
				bufq->switch_point = bp;
				bufq_insert_tail(bufq, bp);
				return;
			}
			/*
			 * If we lie ahead of the current switch point,
			 * insert us before the switch point and move
			 * the switch point.
			 */
			if (bp->b_pblkno < bq->b_pblkno) {
				bufq->switch_point = bp;
				TAILQ_INSERT_BEFORE(bq, bp, b_act);
				return;
			}
		}
		else {
			if (bufq->switch_point != NULL)
				be = TAILQ_PREV(bufq->switch_point, buf_queue, b_act);
			/*
			 * If we lie between last_pblkno and bq,
			 * insert before bq.
			 */
			if (bp->b_pblkno < bq->b_pblkno) {
				TAILQ_INSERT_BEFORE(bq, bp, b_act);
				return;
			}
		}
	}

	/*
	 * Request is at/after our current position in the list.
	 * Optimize for sequential I/O by seeing if we go at the tail.
	 */
	if (bp->b_pblkno > be->b_pblkno) {
		TAILQ_INSERT_AFTER(&bufq->queue, be, bp, b_act);
		return;
	}

	/* Otherwise, insertion sort */
	while ((bn = TAILQ_NEXT(bq, b_act)) != NULL) {
		/*
		 * We want to go after the current request if it is the end
		 * of the first request list, or if the next request is a
		 * larger cylinder than our request.
		 */
		if (bn == bufq->switch_point || bp->b_pblkno < bn->b_pblkno){
			break;
		}
		bq = bn;
	}
	TAILQ_INSERT_AFTER(&bufq->queue, bq, bp, b_act);
}


// ֥åǥХIO
// return : error number
int physio(dev_t dev, struct uio *uio, int ioflag)
{
	int i;
	int error = NOERR;
	u_int iolen;
	struct buf *bp;

	bp = kmalloc(sizeof(*bp));
	if (bp == NULL){
		return -ENOMEM;
	}
	memset(bp, 0, sizeof(*bp));

	for (i = 0; i < uio->uio_iovcnt; i++) {
		while (uio->uio_iov[i].iov_len) {
			if (uio->uio_rw == UIO_READ){
				bp->b_flags = B_PHYS | B_CALL | B_READ;
			}
			else {
				bp->b_flags = B_PHYS | B_CALL | B_WRITE;
			}
			bp->b_dev = dev;
			bp->b_data = uio->uio_iov[i].iov_base;		// 桼Хåե
			bp->b_bcount = uio->uio_iov[i].iov_len;		// IOХȿ
			bp->b_offset = uio->uio_offset;				// Хñ̳ϥեå

			// ϥ֥åֹ
			bp->b_blkno = bp->b_offset >> DEV_BSHIFT;		// 512Хñ̳ϥ֥å
			if ((devsw(dev)->d_flags & D_TYPEMASK) != D_MEM){
				if ((daddr_t)bp->b_blkno != bp->b_blkno) {
					error = -EINVAL; /* blockno overflow */
					goto doerror;
				}
			}

			// ǥХREAD
			BUF_STRATEGY(bp, 0);

			// ХåեžԤ
			while ((bp->b_flags & B_DONE) == 0){
				tsleep((caddr_t)bp, PRIBIO, "physstr", 0);
			}
			iolen = bp->b_bcount - bp->b_resid;			// b_resid = ̤žХȿ
			if (iolen == 0 && !(bp->b_flags & B_ERROR)){
				goto doerror;	/* EOF */
			}
			uio->uio_iov[i].iov_len -= iolen;
			uio->uio_iov[i].iov_base += iolen;
			uio->uio_resid -= iolen;
			uio->uio_offset += iolen;
			if( bp->b_flags & B_ERROR) {
				error = bp->b_error;
				goto doerror;
			}
		}
	}
doerror:
	kfree(bp);
	return (error);
}


//	Finish I/O on a buffer, optionally calling a completion function.
//	This is usually called from an interrupt so process blocking is
//	not allowed.
//	biodone does not mess with B_INVAL, allowing the I/O routine or the
//	initiator to leave B_INVAL set to brelse the buffer out of existance
//	in the biodone routine.
void biodone(struct buf * bp)
{
	int s;

	s = splbio();

	bp->b_flags |= B_DONE;

	/* call optional completion function if requested */
	if (bp->b_flags & B_CALL) {
		bp->b_flags &= ~B_CALL;
		sys_wake(bp->proc);
		splx(s);
		return;
	}

	splx(s);
}


/************************************************************************************************
 *
 * ե륵ӥؿ
 *
 ************************************************************************************************/


STATIC int lookup(const char *path, const int inodeNum, void *inodeObj, uint *o_inodeNum, const char **o_path)
{
	int rest;

	rest = lookupDevInode(path);
	if (rest < 0){
		return rest;
	}
	*o_inodeNum = rest;

	return NOERR;
}


STATIC int creat(const char *path, const int inodeNum, void *inodeObj, const int mode, uint *o_inodeNum)
{
	return -EOPNOTSUPP;
}


STATIC int open(const int din, const uint inodeNum, int oflag, void **o_inodeObj, int *o_type)
{
	dev_t dev = getDevInfo(inodeNum);

	// ǥХopen
	if (inodeNum != DEVICE_ROOT_INODE){
		if (dev->si_devsw->d_open(dev, oflag, 0, NULL) < 0){
			return -EIO;
		}
	}

	// ֥åͤꤹ
	switch (getMode(inodeNum) & S_IFMT){
		case S_IFCHR:
			dev->disk.sectSize = 1;
		case S_IFBLK:
			dev->disk.beginSector = getBegBlock(inodeNum);
			if (0 < getBlocks(inodeNum)){
				dev->disk.sectors = getBlocks(inodeNum);
			}
	}

	*o_inodeObj = (void*)inodeNum;
	*o_type = getMode(inodeNum) & S_IFMT;

	return NOERR;
}


STATIC int close(const int din, void *inodeObj)
{
	int inodeNum = (int)inodeObj;
	dev_t dev = getDevInfo(inodeNum);

	/* ǥХclose */
	if (inodeNum != DEVICE_ROOT_INODE){
		if (dev->si_devsw->d_close(dev, 0, 0, NULL) < 0){
			return -EIO;
		}
	}

	return NOERR;
}


STATIC int read(const int din, void *inodeObj, void *buf, size_t size, size_t begin)
{
	char *readBuf = buf;
	int inodeNum = (int)inodeObj;
	dev_t dev = getDevInfo(inodeNum);
	size_t ioSize;					// žѥ
	int error;

	switch (getMode(inodeNum) & S_IFMT){
		/* 饯ǥХ */
		case S_IFCHR:
			error = convReadDev(dev, readBuf, size, 0, 0, &ioSize);
			if (error != NOERR){
				return error;
			}
			else{
				return ioSize;
			}
		/* ֥åǥХ */
		case S_IFBLK:{
			char *sectBuf = NULL;
			int sectorSize = getDevInfo(inodeNum)->disk.sectSize;	/*  */
			int sectBeg = begin / sectorSize;						/* žϤΥֹ */
			size_t restSize = size;									/* ĤžХȿ */
			size_t sectors;

			/* ǽΥ̤ž */
			if (0 < (begin % sectorSize)){
				int forward = begin % sectorSize;
				int back;

				sectBuf = kmalloc(sectorSize);
				if (sectBuf == NULL){
					return -ENOMEM;
				}
				error = read_cache(inodeNum, sectBuf, 1, sectBeg, &ioSize);
				if (error != NOERR){
					goto ERR;
				}
				if (ioSize == 0){
					goto END;
				}
				back = sectorSize - forward;
				if (size < back){
					back = size;
				}
				memcpy(readBuf, sectBuf + forward, back);
				readBuf += back;
				sectBeg += 1;
				restSize -= back;
			}

			sectors = restSize / sectorSize;
			error = read_cache(inodeNum, readBuf, sectors, sectBeg, &ioSize);
			if (error != NOERR){
				goto ERR;
			}
			restSize -= ioSize * sectorSize;
			if (ioSize < sectors){
				goto END;
			}
			sectBeg += ioSize;
			readBuf += ioSize * sectorSize;

			/* Ĥ꤬̤ž */
			if (0 < restSize){
				if (sectBuf == NULL){
					sectBuf = kmalloc(sectorSize);
					if (sectBuf == NULL){
						return -ENOMEM;
					}
				}
				error = read_cache(inodeNum, sectBuf, 1, sectBeg, &ioSize);
				if (error != NOERR){
					goto ERR;
				}
				if (ioSize == 0){
					goto END;
				}
				memcpy(readBuf, sectBuf, restSize);
				restSize = 0;
			}
END:
			kfree(sectBuf);
			return size - restSize;
ERR:
			kfree(sectBuf);
			return error;
		}
		/* root directory */
		case S_IFDIR:
			return 0;
		default:
			ASSERT(0);
			return -ENOENT;
	}
}


STATIC int write(const int din, void *inodeObj, void *buf, size_t size, size_t begin)
{
	char *writeBuf = buf;
	int inodeNum = (int)inodeObj;
	dev_t dev = getDevInfo(inodeNum);
	size_t ioSize;					// žѥ
	int error;

	switch (getMode(inodeNum) & S_IFMT){
		/* 饯ǥХ */
		case S_IFCHR:
			error = convWriteDev(dev, writeBuf, size, 0, 0, &ioSize);
			if (error != NOERR){
				return error;
			}
			else{
				return ioSize;
			}
		/* ֥åǥХ */
		case S_IFBLK:{
			char *sectBuf = NULL;
			int sectorSize = getDevInfo(inodeNum)->disk.sectSize;	/*  */
			int sectBeg = begin / sectorSize;						/* žϤΥֹ */
			size_t restSize = size;									/* ĤžХȿ */
			size_t sectors;

			/* ǽΥ̤ž */
			if (0 < (begin % sectorSize)){
				int forward = begin % sectorSize;
				int back;

				sectBuf = kmalloc(sectorSize);
				if (sectBuf == NULL){
					return -ENOMEM;
				}
				error = read_cache(inodeNum, sectBuf, 1, sectBeg, &ioSize);
				if (error != NOERR){
					goto ERR;
				}
				if (ioSize == 0){
					goto END;
				}
				back = sectorSize - forward;
				if (size < back){
					back = size;
				}
				memcpy(sectBuf + forward, writeBuf, back);
				error = write_cache(inodeNum, sectBuf, 1, sectBeg, &ioSize);
				if (error != NOERR){
					goto ERR;
				}
				if (ioSize == 0){
					goto END;
				}
				writeBuf += back;
				sectBeg += 1;
				restSize -= back;
			}

			sectors = restSize / sectorSize;
			error = write_cache(inodeNum, writeBuf, sectors, sectBeg, &ioSize);
			if (error != NOERR){
				goto ERR;
			}
			restSize -= ioSize * sectorSize;
			if (ioSize < sectors){
				goto END;
			}
			sectBeg += ioSize;
			writeBuf += ioSize * sectorSize;

			/* Ĥ꤬̤ž */
			if (0 < restSize){
				if (sectBuf == NULL){
					sectBuf = kmalloc(sectorSize);
					if (sectBuf == NULL){
						return -ENOMEM;
					}
				}
				error = read_cache(inodeNum, sectBuf, 1, sectBeg, &ioSize);
				if (error != NOERR){
					goto ERR;
				}
				if (ioSize == 0){
					goto END;
				}
				memcpy(sectBuf, writeBuf, restSize);
				error = write_cache(inodeNum, sectBuf, 1, sectBeg, &ioSize);
				if (error != NOERR){
					goto ERR;
				}
				if (ioSize == 0){
					goto END;
				}
				restSize = 0;
			}
END:
			kfree(sectBuf);
			return size - restSize;
ERR:
			kfree(sectBuf);
			return error;
		}
		/* root directory */
		case S_IFDIR:
			return 0;
		default:
			ASSERT(0);
			return -ENOENT;
	}
}


STATIC int ioctl(const int din, void *inodeObj, int cmd, caddr_t param, int fflag)
{
	int inodeNum = (int)inodeObj;
	dev_t dev = getDevInfo(inodeNum);

	if (inodeNum == DEVICE_ROOT_INODE){
		return -ENOENT;
	}

	return dev->si_devsw->d_ioctl(dev, cmd, param, fflag, NULL);
}


STATIC int poll(void *inodeObj, int events)
{
	int inodeNum = (int)inodeObj;
	dev_t dev = getDevInfo(inodeNum);

	if (inodeNum == DEVICE_ROOT_INODE){
		return -ENOENT;
	}

	return dev->si_devsw->d_poll(dev, events, NULL);
}


STATIC int opendir(const char *path, const int din, void *dirInodeObj, void **o_inodeObj, uint *o_block, int *o_index)
{
	int inodeNum;

	/* ѥθ */
	inodeNum = lookupDevInode(path);
	if (inodeNum < 0){
		return inodeNum;
	}

	*o_inodeObj = (void*)inodeNum;
	*o_block = 0;
	*o_index = -1;

	return NOERR;
}


STATIC int readdir(int din, void *inodeObj, uint *m_block, int *m_index, char *o_name)
{
	const char *name;
	int nodeNum = (int)inodeObj;
	int nextInode;

	if (nodeNum != DEVICE_ROOT_INODE){
		return -ENOENT;
	}

	/* Υȥ꡼򸡺 */
	nextInode = getNextInode(*m_index);
	if (nextInode == ERR){
		return -ENOENT;
	}

	*m_index = nextInode;

	/* ̾ */
	name = getName(nextInode);
	strcpy(o_name, name);
	o_name[strlen(name)] = '\0';

	return 0;
}


STATIC int stat(int din, void *inodeObj, struct stat *m_stat)
{
	int inodeNum = (int)inodeObj;

	return getStat(inodeNum, m_stat);
}


STATIC int mount(const int din, void **o_rootInodeObj)
{
	return -EOPNOTSUPP;
}


STATIC int umount(const int din, void *rootInodeObj)
{
	return -EOPNOTSUPP;
}


STATIC int rename(int din, void *OldDirInodeObj, const char *oldPath, void *newDirInodeObj, const char *newPath)
{
	return -EOPNOTSUPP;
}


STATIC int mkdir(const char *path, const int din, void *dirInodeObj, int mode)
{
	return -EOPNOTSUPP;
}

STATIC int delete(const char *path, const int din, void *dirInodeObj, int type)
{
	return -EOPNOTSUPP;
}

STATIC int chattr(const int din, void *inodeObj, uint mode, uint uid, uint gid, uint atime, uint mtime)
{
	return -EOPNOTSUPP;
}

STATIC int link(const int din, void *dstInodeObj, void *srcInodeObj, const char *srcPath)
{
	return -EOPNOTSUPP;
}

STATIC int symlink(const char *filePath, const int din, void *dirInodeObj, char *linkPath)
{
	return -EOPNOTSUPP;
}

STATIC int statfs(const int din, struct statfs *m_statfs)
{
	return -EOPNOTSUPP;
}


static FS devFsFunc = {
	DEV_FS_NAME,
	mount,
	umount,
	lookup,
	creat,
	open,
	close,
	read,
	write,
	ioctl,
	poll,
	opendir,
	readdir,
	rename,
	mkdir,
	stat,
	delete,
	chattr,
	link,
	symlink,
	statfs,
};


/*
 * å夫ǥХ˥ǡ᤹
 */
int sys_sync()
{
	writeBackAllCache();

	return 0;
}


/*
 * եǥץü̾롣
 */
int sys_ttyname(int fdnum, char *buf)
{
	F_DSC *fd;
	dev_t dev;
	int inodeNum;
	const char *tty;

	if ((fd = getFd(fdnum)) == NULL){
		return -EBADF;
	}

	// ǥХե뤫
	if (fd->vent->din != 0){
		return -ENOTTY;
	}

	// struct cdevswΥ饹ȥ֥åʥС0Υ饯ǥХǧ롣
	inodeNum = (int)fd->vent->inodeObj;
	dev = getDevInfo(inodeNum);
	if (dev->si_devsw->d_flags | D_TTY){
		tty = getName(inodeNum);
		strcpy(buf, tty);
	}
	else{
		return -ENOTTY;
	}

	return 0;
}


/************************************************************************************************
 *
 * 
 *
 ************************************************************************************************/


//================================== PUBLIC =============================================


int initDeviceFs()
{
	/* ǥХե륷ƥν */
	initDevFs();

	return 0;
}


int mountDeviceFs()
{
	int error;

	/* ե륷ƥϿ롣 */
	error = regist_fs(&devFsFunc);
	if (error < 0)
		return error;

	/* ǥХե륷ƥޥȤ롣 */
	error = mount_dev_fs(&devFsFunc);
	if (error != NOERR){
		return error;
	}

	return 0;
}


/****************************************************************************/
void test_device(int irq)
{
	if (irq != 0){
		printk("test_device() irq=%d\n", irq);
	}
}
/*****************************************************************************/
