/*
 * except_i386.c
 *
 * Copyright 2002, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 *
 * 㳰ϥɥ
 */



#include <sys/param.h>
#include <machine/except_i386.h>
#include <machine/segment.h>
#include <kern/lib.h>
#include <kern/proc.h>
#include <kern/interrupt.h>
#include <kern/signal.h>
#include <kern/mm.h>
#include <kern/lock.h>


/************************************************************************************************
 *
 * ǥХå
 *
 ************************************************************************************************/


//================================== PRIVATE ============================================


/*
 * ǥХå쥸ؤɤ߽񤭡
 */
static inline uint readDr0()
{
	uint rest;

	asm volatile(
		"movl	%%dr0,%%eax"
		:"=a"(rest):
	);

	return rest;
}

static inline void writeDr0(uint value)
{
	asm volatile(
		"movl	%%eax,%%dr0"
		::"a"(value)
	);
}

static inline void writeDr6(uint value)
{
	enum{DR6_RESERVE_BIT=0xffff0ff0,};	/* DR6ͽӥåȡ */

	value|=DR6_RESERVE_BIT;
	asm volatile(
		"movl	%%eax,%%dr6"
		::"a"(value)
	);
}

static inline void writeDr7(uint value)
{
	enum{DR7_RESERVE_BIT=0x400,};		/* DR7ͽӥåȡ */

	value|=DR7_RESERVE_BIT;
	asm volatile(
		"movl	%%eax,%%dr7"
		::"a"(value)
	);
}


//================================== PUBLIC =============================================


/*
 * ǥХåꡣ1ĤΥ֥졼ݥȤΤꡣ
 * parameters : ֥졼ݥȥ˥ɥ쥹,֥졼
 */
void setDebug(uint break_addr, int condition)
{
	enum{
		DR7_LEGE_BIT=0x300,			/* DR7 LEGEӥåȡ */
		DR7_LEN0_1BYTE=0x0,			/* DR7 LEN0 1ХĹ */
		DR7_LEN0_4BYTE=0xc0000,		/* DR7 LEN0 4ХĹ */
		DR7_G0_BIT=0x2,				/* DR7 DR0 global break enableӥåȡ */
	};

	cli();

	/* ֥졼ݥȤꡣ */
	writeDr0(break_addr);

	/* ꡣ */
	switch (condition){
		case DEBUG_EXEC:
			condition <<= 16;
			condition |= DR7_LEN0_1BYTE;
		case DEBUG_WRITE:
		case DEBUG_RW:
			condition <<= 16;
			condition |= DR7_LEN0_4BYTE;
		default:
			return;
	}
	condition |= DR7_LEGE_BIT | DR7_G0_BIT;
	writeDr7(condition);

	sti();
}


/************************************************************************************************
 *
 * ǥХԲ㳰
 *
 ************************************************************************************************/


typedef struct{
	PROC *proc;
	uint saveAddr;
}BEFORE_FPU_PROC;


//================================== PROTECTED ==========================================


// ǶΥǥХԲ㳰򵯤ץ
static BEFORE_FPU_PROC beforeFpuProc[MAX_CPU];


//================================== PUBLIC =============================================


// ǶΥǥХԲ㳰򵯤ץ򥯥ꥢ
void clearFpuBeforeProc()
{
	int cpu = get_current_cpu();
	int eflag;
	
	eflag = enterCli();
	if (get_current_task() == beforeFpuProc[cpu].proc){
		beforeFpuProc[cpu].proc = NULL;
	}
	exitCli(eflag);
}


// 桼ϥɥƤӽФFPU֥ɥ쥹ڤؤ
// ץ¤ΤˤFPU֥ե饰ڤؤ롣
void setFpuSaveAddr(uint saveAddr)
{
	int cpu = get_current_cpu();
	PROC *proc = get_current_task();
	int eflag;

	eflag = enterCli();
	{
		if (beforeFpuProc[cpu].proc == proc){
			beforeFpuProc[cpu].saveAddr = saveAddr;
		}
		proc->fpu.saveBefore = proc->fpu.save;
		proc->fpu.save = 0;
	}
	exitCli(eflag);
}


// 桼ϥɥƤӽФFPU֥ɥ쥹ȡ
// ץ¤ΤˤFPU֥ե饰ꥻåȤ롣
void resetFpuSaveAddr()
{
	int cpu = get_current_cpu();
	PROC *proc = get_current_task();
	int eflag;

	eflag = enterCli();
	{
		beforeFpuProc[cpu].saveAddr = KERNEL_ESP_BEG - FPU_SAVE_SIZE;
		proc->fpu.save = proc->fpu.saveBefore;
		proc->fpu.saveBefore = 0;
	}
	exitCli(eflag);
}


/************************************************************************************************
 *
 * 㳰ϥɥ
 *
 ************************************************************************************************/


//================================== PRIVATE ============================================


/*
 * 㳰ϥɥ顣
 * parameters : å,ʥ,㳰ե졼ॢɥ쥹
 */
static void exception(const char *message, int signal, INTERRUPT_FRAME *frame)
{
	printk("\nException! %s.\n",message);
	if ((frame->cs == KERNEL_CODE_DES) || (frame->cs == APM_CODE_DES)){
		uint *stack = &frame->user_esp;

		printk("error=0x%x eip=0x%x cs=0x%x ds=0x%x esp=0x%x proc=0x%x\n",
			frame->error,
			frame->eip,
			frame->cs,
			frame->ds,
			&frame->user_esp,
			get_current_task());
		printk("stack:0x%x 0x%x 0x%x 0x%x 0x%x 0x%x 0x%x 0x%x 0x%x\n",
			stack[0],
			stack[1],
			stack[2],
			stack[3],
			stack[4],
			stack[5],
			stack[6],
			stack[7],
			stack[8]); 
	}
	else{
		printk("error=0x%x eip=0x%x cs=0x%x esp=0x%x,ss=0x%x\n",
			frame->error,
			frame->eip,
			frame->cs,
			frame->user_esp,
			frame->user_ss);
	}
//**************************************
idle();
//**************************************
	forceSendSignal(get_current_task(), signal, TASK_SIGNAL_WAIT);
}


/*
 * 㳰ϥɥ
 * åѤޤ줿ͤɽƥɥ˥פ
 */
void tmp_except(const char *message, INTERRUPT_FRAME *frame)
{
	printk("\nException! %s.\n",message);
	if((frame->cs==KERNEL_CODE_DES)||(frame->cs==APM_CODE_DES)){
		printk("error=0x%x eip=0x%x cs=0x%x esp=0x%x proc=0x%x cpu=%d\n",
			frame->error,frame->eip,frame->cs,&frame->user_esp, get_current_task(), get_current_cpu());
	}
	else{
		printk("error=0x%x eip=0x%x cs=0x%x esp=0x%x,ss=0x%x\n",frame->error,frame->eip,frame->cs,frame->user_esp,frame->user_ss);
	}

	idle();
}


//================================== PUBLIC =============================================


// 顼
void excDivideError(INTERRUPT_FRAME frame)
{
	exception("Divide error",SIGFPE,&frame);
}


// ǥХå
void debug(INTERRUPT_FRAME frame)
{
	if(frame.cs==KERNEL_CODE_DES)
		printk("\nBreak point trap! : break point=0x%x,eip=0x%x,cs=0x%x,esp=0x%x,process=0x%x\n",
			readDr0(),frame.eip,frame.cs,&frame.user_esp,get_current_task());
	else
		printk("\nBreak point trap! : break point=0x%x,eip=0x%x,cs=0x%x,esp=0x%x,process=0x%x\n",
			readDr0(),frame.eip,frame.cs,frame.user_esp,get_current_task());

	/* ǥХå쥸Υꥻåȡ */
	writeDr7(0);
	writeDr6(0);

	idle();
}


// NMIߡ
void excNonmaskableInterrupt(INTERRUPT_FRAME frame)
{
	char value;


	printk("\nException! Nonmaskable interrupt.\n");

	value=inb(0x61);		/* System control portB. */
	if(value&0x80)
		printk("Memory parity error.\n");
	if(value&0x40)
		printk("IO channel parity error.\n");
	if((value&0xc0)==0)
		printk("Unknown NMI error.\n");

	/* Reset NMI interrupt. */
	outb(0x70,0x8f);
	inb(0x71);
	outb(0x70,0x0f);
	inb(0x71);
}


// ֥졼ݥȡ
void excBreakpoint(INTERRUPT_FRAME frame)
{
	exception("Breakpoint",SIGTRAP,&frame);
}


// Сե
void excOverflow(INTERRUPT_FRAME frame)
{
	exception("Overflow",SIGSEGV,&frame);
}


// BOUNDϰϳ
void excBoundsCheck(INTERRUPT_FRAME frame)
{
	exception("Bounds check",SIGSEGV,&frame);
}


// ̵ڥɡ
void excInvalidOpcode(INTERRUPT_FRAME frame)
{
	exception("Invalid opcode",SIGILL,&frame);
}


// ǥХԲ㳰
void excCoprocessorNotAvailable(INTERRUPT_FRAME frame)
{
	int cpu = get_current_cpu();
	PROC *proc = get_current_task();

	cli();
	{
		// TSե饰򥯥ꥢ
		clearTs();

		// ľFPUѥץΥΰfpu쥸򥻡֤롣
		if (beforeFpuProc[cpu].proc != NULL){
			if ((beforeFpuProc[cpu].proc != proc) || (beforeFpuProc[cpu].saveAddr != KERNEL_ESP_BEG - FPU_SAVE_SIZE)){
				uint saveAddr = getPhysicalFromLogicalAddr(beforeFpuProc[cpu].proc, beforeFpuProc[cpu].saveAddr);
				fpuSave(saveAddr);
				if (beforeFpuProc[cpu].saveAddr == KERNEL_ESP_BEG - FPU_SAVE_SIZE){
					beforeFpuProc[cpu].proc->fpu.save = 1;
				}
				else{
					// 桼ϥɥ¹FTP֥ե饰ꤹ롣
					beforeFpuProc[cpu].proc->fpu.saveBefore = 1;
				}
			}
			else{
				beforeFpuProc[cpu].proc->fpu.save = 0;
			}
		}

		beforeFpuProc[cpu].proc = proc;
		beforeFpuProc[cpu].saveAddr = KERNEL_ESP_BEG - FPU_SAVE_SIZE;
	}
	sti();

	// fpu쥸Υ
	if (proc->fpu.save){
		fpuStore(KERNEL_ESP_BEG - FPU_SAVE_SIZE);
		proc->fpu.save = 0;
	}
}


// ֥եȡ
void excDoubleFault(INTERRUPT_FRAME frame)
{
	exception("Double fault",SIGSEGV,&frame);
}


// ץåС
void excCopressorOverrun(INTERRUPT_FRAME frame)
{
	exception("Copressor segment overrun",SIGFPE,&frame);
}


// ̵TSS
void excInvalidTss(INTERRUPT_FRAME frame)
{
	exception("Invalid TSS",SIGSEGV,&frame);
}


// Ժߡ
void excSegmentNotPresent(INTERRUPT_FRAME frame)
{
	exception("Segment not present",SIGBUS,&frame);
}


// åȥեȡ
void excStackException(INTERRUPT_FRAME frame)
{
	exception("Stack exception",SIGBUS,&frame);
}


// ݸ
void excGeneralProtection(INTERRUPT_FRAME frame)
{
	exception("General protection",SIGSEGV,&frame);
}


// FPUư顼
void excFpuError(INTERRUPT_FRAME frame)
{
	exception("FPU error",SIGFPE,&frame);
}


// 饤ȥå
void excAlignmentCheck(INTERRUPT_FRAME frame)
{
	exception("Alignment check",SIGBUS,&frame);
}


// ޥå
void excMachineCheck(INTERRUPT_FRAME frame)
{
	/* ̤ */
	tmp_except("Machine check", &frame);
}


// SIMDư㳰
void excSimdFpuError(INTERRUPT_FRAME frame)
{
	exception("SIMD FPU error",SIGFPE,&frame);
}


// ͽ㳰
void excReserve(INTERRUPT_FRAME frame)
{
	tmp_except("15", &frame);
}


/************************************************************************************************
 *
 * ؿ
 *
 ************************************************************************************************/


//================================== PUBLIC =============================================


void init_except()
{
	set_idt(0,except0,IDT_TRAP);
	set_idt(1,except1,IDT_INTR);
	set_idt(2,except2,IDT_TRAP);
	set_idt(3,except3,IDT_INTR);
	set_idt(4,except4,IDT_TRAP);
	set_idt(5,except5,IDT_TRAP);
	set_idt(6,except6,IDT_TRAP);
	set_idt(7,except7,IDT_TRAP);
	set_idt(8,except8,IDT_TRAP);
	set_idt(9,except9,IDT_TRAP);
	set_idt(10,except10,IDT_TRAP);
	set_idt(11,except11,IDT_TRAP);
	set_idt(12,except12,IDT_TRAP);
	set_idt(13,except13,IDT_TRAP);
	set_idt(14,except14,IDT_INTR);
	set_idt(15,except15,IDT_TRAP);
	set_idt(16,except16,IDT_TRAP);
	set_idt(17,except17,IDT_TRAP);
	set_idt(18,except18,IDT_TRAP);
}
