/*
 * apm.c
 *
 * Copyright 2004, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 */


#include <sys/types.h>
#include <kern/lib.h>
#include <machine/segment.h>
#include <machine/apm/apm.h>


enum{
	/* Power state flag. */
	APM_PWRFLG_ENABL	= 0x0,	/* APM Enabled. */
	APM_PWRFLG_STANDBY	= 0x1,	/* Standby. */
	APM_PWRFLG_SUSPEND	= 0x2,	/* Suspend. */
	APM_PWRFLG_OFF		= 0x3,	/* Power off. */
};


typedef struct{
	uint eax;
	uint ebx;
	uint ecx;
	uint edx;
	uint esi;
	uint edi;
}APM_FUNC_ARGS;


volatile FARCALL_ADDR apmfarCall;	/* Indirect address for APM far call. */
extern ushort apmVersion;
extern ushort apmFlag;
extern uint apmCodeSegBase;		/* Real mode segment base address. */
extern uint apmCode16SegBase;	/* Real mode segment base address. */
extern uint apmDataSegBase;		/* Real mode segment base address. */
extern uint apmOffset;
extern uint apmCodeSegLen;
extern uint apmCode16SegLen;
extern uint apmDataSegLen;


extern int invokeApmFunction(APM_FUNC_ARGS *args);


//================================== PUBLIC =============================================


/*
 * Get APM Driver Version.
 * return : driver version(BCD) or -1
 */
int getApmDriverVersion()
{
	APM_FUNC_ARGS args;
	int error;

	if ((apmFlag & APM_FLG_32MODE) == 0){
		return APM_ERR_NOTPRESEN;
	}

	args.eax = APM_FUNC_DRVVRS & 0xff;
	args.ebx = 0;				/* APM BIOS device number. */
	args.ecx = apmVersion;
	error = invokeApmFunction(&args);
	if (error == 0){
		return args.eax;
	}
	else{
		return -1;
	}
}


// Set power state.
void setPowerOff()
{
	APM_FUNC_ARGS args;
	int error;

	if ((apmFlag & APM_FLG_32MODE) == 0){
		return;
	}

	args.eax = APM_FUNC_SETPWR & 0xff;
	args.ebx = APM_DEV_ALLDEV;
	args.ecx = APM_PWRFLG_OFF;
	args.edx = 0;
	error = invokeApmFunction(&args);
/****************************************************************************************************************/
printk("setPowerState() 0x%x\n", error);
/****************************************************************************************************************/
}


/*
 * 
 */
void InitApm()
{
	if (apmFlag == 0){		/* APM not supported. */
		return;
	}

	/* APM version. */
	printk("APM : BIOS version %d.%d\n", apmVersion >> 8, apmVersion & 0xff);

	if ((apmFlag & APM_FLG_32MODE) == 0){	/* 32-bit protected mode interface not supported. */
		return;
	}

	if(apmOffset == 0){
		printk("APM 32bit mode interface connection error! error coad : %x\n", apmCodeSegBase >> 8);
	}

	/* APMͥѥȤꡣ */
	set_gdt(APM_CODE_DES, apmCodeSegBase << 4, apmCodeSegLen, TYPE_KERNEL_CODE);			/* Code segment. */
	set_gdt(APM_CODE16_DES, apmCode16SegBase << 4, apmCode16SegLen, TYPE_KERNEL_CODE16);	/* 16bit code segment. */
	set_gdt(APM_DATA_DES, apmDataSegBase << 4, apmDataSegLen, TYPE_KERNEL_DATA);			/* Data segment. */
	apmfarCall.des = APM_CODE_DES;
	apmfarCall.addr = apmOffset;
}
