/*
 * keyboard.c
 *
 * Copyright 2007, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 *
 * L[{[h͏
 */


#include <sys/config.h>
#include <sys/types.h>
#include <sys/param.h>
#include <kern/interrupt.h>
#include <dev/console/console.h>
#include <dev/console/keymap.h>
#include <dev/console/keyboard.h>
#include <kern/test.h>
#include <kern/debug.h>


//#define DEBUG_KEYBOARD 1
#ifdef DEBUG_KEYBOARD
	#define STATIC
	#define INLINE
#else
	#define STATIC	static
	#define INLINE	inline
#endif


/******************************************************************************************************
 *
 * L[{[h
 *
 *******************************************************************************************************/

//================================== PRIVATE ============================================

enum{
	IRQ_NO = 1,		/* IRQ number */
};

/*
 * Turn on/off LED.
 */
STATIC INLINE void switchLed(uchar value)
{
	outb(KEYBOARD_OUTPUT_REG, 0xed);		/* Command turn on/of LED. */
	while(inb(KEYBOARD_STAT_REG) & 2);
	outb(KEYBOARD_OUTPUT_REG, value);		/* LED on/off. */
}

/*
 * L[{[hnh[
 * return : YES = [U[ǂݍ݉\ɂȂ or NO = [U[͓ǂݍ݂łȂ
 */
STATIC int keyboardHnadler(
	uchar key_coad)			// scan key coad
{
	enum{
		KEY_BREAK = 0x80,		// key break bit.

		// Add key bit.
		SHT_BIT = 1 << 0,
		CTR_BIT = 1 << 1,

		// LED bit.
		LED_SCRL = 1 << 0,	/* ScrollLock LED. */
		LED_NMLC = 1 << 1,	/* NumLock LED. */
		LED_CPSL = 1 << 2,	/* CapsLock LED. */
	};
	static int add_key = 0;			/* normal = 0,add shift = 1,add ctrl = 2. */
	static int func_key = 0;		/* @\L[tOB(XLR[h0xe0) */
	static int shift_key = 0;		/* Shift key flag. */
	static int led = 0;				/* LED flag, */
	static int ctr_key = 0;			/* Ctrl key. */
	static uchar add_map[] = {
		0,1,2,2};					/* tL[ϊ}bvB */
	uchar map_coad;

	/* @\L[̏B */
	if (func_key) {
		if ((VMAP_START_COAD <= key_coad) && (key_coad <= VMAP_END_COAD)) {
			key_coad += VMAP_ADD;
		}
		func_key = 0;
	}
	else if (key_coad == KEY_FUNC) {
		func_key = 1;
		return NO;
	}

	/* }bvR[h擾B */
	map_coad = keymap[key_coad & ~KEY_BREAK][add_key];
	if (map_coad == 0) {
		return NO;
	}

	/* u[NR[ȟACtrl ShiftȊÔ͎ĂB */
	if (key_coad & KEY_BREAK) {
		switch (map_coad) {
		case MAP_SHT:
			shift_key = 0;
			break;
		case MAP_CTR:
			ctr_key = 0;
			break;
		default:
			return NO;
		}
		add_key = add_map[(shift_key | ctr_key) ^ (led >> 2)];

		return NO;
	}

	/* t@NVL[̏ */
	switch (map_coad) {
	case MAP_SHT:
		shift_key = SHT_BIT;
		add_key = add_map[(shift_key | ctr_key) ^ (led >> 2)];
		return NO;
	case MAP_CPSL:
		switchLed(led ^= LED_CPSL);
		add_key=add_map[(shift_key | ctr_key) ^ (led >> 2)];
		return NO;
	case MAP_CTR:
		ctr_key = CTR_BIT;
		add_key = add_map[(shift_key | ctr_key) ^ (led >> 2)];
		return NO;
	}
	
	return sendConsole(map_coad);
}

/*
 * 荞݃nh[
 * return : task switch on = 1 off = 0
 */
STATIC int keyboardIntr(int irq)
{
	if (keyboardHnadler(inb(KEYBOARD_OUTPUT_REG)) == YES) {
		return 1;
	}

	return 0;
}

//================================== PUBLIC =============================================

/*
 * L[{[hhCo[̏
 */
void initKeyboard()
{
	/* Turn off LED. */
	switchLed(0);

	/* 荞݂̐ݒ */
	irq_entry[IRQ1] = keyboardIntr;

	release_irq_mask(IRQ_NO);

	/* clear data register */
	inb(KEYBOARD_OUTPUT_REG);
}
