/*
 * Copyright (C) 2010-2011 Mtzky.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *         http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.mtzky.reflect;

import static org.mtzky.reflect.IterableUtils.*;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;

import org.mtzky.reflect.IterableUtils.Find;
import org.mtzky.reflect.IterableUtils.Wrap;

/**
 * <p>
 * Descriptors for {@link PropDesc properties} ({@link Method Getter,
 * Setter}, and <strong>public</strong> {@link Field}).
 * </p>
 * 
 * @author mtzky
 * @since 0.1.1
 */
public class PropsDesc {

	private final PropDesc[] descs;

	/**
	 * <p>
	 * Constructs {@link PropsDesc} filtered the specified {@link Annotation}.
	 * </p>
	 * 
	 * @param clazz
	 * @param annotationClass
	 */
	public PropsDesc(final Class<?> clazz,
			final Class<? extends Annotation> annotationClass) {
		final Map<String, PropDesc> descs = new HashMap<String, PropDesc>();
		for (Class<?> c = clazz; (c != Object.class && c != null); c = c
				.getSuperclass()) {
			/* getter/setter methods */
			for (final Method m : c.getMethods()) {
				if (m.getAnnotation(annotationClass) == null) {
					continue;
				}
				final PropDesc desc = new PropDesc(m);
				descs.put(desc.getName(), desc);
			}
			/* public fields */
			for (final Field f : c.getFields()) {
				if (f.getAnnotation(annotationClass) == null) {
					continue;
				}
				final String name = f.getName();
				if (descs.containsKey(name)) {
					continue;
				}
				descs.put(name, new PropDesc(name, f));
			}
		}
		this.descs = descs.values().toArray(new PropDesc[descs.size()]);
		descs.clear();
	}

	public PropDesc[] getDescriptors() {
		return descs;
	}

	/**
	 * @param <T>
	 * @param cb
	 *            {@link WrapCallback#call(Object)}
	 * @param type
	 *            for getting type; Don't pass {@code null}.
	 * @return An array of instance wrapping {@link PropDesc}
	 * @since 0.1.2
	 */
	public <T> T[] wrap(final Wrap<PropDesc, T> cb, final T... type) {
		return each(descs, cb, type);
	}

	/**
	 * @author mtzky
	 * @since 0.1.2
	 */
	public interface WrapCallback<T> extends Wrap<PropDesc, T> {
	}

	/**
	 * <p>
	 * Find the {@link Find#call(Object) matched} {@link PropDesc property
	 * descriptor}.
	 * </p>
	 * 
	 * @param cb
	 *            {@link Find#call(Object)}
	 * @return {@link PropDesc}
	 * @see FindCallback
	 */
	public PropDesc find(final Find<PropDesc> cb) {
		return each(descs, cb);
	}

	/**
	 * @author mtzky
	 */
	public interface FindCallback extends Find<PropDesc> {
	}

}
