/*
 * Copyright (C) 2010-2011 Mtzky.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *         http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.mtzky.reflect;

import static java.util.Collections.*;
import static javax.tools.JavaFileObject.Kind.*;
import static javax.tools.StandardLocation.*;
import static javax.tools.ToolProvider.*;
import static org.mtzky.io.IOUtils.*;
import static org.mtzky.reflect.IterableUtils.*;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.Set;

import javax.tools.DiagnosticCollector;
import javax.tools.JavaFileManager;
import javax.tools.JavaFileObject;
import javax.tools.JavaFileObject.Kind;

import org.mtzky.reflect.IterableUtils.Find;

/**
 * <p>
 * Descriptors for {@link Class} in a {@code package}.
 * </p>
 * 
 * @author mtzky
 * @since 0.1.6
 */
public class PackageDesc {

	private static final Set<Kind> KINDS = EnumSet.of(CLASS);

	private static final Find<Class<?>> ALL = new Find<Class<?>>() {
		@Override
		public boolean call(final Class<?> value) {
			return true;
		}
	};

	private final List<Class<?>> classes;

	/**
	 * @param packageName
	 *            a package name
	 * @throws IOException
	 *             if an I/O error occurred
	 * @throws ClassNotFoundException
	 *             if the class cannot be located
	 */
	public PackageDesc(final String packageName) throws IOException,
			ClassNotFoundException {
		this(packageName, ALL);
	}

	/**
	 * @param packageName
	 *            a package name
	 * @param filter
	 * @throws IOException
	 *             if an I/O error occurred
	 * @throws ClassNotFoundException
	 *             if the class cannot be located
	 */
	public PackageDesc(final String packageName, final Find<Class<?>> filter)
			throws IOException, ClassNotFoundException {
		if (packageName == null) {
			throw new NullPointerException("packageName");
		}
		if (filter == null) {
			throw new NullPointerException("filter");
		}
		final String p = packageName.endsWith(".") ? packageName
				: (packageName + '.');
		JavaFileManager manager = null;
		try {
			final List<Class<?>> classes = new ArrayList<Class<?>>();
			manager = getSystemJavaCompiler().getStandardFileManager(
					new DiagnosticCollector<JavaFileObject>(), null, null);
			for (final JavaFileObject obj : manager.list(CLASS_PATH,
					packageName, KINDS, false)) {
				final String n = obj.getName();
				final Class<?> c = Class.forName(p
						+ n.substring(0, n.lastIndexOf('.')));
				if (filter.call(c)) {
					classes.add(c);
				}
			}
			this.classes = unmodifiableList(classes);
		} finally {
			closeQuietly(manager);
		}
	}

	/**
	 * @return classes in this {@code package} as an
	 *         {@link Collections#unmodifiableList(List) unmodifiable list}
	 */
	public List<Class<?>> getClasses() {
		return classes;
	}

	/**
	 * @param <T>
	 * @param callback
	 * @return a found class in this {@code package}, or {@code null} if not
	 */
	@SuppressWarnings("unchecked")
	public <T> Class<T> findClass(final Find<Class<?>> callback) {
		return (Class<T>) each(classes, callback);
	}

}
