/*
 * Copyright (C) 2010-2011 Mtzky.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *         http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.mtzky.reflect;

import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.List;

import org.mtzky.UtilConstractedAssertionError;

/**
 * <p>
 * {@link Iterable} and array utilities. It is useful to loop in the one-liner
 * statement.
 * </p>
 * 
 * @author mtzky
 * @since 0.1.4
 */
public class IterableUtils {

	private IterableUtils() {
		throw new UtilConstractedAssertionError(IterableUtils.class);
	}

	/**
	 * @param <E>
	 *            An array type
	 * @param a
	 *            An array
	 * @param cb
	 *            {@link Each#call(Object)}
	 * @return The passed array {@code a} itself
	 */
	public static <E> E[] each(final E[] a, final Each<E> cb) {
		for (final E v : a) {
			cb.call(v);
		}
		return a;
	}

	/**
	 * @param <E>
	 *            A {@link Iterable} element type
	 * @param <IT>
	 *            A {@link Iterable} object type
	 * @param it
	 *            A {@link Iterable} object
	 * @param cb
	 *            {@link Each#call(Object)}
	 * @return The passed {@link Iterable} object {@code a} itself
	 */
	public static <E, IT extends Iterable<E>> IT each(final IT it,
			final Each<E> cb) {
		for (final E v : it) {
			cb.call(v);
		}
		return it;
	}

	/**
	 * @param <E>
	 *            passed type
	 * @see IterableUtils
	 * @author mtzky
	 */
	public static interface Each<E> {
		void call(E value);
	}

	/**
	 * @param <E>
	 *            An array type
	 * @param a
	 *            An array
	 * @param cb
	 *            {@link EachWithIndex#call(Object, int)}
	 * @return The passed array {@code a} itself
	 */
	public static <E> E[] each(final E[] a, final EachWithIndex<E> cb) {
		for (int i = 0, len = a.length; i < len; i++) {
			cb.call(a[i], i);
		}
		return a;
	}

	/**
	 * @param <E>
	 *            A {@link Iterable} element type
	 * @param <IT>
	 *            A {@link Iterable} object type
	 * @param it
	 *            A {@link Iterable} object
	 * @param cb
	 *            {@link EachWithIndex#call(Object, int)}
	 * @return The passed {@link Iterable} object {@code a} itself
	 */
	public static <E, IT extends Iterable<E>> IT each(final IT it,
			final EachWithIndex<E> cb) {
		int i = 0;
		for (final E v : it) {
			cb.call(v, i++);
		}
		return it;
	}

	/**
	 * @param <E>
	 *            passed type
	 * @see IterableUtils
	 * @author mtzky
	 */
	public static interface EachWithIndex<E> {
		void call(E value, int index);
	}

	/**
	 * @param <E>
	 *            An array type
	 * @param a
	 *            An array
	 * @param cb
	 *            {@link Find#call(Object)}
	 * @return found element, or null if not found
	 */
	public static <E> E each(final E[] a, final Find<E> cb) {
		for (final E v : a) {
			if (cb.call(v)) {
				return v;
			}
		}
		return null;
	}

	/**
	 * @param <E>
	 *            A {@link Iterable} element type
	 * @param it
	 *            A {@link Iterable} object
	 * @param cb
	 *            {@link Find#call(Object)}
	 * @return found element, or null if not found
	 */
	public static <E> E each(final Iterable<E> it, final Find<E> cb) {
		for (final E v : it) {
			if (cb.call(v)) {
				return v;
			}
		}
		return null;
	}

	/**
	 * @param <E>
	 *            passed type
	 * @see IterableUtils
	 * @author mtzky
	 */
	public static interface Find<E> {
		boolean call(E value);
	}

	/**
	 * @param <E>
	 *            An array type
	 * @param a
	 *            An array
	 * @param cb
	 *            {@link Find#call(Object)}
	 * @return {@code true} if found
	 */
	public static <E> boolean each(final E[] a, final Contain<E> cb) {
		for (final E v : a) {
			if (cb.call(v)) {
				return true;
			}
		}
		return false;
	}

	/**
	 * @param <E>
	 *            A {@link Iterable} element type
	 * @param it
	 *            A {@link Iterable} object
	 * @param cb
	 *            {@link Find#call(Object)}
	 * @return {@code true} if found
	 */
	public static <E> boolean each(final Iterable<E> it, final Contain<E> cb) {
		for (final E v : it) {
			if (cb.call(v)) {
				return true;
			}
		}
		return false;
	}

	/**
	 * @param <E>
	 *            passed type
	 * @see IterableUtils
	 * @author mtzky
	 */
	public static interface Contain<E> {
		boolean call(E value);
	}

	/**
	 * @param <FROM>
	 *            The array type passed
	 * @param <TO>
	 *            The array type returned
	 * @param a
	 *            An array
	 * @param cb
	 *            {@link Wrap#call(Object)}
	 * @param type
	 *            to get a returned type; Don't pass {@code null}.
	 * @return A wrapping array
	 */
	@SuppressWarnings("unchecked")
	public static <FROM, TO> TO[] each(final FROM[] a, final Wrap<FROM, TO> cb,
			final TO... type) {
		if (type == null) {
			throw new NullPointerException("type");
		}
		final Class<TO> t = (Class<TO>) type.getClass().getComponentType();
		final int len = a.length;
		final TO[] wrapped = (TO[]) Array.newInstance(t, len);
		for (int i = 0; i < len; i++) {
			wrapped[i] = cb.call(a[i]);
		}
		return wrapped;
	}

	/**
	 * @param <FROM>
	 *            The {@link Iterable} element type passed
	 * @param <TO>
	 *            The {@link Iterable} element type returned
	 * @param it
	 *            A {@link Iterable} object
	 * @param cb
	 *            {@link Wrap#call(Object)}
	 * @throws IllegalAccessException
	 * @throws InstantiationException
	 */
	@SuppressWarnings("unchecked")
	public static <FROM, TO> TO[] each(final Iterable<FROM> it,
			final Wrap<FROM, TO> cb, final TO... type)
			throws InstantiationException, IllegalAccessException {
		if (type == null) {
			throw new NullPointerException("type");
		}
		final List<TO> wrapped = new ArrayList<TO>();
		for (final FROM v : it) {
			wrapped.add(cb.call(v));
		}
		final Class<TO> t = (Class<TO>) type.getClass().getComponentType();
		return wrapped.toArray((TO[]) Array.newInstance(t, wrapped.size()));
	}

	/**
	 * @param <FROM>
	 *            passed type
	 * @param <TO>
	 *            returned type
	 * @see IterableUtils
	 * @author mtzky
	 */
	public static interface Wrap<FROM, TO> {
		TO call(FROM value);
	}

}
