<?php
/**
 * mod_auth_hmac ticket generator
 * 
 * PHP version 4 and 5
 * 
 * 
 * Copyright 2007 Takayuki Ogiso
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 *
 * @author    Takayuki Ogiso <tatami@users.sourceforge.jp>
 * @copyright 2007 Takayuki Ogiso
 * @license   http://www.apache.org/licenses/LICENSE-2.0  Apache License 2.0
  * @link       http://sourceforge.jp/projects/modauthhmac/
 */

/**
 * mod_auth_hmac ticket generator class
 *
 * @author    Takayuki Ogiso <tatami@users.sourceforge.jp>
 * @copyright 2007 Takayuki Ogiso
 * @license   http://www.apache.org/licenses/LICENSE-2.0  Apache License 2.0
 * @link       http://sourceforge.jp/projects/modauthhmac/
 */
class AuthHmac
{
    /**
     * Generate and return a AuthHmac ticket
     * 
     * @param string $key HMAC key (BASE64 encoded)
     * @param int $expire Expiration (UNIX timestamp)
     * @return string AuthHmac ticket
     */
    function getTicket($key, $expire)
    {
        // RFC 822 format
        $expireString = gmdate('D, d M Y H:i:s', $expire) . ' GMT';
        
        $hmac = $this->hmacSha1($expireString, base64_decode($key));
        $authHmac = $expireString . "\t" . $hmac;
        
        return $authHmac;
    }
    
    /**
     * Generate a HMAC SHA1 digest
     * 
     * @param string $data Message
     * @param string $key Secret key
     * @return string HMAC, lower-case hexadecimal
     */
    function hmacSha1($data, $key)
    {
        $blockSize = 64;
        
        if (strlen($key) > $blockSize) {
            $key = pack('H40', sha1($key));
        }
        $keyArray = preg_split('//', $key, -1, PREG_SPLIT_NO_EMPTY);
        
        $ipad = array_pad($keyArray, $blockSize, "\x00");
        $opad = $ipad;
        
        for ($i = 0; $i < $blockSize; $i++) {
            $ipad[$i] ^= "\x36";
            $opad[$i] ^= "\x5c";
        }
        
        $innerHash = pack('H40', sha1(implode('', $ipad) . $data));
        $hmac = sha1(implode('', $opad) . $innerHash);
        
        return $hmac;
    }
}

?>
