/******************************************************************************
 * Copyright (C) 2006 Tetsuya Kimata <kimata@acapulco.dyndns.org>
 *
 * All rights reserved.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any
 * damages arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any
 * purpose, including commercial applications, and to alter it and
 * redistribute it freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must
 *    not claim that you wrote the original software. If you use this
 *    software in a product, an acknowledgment in the product
 *    documentation would be appreciated but is not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must
 *    not be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 * $Id: ReadLocker.h 2459 2007-06-29 16:38:34Z svn $
 *****************************************************************************/

#ifndef READ_LOCKER_H
#define READ_LOCKER_H

#include "Environment.h"

#include "ReadWriteLocker.h"

#ifdef DEBUG
#include <iostream>
#endif


/**
 * @brief 読み出しロックを表すクラス．
 */
class ReadLocker: public ReadWriteLocker
{
public:
    /**
     * コンストラクタです．
     *
     * @param[out] lock ロック変数
     */
    ReadLocker(apr_atomic_t *lock)
      : ReadWriteLocker(lock)
    {
        apr_size_t try_count;
        apr_uint32_t status;
        apr_uint32_t new_status;

        try_count = 0;
        while (1) {
            try_count++;

            status = apr_atomic_read(lock);
            if (LIKELY(get_read_count(status) != WRITE_LOCKED)) {
                new_status = make_status(get_age(status),
                                         get_read_count(status) + 1);
                if (apr_atomic_cas(lock_, new_status, status) == status) {
                    status_ = new_status;
                    return;
                }
            } else if (((try_count & TIMEOUT_CHECK_MASK) == 0) &&
                       is_timeout(status)) {
                // タイムアウトしたら age を増やす
                new_status = make_status(get_age(status) + 1, 1);
                if (apr_atomic_cas(lock_, new_status, status) == status) {
                    status_ = new_status;
                    return;
                }
            }

            yield();
        }
    };
    ~ReadLocker()
    {
        (void)apr_atomic_cas(lock_, make_status(get_age(status_),
                                                get_read_count(status_) - 1),
                             status_);
    };
    static bool is_locked(apr_atomic_t *lock) {
        return get_read_count(apr_atomic_read(lock)) != NOT_LOCKED;
    };
};

#endif

// Local Variables:
// mode: c++
// coding: utf-8-dos
// End:
