/******************************************************************************
 * Copyright (C) 2006 Tetsuya Kimata <kimata@acapulco.dyndns.org>
 *
 * All rights reserved.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any
 * damages arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any
 * purpose, including commercial applications, and to alter it and
 * redistribute it freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must
 *    not claim that you wrote the original software. If you use this
 *    software in a product, an acknowledgment in the product
 *    documentation would be appreciated but is not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must
 *    not be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 * $Id: ReadWriteLocker.h 2381 2007-02-05 16:19:16Z svn $
 *****************************************************************************/

#ifndef READ_WRITE_LOCKER_H
#define READ_WRITE_LOCKER_H

#include "Environment.h"

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_SCHED_H
#include <sched.h>
#endif

#include "AtomicWrapper.h"
#include "apr_thread_proc.h"

#include "Uncopyable.h"

/**
 * @brief 読み書きロックの設定を集めたクラス．
 */
class ReadWriteLocker: public Uncopyable
{
public:
    static const apr_uint32_t NOT_LOCKED        = 0x000000;

protected:
    /**
     * コンストラクタです．
     *
     * @param[out] read_count 読み出し中のスレッドの数
     */
    ReadWriteLocker(apr_atomic_t *read_count)
      : read_count_(read_count)
    {

    };

    ReadWriteLocker(const ReadWriteLocker& locker)
      : read_count_(locker.read_count_)
    {

    };

    ReadWriteLocker();

    /**
     * プロセッサーを明け渡します．
     */
    void yield()
    {
#if APR_HAS_THREADS
        // APR 0.x だと UNIX 環境の apr_thread_yield の中身は空なので
#if (APR_MAJOR_VERSION == 0) && defined( _POSIX_PRIORITY_SCHEDULING)
        sched_yield();
#else
        apr_thread_yield();
#endif
#endif
    };

    // 注意: ReadWriteLocker 型のポインタ経由で使用されることはないので，
    // デストラクタは virtual にしない．

    // apr_atomic_t に 24bit の制限があるので (APR 0.9.*)
    static const apr_uint32_t WRITE_LOCKED      = 0xFFFFFF;

    /** 読み出し中のスレッドの数へのポインタ */
    apr_atomic_t *read_count_;
};

#endif

// Local Variables:
// mode: c++
// coding: utf-8-dos
// End:
