/******************************************************************************
 * mod_uploader / RFC1867Parser.cpp
 ******************************************************************************
 * Copyright (C) 2004 Tetsuya Kimata <kimata@acapulco.dyndns.org>
 *
 * All rights reserved.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any
 * damages arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any
 * purpose, including commercial applications, and to alter it and
 * redistribute it freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must
 *    not claim that you wrote the original software. If you use this
 *    software in a product, an acknowledgment in the product
 *    documentation would be appreciated but is not bcktuired.
 *
 * 2. Altered source versions must be plainly marked as such, and must
 *    not be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 * $Id: ThumbnailWriter.cpp 704 2005-09-19 19:33:13Z svn $
 *****************************************************************************/

#include "ThumbnailWriter.h"

#ifdef MAKE_THUMBNAIL
#include "UploadItem.h"
#include "Auxiliary.h"
#include "Misc.h"

#include "apr_file_io.h"
#include "apr_strings.h"

#ifdef HAVE_FFMPEG
#include <ffmpeg/avio.h>
#include <ffmpeg/avcodec.h>
#endif

#ifdef WIN32
#undef HAVE_INTTYPES_H
#undef HAVE_STDINT_H
#endif
#undef PACKAGE_BUGREPORT
#undef PACKAGE_NAME
#undef PACKAGE_STRING
#undef PACKAGE_TARNAME
#undef PACKAGE_VERSION
#include <Magick++.h>
#undef PACKAGE_BUGREPORT
#undef PACKAGE_NAME
#undef PACKAGE_STRING
#undef PACKAGE_TARNAME
#undef PACKAGE_VERSION

#ifdef DEBUG
#include <iostream>
#endif

const char ThumbnailWriter::FILE_SUFFIX[]               = "." THUMBNAIL_FILE_TYPE;
const apr_size_t ThumbnailWriter::MAGICK_SIZE_LIMIT     = THUMBNAIL_MAGICK_SIZE_LIMIT;
const apr_size_t ThumbnailWriter::WIDTH_LIMIT           = THUMBNAIL_WIDTH_LIMIT;
const apr_size_t ThumbnailWriter::HEIGHT_LIMIT          = THUMBNAIL_HEIGHT_LIMIT;
const apr_size_t ThumbnailWriter::BUFFER_SIZE           = THUMBNAIL_BUFFER_SIZE;
const double ThumbnailWriter::FRAME_SAMPLE_SEC          = THUMBNAIL_FRAME_SAMPLE_SEC;
const double ThumbnailWriter::FRAME_DELAY_SEC           = THUMBNAIL_FRAME_DELAY_SEC;
const apr_size_t ThumbnailWriter::FRAME_NUMBER          = THUMBNAIL_FRAME_NUMBER;
#ifdef HAVE_FFMPEG
const PixelFormat ThumbnailWriter::FRAME_PIXEL_FORMAT   = PIX_FMT_RGB24;
#endif


/******************************************************************************
 * public ᥽å
 *****************************************************************************/
ThumbnailWriter::ThumbnailWriter(apr_pool_t *pool, const char *file_dir,
                                 const char *thumb_dir)
    : pool_(pool),
      file_dir_(file_dir),
      thumb_dir_(thumb_dir)
{

}

void ThumbnailWriter::write(const char *file_name)
{
    apr_pool_t *pool;

    if (apr_pool_create(&pool, pool_) != APR_SUCCESS) {
        throw "γݤ˼Ԥޤ";
    }

    try {
#ifdef HAVE_FFMPEG
        if (!create_movie_thumb(pool, file_name)) {
#endif
            create_image_thumb(pool, file_name);
#ifdef HAVE_FFMPEG
        }
#endif
        apr_pool_destroy(pool);
    } catch(const char *) {
        apr_pool_destroy(pool);
        throw;
    }
}

const char *ThumbnailWriter::create_thumb_path(apr_pool_t *pool,
                                               const char *thumb_dir,
                                               const char *file_path)
{
    return apr_pstrcat(pool,
                       thumb_dir, "/", file_path, FILE_SUFFIX, NULL);
}


/******************************************************************************
 * private ᥽å
 *****************************************************************************/
bool ThumbnailWriter::create_image_thumb(apr_pool_t *pool, const char *file_name)
{

    try {
        apr_size_t read_size;
        void *file_buffer;

        if (!open_image_file(pool, file_name, &file_buffer, &read_size)) {
            return false;
        }

        Magick::Blob data(file_buffer, read_size);
        Magick::Image image(data);
        Magick::Geometry thumb_size = image.size();

        if ((image.size().width() == 0) || (image.size().height() == 0)) {
            return false;
        }

        adjust_size(thumb_size);

        image.scale(thumb_size);
        image.write(create_thumb_path(pool, thumb_dir_, file_name));

        return true;
    } catch(Magick::Exception &e) {
        throw e.what();
    } catch(exception &e) {
        throw e.what();
    }
}

bool ThumbnailWriter::open_image_file(apr_pool_t *pool, const char *file_name,
                                      void **file_buffer, apr_size_t *size)
{
    UploadItem::header *header;
    apr_file_t *file;
    apr_size_t read_size;

    UploadItemReader reader(pool, file_dir_);
    header = reader.read(file_name, &file);

    if (header->file_size > MAGICK_SIZE_LIMIT) {
        return false;
    }

    APR_PALLOC(*file_buffer, void *, pool, header->file_size);

    if ((apr_file_read_full(file, *file_buffer, header->file_size, &read_size) != APR_SUCCESS) ||
        (read_size != header->file_size)) {
        throw "åץɥեɤ߹ޤǤ";
    }
    apr_file_close(file);

    *size = header->file_size;

    return true;
}

#ifdef HAVE_FFMPEG
bool ThumbnailWriter::create_movie_thumb(apr_pool_t *pool, const char *file_name)
{
    list<Magick::Image> frame_list;
    AVFormatContext *format_context;
    AVCodecContext *codec_context;
    AVCodec *codec;
    AVFrame *frame;
    AVPicture *picture;
    int video_index;
    apr_size_t buffer_size;

    if (!open_movie_context(pool, file_name, &format_context, &codec_context,
                            &codec, &video_index)) {
        return false;
    }

    frame = avcodec_alloc_frame();
    picture = reinterpret_cast<AVPicture *>(avcodec_alloc_frame());
    if (picture == NULL) {
        return false;
    }

    buffer_size = avpicture_get_size(FRAME_PIXEL_FORMAT,
                                     codec_context->width, codec_context->height);

    apr_byte_t buffer[buffer_size];

    avpicture_fill(picture, buffer, FRAME_PIXEL_FORMAT,
                   codec_context->width, codec_context->height);


    read_movie_frame(format_context, codec_context,frame, picture, video_index,
                     FRAME_SAMPLE_SEC, FRAME_DELAY_SEC, FRAME_NUMBER, frame_list);

    writeImages(frame_list.begin(), frame_list.end(),
                create_thumb_path(pool, thumb_dir_, file_name));

    av_free(picture);
    av_free(frame);

    close_movie_context(format_context, codec_context);

    return true;
}

bool ThumbnailWriter::open_movie_file(apr_pool_t *pool, const char *file_name,
                                      AVFormatContext **format_context)
{
    char *file_path;
    AVInputFormat *format;
    AVProbeData probe_data;
    ByteIOContext io_context;
    apr_byte_t buffer[BUFFER_SIZE];

    if (apr_filepath_merge(&file_path, file_dir_, file_name,
                           APR_FILEPATH_NOTABOVEROOT, pool) != APR_SUCCESS) {
        throw "åץɥե̾ǤޤǤ";
    }

    if (url_fopen(&io_context, file_path, URL_RDONLY) < 0) {
        return false;
    }

    io_context.seek(io_context.opaque, UploadItem::ITEM_HEADER_SIZE, SEEK_SET);

    probe_data.filename = file_path;
    probe_data.buf = buffer;
    probe_data.buf_size = 0;
    probe_data.buf_size = get_buffer(&io_context, buffer, BUFFER_SIZE);

    url_fseek(&io_context, 0, SEEK_SET);

    format = av_probe_input_format(&probe_data, 1);

    if (!format) {
        url_fclose(&io_context);
        return false;
    }

    if (av_open_input_stream(format_context, &io_context, file_path, format, NULL)) {
        url_fclose(&io_context);
        return false;
    }

    return true;
}

bool ThumbnailWriter::open_movie_context(apr_pool_t *pool, const char *file_path,
                                         AVFormatContext **format_context,
                                         AVCodecContext **codec_context,
                                         AVCodec **codec, int *video_index)
{
    int i;

    av_register_all();

    if (!open_movie_file(pool, file_path, format_context)) {
        return false;
    }

    if (av_find_stream_info(*format_context) < 0) {
        return false;
    }

    *video_index = -1;
    for (i = 0; i < (*format_context)->nb_streams; i++) {
        if ((*format_context)->streams[i]->codec.codec_type == CODEC_TYPE_VIDEO) {
            *video_index = i;
            break;
        }
    }

    if (*video_index == -1) {
        return false;
    }

    *codec_context = &((*format_context)->streams[*video_index]->codec);

    *codec = avcodec_find_decoder((*codec_context)->codec_id);
    if (*codec == NULL) {
        return false;
    }

    if (avcodec_open(*codec_context, *codec) < 0) {
        return false;
    }

    return true;
}

void ThumbnailWriter::close_movie_context(AVFormatContext *format_context,
                                          AVCodecContext *codec_context)
{
    avcodec_close(codec_context);
    av_close_input_file(format_context);
}

void ThumbnailWriter::read_movie_frame(AVFormatContext *format_context,
                                       AVCodecContext *codec_context,
                                       AVFrame *frame, AVPicture *picture, int video_index,
                                       double frame_sample, double frame_delay,
                                       apr_size_t frame_number,
                                       list<Magick::Image> &frame_list)
{
    AVPacket packet;
    apr_size_t movie_fps;
    int is_frame_finish;
    apr_size_t i;
    apr_size_t j;
    Magick::Geometry movie_size(codec_context->width, codec_context->height);
    Magick::Geometry thumb_size = movie_size;

    adjust_size(thumb_size);
    movie_fps = codec_context->frame_rate / codec_context->frame_rate_base;

    // TODO: av_seek_frame 夤ƤʲΥ롼פ롥

    i = j = 0;
    while (av_read_frame(format_context, &packet) >= 0) {
        if (packet.stream_index != video_index) {
            av_free_packet(&packet);
            continue;
        }

        avcodec_decode_video(codec_context, frame, &is_frame_finish,
                             packet.data, packet.size);

        if (!is_frame_finish || ((++i % static_cast<apr_size_t>(movie_fps*frame_sample)) != 0)) {
            av_free_packet(&packet);
            continue;
        }

        img_convert(picture, FRAME_PIXEL_FORMAT, reinterpret_cast<AVPicture*>(frame),
                    codec_context->pix_fmt, codec_context->width, codec_context->height);

        Magick::Image frame_image(movie_size.width(), movie_size.height(), "RGB",
                                  Magick::CharPixel, picture->data[0]);

        frame_image.scale(thumb_size);
        frame_image.animationDelay(static_cast<apr_size_t>(frame_delay*100));
        frame_image.animationIterations(0);

        frame_list.push_back(frame_image);

        if (++j == frame_number) {
            return;
        }

        av_free_packet(&packet);
    }
}
#endif

void ThumbnailWriter::adjust_size(Magick::Geometry &size) const
{
    if ((size.width() <= WIDTH_LIMIT) && (size.height() <= HEIGHT_LIMIT)) {
        return;
    }

    if ((size.width()/size.height()) > (WIDTH_LIMIT/HEIGHT_LIMIT)) {
        if (size.width() > WIDTH_LIMIT) {
            size.height(size.height()*WIDTH_LIMIT/size.width());
            size.width(WIDTH_LIMIT);
        }
    } else {
        if (size.height() > HEIGHT_LIMIT) {
            size.width(size.width()*HEIGHT_LIMIT/size.height());
            size.height(HEIGHT_LIMIT);
        }
    }
}


/******************************************************************************
 * ƥ
 *****************************************************************************/
#ifdef DEBUG_ThumbnailWriter

#include "apr_general.h"
#include "apr_file_io.h"

static const char THUMB_DIR[]            = "/tmp";

void usage(const char *prog_name)
{
    cerr << "Usage: " << prog_name << " <FILE> <MAGICKDIR>" << endl;
}

int main(int argc, const char * const *argv)
{
    apr_pool_t *pool;

    apr_app_initialize(&argc, &argv, NULL);
    apr_pool_create(&pool, NULL);

    try {
        if (argc != 2) {
            throw "ѤΥե뤬ꤵƤޤ";
        }

        ThumbnailWriter writer(pool, dirname_ex(pool, argv[1]), THUMB_DIR);
        writer.write(basename_ex(argv[1]));
    } catch(const char *message) {
        cerr << "Error: " << message << endl;
        usage(argv[0]);

        return EXIT_FAILURE;
    }

    apr_terminate();

    return EXIT_SUCCESS;
}
#endif

#endif

// Local Variables:
// mode: c++
// buffer-file-coding-system: euc-japan-dos
// End:
