#!/usr/bin/ruby -Ku
#
# mmon
#
# Copyright (c) 2007 sanpo
#
# This program is free software.
# You can redistribute it and/or modify it under the terms of the GPL.

require 'mmon/item'
require 'mmon/util'
require 'mmon/component/label'

class ItemTemp < Item
    NAME = 'Temperature'

    @@color_text = Color.new(1.0, 1.0, 0.0, 1.0)

    def initialize(mmon)
        super(mmon)

        @interval = 1.0
        @use_bg = true

        @width = 72
        @height = 32

        @temp1 = 0.0
        @temp2 = 0.0

        @temp1_input = ''
        @temp2_input = ''

        @temp1_ok = false
        @temp2_ok = false

        # 総当たりでファイル名が temp?_input なファイルを探す  
        # こんな方法でいいのかは不明だがとりあえず動く
        IO.popen("find /sys -name temp?_input") do |io|
            io.each_line do |l|
                l.chomp!
                case l
                when /temp1_input$/ then @temp1_input = l
                when /temp2_input$/ then @temp2_input = l
                end
            end
        end

        @label1 = Label.new(nil, 14, false, true, @@color_text, 0.5 * @width, 0.3 * @height)
        @label1.text = '??'
    
        @label2 = Label.new(nil, 14, false, true, @@color_text, 0.5 * @width, 0.7 * @height)
        @label2.text = '??'

        @sysfs_readable = false

        f = File.open('/proc/version')
        if f.readline =~ /Linux version 2.6.(\d+)/
            @sysfs_readable = true if $1.to_i >= 20

#           puts "itemtemp: kernel version : #{$1}"
        end
#       puts "sysfs_ok : #{Util.bool_to_str(@sysfs_readable)}"

        check_sysfs()
    end

    private

    def render_mask(cc, w, h)
        Util.simple_bg(cc, w, h)
        Util.simple_fg(cc, w, h)
    end

    def render_bg(cc, w, h)
        Util.simple_bg(cc, w, h)
        Util.simple_fg(cc, w, h)
    end

    def render(cc, w, h)
        @label1.render(cc)
        @label2.render(cc)
    end

    def prepare
        #puts 'prepare temp'
        
        if @temp1_input == '' && @temp2_input == ''
            return false
        end

        oldtemp1 = @temp1
        oldtemp2 = @temp2
        begin
            @temp1 = 0.0
            @temp2 = 0.0

            if @sysfs_readable
                if @temp1_ok
                    File.open(@temp1_input) { |f| @temp1 = f.readline.to_f / 1000.0 }
                end

                if @temp2_ok
                    File.open(@temp2_input) { |f| @temp2 = f.readline.to_f / 1000.0 }
                end
            else
                # 素直に読むと f.read した時に止まる
                # ruby-gnome2 のメーリングリストに同じ問題についてのスレッドがある
                # ruby-gnome2 が原因ではないらしい
                if @temp1_ok
                    IO.popen("cat #{@temp1_input}") { |io| @temp1 = io.read.to_f / 1000.0 }
                end

                if @temp2_ok
                    IO.popen("cat #{@temp2_input}") { |io| @temp2 = io.read.to_f / 1000.0 }
                end
            end

            #puts @temp1.to_s + ' ' + @temp2.to_s
        rescue
            p $!
        ensure 
        end

        if oldtemp1 == @temp1 && oldtemp2 == @temp2 
            return false
        else
            @label1.text = sprintf("%4.1f", @temp1)
            @label2.text = sprintf("%4.1f", @temp2)
            return true
        end
    end

    def read_element_localvalue(e)
        e_temp1 = e.elements['temp1_input']
        @temp1_input = e_temp1.text.to_i unless e_temp1.nil?
        e_temp2 = e.elements['temp2_input']
        @temp2_input = e_temp2.text.to_i unless e_temp2.nil?

        check_sysfs()
    end

    def write_element_local(e)
        e_1 = REXML::Element.new('temp1_input')
        e_1.text = @temp1_input
        e_2 = REXML::Element.new('temp2_input')
        e_2.text = @temp2_input

        e.add_element(e_1)
        e.add_element(e_2)
    end

    def setting_widget(x, y)
        label_title = Gtk::Label.new('file location (/sys/.../temp[12]_input)')

        fcb1 = Gtk::FileChooserButton.new('Select "temp1_input"', Gtk::FileChooser::Action::OPEN)
        fcb1.filename = @temp1_input
        fcb2 = Gtk::FileChooserButton.new('Select "temp2_input"', Gtk::FileChooser::Action::OPEN)
        fcb2.filename = @temp2_input

        fcb1.signal_connect('selection-changed') do |w|
            @temp1_input = w.filename
            check_sysfs()
        end

        fcb2.signal_connect('selection-changed') do |w|
            @temp2_input = w.filename
            check_sysfs()
        end

        vbox = Gtk::VBox.new
        vbox.pack_start(fcb1, false, false, 10)
        vbox.pack_start(fcb2, false, false, 10)

        return vbox
    end

    def check_sysfs
        @temp1_ok = false
        @temp2_ok = false
        @temp1_ok = File.exist?(@temp1_input)
        @temp2_ok = File.exist?(@temp2_input)
    end
end

if __FILE__ == $0
    require 'mmon/appmmon'

    app = AppMmon.new
    item = ItemTemp.new(app)
    app.add_item(item)

    Gtk::main()
end
