#!/usr/bin/ruby -Ku
#
# mmon
#
# Copyright (c) 2007 sanpo
#
# This program is free software.
# You can redistribute it and/or modify it under the terms of the GPL.

require 'gtk2'

require 'item'
require 'util'
require 'icon'

class ItemLauncher < Item
    TYPE_V = 0
    TYPE_H = 1

    def initialize(parent = nil)
        super(parent)

        self.add_events(Gdk::Event::Mask::BUTTON_PRESS_MASK |
                        Gdk::Event::Mask::BUTTON_RELEASE_MASK |
                        Gdk::Event::POINTER_MOTION_MASK)
        signal_connect("motion-notify-event")   { |w, e| motion_notify_event(e) }
        signal_connect("leave-notify-event")   { |w, e| leave_notify_event(e) }
		signal_connect("button-press-event")    { |w, e| button_press_event(e) }
		signal_connect("button-release-event")    { |w, e| button_release_event(e) }

        @type = TYPE_V
        @iconlist = []

        @border_item = 8
        @border_icon = 1
        @icon_size = 48
        @tile_size = @icon_size + @border_icon * 2
        @base_size = @tile_size + @border_item * 2
        set_icon_size(@icon_size)
                # 2^n のサイズにするとコンポジットマネージャーを使う環境で白抜きになる

        @cursor_icon = nil
        @pressed = false

        @interval = 0.0
        @use_bg = true
        @use_fg = true
        self.set_item_size(@base_size, @base_size)

        # dnd
        dnd_target = [ ['text/uri-list'    , 0, 0],
                       ['text/plaint'      , 0, 1],
        ]
        Gtk::Drag.dest_set(self, Gtk::Drag::DEST_DEFAULT_ALL, dnd_target, 
                                Gdk::DragContext::Action::COPY | Gdk::DragContext::Action::MOVE)
        signal_connect("drag-data-received") do |w, drag_context, x, y, data, info, time|
            puts "drag_data_received (#{x}, #{y})"
            drag_context.targets.each do |target|
                dnd_target.each do |t|
                    if target.name == t[0] || data.type == Gdk::Selection::TYPE_STRING
                        puts 'target_name : ' +  target.name + '::' + data.data
                        uri = data.data
                        path = uri.strip!.gsub!(/^file:\/\//, '')

                        icon = Icon.new(self, @icon_size, @icon_size, path)
                        @iconlist << icon
                        calc_icon_position()
                        w, h = calc_size()
                        self.set_item_size(w, h)

                        # sid の ruby-gnome2 は古くてバグ持ち # 最新版が来るまでコメントアウト
                        #Gtk::Drag.finish(drag_context, true, false, time)

                        self.redraw_all()
                    end
                end
            end
        end
    end

    private

    def calc_size()
        if @iconlist.size.zero?
            n = 1 
        else
            n = @iconlist.size - 1
        end
        if @type == TYPE_V
            w = @base_size
            h = @base_size + @tile_size * n
        elsif @type == TYPE_H
            w = @base_size + @tile_size * n
            h = @base_size
        end

        return [w, h]
    end

    def calc_icon_position
        x = (@base_size - @tile_size) / 2.0
        y = (@base_size - @tile_size) / 2.0

        @iconlist.each do |icon|
            icon.set_position(x + @border_icon, y + @border_icon)

            case @type
            when TYPE_V then y += @tile_size
            when TYPE_H then x += @tile_size
            end
        end
    end

    def set_type(type)
        @type = type
        calc_icon_position()
        w, h = calc_size()
        self.set_item_size(w, h)
    end

    def set_icon_size(size)
        @icon_size = size
        @tile_size = @icon_size + @border_icon * 2
        @base_size = @tile_size + @border_item * 2

        @iconlist.each do |icon|
            icon.set_size(@icon_size, @icon_size)
        end
    end

    def render_mask(cc, w, h)
        Util.simple_bg(cc, w, h)
        Util.simple_fg(cc, w, h)
    end

    def render_bg(cc, w, h)
        Util.simple_bg(cc, w, h)
    end

    def render_fg(cc, w, h)
        Util.simple_fg(cc, w, h)
    end

    def render(cc, w, h)
        @iconlist.each { |icon| icon.render(cc) }
    end

    def read_element_local(e)
        e_type = e.elements['Type']
        unless e_type.nil?
            case e_type.text
            when 'Vertical'   then @type = TYPE_V
            when 'Horizontal' then @type = TYPE_H
            end
        end

        e_size = e.elements['IconSize']
        set_icon_size(e_size.text.to_i) unless e_size.nil?

        e.each_element('Icon') do |e_icon|
            icon = Icon.new(self, @icon_size, @icon_size)
            icon.read_element(e_icon)
            @iconlist << icon
        end

        calc_icon_position()
        w, h = calc_size()
        self.set_item_size(w, h)
    end

    def write_element_local(e)
        e_type = REXML::Element.new('Type')
        case @type
        when TYPE_V then e_type.text = 'Vertical'
        when TYPE_H then e_type.text = 'Horizontal'
        end

        e.add_element(e_type)

        e_size = REXML::Element.new('IconSize')
        e_size.text = @icon_size.to_s
        e.add_element(e_size)

        @iconlist.each do |icon|
            e_icon = icon.write_element
            e.add_element(e_icon)
        end
    end

    def button_press_event(event)
        case event.button
        when 1 
            @pressed = true

            unless @cursor_icon.nil?
                @cursor_icon.press
                self.redraw_all()
            end
            return true
        end

        return false
    end

    def button_release_event(event)
        case event.button
        when 1 
            @pressed = false

            unless @cursor_icon.nil?
                @cursor_icon.release
                self.redraw_all()
            end
            return true
        end

        return false
    end

    def motion_notify_event(event)
        x = event.x
        y = event.y

        @pressed = true if event.state.button1_mask?

        #puts "mouse_motion #{x} #{y}"

        cursor_icon_old = @cursor_icon
        n =  calc_cursor(x, y)
        if n == -1
            @cursor_icon = nil
        else
            @cursor_icon = @iconlist[n]
            @cursor_icon.move(@pressed) 
        end

        if @cursor_icon != cursor_icon_old 
            cursor_icon_old.leave unless cursor_icon_old.nil?
            @cursor_icon.move(@pressed) unless @cursor_icon.nil?
            self.redraw_all()
        end
        
        return true
    end

    def leave_notify_event(event)
        @pressed = false

        @cursor_icon.leave unless @cursor_icon.nil?

        self.redraw_all()

        return true
    end

    def calc_cursor(x, y)
        base = (@base_size - @tile_size) / 2.0
        x0 = base
        y0 = base
        if @type == TYPE_V
            x1 = base + @tile_size - 1 
            y1 = base + @iconlist.size * @tile_size - 1
        elsif @type == TYPE_H
            x1 = base + @iconlist.size * @tile_size - 1
            y1 = base + @tile_size - 1
        end

        if x < x0 || x > x1 || y < y0 || y > y1
            return -1
        end

        #puts "calc_cursor #{x0} #{y0} #{x1} #{y1}"
            
        if @type == TYPE_V
            n = (y - y0) / @tile_size 
        elsif @type == TYPE_H
            n = (x - x0) / @tile_size 
        end

        return n.to_i
    end

    COLUMN_NAME = 0
    COLUMN_ICON = 1

    def setting_widget(x, y)
        n =  calc_cursor(x, y)

	    rb_type_v = Gtk::RadioButton.new('Vertical')
	    rb_type_h = Gtk::RadioButton.new(rb_type_v, 'Horizontal')
	    rb_type_v.signal_connect('toggled') do |rb|
            set_type(TYPE_V) if rb.active?
	    end
	    rb_type_h.signal_connect('toggled') do |rb|
            set_type(TYPE_H) if rb.active?
	    end

        case @type
        when TYPE_V then rb_type_v.active = true
        when TYPE_H then rb_type_h.active = true
        end

        h0 = Gtk::HBox.new
        h0.pack_start(Gtk::Label.new('Type:'), false, false, 10)
        h0.pack_start(rb_type_v, false, false, 10)
        h0.pack_start(rb_type_h, false, false, 10)

        spin_b = Gtk::SpinButton.new(22.0, 128.0, 2.0)
        spin_b.value = @icon_size

        apply_b = Gtk::Button.new('Apply')
        apply_b.signal_connect('clicked') do |w|
            set_icon_size(spin_b.value) 
            calc_icon_position()
            w, h = calc_size()
            self.set_item_size(w, h)
        end

        h1 = Gtk::HBox.new
        h1.pack_start(Gtk::Label.new('Size:'), false, false, 10)
        h1.pack_start(spin_b, false, false, 10)
        h1.pack_start(apply_b, false, false, 10)

	    liststore = Gtk::ListStore.new(String, Icon)
	    @iconlist.each do |i|
	        iter = liststore.append
	        iter[COLUMN_NAME] = i.name
	        iter[COLUMN_ICON] = i
	    end
	
		column = Gtk::TreeViewColumn.new("Icon", Gtk::CellRendererText.new, :text => COLUMN_NAME)
	
        listview = Gtk::TreeView.new(liststore)
        listview.set_width_request(150)

		scroll = Gtk::ScrolledWindow.new
		scroll.set_policy(Gtk::POLICY_AUTOMATIC,Gtk::POLICY_AUTOMATIC)
		scroll.add(listview)

        b_up = Gtk::Button.new
        b_down = Gtk::Button.new
        b_add = Gtk::Button.new
        b_remove = Gtk::Button.new

	    b_up.signal_connect('clicked') do |w|
			iter0 = listview.selection.selected
            unless iter0.nil?
	            path1 = iter0.path.copy
                path1.prev!
                iter1 = liststore.get_iter(path1)

                unless iter1.nil?
                    liststore.swap(iter0, iter1)
                    listview.set_cursor(path1, nil, false)

                    sync_list(liststore)
                    self.redraw_all()
                end
            end
        end
	
	    b_down.signal_connect('clicked') do |w|
			iter0 = listview.selection.selected
            unless iter0.nil?
	            path1 = iter0.path.copy
	            path1.next!
                iter1 = liststore.get_iter(path1)
            
                unless iter1.nil?
                    liststore.swap(iter0, iter1)

                    sync_list(liststore)
                    listview.set_cursor(path1, nil, false)
                    self.redraw_all()
                end
            end
        end
	
	    b_add.signal_connect('clicked') do |w|
	        newicon = Icon.new(self, @icon_size, @icon_size)
	        iter = liststore.append
	        iter[COLUMN_NAME] = newicon.name
	        iter[COLUMN_ICON] = newicon

	        listview.set_cursor(iter.path, nil, false)

            sync_list(liststore)
            w, h = calc_size()
            self.set_item_size(w, h)
	    end

        icon_setting_widget = IconSettingWidget.new
	
	    b_remove.signal_connect('clicked') do |w|
			iter = listview.selection.selected
            if iter != nil
                liststore.remove(iter)

                icon_setting_widget.icon = nil

                sync_list(liststore)
                w, h = calc_size()
                self.set_item_size(w, h)
            end
	    end

	    b_up.add(Gtk::Image.new(Gtk::Stock::GO_UP, Gtk::IconSize::BUTTON))
	    b_down.add(Gtk::Image.new(Gtk::Stock::GO_DOWN, Gtk::IconSize::BUTTON))
	    b_add.add(Gtk::Image.new(Gtk::Stock::ADD, Gtk::IconSize::BUTTON))
	    b_remove.add(Gtk::Image.new(Gtk::Stock::REMOVE, Gtk::IconSize::BUTTON))
	
	    hb = Gtk::HBox.new
	    hb.pack_start(b_add)
        hb.pack_start(b_remove)
	    hb.pack_start(b_up)
	    hb.pack_start(b_down)

        frame_l = Gtk::Frame.new
        frame_l.add(scroll)
	
        v0 = Gtk::VBox.new
        v0.pack_start(frame_l, true)
        v0.pack_start(hb, false)
	
        frame_r = Gtk::Frame.new
        frame_r.add(icon_setting_widget)
	
	    h_icon = Gtk::HBox.new
	    h_icon.pack_start(v0, false, false, 5)
	    h_icon.pack_start(frame_r, true, true, 5)
	
		listview.headers_visible = false
		listview.append_column(column)
		listview.signal_connect('cursor-changed') do |w|
			iter = w.selection.selected
	        icon_setting_widget.icon = iter[COLUMN_ICON]
		end

	    vall = Gtk::VBox.new
        vall.set_width_request(500)
	    vall.pack_start(h0, false, false, 2)
	    vall.pack_start(h1, false, false, 2)
	    vall.pack_start(Gtk::HSeparator.new, false, false, 2)
	    vall.pack_start(h_icon, true, true)
            
        vall.show_all

        if n == -1
        else
            path = Gtk::TreePath.new(n.to_s) unless n == -1
	        listview.set_cursor(path, nil, false)
        end

        return vall
    end

    def sync_list(liststore)
        @iconlist.clear

        iter = liststore.iter_first

        stat = true
        while stat
            icon = iter[COLUMN_ICON]
            @iconlist << icon

            stat = iter.next!
        end

        calc_icon_position()
    end
end

if __FILE__ == $0
    item = ItemLauncher.new
    item.show

    Gtk::main()
end
