﻿using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Graphics;

namespace MikuMikuDance.XNA.Stages
{
    /// <summary>
    /// MikuMikuDanceで用いるカメラ
    /// </summary>
    public class MMDCamera : GameComponent, IMMDCamera
    {
        // 視錐台(シャドウマップ用)
        BoundingFrustum CameraFrustum;
        /// <summary>
        /// カメラのポジション
        /// </summary>
        public Vector3 CameraPos { get; set; }
        /// <summary>
        /// カメラの視点
        /// </summary>
        public Vector3 CameraTarget { get; set; }
        /// <summary>
        /// カメラの上方向を表すベクトル
        /// </summary>
        public Vector3 CameraUpVector { get; set; }
        /// <summary>
        /// カメラに適応するワールド座標系
        /// </summary>
        public Matrix CameraMatrix { get; set; }
        /// <summary>
        /// 視野角
        /// </summary>
        public float FieldOfView { get; set; }
        /// <summary>
        /// Nearプレーン距離
        /// </summary>
        public float Near { get; set; }
        /// <summary>
        /// Farプレーン距離
        /// </summary>
        public float Far { get; set; }
        /// <summary>
        /// アニメーションプレイヤー
        /// </summary>
        public CameraAnimationPlayer Player { get; private set; }
        /// <summary>
        /// 既定のコンストラクタ
        /// </summary>
        internal MMDCamera(Game game)
            :base(game)
        {
            CameraPos = new Vector3(0, 10, 35);
            CameraTarget = new Vector3(0, 10, 0);
            CameraUpVector = Vector3.Up;
            CameraMatrix = Matrix.Identity;
            FieldOfView = MathHelper.PiOver4;
            Near = 1.0f;
            Far = 300.0f;
            Player = new CameraAnimationPlayer();
            CameraFrustum = new BoundingFrustum(Matrix.Identity);
        }
        /// <summary>
        /// ビューマトリックス取得
        /// </summary>
        /// <returns>ビューマトリックス</returns>
        public Matrix GetViewMatrix()
        {
            return Matrix.CreateLookAt(Vector3.Transform(CameraPos, CameraMatrix),
                Vector3.Transform(CameraTarget, CameraMatrix), Vector3.Transform(CameraUpVector, CameraMatrix));
        }
        /// <summary>
        /// シャドウマップ用ビューマトリックス取得
        /// </summary>
        /// <param name="LightPos">ライトの位置</param>
        /// <returns>シャドウマップ用ビューマトリックス</returns>
        public Matrix GetShadowViewMatrix(Vector3 LightPos)
        {
            return Matrix.CreateLookAt(Vector3.Transform(LightPos, CameraMatrix),
                Vector3.Transform(CameraTarget, CameraMatrix), Vector3.Transform(CameraUpVector, CameraMatrix));
        }
        /// <summary>
        /// プロジェクションマトリックス取得
        /// </summary>
        /// <param name="graphics">グラフィックデバイスオブジェクト</param>
        /// <returns>プロジェクションマトリックス</returns>
        public Matrix GetProjectionMatrix(GraphicsDevice graphics)
        {
            Viewport viewport = graphics.Viewport;
            float aspectRatio = (float)viewport.Width / (float)viewport.Height;
            return Matrix.CreatePerspectiveFieldOfView(FieldOfView, aspectRatio, Near, Far);
        }
        /// <summary>
        /// カメラUpdate
        /// </summary>
        /// <param name="gameTime">ゲームタイムオブジェクト</param>
        public override void Update(GameTime gameTime)
        {
            Player.Update(-1, false, this);
            
        }
        /// <summary>
        /// 視錐台のコーナー情報取得
        /// </summary>
        /// <param name="graphics">グラフィックスデバイス</param>
        /// <param name="shadowDist">シャドウ距離</param>
        /// <param name="frustumCorners">視錐台のコーナー</param>
        public void GetFrustumCorners(GraphicsDevice graphics, float shadowDist, Vector3[] frustumCorners)
        {
            //視錐台の更新
            Viewport viewport = graphics.Viewport;
            float aspectRatio = (float)viewport.Width / (float)viewport.Height;
            Matrix Proj = Matrix.CreatePerspectiveFieldOfView(FieldOfView, aspectRatio, Near, shadowDist);
            CameraFrustum.Matrix = GetViewMatrix() * Proj;
            //視錐台からコーナーの取得
            CameraFrustum.GetCorners(frustumCorners);
        }
    }
}
