﻿using System;
using BulletX.BulletCollision.BroadphaseCollision;
using BulletX.LinerMath;

namespace BulletX.BulletCollision.CollisionShapes
{
    /// <summary>
    /// カプセル型衝突スキン
    /// </summary>
    public class CapsuleShape : ConvexInternalShape
    {
        public override BroadphaseNativeTypes ShapeType
        {
            get { return BroadphaseNativeTypes.CAPSULE_SHAPE_PROXYTYPE; }
        }

        protected int m_upAxis;
        
        protected CapsuleShape() : base() { }
        public CapsuleShape(float radius, float height)
            :base()
        {
            m_upAxis = 1;
            m_implicitShapeDimensions.setValue(radius, 0.5f * height, radius);
        }

        //CollisionShape Interface
        public override void calculateLocalInertia(float mass, out btVector3 inertia)
        {
            btTransform ident = btTransform.Identity;


            float radius = Radius;

            btVector3 halfExtents = new btVector3(radius, radius, radius);
            halfExtents[UpAxis] += HalfHeight;

            float margin = CollisionMargin.CONVEX_DISTANCE_MARGIN;

            float lx = 2f * (halfExtents[0] + margin);
            float ly = 2f * (halfExtents[1] + margin);
            float lz = 2f * (halfExtents[2] + margin);
            float x2 = lx * lx;
            float y2 = ly * ly;
            float z2 = lz * lz;
            float scaledmass = mass * 0.08333333f;

            inertia = new btVector3(scaledmass * (y2 + z2), scaledmass * (x2 + z2), scaledmass * (x2 + y2));
        }

        // btConvexShape Interface
        public override btVector3 localGetSupportingVertexWithoutMargin(btVector3 vec0)
        {
            btVector3 supVec = btVector3.Zero;

            float maxDot = (-BulletGlobal.BT_LARGE_FLOAT);

            btVector3 vec = vec0;
            float lenSqr = vec.Length2;
            if (lenSqr < 0.0001f)
            {
                vec.setValue(1, 0, 0);
            }
            else
            {
                float rlen = 1f / (float)Math.Sqrt(lenSqr);
                vec *= rlen;
            }

            btVector3 vtx;
            float newDot;

            float radius = Radius;


            {
                btVector3 pos = btVector3.Zero;
                pos[UpAxis] = HalfHeight;

                vtx = pos + vec * m_localScaling * (radius) - vec * Margin;
                newDot = vec.dot(vtx);
                if (newDot > maxDot)
                {
                    maxDot = newDot;
                    supVec = vtx;
                }
            }
            {
                btVector3 pos = btVector3.Zero;
                pos[UpAxis] = -HalfHeight;

                vtx = pos + vec * m_localScaling * (radius) - vec * Margin;
                newDot = vec.dot(vtx);
                if (newDot > maxDot)
                {
                    maxDot = newDot;
                    supVec = vtx;
                }
            }

            return supVec;
        }

        public override void batchedUnitVectorGetSupportingVertexWithoutMargin(btVector3[] vectors, btVector3[] supportVerticesOut, int numVectors)
        {
            float radius = Radius;

            for (int j = 0; j < numVectors; j++)
            {
                float maxDot = -BulletGlobal.BT_LARGE_FLOAT;
                btVector3 vec = vectors[j];

                btVector3 vtx;
                float newDot;
                {
                    btVector3 pos = btVector3.Zero;
                    pos[UpAxis] = HalfHeight;
                    vtx = pos + vec * m_localScaling * (radius) - vec * Margin;
                    newDot = vec.dot(vtx);
                    if (newDot > maxDot)
                    {
                        maxDot = newDot;
                        supportVerticesOut[j] = vtx;
                    }
                }
                {
                    btVector3 pos = btVector3.Zero;
                    pos[UpAxis] = -HalfHeight;
                    vtx = pos + vec * m_localScaling * (radius) - vec * Margin;
                    newDot = vec.dot(vtx);
                    if (newDot > maxDot)
                    {
                        maxDot = newDot;
                        supportVerticesOut[j] = vtx;
                    }
                }

            }
        }
        public override float Margin
        {
            set
            {
                //correct the m_implicitShapeDimensions for the margin
                btVector3 oldMargin = new btVector3(Margin, Margin, Margin);
                btVector3 implicitShapeDimensionsWithMargin = m_implicitShapeDimensions + oldMargin;

                base.Margin = value;
                btVector3 newMargin = new btVector3(Margin, Margin, Margin);
                m_implicitShapeDimensions = implicitShapeDimensionsWithMargin - newMargin;
            }
        }

        public override void getAabb(btTransform t, out btVector3 aabbMin, out btVector3 aabbMax)
        {
            btVector3 halfExtents = new btVector3(Radius, Radius, Radius);
            halfExtents[m_upAxis] = Radius + HalfHeight;
            halfExtents += new btVector3(Margin, Margin, Margin);
            btMatrix3x3 abs_b = t.Basis.absolute();
            btVector3 center = t.Origin;
            btVector3 extent = new btVector3(abs_b[0].dot(halfExtents), abs_b[1].dot(halfExtents), abs_b[2].dot(halfExtents));

            aabbMin = center - extent;
            aabbMax = center + extent;
        }

        public override string Name { get { return "CapsuleShape"; } }


        public int UpAxis { get { return m_upAxis; } }

        
        public float Radius
        {
            get
            {
                int radiusAxis = (m_upAxis + 2) % 3;
                return m_implicitShapeDimensions[radiusAxis];
            }
        }
        public float HalfHeight { get { return m_implicitShapeDimensions[m_upAxis]; } }

        public override btVector3 LocalScaling
        {
            set
            {

                btVector3 oldMargin=new btVector3(Margin,Margin,Margin);
		        btVector3 implicitShapeDimensionsWithMargin = m_implicitShapeDimensions+oldMargin;
		        btVector3 unScaledImplicitShapeDimensionsWithMargin = implicitShapeDimensionsWithMargin / m_localScaling;
                base.LocalScaling = value;
            
		        m_implicitShapeDimensions = (unScaledImplicitShapeDimensionsWithMargin * m_localScaling) - oldMargin;

            }
        }

#if false
        virtual	int	calculateSerializeBufferSize() const;

	    ///fills the dataBuffer and returns the struct name (and 0 on failure)
	    virtual	const char*	serialize(void* dataBuffer, btSerializer* serializer) const;
#endif
        
    }
}
