/*
 * The MIT License
 *
 * Copyright 2015 nazo.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package jp.sourceforge.mmd.midiMotion.midi;

import java.util.SortedMap;
import javax.sound.midi.MidiEvent;

/**
 * Note On と Note Off の関連付け. {@link java.util.SortedSet}や{@link SortedMap}に
 * いれて使うことを想定して, {@link Comparable} にしてある.
 * @author nazo
 */
public class MidiRelation implements Comparable<MidiRelation>{
    private MidiEvent noteOn;
    private MidiEvent noteOff;
    private long length;
    private final MidiComparator comparator;

    /**
     * 初期化しかしないコンパレーター.
     */
    public MidiRelation(){
        noteOn=null;
        noteOff=null;
        length=-1;
        comparator=new MidiComparator();
    }

    /** Note On を設定する.
     * @param mm Note On の MidiEvent. Note On じゃ無いときの動作は保障しない.
     */
    public void setNoteOn(MidiEvent mm){
        noteOn=mm;
        if(noteOff!=null)
            length=noteOff.getTick()-noteOn.getTick();
    }

    /**
     * Note Off を設定する.
     * @param mm Note Off の MidiEvent. Note Off じゃ無いときの動作は保障しない.
     * Note Onと同じ ch, Note# を持つものとする.
     */
    public void setNoteOff(MidiEvent mm){
        noteOff=mm;
        if(noteOn!=null)
            length=noteOff.getTick()-noteOn.getTick();
    }

    /**
     * 対応する Note On を得る.
     * @return Note On MidiEvent.
     */
    public MidiEvent getNoteOn(){
        return noteOn;
    }

    /**
     * 対応する Note Off を得る.
     * @return Note Off MidiEvent.
     */
    public MidiEvent getNoteOff(){
        return noteOff;
    }

    /**
     * 長さが MIDI tics 単位で帰ってくる.
     * @return MIDI tics.
     */
    public long getLength(){
        return length;
    }

    /**
     * 比較する.
     * @param o 比較対象.
     * @return 自分が少なければ負, 多ければ正. note On の tics を比べ,
     * 同じだったら note off の tics を比べ,
     * それも同じだったら note number を比較する.
     * 早い順になる.
     */
    @Override
    public int compareTo(MidiRelation o) {
        int ret;
        if(noteOn==null){
            ret=(o.noteOn==null)?0:-1;
        } else {
            ret=(o.noteOn==null)?1:(int)(noteOn.getTick()-o.noteOn.getTick());
        }
        if(ret!=0)return ret;
        if(noteOff==null){
            ret=(o.noteOff==null)?0:-1;
        } else {
            ret=(o.noteOff==null)?1:comparator.compare(noteOff, o.noteOff);
        }
        return ret;
    }
}
