/*
 * The MIT License
 *
 * Copyright 2015 nazo.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package jp.sourceforge.mmd.midiMotion.gui;

import java.awt.ScrollPane;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Map;
import java.util.Properties;
import javax.swing.JFileChooser;
import javax.swing.JOptionPane;
import javax.swing.filechooser.FileNameExtensionFilter;
import javax.swing.Box;
import java.awt.event.KeyEvent;
import java.io.BufferedReader;
import java.io.FilenameFilter;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipInputStream;
import javax.sound.midi.InvalidMidiDataException;
import javax.swing.JDialog;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JTextArea;
import javax.swing.JTextField;
import jp.sourceforge.mmd.midiMotion.midi.MidiResolver;
import jp.sourceforge.mmd.motion.swing.AboutDialog;
import jp.sourceforge.mmd.motion.swing.FileDropEvent;
import jp.sourceforge.mmd.motion.swing.FileDropListener;
import jp.sourceforge.mmd.motion.swing.FileDropTarget;
import jp.sourceforge.mmd.motion.swing.VMD2CSV;
import jp.sourceforge.mmd.motion.swing.MmdDialog;

/**
 * Midi to MMD Motion swing GUI main frame. MID2Motion の
 * 一番根本のウィンドウフレーム.
 * @author nazo
 */
public class MidiMotionFrame extends javax.swing.JFrame
implements FileDropListener, ActionListener {
    private static final long serialVersionUID = 1L;
    protected MidiResolver midiR;
    protected File midiDir;
    protected Properties prop;
    protected ArrayList<MotionPane> mps;
    protected float fps;
    private HashMap <JMenuItem,String> pluginReadme;
    
    /**
     * Creates new form MidiMotionFrame
     */
    @SuppressWarnings("LeakingThisInConstructor")
    public MidiMotionFrame()  {
        try {
            FileInputStream fis=new FileInputStream("./midiMotion.properties");
            prop=new Properties();
            prop.load(fis);
            fis.close();
            for(String s:prop.stringPropertyNames()){
                System.setProperty(s, prop.getProperty(s));
            }
            String s=prop.getProperty("midiMotion.midiDir");
            if(s==null){s="./";}
            midiDir=new File(s);
            MmdDialog.loadDefault();
        } catch (FileNotFoundException ex) {
            midiDir=new File("./");
        } catch (IOException ex) {
            JOptionPane.showMessageDialog(this,"IOエラー:設定ファイルの読み込みに失敗\n"+ex.getLocalizedMessage());
        }

        initComponents();
        pluginReadme=new HashMap<JMenuItem, String>();
        mps=new ArrayList<MotionPane>();
        /** dynamic class loader test */
        URLClassLoader loader;
        MotionPane mp=null;
        File plugins=new File("plugin/");
        File [] list;
        list = plugins.listFiles(new FilenameFilter() {
            @Override
            public boolean accept(File dir, String name) {
                String lower=name.toLowerCase();
                return lower.endsWith(".zip");
            }
        });
        for(final File f:list){
            try{
                loader=URLClassLoader.newInstance(new URL[]{
                    f.toURI().toURL()
                });
                JarFile jf=new JarFile(f);
                Enumeration<JarEntry> e= jf.entries();
                JarEntry je;
                String name=null;
                String readme=null;
                while(e.hasMoreElements()){
                    je=e.nextElement();
                    name=je.getName();
                    if(name.endsWith("MotionPane.class")){
                        name=name.substring(0, name.lastIndexOf('.'));
                        name=name.replace('/', '.');
                        break;
                    } else if(name.equalsIgnoreCase("readme.txt")){
                        readme=name;
                    }
                    name=null;
                }
                if(name!=null){
                    mp=(MotionPane)loader.loadClass(name).newInstance();
                    tabPane.add(mp,mp.getName());
                    mps.add(mp);
                }
                while(mp!=null && readme==null && e.hasMoreElements()){
                    je=e.nextElement();
                    if((name=je.getName())!=null){
                        readme=name;
                        break;
                    }
                }
                jf.close();
                if(mp!=null && readme!=null){
                    JMenu menuFolder=new JMenu(mp.getName());
                    JMenuItem item=new JMenuItem("README.txt");
                    pluginReadme.put(item, f.getName()+"/"+readme);
                    menuFolder.add(item);
                    menuPlugins.add(menuFolder);
                    item.addActionListener(this);
                }
            } catch (ClassNotFoundException ex) {
                JOptionPane.showMessageDialog(this,
                        "plugin エラー: クラスが見つかりません\nfile:"
                                +f+"\n"+ex.getLocalizedMessage());
            } catch (IllegalAccessException ex) {
                JOptionPane.showMessageDialog(this,
                        "読み込み権限がありません。Web start か Applet ですか?\n"
                                +"おそらく署名されてないか、セキュリティー的なことで弾かれてます。\n"
                                +ex.getLocalizedMessage());
            }catch(InstantiationException ex){
                JOptionPane.showMessageDialog(this,
                        "plugin エラー: インスタンスが作れません。壊れてる?\nfile:"
                                +f+"\n"+ex.getLocalizedMessage());
            }catch(MalformedURLException ex){
                JOptionPane.showMessageDialog(this,
                        "URL異常:"+ex.getLocalizedMessage());
            }catch(IOException ex){
                JOptionPane.showMessageDialog(this,
                        "IO異常:"+ex.getLocalizedMessage());
            }
        }

        textMidiFile.setDropTarget(new FileDropTarget(this));
        this.setSize(getPreferredSize());
    }

    private void setFPS(){
        fps=Float.parseFloat(textFPS.getText());
        if(midiR!=null)
            textFPB.setText(String.valueOf(fps*midiR.getTemp()/1000000)+" frames/beat");
    }

    /**
     * This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is always
     * regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;

        tabPane = new javax.swing.JTabbedPane();
        javax.swing.JLabel jLabel1 = new javax.swing.JLabel();
        textMidiFile = new javax.swing.JTextField();
        javax.swing.JLabel jLabel2 = new javax.swing.JLabel();
        textFPS = new javax.swing.JTextField();
        textLength = new javax.swing.JTextField();
        textTemp = new javax.swing.JTextField();
        textBeat = new javax.swing.JTextField();
        textFPB = new javax.swing.JTextField();
        menuBar = new javax.swing.JMenuBar();
        menuFile = new javax.swing.JMenu();
        menuOpen = new javax.swing.JMenuItem();
        menuClose = new javax.swing.JMenuItem();
        menuVisual = new javax.swing.JMenu();
        menuAlwaysOnTop = new javax.swing.JCheckBoxMenuItem();
        menuTools = new javax.swing.JMenu();
        menuVMD2CSV = new javax.swing.JMenuItem();
        menuConcat = new javax.swing.JMenuItem();
        menuPlugins = new javax.swing.JMenu();
        menuHelp = new javax.swing.JMenu();
        menuAbout = new javax.swing.JMenuItem();

        setDefaultCloseOperation(javax.swing.WindowConstants.DO_NOTHING_ON_CLOSE);
        setTitle("MIDI 2 Motion");
        addWindowListener(new java.awt.event.WindowAdapter() {
            public void windowClosing(java.awt.event.WindowEvent evt) {
                formWindowClosing(evt);
            }
        });
        getContentPane().setLayout(new java.awt.GridBagLayout());

        tabPane.setEnabled(false);
        tabPane.setMinimumSize(new java.awt.Dimension(200, 140));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 4;
        gridBagConstraints.gridwidth = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weightx = 0.1;
        gridBagConstraints.weighty = 0.1;
        getContentPane().add(tabPane, gridBagConstraints);

        jLabel1.setText("midi");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        getContentPane().add(jLabel1, gridBagConstraints);

        textMidiFile.setEditable(false);
        textMidiFile.setText("まだ");
        textMidiFile.setToolTipText(".mid をドロップすると読み込みます.");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 0.1;
        getContentPane().add(textMidiFile, gridBagConstraints);

        jLabel2.setText("fps");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        getContentPane().add(jLabel2, gridBagConstraints);

        textFPS.setText("30");
        textFPS.setToolTipText("書き換えられます");
        textFPS.setMinimumSize(new java.awt.Dimension(28, 28));
        textFPS.addFocusListener(new java.awt.event.FocusAdapter() {
            public void focusLost(java.awt.event.FocusEvent evt) {
                textFPSFocusLost(evt);
            }
        });
        textFPS.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                textFPSActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.LINE_START;
        getContentPane().add(textFPS, gridBagConstraints);

        textLength.setEditable(false);
        textLength.setText("秒");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 0.1;
        getContentPane().add(textLength, gridBagConstraints);

        textTemp.setEditable(false);
        textTemp.setText("bpm");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        getContentPane().add(textTemp, gridBagConstraints);

        textBeat.setEditable(false);
        textBeat.setText("beat");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        getContentPane().add(textBeat, gridBagConstraints);

        textFPB.setEditable(false);
        textFPB.setText("frame/beat");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        getContentPane().add(textFPB, gridBagConstraints);

        menuFile.setMnemonic(KeyEvent.VK_F);
        menuFile.setText("ファイル(F)");

        menuOpen.setMnemonic(KeyEvent.VK_O);
        menuOpen.setText("MIDIを開く(O)");
        menuOpen.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                menuOpenActionPerformed(evt);
            }
        });
        menuFile.add(menuOpen);

        menuClose.setMnemonic(KeyEvent.VK_X);
        menuClose.setText("閉じる(X)");
        menuClose.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                menuCloseActionPerformed(evt);
            }
        });
        menuFile.add(menuClose);

        menuBar.add(menuFile);

        menuVisual.setText("表示(V)");

        menuAlwaysOnTop.setText("常に手前に表示(L)");
        menuAlwaysOnTop.setToolTipText("常にどのウィンドウよりも上");
        menuAlwaysOnTop.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                menuAlwaysOnTopActionPerformed(evt);
            }
        });
        menuVisual.add(menuAlwaysOnTop);

        menuBar.add(menuVisual);

        menuTools.setMnemonic(KeyEvent.VK_T);
        menuTools.setText("ツール(T)");

        menuVMD2CSV.setMnemonic(KeyEvent.VK_V);
        menuVMD2CSV.setText("VMD2CSV(V)");
        menuVMD2CSV.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                menuVMD2CSVActionPerformed(evt);
            }
        });
        menuTools.add(menuVMD2CSV);

        menuConcat.setMnemonic(KeyEvent.VK_C);
        menuConcat.setText("MIDI Concatenate(C)");
        menuConcat.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                menuConcatActionPerformed(evt);
            }
        });
        menuTools.add(menuConcat);

        menuBar.add(menuTools);

        menuPlugins.setText("プラグイン(P)");
        menuBar.add(menuPlugins);

        menuHelp.setMnemonic(KeyEvent.VK_H);
        menuHelp.setText("ヘルプ(H)");

        menuAbout.setMnemonic(KeyEvent.VK_A);
        menuAbout.setText("MID2Motion について(A)");
        menuAbout.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                menuAboutActionPerformed(evt);
            }
        });
        menuHelp.add(menuAbout);

        menuBar.add(Box.createHorizontalGlue());

        menuBar.add(menuHelp);

        setJMenuBar(menuBar);

        pack();
    }// </editor-fold>//GEN-END:initComponents

    private void menuOpenActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_menuOpenActionPerformed
        JFileChooser jfc=new JFileChooser(midiDir);
        jfc.setFileFilter(new FileNameExtensionFilter("MIDI", "mid"));
        if(jfc.showOpenDialog(this)!=JFileChooser.APPROVE_OPTION)
            return;
        File f=jfc.getSelectedFile();
        openMidiFile(f);
    }//GEN-LAST:event_menuOpenActionPerformed

    private void openMidiFile(File midiFile){
        midiDir=midiFile.getParentFile();
        try {
            midiR=new MidiResolver(midiFile);
            if(midiR!=null){
                textMidiFile.setText(midiFile.getName());
                textLength.setText(((double)midiR.getMidiSeq().getMicrosecondLength()/1000000)+"秒");
                textTemp.setText(((double)60000000/midiR.getTemp())+" bpm");
                textBeat.setText(((double)midiR.getMidiSeq().getTickLength()/midiR.getReso())+" beats");
                setFPS();
                tabPane.setEnabled(true);
                for(MotionPane mp:mps){
                    mp.setMidiResolver(midiR);
                }
            }
        } catch(IOException ex){
            JOptionPane.showMessageDialog(this, "読み込みIOエラー\nfile: "+midiFile.toString()+"\nerr: "+ex.getMessage());
        } catch (InvalidMidiDataException ex){
            JOptionPane.showMessageDialog(this, "異常なMIDIデータ\nfile: "+midiFile.toString()+"\nerr: "+ex.getMessage());
        }
    }

    @Override
    public void actionPerformed(ActionEvent e){
        String fname=pluginReadme.get((JMenuItem)e.getSource());
        String zipName=fname.substring(0, fname.indexOf('/'));
        String readme=fname.substring(fname.indexOf('/')+1);

        JDialog dialog=new JDialog(this);
        dialog.setTitle(readme+" in "+zipName);
        ScrollPane sp=new ScrollPane();
        dialog.add(sp);
        JTextArea jtext=new JTextArea();
        sp.setSize(450, 200);
        sp.add(jtext);

        try {
            ZipFile zip = new ZipFile("plugin/"+zipName);
            ZipEntry ze=zip.getEntry(readme);
            InputStream is=zip.getInputStream(ze);
            BufferedReader br=new BufferedReader(new InputStreamReader(is,"MS932"));
            while(br.ready()){
                jtext.append(br.readLine()+"\n");
            }
            is.close();
            zip.close();
            dialog.setSize(dialog.getPreferredSize());
            dialog.setVisible(true);
        } catch (IOException ex) {
            JOptionPane.showMessageDialog(this,"Zipファイル"+zipName+"が読み込めません: "+ex);
        }
    }

    private void menuCloseActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_menuCloseActionPerformed
        confirmClose();
    }//GEN-LAST:event_menuCloseActionPerformed

    private void confirmClose(){
        int i=JOptionPane.showConfirmDialog(this,"終了していいですか","MIDI2Motion",JOptionPane.YES_NO_OPTION);
        if(i!=JOptionPane.OK_OPTION){
            return;
        }
        FileOutputStream fos;
        Properties p=new Properties();
        MmdDialog.saveDefault();

        String key;
        for(Map.Entry<Object,Object> e:System.getProperties().entrySet()){
            key=(String)e.getKey();
            if(key.startsWith("midiMotion.") || key.startsWith("MMDMotion.")){
                p.setProperty(key, (String)e.getValue());
            }
        }
        try {
            p.setProperty("midiMotion.midiDir", midiDir.getCanonicalPath());
            fos = new FileOutputStream("midiMotion.properties");
            p.store(fos, "MidiMotion Setting File");
            fos.close();
        } catch (FileNotFoundException ex) {
        } catch (IOException ex) {
            JOptionPane.showMessageDialog(this,"IOエラー:設定ファイルの書き込みに失敗\n"+ex.getLocalizedMessage());
        }
        setVisible(false);
        dispose();
    }

    private void menuVMD2CSVActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_menuVMD2CSVActionPerformed
        VMD2CSV v2c=new VMD2CSV();
        v2c.setVisible(true);
    }//GEN-LAST:event_menuVMD2CSVActionPerformed

    private void textFPSActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_textFPSActionPerformed
        setFPS();
    }//GEN-LAST:event_textFPSActionPerformed

    private void textFPSFocusLost(java.awt.event.FocusEvent evt) {//GEN-FIRST:event_textFPSFocusLost
        setFPS();
    }//GEN-LAST:event_textFPSFocusLost

    private void formWindowClosing(java.awt.event.WindowEvent evt) {//GEN-FIRST:event_formWindowClosing
        confirmClose();
    }//GEN-LAST:event_formWindowClosing

    private void menuAboutActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_menuAboutActionPerformed
         new AboutDialog(this,true,this.getClass(),
                 "MIDI to Motion は Standard MIDI File (.mid) から "
                         + "MMDモーションを生成する JavaライブラリーのGUIです。"
         ).show("MIDI to Motion");
    }//GEN-LAST:event_menuAboutActionPerformed

    private void menuConcatActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_menuConcatActionPerformed
        new MidiConcat().setVisible(true);
    }//GEN-LAST:event_menuConcatActionPerformed

    private void menuAlwaysOnTopActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_menuAlwaysOnTopActionPerformed
        setAlwaysOnTop(menuAlwaysOnTop.isSelected());
    }//GEN-LAST:event_menuAlwaysOnTopActionPerformed

    /**
     * {@link FileDropTarget} から呼ばれる{@link FileDropListener}のメソッド.
     * @param e {@link FileDropEvent#getSource()} でドロップされた swing/awt
     * コンポーネント. {@link FileDropEvent#getFiles()}でドロップで送られてきた
     * ファイルのリストを入手することができる.
     */
    @Override
    public void fileDrop(FileDropEvent e) {
        Object src=e.getSource();
        File file=e.getFiles()[0];
        if(src==textMidiFile){
            openMidiFile(file);
        }
    }

    /**
     * main method (初実行用)
     * @param args コマンドオプションは, 最初のargをMIDI ファイルとみなす.
     */
    public static void main(final String args[]) {
        /* Set the Nimbus look and feel */
        //<editor-fold defaultstate="collapsed" desc=" Look and feel setting code (optional) ">
        /* If Nimbus (introduced in Java SE 6) is not available, stay with the default look and feel.
         * For details see http://download.oracle.com/javase/tutorial/uiswing/lookandfeel/plaf.html 
         */
        try {
            for (javax.swing.UIManager.LookAndFeelInfo info : javax.swing.UIManager.getInstalledLookAndFeels()) {
                if ("Nimbus".equals(info.getName())) {
                    javax.swing.UIManager.setLookAndFeel(info.getClassName());
                    break;
                }
            }
        } catch (ClassNotFoundException ex) {
            java.util.logging.Logger.getLogger(MidiMotionFrame.class.getName()).log(java.util.logging.Level.SEVERE, null, ex);
        } catch (InstantiationException ex) {
            java.util.logging.Logger.getLogger(MidiMotionFrame.class.getName()).log(java.util.logging.Level.SEVERE, null, ex);
        } catch (IllegalAccessException ex) {
            java.util.logging.Logger.getLogger(MidiMotionFrame.class.getName()).log(java.util.logging.Level.SEVERE, null, ex);
        } catch (javax.swing.UnsupportedLookAndFeelException ex) {
            java.util.logging.Logger.getLogger(MidiMotionFrame.class.getName()).log(java.util.logging.Level.SEVERE, null, ex);
        }
        //</editor-fold>
        /* Create and display the form */
        java.awt.EventQueue.invokeLater(new Runnable() {
            @Override
            public void run() {
                MidiMotionFrame mmf=new MidiMotionFrame();
                if(args.length>=1){
                    File f=new File(args[0]);
                    if(f.exists())
                        mmf.openMidiFile(f);
                }
                mmf.setVisible(true);
            }
        });
    }

    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JMenuItem menuAbout;
    private javax.swing.JCheckBoxMenuItem menuAlwaysOnTop;
    private javax.swing.JMenuBar menuBar;
    private javax.swing.JMenuItem menuClose;
    private javax.swing.JMenuItem menuConcat;
    private javax.swing.JMenu menuFile;
    private javax.swing.JMenu menuHelp;
    private javax.swing.JMenuItem menuOpen;
    private javax.swing.JMenu menuPlugins;
    private javax.swing.JMenu menuTools;
    private javax.swing.JMenuItem menuVMD2CSV;
    private javax.swing.JMenu menuVisual;
    private javax.swing.JTabbedPane tabPane;
    private javax.swing.JTextField textBeat;
    private javax.swing.JTextField textFPB;
    private javax.swing.JTextField textFPS;
    private javax.swing.JTextField textLength;
    private javax.swing.JTextField textMidiFile;
    private javax.swing.JTextField textTemp;
    // End of variables declaration//GEN-END:variables
}
