/*
 * The MIT License
 *
 * Copyright 2015 nazo.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package jp.sourceforge.mmd.motion.model;

import java.io.FileInputStream;
import java.io.IOException;
import jp.sfjp.mikutoga.bin.parser.MmdFormatException;
import jp.sourceforge.mmd.motion.geo.Matrix;
import jp.sourceforge.mmd.motion.geo.Vector3D;
import jp.sourceforge.mmd.motion.BonePose;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import static org.junit.Assert.*;

/**
 *
 * @author nazo
 */
public class ModelTest {
    static private double delta=1e-12;
    private  Model model;
    public ModelTest() {
    }
    
    @Before
    public void setUp() throws IOException{
        try {
            FileInputStream fis=new FileInputStream("target/test-classes/coordinate.pmx");
            model=Model.fromPMX(fis);
            fis.close();
        } catch (MmdFormatException ex) {
            System.err.println("Test Model file is broken");
        }
    }
    
    @After
    public void tearDown() {
        model=null;
    }

    @Test
    public void modelPropertyTest() {
        assertEquals("Coordinate",model.getName());
        assertNull("Non existent",model.get("abc"));
        assertNotNull(model.get("センター"));
        assertNotNull(model.get("原点1"));
    }

    @Test
    public void boneTranslationTest(){
        model.resetChanged();
        Bone c=model.get("センター");
        Bone d=model.get("原点1");
        assertArrayEquals(new Vector3D().toDouble(),d.getPos().toDouble(),delta);// (0,0,0)
        d.translationTo(new Vector3D(2,3,4));
        assertEquals(1,model.getChanged().length);
        assertArrayEquals(new Vector3D(2,3,4).toDouble(),d.getPos().toDouble(),delta);// (2,3,4)
        c.translationTo(new Vector3D(1,1,1));
        assertArrayEquals(new Vector3D(1,1,1).toDouble(),c.getPos().toDouble(),delta);
        assertArrayEquals(new Vector3D(3,4,5).toDouble(),d.getPos().toDouble(),delta);
        d.translationTo(new Vector3D(-1,-1,-1));
        assertArrayEquals(new Vector3D().toDouble(),d.getPos().toDouble(),delta);// (2,3,4)
        assertEquals(2,model.getChanged().length);
    }

    @Test
    public void boneRotation1Test(){
        Bone c=model.get("センター");
        Bone d=model.get("原点1");

        d.translationTo(new Vector3D(0,0,1));
        assertArrayEquals(new Matrix().toDouble(), d.getMatrix().toDouble(), delta);
        assertArrayEquals(new Matrix().toDouble(), d.getGMatrix().toDouble(), delta);
        assertArrayEquals(new Vector3D(0,0,1).toDouble(), d.getPos().toDouble(), delta);

        c.rotate(Matrix.rotation(0, 90, 0));
        assertArrayEquals(new Matrix().toDouble(), d.getMatrix().toDouble(), delta);
        assertArrayEquals(Matrix.rotation(0,90,0).toDouble(), d.getGMatrix().toDouble(), delta);
        assertArrayEquals(new Vector3D(-1,0,0).toDouble(), d.getPos().toDouble(), delta);

        c.translationTo(new Vector3D(0,0,1));
        assertArrayEquals(new Vector3D(-1,0,1).toDouble(), d.getPos().toDouble(), delta);

        d.rotate(Matrix.rotation(0, -90, 0));
        assertArrayEquals(Matrix.rotation(0, -90, 0).toDouble(), d.getMatrix().toDouble(), delta);
        assertArrayEquals(new Matrix().toDouble(), d.getGMatrix().toDouble(), delta);
        assertArrayEquals(new Vector3D(-1,0,1).toDouble(), d.getPos().toDouble(), delta);

        d.translationTo(new Vector3D(0,0,0));
        assertArrayEquals(new Vector3D(0,0,1).toDouble(), d.getPos().toDouble(), delta);

        d.rotateToG(Matrix.rotation(30,0,60));
        assertArrayEquals(Matrix.rotation(30, 0, 60).toDouble(), d.getGMatrix().toDouble(), delta);
        assertArrayEquals(Matrix.rotation(30, -90, 60).toDouble(), d.getMatrix().toDouble(), delta);
    }

    @Test
    public void boneRotation2Test(){
        Bone c=model.get("センター");
        Bone d=model.get("原点1");

        d.translationTo(new Vector3D(0,0,1));

        c.rotate(new Vector3D(0, -1, 0),0);
        assertArrayEquals(new Matrix().toDouble(), d.getMatrix().toDouble(), delta);
        assertArrayEquals(Matrix.rotation(0,90,0).toDouble(), d.getGMatrix().toDouble(), delta);
        assertArrayEquals(new Vector3D(-1,0,0).toDouble(), d.getPos().toDouble(), delta);

        c.translationTo(new Vector3D(0,0,1));
        assertArrayEquals(new Vector3D(-1,0,1).toDouble(), d.getPos().toDouble(), delta);

        d.rotate(new Vector3D(1, 0, 0),0);
        assertArrayEquals(Matrix.rotation(0,  0, 90).toDouble(), d.getMatrix().toDouble(), delta);
        assertArrayEquals(Matrix.rotation(0, 90, 90).toDouble(), d.getGMatrix().toDouble(), delta);
        assertArrayEquals(new Vector3D(-1,0,1).toDouble(), d.getPos().toDouble(), delta);

        d.translationTo(new Vector3D(1,0,0));
        assertArrayEquals(new Vector3D(0,0,2).toDouble(), d.getPos().toDouble(), delta);

        d.rotateToG(Matrix.rotation(30,0,60));
        assertArrayEquals(Matrix.rotation(30, 0, 60).toDouble(), d.getGMatrix().toDouble(), delta);
        assertArrayEquals(Matrix.rotation(30, -90, 60).toDouble(), d.getMatrix().toDouble(), delta);
    }

    @Test
    public void bonePoseTest(){
        Bone c=model.get("センター");
        Bone d=model.get("原点1");

        d.translationTo(new Vector3D(0,0,1));
        c.rotate(new Vector3D(0, -1, 0),0);

        BonePose bp=d.getPose();
        assertNotNull(bp);
        assertEquals(d.getName(),bp.nameOfBone);
        assertArrayEquals(bp.v.toDouble(),new double[] {0,0,1},1e-9);
        assertArrayEquals(bp.mr.toDouble(),new Matrix().toDouble(),1e-9);

        c.translationTo(new Vector3D(0,0,1));
        d.rotate(new Vector3D(1, 0, 0),0);
        d.translationTo(new Vector3D(1,0,0));
        assertArrayEquals(Matrix.rotation(0, 0, 90).toDouble(), d.getMatrix().toDouble(), delta);
        assertArrayEquals(Matrix.rotation(0,90, 90).toDouble(), d.getGMatrix().toDouble(), delta);
        assertArrayEquals(new Vector3D(0,0,2).toDouble(), d.getPos().toDouble(), delta);

        d.setPose(bp);
        assertArrayEquals(Matrix.rotation(0, 90, 0).toDouble(), d.getGMatrix().toDouble(), delta);
        assertArrayEquals(Matrix.rotation(0, 0, 0).toDouble(), d.getMatrix().toDouble(), delta);
        assertArrayEquals(new Vector3D(-1,0, 1.0).toDouble(), d.getPos().toDouble(), delta);
    }
}
