/*
 * The MIT License
 *
 * Copyright 2015 nazo.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package jp.sourceforge.mmd.motion.model;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Set;
import java.util.TreeMap;
import jp.sourceforge.mmd.motion.BonePose;
import jp.sourceforge.mmd.motion.Pose;
import jp.sfjp.mikutoga.bin.parser.MmdFormatException;
import jp.sfjp.mikutoga.pmd.parser.PmdParser;
import jp.sfjp.mikutoga.pmx.parser.PmxParser;
import jp.sourceforge.mmd.motion.MorphPose;

/**
 * MMD モデルの記録用.
 * また、このクラスは大体{@link Bone}と{@link Morph}のコンテナである.
 * いろいろ複雑なのでコンストラクターは{@code protected} にして、モデルの構築は
 * {@link #fromCSV(InputStream)}や{@link #fromPMD(InputStream)}を使ってでしか
 * 構築できないようにしてある.
 * @author nazo
 */
public class Model {
    private String name;
    private TreeMap<String,Bone> boneMap;
    private TreeMap<String,Morph> morphMap;

    /**
     * 空のモデルコンテナを用意する. モデル名は無い.
     */
    protected Model(){
        super();
        boneMap=new TreeMap<String, Bone>();
        morphMap=new TreeMap<String, Morph>();
    }

    /** CSV テキスト(複数行)からモデルを構築する.
     * CSV は pmx editor で生成するフォーマット.
     * @param csv CSVデーターのInputStream
     * @return 構築されたモデル
     * @throws IOException 途中で読み込みに失敗したなど
     * @throws MmdFormatException CSVフォーマットとして異常な時
     */
    static public Model fromCSV(InputStream csv) throws IOException,MmdFormatException{
        BufferedReader br=null;
        String line;
        int i,j;
        int bid=0;
        int mid=0;
        Bone tb;
        Morph tm;
        try{
            br=new BufferedReader(new InputStreamReader(csv,"MS932"));
        }catch(UnsupportedEncodingException ex){
            System.err.println("Syntax error in Model.java:"+ex.getMessage());
            System.exit(-1);
        }
        Model ret=new Model();
        while(br.ready()){
            line=br.readLine();
            if(line.startsWith("Bone,")){
                tb=Bone.fromCSV(ret,line);
                tb.id=bid;
                bid++;
                if(tb!=null){
                    ret.put(tb);
                }
            } else if(line.startsWith("Morph,")){
                tm=Morph.fromCSV(ret,line);
                tm.id=mid;
                mid++;
                if(tm!=null){
                    ret.put(tm);
                }
            } else if(line.startsWith("BoneMorph,") || line.startsWith("GroupMorph,")){
                Morph.fromCSVList(ret,line);
            } else if(line.startsWith("ModelInfo,")){
                i=line.indexOf(',');
                j=line.indexOf(',',i+1);
                ret.name=line.substring(i+2, j-1);
            }
        }
        ret.resetChanged();
        return ret;
    }

    /** PMDからモデルを構築する.
     * @param is PMDのInputStream
     * @return 構築されたモデル
     * @throws IOException 途中で読み込みに失敗したなど
     * @throws MmdFormatException PMDフォーマットとして異常な時
     */
    static public Model fromPMD(InputStream is) throws IOException, MmdFormatException{
        Model mo=new Model();
        PmdParser pmdp=new PmdParser(is);
        PmdFileHander parser = new PmdFileHander(mo);

        pmdp.setBasicHandler(parser);
        pmdp.setBoneHandler(parser);
        pmdp.setMorphHandler(parser);

        pmdp.parsePmd();

        return mo;
    }

    /** PMXからモデルを構築する.
     * @param is PMXのInputStream
     * @return 構築されたモデル
     * @throws IOException 途中で読み込みに失敗したなど
     * @throws MmdFormatException PMXフォーマットとして異常な時
     */
    static public Model fromPMX(InputStream is) throws IOException, MmdFormatException{
        Model mo=new Model();
        PmxParser pmxp=new PmxParser(is);
        PmxFileHander parser = new PmxFileHander(mo);

        pmxp.setBasicHandler(parser);
        pmxp.setBoneHandler(parser);
        pmxp.setMorphHandler(parser);

        pmxp.parsePmx();

        return mo;
    }

    /**
     * ボーン名から Bone を取得する
     * @param name ボーン名
     * @return その名前のBone. 見つからないときは {@code null}.
     */
    public Bone get(String name){
        return boneMap.get(name);
    }

    /**
     * モーフ名から Morph を取得する
     * @param name モーフ名
     * @return その名前のMorph. 見つからないときは {@code null}.
     */
    public Morph getMorph(String name){
        return morphMap.get(name);
    }

    /**
     * Boneを追加する
     * @param b 追加するボーンオブジェクト.
     * 必ず{@link #setName(java.lang.String)}で名前をつけること. null 不可.
     */
    protected void put(Bone b){
        boneMap.put(b.getName(), b);
    }

    /**
     * Morphを追加する
     * @param m 追加するモーフオブジェクト.
     * 必ず{@link #setName(java.lang.String)}で名前をつけること. null 不可.
     */
    protected void put(Morph m){
        morphMap.put(m.getName(), m);
    }

    /**
     * ボーンオブジェクトのリストを返す.
     * @return ボーンのリスト. ボーン名のユニコード順.(大体五十音順)
     */
    public Collection<Bone> values(){
        return boneMap.values();
    }

    /**
     * ボーン名のセットを返す.
     * @return Bone のセット
     */
    public Set<String> keys(){
        return boneMap.keySet();
    }
    
    /**
     * モーフオブジェクトのリストを返す.
     * @return モーフのリスト. モーフ名のユニコード順.(大体五十音順)
     */
    public Collection<Morph> morphValues(){
        return morphMap.values();
    }

    /**
     * モーフ名のセットを返す.
     * @return Bone のセット
     */
    public Set<String> morphKeys(){
        return morphMap.keySet();
    }
    
    /**
     * モデル名を返す.
     * @return モデル名
     */
    public String getName() {
        return name;
    }

    /**
     * モデル名を設定する.
     * @param name モデル名. {@code null}はだめ.
     */
    public void setName(String name) {
        this.name = name;
    }
    
    /**
     * ポーズを設定する(フレーム番は無視).
     * @param mof モーションのアレイ. null でも可能.
     */
    public void setPoses(Pose [] mof){
        if(mof==null)return;
        Bone b;
        Morph m;
        for(Pose p:mof){
            if(p instanceof BonePose){
                if((b=get(p.nameOfBone))!=null){
                    b.setPose((BonePose)p);
                }
            }
            if(p instanceof MorphPose){
                if((m=getMorph(p.nameOfBone))!=null){
                    m.setPose((MorphPose)p);
                }
            }
        }
    }

    /**
     * 変更があったボーンのポーズを取得する. {@link #resetChanged()} から後の変更が適応される.
     * @return 変更があったBone の Pose データ集.
     */
    public Pose [] getChanged(){
        ArrayList<Pose> al=new ArrayList<Pose>();
        for(Morph m:morphMap.values()){
            if(m.getChanged()){
                al.add(m.getPose());
            }
        }
        for(Bone b:boneMap.values()){
            if(b.getChanged()){
                al.add(b.getPose());
            }
        }
        return al.toArray(new Pose[al.size()]);
    }

    /**
     * モデルに属する全ての変更フラグをリセットする.
     * モデル読み込み時には、全てのBone にフラグがたっているので、
     * 読み込み後に良く使う.
     */
    public void resetChanged(){
        for(Bone b:boneMap.values()){
            b.resetChanged();
        }
    }
}
