/*
 * The MIT License
 *
 * Copyright 2015 nazo.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package jp.sourceforge.mmd.motion.geo;

/**
 * 3次元ベクトルクラス.
 * @author nazo
 */
public class Vector3D implements Cloneable {
    /** 単位X. (1,0,0) */
    final static public Vector3D unitX=new Vector3D(1,0,0);
    /** 単位Y. (0,1,0) */
    final static public Vector3D unitY=new Vector3D(0,1,0);
    /** 単位Z. (0,0,1) */
    final static public Vector3D unitZ=new Vector3D(0,0,1);

    /** 3成分 */
    protected double [] r=new double[3];

    /** 初期値(0,0,0) */
    public Vector3D(){
        r[0]=0;
        r[1]=0;
        r[2]=0;
    }

    /** 初期値(v[0],v[1],v[2]).
     * @param v (v[0],v[1],v[2]) だけみる. {@code null} はだめ.
     */
    public Vector3D(double [] v){
        this.set(v);
    }

    /** 初期値(x,y,z).
     * @param x 第1成分
     * @param y 第2成分
     * @param z 第3成分
     */    
    public Vector3D(double x,double y,double z){
        this.set(x, y, z);
    }

    /**
     * ベクトルのクローン.
     * @param b クローン元
     */
    public Vector3D(Vector3D b){
        this.set(b.r);
    }

    /**
     * 3成分の設定
     * @param v (v[0],v[1],v[2]) だけみる. {@code null} はだめ.
     */
    public final void set(double []v){
        r[0]=v[0];
        r[1]=v[1];
        r[2]=v[2];
    }

    /**
     * 3成分の設定
     * @param x 第1成分
     * @param y 第2成分
     * @param z 第3成分
     */
    public final void set(double x,double y,double z){
        r[0]=x;
        r[1]=y;
        r[2]=z;        
    }

    /**
     * 等しいかを確認する. 
     * @param e 比較先. {@code null} はだめ.
     * @return 3成分全てが等しい時に {@code true}.
     */
    public boolean equals(Vector3D e){
        return (r[0]==e.r[0]) && (r[1]==e.r[1]) && (r[2]==e.r[2]);
    }
    
    /**
     * ベクトルのクローン.
     * @return クローンされたベクトル.
     */
    @Override
    @SuppressWarnings({"CloneDoesntCallSuperClone", "CloneDeclaresCloneNotSupported"})
    public Vector3D clone() {
        return new Vector3D(this);
    }
    
    /** ベクトル加算.
     * @param b 加算するベクトル.
     * @return 加算された新しいベクトル.
     */
    public Vector3D add(Vector3D b){
        return new Vector3D(r[0]+b.r[0],r[1]+b.r[1],r[2]+b.r[2]);
    }

    /** ベクトル減算.
     * @param b 減算するベクトル.
     * @return 減算された新しいベクトル.
     */
    public Vector3D sub(Vector3D b){
        return new Vector3D(r[0]-b.r[0],r[1]-b.r[1],r[2]-b.r[2]);
    }

    /** 反ベクトル. 反対向きのベクトル.
     * @return 新しい反ベクトル.
     */
    public Vector3D inverse(){
        return new Vector3D(-r[0],-r[1],-r[2]);
    }

    /** 割り算. ノーマライズ(正規化)に使う.
     * @param d 割るスカラー
     * @return 新しいベクトル.
     */
    public Vector3D divide(double d){
        return new Vector3D(r[0]/d,r[1]/d,r[2]/d);
    }

    /**
     * スカラー掛け算. 
     * @param m スカラー
     * @return 新しいベクトル
     */
    public Vector3D times(double m){
        return new Vector3D(r[0]*m,r[1]*m,r[2]*m);
    }

    /** ドット積(内積).
     * @param b 積にするベクトル
     * @return ドット積されたスカラー.
     */
    public double times(Vector3D b){
        return r[0]*b.r[0]+r[1]*b.r[1]+r[2]*b.r[2];
    }

    /** クロス積(外積).
     * @param b 積にするベクトル.
     * @return クロス積された新しいベクトル.
     */
    public Vector3D cross(Vector3D b){
        return new Vector3D(
        r[1]*b.r[2]-r[2]*b.r[1],
        r[2]*b.r[0]-r[0]*b.r[2],
        r[0]*b.r[1]-r[1]*b.r[0]);
    }

    /**
     * ノルム(絶対値)
     * @return ノルムのスカラー
     */
    public double norm(){
        return Math.sqrt(this.norm2());
    }

    /**
     * ノルムの2乗.
     * ノルムの計算より早く算出されるので, 2乗でもいい場合は使用する.
     * @return ノルム2乗
     */    
    public double norm2(){
        return r[0]*r[0]+r[1]*r[1]+r[2]*r[2];
    }

    /** ベクトル周りの回転をする.
     * @param axis_sin 回転軸でノルムが回転分のsine 成分を持つもの.
     *                  よってノルムは, 0以上, 1以下でなければならない.
     * @param c cosine 成分のスカラー -1以上, 1以下でなければならない.
     * 　　　　　　　　　またaxis_sin のノルムと対応しなければ、挙動は保障されない.
     * @return 回転後のベクトル.
     */
    public Vector3D rotate_vector(Vector3D axis_sin,double c){
        double norm2=axis_sin.norm2();
        if(norm2==0){
            return new Vector3D(this);
        }
        double ns=axis_sin.times(this);
        Vector3D nn=axis_sin.times(ns/norm2);

        Vector3D ca=axis_sin.cross(this);
        return this.sub(nn).times(c).add(nn).add(ca);
    }

    /**
     * CSV 化文字列
     * @return CSV String
     */
    @Override
    public String toString(){
        return Double.toString(r[0])+","+r[1]+","+r[2];
    }

    /**
     * double [3]をえる.
     * @return 3成分が, double [3] で帰ってくる.
     */
    public double []  toDouble(){
        double [] ret=new double[3];
        ret[0]=r[0];
        ret[1]=r[1];
        ret[2]=r[2];
        return ret;
    }
}
