/*
 * The MIT License
 *
 * Copyright 2015 nazo.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package jp.sourceforge.mmd.motion.model;

import java.util.ArrayList;
import java.util.Map;
import java.util.TreeMap;
import jp.sourceforge.mmd.motion.geo.Matrix;
import jp.sourceforge.mmd.motion.geo.Vector3D;
import jp.sfjp.mikutoga.bin.parser.MmdFormatException;
import jp.sfjp.mikutoga.bin.parser.ParseStage;
import jp.sfjp.mikutoga.pmx.parser.PmxBasicHandler;
import jp.sfjp.mikutoga.pmx.parser.PmxBoneHandler;
import jp.sfjp.mikutoga.pmx.BoneFlags;

/**
 * PMX データを読み出すのに使用するハンドラ.
 * @author nazo
 */
public class PmxFileHander implements PmxBoneHandler,PmxBasicHandler{
    static protected Vector3D absoluteZ=Vector3D.unitZ;

    protected Model model;
    protected Bone bone;
    protected short boneFlags;
    protected ArrayList<String> idToName;

    protected TreeMap<Integer,Integer> childToParent;
    protected TreeMap<Integer,Number []> childToLinkParent;
    protected TreeMap<Integer,Integer> parentToTail;
    protected TreeMap<Integer,Vector3D> parentToOffset;
    
    public PmxFileHander(Model model){
        super();
        this.model=model;
    }

    @Override
    public void pmxParseStart() {
        bone=null;
        boneFlags=0;
    }

    @Override
    public void pmxHeaderInfo(float version, int encode, int uv) throws MmdFormatException {
        if(version<2){
            throw new MmdFormatException("Too old PMX file.");
        }
    }

    @Override
    public void pmxModelName(String name, String nameE) {
        model.setName(name);
    }

    @Override
    public void pmxModelDescription(String description, String descriptionE) {
    }

    @Override
    public void loopStart(ParseStage stage, int loops) {
        if(stage==PmxBoneHandler.BONE_LIST){
            idToName=new ArrayList<String>();
            childToParent=new TreeMap<Integer, Integer>();
            childToLinkParent=new TreeMap<Integer, Number[]>();
            parentToTail=new TreeMap<Integer, Integer>();
            parentToOffset=new TreeMap<Integer, Vector3D>();
        }
    }

    @Override
    public void pmxBoneInfo(String name, String nameE) {
        bone=new Bone(model);
        bone.name=name;
        idToName.add(name);
    }

    @Override
    public void pmxBonePosition(float xPos, float yPos, float zPos) {
        bone.gv=new Vector3D(xPos,yPos,zPos);
//        bone.v=new Vector3D();
//        bone.g_mr=new Matrix();
    }

    @Override
    public void pmxBoneStructure(int parentId, int depth) {
        if(parentId==-1){
            bone.parent=null;
        }else {
            childToParent.put(idToName.size()-1,parentId);
        }
    }

    @Override
    public void pmxBoneFlags(short flags) {
        this.boneFlags=flags;
        if(!BoneFlags.LINK_FLAG.check(flags)){
            bone.linkParent=null;
        }
        if(!BoneFlags.AXIS_ROTATE.check(flags)){
            bone.limitRot=null;
        }
        if(!BoneFlags.LOCAL_AXIS.check(flags)){
            bone.mr=new Matrix();
            bone.ini_mr=new Matrix();
        }
    }

    @Override
    public void pmxBoneOffset(float offX, float offY, float offZ) {
        parentToOffset.put(idToName.size()-1, new Vector3D(offX,offY,offZ));
    }

    @Override
    public void pmxBoneArrowhead(int arrowId) {
        parentToTail.put(idToName.size()-1, arrowId);
    }

    @Override
    public void pmxBoneLink(int linkParent, float ratio) throws MmdFormatException {
        childToLinkParent.put(idToName.size()-1, new Number[]{
            linkParent,
            (BoneFlags.MOVE_LINK.check(boneFlags)?ratio:0),
            (BoneFlags.ROTATE_LINK.check(boneFlags)?ratio:0)
        });
    }

    @Override
    public void pmxBoneLocalAxis(float xx, float xy, float xz, float zx, float zy, float zz) {
        bone.mr=new Matrix(new Vector3D(xx, xy, xz),new Vector3D(-zx, -zy, -zz));
        bone.ini_mr=new Matrix(bone.mr);
    }

    @Override
    public void pmxBoneRotateAxe(float x, float y, float z) {
        bone.limitRot=new Vector3D(x,y,z);
    }

    @Override
    public void pmxBoneExtraParent(int extraParent) {
    }

    @Override
    public void pmxBoneIKInfo(int targetId, int depth, float weight) {
    }

    @Override
    public void pmxIKChainInfo(int childId, float[] limit_rotation) {
    }

    @Override
    public void loopNext(ParseStage stage) {
        if(bone != null){
            model.put(bone);
        }
        bone=null;
    }

    @Override
    public void loopEnd(ParseStage stage) throws MmdFormatException {
        if(stage==PmxBoneHandler.BONE_LIST){
            for(Map.Entry<Integer,Integer> e:childToParent.entrySet()){
                Bone child=model.get(idToName.get(e.getKey()));
                child.parent=idToName.get(e.getValue());
                model.get(child.parent).addChild(child);
            }
            for(Map.Entry<Integer,Number[]> e:childToLinkParent.entrySet()){
                Bone child=model.get(idToName.get(e.getKey()));
                Number [] link=e.getValue();
                child.linkParent=idToName.get((Integer)link[0]);
                if(child.linkParent==null)
                    throw new MmdFormatException("Link parent not found. id:"
                            +child.linkParent+" at bone:"+child.name);
                model.get(child.linkParent).addLinkChild(child,
                        (Float)link[1],(Float)link[2]
                );
            }
            for(Map.Entry<Integer,Vector3D> e:parentToOffset.entrySet()){
                String name=idToName.get(e.getKey());
                if(model.get(name+"先")==null){
                    Bone parent=model.get(name);
                    Bone arrowhead=new Bone(model);
                    arrowhead.setName(name+"先");
                    arrowhead.gv=parent.getPos().add(e.getValue());
                    arrowhead.mr=new Matrix();
                    arrowhead.ini_mr=new Matrix();
                    arrowhead.parent=name;
                    parent.addChild(arrowhead);

                    idToName.add(arrowhead.getName());
                    parentToTail.put(e.getKey(),idToName.size()-1);

                    model.put(arrowhead);
                }
            }
            for(Map.Entry<Integer,Integer> e:parentToTail.entrySet()){
                String name=idToName.get(e.getKey());
                if(name.matches("((左|右)(腕|ひじ|手首)|.*(親|人|中|薬|小)指.*)")){
                    Bone parent=model.get(name);
                    if(parent.mr.equals(new Matrix())){ // Local Axis 最優先
                        String tail=idToName.get(e.getValue());
                        Vector3D arrow= model.get(tail).getPos().sub(parent.getPos());
                        arrow=arrow.sub(absoluteZ.times(arrow.times(absoluteZ)));
                        double norm=arrow.norm();
                        if(norm>0){
                            parent.mr=new Matrix(arrow.divide(norm), absoluteZ);
                            parent.ini_mr=new Matrix(parent.mr);
                        }
                    }
                }
            }
        }
    }    
    
    @Override
    public void pmxParseEnd(boolean hasMoreData) {
    }

}
