/*
 * The MIT License
 *
 * Copyright 2015 nazo.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package jp.sourceforge.mmd.motion.model;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Set;
import java.util.TreeMap;
import jp.sourceforge.mmd.motion.BonePose;
import jp.sourceforge.mmd.motion.Pose;
import jp.sfjp.mikutoga.bin.parser.MmdFormatException;
import jp.sfjp.mikutoga.pmd.parser.PmdParser;
import jp.sfjp.mikutoga.pmx.parser.PmxParser;

/**
 * MMD モデルの記録用.
 * また、このクラスは大体{@link Bone}のコンテナである. コンストラクターは
 * {@code protected} なので、{@link #fromCSV(InputStream)}や
 * {@link #fromPMD(InputStream)}を使って構築するしかない.
 * @author nazo
 */
public class Model {
    private String name;
    private TreeMap<String,Bone> namemap;
    private Model(){
        super();
        namemap=new TreeMap<String, Bone>();
    }

    /** CSV テキスト(複数行)からモデルを構築する.
     * CSV は pmx editor で生成するフォーマット.
     * @param csv CSVデーターのInputStream
     * @return 構築されたモデル
     * @throws IOException 途中で読み込みに失敗したなど
     * @throws MmdFormatException CSVフォーマットとして異常な時
     */
    static public Model fromCSV(InputStream csv) throws IOException,MmdFormatException{
        BufferedReader br=null;
        String line;
        int i,j;
        Bone tb;
        try{
            br=new BufferedReader(new InputStreamReader(csv,"MS932"));
        }catch(UnsupportedEncodingException ex){
            System.err.println("Syntax error in Model.java:"+ex.getMessage());
            System.exit(-1);
        }
        Model ret=new Model();
        while(br.ready()){
            line=br.readLine();
            if(line.startsWith("Bone,")){
                tb=Bone.fromCSV(ret,line);
                if(tb!=null){
                    ret.put(tb);
                }
            }else if(line.startsWith("ModelInfo,")){
                i=line.indexOf(',');
                j=line.indexOf(',',i+1);
                ret.name=line.substring(i+2, j-1);
            }
        }
        ret.resetChanged();
        return ret;
    }

    /** PMDからモデルを構築する.
     * @param is PMDのInputStream
     * @return 構築されたモデル
     * @throws IOException 途中で読み込みに失敗したなど
     * @throws MmdFormatException PMDフォーマットとして異常な時
     */
    static public Model fromPMD(InputStream is) throws IOException, MmdFormatException{
        Model mo=new Model();
        PmdParser pmdp=new PmdParser(is);
        PmdFileHander parser = new PmdFileHander(mo);

        pmdp.setBasicHandler(parser);
        pmdp.setBoneHandler(parser);

        pmdp.parsePmd();

        return mo;
    }

    /** PMXからモデルを構築する.
     * @param is PMXのInputStream
     * @return 構築されたモデル
     * @throws IOException 途中で読み込みに失敗したなど
     * @throws MmdFormatException PMXフォーマットとして異常な時
     */
    static public Model fromPMX(InputStream is) throws IOException, MmdFormatException{
        Model mo=new Model();
        PmxParser pmxp=new PmxParser(is);
        PmxFileHander parser = new PmxFileHander(mo);

        pmxp.setBasicHandler(parser);
        pmxp.setBoneHandler(parser);

        pmxp.parsePmx();

        return mo;
    }

    /**
     * ボーン名からBoneを取得する
     * @param name ボーン名
     * @return その名前のBone. 見つからないときは {@code null}.
     */
    public Bone get(String name){
        return namemap.get(name);
    }

    /**
     * Boneを追加する
     * @param b Bone 追加するボーンオブジェクト.
     * 必ず{@link #setName(java.lang.String)}で名前をつけること. null 不可.
     */
    protected void put(Bone b){
        namemap.put(b.getName(), b);
    }

    /**
     * ボーン名のリストを返す.
     * @return ボーン名のリスト. ボーン名のユニコード順.(大体五十音順)
     */
    public Collection<Bone> values(){
        return namemap.values();
    }

    /**
     * Bone のセットを返す.
     * @return Bone のセット
     */
    public Set<String> keys(){
        return namemap.keySet();
    }
    
    /**
     * モデル名を返す.
     * @return モデル名
     */
    public String getName() {
        return name;
    }

    /**
     * モデル名を設定する.
     * @param name モデル名. {@code null}はだめ.
     */
    public void setName(String name) {
        this.name = name;
    }
    
    /**
     * ポーズを設定する(フレーム番は無視).
     * @param mof モーションのアレイ. null でも可能.
     */
    public void setPoses(Pose [] mof){
        if(mof==null)return;
        Bone b;
        for(Pose p:mof){
            if(p instanceof BonePose){
                if((b=get(p.nameOfBone))!=null){
                    b.setPose((BonePose)p);
                }
            }
        }
    }

    /**
     * 変更があったボーンのポーズを取得する. {@link resetChanged} から後の変更が適応される.
     * @return 変更があったBone の Pose データ集.
     */
    public Pose [] getChanged(){
        ArrayList<Pose> al=new ArrayList<Pose>();
        for(Bone b:namemap.values()){
            if(b.getChanged()){
                al.add(b.getPose());
            }
        }
        return al.toArray(new Pose[al.size()]);
    }

    /**
     * モデルに属する全ての変更フラグをリセットする.
     * モデル読み込み時には、全てのBone にフラグがたっているので、
     * 読み込み後に良く使う.
     */
    public void resetChanged(){
        for(Bone b:namemap.values()){
            b.resetChanged();
        }
    }
}
