/*
 * The MIT License
 *
 * Copyright 2015 nazo.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package jp.sourceforge.mmd.motion;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.ArrayList;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import jp.sfjp.mikutoga.bin.parser.MmdFormatException;
import jp.sfjp.mikutoga.vmd.parser.VmdParser;

/**
 * MMD モーションの管理.
 * モーションはフレーム番とボーン名の両方で管理されている.
 * @author nazo
 */
@SuppressWarnings("FieldMayBeFinal")
public class Motion {
    private String modelName;
    private int max_frame;
    private TreeMap<Integer,MoveOnFrame> frame_map;
    private TreeMap<String,MoveOnBone> bone_map;
    private TreeMap<String,MoveOnBone> morph_map;
    private ArrayList<CameraPose> cameraArray;
    private ArrayList<LightPose> lightArray;
    private ArrayList<ShadowPose> shadowArray;
    private ArrayList<BooleanPose> booleanArray;

    /**
     * 空で名前のないモーションをつくる.
     */
    public Motion(){
        modelName=null;
        max_frame=0;
        frame_map = new TreeMap<Integer,MoveOnFrame> ();
        bone_map = new TreeMap<String,MoveOnBone>();
        morph_map = new TreeMap<String,MoveOnBone>();
        cameraArray=new ArrayList<CameraPose>();
        lightArray=new ArrayList<LightPose>();
        shadowArray=new ArrayList<ShadowPose>();
        booleanArray=new ArrayList<BooleanPose>();
    }

    /**
     * 空でモデル名があるモーションをつくる.
     * @param model_name モデル名.
     */
    public Motion(String model_name){
        this();
        this.modelName=model_name;
    }

    /**
     * Poseを登録する.
     * フレーム番と名前は, Pose から読み取られる.
     * @param p 登録するPose
     */
    public void put(Pose p){
        if(p instanceof BonePose){
            MoveOnBone mob=bone_map.get(p.nameOfBone);
            if(mob==null){
                mob=new MoveOnBone(p.nameOfBone);
                bone_map.put(p.nameOfBone, mob);
            }
            mob.put(p);
        }else if(p instanceof MorphPose){
            MoveOnBone mob=morph_map.get(p.nameOfBone);
            if(mob==null){
                mob=new MoveOnBone(p.nameOfBone);
                morph_map.put(p.nameOfBone, mob);
            }
            mob.put(p);
        }else if(p instanceof CameraPose){
            cameraArray.add((CameraPose)p);
        }else if(p instanceof LightPose){
            lightArray.add((LightPose)p);
        }else if(p instanceof ShadowPose){
            shadowArray.add((ShadowPose)p);
        }else if(p instanceof BooleanPose){
            booleanArray.add((BooleanPose)p);
        }

        MoveOnFrame mof=frame_map.get(p.frame);
        if(mof==null){
            mof=new MoveOnFrame(p.frame);
            frame_map.put(p.frame, mof);
            if(p.frame>max_frame){
                max_frame=p.frame;
            }
        }
        mof.put(p);
    }

    /**
     * 複数のPoseを登録する.
     * 名前は, Pose から読み取られる. フレーム番は書き換えられる.
     * @param ps 登録するPoseのアレイ. null でも可.
     * @param frame 登録するフレーム番.
     */
    public void putAll(Pose [] ps,int frame){
        if(ps==null)return;
        for(Pose p:ps){
            p.frame=frame;
            this.put(p);
        }
    }
    
    /**
     * 指定したボーン名のポーズを全て得る.
     * @param bone ボーン名.
     * @return あてはまるポーズのクローンがアレイで帰る. ないときは{@code null}.
     */
    public Pose [] get(String bone){
        if(bone==null)return null;
        MoveOnBone mob=bone_map.get(bone);
        if(mob==null){
            mob=morph_map.get(bone);
            if(mob==null)
                return null;
        }
        return mob.toArray();
    }

    /**
     * 指定したフレームのポーズを全て得る.
     * @param frame フレーム番
     * @return あてはまるポーズのクローンがアレイで帰る. ないときは{@code null}. 
     */
    public Pose [] get(int frame){
        MoveOnFrame mof=frame_map.get(frame);
        if(mof==null)
            return null;
        return mof.toArray();
    }

    /**
     * 指定したフレームの補間ポーズを全て得る.
     * @param frame フレーム番
     * @return 補間したポーズがアレイで帰る. 
     */
    public Pose[] getInterporate(int frame) {
        String key;
        int i=0;
        Pose [] ret=new Pose[bone_map.size()];
        for(MoveOnBone e:bone_map.values()){
            ret[i]=e.getInterporate(frame);
            i++;
        }
        return ret;
    }

    /**
     * ごみ掃除. 無意味なポーズを削除する.
     */
    public void gc(){
        Integer [] list;
        for(Map.Entry<String,MoveOnBone> mob:bone_map.entrySet()){
            list=mob.getValue().gc();
            for(Integer i : list){
                frame_map.get(i).remove(mob.getKey());
            }
        }
    }

    /**
     * CSV化
     * @param os CSV化された文字列の出力先.
     * @throws java.io.IOException 書き込みできないなど.
     */
    public void toCSV(OutputStream os) throws IOException{
        MoveOnBone mob;
        OutputStreamWriter osw=null;
        try {
            osw=new OutputStreamWriter(os,"MS932");
        } catch (UnsupportedEncodingException ex) {
            /* never called. */
        }
        osw.write("Vocaloid Motion Data 0002,0\n"+modelName+"\n");

        int num_bone=0;
        StringBuilder list=new StringBuilder(4096);
        for(Map.Entry<String,MoveOnBone> e:bone_map.entrySet()){
            mob=e.getValue();
            num_bone+=mob.size();
            list.append(mob.toCSV());
        }
        osw.write(num_bone+"\n"+list);

        num_bone=0; //morph
        list=new StringBuilder(4096);
        for(Map.Entry<String,MoveOnBone> e:morph_map.entrySet()){
            mob=e.getValue();
            num_bone+=mob.size();
            list.append(mob.toCSV());
        }
        osw.write(num_bone+"\n"+list);
        
        num_bone=0; // camera
        list=new StringBuilder(4096);
        for(CameraPose e:cameraArray){
            num_bone++;
            list.append(e.toCSV());
        }
        osw.write(num_bone+"\n"+list);

        num_bone=0;// light
        list=new StringBuilder(1024);
        for(LightPose e:lightArray){
            num_bone++;
            list.append(e.toCSV());
        }
        osw.write(num_bone+"\n"+list);

        num_bone=0;// shadow
        list=new StringBuilder(1024);
        for(ShadowPose e:shadowArray){
            num_bone++;
            list.append(e.toCSV());
        }
        osw.write(num_bone+"\n"+list);

        num_bone=0;// boolean
        list=new StringBuilder(1024);
        for(BooleanPose e:booleanArray){
            num_bone++;
            list.append(e.toCSV());
        }
        osw.write(num_bone+"\n"+list);
        osw.flush();
        os.flush();
    }

    /**
     * VMD化.
     * @param os 書き込み先のOutputStream.
     * @throws IOException 書き込み時に不具合が発生したとき.
     */
    public void toVMD(OutputStream os) throws IOException{
        ByteBuffer bbInt=ByteBuffer.allocate(4).order(ByteOrder.LITTLE_ENDIAN);
        byte [] temp;
        int i;
        try {
            os.write("Vocaloid Motion Data 0002".getBytes("MS932"));
        } catch (UnsupportedEncodingException ex) {
            System.err.println("Syntax error in toVMD in Motion class.");
            System.exit(-1);
        }
        os.write(0);
        os.write(bbInt.putInt(0).array());
        temp=modelName.getBytes("MS932");
        os.write(temp);
        for(i=temp.length;i<20;i++){
            os.write(0);
        }

        MoveOnBone mob;// bone
        int number=0;
        ArrayList<byte[]> list=new ArrayList<byte[]>();
        for(Map.Entry<String,MoveOnBone> e:bone_map.entrySet()){
            mob=e.getValue();
            number+=mob.size();
            list.add(mob.toVMD());
        }
        os.write(bbInt.putInt(0,number).array());
        for(byte [] line:list){
            os.write(line);
        }

        number=0;// morph
        list.clear();
        for(Map.Entry<String,MoveOnBone> e:morph_map.entrySet()){
            mob=e.getValue();
            number+=mob.size();
            list.add(mob.toVMD());
        }
        os.write(bbInt.putInt(0,number).array());
        for(byte [] line:list){
            os.write(line);
        }

        number=0;// camera
        list.clear();
        for(CameraPose e:cameraArray){
            number++;
            list.add(e.toVMD());
        }
        os.write(bbInt.putInt(0,number).array());
        for(byte [] line:list){
            os.write(line);
        }
        
        number=0;// light
        list.clear();
        for(LightPose e:lightArray){
            number++;
            list.add(e.toVMD());
        }
        os.write(bbInt.putInt(0,number).array());
        for(byte [] line:list){
            os.write(line);
        }

        number=0;// shadow
        list.clear();
        for(ShadowPose e:shadowArray){
            number++;
            list.add(e.toVMD());
        }
        os.write(bbInt.putInt(0,number).array());
        for(byte [] line:list){
            os.write(line);
        }

        number=0;// boolean
        list.clear();
        for(BooleanPose e:booleanArray){
            number++;
            list.add(e.toVMD());
        }
        os.write(bbInt.putInt(0,number).array());
        for(byte [] line:list){
            os.write(line);
        }
        os.flush();
    }

    /**
     * CSVからMotion を追加する.
     * @param is CSVのInputStream
     * @return 追加後のモーション. {@code this}と同じ.
     * @throws IOException 書き込み時に不具合が発生したとき.
     * @throws MmdFormatException MMDモーションのフォーマットじゃないとき.
     */
    public Motion fromCSV(InputStream is)throws IOException,MmdFormatException{
        return fromCSV(is,0);
    }

    /**
     * CSVからMotion をフレーム番ずらして追加する.
     * @param is CSVのInputStream
     * @param offsetF ずらすオフセット. 負数は推奨されない.
     * @return 追加後のモーション. {@code this}と同じ.
     * @throws IOException 書き込み時に不具合が発生したとき.
     * @throws MmdFormatException MMDモーションのフォーマットじゃないとき.
     */
    public Motion fromCSV(InputStream is,int offsetF) throws IOException,MmdFormatException{
        BufferedReader br=null;
        try {
            br = new BufferedReader(new InputStreamReader(is,"MS932"));
        } catch (UnsupportedEncodingException ex) {
            System.err.println("Syntax err in class Motion.fromCSV.");
            System.exit(-1);
        }
        String line;
        int l,i;

        line=br.readLine();
        if(!line.startsWith("Vocaloid Motion Data")){
            throw new MmdFormatException("Header error",0);
        }
        line=br.readLine();
        i=line.indexOf(',');
        if(i>1)
            line=line.substring(0, i-1);
        if(modelName==null){
            modelName=line;
        }else if(modelName.compareTo(line)!=0){
            throw new MmdFormatException("Not same model.");            
        }

        Pose p;
        line=br.readLine(); // Bones
        i=line.indexOf(',');
        if(i>1)
            line=line.substring(0, i-1);
        l=Integer.parseInt(line);
        for(i=0;i<l;i++){
            line=br.readLine();
            p=BonePose.fromCSV(line);
            if(p!=null){
                p.frame+=offsetF;
                this.put(p);
            }
        }

        line=br.readLine();  //Morphs
        if(line==null)return this;
        i=line.indexOf(',');
        if(i>1)
            line=line.substring(0, i-1);
        l=Integer.parseInt(line);
        for(i=0;i<l;i++){
            line=br.readLine();
            p=MorphPose.fromCSV(line);
            if(p!=null){
                p.frame+=offsetF;
                this.put(p);
            }
        }

        line=br.readLine();  //camera
        if(line==null)return this;
        i=line.indexOf(',');
        if(i>1)
            line=line.substring(0, i-1);
        l=Integer.parseInt(line);
        for(i=0;i<l;i++){
            line=br.readLine();
            p=CameraPose.fromCSV(line);
            if(p!=null){
                p.frame+=offsetF;
                this.put(p);
            }
        }

        line=br.readLine();  //light
        if(line==null)return this;
        i=line.indexOf(',');
        if(i>1)
            line=line.substring(0, i-1);
        l=Integer.parseInt(line);
        for(i=0;i<l;i++){
            line=br.readLine();
            p=LightPose.fromCSV(line);
            if(p!=null){
                p.frame+=offsetF;
                this.put(p);
            }
        }

        line=br.readLine();  //shadow
        if(line==null)return this;
        i=line.indexOf(',');
        if(i>1)
            line=line.substring(0, i-1);
        l=Integer.parseInt(line);
        for(i=0;i<l;i++){
            line=br.readLine();
            p=ShadowPose.fromCSV(line);
            if(p!=null){
                p.frame+=offsetF;
                this.put(p);
            }
        }

        line=br.readLine();  //boolean
        if(line==null)return this;
        i=line.indexOf(',');
        if(i>1)
            line=line.substring(0, i-1);
        l=Integer.parseInt(line);
        for(i=0;i<l;i++){
            line=br.readLine();
            p=BooleanPose.fromCSV(line);
            if(p!=null){
                p.frame+=offsetF;
                this.put(p);
            }
        }
        return this;
    }

    /**
     * VMDからMotion を追加する.
     * @param is VMDのInputStream
     * @return 追加後のモーション. {@code this}と同じ.
     * @throws IOException 書き込み時に不具合が発生したとき.
     * @throws MmdFormatException MMDモーションのフォーマットじゃないとき.
     */
    public Motion fromVMD(InputStream is) throws IOException,MmdFormatException{
        return fromVMD(is,0);
    }

    /**
     * VMDからMotion をフレーム番ずらして追加する.
     * @param is VMDのInputStream
     * @param frame ずらすオフセット. 負数は推奨されない.
     * @return 追加後のモーション. {@code this}と同じ.
     * @throws IOException 書き込み時に不具合が発生したとき.
     * @throws MmdFormatException MMDモーションのフォーマットじゃないとき.
     */
    public Motion fromVMD(InputStream is,int frame) throws IOException,MmdFormatException{
        VmdParser vmdp=new VmdParser(is);
        VMDMotionHander parser = new VMDMotionHander(this,frame);

        vmdp.setBasicHandler(parser);
        vmdp.setCameraHandler(parser);
        vmdp.setLightingHandler(parser);
        vmdp.setBoolHandler(parser);

        vmdp.parseVmd();

        return this;
    }

    /**
     * モデル名を得る.
     * @return モデル名.
     */
    public String getModelName() {
        return modelName;
    }

    /**
     * 最終フレーム番を得る.
     * @return 最終フレーム番.
     */
    public int getMaxFrame(){
        return max_frame;
    }

    /**
     * モデル名を書き換える.
     * @param modelName 新しいモデル名.
     */
    public void setModelName(String modelName) {
        this.modelName = modelName;
    }

    /**
     * ボーン名のSetを返す.
     * @return ボーン名のSet.
     */
    public Set<String> listOfBone() {
        return bone_map.keySet();
    }
}
