/*
 * The MIT License
 *
 * Copyright 2015 nazo.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package jp.sourceforge.mmd.midiMotion.gui;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import javax.swing.JFileChooser;
import javax.swing.JOptionPane;
import javax.swing.filechooser.FileNameExtensionFilter;
import jp.sourceforge.mmd.midiMotion.BassMotionBuilder;
import jp.sourceforge.mmd.midiMotion.MidiResolver;
import jp.sourceforge.mmd.midiMotion.ProgressLisner;
import jp.sourceforge.mmd.motion.Motion;
import jp.sourceforge.mmd.motion.swing.MmdDialog;
import jp.sfjp.mikutoga.bin.parser.MmdFormatException;

/**
 * Bass の Pane.
 * @author nazo
 */
public class BassMotionPane extends MotionPane implements ProgressLisner {
    protected BassMotionBuilder bmb;
    protected File convertTable;
    private Motion template;

    /**
     * SampleMotionBuilder 用のJPane 
     */
    public BassMotionPane() {
        super("Bass");
        bmb=new BassMotionBuilder();

        convertTable=null;
        String s=System.getProperty("midiMotion.bassMotion.convertTable");
        if(s!=null){
            convertTable=new File(s);
            if(!convertTable.exists()){
                log("ファイルが存在しない","bass convet table読み込み。\nFile:"+s);
                convertTable=null;
            }
        }

        s=System.getProperty("midiMotion.bassMotion.template");
        template=null;
        if(s!=null){
            template=new Motion();
            FileInputStream fis;
            try {
                template.fromVMD(fis=new FileInputStream(s));
                fis.close();
            } catch (IOException ex) {
                log("IOエラー","bassテンプレート読み込み。\nFile:"+s+"\n"+ex);
            } catch (MmdFormatException ex) {
                log("MMDモーションじゃありません","bassテンプレート読み込み。\nFile:"+s+"\n"+ex);
            }
        }
        initComponents();
    }

    @Override
    public void setMidiResolver(MidiResolver midiR) {
        super.setMidiResolver(midiR);
        buttonProgress.setEnabled(true);
    }

    @Override
    void setFPS(float fps) {
        bmb.setFPS(fps);
    }

    @Override
    public void progressStart(int end) {
        progressBar.setEnabled(true);
        progressBar.setMaximum(end);
        progressBar.setValue(0);
    }

    @Override
    public void progressMid(int s) {
        progressBar.setValue(s);
        progressBar.setString(String.valueOf(s));
    }

    @Override
    public void progressEnd() {
        progressBar.setEnabled(false);
        progressBar.setMaximum(0);
        buttonProgress.setEnabled(true);

        // output
        Motion motion=bmb.getMotion();

        String s=System.getProperty("midiMotion.bassMotion.output");
        if(s==null){s=".";}
        JFileChooser jfc=new JFileChooser(s);
        jfc.setFileSelectionMode(JFileChooser.FILES_ONLY);
        jfc.setToolTipText("ファイル名を選ぶ");
        jfc.setFileFilter(new FileNameExtensionFilter("MMD Motion File", "vmd","csv"));
        if(jfc.showSaveDialog(this)!=JFileChooser.APPROVE_OPTION){
            return;
        }
        File f=jfc.getSelectedFile();
        System.setProperty("midiMotion.bassMotion.output",f.getParent());
        try {
            FileOutputStream fos;
            if(f.getName().toLowerCase().endsWith(".vmd")){
                fos=new FileOutputStream(f);
                motion.toVMD(fos);
                fos.close();
            }else {
                fos=new FileOutputStream(f);
                motion.toCSV(fos);
                fos.close();
            }
        } catch(FileNotFoundException ex){
            JOptionPane.showMessageDialog(this, "ファイルが見つかりません\n"+ex);
        } catch (IOException ex) {
            JOptionPane.showMessageDialog(this, "IOエラー\n"+ex);
        }
        bmb.resetMotion();
    }

    /**
     * This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is always
     * regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;

        buttonProgress = new javax.swing.JButton();
        javax.swing.JLabel jLabel1 = new javax.swing.JLabel();
        buttonDefine = new javax.swing.JButton();
        progressBar = new javax.swing.JProgressBar();
        textConvert = new javax.swing.JTextField();
        javax.swing.JLabel jLabel2 = new javax.swing.JLabel();
        textMotion = new javax.swing.JTextField();
        buttonMotion = new javax.swing.JButton();

        setLayout(new java.awt.GridBagLayout());

        buttonProgress.setText("do");
        buttonProgress.setEnabled(false);
        buttonProgress.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                buttonProgressActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        add(buttonProgress, gridBagConstraints);

        jLabel1.setText("定義ファイル");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        add(jLabel1, gridBagConstraints);

        buttonDefine.setText("open");
        buttonDefine.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                buttonDefineActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 1;
        add(buttonDefine, gridBagConstraints);

        progressBar.setString("0");
        progressBar.setStringPainted(true);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 0.2;
        add(progressBar, gridBagConstraints);

        textConvert.setText((convertTable==null)?"まだ":convertTable.getName());
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 0.1;
        add(textConvert, gridBagConstraints);

        jLabel2.setText("ベーシストテンプレート");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        add(jLabel2, gridBagConstraints);

        textMotion.setText((template==null)?"まだ":template.getModelName());
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        add(textMotion, gridBagConstraints);

        buttonMotion.setText("open");
        buttonMotion.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                buttonMotionActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 0;
        add(buttonMotion, gridBagConstraints);
    }// </editor-fold>//GEN-END:initComponents

    private void buttonDefineActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_buttonDefineActionPerformed
        JFileChooser jfc;
        String s=System.getProperty("midiMotion.bassMotion.sampleTemplate");
        if(s==null){
            jfc=new JFileChooser();
        } else {
            jfc=new JFileChooser(new File(s).getParentFile());
        }
        jfc.setFileFilter(new FileNameExtensionFilter("CSV convert table", "csv"));
        if(jfc.showOpenDialog(this)!=JFileChooser.APPROVE_OPTION)
            return;
        convertTable=jfc.getSelectedFile();
        textConvert.setText(convertTable.getName());
        System.setProperty("midiMotion.bassMotion.convertTable",convertTable.getAbsolutePath());
    }//GEN-LAST:event_buttonDefineActionPerformed

    private void buttonProgressActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_buttonProgressActionPerformed
        FileInputStream fis;
        try{
            fis=new FileInputStream(convertTable);
            bmb.loadFrameMap(fis);
            fis.close();
        }catch(FileNotFoundException ex){
            log("ファイルが見つかりません", "conver table:"+convertTable.getAbsolutePath());
            return;
        } catch (IOException ex) {
            log("IOエラー","conver table:"+convertTable.getAbsolutePath()+"\n"+ex.getLocalizedMessage());
            return;
        }
        bmb.setTemplate(template);
        
        bmb.addProgressListener(this);
        buttonProgress.setEnabled(false);
        new Thread(new Runnable() {
            private ProgressLisner perent;
            public Runnable setP(ProgressLisner l){
                perent=l;
                return this;
            }
            @Override
            public void run() {
                bmb.setMidiResolver(midiR);
                bmb.loadMessages();
                bmb.removeProgressListener(perent);
            }
        }.setP(this)).start();
    }//GEN-LAST:event_buttonProgressActionPerformed

    private void buttonMotionActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_buttonMotionActionPerformed
        Motion m=MmdDialog.loadMotion(this);
        if(m==null)return;
        template=m;
        textMotion.setText(m.getModelName());

        System.setProperty("midiMotion.bassMotion.sampleTemplate", MmdDialog.lastSelected.getAbsolutePath());
    }//GEN-LAST:event_buttonMotionActionPerformed

    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JButton buttonDefine;
    private javax.swing.JButton buttonMotion;
    private javax.swing.JButton buttonProgress;
    private javax.swing.JProgressBar progressBar;
    private javax.swing.JTextField textConvert;
    private javax.swing.JTextField textMotion;
    // End of variables declaration//GEN-END:variables

}
