/*
 * The MIT License
 *
 * Copyright 2015 nazo.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package jp.sourceforge.mmd.midiMotion;

import javax.sound.midi.MidiEvent;
import javax.sound.midi.ShortMessage;
import jp.sourceforge.mmd.ik_solver.IKSolver;
import jp.sourceforge.mmd.motion.model.Bone;
import jp.sourceforge.mmd.motion.BonePose;
import jp.sourceforge.mmd.motion.geo.Matrix;
import jp.sourceforge.mmd.motion.model.Model;
import jp.sourceforge.mmd.motion.Motion;
import jp.sourceforge.mmd.motion.Pose;
import jp.sourceforge.mmd.motion.geo.Vector3D;

/**
 * Standard MIDI File からモーションを作成する. setPlayer を指定しなければ、鍵盤の動きしか生成しない。
 *
 * @author nazo
 */
public class KeyBoardMotionBuilder extends MotionBuilder {

    static private String[] rightHands = new String[]{"右人指３先", "右人指１", "右手首", "右ひじ", "右腕"};
    static private String[] leftHands = new String[]{"左人指３先", "左人指１", "左手首", "左ひじ", "左腕"};
    protected Motion player = null;
    protected Model playerModel = null;
    protected Model keyModel = null;
    protected String bendW = null;
    protected String programChange = null;
    private Vector3D offset = new Vector3D(0, 0, -3.2);
    protected int center = 61;

    /**
     * キーボード
     */
    public KeyBoardMotionBuilder() {
        super();
        motion.setModelName("Keyboard");
    }

    /**
     *
     * @param init キーボードの初期モーション
     */
    public KeyBoardMotionBuilder(Motion init) {
        super(init);
    }

    /**
     * キーボードプレイヤーのモデルを指定
     *
     * @param motion nullにすると初期状態になる。
     * @param model null にできない。プレイヤーのモデル。
     */
    public void setPlayer(Motion motion, Model model) {
        if (motion == null) {
            this.player = new Motion(model.getName());
        } else {
            this.player = motion;
        }
        playerModel = model;
    }

    /**
     * キーボードのモデルを指定
     *
     * @param motion nullにすると初期状態になる。
     * @param model null にできない。キーボードのモデル。
     */
    public void setKeyboard(Motion motion, Model model) {
        if (motion != null) {
            this.motion = motion;
        }
        keyModel = model;
        this.motion.setModelName(model.getName());
    }

    /**
     * 右手・左手の境目のnote number を設定
     *
     * @return the center note number
     */
    public int getCenter() {
        return center;
    }

    /**
     * 右手・左手の境目のnote number を設定
     *
     * @param aCenter the center note number to set
     */
    public void setCenter(int aCenter) {
        center = aCenter;
    }

    /**
     * Midiメッセージからモーションを作るのを開始する
     *
     */
    @Override
    public void loadMessages() {
        MidiEvent[] keys = mm.getChannelMessage(0);
        if (keys == null) {
            keys = mm.getChannelMessage(1);
        }
        if (keys == null) {
            keys = mm.getChannelMessage(2);
        }
        ShortMessage sm;
        int com;
        int note;
        int velo;
        int frame;
        int pitchBend = 0;
        int loaded = 0;
        BonePose p;
        Pose[] ps;
        IKSolver iks;
        Vector3D finger;
        Vector3D key_surface;

        IKSolver iksR = null, iksL = null;

        if (playerModel != null) {
            iksR = new IKSolver();
            iksR.setBones(new Bone[]{playerModel.get(rightHands[2]),
                playerModel.get(rightHands[3]),
                playerModel.get(rightHands[4])});
            iksR.setLimits(0,IKSolver.RLIMIT_NOGlobalR);
            iksR.setLimits(1, IKSolver.RLIMIT_NOZ | IKSolver.RLIMIT_NOX);

            iksL = new IKSolver();
            iksL.setBones(new Bone[]{playerModel.get(leftHands[2]),
                playerModel.get(leftHands[3]),
                playerModel.get(leftHands[4])});
            iksL.setLimits(0,IKSolver.RLIMIT_NOGlobalR);
            iksL.setLimits(1, IKSolver.RLIMIT_NOZ | IKSolver.RLIMIT_NOX);

            playerModel.setPoses(player.get(0));
            playerModel.resetChanged();
            keyModel.setPoses(motion.get(0));
            keyModel.resetChanged();
        }
        reportProgressStart();

        for (MidiEvent me : keys) {
            sm = (ShortMessage) me.getMessage();
            frame = ticsToframe(me.getTick());
            reportProgress(frame);
            if (loaded < frame) {
                if (keyModel != null) {
                    keyModel.setPoses(motion.getInterporate(frame));
                    keyModel.resetChanged();
                }
                if (playerModel != null) {
                    ps = player.getInterporate(frame);
                    playerModel.setPoses(ps);
                    playerModel.resetChanged();
                }
                loaded = frame;
            }
            if ((com = sm.getCommand()) == ShortMessage.NOTE_ON) {
                note = sm.getData1();
                velo = sm.getData2();
                p = new BonePose();
                p.nameOfBone = String.format("nt%03d", note);
                p.frame = frame - 1;
                motion.put(new BonePose(p));

                p.frame++;
                p.mr = Matrix.rotation(-4, 0, 0);
                motion.put(p);
                keyModel.get(p.nameOfBone).setPose(p);

                if (iksR != null) {
                    if (note > center) {
                        iks = iksR;
                        finger=playerModel.get(rightHands[2]).getPos().sub(playerModel.get(rightHands[0]).getPos());
                    } else {
                        iks = iksL;
                        finger=playerModel.get(leftHands[2]).getPos().sub(playerModel.get(leftHands[0]).getPos());
                    }
                    iks.solve(getVecter(note, 0.5D * velo / 100).add(finger));
                    ps = playerModel.getChanged();
                    playerModel.resetChanged();
                    player.putAll(ps, frame - 1);

                    iks.solve(getVecter(note,0).add(finger));
                    ps = playerModel.getChanged();
                    playerModel.resetChanged();
                    player.putAll(ps, frame);
                }
            } else if (com == ShortMessage.NOTE_OFF) {
                note = sm.getData1();
                p = new BonePose();
                p.nameOfBone = String.format("nt%03d", note);
                p.frame = frame;
                motion.put(new BonePose(p));
                p.frame--;
                p.mr = Matrix.rotation(-4, 0, 0);
                motion.put(p);
                keyModel.get(p.nameOfBone).setPose(p);
                
                if (playerModel != null) {
                    if (note > center) {
                        iks = iksR;
                        finger=playerModel.get(rightHands[2]).getPos().sub(playerModel.get(rightHands[0]).getPos());
                    } else {
                        iks = iksL;
                        finger=playerModel.get(leftHands[2]).getPos().sub(playerModel.get(leftHands[0]).getPos());
                    }
                    iks.solve(getVecter(note,0).add(finger));
                    ps = playerModel.getChanged();
                    playerModel.resetChanged();
                    player.putAll(ps, frame - 1);

                    iks.solve(getVecter(note,0.3D).add(finger));
                    ps = playerModel.getChanged();
                    playerModel.resetChanged();
                    player.putAll(ps, frame);
                }
            } else if (com == ShortMessage.PITCH_BEND && bendW != null) {
                p = new BonePose();
                p.nameOfBone = bendW;
                p.frame = frame - 1;
                p.mr = Matrix.rotation(60D * pitchBend / 8192, 0, 0);
                motion.put(new BonePose(p));

                if (playerModel != null) {
                    finger=playerModel.get(leftHands[2]).getPos().sub(playerModel.get(leftHands[0]).getPos());
                    key_surface = keyModel.get(bendW).getPos();
                    iksL.solve(key_surface
                            .add(new Vector3D(0, 0.4, 0)
                                    .rotate_vector(new Vector3D(
                                            Math.sin(-(double) pitchBend / 8192), 0, 0),
                                            Math.cos((double) pitchBend / 8192)))
                            .add(finger));
                    ps = playerModel.getChanged();
                    playerModel.resetChanged();
                    player.putAll(ps, frame - 1);

                    pitchBend = sm.getData1() + sm.getData2() * 128 - 8192;

                    iksL.solve(key_surface
                            .add(new Vector3D(0, 0.4, 0)
                                    .rotate_vector(new Vector3D(
                                            Math.sin(-(double)pitchBend / 8192), 0, 0),
                                            Math.cos((double)pitchBend / 8192)))
                            .add(finger));
                    ps = playerModel.getChanged();
                    playerModel.resetChanged();
                    player.putAll(ps, frame);
                }
                
                p.frame = frame;
                p.mr = Matrix.rotation(60D * pitchBend / 8192, 0, 0);
                motion.put(p);

            }
        }
        reportProgressEnd();
    }

    public Motion getMotionPlayer() {
        return player;
    }

    public void setBendW(String s) {
        bendW = s;
    }

    public void setProgramChange(String s) {
        programChange = s;
    }

    /**
     * 
     * @param note note number 
     * @param y local y
     * @return finger 
     */
    private Vector3D getVecter(int note,double y) {
        Bone b = keyModel.get(String.format("nt%03d", note));
        return b.getPos().add(b.getGMatrix().times(offset.add(new Vector3D(0,y,0))));
    }

    /**
     * @return the offset
     */
    public Vector3D getOffset() {
        return offset;
    }

    /**
     * @param offset the offset to set
     */
    public void setOffset(Vector3D offset) {
        this.offset = offset;
    }

}
