/*
 * MMap+ - 3d image viewer
 * Copyright 2005, 2006 Masahide Miyake
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */


/*
#define DB(x) (x)
*/
#define DB(x)

#include <GL/gl.h>

#include "config.h"

#include "mmap.h"
#include "ww_mesh.h"
#include "mesh.h"
#include "glarea.h"
#include "ww_terrainaccessor.h"

typedef struct _WwMeshPrivate WwMeshPrivate;
struct _WwMeshPrivate {
	gboolean dispose_has_run;

	gdouble distance_above_surface;
	gboolean terrain_mapped;

	gint nx;					/* 分割数 */
	gint ny;

	gint nv;					/* 頂点数 */
	gint ni;					/* インデックス数 */

	gdouble *v_deg;
	gdouble *v_xyz;
	guint *index;
};

#define WW_MESH_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), WW_TYPE_MESH, WwMeshPrivate))

static GObjectClass *parent_class = NULL;

static void ww_mesh_class_init (WwMeshClass * klass);
static void ww_mesh_init (WwMesh * object);
static void ww_mesh_finalize (GObject * object);
static void ww_mesh_dispose (GObject * object);
static GObject *ww_mesh_constructor (GType type, guint n_props, GObjectConstructParam * props);

/*****************************************************************************/

GType
ww_mesh_get_type (void)
{
	static GType type = 0;

	if (type == 0) {
		static const GTypeInfo info = {
			sizeof (WwMeshClass),
			NULL,				/* base_init */
			NULL,				/* base_finalize */
			(GClassInitFunc) ww_mesh_class_init,
			NULL,				/* class_finalize */
			NULL,				/* class_data */
			sizeof (WwMesh),
			0,					/* n_preallocs */
			(GInstanceInitFunc) ww_mesh_init
		};

		type = g_type_register_static (G_TYPE_OBJECT, "WwMesh", &info, 0);
	}

	return type;
}

static void
ww_mesh_class_init (WwMeshClass * klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	/*
	   g_print ("ww_mesh_class_init:c:%p:\n", klass);
	 */
	g_type_class_add_private (klass, sizeof (WwMeshPrivate));

	parent_class = g_type_class_peek_parent (klass);
	object_class->constructor = ww_mesh_constructor;
	object_class->finalize = ww_mesh_finalize;
	object_class->dispose = ww_mesh_dispose;

}

static void
ww_mesh_init (WwMesh * self)
{
	WwMeshPrivate *priv = WW_MESH_GET_PRIVATE (self);

	/*
	   g_print ("ww_mesh_init:o:%p:\n", self);
	 */

	priv->dispose_has_run = FALSE;
	priv->distance_above_surface = 0.0;
	priv->terrain_mapped = FALSE;
	priv->nx = 0;
	priv->ny = 0;
	priv->nv = 0;
	priv->ni = 0;
	priv->v_deg = NULL;
	priv->v_xyz = NULL;
	priv->index = NULL;
}

static void
ww_mesh_dispose (GObject * obj)
{
	WwMesh *self = WW_MESH (obj);
	WwMeshPrivate *priv = WW_MESH_GET_PRIVATE (self);
	/*
	   g_print ("ww_mesh_dispose\n");
	 */
	if (priv->dispose_has_run) {
		return;
	}
	priv->dispose_has_run = TRUE;

	G_OBJECT_CLASS (parent_class)->dispose (obj);
}

static void
ww_mesh_finalize (GObject * obj)
{
	WwMesh *self = WW_MESH (obj);
	WwMeshPrivate *priv = WW_MESH_GET_PRIVATE (self);
	/*
	   g_print ("ww_mesh_finalize\n");
	 */
	g_free (priv->v_deg);
	g_free (priv->v_xyz);
	g_free (priv->index);

	G_OBJECT_CLASS (parent_class)->finalize (obj);
}

static GObject *
ww_mesh_constructor (GType type, guint n_props, GObjectConstructParam * props)
{
	GObject *object;
	GObjectClass *object_class = G_OBJECT_CLASS (parent_class);
	/*
	   g_print ("ww_mesh_constructor\n");
	 */
	object = object_class->constructor (type, n_props, props);

	return object;
}

WwMesh *
ww_mesh_new (void)
{
	GObject *object;

	object = g_object_new (WW_TYPE_MESH, NULL);
	/*
	   g_print ("ww_mesh_new:o:%p\n", object);
	 */
	return WW_MESH (object);
}

/********************************************************************/

WwMesh *
ww_mesh_new_with_terrain (gdouble x0, gdouble y0, gdouble x1, gdouble y1,
						  gint nx, gint ny, gdouble above_surface, WwTerrainaccessor * ta)
{
	WwMesh *mesh;
	WwMeshPrivate *priv;
	gdouble multi;
	/*
	   g_print ("ww_mesh_new_with_terrain:\n");
	 */
	mesh = ww_mesh_new ();
	priv = WW_MESH_GET_PRIVATE (mesh);

	priv->distance_above_surface = above_surface;
	priv->terrain_mapped = TRUE;
	priv->nx = nx;
	priv->ny = ny;
	priv->nv = (nx + 1) * (ny + 1);
	priv->ni = ((nx + 1) * 2 + 2) * ny;

	if (ta == NULL) {
		multi = 1.0;
	} else {
		multi = mmap_get_vertical_exaggeration ();
	}

	priv->v_deg = mesh_create_deg (x0, y0, x1, y1, nx, ny, ta);
	priv->v_xyz = mesh_create_xyz (priv->v_deg, nx, ny, above_surface, multi);
	priv->index = mesh_create_index (nx, ny);

	return mesh;
};

/* tx0, ty0, tx1, ty1 ::: コピー元テクスチャーの範囲
 * mx0, my0, mx1, my1 ::: コピー元メッシュの範囲
 * cx0, cy0, cx1, cy1 ::: コピー先の範囲 */
WwMesh *
ww_mesh_new_copy (gdouble px0, gdouble py0, gdouble px1, gdouble py1,
				  gdouble cx0, gdouble cy0, gdouble cx1, gdouble cy1,
				  gint nx, gint ny, gdouble above_surface, gboolean terrain_mapped, WwMesh * mesh_parent)
{
	WwMesh *mesh;
	WwMeshPrivate *priv;
	WwMeshPrivate *priv_p;
	gdouble multi;
	/*  
	   g_print ("ww_mesh_new_copy\n");
	 */
	mesh = ww_mesh_new ();
	priv = WW_MESH_GET_PRIVATE (mesh);

	priv->distance_above_surface = above_surface;
	priv->terrain_mapped = terrain_mapped;
	priv->nx = nx;
	priv->ny = ny;
	priv->nv = (nx + 1) * (ny + 1);
	priv->ni = ((nx + 1) * 2 + 2) * ny;

	if (terrain_mapped == TRUE) {
		multi = mmap_get_vertical_exaggeration ();
	} else {
		multi = 1.0;
	}

	priv_p = WW_MESH_GET_PRIVATE (mesh_parent);
	priv->v_deg = mesh_create_deg_copy (px0, py0, px1, py1, cx0, cy0, cx1, cy1, nx, ny, priv_p->v_deg);
	priv->v_xyz = mesh_create_xyz (priv->v_deg, nx, ny, above_surface, multi);
	priv->index = mesh_create_index (nx, ny);

	return mesh;
}

void
ww_mesh_change_vertical_exaggeration (WwMesh * mesh)
{
	WwMeshPrivate *priv;
	gdouble multi;

	if (mesh == NULL) {
		return;
	}

	priv = WW_MESH_GET_PRIVATE (mesh);

	multi = mmap_get_vertical_exaggeration ();

	mesh_change_xyz (priv->v_deg, priv->v_xyz, priv->nx, priv->ny, priv->distance_above_surface, multi);
}

void
ww_mesh_render (WwMesh * mesh, WwMeshRenderType type, gboolean is_in_progress, gint color)
{
	WwMeshPrivate *priv;
	gdouble m = 1.0;			/* debug 用 */

	if (mesh == NULL) {
		return;
	}

	if (is_in_progress == TRUE) {	/* debug 用 */
		m = 0.5;
		color *= 0.5;
	}

	priv = WW_MESH_GET_PRIVATE (mesh);

	if (type == WW_MESH_RENDER_TYPE_TEXTURE) {
		/* テクスチャーを表示 */
		glColor3d (1.0, 1.0, 1.0);
		glVertexPointer (3, GL_DOUBLE, 0, priv->v_xyz);
		glDrawElements (GL_TRIANGLE_STRIP, priv->ni, GL_UNSIGNED_INT, priv->index);
		/*
		   glDrawElements (GL_LINE_STRIP, priv->ni, GL_UNSIGNED_INT, priv->index);
		 */

		glDisable (GL_TEXTURE_2D);
		glDisable (GL_DEPTH_TEST);

		if (color == 0) {
			glColor3d (m, 0.0, 0.0);
		} else if (color == 1) {
			glColor3d (0.0, m, 0.0);
		} else if (color == 2) {
			glColor3d (0.0, m, m);
		} else if (color == 3) {
			glColor3d (m, m, 0.0);
		} else {
			glColor3d (0.0, m, m);
		}

		if (is_in_progress == TRUE) {
			glLineWidth (2.0);
			glBegin (GL_LINE_LOOP);

			glVertex3dv (priv->v_xyz);
			glVertex3dv (priv->v_xyz + 3 * priv->nx);
			glVertex3dv (priv->v_xyz + 3 * (priv->nx + priv->ny * (priv->nx + 1)));
			glVertex3dv (priv->v_xyz + 3 * (0 + priv->ny * (priv->nx + 1)));
			glEnd ();
		}

		/*  
		   glLineWidth (1.0);
		   glBegin (GL_LINE_STRIP);
		   glVertex3dv (priv->v_xyz);
		   glVertex3dv (priv->v_xyz + 3 * (priv->nx + priv->ny * (priv->nx + 1)));
		   glVertex3dv (priv->v_xyz + 3 * priv->nx);
		   glVertex3dv (priv->v_xyz + 3 * (0 + priv->ny * (priv->nx + 1)));
		   glEnd ();
		 */

		glLineWidth (1.0);
		glEnable (GL_DEPTH_TEST);
		glEnable (GL_TEXTURE_2D);

	} else if (type == WW_MESH_RENDER_TYPE_BOX) {
		glDisable (GL_TEXTURE_2D);
		glDisable (GL_DEPTH_TEST);

		if (color == 0) {
			glColor3d (m, 0.0, 0.0);
		} else if (color == 1) {
			glColor3d (0.0, m, 0.0);
		} else if (color == 2) {
			glColor3d (0.0, m, m);
		} else if (color == 3) {
			glColor3d (m, m, 0.0);
		} else {
			glColor3d (0.0, m, m);
		}

		glLineWidth (2.0);
		glBegin (GL_LINE_LOOP);

		glVertex3dv (priv->v_xyz);
		glVertex3dv (priv->v_xyz + 3 * priv->nx);
		glVertex3dv (priv->v_xyz + 3 * (priv->nx + priv->ny * (priv->nx + 1)));
		glVertex3dv (priv->v_xyz + 3 * (0 + priv->ny * (priv->nx + 1)));
		glEnd ();

		/*
		   glLineWidth (1.0);
		   glBegin (GL_LINE_STRIP);
		   glVertex3dv (priv->v_xyz);
		   glVertex3dv (priv->v_xyz + 3 * (priv->nx + priv->ny * (priv->nx + 1)));
		   glVertex3dv (priv->v_xyz + 3 * priv->nx);
		   glVertex3dv (priv->v_xyz + 3 * (0 + priv->ny * (priv->nx + 1)));
		   glEnd ();
		 */

		glEnable (GL_DEPTH_TEST);
		glEnable (GL_TEXTURE_2D);

	} else {
		;
	}
    /*
	glDisable (GL_TEXTURE_2D);
	glDisable (GL_DEPTH_TEST);
	glLineWidth (2.0);
	glBegin (GL_LINE_LOOP);

	glVertex3dv (priv->v_xyz);
	glVertex3dv (priv->v_xyz + 3 * priv->nx);
	glVertex3dv (priv->v_xyz + 3 * (priv->nx + priv->ny * (priv->nx + 1)));
	glVertex3dv (priv->v_xyz + 3 * (0 + priv->ny * (priv->nx + 1)));
	glEnd ();
	glEnable (GL_DEPTH_TEST);
	glEnable (GL_TEXTURE_2D);
    */

}
