/*
 * MMap+ - 3d image viewer
 * Copyright 2005, 2006 Masahide Miyake
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <sys/stat.h>
#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include <glib/gprintf.h>
#include <stdlib.h>

#include "mmap.h"
#include "layermanager.h"
#include "color.h"
#include "mmapconfig.h"
#include "mmapsearch.h"
#include "glarea.h"
#include "light.h"
#include "search.h"
#include "mark_dialog.h"
#include "mark.h"
#include "camera.h"
#include "util.h"
#include "text.h"
#include "nmapdata.h"
#include "route.h"

enum {
	RADIO_CAMERA_TYPE_NWW,
	RADIO_CAMERA_TYPE_MMAP,
};

enum {
	RADIO_VE_X000,
	RADIO_VE_X010,
	RADIO_VE_X015,
	RADIO_VE_X020,
	RADIO_VE_X030,
	RADIO_VE_X050,
	RADIO_VE_X070,
	RADIO_VE_X100,
};

/*****************************/

static GtkWidget *window = NULL;
static GtkUIManager *ui_manager = NULL;
static GtkActionGroup *action_group = NULL;
static GtkWidget *size_label = NULL;

static guint time_id = 0;

/*****************************/

static gboolean
configure_event_cb (GtkWidget * widget, GdkEventConfigure * event, gpointer user_data)
{
	gint w, h;

	gtk_window_get_size (GTK_WINDOW (window), &w, &h);

	/*
	   g_print ("configure_event_cb:%d %d\n", w, h);
	 */

	mmap_config_set_int ("window_size_w", w);
	mmap_config_set_int ("window_size_h", h);

	return FALSE;
}

static void
window_destroy_cb (GtkWidget * widget, gpointer data)
{
	g_print ("window_destroy_cb\n");
	gtk_main_quit ();
}

static void
mmap_quit (void)
{
	g_print ("mmap_quit\n");

	g_source_remove (time_id);

	gtk_widget_destroy (window);
}

static gboolean
window_delete_event_cb (GtkWidget * widget, GdkEvent * event, gpointer user_data)
{
	g_print ("window_delete_event_cb\n");

	mmap_quit ();

	/* FALSE を返すと destroy イベントが出る？ */
	return FALSE;
}

/*****************************/

static void
open_world_action (GtkAction * action)
{
	GtkWidget *dialog;

	dialog = gtk_file_chooser_dialog_new (_("Open World"), GTK_WINDOW (window), GTK_FILE_CHOOSER_ACTION_OPEN,
										  GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT, NULL);

	gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (dialog), mmap_dir_pkg_config);

	if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT) {
		gchar *filename;

		filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));

		g_print ("open_world_action:%s\n", filename);
		mmap_set_world (filename);

		g_free (filename);
	}

	gtk_widget_destroy (dialog);

	glarea_force_update_and_render ();
}

static void
save_image_action (GtkAction * action)
{
	glarea_saveimage ();
}

static void
mmap_quit_action (GtkAction * action)
{
	mmap_quit ();
}

static void
view_reset_action (GtkAction * action)
{
	camera_reset ();

	glarea_force_render ();
}

static void
mark_edit_action (GtkAction * action)
{
	mark_dialog_create (GTK_WINDOW (window));
}

static void
mark_start_action (GtkAction * action)
{
	mark_start (GTK_WINDOW (window));
}

static void
search_nww_action (GtkAction * action)
{
	search_dialog_create (action);
}

static void
search_gsi_action (GtkAction * action)
{
	mmapsearch_dialog_create (action);
}

static void
color_action (GtkAction * action)
{
	color_dialog_create (GTK_WINDOW (window));
}

static void
light_action (GtkAction * action)
{
	light_dialog_create (window);
}

static void
selector_action (GtkAction * action)
{
    nmapdata_selector_show();
}

static void
route_action (GtkAction * action)
{
    route_dialog_create();
}

static void
inertia_action (GtkAction * action)
{
	gboolean is_active;

	is_active = gtk_toggle_action_get_active (GTK_TOGGLE_ACTION (action));
	util_print_bool ("inertia_action::", is_active);

	camera_set_inertia (is_active);
}

static void
camera_type_action (GtkAction * action)
{
	if (gtk_radio_action_get_current_value (GTK_RADIO_ACTION (action)) == RADIO_CAMERA_TYPE_NWW) {
		camera_set_type (CAMERA_TYPE_NWW);
	} else {
		camera_set_type (CAMERA_TYPE_MMAP);
	}
}

static void
ve_action (GtkAction * action)
{
	gint value = gtk_radio_action_get_current_value (GTK_RADIO_ACTION (action));
	gdouble x;

	if (value == RADIO_VE_X000) {
        x = 0.0;
    } else if (value == RADIO_VE_X010) {
		x = 1.0;
	} else if (value == RADIO_VE_X015) {
		x = 1.5;
	} else if (value == RADIO_VE_X020) {
		x = 2.0;
	} else if (value == RADIO_VE_X030) {
		x = 3.0;
	} else if (value == RADIO_VE_X050) {
		x = 5.0;
	} else if (value == RADIO_VE_X070) {
		x = 7.0;
	} else if (value == RADIO_VE_X100) {
		x = 10.0;
	} else {
		x = 1.0;
	}

	mmap_set_vertical_exaggeration (x);
	glarea_force_update_and_render ();
}

/*****************************/

static GtkActionEntry entries[] = {
	{"MenuFile", NULL, "File"},
	{"MenuView", NULL, "View"},
	{"MenuMark", NULL, "Mark"},
	{"MenuSearch", NULL, "Search"},
	{"MenuSetting", NULL, "Setting"},
	{"MenuSettingCameraType", NULL, "Camera Type"},
	{"MenuSettingVerticalExaggeration", NULL, "Vertical Exaggeration"},

	{"open-world", NULL, "Open World", NULL, NULL, G_CALLBACK (open_world_action)},
	{"save-image", NULL, "Save Image", NULL, NULL, G_CALLBACK (save_image_action)},
	{"quit", GTK_STOCK_QUIT, NULL, "<control>Q", "Quit the application", G_CALLBACK (mmap_quit_action)},

	{"view-reset", NULL, "Reset", "space", NULL, G_CALLBACK (view_reset_action)},

	{"mark-edit", NULL, "Edit Mark", NULL, NULL, G_CALLBACK (mark_edit_action)},
	{"mark-start", NULL, "Mark Start", NULL, NULL, G_CALLBACK (mark_start_action)},

	{"search-gsi", NULL, "GSI Search", NULL, NULL, G_CALLBACK (search_gsi_action)},
	{"search-nww", NULL, "NWW Search", NULL, NULL, G_CALLBACK (search_nww_action)},

	{"color", NULL, "Color(GSI)", NULL, NULL, G_CALLBACK (color_action)},
	{"light", NULL, "Light(GSI)", NULL, NULL, G_CALLBACK (light_action)},
	{"selector", NULL, "表示選択(GSI)", NULL, NULL, G_CALLBACK (selector_action)},
	{"route", NULL, "ルート検索(GSI)", NULL, NULL, G_CALLBACK (route_action)},
};
static guint n_entries = G_N_ELEMENTS (entries);

/*****************************/

static GtkToggleActionEntry toggle_entries[] = {
	{"inertia", NULL, "Inertia", NULL, NULL, G_CALLBACK (inertia_action), TRUE},
};
static guint n_toggle_entries = G_N_ELEMENTS (toggle_entries);

/*****************************/

static GtkRadioActionEntry camera_entries[] = {
	{"camera-nww", NULL, "World Wind", NULL, NULL, RADIO_CAMERA_TYPE_NWW},
	{"camera-mmap", NULL, "MMap", NULL, NULL, RADIO_CAMERA_TYPE_MMAP},
};
static guint n_camera_entries = G_N_ELEMENTS (camera_entries);

/*****************************/

static GtkRadioActionEntry ve_entries[] = {
	{"ve-x000", NULL, "x 0.0", NULL, NULL, RADIO_VE_X000},
	{"ve-x010", NULL, "x 1.0", NULL, NULL, RADIO_VE_X010},
	{"ve-x015", NULL, "x 1.5", NULL, NULL, RADIO_VE_X015},
	{"ve-x020", NULL, "x 2.0", NULL, NULL, RADIO_VE_X020},
	{"ve-x030", NULL, "x 3.0", NULL, NULL, RADIO_VE_X030},
	{"ve-x050", NULL, "x 5.0", NULL, NULL, RADIO_VE_X050},
	{"ve-x070", NULL, "x 7.0", NULL, NULL, RADIO_VE_X070},
	{"ve-x100", NULL, "x10.0", NULL, NULL, RADIO_VE_X100},
};
static guint n_ve_entries = G_N_ELEMENTS (ve_entries);


/****************************************************/

static const gchar *ui_info =
	"  <menubar>\n"
	"    <menu name=\"File\" action=\"MenuFile\">\n"
	"      <menuitem name=\"open-world\" action=\"open-world\" />\n"
	"      <menuitem name=\"save-image\" action=\"save-image\" />\n"
	"      <separator name=\"sep1\" />\n"
	"      <menuitem name=\"quit\" action=\"quit\" />\n"
	"    </menu>\n"
	"    <menu name=\"view\" action=\"MenuView\">\n"
	"      <menuitem name=\"view-reset\" action=\"view-reset\" />\n"
	"    </menu>\n"
	"    <menu name=\"mark\" action=\"MenuMark\">\n"
	"      <menuitem name=\"mark-edit\" action=\"mark-edit\" />\n"
	"      <menuitem name=\"mark-start\" action=\"mark-start\" />\n"
	"      <separator name=\"sep2\"/>\n"
	"    </menu>\n"
	"    <menu name=\"search\" action=\"MenuSearch\">\n"
	"      <menuitem name=\"search_gsi\" action=\"search-gsi\" />\n"
	"      <menuitem name=\"search_nww\" action=\"search-nww\" />\n"
	"    </menu>\n"
	"    <menu name=\"setting\" action=\"MenuSetting\">\n"
	"      <menu name=\"camera-type\" action=\"MenuSettingCameraType\">\n"
	"        <menuitem name=\"camera-nww\" action=\"camera-nww\" />\n"
	"        <menuitem name=\"camera-mmap\" action=\"camera-mmap\" />\n"
	"      </menu>\n"
	"      <menu name=\"ve\" action=\"MenuSettingVerticalExaggeration\">\n"
	"        <menuitem name=\"ve-x000\" action=\"ve-x000\" />\n"
	"        <menuitem name=\"ve-x010\" action=\"ve-x010\" />\n"
	"        <menuitem name=\"ve-x015\" action=\"ve-x015\" />\n"
	"        <menuitem name=\"ve-x020\" action=\"ve-x020\" />\n"
	"        <menuitem name=\"ve-x030\" action=\"ve-x030\" />\n"
	"        <menuitem name=\"ve-x050\" action=\"ve-x050\" />\n"
	"        <menuitem name=\"ve-x070\" action=\"ve-x070\" />\n"
	"        <menuitem name=\"ve-x100\" action=\"ve-x100\" />\n"
	"      </menu>\n"
	"      <menuitem name=\"color\" action=\"color\" />\n"
	"      <menuitem name=\"light\" action=\"light\" />\n"
	"      <menuitem name=\"selector\" action=\"selector\" />\n"
	"      <menuitem name=\"route\" action=\"route\" />\n"
	"      <menuitem name=\"nertia\" action=\"inertia\" />\n" "    </menu>\n" "  </menubar>\n";


/****************************************************/

void
window_menu_mark_sensitive (gboolean bool)
{
	GtkAction *mark_edit;
	GtkAction *mark_start;

	mark_edit = gtk_action_group_get_action (action_group, "mark-edit");
	mark_start = gtk_action_group_get_action (action_group, "mark-start");

	gtk_action_set_sensitive (mark_edit, bool);
	gtk_action_set_sensitive (mark_start, bool);
}

void
window_menu_saveimage_sensitive (gboolean bool)
{
	GtkAction *saveimage;

	saveimage = gtk_action_group_get_action (action_group, "save-image");

	gtk_action_set_sensitive (saveimage, bool);
}

void
window_menu_selector_sensitive (gboolean bool)
{
	GtkAction *selector;

	selector = gtk_action_group_get_action (action_group, "selector");

	gtk_action_set_sensitive (selector, bool);
}

void
window_menu_route_sensitive (gboolean bool)
{
	GtkAction *route;

	route = gtk_action_group_get_action (action_group, "route");

	gtk_action_set_sensitive (route, bool);
}



/****************************************************/


static GtkActionGroup *mark_action_group = NULL;
static guint ui_id = 0;

void
window_set_mark_submenu (void)
{
	g_print ("window_set_mark_submenu\n");

	if (ui_id != 0 || mark_action_group != NULL) {
		/* 古いマークを全て削除 */
		gtk_ui_manager_remove_ui (ui_manager, ui_id);
		gtk_ui_manager_ensure_update (ui_manager);
		ui_id = 0;

		gtk_ui_manager_remove_action_group (ui_manager, mark_action_group);
		g_object_unref (mark_action_group);
		mark_action_group = NULL;
	}

	mark_action_group = gtk_action_group_new ("MarkActions");
	gtk_ui_manager_insert_action_group (ui_manager, mark_action_group, 0);

	ui_id = gtk_ui_manager_new_merge_id (ui_manager);

	mark_append_menu (ui_manager, mark_action_group, ui_id);

	gtk_ui_manager_ensure_update (ui_manager);

	/*
	   {
	   gchar *dump;

	   dump = gtk_ui_manager_get_ui (ui_manager);
	   g_message (dump);
	   g_free (dump);
	   }
	 */
}

/******************************************/


static void
add_widget (GtkUIManager * ui_manager, GtkWidget * widget, GtkContainer * container)
{
	gtk_box_pack_start (GTK_BOX (container), widget, FALSE, FALSE, 0);
	gtk_widget_show (widget);
}

/*
static void
add_home_kay_func(void)
{
    GtkAction *action;

    action = gtk_action_group_get_action (action_group, "view-reset");
    gtk_action_group_add_action_with_accel (action_group, action, "Home");
}
*/

/**********************************************/


static gint size_old = 0;

gboolean
window_set_cache_size (gpointer data)
{
	gint size;
	gchar *text;

	size = util_get_cache_size ();

	text = g_strdup_printf ("Disk Cache :: %d Mbyte", size / 1024);

	size_old = size;

	gtk_label_set_text (GTK_LABEL (size_label), text);

	g_free (text);

	return TRUE;
}

static void
realize_cb (GtkWidget * widget, gpointer user_data)
{
	guint interval;

	interval = 5 * 60 * 1000;	/* msec    ::: 5 分 */

	time_id = g_timeout_add (interval, window_set_cache_size, NULL);

    text_init(widget);
}


/**********************************************/


void
window_create (void)
{
	GtkWidget *vbox;
	GtkWidget *vbox2;
	GtkWidget *hpaned;
	GtkWidget *layermanager;
	GtkWidget *glarea;
	gint window_size_w;
	gint window_size_h;
	GError *err = NULL;

	glarea = glarea_create ();

	action_group = gtk_action_group_new ("MMapActions");
	gtk_action_group_add_actions (action_group, entries, n_entries, NULL);
	gtk_action_group_add_toggle_actions (action_group, toggle_entries, n_toggle_entries, NULL);
	gtk_action_group_add_radio_actions (action_group, camera_entries, n_camera_entries, RADIO_CAMERA_TYPE_NWW,
										G_CALLBACK (camera_type_action), NULL);
	gtk_action_group_add_radio_actions (action_group, ve_entries, n_ve_entries, RADIO_VE_X010, G_CALLBACK (ve_action), NULL);

	ui_manager = gtk_ui_manager_new ();

	layermanager = layermanager_create ();

	window_size_w = mmap_config_get_int ("window_size_w", NULL);
	if (window_size_w < 10) {
		window_size_w = 900;
	}
	window_size_h = mmap_config_get_int ("window_size_h", NULL);
	if (window_size_h < 10) {
		window_size_h = 700;
	}

	window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title (GTK_WINDOW (window), "MMap+");
	g_object_set (G_OBJECT (window), "allow_shrink", TRUE, NULL);
	gtk_window_set_default_size (GTK_WINDOW (window), window_size_w, window_size_h);
	g_signal_connect (window, "delete_event", G_CALLBACK (window_delete_event_cb), NULL);
	g_signal_connect (window, "destroy", G_CALLBACK (window_destroy_cb), NULL);
	g_signal_connect (window, "realize", G_CALLBACK (realize_cb), NULL);
	g_signal_connect (window, "configure_event", G_CALLBACK (configure_event_cb), NULL);
	g_signal_connect_swapped (window, "destroy", G_CALLBACK (g_object_unref), ui_manager);

	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (window), vbox);

	gtk_ui_manager_insert_action_group (ui_manager, action_group, 0);
	g_signal_connect (ui_manager, "add_widget", G_CALLBACK (add_widget), vbox);


	gtk_window_add_accel_group (GTK_WINDOW (window), gtk_ui_manager_get_accel_group (ui_manager));
	if (!gtk_ui_manager_add_ui_from_string (ui_manager, ui_info, -1, &err)) {
		g_message ("building menus failed: %s", err->message);
		g_error_free (err);
		exit (1);
	}

	{
		size_label = gtk_label_new ("");

		window_set_cache_size (NULL);

		vbox2 = gtk_vbox_new (FALSE, 0);
		gtk_box_pack_start (GTK_BOX (vbox2), layermanager, TRUE, TRUE, 0);
		gtk_box_pack_start (GTK_BOX (vbox2), size_label, FALSE, FALSE, 0);
	}

	hpaned = gtk_hpaned_new ();
	gtk_paned_add1 (GTK_PANED (hpaned), vbox2);
	gtk_paned_add2 (GTK_PANED (hpaned), glarea);

	gtk_box_pack_end (GTK_BOX (vbox), hpaned, TRUE, TRUE, 0);

	window_set_mark_submenu ();
	/*
	   add_home_kay_func();
	 */

	gtk_widget_show_all (window);
}
