/*
 * MMap+ - 3d image viewer
 * Copyright 2005, 2006 Masahide Miyake
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/*
#define DB(x) (x)
*/
#define DB(x)

#include <glib.h>
#include <glib/gprintf.h>
#include <stdlib.h>
#include <math.h>

#include "../route_common_def.h"
#include "disk.h"
#include "gsi.h"
#include "data.h"

static gint n_node = 0;
static gint n_edge = 0;

/*********************************************************/

static void
create_alledge_allnode (void)
{
	GIOChannel *ch;
	GIOChannel *ch_node;
	GIOChannel *ch_edge;
	GIOStatus status;
	GError *err = NULL;
	gchar *line;

	system ("ls *.slm > files");

	ch_node = disk_channel_open ("allnode", WRITE);
	ch_edge = disk_channel_open ("alledge", WRITE);

	ch = disk_channel_open ("files", READ);
	status = g_io_channel_read_line (ch, &line, NULL, NULL, &err);
	while (status != G_IO_STATUS_EOF) {
		gint64 x0, y0;
		gint n;
		Slp *slp;

		if (status == G_IO_STATUS_ERROR) {
			g_print ("err:files\n");
			exit (1);
		}

		line[5] = '\0';			/* line: "11111.slm" */

		/* slm からの基準点と頂点数の取得 */
		slm (line, &x0, &y0, &n);
		g_print ("name:%s x0:%Ld y0:%Ld n:%d\n", line, x0, y0, n);

		/* slp から市の経度緯度の配列をつくる */
		slp = slp_create (line, x0, y0, n);

		/* dk から必要なデータを選択し、エッジとノードの全リストに書き出す */
		edge_create (line, slp, ch_edge, ch_node);

		slp_free (slp);
		g_free (line);
		status = g_io_channel_read_line (ch, &line, NULL, NULL, &err);
	}
	g_free (line);
	g_io_channel_unref (ch);
	g_io_channel_unref (ch_node);
	g_io_channel_unref (ch_edge);
}

/*********************************************************/

/* 経度緯度の組をキーにして、その経度緯度が allnode.sort.uniq 中で何番目にあるかをあらわす値を検索するためのハッシュ */

typedef struct {
    gint64 lon;
    gint64 lat;
}HashKey;

static gboolean
g_node_equal (gconstpointer v1, gconstpointer v2)
{
	const HashKey *key1 = v1;
	const HashKey *key2 = v2;

    return (key1->lon == key2->lon) && (key1->lat == key2->lat);
}

static guint
g_node_hash (gconstpointer v)
{
	const HashKey *key = v;

	return (guint)(key->lon / 1000);
}

static GHashTable *
hash_node_create ()
{
	GIOChannel *ch_node;
	GIOChannel *ch_node_bin;
	GIOStatus status;
	GError *err = NULL;
	GHashTable *hash_node;
	gchar *line;
	gint i = 0;

	g_print ("#################### create node hash\n");

	ch_node = disk_channel_open ("allnode.sort.uniq", READ);
	ch_node_bin = disk_channel_open ("allnode.bin", WRITE);

	hash_node = g_hash_table_new (g_node_hash, g_node_equal);

	status = g_io_channel_read_line (ch_node, &line, NULL, NULL, &err);
	while (status != G_IO_STATUS_EOF) {
		HashKey *key;
		gint64 lon, lat;
		AllNodeBin allnodebin;
        gint *value;

		if (status == G_IO_STATUS_ERROR) {
			g_print ("err:allnode.sort.uniq\n");
			exit (1);
		}

		line = g_strchomp (line);
		sscanf (line, "%10Ld%10Ld", &lon, &lat);

		key = g_new (HashKey, 1);
		key->lon = lon;
		key->lat = lat;
		value = g_new(gint, 1);
        *value = i;

		g_hash_table_insert (hash_node, key, value);

		/* 始点・終点取得用の経度緯度データ */
		allnodebin.lon = (gfloat) lon / 10000.0;
		allnodebin.lat = (gfloat) lat / 10000.0;
		g_io_channel_write_chars (ch_node_bin, (gchar *) & allnodebin, sizeof (AllNodeBin), NULL, NULL);

		++i;
		g_free (line);
		status = g_io_channel_read_line (ch_node, &line, NULL, NULL, &err);
	}
	g_free (line);
	g_io_channel_unref (ch_node);
	g_io_channel_unref (ch_node_bin);

	g_print ("#################### %d\n", g_hash_table_size (hash_node));

	n_node = i;

	return hash_node;
}

/*********************************************************/

static void
write_char_sub (GIOChannel * ch, gdouble lon, gdouble lat)
{
	LonLat lonlat;

	lonlat.lon = (gfloat) lon;
	lonlat.lat = (gfloat) lat;

	g_io_channel_write_chars (ch, (gchar *) & lonlat, sizeof (LonLat), NULL, NULL);
	/*
	   g_print ("write_char_sub:%.2f %.2f\n", lonlat.lon, lonlat.lat);
	 */
}

/* lon,lat: sec */
void
get_xyz_from_text (const gchar * text, gdouble * lon, gdouble *lat, gdouble * x, gdouble * y, gdouble * z)
{
	gint64 lon64, lat64;

	sscanf (text, "%10Ld%10Ld", &lon64, &lat64);
	/*
	   g_print ("%Ld %Ld\n",lon64, lat64);
	 */
	*lon = (gdouble) lon64 / 10000.0;
	*lat = (gdouble) lat64 / 10000.0;
	gsi_keidoido_to_xyz (*lon, *lat, 0.0, x, y, z);
}

/* エッジ１本の経度・緯度の列をｘｙｚの座標に変換して書き出すとともに、エッジの長さを求める */
/**********************************************************************************
 * 現在の MMap+ の地球は球で、楕円体でない。
 * しかし、ここでは楕円体上の座標を使って距離 l を求める。
 * 少しは実際の距離に近くなるかと思ったが、標高を考慮してない現状では自己満足。
 * *******************************************************************************/
static gdouble
calc_edge_l (gchar ** token, gint n, GIOChannel * ch_lonlat_bin)
{
	gint k;
	gdouble x0, y0, z0;
    gdouble lon, lat;
	gdouble l;

	/* token[0] は幅員情報なので省く */
	get_xyz_from_text (token[1], &lon, &lat, &x0, &y0, &z0);
	write_char_sub (ch_lonlat_bin, lon, lat);

	l = 0;
	for (k = 2; k < n; ++k) {
		gdouble x1, y1, z1;
		gdouble dx, dy, dz;

	    get_xyz_from_text (token[k], &lon, &lat, &x1, &y1, &z1);
	    write_char_sub (ch_lonlat_bin, lon, lat);

		dx = x1 - x0;
		dy = y1 - y0;
		dz = z1 - z0;
		l += sqrt (dx * dx + dy * dy + dz * dz);

		x0 = x1;
		y0 = y1;
		z0 = z1;
	}
	return l;
}

static void
create_edge_l_index (GHashTable * hash_node)
{
	GIOChannel *ch_e;
	GIOChannel *ch_struct_bin;
	GIOChannel *ch_start_bin;
	GIOChannel *ch_lonlat_bin;
	GIOStatus status;
	GError *err = NULL;
	gchar *line;
	gint count_total = 0;
	gint n_line = 0;;

	ch_struct_bin = disk_channel_open ("edge_struct.bin", WRITE);
	ch_start_bin = disk_channel_open ("edge_start_n.bin", WRITE);
	ch_lonlat_bin = disk_channel_open ("edge_lonlat.bin", WRITE);

	ch_e = disk_channel_open ("alledge", READ);
	status = g_io_channel_read_line (ch_e, &line, NULL, NULL, &err);
	while (status != G_IO_STATUS_EOF) {
		gchar **token = NULL;
		gint count = 0;			/* token[] の数。先頭の幅員の情報１つも含まれている */
		gint64 lon0, lat0, lon1, lat1;
		Edge edge;
		EdgeStartN startn;
		gint multiple;
		gdouble l;
		HashKey key0, key1;
		gint *value0, *value1;

		if (status == G_IO_STATUS_ERROR) {
			g_print ("err:files\n");
			exit (1);
		}
		line = g_strchomp (line);

		token = g_strsplit (line, ",", -1);
		count = 0;
		/* トークンの数を数える */
		while (token[count] != NULL) {
            /*
			g_print ("tokne:%d:%s\n",count, token[count]); 
            */
			++count;
		}

		/* 道の重み：高速は 5、細道は 30 でこれを距離にかけて使う */
		sscanf (token[0]        , "%d",         &multiple);
		sscanf (token[1]        , "%10Ld%10Ld", &lon0, &lat0);
		sscanf (token[count - 1], "%10Ld%10Ld", &lon1, &lat1);

        key0.lon = lon0;
        key0.lat = lat0;
        key1.lon = lon1;
        key1.lat = lat1;

		value0 = g_hash_table_lookup (hash_node, &key0);
		value1 = g_hash_table_lookup (hash_node, &key1);

		edge.node_index0 = *value0;
		edge.node_index1 = *value1;
		l = calc_edge_l (token, count, ch_lonlat_bin) * (gdouble) multiple / 10.0;
		edge.l = (gfloat) l;

		startn.start = count_total;
		startn.n = count - 1;
		startn.l = (gfloat) l;

		g_io_channel_write_chars (ch_struct_bin, (gchar *) & edge, sizeof (Edge), NULL, NULL);
		g_io_channel_write_chars (ch_start_bin, (gchar *) & startn, sizeof (EdgeStartN), NULL, NULL);

		count_total += (count - 1);
		++n_line;

		g_strfreev (token);
		g_free (line);
		status = g_io_channel_read_line (ch_e, &line, NULL, NULL, &err);
	}
	g_free (line);
	g_io_channel_unref (ch_e);
	g_io_channel_unref (ch_struct_bin);
	g_io_channel_unref (ch_start_bin);
	g_io_channel_unref (ch_lonlat_bin);

	n_edge = n_line;

	{
		GIOChannel *ch_size;
		gchar buf[100];

		ch_size = disk_channel_open ("data_size", WRITE);
		g_sprintf (buf, "%d,node\n", n_node);
		g_io_channel_write_chars (ch_size, buf, -1, NULL, NULL);
		g_sprintf (buf, "%d,edge\n", n_edge);
		g_io_channel_write_chars (ch_size, buf, -1, NULL, NULL);
		g_io_channel_unref (ch_size);
	}
}

/*********************************************************/

typedef struct {
	GSList *slist_edge;
} NodeTmp;

static void
node_add_curve (NodeTmp * nodetmp, Edge * edge)
{
	nodetmp->slist_edge = g_slist_prepend (nodetmp->slist_edge, edge);
}

static void
create_node_list (void)
{
	GIOChannel *ch_index_bin;
	GIOChannel *ch_start_bin;
	NodeTmp *nodetmp;
	Edge *edge;
	gint i;
	gint dummy;
	gint count_sum;

	g_print ("create node list:n_node:%d\n", n_node);

	nodetmp = g_new0 (NodeTmp, n_node);
	edge = disk_bin_from_disk ("edge_struct.bin", &dummy);

	for (i = 0; i < n_edge; ++i) {
		Edge *e = &edge[i];
		/*
		g_print ("index0:%d index1:%d\n", e->node_index0, e->node_index1);
		 */

		node_add_curve (&nodetmp[e->node_index0], e);
		node_add_curve (&nodetmp[e->node_index1], e);
	}

	/* 書き出し */

	ch_index_bin = disk_channel_open ("node_index.bin", WRITE);
	ch_start_bin = disk_channel_open ("node_start_n.bin", WRITE);
	g_io_channel_set_buffered (ch_index_bin, FALSE);
	g_io_channel_set_buffered (ch_start_bin, FALSE);

	count_sum = 0;
	for (i = 0; i < n_node; ++i) {
		NodeTmp *n = &nodetmp[i];
		GSList *sl;
		NodeStartN startn;
		gint count;

		for (sl = n->slist_edge, count = 0; sl != NULL; sl = sl->next, ++count) {
			Edge *edge_tmp = sl->data;
			gint index;

			index = edge_tmp - edge;
			g_io_channel_write_chars (ch_index_bin, (gchar *) & index, sizeof (gint), NULL, NULL);
		}

		startn.start = count_sum;
		startn.n = count;
		g_io_channel_write_chars (ch_start_bin, (gchar *) & startn, sizeof (NodeStartN), NULL, NULL);

		count_sum += count;
	}

	g_io_channel_unref (ch_index_bin);
	g_io_channel_unref (ch_start_bin);
}

/*********************************************************/

int
main (int argc, char *argv[])
{
	GHashTable *hash_node;

	/* 検索対象にするノードとエッジを全てファイル("allnode","alledge")に書き出す */
	create_alledge_allnode ();

	/* 書き出したノードのファイルには、市町村の境界で重複しているノードがあるので、重複分を削除する */
	g_print ("sort allnode | uniq > allnode.sort.uniq\n");
	system ("sort allnode | uniq > allnode.sort.uniq");

	/* 重複のなくなったノードを検索しやすくするためハッシュをつくるとともに、バイナリ形式の "allnode.bin" を作る */
	hash_node = hash_node_create ();

	/* エッジの長さ、その両端のノード番号、他をファイルに書き出す */
	create_edge_l_index (hash_node);

	/* ノードにつながるエッジのリストをまとめる */
	create_node_list ();

	return 0;
}
