/*
 * MMap+ - 3d image viewer
 * Copyright 2005, 2006 Masahide Miyake
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/*
#define DB(x) (x)
*/
#define DB(x)

#include <gtk/gtk.h>
#include <gtk/gtkgl.h>
#include <glib/gprintf.h>
#include <stdlib.h>
#include <string.h>
#include <GL/gl.h>

#include "mmap.h"
#include "window.h"
#include "disk.h"
#include "util.h"
#include "glarea.h"
#include "camera.h"
#include "route.h"
#include "route_common_def.h"

/*********************************************/
/* ファイル一覧 */

/* ノードとエッジのデータ数 */
static gchar *DATA_SIZE = "data_size";

/* ノードの座標の列。lon lat 各 10 桁の合計 20 文字で１行となる文字列 */
static gchar *ALLNODE_BIN = "allnode.bin";

/* エッジのカーブの座標がならんでいる。 (gfloat lon, gfloat lat) の列 */
static gchar *EDGE_LONLAT_BIN = "edge_lonlat.bin";

/* strcut Edge の列 */
static gchar *EDGE_STRUCT_BIN = "edge_struct.bin";

/* struct Edge の N 番目のカーブデータが edge_lonlat.bin のどこからはじまり何個あるか */
/* (int start, int n) の列 */
static gchar *EDGE_START_N_BIN = "edge_start_n.bin";


/* struct Node の Edge へのインデックスが、node_index.bin のどこからはじまり何個あるか */
static gchar *NODE_START_N_BIN = "node_start_n.bin";

static gchar *NODE_INDEX_BIN = "node_index.bin";

/******************************************************/

static GtkWidget *label_start_lon;
static GtkWidget *label_start_lat;
static GtkWidget *label_goal_lon;
static GtkWidget *label_goal_lat;
static GtkWidget *label_result;

static gint index_start = -1;
static gint index_goal = -1;
static gdouble start_x;
static gdouble start_y;
static gdouble start_z;
static gdouble goal_x;
static gdouble goal_y;
static gdouble goal_z;
static gfloat start_to_goal = 0.0;

/*********************************************/


/* ルート検索用のデータがあれば TRUE */
static gboolean route_data = FALSE;

/* 検索に使用するノードとエッジの数 */
static gint n_node;
static gint n_edge;

/* 与えられた経度緯度(秒)に近いノードのインデックスを返す。
 * 意味のあるインデックスを返せない時には -1 を返す。 */
static gint
route_get_near_index (gdouble lon, gdouble lat)
{
	GIOChannel *ch;
	gint i = 0;
	gint index = -1;
	gfloat dist = 1000000000.0;
	gfloat lon_node = 0.0;
	gfloat lat_node = 0.0;
	gchar *fullpath;

    /*
	g_print ("route_get_near_index:%.2f %.2f\n", lon, lat);
    */
	if (route_data == FALSE) {
		return -1;
	}

	fullpath = g_strconcat (mmap_dir_gsi_vn, ALLNODE_BIN, NULL);
	ch = disk_channel_open (fullpath, READ_BIN);
	g_free (fullpath);

	g_io_channel_set_buffered (ch, FALSE);

	for (i = 0; i < n_node; ++i) {
		AllNodeBin allnodebin;
		gfloat dlon, dlat;
		gfloat dist_node;
		gint offset;

		offset = i * sizeof (AllNodeBin);
		g_io_channel_seek_position (ch, offset, G_SEEK_SET, NULL);
		g_io_channel_read_chars (ch, (gchar *) & allnodebin, sizeof (AllNodeBin), NULL, NULL);

        /*
	    g_print ("allnodebin lon:%.2f lat:%.2f\n", allnodebin.lon, allnodebin.lat);
        */

		dlon = allnodebin.lon - (gfloat) lon;
		dlat = allnodebin.lat - (gfloat) lat;

		/* 以下の二つは allnodebin が経度の順でならんでいるから使える */

		/* クリック地点の経度より２０００秒(約5０キロ)以上東は見ない */
		if (dlon > 2000.0) {
			break;
		}
		/* クリック地点の経度より２０００秒(約5０キロ)以上西なら大きく進む */
		if (dlon < -2000.0) {
			i += 100;
			continue;
		}

		if (dlon < 0) {
			dlon = -dlon;
		}
		if (dlat < 0) {
			dlat = -dlat;
		}
		dist_node = dlon + dlat;

		if (dist > dist_node) {
			index = i;
			lon_node = allnodebin.lon;
			lat_node = allnodebin.lat;
			dist = dist_node;
		}
	}
	g_io_channel_unref (ch);

    /*
	g_print ("index:%d dist:%.2f    %.2f %.2f\n", index, dist, lon_node, lat_node);
    */

	return index;
}

/* ~/.mmap/gsi/vn/data_size に書かれているノードとエッジの数を取り込む */
static void
check_data_size (void)
{
	GIOChannel *ch;
	GIOStatus status;
	GError *err = NULL;
	gchar *line;
	gchar *fullpath;

	fullpath = g_strconcat (mmap_dir_gsi_vn, DATA_SIZE, NULL);

	ch = disk_channel_open (fullpath, READ_LOCAL);
	status = g_io_channel_read_line (ch, &line, NULL, NULL, &err);
	while (status != G_IO_STATUS_EOF) {
		gchar **token = NULL;

		if (status == G_IO_STATUS_ERROR) {
			g_print ("err:check_data_size\n");
			exit (1);
		}

		line = g_strchomp (line);

		token = g_strsplit (line, ",", -1);
		if (token[1] == NULL) {
			g_strfreev (token);
			break;
		}

		if (g_ascii_strcasecmp (token[1], "node") == 0) {
			n_node = atoi (token[0]);
		} else if (g_ascii_strcasecmp (token[1], "edge") == 0) {
			n_edge = atoi (token[0]);
		} else {
			;
		}

		g_strfreev (token);

		g_free (line);
		status = g_io_channel_read_line (ch, &line, NULL, NULL, &err);
	}
	g_free (line);
	g_io_channel_unref (ch);

    g_free(fullpath);

	g_print ("n_node:%d    n_edge:%d\n", n_node, n_edge);
}

static gboolean
route_data_check_sub (const gchar * name)
{
	gchar *fullpath;
	gboolean file_exist;

	fullpath = g_strconcat (mmap_dir_gsi_vn, name, NULL);
	file_exist = filetest (fullpath);
	g_free (fullpath);

	return file_exist;
}

void
route_data_check (void)
{
	gboolean allnode;
	gboolean edge_lonlat_bin;
	gboolean edge_struct_bin;
	gboolean edge_start_n_bin;
	gboolean data_size;

	allnode = route_data_check_sub (ALLNODE_BIN);
	edge_lonlat_bin = route_data_check_sub (EDGE_LONLAT_BIN);
	edge_struct_bin = route_data_check_sub (EDGE_STRUCT_BIN);
	edge_start_n_bin = route_data_check_sub (EDGE_START_N_BIN);
	data_size = route_data_check_sub (DATA_SIZE);

	if (allnode && edge_lonlat_bin && edge_struct_bin && edge_start_n_bin && data_size) {
		check_data_size ();
		route_data = TRUE;
	} else {
		route_data = FALSE;
	}

    /*
	g_print ("route_data_check:n_node:%d\n", n_node);
    */
}

/*********************************/

typedef struct {
    gfloat lonlat[2];
    gfloat xyz[3];
} RPoint;

typedef struct {
	gfloat l;					/* カーブの距離(m) */
	GSList *sl_rpoint;
} RouteResult;

GSList *sl_result = NULL;

void
result_free (void)
{
	GSList *sl;

	for (sl = sl_result; sl != NULL; sl = sl->next) {
		GSList *s;
		RouteResult *rr = sl->data;

		for (s = rr->sl_rpoint; s != NULL; s = s->next) {
            RPoint *rp = s->data;

			g_free (rp);
		}
		g_slist_free (rr->sl_rpoint);
		g_free (rr);
	}

	g_slist_free (sl_result);

	sl_result = NULL;
}

void
route_render (void)
{
	GSList *sl;

	glDisable (GL_DEPTH_TEST);

	glLineWidth (3.0);
	glColor3d (1.0, 1.0, 0.0);
	for (sl = sl_result; sl != NULL; sl = sl->next) {
		GSList *s;
		RouteResult *rr = sl->data;

		glBegin (GL_LINE_STRIP);
		for (s = rr->sl_rpoint; s != NULL; s = s->next) {
			RPoint *rp = s->data;

			glVertex3fv (rp->xyz);
		}
		glEnd ();
	}
	glLineWidth (1.0);

	glEnable (GL_DEPTH_TEST);
}

/*********************************/

static const gint FAR_ENOUGH = 10000000;

/**************/

static gint *
edge_index_create (void)
{
	gchar *fullpath;
	gint *edge_index;
	gint size;

	fullpath = g_strconcat (mmap_dir_gsi_vn, NODE_INDEX_BIN, NULL);
	edge_index = disk_load_bin (fullpath, &size);
	g_free (fullpath);

	return edge_index;
}

static Node *
node_create (gint * edge_index)
{
	Node *node;
	gchar *fullpath;
	GIOChannel *ch = NULL;
	gint i;

	node = g_new (Node, n_node);

	fullpath = g_strconcat (mmap_dir_gsi_vn, NODE_START_N_BIN, NULL);
	ch = disk_channel_open (fullpath, READ_BIN);
	g_free (fullpath);

	for (i = 0; i < n_node; ++i) {
		Node *n = &node[i];
		NodeStartN startn;

		g_io_channel_read_chars (ch, (gchar *) & startn, sizeof (NodeStartN), NULL, NULL);
		n->edge_index = &edge_index[startn.start];
		n->n = startn.n;
		n->l = FAR_ENOUGH;
	}

	g_io_channel_unref (ch);

	return node;
}

void
node_free (Node * node, gint * edge_index)
{
	g_free (node);
	g_free (edge_index);
}

/**************/

/* カーブの一方のノード番号からもう一方のノード番号を調べる */
static Node *
edge_get_other_side (Edge * edge, Node * node_all, Node * node)
{
	if (&node_all[edge->node_index0] == node) {
		return &node_all[edge->node_index1];
	} else {
		return &node_all[edge->node_index0];
	}
}

void
edge_free (Edge * edge)
{
	g_free (edge);
}

/**************/

static Edge *
edge_create (void)
{
	gchar *fullpath;
	Edge *edge;
	gint size;

	fullpath = g_strconcat (mmap_dir_gsi_vn, EDGE_STRUCT_BIN, NULL);
	edge = disk_load_bin (fullpath, &size);
	g_free (fullpath);

	return edge;
}

/******************************/

/*
#define DB_TREE(x) (x)
*/
#define DB_TREE(x)

static GTree *tree = NULL;
static gint n_tree = 0;

static gboolean
tree_traverse_func (gpointer key, gpointer value, gpointer data)
{
	Node *node = value;
	Node **node_return = data;

	DB_TREE (g_print ("tree_traverse:node %.2f\n", node->l));

	*node_return = node;

	return TRUE;
}

static Node *
tree_get_next_node (void)
{
	Node *next;

	g_tree_foreach (tree, (GTraverseFunc) tree_traverse_func, &next);

	DB_TREE (g_print ("next node:dist %.2f\n", next->l));

	return next;
}

static void
tree_add_node (Node * node)
{
	{
		Node *tmp;

		/* 同じ距離のものがある場合に上書きされるので、追加ノードの距離に少しづつ足していって、上書き距離にする */
		tmp = g_tree_lookup (tree, &(node->l));
		while (tmp != NULL) {
			/*
			g_print("tree add:node:%.2f %p\n", node->l, node);
			g_print("tree add:tmp :%.2f %p\n", tmp->l, tmp);
			*/

			node->l += 0.1;
			/*
			g_print("+0.01         :%.2f %p\n", node->l, node);
			*/
			tmp = g_tree_lookup (tree, &(node->l));
		}
	}
	g_tree_insert (tree, &(node->l), node);
	++n_tree;

	DB_TREE (g_print ("tree_add:%d\n", n_tree));
}

static void
tree_del_node (Node * node)
{
	g_tree_remove (tree, &(node->l));
	--n_tree;

	DB_TREE (g_print ("tree_del:%d\n", n_tree));
}

gint
key_compare_func (gconstpointer a, gconstpointer b)
{
	gfloat d;
	const gfloat *la = a;
	const gfloat *lb = b;
	gint retval;

	d = *la - *lb;
	if (d < 0) {
		retval = -1;
	} else if (d > 0) {
		retval = 1;
	} else {
		retval = 0;
	}
	/*
	g_print ("tree compare:%.2f %.2f    retval:%d\n", *la, *lb, retval);
	*/
	return retval;
}

static void
tree_seek_free (GTree * tree)
{
	g_tree_destroy (tree);
}

static void
tree_new (void)
{
	tree = g_tree_new ((GCompareFunc) key_compare_func);
}

/******************************/

static GSList *
get_lonlat (GIOChannel * ch, gint start, gint n)
{
	gint i;
	GSList *sl_rpoint = NULL;
	/*
	   g_print ("get_lonlat:start %d   n %d\n", start, n);
	 */
	for (i = 0; i < n; ++i) {
        RPoint *rpoint;
		gint64 offset;
        gdouble x, y, z;

		offset = (start + i) * sizeof (LonLat);
		g_io_channel_seek_position (ch, offset, G_SEEK_SET, NULL);

        rpoint = g_new(RPoint, 1);
		g_io_channel_read_chars (ch, (gchar *) (rpoint->lonlat), sizeof (gfloat) * 2, NULL, NULL);

        mmap_deg_to_xyz (rpoint->lonlat[0] / 3600.0, rpoint->lonlat[1] / 3600.0, 0.0, &x, &y, &z);
        rpoint->xyz[0] = (gfloat)x;
        rpoint->xyz[1] = (gfloat)y;
        rpoint->xyz[2] = (gfloat)z;

		/*
		   g_print ("get_lonlat:%.2f %.2f\n", lonlat->lon, lonlat->lat);
		 */
		sl_rpoint = g_slist_prepend (sl_rpoint, rpoint);
	}

	return sl_rpoint;
}

static void
get_start_n_l (GIOChannel * ch, gint index, gint * start, gint * n, gfloat * l)
{
	gint offset;
	EdgeStartN startn;

	offset = index * sizeof (EdgeStartN);
	g_io_channel_seek_position (ch, offset, G_SEEK_SET, NULL);
	g_io_channel_read_chars (ch, (gchar *) & startn, sizeof (EdgeStartN), NULL, NULL);
	*start = startn.start;
	*n = startn.n;
	*l = startn.l;
	/*
	   g_print("get_start_n:%d  start:%d  n:%d l:%.2f\n", index, *start, *n, *l);
	 */
}


/* 通って来た経路をゴールからスタートにたどって行きながら、sl_result に追加していく。距離は後で計算する。 */
static void
create_result (Node * start_node, Node * goal_node, Node * node, Edge * curve)
{
	Node *n;
	int count = 0;
	gchar *fullpath;
	GIOChannel *ch_start = NULL;
	GIOChannel *ch_lonlat = NULL;

	fullpath = g_strconcat (mmap_dir_gsi_vn, EDGE_START_N_BIN, NULL);
	ch_start = disk_channel_open (fullpath, READ_BIN);
	g_free (fullpath);

	fullpath = g_strconcat (mmap_dir_gsi_vn, EDGE_LONLAT_BIN, NULL);
	ch_lonlat = disk_channel_open (fullpath, READ_BIN);
	g_free (fullpath);

	g_io_channel_set_buffered (ch_start, FALSE);
	g_io_channel_set_buffered (ch_lonlat, FALSE);

	result_free ();

	start_to_goal = 0.0;

	n = goal_node;
	while (n != start_node) {
		gint i;
		gfloat ll = 10000000.0;
		Node *near_node = NULL;
		Edge *near_curve = NULL;
		RouteResult *rr;
		gfloat l;
		gint start_start;
		gint start_n;

		for (i = 0; i < n->n; ++i) {
			gint edge_index = n->edge_index[i];
			Edge *c = &curve[edge_index];
			Node *n_tmp;

			n_tmp = edge_get_other_side (c, node, n);
			/*
			   g_print ("               now:%.2f  c:%.2f n:%.2f\n", n->l, c->l, n_tmp->l);
			 */
			if (ll > n_tmp->l + c->l) {
				ll = n_tmp->l + c->l;
				near_node = n_tmp;
				near_curve = c;
			}
		}

		get_start_n_l (ch_start, near_curve - curve, &start_start, &start_n, &l);

		start_to_goal += l;

		rr = g_new (RouteResult, 1);
		rr->l = l;
		rr->sl_rpoint = get_lonlat (ch_lonlat, start_start, start_n);

		sl_result = g_slist_prepend (sl_result, rr);

		++count;
		/*
		g_print ("%d:node:%d  curve:%d:  %.2f %.2f", count, near_node - node, near_curve - curve, near_curve->l, n->l);
		*/
		n = near_node;
	}
	g_io_channel_unref (ch_start);
	g_io_channel_unref (ch_lonlat);

	g_print ("route edge n:%d\n", count);
}


static void
route_search (void)
{
	Edge *curve;
	Node *node;
	gint *edge_index;
	Node *now;					/* 現在位置 */
	Node *start_node;
	Node *goal_node;
	GTimeVal start;
	GTimeVal stop;

	/* 空のカーブデータを埋める */
	g_print ("edge_create\n");
	g_get_current_time (&start);
	curve = edge_create ();
	g_get_current_time (&stop);
    g_print("edge_create:%.2f \n", util_diff_gtimeval(&start, &stop));

	g_get_current_time (&start);
	edge_index = edge_index_create ();
	g_get_current_time (&stop);
    g_print("edge_index_create:%.2f \n", util_diff_gtimeval(&start, &stop));

	/* 空のノードデータを埋める */
	g_print ("node_create\n");
	g_get_current_time (&start);
	node = node_create (edge_index);
	g_get_current_time (&stop);
    g_print("node_create:%.2f \n", util_diff_gtimeval(&start, &stop));

	start_node = &node[index_start];
	goal_node = &node[index_goal];

	g_print ("search start!!!!!!\n");

	/* 現在いる node をスタート位置に設定する */
	now = start_node;
	now->l = 0.0;

	tree_new ();
	tree_add_node (now);

	g_get_current_time (&start);
	while (now != goal_node) {
		gint i;

		/* 行ける node の仮距離を計算し、書き換えられた場所を seek のリストに追加する */
		for (i = 0; i < now->n; ++i) {
			gint edge_index = now->edge_index[i];
			Edge *c = &curve[edge_index];
			Node *n;
			gfloat new_l;
			/*
			   g_print ("now:n:%d   %.2f\n", now->n, now->l);
			 */
			n = edge_get_other_side (c, node, now);

			new_l = now->l + c->l;
			/*
			   g_print ("node:l:%.2f   new_l:%.0f\n", n->l, new_l);
			 */
			if (n->l > new_l) {
				if (n->l == FAR_ENOUGH) {
					n->l = new_l;
					tree_add_node (n);
				} else {
					/* 並べかえさせる */
					tree_del_node (n);
					n->l = new_l;
					tree_add_node (n);
				}
			}
		}

		tree_del_node (now);
		if (n_tree == 0) {
			node_free (node, edge_index);
			edge_free (curve);
			g_print ("route search fail\n");
			return;
		}

		/* seek のリストから距離の最も短いものを探す */
		now = tree_get_next_node ();

	}
	tree_seek_free (tree);
	g_get_current_time (&stop);
	g_print ("finish:%.0f\n", now->l);
    g_print("search:%.2f \n", util_diff_gtimeval(&start, &stop));


	/* 経路の抜きだし */
	g_print ("result create\n");
	g_get_current_time (&start);
	create_result (start_node, goal_node, node, curve);
	g_get_current_time (&stop);
    g_print("result:%.2f \n", util_diff_gtimeval(&start, &stop));

	node_free (node, edge_index);
	edge_free (curve);

	g_print ("finish !!!!!\n");
}

/*********************************/

void
route_render_start_goal (void)
{
    gdouble h;

    if(start_x == 0.0 && start_y == 0.0 && start_z == 0.0 &&
       goal_x == 0.0 && goal_y == 0.0 && goal_z == 0.0){
        return;
    }

	h = camera_get_dist ();

	glDisable (GL_DEPTH_TEST);

	glPushMatrix ();
	glTranslated (start_x, start_y, start_z);
	glColor3d (0.0, 0.0, 1.0);
	gdk_gl_draw_sphere (TRUE, h / 200.0, 12, 6);
	glPopMatrix ();

	glPushMatrix ();
	glTranslated (goal_x, goal_y, goal_z);
	glColor3d (1.0, 0.0, 0.0);
	gdk_gl_draw_sphere (TRUE, h / 200.0, 12, 6);
	glPopMatrix ();

	glEnable (GL_DEPTH_TEST);
}

static void
button_start_clicked_cb (GtkButton * button, gpointer user_data)
{
    gdouble lon, lat;
	gint deg, min;
	gdouble sec;
	gchar text_lon[100];
	gchar text_lat[100];

	lon = camera_get_lon ();
	lat = camera_get_lat ();

    util_sec_to_deg_min_sec (lon * 3600.0, &deg, &min, &sec);
	g_sprintf (text_lon, "経度：E %3d°%2d’%2.0f", deg, min, sec);
    util_sec_to_deg_min_sec (lat * 3600.0, &deg, &min, &sec);
	g_sprintf (text_lat, "緯度：N %3d°%2d’%2.0f", deg, min, sec);

	index_start = route_get_near_index (lon * 3600.0, lat * 3600.0);

    mmap_deg_to_xyz (lon, lat, 0.0, &start_x, &start_y, &start_z);

    g_print ("%.2f %.2f %.2f\n", start_x, start_y, start_z);

	gtk_label_set_text (GTK_LABEL (label_start_lon), text_lon);
	gtk_label_set_text (GTK_LABEL (label_start_lat), text_lat);

    glarea_force_render();
}

static void
button_goal_clicked_cb (GtkButton * button, gpointer user_data)
{
    gdouble lon, lat;
	gint deg, min;
	gdouble sec;
	gchar text_lon[100];
	gchar text_lat[100];

	lon = camera_get_lon ();
	lat = camera_get_lat ();

    util_sec_to_deg_min_sec (lon * 3600.0, &deg, &min, &sec);
	g_sprintf (text_lon, "経度：E %3d°%2d’%2.0f", deg, min, sec);
    util_sec_to_deg_min_sec (lat * 3600.0, &deg, &min, &sec);
	g_sprintf (text_lat, "緯度：N %3d°%2d’%2.0f", deg, min, sec);

	index_goal = route_get_near_index (lon * 3600.0, lat * 3600.0);

    mmap_deg_to_xyz (lon, lat, 0.0, &goal_x, &goal_y, &goal_z);

	gtk_label_set_text (GTK_LABEL (label_goal_lon), text_lon);
	gtk_label_set_text (GTK_LABEL (label_goal_lat), text_lat);

    glarea_force_render();
}

static void
button_search_clicked_cb (GtkButton * button, gpointer user_data)
{
	gchar text_l[100];

	if (index_start == -1 || index_goal == -1) {
		return;
	}
	route_search ();

	g_sprintf (text_l, "距離：%8.1f(km)", start_to_goal / 1000.0);
	gtk_label_set_text (GTK_LABEL (label_result), text_l);

    glarea_force_render();
}

static void
button_clear_clicked_cb (GtkButton * button, gpointer user_data)
{
	gchar text_lon[] = "経度：";
	gchar text_lat[] = "緯度：";
	gchar text_l[] = "距離：";

	index_start = -1;
	index_goal = -1;
	start_x = 0.0;
	start_y = 0.0;
	start_z = 0.0;
	goal_x = 0.0;
	goal_y = 0.0;
	goal_z = 0.0;
	start_to_goal = 0.0;
	gtk_label_set_text (GTK_LABEL (label_start_lon), text_lon);
	gtk_label_set_text (GTK_LABEL (label_start_lat), text_lat);
	gtk_label_set_text (GTK_LABEL (label_goal_lon), text_lon);
	gtk_label_set_text (GTK_LABEL (label_goal_lat), text_lat);
	gtk_label_set_text (GTK_LABEL (label_result), text_l);

	result_free ();
    glarea_force_render();
}

static void
dialog_response_cb (GtkDialog * dialog, gint arg1, gpointer user_data)
{
    g_print ("dialog_response_cb\n");

    gtk_widget_destroy (GTK_WIDGET (dialog));

    window_menu_route_sensitive (TRUE);
}

void
route_dialog_create (void)
{
	GtkWidget *dialog;
    GtkWidget *button_start;
    GtkWidget *button_goal;
    GtkWidget *button_search;
    GtkWidget *button_clear;
	GtkWidget *vbox;
	GtkWidget *vbox_start;
	GtkWidget *vbox_goal;
	GtkWidget *vbox_result;
	GtkWidget *separator;
	GtkWidget *label;
	GtkWidget *frame_start;
	GtkWidget *frame_goal;
	GtkWidget *frame_result;

    window_menu_route_sensitive (FALSE);

    route_data_check ();

	separator = gtk_hseparator_new ();
	label = gtk_label_new ("経路検索");

	label_start_lon = gtk_label_new ("経度：");
	label_start_lat = gtk_label_new ("緯度：");
	gtk_misc_set_alignment (GTK_MISC (label_start_lon), 0.0, 0.0);
	gtk_misc_set_alignment (GTK_MISC (label_start_lat), 0.0, 0.0);

	label_goal_lon = gtk_label_new ("経度：");
	label_goal_lat = gtk_label_new ("緯度：");
	gtk_misc_set_alignment (GTK_MISC (label_goal_lon), 0.0, 0.0);
	gtk_misc_set_alignment (GTK_MISC (label_goal_lat), 0.0, 0.0);

	label_result = gtk_label_new ("距離：");
	gtk_misc_set_alignment (GTK_MISC (label_result), 0.0, 0.0);

	frame_start = gtk_frame_new ("スタート");
	vbox_start = gtk_vbox_new (FALSE, 3);
	gtk_box_pack_start (GTK_BOX (vbox_start), label_start_lon, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox_start), label_start_lat, FALSE, FALSE, 0);
	gtk_container_add (GTK_CONTAINER (frame_start), vbox_start);

	frame_goal = gtk_frame_new ("ゴール");
	vbox_goal = gtk_vbox_new (FALSE, 3);
	gtk_box_pack_start (GTK_BOX (vbox_goal), label_goal_lon, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox_goal), label_goal_lat, FALSE, FALSE, 0);
	gtk_container_add (GTK_CONTAINER (frame_goal), vbox_goal);

	frame_result = gtk_frame_new ("結果");
	vbox_result = gtk_vbox_new (FALSE, 20);
	gtk_box_pack_start (GTK_BOX (vbox_result), label_result, FALSE, FALSE, 0);
	gtk_container_add (GTK_CONTAINER (frame_result), vbox_result);

	button_start = gtk_button_new_with_label ("スタート地点設定");
	g_signal_connect (button_start, "clicked", G_CALLBACK (button_start_clicked_cb), NULL);
	button_goal = gtk_button_new_with_label ("ゴール地点設定");
	g_signal_connect (button_goal, "clicked", G_CALLBACK (button_goal_clicked_cb), NULL);

	button_search = gtk_button_new_with_label ("検索");
	g_signal_connect (button_search, "clicked", G_CALLBACK (button_search_clicked_cb), NULL);
	button_clear = gtk_button_new_with_label ("クリアー");
	g_signal_connect (button_clear, "clicked", G_CALLBACK (button_clear_clicked_cb), NULL);


	vbox = gtk_vbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), separator, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), frame_start, FALSE, FALSE, 2);
	gtk_box_pack_start (GTK_BOX (vbox), frame_goal, FALSE, FALSE, 2);
	gtk_box_pack_start (GTK_BOX (vbox), frame_result, FALSE, FALSE, 4);
	gtk_box_pack_start (GTK_BOX (vbox), button_start, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), button_goal, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), button_search, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), button_clear, FALSE, FALSE, 0);

    dialog = gtk_dialog_new_with_buttons ("ルート検索", NULL,
                                           GTK_DIALOG_DESTROY_WITH_PARENT, GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE, NULL);
    g_signal_connect (G_OBJECT (dialog), "response", G_CALLBACK (dialog_response_cb), NULL);

    gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), vbox);
	gtk_window_set_default_size (GTK_WINDOW (dialog), 220, 300);
    gtk_widget_show_all (dialog);
}
