/*
 * MMap+ - 3d image viewer
 * Copyright 2005, 2006 Masahide Miyake
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/*
#define DB(x) (x)
*/
#define DB(x)

#include <gtk/gtk.h>
#include <GL/gl.h>
#include <GL/glext.h>
#include <stdlib.h>
#include <string.h>

#include "mmap.h"
#include "text.h"
#include "disk.h"
#include "util.h"
#include "nmapdata.h"
#include "nmapdata_all.h"
#include "gsi.h"

/*****************************************/

static void
data_common_vertices_debug_print(Vertices *vertices)
{
    gint i;

    g_print("data_common_vertices_debug_print:n:%d\n", vertices->n);
    for(i = 0; i < vertices->n; ++i){
        gfloat *v = (vertices->v) + i * 3;
        g_print("data_common_vertices_debug_print:%10d:%.2f %.2f %.2f\n", i, v[0], v[1], v[2]);
    }
}

Vertices *
data_common_vertices_new(void)
{
    Vertices *vertices;

    vertices = g_new(Vertices, 1);
	vertices->v = NULL;
    vertices->n = 0;

    return vertices;
}

void
data_common_vertices_add(Vertices *vertices, gfloat *v, gint n)
{
    if (vertices->v == NULL) {
        vertices->v = g_memdup(v, n * sizeof (gfloat) * 3);
        vertices->n = n;
    } else {
        vertices->v = g_realloc (vertices->v, (vertices->n + n) * sizeof (gfloat) * 3);
        g_memmove (vertices->v + vertices->n * 3, v, n * sizeof (gfloat) * 3);
        vertices->n += n;
    }
}

void
data_common_vertices_free(Vertices *vertices)
{
    if (vertices == NULL){
        return;
    }
    g_free(vertices->v);
    g_free(vertices);
}

/**************************/

typedef struct {
	GArray *first;    
	GArray *count;
    /* v の first[n] バイト目を先頭に count[n] 個がの座標データでひとまとまり */
}LinesColor;

struct _Lines {
    gint n;
    LinesColor *linescolor;
};

static void
data_common_lines_debug_print(Lines *lines)
{
    gint i, j;

    for(i = 0; i < lines->n; ++i){
        LinesColor *lc = &(lines->linescolor[i]);
        gint sum = 0;

        g_print("data_common_lines_debug_print:%10d: n:%d %d\n", i, lc->first->len, lc->count->len);

        for(j = 0; j < lc->first->len; ++j){

            gint first = g_array_index(lc->first, gint, j);
            gint count = g_array_index(lc->count, gint, j);

            sum += count;

            g_print("data_common_lines_debug_print:%10d:%10d: first %d   count %d\n", i, sum, first, count);
        }
    }
}

/* n このデータを持ち、add の時に振り分けることで色や線の太さを変えることができる */
Lines * 
data_common_lines_new(gint n)
{
    Lines *lines;
    gint i;

    lines = g_new(Lines, 1);
    lines->n = n;
    lines->linescolor = g_new(LinesColor, n);
        
    for(i = 0; i < n; ++i){
        LinesColor *lc = &(lines->linescolor[i]);

	    lc->first = g_array_new (FALSE, FALSE, sizeof (gint));
	    lc->count = g_array_new (FALSE, FALSE, sizeof (gint));
    }

    return lines;
}

void
data_common_lines_add(Lines *lines, gint n, gint first, gint count)
{
    LinesColor *lc = &(lines->linescolor[n]);

    g_array_append_val (lc->first, first);
	g_array_append_val (lc->count, count);
}

void
data_common_lines_free(Lines *lines)
{
    gint i;

    if (lines == NULL){
        return;
    }

    for(i = 0; i < lines->n; ++i){
        LinesColor *lc = &(lines->linescolor[i]);

	    g_array_free (lc->first, TRUE);
	    g_array_free (lc->count, TRUE);
    }
    g_free(lines->linescolor);
    g_free(lines);
}

void
data_common_lines_draw(Lines *lines, Vertices *vertices, LineType *linetype)
{
    gint i;

    if(lines == NULL){
        return;
    }

/*
data_common_vertices_debug_print(vertices);
*/
    
	glDisableClientState (GL_TEXTURE_COORD_ARRAY);

	glDisable (GL_TEXTURE_2D);

	glVertexPointer (3, GL_FLOAT, 0, vertices->v);
    for(i = 0; i < lines->n; ++i){
        LinesColor *lc = &(lines->linescolor[i]);
        LineType lt = linetype[i];

        if(lc->first->len == 0 || lc->count->len == 0){
            continue;
        }
/*
        g_print("data_common_lines_draw2: %d\n", i);
        g_print("data_common_lines_draw: linewidth %.2f\n", lt.linewidth);
        g_print("data_common_lines_draw: color %.2f %.2f %.2f\n", lt.color[0], lt.color[1], lt.color[2]);
        g_print("data_common_lines_draw: count->len %d  %p %p\n", lc->count->len, lc->first->data, lc->count->data);
        g_print("data_common_lines_draw: v %p\n", vertices->v);
data_common_lines_debug_print(lines);
*/

	    glLineWidth (lt.linewidth);
	    glColor3fv (lt.color);
	    glMultiDrawArrays (GL_LINE_STRIP, (gint *)(lc->first->data), (gint *)(lc->count->data), lc->count->len);
    }

	glEnable (GL_TEXTURE_2D);

	glEnableClientState (GL_TEXTURE_COORD_ARRAY);
}

/**************************/

Text *
data_common_text_new(void)
{
    Text *text;

    text = g_new(Text, 1);
	text->index = g_array_new (FALSE, FALSE, sizeof (guint16));
    text->pixtext = g_ptr_array_new ();

    return text;
}

void
data_common_text_add(Text *text, guint16 index, PixText *pixtext)
{
    g_array_append_val (text->index, index);
    if(pixtext != NULL){
        /* ここにくる場合は、全ての pixtext が NULL になるはずなので、 text->pixtext->len がゼロになる */
        /* ということで、不整合はおきないはず・・ */
        g_ptr_array_add (text->pixtext, pixtext);
    }
}

void
data_common_text_free_pixtext(Text *text)
{
    if (text == NULL || text->pixtext == NULL){
        return;
    }
    g_ptr_array_free (text->pixtext, FALSE);
    text->pixtext = NULL;
}

void
data_common_text_free(Text *text)
{
    if (text == NULL){
        return;
    }
	g_array_free (text->index, TRUE);
    data_common_text_free_pixtext(text);
    g_free(text);
}

void
data_common_text_draw_text(Text *text, Vertices *vertices)
{
    gint i;
    gint n;

    if(text == NULL || text->index->len == 0 || text->pixtext == NULL || text->pixtext->len == 0){
        return;
    }

    n = text->index->len;

	glDisable (GL_TEXTURE_2D);
    glEnable (GL_BLEND);

    for (i = 0; i < n; ++i) {
        gint index = g_array_index(text->index, guint16, i);
        PixText *pixtext = g_ptr_array_index(text->pixtext, i);

        if(pixtext == NULL){
            continue;
        }

        glRasterPos3fv (vertices->v + index * 3);
        glDrawPixels (pixtext->w, pixtext->h, GL_RGBA, GL_UNSIGNED_INT_8_8_8_8_REV, pixtext->pix);
    }

    glDisable (GL_BLEND);
	glEnable (GL_TEXTURE_2D);
}

void
data_common_text_draw_points(Text *text, Vertices *vertices, gfloat *color)
{
    if(text == NULL || text->index->len == 0){
        return;
    }

	glDisableClientState (GL_TEXTURE_COORD_ARRAY);
	glDisable (GL_TEXTURE_2D);

	glVertexPointer (3, GL_FLOAT, 0, vertices->v);
	glColor3fv (color);
	glDrawElements (GL_POINTS, text->index->len, GL_UNSIGNED_SHORT, text->index->data);

	glEnableClientState (GL_TEXTURE_COORD_ARRAY);
	glEnable (GL_TEXTURE_2D);
}

/**************************/

GIOChannel *
data_common_channel_open(const gchar *name, const gchar *type)
{
	GIOChannel *ch;
    gchar *fullpath;

	fullpath = g_strconcat (mmap_dir_gsi_vn, name, type, NULL);
	ch = disk_channel_open (fullpath, READ_BIN);
	g_free (fullpath);

    return ch;
}

/**************************/

#if 0

typedef gint (*LineTypeNFunc)(void);
typedef GIOChannel *(*IOChannelFunc)(void);
typedef gpointer (*SubNewFunc)(void);
typedef gpointer (*SubFillFunc)(SData *, Slp *, gpointer, gchar *);
typedef void (*SubFreeFunc)(gpointer);
typedef LineType *(*LineTypeFunc)(void);
typedef gfloat *(*PointColorFunc)(void);

typedef struct {
    kkline_type
}SDataFunc;

typedef struct {
    NMapDataType type;

    GSList *sub;

    Vertices *vertices;
    Lines *lines;
    Text *text;
}SData;

void
sdata_draw (SData *sdata)
{
    SdataFunc *sdf = sdf_all[sdf->type];

    LineType *linetype = sdf->linetype();
    gfloat *color = sdf->pointcolor();;

    data_common_lines_draw(sdata->lines, sdata->vertices, linetype);
    data_common_text_draw_points(sdata->text, sdata->vertices, color);
    data_common_text_draw_text(sdata->text, sdata->vertices);
}

void
sdata_free(Sdata *sdata)
{
	GSList *sl;
    SdataFunc *sdf = sdf_all[sdf->type];

	for (sl = sdata->sub; sl != NULL; sl = sl->next) {
		gpointer sub = sl->data;

        sdf->sub_free(sub);
	}
	g_slist_free (sdata->sub);

    data_common_vertices_free(sdata->vertices);
    data_common_lines_free(sdata->lines);
    data_common_text_free(sdata->text);
	g_free (sdata);
}

SData *
sdata_create (SDataType type, const gchar * name, Slp * slp)
{
	gchar *line;
	GIOChannel *ch;
	GIOStatus status;
	GError *err = NULL;
    SData *sdata;
    SdataFunc *sdf = sdf_all[type];

	sdata = g_new0 (SData, 1);
    sdata->vertices = data_common_vertices_new();
    sdata->lines = data_common_lines_new(sdf->sdata_line_type_n());
    sdata->text = data_common_text_new();

	ch = sdf->io_channel();
	status = g_io_channel_read_line (ch, &line, NULL, NULL, &err);
	while (status != G_IO_STATUS_EOF) {
		gpointer sub;
		const gchar *p;
		Data25k top;

		if (status == G_IO_STATUS_ERROR) {
			g_print ("err:cm_create:%s:\n", name);

			g_free (line);
			status = g_io_channel_read_line (ch, &line, NULL, NULL, &err);
			continue;
		}
		line = g_strchomp (line);

		sub = sdf->sub_new();

		gsi25k_get_data (line, &top);
        sdf->sub_fill(sdata, slp, sub, top.value);
		g_free (top.name);
		g_free (top.value);

		sdata->sub = g_slist_prepend (sdata->sub, sub);

		g_free (line);
		status = g_io_channel_read_line (ch, &line, NULL, NULL, &err);
	}
	g_free (line);
	g_io_channel_unref (ch);

	return cm;
}

#endif

/*****************************************/

static gchar *
slm_get_line (GIOChannel * ch, const gchar * name)
{
	gchar *line;
	GIOStatus status;
	GError *err = NULL;

	status = g_io_channel_read_line (ch, &line, NULL, NULL, &err);
	if (status == G_IO_STATUS_ERROR) {
		g_print ("error:slm read:%s:%s\n", name, err->message);
		g_free (line);
		exit (1);
	}

	line = g_strchomp (line);

	return line;
}

void
slm_free (Slm * slm)
{
	g_free (slm);
}

/* name:11111 */
Slm *
slm_create (const gchar * name)
{
	gchar *line = NULL;
	GIOChannel *ch;
	Slm *slm;
	gdouble x, y;
	gdouble nx, ny;
	gint n_slp, n_mhslp;
	gint64 x64, y64;

	ch = data_common_channel_open(name, ".slm");

	line = slm_get_line (ch, name);
	sscanf (line, "%Ld,%Ld", &x64, &y64);
	x = (gdouble) x64 / 10000.0;
	y = (gdouble) y64 / 10000.0;
	g_free (line);

	line = slm_get_line (ch, name);
	sscanf (line, "%lf,%lf", &nx, &ny);
	g_free (line);

	line = slm_get_line (ch, name);
	sscanf (line, "%d", &n_slp);
	g_free (line);

	line = slm_get_line (ch, name);
	sscanf (line, "%d", &n_mhslp);
	g_free (line);

	g_io_channel_unref (ch);

	/*
	DB (g_print ("slm :%.4f:%.4f:%.4f:%.4f:%d:%d\n", x, y, nx / 10000.0, ny / 1000.0, n_slp, n_mhslp));
	*/

	slm = g_new (Slm, 1);
	slm->x = x;
	slm->y = y;
	slm->nx = nx / 10000.0;
	slm->ny = ny / 10000.0;
	slm->n_slp = n_slp;
	slm->n_mhslp = n_mhslp;

	return slm;
}

/*****************************************/

void
slp_free (Slp * slp)
{
	g_free (slp->v);
	g_free (slp);
}

Slp *
slp_create (const gchar * name, gint n, gdouble x0, gdouble y0)
{
	gint i;
	gchar *fullpath;
	gfloat *v;
	gchar *bin;
	gchar *p_bin;
	gint size;
	Slp *slp;
	gint line_n;

	fullpath = g_strconcat (mmap_dir_gsi_vn, name, ".slp", NULL);
    bin = disk_load_bin (fullpath, &size);
	g_free (fullpath);

	line_n = size / n;

	slp = g_new (Slp, 1);
	slp->n = n;
	slp->v = g_new (gfloat, 3 * n);

	p_bin = bin;
	v = slp->v;

	for (i = 0; i < n; ++i, v += 3, p_bin += line_n) {
		gdouble x_offset = 0.0;
		gdouble y_offset = 0.0;
		gdouble lon, lat, alt;
		gdouble x, y, z;

		x_offset += (p_bin[0] - '0') * 1000.0;
		x_offset += (p_bin[1] - '0') * 100.0;
		x_offset += (p_bin[2] - '0') * 10.0;
		x_offset += (p_bin[3] - '0') * 1.0;
		x_offset += (p_bin[4] - '0') / 10.0;
		x_offset += (p_bin[5] - '0') / 100.0;
		x_offset += (p_bin[6] - '0') / 1000.0;
		x_offset += (p_bin[7] - '0') / 10000.0;

		y_offset += (p_bin[8] - '0') * 1000.0;
		y_offset += (p_bin[9] - '0') * 100.0;
		y_offset += (p_bin[10] - '0') * 10.0;
		y_offset += (p_bin[11] - '0') * 1.0;
		y_offset += (p_bin[12] - '0') / 10.0;
		y_offset += (p_bin[13] - '0') / 100.0;
		y_offset += (p_bin[14] - '0') / 1000.0;
		y_offset += (p_bin[15] - '0') / 10000.0;

		lon = (x0 + x_offset) / 3600.0;
		lat = (y0 + y_offset) / 3600.0;
		alt = 0.0;

        /*
        g_print("slp_create:lon %.2f lat %.2f alt %.2f\n", lon, lat, alt);
        */

        mmap_deg_to_xyz (lon, lat, alt, &x, &y, &z);

		*(v + 0) = (gfloat) x;
		*(v + 1) = (gfloat) y;
		*(v + 2) = (gfloat) z;
		/*
		g_print ("slp :x0:%.2f  y0:%.2f   xoff:%.2f yoff:%.2f", x0, y0, x_offset, y_offset);
		g_print ("slp :lonlatalt:%.2f %.2f %.2f::: xyz:%.2f %.2f %.2f\n", lon,lat,alt, x, y, z);
		 */
	}

	g_free (bin);

	return slp;
}

/*****************************************/

enum {
    GK_TYPE_0,
    GK_TYPE_1,
    GK_TYPE_N
};

void
gk_state_change (Gk * gk)
{
	GSList *sl;

	DB (g_print ("gk_state_change\n"));

    data_common_lines_free(gk->lines);
    gk->lines = data_common_lines_new(GK_TYPE_N);

	for (sl = gk->sub; sl != NULL; sl = sl->next) {
		Gksub *sub = sl->data;
		DB (g_print ("gk_state_change:%x %x %d %d\n", sub->sr, sub->jt, sub->first, sub->count));

		if (nmapdata_selector_check_onoff(sub->sr) == TRUE && nmapdata_selector_check_onoff(sub->jt) == TRUE) {
			if (sub->sr == 0x29) {
                data_common_lines_add(gk->lines, GK_TYPE_0, sub->first, sub->count);
			} else {
                data_common_lines_add(gk->lines, GK_TYPE_1, sub->first, sub->count);
			}
		}
	}
}

void
gk_draw (Gk * gk)
{
    LineType linetype[] = { {2.0, { 1.0, 0.1, 0.1 }}, {2.0, { 1.0, 0.5, 0.5 }} };
    /*
	g_print ("gk_draw\n");
    */
    data_common_lines_draw(gk->lines, gk->vertices, linetype);
}

void
gk_free (Gk * gk)
{
	GSList *sl;

	for (sl = gk->sub; sl != NULL; sl = sl->next) {
		Gksub *sub = sl->data;
		g_free (sub);
	}
	g_slist_free (gk->sub);
    data_common_vertices_free(gk->vertices);
    data_common_lines_free(gk->lines);
	g_free (gk);
}

Gk *
gk_create (const gchar * name, Slp * slp)
{
	gchar *line;
	GIOChannel *ch;
	GIOStatus status;
	GError *err = NULL;
	Gk *gk;

	gk = g_new0 (Gk, 1);
    gk->vertices = data_common_vertices_new();
    gk->lines = data_common_lines_new(GK_TYPE_N);

	ch = data_common_channel_open(name, "gk.sal");
	status = g_io_channel_read_line (ch, &line, NULL, NULL, &err);
	while (status != G_IO_STATUS_EOF) {
		Gksub *sub;
		const gchar *p;
		Data25k top;

		if (status == G_IO_STATUS_ERROR) {
			g_print ("err:create_gk:%s:\n", name);
			exit (1);
		}
		line = g_strchomp (line);

		sub = g_new0 (Gksub, 1);        /* 無いデータもあるのでゼロの初期化必須 */

		gsi25k_get_data (line, &top);

		p = top.value;
		do {
			Data25k d25k;

			p = gsi25k_get_data (p, &d25k);
			if (d25k.id[0] == 'S' && d25k.id[1] == 'R') {
				sub->sr = gsi25k_hex_to_int (d25k.value);

			} else if (d25k.id[0] == 'J' && d25k.id[1] == 'T') {
				sub->jt = gsi25k_hex_to_int (d25k.value);

			} else if (d25k.id[0] == 'C' && d25k.id[1] == 'V') {
				gint n;
				gfloat *v;

				v = gsi25k_get_curve (d25k.value, slp, &n);

				sub->first = gk->vertices->n;
				sub->count = n;
                /*
				g_print("!!!!!!:first:%d count:%d v:%p\n",sub->first, sub->count, v);
                */
                data_common_vertices_add(gk->vertices, v, n);

				g_free (v);

			} else {
				DB (g_print ("line_to_data:err:%s:\n", p));
			}

			g_free (d25k.name);
			g_free (d25k.value);
		} while (*p != '\0');
		g_free (top.name);
		g_free (top.value);

		gk->sub = g_slist_prepend (gk->sub, sub);


		g_free (line);
		status = g_io_channel_read_line (ch, &line, NULL, NULL, &err);
	}
	g_free (line);
	g_io_channel_unref (ch);

	return gk;
}

/*****************************************/

enum {
    SK_TYPE_0,
    SK_TYPE_1,
    SK_TYPE_N
};

void
sk_state_change (Sk * sk)
{
	GSList *sl;

	DB (g_print ("sk_state_change\n"));

    data_common_lines_free(sk->lines);
    sk->lines = data_common_lines_new(SK_TYPE_N);

	for (sl = sk->sub; sl != NULL; sl = sl->next) {
		Sksub *sub = sl->data;
		DB (g_print ("sk_state_change:%x %x %d %d\n", sub->sr, sub->jt, sub->first, sub->count));

		if (nmapdata_selector_check_onoff(sub->sr) == TRUE) {
			if (sub->sr == 0x51) {
                data_common_lines_add(sk->lines, SK_TYPE_0, sub->first, sub->count);
			} else {
                data_common_lines_add(sk->lines, SK_TYPE_1, sub->first, sub->count);
			}
		}
	}
}

void
sk_draw (Sk * sk)
{
    LineType linetype[] = { {1.0, { 0.5, 0.5, 1.0 }}, 
                            {2.0, { 0.4, 0.4, 1.0 }} };

    data_common_lines_draw(sk->lines, sk->vertices, linetype);
}

void
sk_free (Sk * sk)
{
	GSList *sl;

	for (sl = sk->sub; sl != NULL; sl = sl->next) {
		Sksub *sub = sl->data;
		g_free (sub);
	}
	g_slist_free (sk->sub);
    data_common_vertices_free(sk->vertices);
    data_common_lines_free(sk->lines);
	g_free (sk);
}

Sk *
sk_create (const gchar * name, Slp * slp)
{
	gchar *line;
	GIOChannel *ch;
	GIOStatus status;
	GError *err = NULL;
	Sk *sk;

	sk = g_new0 (Sk, 1);
    sk->vertices = data_common_vertices_new();
    sk->lines = data_common_lines_new(SK_TYPE_N);

	ch = data_common_channel_open(name, "sk.sal");
	status = g_io_channel_read_line (ch, &line, NULL, NULL, &err);
	while (status != G_IO_STATUS_EOF) {
		Sksub *sub;
		const gchar *p;
		Data25k top;

		if (status == G_IO_STATUS_ERROR) {
			g_print ("err:make_sk:%s:\n", name);
			exit (1);
		}
		line = g_strchomp (line);

		/* 無いデータもあるのでゼロの初期化必須 */
		sub = g_new0 (Sksub, 1);

		gsi25k_get_data (line, &top);

		p = top.value;
		do {
			Data25k d25k;

			p = gsi25k_get_data (p, &d25k);
			if (d25k.id[0] == 'S' && d25k.id[1] == 'R') {
				sub->sr = gsi25k_hex_to_int (d25k.value);

			} else if (d25k.id[0] == 'C' && d25k.id[1] == 'V') {
				gint n;
				gfloat *v;

				v = gsi25k_get_curve (d25k.value, slp, &n);

				sub->first = sk->vertices->n;
				sub->count = n;
				/*
				   g_print("!!!!!!:first:%d count:%d v:%p\n",sub->first, sub->count, v);
				 */
                data_common_vertices_add(sk->vertices, v, n);

				g_free (v);

			} else {
				DB (g_print ("line_to_data:err:%s:\n", p));
			}

			g_free (d25k.name);
			g_free (d25k.value);
		} while (*p != '\0');
		g_free (top.name);
		g_free (top.value);

		sk->sub = g_slist_prepend (sk->sub, sub);


		g_free (line);
		status = g_io_channel_read_line (ch, &line, NULL, NULL, &err);
	}
	g_free (line);
	g_io_channel_unref (ch);

	return sk;
}

/*****************************************/

enum {
    DK_TYPE_0,
    DK_TYPE_1,
    DK_TYPE_2,
    DK_TYPE_3,
    DK_TYPE_4,
    DK_TYPE_N
};

void
dk_state_change (Dk * dk)
{
	GSList *sl;

	DB (g_print ("dk_state_change\n"));

    data_common_lines_free(dk->lines);
    dk->lines = data_common_lines_new(DK_TYPE_N);
    /*
    data_common_text_free(dk->text);
    dk->text = data_common_text_new();
    */

	for (sl = dk->sub; sl != NULL; sl = sl->next) {
		Dksub *sub = sl->data;
		DB (g_print ("dk_state_change:%x %d %d\n", sub->sb, sub->first, sub->count));

		if (nmapdata_selector_check_onoff(sub->sb) == TRUE && nmapdata_selector_check_onoff(sub->jt) == TRUE && 
                nmapdata_selector_check_onoff(sub->yu) == TRUE && nmapdata_selector_check_onoff(sub->fi) == TRUE) {
			if (sub->kb != 0) {	                                        /* 国道 */
                data_common_lines_add(dk->lines, DK_TYPE_0, sub->first, sub->count);
			} else if (sub->yu == 0x1c) {	                            /* 有料道路 */
                data_common_lines_add(dk->lines, DK_TYPE_1, sub->first, sub->count);
			} else if (sub->fi == 0x17 || sub->fi == 0x18) {	        /* 幅員 3.0m 未満 */
                data_common_lines_add(dk->lines, DK_TYPE_2, sub->first, sub->count);
			} else if (sub->fi == 0x19) {	                            /* 幅員 3.0-5.5m */
                data_common_lines_add(dk->lines, DK_TYPE_3, sub->first, sub->count);
			} else {
                data_common_lines_add(dk->lines, DK_TYPE_4, sub->first, sub->count);
			}
			/*
			   if (sub->kb != NULL && sub->pixtext != NULL) {
                data_common_text_add(dk->text, sub->first, sub->pixtext);
			   }
			 */
		}
	}
}

void
dk_draw (Dk * dk)
{
    LineType linetype[] = {
        {3.0, { 1.0, 0.6, 0.3 }}, 
        {3.0, { 1.0, 0.3, 1.0 }}, 
        {1.0, { 0.8, 0.8, 0.8 }}, 
        {1.0, { 0.6, 0.6, 0.6 }}, 
        {2.0, { 0.6, 0.6, 0.6 }} };

    data_common_lines_draw(dk->lines, dk->vertices, linetype);
	/*
    data_common_text_draw(dk->text, dk->vertices);
	 */
}

/*
void
dk_add_text (Dk * dk)
{
	GSList *sl;

	for (sl = dk->sub; sl != NULL; sl = sl->next) {
		Dksub *sub = sl->data;

        if( sub->kb != NULL && sub->kb[0] != '\0'){
		    sub->pixtext = text_create (sub->kb);
        }
	}
}

void
dk_del_text (Dk * dk)
{
	GSList *sl;

	for (sl = dk->sub; sl != NULL; sl = sl->next) {
		Dksub *sub = sl->data;

        if(sub->pixtext != NULL){
            text_free(sub->pixtext);
        }
		sub->pixtext = NULL;
	}
    data_common_text_free_pixtext(dk->text);
}
*/

void
dk_free (Dk * dk)
{
	GSList *sl;

	for (sl = dk->sub; sl != NULL; sl = sl->next) {
		Dksub *sub = sl->data;
		GSList *k;

		for (k = sub->nm; k != NULL; k = k->next) {
			gchar *nm = k->data;

			g_free (nm);
		}

		g_slist_free (sub->nm);
		/*
		   g_free (sub->kb);
		   if (sub->pixtext != NULL) {
            text_free(sub->pixtext);
		   }
		 */
		g_free (sub);
	}
	g_slist_free (dk->sub);
    data_common_vertices_free(dk->vertices);
    data_common_lines_free(dk->lines);
	/*
    data_common_text_free(dk->text);
	 */
	g_free (dk);
}

Dk *
dk_create (const gchar * name, Slp * slp)
{
	gchar *line;
	GIOChannel *ch;
	GIOStatus status;
	GError *err = NULL;
	Dk *dk;

	dk = g_new0 (Dk, 1);
    dk->vertices = data_common_vertices_new();
    dk->lines = data_common_lines_new(DK_TYPE_N);
    /*
    dk->text = data_common_text_new();
    */

	ch = data_common_channel_open(name, "dk.sal");
	status = g_io_channel_read_line (ch, &line, NULL, NULL, &err);
	while (status != G_IO_STATUS_EOF) {
		Dksub *sub;
		const gchar *p;
		Data25k top;

		if (status == G_IO_STATUS_ERROR) {
			g_print ("err:dk_create:%s:\n", name);
			g_free (line);
			status = g_io_channel_read_line (ch, &line, NULL, NULL, &err);
			continue;
		}
		line = g_strchomp (line);

		sub = g_new0 (Dksub, 1);             /* 無いデータもあるのでゼロの初期化必須 */

		gsi25k_get_data (line, &top);

		p = top.value;
		do {
			Data25k d25k;

			p = gsi25k_get_data (p, &d25k);
            if (gsi25k_compare_id(&d25k, "SB") == TRUE){
				sub->sb = gsi25k_hex_to_int (d25k.value);

			} else if (gsi25k_compare_id(&d25k, "JT") == TRUE){
				sub->jt = gsi25k_hex_to_int (d25k.value);

			} else if (gsi25k_compare_id(&d25k, "YU") == TRUE){
				sub->yu = gsi25k_hex_to_int (d25k.value);

			} else if (gsi25k_compare_id(&d25k, "FI") == TRUE){
				sub->fi = gsi25k_hex_to_int (d25k.value);

			} else if (gsi25k_compare_id(&d25k, "KB") == TRUE){
				sub->kb = (guint16) atoi (d25k.value);
				/*
				   if (d25k.value != NULL) {
				   sub->kb = g_strdup (d25k.value);
				   }
				 */

			} else if (gsi25k_compare_id(&d25k, "NM") == TRUE){
				gchar *name;

				name = util_to_utf (d25k.value, SJIS);
				sub->nm = g_slist_prepend (sub->nm, name);

			} else if (gsi25k_compare_id(&d25k, "HA") == TRUE){
				sub->hatoyo = HA;

			} else if (gsi25k_compare_id(&d25k, "TO") == TRUE){
				sub->hatoyo = TO;

			} else if (gsi25k_compare_id(&d25k, "YO") == TRUE){
				sub->hatoyo = YO;

			} else if (gsi25k_compare_id(&d25k, "CV") == TRUE){
				gint n;
				gfloat *v;

				v = gsi25k_get_curve (d25k.value, slp, &n);

				sub->first = dk->vertices->n;
				sub->count = (guint16) n;
				/*
				   g_print("!!!!!!:first:%d count:%d v:%p\n",sub->first, sub->count, v);
				 */
                data_common_vertices_add(dk->vertices, v, n);

				g_free (v);

			} else {
				DB (g_print ("line_to_data:err:%s:\n", p));
			}

			g_free (d25k.name);
			g_free (d25k.value);
		} while (*p != '\0');
		g_free (top.name);
		g_free (top.value);

		dk->sub = g_slist_prepend (dk->sub, sub);

		g_free (line);
		status = g_io_channel_read_line (ch, &line, NULL, NULL, &err);
	}
	g_free (line);
	g_io_channel_unref (ch);

	return dk;
}

/*****************************************/

gchar **
ek_create (const gchar * name)
{
	gchar *line;
	GIOChannel *ch;
	GIOStatus status;
	GError *err = NULL;
	GPtrArray *array = g_ptr_array_new ();

	ch = data_common_channel_open(name, "ek.sal");
	status = g_io_channel_read_line (ch, &line, NULL, NULL, &err);
	while (status != G_IO_STATUS_EOF) {
		const gchar *p;
		Data25k top;

		if (status == G_IO_STATUS_ERROR) {
			g_print ("err:make_ek:%s:\n", name);
			exit (1);
		}
		line = g_strchomp (line);

		gsi25k_get_data (line, &top);

		p = top.value;
		do {
			Data25k d25k;

			p = gsi25k_get_data (p, &d25k);
			if (d25k.id[0] == 'N' && d25k.id[1] == 'M') {
				gchar *name;

				name = util_to_utf (d25k.value, SJIS);
				g_ptr_array_add (array, name);

				g_free (d25k.name);
				g_free (d25k.value);
				break;

			} else {
				;
				DB (g_print ("line_to_data:err:%s:\n", p));
			}

			g_free (d25k.name);
			g_free (d25k.value);
		} while (*p != '\0');
		g_free (top.name);
		g_free (top.value);

		g_free (line);
		status = g_io_channel_read_line (ch, &line, NULL, NULL, &err);
	}
	g_free (line);
	g_io_channel_unref (ch);

	return (gchar **) g_ptr_array_free (array, FALSE);
}

enum {
    TK_TYPE_0,
    TK_TYPE_1,
    TK_TYPE_2,
    TK_TYPE_N
};

void
tk_state_change (Tk * tk)
{
	GSList *sl;

	DB (g_print ("tk_state_change\n"));

    data_common_lines_free(tk->lines);
    data_common_text_free(tk->text);
    tk->lines = data_common_lines_new(TK_TYPE_N);
    tk->text = data_common_text_new();

	for (sl = tk->sub; sl != NULL; sl = sl->next) {
		Tksub *sub = sl->data;
		DB (g_print ("tk_state_change:%x %d %d\n", sub->sb, sub->first, sub->count));

		if (nmapdata_selector_check_onoff(sub->sb) == TRUE && nmapdata_selector_check_onoff(sub->jt) == TRUE) {
			if (sub->er != NULL) {	/* 駅 */
                data_common_lines_add(tk->lines, TK_TYPE_0, sub->first, sub->count);
			} else if (sub->sb == 0x43 || sub->sb == 0x44) {	/* いわゆる鉄道 */
                data_common_lines_add(tk->lines, TK_TYPE_1, sub->first, sub->count);
			} else {
                data_common_lines_add(tk->lines, TK_TYPE_2, sub->first, sub->count);
			}
			if (sub->er != NULL && sub->pixtext != NULL) {
                data_common_text_add(tk->text, sub->first, sub->pixtext);
			}
		}
	}
}

void
tk_draw (Tk * tk)
{
    LineType linetype[] = {
        {4.0, { 0.1, 0.5, 0.1 }}, 
        {2.0, { 0.2, 0.7, 0.2 }}, 
        {1.0, { 0.0, 0.9, 0.0 }} };

    data_common_lines_draw(tk->lines, tk->vertices, linetype);
    data_common_text_draw_text(tk->text, tk->vertices);
}

void
tk_add_text (Tk * tk)
{
	GSList *sl;

	for (sl = tk->sub; sl != NULL; sl = sl->next) {
		Tksub *sub = sl->data;

		if (sub->er != NULL && sub->er[0] != '\0') {
			sub->pixtext = text_create (sub->er);
		}
	}
}

void
tk_del_text (Tk * tk)
{
	GSList *sl;

	for (sl = tk->sub; sl != NULL; sl = sl->next) {
		Tksub *sub = sl->data;

		if (sub->pixtext != NULL) {
            text_free(sub->pixtext);
		}
		sub->pixtext = NULL;
	}
    data_common_text_free(tk->text);
    tk->text = NULL;
}

void
tk_free (Tk * tk)
{
	GSList *sl;

	for (sl = tk->sub; sl != NULL; sl = sl->next) {
		Tksub *sub = sl->data;
		GSList *k;

		for (k = sub->nm; k != NULL; k = k->next) {
			gchar *nm = k->data;

			g_free (nm);
		}

		g_slist_free (sub->nm);
		g_free (sub->er);
		if (sub->pixtext != NULL) {
            text_free(sub->pixtext);
		}

		g_free (sub);
	}
	g_slist_free (tk->sub);
    data_common_vertices_free(tk->vertices);
    data_common_lines_free(tk->lines);
    data_common_text_free(tk->text);
	g_free (tk);
}

Tk *
tk_create (const gchar * name, Slp * slp)
{
	gchar *line;
	GIOChannel *ch;
	GIOStatus status;
	GError *err = NULL;
	Tk *tk;
	gchar **ek_char;

	tk = g_new0 (Tk, 1);
    tk->vertices = data_common_vertices_new();
    tk->lines = data_common_lines_new(TK_TYPE_N);
    tk->text = data_common_text_new();

	ek_char = ek_create (name);

    ch = data_common_channel_open(name, "tk.sal");
	status = g_io_channel_read_line (ch, &line, NULL, NULL, &err);
	while (status != G_IO_STATUS_EOF) {
		Tksub *sub;
		const gchar *p;
		Data25k top;

		if (status == G_IO_STATUS_ERROR) {
			g_print ("err:tk_create:%s:\n", name);
			g_free (line);
			status = g_io_channel_read_line (ch, &line, NULL, NULL, &err);
			continue;
		}
		line = g_strchomp (line);

		sub = g_new0 (Tksub, 1);

		gsi25k_get_data (line, &top);

		p = top.value;
		do {
			Data25k d25k;

			p = gsi25k_get_data (p, &d25k);
			if (d25k.id[0] == 'S' && d25k.id[1] == 'B') {
				sub->sb = gsi25k_hex_to_int (d25k.value);

			} else if (d25k.id[0] == 'J' && d25k.id[1] == 'T') {
				sub->jt = gsi25k_hex_to_int (d25k.value);

			} else if (d25k.id[0] == 'N' && d25k.id[1] == 'M') {
				gchar *name;

				name = util_to_utf (d25k.value, SJIS);
				sub->nm = g_slist_prepend (sub->nm, name);

			} else if (d25k.id[0] == 'H' && d25k.id[1] == 'A') {
				sub->hatoyo = HA;

			} else if (d25k.id[0] == 'T' && d25k.id[1] == 'O') {
				sub->hatoyo = TO;

			} else if (d25k.id[0] == 'Y' && d25k.id[1] == 'O') {
				sub->hatoyo = YO;

			} else if (d25k.id[0] == 'E' && d25k.id[1] == 'R') {
				Data25k d25kek;
				gchar *p;
				gint index = 0;

				/* d25k.name:IR{EK11111000001} */
				gsi25k_get_data (d25k.name, &d25kek);
				/* d25kek.value:EK11111000001 */
				p = d25kek.value;
				p += 7;
				index += (p[0] - '0') * 100000;
				index += (p[1] - '0') * 10000;
				index += (p[2] - '0') * 1000;
				index += (p[3] - '0') * 100;
				index += (p[4] - '0') * 10;
				index += (p[5] - '0');
				--index;		/* 国土地理院のインデックスはゼロから */

				sub->er = ek_char[index];

				g_free (d25kek.name);
				g_free (d25kek.value);
			} else if (d25k.id[0] == 'C' && d25k.id[1] == 'V') {
				gint n;
				gfloat *v;

				v = gsi25k_get_curve (d25k.value, slp, &n);

				sub->first = tk->vertices->n;
				sub->count = (guint16) n;

                data_common_vertices_add(tk->vertices, v, n);

				g_free (v);

			} else {
				DB (g_print ("line_to_data:err:%s:\n", p));
			}

			g_free (d25k.name);
			g_free (d25k.value);
		} while (*p != '\0');
		g_free (top.name);
		g_free (top.value);

		tk->sub = g_slist_prepend (tk->sub, sub);


		g_free (line);
		status = g_io_channel_read_line (ch, &line, NULL, NULL, &err);
	}
	g_free (line);
	g_io_channel_unref (ch);
	g_free (ek_char);

	return tk;
}

/*****************************************/

enum {
    KK_TYPE_0,
    KK_TYPE_N
};

void
kk_state_change (Kk * kk)
{
	GSList *sl;

	DB (g_print ("kk_state_change\n"));

    data_common_lines_free(kk->lines);
    kk->lines = data_common_lines_new(KK_TYPE_N);

	for (sl = kk->sub; sl != NULL; sl = sl->next) {
		Kksub *sub = sl->data;

		if (nmapdata_selector_check_onoff(sub->sb) == TRUE) {
            data_common_lines_add(kk->lines, KK_TYPE_0, sub->first, sub->count);
		}
	}
}

void
kk_draw (Kk * kk)
{
    LineType linetype[] = { {1.0, { 0.5, 0.5, 1.0 }} };

    data_common_lines_draw(kk->lines, kk->vertices, linetype);
}

void
kk_free (Kk * kk)
{
	GSList *sl;

	for (sl = kk->sub; sl != NULL; sl = sl->next) {
		Kksub *sub = sl->data;
		GSList *k;

		for (k = sub->nm; k != NULL; k = k->next) {
			gchar *nm = k->data;

			g_free (nm);
		}

		g_slist_free (sub->nm);
		g_free (sub);
	}
	g_slist_free (kk->sub);
    data_common_vertices_free(kk->vertices);
    data_common_lines_free(kk->lines);
	g_free (kk);
}

Kk *
kk_create (const gchar * name, Slp * slp)
{
	gchar *line;
	GIOChannel *ch;
	GIOStatus status;
	GError *err = NULL;
	Kk *kk;

	kk = g_new0 (Kk, 1);
    kk->vertices = data_common_vertices_new();
    kk->lines = data_common_lines_new(KK_TYPE_N);

	ch = data_common_channel_open(name, "kk.sal");
	status = g_io_channel_read_line (ch, &line, NULL, NULL, &err);
	while (status != G_IO_STATUS_EOF) {
		Kksub *sub;
		const gchar *p;
		Data25k top;

		if (status == G_IO_STATUS_ERROR) {
			g_print ("err:make_kk:%s:\n", name);
			exit (1);
		}
		line = g_strchomp (line);

		/* 無いデータもあるのでゼロの初期化必須 */
		sub = g_new0 (Kksub, 1);

		gsi25k_get_data (line, &top);

		p = top.value;
		do {
			Data25k d25k;

			p = gsi25k_get_data (p, &d25k);
			if (d25k.id[0] == 'S' && d25k.id[1] == 'B') {
				sub->sb = gsi25k_hex_to_int (d25k.value);

			} else if (d25k.id[0] == 'N' && d25k.id[1] == 'M') {
				gchar *name;

				name = util_to_utf (d25k.value, SJIS);
				sub->nm = g_slist_prepend (sub->nm, name);

			} else if (d25k.id[0] == 'H' && d25k.id[1] == 'A') {
				sub->hatoyo = HA;

			} else if (d25k.id[0] == 'T' && d25k.id[1] == 'O') {
				sub->hatoyo = TO;

			} else if (d25k.id[0] == 'C' && d25k.id[1] == 'V') {
				gint n;
				gfloat *v;

				v = gsi25k_get_curve (d25k.value, slp, &n);

				sub->first = kk->vertices->n;
				sub->count = n;

                data_common_vertices_add(kk->vertices, v, n);

				g_free (v);

			} else {
				DB (g_print ("line_to_data:err:%s:\n", p));
			}

			g_free (d25k.name);
			g_free (d25k.value);
		} while (*p != '\0');
		g_free (top.name);
		g_free (top.value);

		kk->sub = g_slist_prepend (kk->sub, sub);


		g_free (line);
		status = g_io_channel_read_line (ch, &line, NULL, NULL, &err);
	}
	g_free (line);
	g_io_channel_unref (ch);

	return kk;
}

/*****************************************/

void
kj_state_change (Kj * kj)
{
	GSList *sl;

    data_common_text_free(kj->text);
    kj->text = data_common_text_new();

	for (sl = kj->sub; sl != NULL; sl = sl->next) {
		Kjsub *sub = sl->data;
		DB (g_print ("kj_state_change:%x %d\n", sub->sb, sub->index));

		if (nmapdata_selector_check_onoff(sub->sr) == TRUE) {
            data_common_text_add(kj->text, sub->index, sub->pixtext);
		}
	}
}

void
kj_draw (Kj * kj)
{
	gfloat c0[] = { 1.0, 0.0, 0.0 };

    data_common_text_draw_points(kj->text, kj->vertices, c0);
    data_common_text_draw_text(kj->text, kj->vertices);
}

void
kj_add_text (Kj * kj)
{
	GSList *sl;

	for (sl = kj->sub; sl != NULL; sl = sl->next) {
		Kjsub *sub = sl->data;

		sub->pixtext = text_create (sub->hk_char);
	}
}

void
kj_del_text (Kj * kj)
{
	GSList *sl;

	for (sl = kj->sub; sl != NULL; sl = sl->next) {
		Kjsub *sub = sl->data;

        text_free(sub->pixtext);
		sub->pixtext = NULL;
	}
    data_common_text_free_pixtext(kj->text);
}

void
kj_free (Kj * kj)
{
	GSList *sl;

	for (sl = kj->sub; sl != NULL; sl = sl->next) {
		Kjsub *sub = sl->data;

		g_free (sub->hk_char);
		if (sub->pixtext != NULL) {
            text_free(sub->pixtext);
		}
		g_free (sub);
	}
	g_slist_free (kj->sub);
    data_common_vertices_free(kj->vertices);
    data_common_text_free(kj->text);
	g_free (kj);
}

Kj *
kj_create (const gchar * name, Slp * slp)
{
	gchar *line;
	GIOChannel *ch;
	GIOStatus status;
	GError *err = NULL;
	Kj *kj;

	kj = g_new0 (Kj, 1);
    kj->vertices = data_common_vertices_new();
    kj->text = data_common_text_new();

	ch = data_common_channel_open(name, "kj.sal");
	status = g_io_channel_read_line (ch, &line, NULL, NULL, &err);
	while (status != G_IO_STATUS_EOF) {
		Kjsub *sub;
		const gchar *p;
		Data25k top;

		if (status == G_IO_STATUS_ERROR) {
			g_print ("err:make_kj:%s:\n", name);
			exit (1);
		}
		line = g_strchomp (line);

		/* 無いデータもあるのでゼロの初期化必須 */
		sub = g_new0 (Kjsub, 1);

		gsi25k_get_data (line, &top);

		p = top.value;
		do {
			Data25k d25k;

			p = gsi25k_get_data (p, &d25k);
			if (d25k.id[0] == 'S' && d25k.id[1] == 'R') {
				sub->sr = gsi25k_hex_to_int (d25k.value);

			} else if (d25k.id[0] == 'N' && d25k.id[1] == 'M') {
				;				/* 数字か名称不明しかないので不使用 */

			} else if (d25k.id[0] == 'H' && d25k.id[1] == 'K') {
				sub->hk_char = g_strdup (d25k.value);

			} else if (d25k.id[0] == 'P' && d25k.id[1] == 'T') {
				gint n;
				gfloat *v;

				v = gsi25k_get_curve (d25k.value, slp, &n);

				sub->index = (guint16) (kj->vertices->n);

                data_common_vertices_add(kj->vertices, v, n);

				g_free (v);

			} else {
				DB (g_print ("line_to_data:err:%s:\n", p));
			}

			g_free (d25k.name);
			g_free (d25k.value);
		} while (*p != '\0');
		g_free (top.name);
		g_free (top.value);

		kj->sub = g_slist_prepend (kj->sub, sub);

		g_free (line);
		status = g_io_channel_read_line (ch, &line, NULL, NULL, &err);
	}
	g_free (line);
	g_io_channel_unref (ch);

	return kj;
}

/*****************************************/

void
ko_state_change (Ko * ko)
{
	GSList *sl;

    data_common_text_free(ko->text);
    ko->text = data_common_text_new();

	for (sl = ko->sub; sl != NULL; sl = sl->next) {
		Kosub *sub = sl->data;

		if (nmapdata_selector_check_onoff(sub->sr) == TRUE) {
            data_common_text_add(ko->text, sub->index, sub->pixtext);
		}
	}
}

void
ko_draw (Ko * ko)
{
	gfloat c0[] = { 1.0, 0.5, 1.0 };

	DB (g_print ("ko_draw:%d\n", ko->n));

    data_common_text_draw_points(ko->text, ko->vertices, c0);
    data_common_text_draw_text(ko->text, ko->vertices);
}

void
ko_add_text (Ko * ko)
{
	GSList *sl;

	for (sl = ko->sub; sl != NULL; sl = sl->next) {
		Kosub *sub = sl->data;

		sub->pixtext = text_create (sub->nm);
	}
}

void
ko_del_text (Ko * ko)
{
	GSList *sl;

	for (sl = ko->sub; sl != NULL; sl = sl->next) {
		Kosub *sub = sl->data;

        text_free(sub->pixtext);
		sub->pixtext = NULL;
	}
    data_common_text_free_pixtext(ko->text);
}

void
ko_free (Ko * ko)
{
	GSList *sl;

	for (sl = ko->sub; sl != NULL; sl = sl->next) {
		Kosub *sub = sl->data;

		g_free (sub->nm);
		g_free (sub->js);
		if (sub->pixtext != NULL) {
            text_free(sub->pixtext);
		}
		g_free (sub);
	}
	g_slist_free (ko->sub);
    data_common_vertices_free(ko->vertices);
    data_common_text_free(ko->text);
	g_free (ko);
}

Ko *
ko_create (const gchar * name, Slp * slp)
{
	gchar *line;
	GIOChannel *ch;
	GIOStatus status;
	GError *err = NULL;
	Ko *ko;

	ko = g_new0 (Ko, 1);
    ko->vertices = data_common_vertices_new();
    ko->text = data_common_text_new();

	ch = data_common_channel_open(name, "ko.sal");
	status = g_io_channel_read_line (ch, &line, NULL, NULL, &err);
	while (status != G_IO_STATUS_EOF) {
		Kosub *sub;
		const gchar *p;
		Data25k top;

		if (status == G_IO_STATUS_ERROR) {
			g_print ("err:ko_create:%s:\n", name);
			g_free (line);
			status = g_io_channel_read_line (ch, &line, NULL, NULL, &err);
			continue;
		}
		line = g_strchomp (line);

		/* 無いデータもあるのでゼロの初期化必須 */
		sub = g_new0 (Kosub, 1);

		gsi25k_get_data (line, &top);

		p = top.value;
		do {
			Data25k d25k;

			p = gsi25k_get_data (p, &d25k);
			if (d25k.id[0] == 'S' && d25k.id[1] == 'R') {
				sub->sr = gsi25k_hex_to_int (d25k.value);

			} else if (d25k.id[0] == 'N' && d25k.id[1] == 'M') {
				sub->nm = util_to_utf (d25k.value, SJIS);

			} else if (d25k.id[0] == 'P' && d25k.id[1] == 'T') {
				gint n;
				gfloat *v;

				v = gsi25k_get_curve (d25k.value, slp, &n);

				sub->index = (guint16) (ko->vertices->n);

                data_common_vertices_add(ko->vertices, v, n);

				g_free (v);

			} else {
				DB (g_print ("line_to_data:err:%s:\n", p));
			}

			g_free (d25k.name);
			g_free (d25k.value);
		} while (*p != '\0');
		g_free (top.name);
		g_free (top.value);

		ko->sub = g_slist_prepend (ko->sub, sub);


		g_free (line);
		status = g_io_channel_read_line (ch, &line, NULL, NULL, &err);
	}
	g_free (line);
	g_io_channel_unref (ch);

	return ko;
}

/*****************************************/

void
cm_state_change (Cm * cm)
{
	GSList *sl;

    data_common_text_free(cm->text);
    cm->text = data_common_text_new();

	for (sl = cm->sub; sl != NULL; sl = sl->next) {
		Cmsub *sub = sl->data;

		if (nmapdata_selector_check_onoff(sub->sr) == TRUE) {
            data_common_text_add(cm->text, sub->index, sub->pixtext);
		}
	}
}

void
cm_draw (Cm * cm)
{
	gfloat c0[] = { 1.0, 0.6, 0.0 };

    data_common_text_draw_points(cm->text, cm->vertices, c0);
    data_common_text_draw_text(cm->text, cm->vertices);
}

void
cm_add_text (Cm * cm)
{
	GSList *sl;

	for (sl = cm->sub; sl != NULL; sl = sl->next) {
		Cmsub *sub = sl->data;

		sub->pixtext = text_create (sub->nm);
	}
}

void
cm_del_text (Cm * cm)
{
	GSList *sl;

	for (sl = cm->sub; sl != NULL; sl = sl->next) {
		Cmsub *sub = sl->data;

		if (sub->pixtext != NULL) {
            text_free(sub->pixtext);
		}
		sub->pixtext = NULL;
	}
    data_common_text_free_pixtext(cm->text);
}

void
cm_free (Cm * cm)
{
	GSList *sl;

	for (sl = cm->sub; sl != NULL; sl = sl->next) {
		Cmsub *sub = sl->data;

		g_free (sub->nm);
		if (sub->pixtext != NULL) {
            text_free(sub->pixtext);
		}
		g_free (sub);
	}
	g_slist_free (cm->sub);
    data_common_vertices_free(cm->vertices);
    data_common_text_free(cm->text);
	g_free (cm);
}

Cm *
cm_create (const gchar * name, Slp * slp)
{
	gchar *line;
	GIOChannel *ch;
	GIOStatus status;
	GError *err = NULL;
	Cm *cm;

	cm = g_new0 (Cm, 1);
    cm->vertices = data_common_vertices_new();
    cm->text = data_common_text_new();

	ch = data_common_channel_open(name, "cm.sal");
	status = g_io_channel_read_line (ch, &line, NULL, NULL, &err);
	while (status != G_IO_STATUS_EOF) {
		Cmsub *sub;
		const gchar *p;
		Data25k top;

		if (status == G_IO_STATUS_ERROR) {
			g_print ("err:cm_create:%s:\n", name);

			g_free (line);
			status = g_io_channel_read_line (ch, &line, NULL, NULL, &err);
			continue;
		}
		line = g_strchomp (line);

		sub = g_new0 (Cmsub, 1); /* 無いデータもあるのでゼロの初期化必須 */

		gsi25k_get_data (line, &top);

		p = top.value;
		do {
			Data25k d25k;

			p = gsi25k_get_data (p, &d25k);
			if (gsi25k_compare_id(&d25k, "SR") == TRUE) {
				sub->sr = gsi25k_hex_to_int (d25k.value);

			} else if (gsi25k_compare_id(&d25k, "NM") == TRUE) {
				sub->nm = util_to_utf (d25k.value, SJIS);

			} else if (gsi25k_compare_id(&d25k, "PT") == TRUE) {
				gint n;
				gfloat *v;

				sub->index = (guint16) (cm->vertices->n);

				v = gsi25k_get_curve (d25k.value, slp, &n);
                data_common_vertices_add(cm->vertices, v, n);

				g_free (v);

			} else {
				DB (g_print ("line_to_data:err:%s:\n", p));
			}

			g_free (d25k.name);
			g_free (d25k.value);
		} while (*p != '\0');
		g_free (top.name);
		g_free (top.value);

		cm->sub = g_slist_prepend (cm->sub, sub);

		g_free (line);
		status = g_io_channel_read_line (ch, &line, NULL, NULL, &err);
	}
	g_free (line);
	g_io_channel_unref (ch);

	return cm;
}

/*****************************************/

void
mh_free (Mh * mh)
{
	g_hash_table_destroy (mh->hash);
    g_free (mh->name);
	g_free (mh->keyval);
	g_free (mh);
}

/* 経度緯度の下４桁だけを使い、経度を１００００倍した物と緯度を足してキーとする */
/* ちなみに３桁だけ使うようにしたら、約５０ｋｍ毎に同じキーを持つ値ができてしまったのでボツ。
 * 一つの市や町で５００ｋｍ以上あるところがあれば、大きな段差になって出てしまう */
gint
mh_create_hash_key_from_lonlat (gint lon, gint lat)
{
	return (lon % 10000) * 10000 + (lat % 10000);
}

/* 11111 の名前をもらって、そのデータをメモリー中に読み込む */
Mh *
mh_create (const gchar * name, gdouble x0, gdouble y0)
{
    Mh *mh;
	gchar *sal;
	gchar *slp;

	gint i;
	gchar *line_sal = NULL;
	GIOChannel *ch_sal;
	GIOStatus status_sal;
	GError *err = NULL;
	gint x0l, y0l;				/* x0 y0 の int 版。必ず偶数にする */
	gint xmax = 0;
	gint ymax = 0;

	gchar *slp_data;
	gint slp_size;
	gint num_line;
	gchar *p_slp;
	KeyVal *p_keyval;

	DB (g_print ("mh_create:%s\n", city->path));

	sal = g_strconcat (mmap_dir_gsi_vn, name, "mh.sal", NULL);
	slp = g_strconcat (mmap_dir_gsi_vn, name, "mh.slp", NULL);

	slp_data = disk_load_bin (slp, &slp_size);
	p_slp = slp_data;
	num_line = slp_size / 17;	/* １行１７文字決め打ち */

	mh = g_new (Mh, 1);
    mh->name = g_strdup(name);
	mh->keyval = g_new (KeyVal, num_line);
	mh->hash = g_hash_table_new (g_int_hash, g_int_equal);
	mh->ref_count = 1;

	p_keyval = mh->keyval;

	/* slm の基準点は、偶数も奇数も小数もある。経度が奇数で緯度が偶数とかもある(43343)。
	 * mh.slp のオフセットにも偶数と奇数と小数がある。これも経度が偶数で緯度が奇数がある。
	 * 基準点が奇数の時は、オフセットが偶数で
	 * 基準点が偶数の時は、オフセットが奇数となる。
	 * 小数の場合も同じで基準とオフセットの和が奇数になるようにしてある。
	 *
	 * 基準点が小数の場合、オフセットは 0.**** と１以下からはじまっている。
	 *
	 * 基準が奇数の時でも、オフセットの最小値にゼロはなく１。
	 *
	 * まとめると、
	 * 基準点はいろいろな値をとるが、標高値を持つのは奇数の点だけだということ。
	 *
	 * 基準点が整数の時には、オフセットも整数で、足すと必ず奇数になる。
	 * 基準点が小数の時には、オフセットも小数だが、それぞれ小数点以下を
	 * 切り捨ててしまっているので、足すと偶数になり、切捨て分の１を足すのが正解。
	 */
	x0l = (gint) x0;
	y0l = (gint) y0;
	if (x0 - x0l > 0) {
		++x0l;
	}
	if (y0 - y0l > 0) {
		++y0l;
	}
	DB (g_print ("x0:%.2f y0:%.2f    x0l:%d y0l:%d\n", x0, y0, x0l, y0l));

	ch_sal = disk_channel_open (sal, READ_SJIS);

	status_sal = g_io_channel_read_line (ch_sal, &line_sal, NULL, NULL, &err);
	for (i = 0; i < num_line; ++i, ++p_keyval, p_slp += 17) {
		gint x_offset = 0;
		gint y_offset = 0;
		gint x, y, z;

		if (status_sal == G_IO_STATUS_ERROR) {
			g_print ("error:sal read:%s:%s\n", line_sal, err->message);
			exit (1);
		}

		/* 小数点以下は切捨てる */
		x_offset += (p_slp[0] - '0') * 1000;
		x_offset += (p_slp[1] - '0') * 100;
		x_offset += (p_slp[2] - '0') * 10;
		x_offset += (p_slp[3] - '0');

		y_offset += (p_slp[8] - '0') * 1000;
		y_offset += (p_slp[9] - '0') * 100;
		y_offset += (p_slp[10] - '0') * 10;
		y_offset += (p_slp[11] - '0');

		/*
		   DB(g_print ("x0l:%d y0l:%d x_off:%d y_off:%d\n", x0l,y0l,x_offset,y_offset));
		 */
		x = x0l + x_offset;
		y = y0l + y_offset;

		sscanf (line_sal + 20, "%d", &z);

		if (xmax < x_offset)
			xmax = x_offset;
		if (ymax < y_offset)
			ymax = y_offset;

		p_keyval->key = mh_create_hash_key_from_lonlat (x, y);
		p_keyval->z = z;
		g_hash_table_insert (mh->hash, &(p_keyval->key), p_keyval);
		/*
		   g_print ("to hash:%d %d %d\n", x,y,z);
		 */
		g_free (line_sal);
		status_sal = g_io_channel_read_line (ch_sal, &line_sal, NULL, NULL, &err);
	}
	g_free (line_sal);


	mh->x0 = x0;
	mh->y0 = y0;
	mh->x1 = x0 + xmax;
	mh->y1 = y0 + ymax;

	g_io_channel_unref (ch_sal);

	g_free (slp_data);

	g_free (sal);
	g_free (slp);

    return mh;
}


