/*
 * MMap+ - 3d image viewer
 * Copyright 2005, 2006 Masahide Miyake
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/*
#define DB(x) (x)
*/
#define DB(x)

#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "mmap.h"
#include "nmapdata.h"
#include "nmapdata_all.h"
#include "disk.h"
#include "net.h"
#include "gsi.h"
#include "camera.h"
#include "util.h"
#include "glarea.h"
#include "window.h"
#include "ww_gsiloader.h"

/******************************************/

/* 中身は Mh * で、標高の検索で毎回全部の city をなめるのは無駄なのでデータのあるものだけまとめる */
static GSList *sl_mh = NULL;

/* lon,lat の単位は度 */
gdouble
city_get_alt (gdouble lon, gdouble lat)
{
	GSList *sl;
	gint key;
	gint x, y;
	gdouble alt = -1.0;

	/* lon, lat ともに整数で奇数のはずだがキャスト時に偶数になることがあるので 0.01 足してある */
	x = (gint) (lon * 3600.0 + 0.01);
	y = (gint) (lat * 3600.0 + 0.01);

	/*
	g_print ("city_get_alt:x:%d y:%d :::::: ", x, y);
	*/

    key = mh_create_hash_key_from_lonlat (x, y);

	for (sl = sl_mh; sl != NULL; sl = sl->next) {
		Mh *mh = sl->data;
		KeyVal *keyval;

		keyval = g_hash_table_lookup (mh->hash, &key);
		if (keyval == NULL) {
			;
		} else {
			alt = keyval->z;
		}
	}
	/*
	g_print ("alt:%.2f\n", alt);
	*/

	return alt;
}

void
city_create_mh (NCity *city)
{
    if(city->mh == NULL){
        Slm *slm;
        Mh *mh;

        /*
        g_print("city_create_mh:%s newly created\n", city->name);
        */

        slm = slm_create (city->name);
        mh = mh_create(city->name, slm->x, slm->y);
        slm_free(slm);

        sl_mh = g_slist_prepend(sl_mh, mh);
        city->mh = mh;
    }else{
        g_print("city_create_mh:%s already created\n", city->name);
    }
}

void
city_free_mh (NCity *city)
{
    city->mh_counter -= 1;

    /*
    g_print("city_free_mh:%d:%s\n", city->mh_counter, city->name);
    */

    if(city->mh_counter == 0){
        /*
        g_print("mh_free:%s\n", city->name);
        */

	    sl_mh = g_slist_remove(sl_mh, city->mh);

	    mh_free (city->mh);
        city->mh = NULL;
    }
}

void
city_create_other (NCity * city)
{
	Data *data;
	Slp *slp;

    /*
	g_print ("city_create_other:%s:\n", city->path);
    */

	data = g_new0 (Data, 1);
	data->slm = slm_create (city->name);
	slp = slp_create (city->name, data->slm->n_slp, data->slm->x, data->slm->y);
	data->gk = gk_create (city->name, slp);
	data->sk = sk_create (city->name, slp);
	data->dk = dk_create (city->name, slp);
	data->tk = tk_create (city->name, slp);
	data->kk = kk_create (city->name, slp);
	data->kj = kj_create (city->name, slp);
	data->ko = ko_create (city->name, slp);
	data->cm = cm_create (city->name, slp);

	slp_free (slp);

    gdk_threads_enter();
    {
    /*
    if(nmapdata_selector_check_onoff_text(TEXT_DK) == TRUE){
		dk_add_text (data->dk);
    }
    */
    if(nmapdata_selector_check_onoff_text(TEXT_TK) == TRUE){
		tk_add_text (data->tk);
    }
    if(nmapdata_selector_check_onoff_text(TEXT_KJ) == TRUE){
		kj_add_text (data->kj);
    }
    if(nmapdata_selector_check_onoff_text(TEXT_KO) == TRUE){
		ko_add_text (data->ko);
    }
    if(nmapdata_selector_check_onoff_text(TEXT_CM) == TRUE){
		cm_add_text (data->cm);
    }
    }
    gdk_threads_leave();

    nmapdata_selector_lock();
    {
	gk_state_change (data->gk);
	sk_state_change (data->sk);
	dk_state_change (data->dk);
	tk_state_change (data->tk);
	kk_state_change (data->kk);
	kj_state_change (data->kj);
	ko_state_change (data->ko);
	cm_state_change (data->cm);
    }
    nmapdata_selector_unlock();

    g_mutex_lock(city->mutex_data);
    {
        city->data = data;
    }
    g_mutex_unlock(city->mutex_data);
}

static void
city_free_other (Data * data)
{
	slm_free (data->slm);
	/*
	slp_free (data->slp);
	*/
	gk_free (data->gk);
	sk_free (data->sk);
	dk_free (data->dk);
	tk_free (data->tk);
	kk_free (data->kk);
	kj_free (data->kj);
	ko_free (data->ko);
	cm_free (data->cm);
	g_free (data);
}

/*********************************************/

static GSList *sl_city = NULL;

void
citylist_free(void)
{
    GSList *sl;

	for (sl = sl_city; sl != NULL; sl = sl->next) {
		NCity *city = (NCity *) sl->data;

        g_free(city->data);
        g_free(city->path);
        g_free(city->city);

        g_mutex_free(city->mutex_data);
        city_free_other(city->data);
	}
	g_slist_free (sl_city);

    sl_city = NULL;
}

void
citylist_init(void)
{
	gchar *fullpath;
	gchar *line = NULL;
	GIOChannel *ch = NULL;
	GError *err = NULL;
	GIOStatus status;

	fullpath = g_strconcat (mmap_dir_pkg, "/alt.dat", NULL);
	ch = disk_channel_open (fullpath, READ_UTF8);
	g_free (fullpath);

	status = g_io_channel_read_line (ch, &line, NULL, NULL, &err);
	while (status != G_IO_STATUS_EOF) {
		NCity *city;
		gchar path[50];
		gint64 x64, y64;
		gdouble x, y;
		gint nx, ny;
		gint nall, nalt;
		gchar pref[30];
		gchar cityname[150];
		gchar *tmp;

		if (status == G_IO_STATUS_ERROR) {
			g_print ("error:g_io_channel_read_line:%s:%s\n", line, err->message);
			exit (-1);
		}
		if (line[0] == '#') {	/* コメント行をとばす */
			continue;
		}

		/* 最後の市が空白で区切られていて最初しかとれないが、使ってないのでこのまま行く。 */
		/* 経度緯度は９桁の場合もある */
		sscanf (line, "%s %Ld %Ld %d %d %d %d %s %s", path, &x64, &y64, &nx, &ny, &nall, &nalt, pref, cityname);
		x = (gdouble) x64 / 10000, 0;
		y = (gdouble) y64 / 10000, 0;

		tmp = g_strrstr (path, "/");
		++tmp;

		city = g_new (NCity, 1);
		city->name = g_strdup (tmp);
		city->path = g_strdup (path);
		city->x0 = x;
		city->y0 = y;
		city->x1 = x + nx / 10000.0;
		city->y1 = y + ny / 10000.0;
		city->city = g_strdup (cityname);
        city->data = NULL;
        city->mh = NULL;
        city->mh_counter = 0;
        city->loader = NULL;
        city->mutex_data = g_mutex_new();

		DB (g_print ("%s\n", line));
		DB (g_print ("%s:%.0f:%.0f:%.0f:%.0f:%s\n", city->path, city->x0, city->y0, city->x1, city->y1, city->city));

		sl_city = g_slist_prepend (sl_city, city);

		g_free (line);
		status = g_io_channel_read_line (ch, &line, NULL, NULL, &err);
	}
	g_free (line);

	g_io_channel_unref (ch);
}

/* 引数は全部度 */
void
citylist_clear(void)
{
	GSList *sl;

	DB (g_print ("citylist_clear\n"));

	for (sl = sl_city; sl != NULL; sl = sl->next) {
		NCity *city = sl->data;

	    if (city->data != NULL) {
            Data *del_data;

            g_mutex_lock(city->mutex_data);
            {
                del_data = city->data;
                city->data = NULL;
            }
            g_mutex_unlock(city->mutex_data);

         city_free_other(del_data);
        }

        if(city->loader != NULL){
            g_object_unref(city->loader);
            city->loader = NULL;
        }
	}
}

/* 引数は全部度 */
void
citylist_update (gdouble x0, gdouble y0, gdouble x1, gdouble y1)
{
	GSList *sl;

	DB (g_print ("citylist_update:%.0f %.0f %.0f %.0f\n", x0, y0, x1, y1));

	for (sl = sl_city; sl != NULL; sl = sl->next) {
		NCity *city = sl->data;
		gdouble cx0, cy0, cx1, cy1;	/* 市の左下と右上の経度緯度 */

		cx0 = city->x0 / 3600.0;
		cy0 = city->y0 / 3600.0;
		cx1 = city->x1 / 3600.0;
		cy1 = city->y1 / 3600.0;

        /*
	    g_print ("citylist_update:camera:%.0f %.0f %.0f %.0f\n", x0, y0, x1, y1);
	    g_print ("citylist_update:city  :%.0f %.0f %.0f %.0f\n", cx0, cy0, cx1, cy1);
        */
        if (util_check_overlap (x0, y0, x1, y1, cx0, cy0, cx1, cy1) == TRUE){
		    if (city->data == NULL){
                if(city->loader == NULL) {
                    city->loader = ww_gsiloader_new();
                    ww_gsiloader_set_owner(city->loader, NULL);
                    ww_gsiloader_add_source(city->loader, city);
                    ww_gsiloader_start(city->loader);
                }
            }else{
                if(city->loader != NULL) {
                    g_object_unref(city->loader);
                    city->loader = NULL;
                }
            }
        }else{
		    if (city->data != NULL) {
                Data *del_data;

                g_mutex_lock(city->mutex_data);
                {
                    del_data = city->data;
                    city->data = NULL;
                }
                g_mutex_unlock(city->mutex_data);

	            city_free_other(del_data);
            }

            if(city->loader != NULL){
                g_object_unref(city->loader);
                city->loader = NULL;
            }
        }
	}
}

/* 引数は全部秒 */
GSList *
citylist_lookup(gdouble x0, gdouble y0, gdouble x1, gdouble y1)
{
	GSList *sl;
	GSList *rlist = NULL;

    x0 /= 3600.0;
    y0 /= 3600.0;
    x1 /= 3600.0;
    y1 /= 3600.0;

	g_print ("citylist_lookup:%.0f %.0f %.0f %.0f\n", x0, y0, x1, y1);

	for (sl = sl_city; sl != NULL; sl = sl->next) {
		NCity *city = sl->data;
		gdouble cx0, cy0, cx1, cy1;	/* 市の左下と右上の経度緯度 */

		cx0 = city->x0 / 3600.0;
		cy0 = city->y0 / 3600.0;
		cx1 = city->x1 / 3600.0;
		cy1 = city->y1 / 3600.0;

        if (util_check_overlap (x0, y0, x1, y1, cx0, cy0, cx1, cy1) == TRUE){
			rlist = g_slist_prepend (rlist, city);	/* 重なってる */
            city->mh_counter += 1;

			g_print ("citylist_lookup: ++ref :%d:%s\n", city->mh_counter, city->name);

			DB (g_print ("%.2f %.2f %.2f %.2f %s\n", cx0, cy0, cx1, cy1, city->city));
		} else {
			;					/* 重なってない */
		}
	}

	return rlist;
}

void
citylist_render (void)
{
	GSList *sl;
    /*
    g_print("city_list_render\n");
    */
	for (sl = sl_city; sl != NULL; sl = sl->next) {
		NCity *city = sl->data;
        Data *data = city->data;


        if(data == NULL){
            continue;
        }

        if(g_mutex_trylock(city->mutex_data) == TRUE){

            /*
            g_print("render :::: %s\n", city->city);
            */
		    gk_draw (data->gk);
		    sk_draw (data->sk);
		    dk_draw (data->dk);
		    tk_draw (data->tk);
		    kk_draw (data->kk);
		    kj_draw (data->kj);
		    ko_draw (data->ko);
		    cm_draw (data->cm);

            g_mutex_unlock(city->mutex_data);
        }else{
            g_print("trylock:fail\n");
            ;
        }
	}
}

/*********************************************/

static void
city_add_text (NCityTextType type)
{
	GSList *sl;

	for (sl = sl_city; sl != NULL; sl = sl->next) {
		NCity *city = sl->data;
		Data *data = city->data;

        if(data == NULL){
            continue;
        }

        g_mutex_lock(city->mutex_data);
        {
            /*
	        if (type == TEXT_DK) {
	            dk_add_text (data->dk);
	            dk_state_change (data->dk);
            */

    	    if (type == TEXT_TK) {
        		tk_add_text (data->tk);
			    tk_state_change (data->tk);
	        } else if (type == TEXT_KJ) {
			    kj_add_text (data->kj);
			    kj_state_change (data->kj);
	        } else if (type == TEXT_KO) {
			    ko_add_text (data->ko);
			    ko_state_change (data->ko);
	        } else if (type == TEXT_CM) {
			    cm_add_text (data->cm);
			    cm_state_change (data->cm);
	        }
        }
        g_mutex_unlock(city->mutex_data);
    }
}

static void
city_del_text (NCityTextType type)
{
	GSList *sl;

	for (sl = sl_city; sl != NULL; sl = sl->next) {
		NCity *city = sl->data;
		Data *data = city->data;

        if(data == NULL){
            continue;
        }

        g_mutex_lock(city->mutex_data);
        {
            /*
	        if (type == TEXT_DK) {
	            dk_del_text (city->dk);
            */
	        if (type == TEXT_TK) {
			    tk_del_text (data->tk);
	        } else if (type == TEXT_KJ) {
			    kj_del_text (data->kj);
	        } else if (type == TEXT_KO) {
			    ko_del_text (data->ko);
	        } else if (type == TEXT_CM) {
			    cm_del_text (data->cm);
	        }
        }
        g_mutex_unlock(city->mutex_data);
	}
}


void
city_change (NCityDataType type, gint num)
{
	GSList *sl;

    if(type == TEXT){
        gboolean onoff;

        onoff = nmapdata_selector_check_onoff(num);
        if(onoff == TRUE){
            city_add_text (num);
        }else{
            city_del_text (num);
        }

    }else{
	for (sl = sl_city; sl != NULL; sl = sl->next) {
		NCity *city = sl->data;
        Data *data = city->data;

        if(data == NULL){
            continue;
        }

        g_mutex_lock(city->mutex_data);
        {
	        if (type == GK) {
			    gk_state_change (data->gk);
	        } else if (type == SK) {
			    sk_state_change (data->sk);
	        } else if (type == DK) {
			    dk_state_change (data->dk);
	        } else if (type == TK) {
			    tk_state_change (data->tk);
	        } else if (type == KK) {
			    kk_state_change (data->kk);
	        } else if (type == KJ) {
			    kj_state_change (data->kj);
	        } else if (type == KO) {
			    ko_state_change (data->ko);
	        } else if (type == CM) {
			    cm_state_change (data->cm);
		    }
        }
        g_mutex_unlock(city->mutex_data);
    }
    }
}

/*********************************************/

/* オンオフ情報 */

typedef enum {
	GK_SR,
	GK_JT,

	SK_SR,

	DK_JT,
	DK_YU,
	DK_SB,
	DK_FI,

	TK_JT,
	TK_SB,

	KK_SB,

	KJ_SR,

	KO_SR,

	CM_SR,

    TEXT_INFO,

    CITY_INFO_N
}CityDataInfo;

typedef struct {
    gchar *data_name;
    gint n_info;
    CityDataInfo *data_info;
}CityInfo;

static CityDataInfo cdigk[] = { GK_SR, GK_JT };
static CityDataInfo cdisk[] = { SK_SR };
static CityDataInfo cdidk[] = { DK_JT, DK_YU, DK_SB, DK_FI };
static CityDataInfo cditk[] = { TK_JT, TK_SB };
static CityDataInfo cdikk[] = { KK_SB };
static CityDataInfo cdikj[] = { KJ_SR };
static CityDataInfo cdiko[] = { KO_SR };
static CityDataInfo cdicm[] = { CM_SR };
static CityDataInfo cditext[] = { TEXT_INFO };

static CityInfo city_info_real[CITY_DATA_N] = {
    {"行政界",   2, cdigk },                            /* gk */ 
    {"水域界",   1, cdisk },                            /* sk */ 
    {"道路",     4, cdidk },                            /* dk */
    {"鉄道",     2, cditk },                            /* tk */
    {"河川",     1, cdikk },                            /* kk */
    {"基準点",   1, cdikj },                            /* kj */
    {"公共施設", 1, cdiko },                            /* ko */
    {"地名",     1, cdicm },                            /* cm */

    {"文字",     1, cditext },                          /* text */
};

static CityInfo *city_info = city_info_real;

static gchar *info_name[CITY_INFO_N] = {
	"種類",     /* GK_SR */
	"状態",     /* GK_JT */

	"種類",     /* SK_SR */

	"状態",     /* DK_JT */
	"有料無料", /* DK_YU */
	"種別",     /* DK_SB */
	"幅員",     /* DK_FI */

	"状態",     /* TK_JT */
	"種別",     /* TK_SB */

	"種別",     /* KK_SB */

	"種類",     /* KJ_SR */

	"種類",     /* KO_SR */

	"種類",     /* CM_SR */

	"",         /* TEXT_INFO */
};

typedef struct {
	guint8 n;					/* 国土地理院が決めている番号 */
	gboolean onoff;				/* 起動直後の初期状態 */
	gchar *text;				/* その番号の説明文 */
    NCityDataType type;
    CityDataInfo info;
	GtkWidget *cb;				/* その番号をオンオフするチェックボタン */
    gulong handler_id;
} Selector;

const gint N_SELECTOR = 0xaf;
static Selector sel_ar[] = {
	{0x11, TRUE, "供用中", DK, DK_JT, NULL, 0},
	{0x12, TRUE, "建設中", DK, DK_JT, NULL, 0},
	{0x13, TRUE, "庭園路", DK, DK_SB, NULL, 0},
	{0x14, TRUE, "石段", DK, DK_SB, NULL, 0},
	{0x15, TRUE, "一般道", DK, DK_SB, NULL, 0},
	{0x16, TRUE, "高速道路", DK, DK_SB, NULL, 0},
	{0x17, TRUE, "1.5m未満", DK, DK_FI,  NULL, 0},
	{0x18, TRUE, "1.5m以上3.0m未満", DK, DK_FI, NULL, 0},
	{0x19, TRUE, "3.0m以上5.5m未満", DK, DK_FI,  NULL, 0},
	{0x1a, TRUE, "5.5m以上13.0未満", DK, DK_FI, NULL, 0},
	{0x1b, TRUE, "13.0m以上", DK, DK_FI, NULL, 0},
	{0x1c, TRUE, "有料", DK, DK_YU, NULL, 0},
	{0x1d, TRUE, "無料", DK, DK_YU, NULL, 0},
	{0x1e, TRUE, "真幅道路等", DK, DK_FI, NULL, 0},    /* 続きでない。注意 */

	{0x27, TRUE, "確定境界", GK, GK_JT, NULL, 0},
	{0x28, TRUE, "未定境界", GK, GK_JT, NULL, 0},
	{0x29, TRUE, "都道府県界", GK, GK_SR, NULL, 0},
	{0x2a, TRUE, "北海道の支庁界", GK, GK_SR, NULL, 0},
	{0x2b, TRUE, "都市東京都の区界", GK, GK_SR, NULL, 0},
	{0x2c, TRUE, "町村指定都市の区界", GK, GK_SR, NULL, 0},

	{0x41, TRUE, "運行中", TK, TK_JT, NULL, 0},
	{0x42, TRUE, "建設中", TK, TK_JT, NULL, 0},
	{0x43, TRUE, "普通電車(JR)", TK, TK_SB, NULL, 0},
	{0x44, TRUE, "普通電車", TK, TK_SB, NULL, 0},
	{0x45, TRUE, "路面電車", TK, TK_SB, NULL, 0},
	{0x46, TRUE, "地下式鉄道", TK, TK_SB, NULL, 0},
	{0x47, TRUE, "その他", TK, TK_SB, NULL, 0},

	{0x51, TRUE, "水涯線または湖岸線", SK, SK_SR, NULL, 0},
	{0x52, TRUE, "海岸線", SK, SK_SR, NULL, 0},
	{0x53, FALSE, "河口", SK, SK_SR, NULL, 0},
	{0x54, FALSE, "湖沼と河川の境界", SK, SK_SR, NULL, 0},
	{0x55, TRUE, "一条河川", KK, KK_SB, NULL, 0},
	{0x56, TRUE, "二条河川", KK, KK_SB, NULL, 0},
	{0x57, TRUE, "一条かれ川", KK, KK_SB, NULL, 0},
	{0x58, TRUE, "二条かれ川", KK, KK_SB, NULL, 0},
	{0x59, FALSE, "湖沼域内中心線", KK, KK_SB, NULL, 0},

	{0x61, TRUE, "水準点", KJ, KJ_SR, NULL, 0},
	{0x62, TRUE, "三角点", KJ, KJ_SR, NULL, 0},
	{0x63, TRUE, "電子基準点", KJ, KJ_SR, NULL, 0},

	{0x81, TRUE, "国の機関",        KO, KO_SR, NULL, 0},
	{0x82, TRUE, "地方公共団体",    KO, KO_SR, NULL, 0},
	{0x83, TRUE, "厚生機関",        KO, KO_SR, NULL, 0},
	{0x84, TRUE, "警察機関",        KO, KO_SR, NULL, 0},
	{0x85, TRUE, "消防署",          KO, KO_SR, NULL, 0},
	{0x86, TRUE, "学校",            KO, KO_SR, NULL, 0},
	{0x87, TRUE, "病院",            KO, KO_SR, NULL, 0},
	{0x88, TRUE, "郵便局",          KO, KO_SR, NULL, 0},

	{0x91, TRUE, "自然地名",        CM, CM_SR, NULL, 0},
	{0x92, TRUE, "土地の利用景",    CM, CM_SR, NULL, 0},
	{0x93, TRUE, "居住地名",        CM, CM_SR, NULL, 0},

	{TEXT_TK, FALSE, "駅",          TEXT, TEXT_INFO, NULL, 0},
	/*
	{TEXT_DK, FALSE,"国道番号",     TEXT, TEXT_INFO, NULL, 0},
	*/
	{TEXT_KJ, FALSE, "基準点",      TEXT, TEXT_INFO, NULL, 0},
	{TEXT_KO, FALSE, "公共施設",    TEXT, TEXT_INFO, NULL, 0},
	{TEXT_CM, FALSE, "地名",        TEXT, TEXT_INFO, NULL, 0},
};

static Selector **selector = NULL;

/************************************************/

static void
data_init(void)
{
    gint i;
    gint n;

    selector = g_malloc0(sizeof(Selector *) * N_SELECTOR);

    n = sizeof sel_ar / sizeof(Selector);
    for(i = 0; i < n; ++i){
        Selector *s = &(sel_ar[i]);
        gint num = s->n;

        selector[num] = s;
    }
}

/************************************************/

static void
check_onoff (CityDataInfo info, gboolean onoff)
{
    gint i;

	for (i = 0; i < N_SELECTOR; ++i) {
        Selector *sel = selector[i];

        if(sel != NULL && sel->info == info){
            g_signal_handler_block(sel->cb, sel->handler_id);

            gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (sel->cb), onoff);
            sel->onoff = onoff;

            g_signal_handler_unblock(sel->cb, sel->handler_id);
        }
    }
}

static gboolean
up_check (gdouble old, gdouble new, gdouble val)
{
	return (old < val && new >= val);
}

static gboolean
down_check (gdouble old, gdouble new, gdouble val)
{
	return (old >= val && new < val);
}

void
nmapdata_selector_check_alt_up (gdouble old, gdouble new)
{
	if (up_check (old, new, 10000.0) == TRUE) {
        nmapdata_selector_lock();
        {
		    check_onoff (TEXT_INFO,FALSE);
        }
        nmapdata_selector_unlock();

        city_change(TEXT, TEXT_TK);
        /*
        city_change(TEXT, TEXT_DK);
        */
        city_change(TEXT, TEXT_KJ);
        city_change(TEXT, TEXT_KO);
        city_change(TEXT, TEXT_CM);
	} else if (up_check (old, new, 25000.0) == TRUE) {
        nmapdata_selector_lock();
        {
		    check_onoff (KJ_SR, FALSE);
		    check_onoff (KO_SR, FALSE);
		    check_onoff (CM_SR, FALSE);
        }
        nmapdata_selector_unlock();

	    city_change (KJ, KJ_SR);
	    city_change (KO, KO_SR);
	    city_change (CM, CM_SR);

	    glarea_force_render();
	}
}

void
nmapdata_selector_check_alt_down (gdouble old, gdouble new)
{
	if (down_check (old, new, 25000.0) == TRUE) {
        nmapdata_selector_lock();
        {
		    check_onoff (KJ_SR, TRUE);
		    check_onoff (KO_SR, TRUE);
		    check_onoff (CM_SR, TRUE);
        }
        nmapdata_selector_unlock();

	    city_change (KJ, KJ_SR);
	    city_change (KO, KO_SR);
	    city_change (CM, CM_SR);

	    glarea_force_render();
	}
}

/************************************************/

static GtkWidget *nmap_dialog = NULL;
static GMutex *mutex_onoff;

void
nmapdata_selector_lock (void)
{
	g_mutex_lock (mutex_onoff);
}

void
nmapdata_selector_unlock (void)
{
	g_mutex_unlock (mutex_onoff);
}

gboolean 
nmapdata_selector_check_onoff(gint num)
{
    return selector[num]->onoff;
}

gboolean 
nmapdata_selector_check_onoff_text(gint num)
{
    return selector[num]->onoff;
}

/**********************************************/

static void
check_button_toggled_cb (GtkToggleButton * tb, gpointer user_data)
{
    Selector *sel = (Selector *)user_data;

    nmapdata_selector_lock();
    {
	    sel->onoff = gtk_toggle_button_get_active (tb);
    }
    nmapdata_selector_unlock();

	city_change (sel->type, sel->n);

    glarea_force_render();
}

static void
check_button_init (void)
{
    gint i;

	for (i = 0; i < N_SELECTOR; ++i) {
        Selector *sel = selector[i];

        if(sel != NULL){
		    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (sel->cb), sel->onoff);
		    sel->handler_id = g_signal_connect (GTK_TOGGLE_BUTTON(sel->cb), "toggled", 
                    G_CALLBACK(check_button_toggled_cb), sel);
        }
    }
}

static GtkWidget *
city_info_frame_create(CityDataInfo info)
{
    gint i;
	GtkWidget *vbox;
    GtkWidget *frame;

	vbox = gtk_vbox_new (FALSE, 0);

	for (i = 0; i < N_SELECTOR; ++i) {
        Selector *sel = selector[i];

        if(sel != NULL && sel->info == info){
            sel->cb = gtk_check_button_new_with_label (sel->text);
            /* シグナルは、realize 後にする初期化のタイミングで接続する */

		    gtk_box_pack_start (GTK_BOX (vbox), sel->cb, FALSE, FALSE, 0);
        }
	}

	frame = gtk_frame_new (info_name[info]);
	gtk_container_add (GTK_CONTAINER (frame), vbox);

    return frame;
}

static GtkWidget *
city_info_box_create(NCityDataType type)
{
    gint i;
	GtkWidget *label;
	GtkWidget *vbox;
    CityInfo *ci = &(city_info[type]);

	vbox = gtk_vbox_new (FALSE, 0);

	label = gtk_label_new (ci->data_name);
	gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, FALSE, 0);

    for(i = 0; i< ci->n_info; ++i){
	    GtkWidget *frame;
        CityDataInfo cdi = ci->data_info[i];

	    frame = city_info_frame_create(cdi);
	    gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);
    }

    return vbox;
}

/**********************************************/
static void
notebook_realize_cb (GtkWidget * widget, gpointer user_data)
{
    g_print("notebook_realize_cb\n");

	check_button_init ();
}
static void
nmap_dialog_response_cb (GtkDialog * dialog, gint arg1, gpointer user_data)
{
    g_print("nmap_dialog_close_cb\n");

	gtk_widget_hide_all (nmap_dialog);
    window_menu_selector_sensitive (TRUE);

    nmap_dialog = NULL;
}

void
nmapdata_selector_init (void)
{
	mutex_onoff = g_mutex_new ();
    data_init();
}

void
nmapdata_selector_show (void)
{
	GtkWidget *tab;
	GtkWidget *detail;
	GtkWidget *vbox;
	GtkWidget *label;

    window_menu_selector_sensitive (FALSE);

	tab = gtk_notebook_new ();
	g_signal_connect_after (tab, "realize", G_CALLBACK (notebook_realize_cb), NULL);

	vbox = gtk_vbox_new (FALSE, 0);
	{
        detail = city_info_box_create(GK);
		gtk_box_pack_start (GTK_BOX (vbox), detail, FALSE, FALSE, 0);
        detail = city_info_box_create(SK);
		gtk_box_pack_start (GTK_BOX (vbox), detail, FALSE, FALSE, 0);
	}
	label = gtk_label_new ("境界");
	gtk_notebook_append_page (GTK_NOTEBOOK (tab), vbox, label);

    detail = city_info_box_create(DK);
	label = gtk_label_new ("道路");
	gtk_notebook_append_page (GTK_NOTEBOOK (tab), detail, label);

    detail = city_info_box_create(TK);
	label = gtk_label_new ("鉄道");
	gtk_notebook_append_page (GTK_NOTEBOOK (tab), detail, label);


    detail = city_info_box_create(KK);
	label = gtk_label_new ("河川");
	gtk_notebook_append_page (GTK_NOTEBOOK (tab), detail, label);

	vbox = gtk_vbox_new (FALSE, 0);
	{
        detail = city_info_box_create(KJ);
		gtk_box_pack_start (GTK_BOX (vbox), detail, FALSE, FALSE, 0);
        detail = city_info_box_create(KO);
		gtk_box_pack_start (GTK_BOX (vbox), detail, FALSE, FALSE, 0);
        detail = city_info_box_create(CM);
		gtk_box_pack_start (GTK_BOX (vbox), detail, FALSE, FALSE, 0);
        detail = city_info_box_create(TEXT);
		gtk_box_pack_start (GTK_BOX (vbox), detail, FALSE, FALSE, 0);
	}
	label = gtk_label_new (" 他 ");
	gtk_notebook_append_page (GTK_NOTEBOOK (tab), vbox, label);

	nmap_dialog = gtk_dialog_new_with_buttons (_("表示選択(GSI)"), NULL,
										  GTK_DIALOG_DESTROY_WITH_PARENT, GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE, NULL);
	g_signal_connect (G_OBJECT (nmap_dialog), "response", G_CALLBACK (nmap_dialog_response_cb), NULL);
	gtk_container_add (GTK_CONTAINER (GTK_DIALOG (nmap_dialog)->vbox), tab);
	gtk_window_set_default_size (GTK_WINDOW (nmap_dialog), 220, 600);

	gtk_widget_show_all (nmap_dialog);
}

