/*
 * MMap+ - 3d image viewer
 * Copyright 2005, 2006 Masahide Miyake
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <sys/stat.h>

#include <gtk/gtk.h>

#include "mmap.h"
#include "util.h"
#include "xmlutil.h"
#include "camera.h"
#include "ww_world.h"
#include "ww_terrainaccessor.h"

gchar *mmap_dir_pkg = NULL;		/* /usr/local/share/mmap */
gchar *mmap_dir_pkg_config = NULL;	/* /usr/local/share/mmap/Config */
gchar *mmap_dir_pkg_data = NULL;	/* /usr/local/share/mmap/Data */

gchar *mmap_dir = NULL;			/* ~/.mmap/ */
gchar *mmap_dir_cache = NULL;	/* ~/.mmap/Cache/ */
gchar *mmap_dir_data = NULL;	/* ~/.mmap/Data/ */
gchar *mmap_dir_gsi = NULL;		/* ~/.mmap/gsi */
gchar *mmap_dir_gsi_tex = NULL;	/* ~/.mmap/gsi/tex */
gchar *mmap_dir_gsi_vn = NULL;	/* ~/.mmap/gsi/vn */


static void
check_dir (const gchar * dir)
{
	if (!g_file_test (dir, G_FILE_TEST_IS_DIR | G_FILE_TEST_EXISTS)) {
		mkdir (dir, S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH);
	}
}

void
mmap_directory_set (void)
{
	mmap_dir_pkg = MMAP_DATA_DIR;
	mmap_dir_pkg_config = g_strconcat (mmap_dir_pkg, "/Config", NULL);
	mmap_dir_pkg_data = g_strconcat (mmap_dir_pkg, "/Data", NULL);

	mmap_dir = g_strconcat (g_get_home_dir (), "/.mmap/", NULL);
	check_dir (mmap_dir);

	mmap_dir_cache = g_strconcat (mmap_dir, "Cache/", NULL);
	util_mkdir_p (mmap_dir_cache);

	mmap_dir_data = g_strconcat (mmap_dir, "Data/", NULL);
	util_mkdir_p (mmap_dir_data);

	mmap_dir_gsi = g_strconcat (mmap_dir, "gsi/", NULL);
	util_mkdir_p (mmap_dir_gsi);

	mmap_dir_gsi_tex = g_strconcat (mmap_dir_gsi, "tex/", NULL);
	util_mkdir_p (mmap_dir_gsi_tex);

	mmap_dir_gsi_vn = g_strconcat (mmap_dir_gsi, "vn/", NULL);
	util_mkdir_p (mmap_dir_gsi_vn);
}

/*****************************************************/

static gdouble vertical_exaggeration = 1.0;
static MmapViewMode view_mode = 3.0;

gdouble
mmap_get_vertical_exaggeration (void)
{
	return vertical_exaggeration;
}

void
mmap_set_vertical_exaggeration (gdouble n)
{
	vertical_exaggeration = n;
}

MmapViewMode
mmap_get_view_mode (void)
{
	return view_mode;
}

void
mmap_set_view_mode (MmapViewMode mode)
{
	view_mode = mode;
}

/*****************************************************/

static WwWorld *current_world = NULL;

void
mmap_set_world (const gchar * path_xml)
{

	if (current_world != NULL) {
		g_object_unref (current_world);
	}

	current_world = xmlutil_data_build (path_xml);

	camera_set_dist_direct (ww_world_get_equatorial_radius (current_world) * 3);

	ww_world_create_layer (current_world);
}

/*****************************************************/

gdouble
mmap_get_world_radius (void)
{
	return ww_world_get_equatorial_radius (current_world);
}

void
mmap_deg_to_xyz (gdouble lon, gdouble lat, gdouble alt, gdouble * x, gdouble * y, gdouble * z)
{
	ww_world_deg_to_xyz (current_world, lon, lat, alt, x, y, z);
}

const gchar *
mmap_get_world_name (void)
{
	return ww_world_get_name (current_world);
}

WwTerrainaccessor *
mmap_get_terrainaccessor (gdouble x0, gdouble y0, gdouble x1, gdouble y1)
{
	return ww_world_get_terrainaccessor (current_world, x0, y0, x1, y1);
}

void
mmap_render (void)
{
	ww_world_render (current_world);
}

void
mmap_update (void)
{
	ww_world_update (current_world);
}

/****************************************/

typedef struct {
	MmapIdleFunc func;
	GObject *obj;
} MmapIdleData;

static gboolean
mmap_idle_func_single (gpointer data)
{
	MmapIdleData *idle_data = data;

	idle_data->func (idle_data->obj);

	g_object_unref (idle_data->obj);
	g_free (idle_data);

	return FALSE;				/* FALSE を返して一回で終了 */
}

void
mmap_idle_add_single (MmapIdleFunc func, GObject * obj)
{
	MmapIdleData *data;

	g_object_ref (obj);

	data = g_new (MmapIdleData, 1);
	data->func = func;
	data->obj = obj;

	g_idle_add (mmap_idle_func_single, data);
}
