/*
 * MMap+ - 3d image viewer
 * Copyright 2005, 2006 Masahide Miyake
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
/*
#define DB(x) (x)
*/
#define DB(x)

#include <gtk/gtk.h>
#include <string.h>
#include <stdio.h>
#include <glib/gi18n.h>

#include "mark.h"
#include "camera.h"
#include "util.h"
#include "glarea.h"
#include "window.h"

enum {
	MARK_TEXT,
	MARK_NO,					/* マーク内の順番(ゼロから始まる) */
	MARK_N
};

enum {
	POINT_TEXT,
	POINT_LON,
	POINT_LAT,
	POINT_ALT,
	POINT_NO,
	POINT_N
};

static GtkWidget *list_mark = NULL;
static GtkWidget *list_point = NULL;

static GtkWidget *entry_name;

static GtkWidget *spin_lon_deg;
static GtkWidget *spin_lon_min;
static GtkWidget *spin_lon_sec;
static GtkWidget *spin_lat_deg;
static GtkWidget *spin_lat_min;
static GtkWidget *spin_lat_sec;
static GtkWidget *spin_alt;
static GtkWidget *entry_text;

static GtkWidget *mbutton_up;
static GtkWidget *mbutton_down;
static GtkWidget *mbutton_del;
static GtkWidget *mbutton_add;

static GtkWidget *pbutton_up;
static GtkWidget *pbutton_down;
static GtkWidget *pbutton_del;
static GtkWidget *pbutton_add;

static GtkWidget *button_change;

static gint mark_current = -1;
static gint point_current = -1;

/********************************************************************/

/* 編集領域のマーク名の sensitive */
static void
mark_widget_sensitive (gboolean bool)
{
	gtk_widget_set_sensitive (entry_name, bool);
}

/* 編集領域のマーク名以外の sensitive */
static void
point_widget_sensitive (gboolean bool)
{
	gtk_widget_set_sensitive (spin_lon_deg, bool);
	gtk_widget_set_sensitive (spin_lon_min, bool);
	gtk_widget_set_sensitive (spin_lon_sec, bool);
	gtk_widget_set_sensitive (spin_lat_deg, bool);
	gtk_widget_set_sensitive (spin_lat_min, bool);
	gtk_widget_set_sensitive (spin_lat_sec, bool);
	gtk_widget_set_sensitive (spin_alt, bool);
	gtk_widget_set_sensitive (entry_text, bool);
}

/********************************************************************/

/* カーソルをあわせる */
/* cursor-changed シグナルが出る */

void
cursor_set_mark_current (void)
{
	if (mark_current >= 0) {
		GtkTreePath *path;

		path = gtk_tree_path_new_from_indices (mark_current, -1);
		gtk_tree_view_set_cursor (GTK_TREE_VIEW (list_mark), path, NULL, FALSE);
		gtk_tree_path_free (path);
	}
}

void
cursor_set_point_current (void)
{
	if (point_current >= 0) {
		GtkTreePath *path;

		path = gtk_tree_path_new_from_indices (point_current, -1);
		gtk_tree_view_set_cursor (GTK_TREE_VIEW (list_point), path, NULL, FALSE);
		gtk_tree_path_free (path);
	}
}

/********************************************************************/
/* 何かのアクションがある毎に実行して、編集領域を全部書き直す。
 * sensitive の変更もここでする。
 */
static void
make_edit_area (void)
{
	gint lon_deg = 0;
	gint lon_min = 0;
	gdouble lon_sec = 0.0;
	gint lat_deg = 0;
	gint lat_min = 0;
	gdouble lat_sec = 0.0;
	gdouble alt = 0.0;
	gchar *name = "";
	gchar *text = "";

	DB (g_print ("make_edit_area\n"));

	if (mark_current < 0) {
		/* 上の初期値のまま */
		mark_widget_sensitive (FALSE);
		point_widget_sensitive (FALSE);
	} else {
		Mark *mark = mark_get_mark (mark_current);

		name = mark->text;
		mark_widget_sensitive (TRUE);

		if (point_current < 0) {
			/* 上の初期値のまま */
			point_widget_sensitive (FALSE);
		} else {
			MarkPoint *point = mark_get_point (mark_current, point_current);

			util_sec_to_deg_min_sec (point->lon, &lon_deg, &lon_min, &lon_sec);
			util_sec_to_deg_min_sec (point->lat, &lat_deg, &lat_min, &lat_sec);

			alt = point->alt;
			text = point->text;

			point_widget_sensitive (TRUE);
		}
	}

	gtk_entry_set_text (GTK_ENTRY (entry_name), name);

	gtk_spin_button_set_value (GTK_SPIN_BUTTON (spin_lon_deg), lon_deg);
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (spin_lon_min), lon_min);
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (spin_lon_sec), lon_sec);

	gtk_spin_button_set_value (GTK_SPIN_BUTTON (spin_lat_deg), lat_deg);
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (spin_lat_min), lat_min);
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (spin_lat_sec), lat_sec);

	gtk_spin_button_set_value (GTK_SPIN_BUTTON (spin_alt), alt);

	gtk_entry_set_text (GTK_ENTRY (entry_text), text);
}

/* mark_current が指しているポイントのリストを作成する。
 * mark_current が何も指してない、またはポイントリストがない、クリアだけする。 */
static void
make_pointlist_data (void)
{
	GtkTreeView *treeview = GTK_TREE_VIEW (list_point);
	GtkTreeModel *model;
	GtkTreeIter iter;
	gint n_point;
	gint i;

	DB (g_print ("make_pointlist_data\n"));

	model = gtk_tree_view_get_model (treeview);
	gtk_list_store_clear (GTK_LIST_STORE (model));

	if (mark_current < 0) {
		return;
	}

	n_point = mark_get_point_n (mark_current);

	for (i = 0; i < n_point; ++i) {
		MarkPoint *mp;
		gchar *lon;
		gchar *lat;
		gchar *alt;
		gint deg, min;
		gdouble sec;

		mp = mark_get_point (mark_current, i);

		util_sec_to_deg_min_sec (mp->lon, &deg, &min, &sec);
		if (mp->lon < 0.0) {
			lon = g_strdup_printf (" W %3d°%2d’%3.1f ", -deg, -min, -sec);
		} else {
			lon = g_strdup_printf (" E %3d°%2d’%3.1f ", deg, min, sec);
		}

		util_sec_to_deg_min_sec (mp->lat, &deg, &min, &sec);
		if (mp->lat < 0.0) {
			lat = g_strdup_printf (" S %3d°%2d’%3.1f ", -deg, -min, -sec);
		} else {
			lat = g_strdup_printf (" N %3d°%2d’%3.1f ", deg, min, sec);
		}

		alt = g_strdup_printf (" %.1f ", mp->alt);

		gtk_list_store_append (GTK_LIST_STORE (model), &iter);
		gtk_list_store_set (GTK_LIST_STORE (model), &iter, POINT_TEXT, mp->text, -1);
		gtk_list_store_set (GTK_LIST_STORE (model), &iter, POINT_LON, lon, -1);
		gtk_list_store_set (GTK_LIST_STORE (model), &iter, POINT_LAT, lat, -1);
		gtk_list_store_set (GTK_LIST_STORE (model), &iter, POINT_ALT, alt, -1);
		gtk_list_store_set (GTK_LIST_STORE (model), &iter, POINT_NO, i, -1);

		g_free (lon);
		g_free (lat);
		g_free (alt);
	}
}

/* マークのリストを作成する。
 * ないならクリアだけする。 */
static void
make_marklist_data (void)
{
	GtkTreeView *treeview = GTK_TREE_VIEW (list_mark);
	GtkTreeModel *model;
	GtkTreeIter iter;
	gint n_mark;
	gint i;

	DB (g_print ("make_marklist_data\n"));

	model = gtk_tree_view_get_model (treeview);
	gtk_list_store_clear (GTK_LIST_STORE (model));

	n_mark = mark_get_mark_n ();

	for (i = 0; i < n_mark; ++i) {
		Mark *mk;

		mk = mark_get_mark (i);

		gtk_list_store_append (GTK_LIST_STORE (model), &iter);
		gtk_list_store_set (GTK_LIST_STORE (model), &iter, MARK_TEXT, mk->text, -1);
		gtk_list_store_set (GTK_LIST_STORE (model), &iter, MARK_NO, i, -1);
	}
}

static void
point_draw_and_set_cursor (void)
{
	DB (g_print ("remake_point:p:%d\n", point_current));

	/* リストを再描画 */
	make_pointlist_data ();

	/* カーソルをあわせる  cursor-changed シグナルが出る */
	cursor_set_point_current ();

	make_edit_area ();
}

static void
mark_draw_and_set_cursor (void)
{
	DB (g_print ("remake_mark:m:%d\n", mark_current));

	/* リストを再描画 */
	make_marklist_data ();

	/* カーソルをあわせる  cursor-changed シグナルが出る */
	cursor_set_mark_current ();

	point_draw_and_set_cursor ();
}

static void
point_list_cursor_changed_cb (GtkTreeView * treeview, gpointer user_data)
{
	guint no;
	GtkTreeIter iter;
	GtkTreeModel *model;
	GtkTreeSelection *selection;
	GList *glist;

	DB (g_print ("point_list_cursor_changed_cb::m:%d p:%d\n", mark_current, point_current));

	selection = gtk_tree_view_get_selection (treeview);
	glist = gtk_tree_selection_get_selected_rows (selection, &model);
	if (glist == NULL) {
		point_current = -1;

		make_edit_area ();

		return;
	}

	gtk_tree_model_get_iter (model, &iter, glist->data);
	gtk_tree_model_get (model, &iter, POINT_NO, &no, -1);
	DB (g_print ("point no:%d\n", no));

	if (no == point_current) {
		;
	} else {
		point_current = no;

		make_edit_area ();
	}
}

static void
point_list_row_activated_cb (GtkTreeView * treeview, GtkTreePath * path, GtkTreeViewColumn * column, gpointer user_data)
{
	guint no;
	GtkTreeIter iter;
	GtkTreeModel *model;
	MarkPoint *mp;

	DB (g_print ("mark_list_row_activated_cb::m:%d p:%d\n", mark_current, point_current));

	model = gtk_tree_view_get_model (treeview);
	gtk_tree_model_get_iter (model, &iter, path);
	gtk_tree_model_get (model, &iter, POINT_NO, &no, -1);

	mp = mark_get_point (mark_current, point_current);

	if (mp != NULL) {
		camera_set_position (mp->lon / 3600.0, mp->lat / 3600.0);
		glarea_force_update_and_render ();
	}
}

static void
mark_list_cursor_changed_cb (GtkTreeView * treeview, gpointer user_data)
{
	guint no;
	GtkTreeIter iter;
	GtkTreeModel *model;
	GtkTreeSelection *selection;
	GList *glist;

	DB (g_print ("mark_list_cursor_changed_cb::m:%d p:%d\n", mark_current, point_current));

	selection = gtk_tree_view_get_selection (treeview);
	glist = gtk_tree_selection_get_selected_rows (selection, &model);
	if (glist == NULL) {
		mark_current = -1;
		point_current = -1;

		make_pointlist_data ();

		make_edit_area ();

		return;
	}

	gtk_tree_model_get_iter (model, &iter, glist->data);
	gtk_tree_model_get (model, &iter, MARK_NO, &no, -1);
	DB (g_print ("mark no:%d\n", no));

	if (mark_current == no) {
		;
	} else {
		mark_current = no;
		point_current = -1;

		make_pointlist_data ();

		make_edit_area ();
	}
}

static void
mark_list_row_activated_cb (GtkTreeView * treeview, GtkTreePath * path, GtkTreeViewColumn * column, gpointer user_data)
{
	guint no;
	GtkTreeIter iter;
	GtkTreeModel *model;
	MarkPoint *point;

	DB (g_print ("mark_list_row_activated_cb::m:%d p:%d\n", mark_current, point_current));

	model = gtk_tree_view_get_model (treeview);
	gtk_tree_model_get_iter (model, &iter, path);
	gtk_tree_model_get (model, &iter, MARK_NO, &no, -1);

	point = mark_get_point (no, 0);
	if (point != NULL) {
		camera_set_position (point->lon / 3600.0, point->lat / 3600.0);
		glarea_force_update_and_render ();
	}
}

/* 変更をマークとポイントの両方のリストに反映させる */
static void
change_button_clicked_cb (GtkButton * button, gpointer user_data)
{
	DB (g_print ("change_button_clicked_cb\n"));

	if (mark_current < 0) {
		return;
	} else {
		const gchar *name;

		name = gtk_entry_get_text (GTK_ENTRY (entry_name));
		if (*name == '\0') {
			gtk_entry_set_text (GTK_ENTRY (entry_name), "?????");
			name = gtk_entry_get_text (GTK_ENTRY (entry_name));
		}

		mark_set_mark (mark_current, name);
	}

	if (point_current < 0) {
		;
	} else {
		const gchar *text;
		gint lon_deg, lon_min;
		gdouble lon_sec;
		gint lat_deg, lat_min;
		gdouble lat_sec;
		gdouble lon, lat, alt;

		lon_deg = (gint) gtk_spin_button_get_value (GTK_SPIN_BUTTON (spin_lon_deg));
		lon_min = (gint) gtk_spin_button_get_value (GTK_SPIN_BUTTON (spin_lon_min));
		lon_sec = gtk_spin_button_get_value (GTK_SPIN_BUTTON (spin_lon_sec));
		lon = lon_deg * 3600.0 + lon_min * 60.0 + lon_sec;

		lat_deg = (gint) gtk_spin_button_get_value (GTK_SPIN_BUTTON (spin_lat_deg));
		lat_min = (gint) gtk_spin_button_get_value (GTK_SPIN_BUTTON (spin_lat_min));
		lat_sec = gtk_spin_button_get_value (GTK_SPIN_BUTTON (spin_lat_sec));
		lat = lat_deg * 3600.0 + lat_min * 60.0 + lat_sec;

		alt = gtk_spin_button_get_value (GTK_SPIN_BUTTON (spin_alt));

		text = gtk_entry_get_text (GTK_ENTRY (entry_text));

		mark_set_point (mark_current, point_current, lon, lat, alt, text);
	}

	/* 全部再描画 */
	mark_draw_and_set_cursor ();

	glarea_force_render ();
}

/*********************************************************************/

/******** ポイントの編集 ********/
static void
up_point_button_clicked_cb (GtkButton * button, gpointer user_data)
{
	DB (g_print ("up_point_button_clicked_cb::m:%d p:%d\n", mark_current, point_current));

	if (point_current < 0) {
		return;
	}

	if (point_current == 0) {
		return;					/* 既に一番上 */
	}

	/* データの入れ換え */
	mark_swap_point (mark_current, point_current - 1, point_current);

	point_current -= 1;

	point_draw_and_set_cursor ();
}

static void
down_point_button_clicked_cb (GtkButton * button, gpointer user_data)
{
	guint n_point;

	DB (g_print ("down_point_button_clicked_cb::m:%d p:%d\n", mark_current, point_current));

	if (point_current < 0) {
		return;
	}

	n_point = mark_get_point_n (mark_current);

	if (point_current == (n_point - 1)) {
		return;					/* 既に一番下 */
	}

	/* データの入れ換え */
	mark_swap_point (mark_current, point_current, point_current + 1);

	point_current += 1;

	point_draw_and_set_cursor ();
}

static void
add_point_button_clicked_cb (GtkButton * button, gpointer user_data)
{
	guint n;

	DB (g_print ("add_point_button_clicked_cb::m:%d p:%d\n", mark_current, point_current));

	if (mark_current < 0) {
		return;
	}

	n = mark_add_point (mark_current);

	point_current = n;

	point_draw_and_set_cursor ();

	glarea_force_render ();
}

static void
del_point_button_clicked_cb (GtkButton * button, gpointer user_data)
{
	guint n_point;

	DB (g_print ("del_point_button_clicked_cb::m:%d p:%d\n", mark_current, point_current));

	if (point_current < 0) {
		return;
	}

	/* データから削除 */
	mark_del_point (mark_current, point_current);

	n_point = mark_get_point_n (mark_current);
	if (n_point == 0) {			/* 最後の一つ */
		point_current = -1;
	}

	if (point_current == n_point) {	/* 最後尾 */
		point_current -= 1;
	} else {
		;
	}

	point_draw_and_set_cursor ();

	glarea_force_render ();
}


/******** マークの編集 ********/
static void
up_mark_button_clicked_cb (GtkButton * button, gpointer user_data)
{
	DB (g_print ("up_mark_button_clicked_cb::m:%d p:%d\n", mark_current, point_current));

	if (mark_current < 0) {
		return;
	}

	if (mark_current == 0) {
		return;					/* 既に一番上 */
	}

	/* データの入れ換え */
	mark_swap_mark (mark_current - 1, mark_current);

	mark_current -= 1;

	mark_draw_and_set_cursor ();
}

static void
down_mark_button_clicked_cb (GtkButton * button, gpointer user_data)
{
	guint n_mark;

	DB (g_print ("down_mark_button_clicked_cb::m:%d p:%d\n", mark_current, point_current));

	if (mark_current < 0) {
		return;
	}

	n_mark = mark_get_mark_n ();

	if (mark_current == (n_mark - 1)) {
		return;					/* 既に一番下 */
	}

	/* データの入れ換え */
	mark_swap_mark (mark_current, mark_current + 1);

	mark_current += 1;

	mark_draw_and_set_cursor ();
}

static void
add_mark_button_clicked_cb (GtkButton * button, gpointer user_data)
{
	guint n;

	DB (g_print ("add_mark_button_clicked_cb::m:%d p:%d\n", mark_current, point_current));

	n = mark_add_mark ();

	mark_current = n;
	point_current = -1;

	mark_draw_and_set_cursor ();

	glarea_force_render ();
}

static void
del_mark_button_clicked_cb (GtkButton * button, gpointer user_data)
{
	guint n_mark;				/* 削除後のマークの個数 */

	DB (g_print ("del_mark_button_clicked_cb::m:%d p:%d\n", mark_current, point_current));

	if (mark_current < 0) {
		return;
	}

	/* データから削除 */
	mark_del_mark (mark_current);

	n_mark = mark_get_mark_n ();
	if (n_mark == 0) {			/* 最後の一つ */
		mark_current = -1;
	}

	if (mark_current == n_mark) {	/* 最後尾 */
		mark_current -= 1;
	} else {
		;
	}

	point_current = -1;

	mark_draw_and_set_cursor ();

	glarea_force_render ();
}

/**************************************************************************/

static GtkWidget *
create_mark (void)
{
	GtkWidget *frame_mark;
	GtkWidget *vbox;
	GtkWidget *frame_list;
	GtkWidget *scroll;
	GtkListStore *list_store;
	GtkTreeViewColumn *column;
	GtkCellRenderer *renderer;
	GtkWidget *vbox_button;

	frame_mark = gtk_frame_new (_("Mark"));
	gtk_frame_set_shadow_type (GTK_FRAME (frame_mark), GTK_SHADOW_ETCHED_IN);

	vbox = gtk_vbox_new (FALSE, 0);

	frame_list = gtk_frame_new (NULL);
	gtk_frame_set_shadow_type (GTK_FRAME (frame_list), GTK_SHADOW_IN);

	scroll = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);


	list_store = gtk_list_store_new (MARK_N, G_TYPE_STRING, G_TYPE_INT);
	list_mark = gtk_tree_view_new_with_model (GTK_TREE_MODEL (list_store));
	g_signal_connect (list_mark, "row-activated", G_CALLBACK (mark_list_row_activated_cb), NULL);
	g_signal_connect (list_mark, "cursor-changed", G_CALLBACK (mark_list_cursor_changed_cb), NULL);
	g_object_unref (G_OBJECT (list_store));

	column = gtk_tree_view_column_new ();

	gtk_tree_view_column_set_title (column, "マーク名");

	renderer = gtk_cell_renderer_text_new ();
	gtk_tree_view_column_pack_start (column, renderer, TRUE);
	gtk_tree_view_column_set_attributes (column, renderer, "text", MARK_TEXT, NULL);
	gtk_tree_view_append_column (GTK_TREE_VIEW (list_mark), column);

	vbox_button = gtk_vbox_new (FALSE, 0);

	pbutton_up = gtk_button_new_with_label (_("Up"));
	pbutton_down = gtk_button_new_with_label (_("Down"));
	pbutton_add = gtk_button_new_with_label (_("Add"));
	pbutton_del = gtk_button_new_with_label (_("Remove"));

	g_signal_connect (pbutton_up, "clicked", G_CALLBACK (up_mark_button_clicked_cb), NULL);
	g_signal_connect (pbutton_down, "clicked", G_CALLBACK (down_mark_button_clicked_cb), NULL);
	g_signal_connect (pbutton_add, "clicked", G_CALLBACK (add_mark_button_clicked_cb), NULL);
	g_signal_connect (pbutton_del, "clicked", G_CALLBACK (del_mark_button_clicked_cb), NULL);

	gtk_container_add (GTK_CONTAINER (scroll), list_mark);
	gtk_container_add (GTK_CONTAINER (frame_list), scroll);

	gtk_box_pack_start (GTK_BOX (vbox_button), pbutton_up, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox_button), pbutton_down, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox_button), pbutton_add, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox_button), pbutton_del, FALSE, FALSE, 0);

	gtk_box_pack_start (GTK_BOX (vbox), frame_list, TRUE, TRUE, 5);
	gtk_box_pack_start (GTK_BOX (vbox), vbox_button, FALSE, FALSE, 5);

	gtk_container_add (GTK_CONTAINER (frame_mark), vbox);

	return frame_mark;
}

static GtkWidget *
create_mark_point (void)
{
	GtkWidget *frame_point;
	GtkWidget *vbox;
	GtkWidget *hbox;
	GtkWidget *frame_list;
	GtkWidget *scroll;
	GtkListStore *list_store;
	GtkTreeViewColumn *column_lon;
	GtkTreeViewColumn *column_lat;
	GtkTreeViewColumn *column_alt;
	GtkTreeViewColumn *column_text;
	GtkCellRenderer *renderer;

	frame_point = gtk_frame_new ("ポイント");
	gtk_frame_set_shadow_type (GTK_FRAME (frame_point), GTK_SHADOW_ETCHED_IN);

	vbox = gtk_vbox_new (FALSE, 0);

	frame_list = gtk_frame_new (NULL);
	gtk_frame_set_shadow_type (GTK_FRAME (frame_list), GTK_SHADOW_IN);

	scroll = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);


	list_store = gtk_list_store_new (POINT_N, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_UINT);
	list_point = gtk_tree_view_new_with_model (GTK_TREE_MODEL (list_store));
	g_signal_connect (list_point, "row-activated", G_CALLBACK (point_list_row_activated_cb), NULL);
	g_signal_connect (list_point, "cursor-changed", G_CALLBACK (point_list_cursor_changed_cb), NULL);
	g_object_unref (G_OBJECT (list_store));

	renderer = gtk_cell_renderer_text_new ();
	column_lon = gtk_tree_view_column_new_with_attributes ("経度", renderer, "text", POINT_LON, NULL);
	column_lat = gtk_tree_view_column_new_with_attributes ("緯度", renderer, "text", POINT_LAT, NULL);
	column_alt = gtk_tree_view_column_new_with_attributes ("高度", renderer, "text", POINT_ALT, NULL);
	column_text = gtk_tree_view_column_new_with_attributes (_("Comment"), renderer, "text", POINT_TEXT, NULL);

	gtk_tree_view_append_column (GTK_TREE_VIEW (list_point), column_lon);
	gtk_tree_view_append_column (GTK_TREE_VIEW (list_point), column_lat);
	gtk_tree_view_append_column (GTK_TREE_VIEW (list_point), column_alt);
	gtk_tree_view_append_column (GTK_TREE_VIEW (list_point), column_text);

	gtk_container_add (GTK_CONTAINER (scroll), list_point);
	gtk_container_add (GTK_CONTAINER (frame_list), scroll);

	hbox = gtk_hbox_new (FALSE, 0);

	mbutton_up = gtk_button_new_with_label (_("Up"));
	mbutton_down = gtk_button_new_with_label (_("Down"));
	mbutton_add = gtk_button_new_with_label (_("Add"));
	mbutton_del = gtk_button_new_with_label (_("Remove"));

	g_signal_connect (mbutton_up, "clicked", G_CALLBACK (up_point_button_clicked_cb), NULL);
	g_signal_connect (mbutton_down, "clicked", G_CALLBACK (down_point_button_clicked_cb), NULL);
	g_signal_connect (mbutton_add, "clicked", G_CALLBACK (add_point_button_clicked_cb), NULL);
	g_signal_connect (mbutton_del, "clicked", G_CALLBACK (del_point_button_clicked_cb), NULL);

	gtk_box_pack_start (GTK_BOX (hbox), mbutton_up, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), mbutton_down, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), mbutton_add, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), mbutton_del, TRUE, TRUE, 0);

	gtk_box_pack_start (GTK_BOX (vbox), frame_list, TRUE, TRUE, 5);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 5);

	gtk_container_add (GTK_CONTAINER (frame_point), vbox);

	return frame_point;
}


static GtkWidget *
create_edit (void)
{
	GtkWidget *frame;
	GtkWidget *vbox;
	GtkWidget *hbox_name;
	GtkWidget *hbox_lon;
	GtkWidget *hbox_lat;
	GtkWidget *hbox_alt;
	GtkWidget *hbox_text;
	GtkWidget *label_name;
	GtkWidget *label_lon;
	GtkWidget *label_lon_e;
	GtkWidget *label_lon_deg;
	GtkWidget *label_lon_min;
	GtkWidget *label_lon_sec;
	GtkWidget *label_lat;
	GtkWidget *label_lat_n;
	GtkWidget *label_lat_deg;
	GtkWidget *label_lat_min;
	GtkWidget *label_lat_sec;
	GtkWidget *label_alt;
	GtkWidget *label_alt_m;
	GtkWidget *label_text;

	frame = gtk_frame_new ("変更");
	gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);

	vbox = gtk_vbox_new (FALSE, 0);

	label_name = gtk_label_new ("マーク名：");

	label_lon = gtk_label_new ("経度：");
	label_lon_e = gtk_label_new ("東経");
	label_lon_deg = gtk_label_new ("度");
	label_lon_min = gtk_label_new ("分");
	label_lon_sec = gtk_label_new ("秒");

	label_lat = gtk_label_new ("緯度：");
	label_lat_n = gtk_label_new ("北緯");
	label_lat_deg = gtk_label_new ("度");
	label_lat_min = gtk_label_new ("分");
	label_lat_sec = gtk_label_new ("秒");

	label_alt = gtk_label_new ("高度：");
	label_alt_m = gtk_label_new ("メートル");

	label_text = gtk_label_new ("コメント：");

	spin_lon_deg = gtk_spin_button_new_with_range (0.0, 180.0, 1.0);
	spin_lon_min = gtk_spin_button_new_with_range (0.0, 60.0, 1.0);
	spin_lon_sec = gtk_spin_button_new_with_range (0.0, 60.0, 0.1);

	spin_lat_deg = gtk_spin_button_new_with_range (0.0, 90.0, 1.0);
	spin_lat_min = gtk_spin_button_new_with_range (0.0, 60.0, 1.0);
	spin_lat_sec = gtk_spin_button_new_with_range (0.0, 60.0, 0.1);

	spin_alt = gtk_spin_button_new_with_range (0.0, 10000.0, 1.0);

	entry_name = gtk_entry_new ();
	entry_text = gtk_entry_new ();

	hbox_name = gtk_hbox_new (FALSE, 3);
	hbox_lon = gtk_hbox_new (FALSE, 3);
	hbox_lat = gtk_hbox_new (FALSE, 3);
	hbox_alt = gtk_hbox_new (FALSE, 3);
	hbox_text = gtk_hbox_new (FALSE, 3);

	button_change = gtk_button_new_with_label ("変更を反映させる");
	g_signal_connect (button_change, "clicked", G_CALLBACK (change_button_clicked_cb), NULL);

	gtk_box_pack_start (GTK_BOX (hbox_name), label_name, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox_name), entry_name, TRUE, TRUE, 0);

	gtk_box_pack_start (GTK_BOX (hbox_lon), label_lon, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox_lon), label_lon_e, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox_lon), spin_lon_deg, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox_lon), label_lon_deg, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox_lon), spin_lon_min, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox_lon), label_lon_min, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox_lon), spin_lon_sec, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox_lon), label_lon_sec, FALSE, FALSE, 0);

	gtk_box_pack_start (GTK_BOX (hbox_lat), label_lat, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox_lat), label_lat_n, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox_lat), spin_lat_deg, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox_lat), label_lat_deg, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox_lat), spin_lat_min, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox_lat), label_lat_min, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox_lat), spin_lat_sec, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox_lat), label_lat_sec, FALSE, FALSE, 0);

	gtk_box_pack_start (GTK_BOX (hbox_alt), label_alt, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox_alt), spin_alt, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox_alt), label_alt_m, FALSE, FALSE, 0);

	gtk_box_pack_start (GTK_BOX (hbox_text), label_text, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox_text), entry_text, TRUE, TRUE, 0);

	gtk_box_pack_start (GTK_BOX (vbox), hbox_name, FALSE, FALSE, 3);
	gtk_box_pack_start (GTK_BOX (vbox), hbox_lon, FALSE, FALSE, 3);
	gtk_box_pack_start (GTK_BOX (vbox), hbox_lat, FALSE, FALSE, 3);
	gtk_box_pack_start (GTK_BOX (vbox), hbox_alt, FALSE, FALSE, 3);
	gtk_box_pack_start (GTK_BOX (vbox), hbox_text, FALSE, FALSE, 3);
	gtk_box_pack_start (GTK_BOX (vbox), button_change, FALSE, FALSE, 3);

	gtk_container_add (GTK_CONTAINER (frame), vbox);

	return frame;
}

static void
mark_dialog_realize_cb (GtkWidget * widget, gpointer user_data)
{
	DB (g_print ("mark_dialog_realize_cb\n"));

	point_widget_sensitive (FALSE);
}


void
mark_dialog_create (GtkWindow * parent)
{
	GtkWidget *dialog;
	GtkWidget *mark;
	GtkWidget *mark_point;
	GtkWidget *mark_edit;
	GtkWidget *vbox;
	GtkWidget *hpaned;
	gint response;

	mark = create_mark ();
	mark_point = create_mark_point ();
	mark_edit = create_edit ();

	vbox = gtk_vbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), mark_point, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), mark_edit, FALSE, FALSE, 0);

	hpaned = gtk_hpaned_new ();
	g_signal_connect_after (hpaned, "realize", G_CALLBACK (mark_dialog_realize_cb), NULL);
	gtk_container_set_border_width (GTK_CONTAINER (hpaned), 5);
	gtk_paned_set_position (GTK_PANED (hpaned), 120);

	gtk_paned_pack1 (GTK_PANED (hpaned), mark, FALSE, TRUE);
	gtk_paned_pack2 (GTK_PANED (hpaned), vbox, TRUE, TRUE);

	mark_draw_and_set_cursor ();

	dialog =
		gtk_dialog_new_with_buttons ("マークの編集", parent, GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
									 GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE, NULL);
	gtk_window_set_default_size (GTK_WINDOW (dialog), 550, 600);

	gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), hpaned);
	gtk_widget_show_all (dialog);

	response = gtk_dialog_run (GTK_DIALOG (dialog));

	if (response == GTK_RESPONSE_CLOSE) {
	}
	gtk_widget_destroy (dialog);

	window_set_mark_submenu ();
}
