/*
 * MMap+ - 3d image viewer
 * Copyright 2005, 2006 Masahide Miyake
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <math.h>

#include <gtk/gtk.h>
#include <gtk/gtkgl.h>

#include <GL/gl.h>
#include <GL/glu.h>
#include <GL/glext.h>

#include "light.h"
#include "glarea.h"

/********************************************/

static gfloat light_pos[] = { -0.2, 1.0, 0.3, 0.0 };
static gfloat ambient[] = { 0.7, 0.7, 0.7, 1.0 };
static gfloat diffuse[] = { 0.3, 0.3, 0.3, 1.0 };
static gfloat specular[] = { 0.0, 0.0, 0.0, 1.0 };
static gfloat model_ambient[] = { 0.0, 0.0, 0.0, 0.0 };

void
light_on (void)
{
	glEnable (GL_LIGHTING);
	glEnable (GL_LIGHT0);

	glLightModelfv (GL_LIGHT_MODEL_AMBIENT, model_ambient);

	glLightfv (GL_LIGHT0, GL_AMBIENT, ambient);
	glLightfv (GL_LIGHT0, GL_DIFFUSE, diffuse);
	glLightfv (GL_LIGHT0, GL_SPECULAR, specular);

	glLightfv (GL_LIGHT0, GL_POSITION, light_pos);

}

void
light_off (void)
{
	glDisable (GL_LIGHT0);
	glDisable (GL_LIGHTING);
}

static void
angle_to_xyz (gdouble lon, gdouble lat)
{
	gdouble r;
	gdouble x, y, z;

	r = cos (2.0 * G_PI * lat / 360.0);

	x = r * cos (2.0 * G_PI * lon / 360.0);
	y = r * sin (2.0 * G_PI * lon / 360.0);
	z = sin (2.0 * G_PI * lat / 360.0);

	light_pos[0] = x;
	light_pos[1] = y;
	light_pos[2] = z;

    /*
	g_print ("angle_to_xyz:x:%f y:%f z:%f\n", light_pos[0], light_pos[1], light_pos[2]);
    */
}

static void
xyz_to_angle (gdouble * lon, gdouble * lat)
{
	gdouble x, y, z;

	x = light_pos[0];
	y = light_pos[1];
	z = light_pos[2];

	if (x > 0.0) {
		*lon = 360.0 / (2.0 * G_PI) * atan (y / x);
	} else {
		*lon = 360.0 / (2.0 * G_PI) * atan (y / x) + 180.0;
	}
	*lat = 360.0 / (2.0 * G_PI) * asin (z / 1.0);

	g_print ("xyz_to_angle:lon:%f lat:%f\n", *lon, *lat);
}

/********************************************/

static GtkWidget *spin_lon;
static GtkWidget *spin_lat;

static void
set_lightpos (void)
{
	gdouble lon, lat;

	lon = gtk_spin_button_get_value (GTK_SPIN_BUTTON (spin_lon));
	lat = gtk_spin_button_get_value (GTK_SPIN_BUTTON (spin_lat));

	angle_to_xyz (lon, lat);

    /*
	g_print ("set_lightpos:lon:%.0f  lat:%.0f\n", lon, lat);
    */

	glarea_force_render ();
}

static void
value_changed_cb (GtkSpinButton * spinbutton, gpointer user_data)
{
	set_lightpos ();
}

static GtkWidget *
widget_light (void)
{
	GtkWidget *hbox;
	GtkWidget *label_lon;
	GtkWidget *label_lat;
	gdouble lon;
	gdouble lat;

	hbox = gtk_hbox_new (FALSE, 0);

	spin_lon = gtk_spin_button_new_with_range (-180.0, 180.0, 1.0);
	spin_lat = gtk_spin_button_new_with_range (-90.0, 90.0, 1.0);

	xyz_to_angle (&lon, &lat);
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (spin_lon), lon);
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (spin_lat), lat);

	g_signal_connect (spin_lon, "value_changed", G_CALLBACK (value_changed_cb), NULL);
	g_signal_connect (spin_lat, "value_changed", G_CALLBACK (value_changed_cb), NULL);

	gtk_entry_set_alignment (GTK_ENTRY (spin_lon), 1.0);
	gtk_entry_set_alignment (GTK_ENTRY (spin_lat), 1.0);

	label_lon = gtk_label_new ("経度:");
	label_lat = gtk_label_new ("  緯度:");

	gtk_box_pack_start (GTK_BOX (hbox), label_lon, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), spin_lon, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), label_lat, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), spin_lat, FALSE, FALSE, 0);

	return hbox;
}

static void
dialog_realize_cb (GtkWidget * widget, gpointer user_data)
{
	;
}

void
light_dialog_create (GtkWidget * window)
{
	GtkWidget *dialog;
	GtkWidget *light;
	GtkWidget *vbox;
	gint response;
	GtkWidget *toplevel;

	light = widget_light ();

	vbox = gtk_vbox_new (FALSE, 10);
	gtk_container_set_border_width (GTK_CONTAINER (vbox), 10);
	gtk_box_pack_start (GTK_BOX (vbox), light, FALSE, FALSE, 0);

	toplevel = gtk_widget_get_toplevel (window);
	dialog = gtk_dialog_new_with_buttons ("光源の位置", GTK_WINDOW (toplevel),
										  GTK_DIALOG_DESTROY_WITH_PARENT, GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE, NULL);
	g_signal_connect (dialog, "realize", G_CALLBACK (dialog_realize_cb), NULL);

	gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), vbox);
	gtk_widget_show_all (dialog);

	response = gtk_dialog_run (GTK_DIALOG (dialog));

	if (response == GTK_RESPONSE_CLOSE) {
		set_lightpos ();
	} else {
		;
	}

	gtk_widget_destroy (dialog);

	glarea_force_render ();
}
