/*
    Melody maid
    copyright (c) 1998-2005 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "general.h"
#include "mm_list.h"
#include "misc/charuty.h"
#include "misc/misc.h"
#include <gdk/gdkkeysyms.h>


static void mm_list_class_init (MmListClass *klass);
static void mm_list_init       (MmList      *list);
static void mm_list_destroy    (GtkObject   *object);


enum
{
  CHANGED_SIGNAL,
  DELETED_SIGNAL,
  EDITED_SIGNAL,
  INSERTED_SIGNAL,
  MOVED_SIGNAL,
  SORTED_SIGNAL,
  TOGGLED_SIGNAL,
  LAST_SIGNAL
};
enum
{
  TARGET_MMAID_ID3TAG
};


static GtkWindowClass *parent_class = NULL;
static gint mm_list_signals[LAST_SIGNAL] = {0};


/******************************************************************************
*                                                                             *
******************************************************************************/
GtkType
mm_list_get_type (void)
{
  static GType type = 0;

  if (!type)
    {
      const static GTypeInfo info =
      {
        sizeof (MmListClass),
        NULL,               /* base_init */
        NULL,               /* base_finalize */
        (GClassInitFunc)mm_list_class_init,
        NULL,               /* class_finalize */
        NULL,               /* class_data */
        sizeof (MmList),
        0,              /* n_preallocs */
        (GInstanceInitFunc)mm_list_init,
      };

      type = g_type_register_static (GTK_TYPE_TREE_VIEW, "MmList", &info, 0);
    }

  return type;
}


static void
mm_list_class_init (MmListClass *klass)
{
  GObjectClass *gobject_class;
  GtkObjectClass *object_class;
  GtkWidgetClass *widget_class;

  gobject_class = (GObjectClass*) klass;
  object_class = (GtkObjectClass*) klass;
  widget_class = (GtkWidgetClass*) klass;

  parent_class = g_type_class_peek_parent (klass);

  object_class->destroy = mm_list_destroy;

  klass->changed  = NULL;
  klass->deleted  = NULL;
  klass->edited   = NULL;
  klass->inserted = NULL;
  klass->moved    = NULL;
  klass->sorted  = NULL;
  klass->toggled  = NULL;

  mm_list_signals[CHANGED_SIGNAL]
                = g_signal_new ("changed",
                                G_TYPE_FROM_CLASS (klass),
                                G_SIGNAL_RUN_FIRST | G_SIGNAL_DETAILED,
                                G_STRUCT_OFFSET (MmListClass, changed),
                                NULL, NULL,
                                g_cclosure_marshal_VOID__VOID,
                                G_TYPE_NONE, 0);
  mm_list_signals[DELETED_SIGNAL]
                = g_signal_new ("deleted",
                                G_TYPE_FROM_CLASS (klass),
                                G_SIGNAL_RUN_FIRST | G_SIGNAL_DETAILED,
                                G_STRUCT_OFFSET (MmListClass, deleted),
                                NULL, NULL,
                                g_cclosure_marshal_VOID__VOID,
                                G_TYPE_NONE, 0);
  mm_list_signals[EDITED_SIGNAL]
                = g_signal_new ("edited",
                                G_TYPE_FROM_CLASS (klass),
                                G_SIGNAL_RUN_FIRST | G_SIGNAL_DETAILED,
                                G_STRUCT_OFFSET (MmListClass, edited),
                                NULL, NULL,
                                gtk_marshal_VOID__INT_INT,
                                G_TYPE_NONE, 2, G_TYPE_INT, G_TYPE_INT);
  mm_list_signals[INSERTED_SIGNAL]
                = g_signal_new ("inserted",
                                G_TYPE_FROM_CLASS (klass),
                                G_SIGNAL_RUN_FIRST | G_SIGNAL_DETAILED,
                                G_STRUCT_OFFSET (MmListClass, inserted),
                                NULL, NULL,
                                g_cclosure_marshal_VOID__INT,
                                G_TYPE_NONE, 1, G_TYPE_INT);
  mm_list_signals[MOVED_SIGNAL]
                = g_signal_new ("moved",
                                G_TYPE_FROM_CLASS (klass),
                                G_SIGNAL_RUN_FIRST | G_SIGNAL_DETAILED,
                                G_STRUCT_OFFSET (MmListClass, moved),
                                NULL, NULL,
                                gtk_marshal_VOID__INT_INT,
                                G_TYPE_NONE, 2, G_TYPE_INT, G_TYPE_INT);
  mm_list_signals[SORTED_SIGNAL]
                = g_signal_new ("sorted",
                                G_TYPE_FROM_CLASS (klass),
                                G_SIGNAL_RUN_FIRST | G_SIGNAL_DETAILED,
                                G_STRUCT_OFFSET (MmListClass, sorted),
                                NULL, NULL,
                                g_cclosure_marshal_VOID__VOID,
                                G_TYPE_NONE, 0);
  mm_list_signals[TOGGLED_SIGNAL]
                = g_signal_new ("toggled",
                                G_TYPE_FROM_CLASS (klass),
                                G_SIGNAL_RUN_FIRST | G_SIGNAL_DETAILED,
                                G_STRUCT_OFFSET (MmListClass, toggled),
                                NULL, NULL,
                                g_cclosure_marshal_VOID__INT,
                                G_TYPE_NONE, 1, G_TYPE_INT);
}


/* ja:ツリービュー削除 */
static void
mm_list_destroy (GtkObject *object)
{
  MM_LIST (object)->signal = FALSE;
  mm_list_delete_all (MM_LIST (object));
  GTK_OBJECT_CLASS (parent_class)->destroy (object);
}


/* ja:選択が変化した */
void
mm_list_changed (GtkTreeSelection *select,
                 MmList           *list)
{
  if (list->signal)
    g_signal_emit_by_name (G_OBJECT (list), "changed");
}


/* ja:コラムがクリックされた */
static void
mm_list_clicked (GtkTreeViewColumn *column,
                 MmList            *list)
{
  gboolean visible, sensitive, toggle;
  gint i, count, c_visible, c_sensitive, c_toggle;
  GtkTreeIter iter;

  c_toggle = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (column),
                                                                "user_data"));
  switch (c_toggle)
    {
      case MM_LIST_ADD_TOGGLE:
        c_visible   = MM_LIST_ADD_VISIBLE;
        c_sensitive = MM_LIST_ADD_SENSITIVE;
        break;
      case MM_LIST_REFRESH_TOGGLE:
        c_visible   = MM_LIST_REFRESH_VISIBLE;
        c_sensitive = MM_LIST_REFRESH_SENSITIVE;
        break;
      case MM_LIST_DELETE_TOGGLE:
        c_visible   = MM_LIST_DELETE_VISIBLE;
        c_sensitive = MM_LIST_DELETE_SENSITIVE;
        break;
      default:
        if (list->signal)
          g_signal_emit_by_name (G_OBJECT (list), "sorted");
        return;
    }
  count = gtk_tree_model_iter_n_children (GTK_TREE_MODEL (list->store), NULL);
  for (i = 0; i < count; i++)
    {
      gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (list->store),
                                                            &iter, NULL, i);
      gtk_tree_model_get (GTK_TREE_MODEL (list->store), &iter,
                                                    c_visible,   &visible,
                                                    c_sensitive, &sensitive,
                                                    c_toggle,   &toggle, -1);
      if (visible && sensitive && !toggle)
        break;
    }
  toggle = i < count;
  for (i = 0; i < count; i++)
    {
      gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (list->store),
                                                            &iter, NULL, i);
      gtk_tree_model_get (GTK_TREE_MODEL (list->store), &iter,
                            c_visible, &visible, c_sensitive, &sensitive, -1);
      if (visible && sensitive)
        gtk_list_store_set (list->store, &iter, c_toggle, toggle,
            c_toggle == MM_LIST_ADD_TOGGLE    ? MM_LIST_REFRESH_SENSITIVE
                                    : MM_LIST_ADD_SENSITIVE, !toggle,
            c_toggle == MM_LIST_DELETE_TOGGLE ? MM_LIST_REFRESH_SENSITIVE
                                    : MM_LIST_DELETE_SENSITIVE, !toggle, -1);
    }
  if (list->signal)
    g_signal_emit_by_name (G_OBJECT (list), "toggled", c_toggle);
}


/* ja:チェック項目が変化したとき */
static void
mm_list_toggled (GtkCellRendererToggle *cell,
                 const gchar           *path_string,
                 MmList                *list)
{
  gboolean toggle;
  gint c;
  GtkTreeIter iter;
  GtkTreePath *path;

  c = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (cell), "user_data"));
  path = gtk_tree_path_new_from_string (path_string);
  gtk_tree_model_get_iter (GTK_TREE_MODEL (list->store), &iter, path);
  gtk_tree_path_free (path);
  gtk_tree_model_get (GTK_TREE_MODEL (list->store), &iter, c, &toggle, -1);
  gtk_list_store_set (list->store, &iter, c, !toggle,
            c == MM_LIST_ADD_TOGGLE    ? MM_LIST_REFRESH_SENSITIVE
                                       : MM_LIST_ADD_SENSITIVE, toggle,
            c == MM_LIST_DELETE_TOGGLE ? MM_LIST_REFRESH_SENSITIVE
                                       : MM_LIST_DELETE_SENSITIVE, toggle, -1);
  if (list->signal)
    g_signal_emit_by_name (G_OBJECT (list), "toggled", c);
}


/* ja:編集されたとき */
static void
mm_list_edited (GtkCellRendererText *cell,
                const gchar         *path_string,
                const gchar         *text,
                MmList              *list)
{
  gint i, c;
  GtkTreeIter iter;
  GtkTreePath *path;
  ID3Tag *id3tag;

  c = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (cell), "user_data"));
  switch (c)
    {
      case MM_LIST_FILE:
        if (!mm_list_is_valid_file (list, -1, text))
            return;
        break;
      case MM_LIST_CHARSET:
        if (text && text[0] != '\0' && !charuty_is_valid (text))
          return;
    }
  path = gtk_tree_path_new_from_string (path_string);
  gtk_tree_model_get_iter (GTK_TREE_MODEL (list->store), &iter, path);
  gtk_tree_path_free (path);
  gtk_tree_model_get (GTK_TREE_MODEL (list->store), &iter,
                                                MM_LIST_ID3TAG, &id3tag, -1);
  for (i = gtk_tree_model_iter_n_children
                        (GTK_TREE_MODEL (list->store), NULL) - 1; i >= 0; i--)
    {
      ID3Tag *id3tag0;

      gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (list->store),
                                                            &iter, NULL, i);
      gtk_tree_model_get (GTK_TREE_MODEL (list->store), &iter,
                                                MM_LIST_ID3TAG, &id3tag0, -1);
      if (id3tag == id3tag0)
        break;
    }
  if (c == MM_LIST_TRACK)
    {
      gint track;

      if (!misc_str_to_val (&track, text, 10, TRUE)
                                                || track < -1 || 255 < track)
        track = -1;
      mm_list_change (list, i, MM_LIST_TRACK, track, -1);
    }
  else
    {
      gchar *str;

      str = text && text[0] != '\0' ? g_strdup (text) : NULL;
      mm_list_change (list, i, c, str, -1);
      if (c == MM_LIST_CHARSET && str)
        charlist_renewal_all (str);
    }
  if (list->signal)
    g_signal_emit_by_name (G_OBJECT (list), "edited", i, c);
}


/* ja:ドロップされたとき */
static void
mm_list_drag_end (GtkWidget      *widget,
                  GdkDragContext *drag_context,
                  gpointer        user_data)
{
  if (MM_LIST (widget)->signal)
    g_signal_emit_by_name (G_OBJECT (widget), "sorted");
}


static void
mm_list_init (MmList *list)
{
  gint i;
  GtkTreeSelection *select;
  static gchar *title[] = {N_("File"),  N_("Title"), N_("Artist"),
                           N_("Album"), N_("Year"),  N_("Comment"),
                           N_("Track"), N_("Genre"), N_("Character Encoding")};
  GtkTargetEntry dnd_entries[1] = {{"GTK_TREE_MODEL_ROW", GTK_TARGET_SAME_WIDGET | GTK_TARGET_SAME_WIDGET, TARGET_MMAID_ID3TAG}};

  list->editable       = FALSE;
  list->editable_file  = FALSE;
  list->signal         = TRUE;
  list->column_add     = NULL;
  list->column_refresh = NULL;
  list->column_delete  = NULL;
  list->column_status  = NULL;
  /* ja:リストボックス */
  list->store = gtk_list_store_new (MM_LIST_N_COLUMNS,
                                G_TYPE_POINTER, /* MM_LIST_ID3TAG */
                                G_TYPE_STRING,  /* MM_LIST_FILE */
                                G_TYPE_STRING,  /* MM_LIST_TITLE */
                                G_TYPE_STRING,  /* MM_LIST_ARTIST */
                                G_TYPE_STRING,  /* MM_LIST_ALBUM */
                                G_TYPE_STRING,  /* MM_LIST_YEAR */
                                G_TYPE_STRING,  /* MM_LIST_COMMENT */
                                G_TYPE_STRING,  /* MM_LIST_TRACK */
                                G_TYPE_STRING,  /* MM_LIST_GENRE */
                                G_TYPE_STRING,  /* MM_LIST_CHARSET */
                                G_TYPE_STRING,  /* MM_LIST_STATUS */
                                G_TYPE_BOOLEAN, /* MM_LIST_EDITED */
                                G_TYPE_BOOLEAN, /* MM_LIST_EDITABLE */
                                G_TYPE_BOOLEAN, /* MM_LIST_EDITABLE_FILE */
                                G_TYPE_BOOLEAN, /* MM_LIST_ADD_VISIBLE */
                                G_TYPE_BOOLEAN, /* MM_LIST_ADD_SENSITIVE */
                                G_TYPE_BOOLEAN, /* MM_LIST_ADD_TOGGLE */
                                G_TYPE_BOOLEAN, /* MM_LIST_REFRESH_VISIBLE */
                                G_TYPE_BOOLEAN, /* MM_LIST_REFRESH_SENSITIVE */
                                G_TYPE_BOOLEAN, /* MM_LIST_REFRESH_TOGGLE */
                                G_TYPE_BOOLEAN, /* MM_LIST_DELETE_VISIBLE */
                                G_TYPE_BOOLEAN, /* MM_LIST_DELETE_SENSITIVE */
                                G_TYPE_BOOLEAN);/* MM_LIST_DELETE_TOGGLE */
  gtk_tree_view_set_model (GTK_TREE_VIEW (list), GTK_TREE_MODEL (list->store));
  gtk_tree_view_set_enable_search (GTK_TREE_VIEW (list), FALSE);
  for (i = 0; i < 9; i++)
    {
      GtkCellRenderer *renderer;
      GtkTreeViewColumn *column;

      renderer = gtk_cell_renderer_text_new ();
      g_signal_connect (renderer, "edited", G_CALLBACK (mm_list_edited), list);
      g_object_set_data (G_OBJECT (renderer), "user_data",
                                                    GINT_TO_POINTER (i + 1));
      column = gtk_tree_view_column_new_with_attributes (_(title[i]), renderer,
                "text", i + 1,
                "editable", i == 0 ? MM_LIST_EDITABLE_FILE : MM_LIST_EDITABLE,
                NULL);
      gtk_tree_view_column_set_resizable (column, TRUE);
      gtk_tree_view_column_set_sort_column_id (column, i + 1);
      gtk_tree_view_append_column (GTK_TREE_VIEW (list), column);
      g_signal_connect (G_OBJECT (column), "clicked",
                                        G_CALLBACK (mm_list_clicked), list);
      g_object_set_data (G_OBJECT (column), "user_data",
                                                    GINT_TO_POINTER (i + 1));
    }
  select = gtk_tree_view_get_selection (GTK_TREE_VIEW (list));
  gtk_tree_selection_set_mode (select, GTK_SELECTION_MULTIPLE);
  g_signal_connect (G_OBJECT (select), "changed",
                                        G_CALLBACK (mm_list_changed), list);
  /* en:Drag & Drop */
  gtk_tree_view_enable_model_drag_dest (GTK_TREE_VIEW (list),
                                            dnd_entries, 1, GDK_ACTION_MOVE);
  gtk_tree_view_enable_model_drag_source (GTK_TREE_VIEW (list),
                            GDK_BUTTON1_MASK, dnd_entries, 1, GDK_ACTION_MOVE);
  g_signal_connect (G_OBJECT (list), "drag-end",
                                        G_CALLBACK (mm_list_drag_end), NULL);
}


/******************************************************************************
*                                                                             *
* ja:リスト関数群                                                             *
*                                                                             *
******************************************************************************/
/*  ja:新規作成
    RET,ウィジェット                                                        */
GtkWidget*
mm_list_new (void)
{
  return GTK_WIDGET (g_object_new (MM_TYPE_LIST, NULL));
}


/*  ja:項目を追加する
      list,ウィジェット
    id3tag,タグ
       RET,項目,-1:失敗                                                     */
gint
mm_list_append (MmList *list,
                ID3Tag *id3tag)
{
  gint i;
  GtkTreeIter iter;

  gtk_list_store_append (list->store, &iter);
  gtk_list_store_set (list->store, &iter,
        MM_LIST_ID3TAG,            id3tag,
        MM_LIST_FILE,              id3tag->file,
        MM_LIST_TITLE,             id3tag->title,
        MM_LIST_ARTIST,            id3tag->artist,
        MM_LIST_ALBUM,             id3tag->album,
        MM_LIST_YEAR,              id3tag->year,
        MM_LIST_COMMENT,           id3tag->comment,
        MM_LIST_TRACK,             0 <= id3tag->track && id3tag->track <= 255
                                    ? g_strdup_printf ("%03d", id3tag->track)
                                    : NULL,
        MM_LIST_GENRE,             id3tag->genre,
        MM_LIST_CHARSET,           id3tag->charset,
        MM_LIST_STATUS,            NULL,
        MM_LIST_EDITED,            FALSE,
        MM_LIST_EDITABLE,          list->editable,
        MM_LIST_EDITABLE_FILE,     list->editable_file,
        MM_LIST_ADD_VISIBLE,       FALSE,
        MM_LIST_ADD_SENSITIVE,     TRUE,
        MM_LIST_ADD_TOGGLE,        FALSE,
        MM_LIST_REFRESH_VISIBLE,   FALSE,
        MM_LIST_REFRESH_SENSITIVE, TRUE,
        MM_LIST_REFRESH_TOGGLE,    FALSE,
        MM_LIST_DELETE_VISIBLE,    FALSE,
        MM_LIST_DELETE_SENSITIVE,  TRUE,
        MM_LIST_DELETE_TOGGLE,     FALSE,
        -1);
  for (i = gtk_tree_model_iter_n_children
                        (GTK_TREE_MODEL (list->store), NULL) - 1; i >= 0 ; i--)
    {
      ID3Tag *id3tag0;

      gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (list->store),
                                                            &iter, NULL, i);
      gtk_tree_model_get (GTK_TREE_MODEL (list->store), &iter,
                                                MM_LIST_ID3TAG, &id3tag0, -1);
      if (id3tag == id3tag0)
        break;
    }
  if (i >= 0 && list->signal)
    g_signal_emit_by_name (G_OBJECT (list), "inserted", i);
  return i;
}


/*  ja:項目を挿入する
      list,ウィジェット
         n,項目(この項目の直後に挿入,-1:先頭)
    id3tag,タグ
       RET,項目,-1:失敗                                                     */
gint
mm_list_insert (MmList     *list,
                const gint  n,
                ID3Tag     *id3tag)
{
  gint i;
  GtkTreeIter iter;

  if (n >= 0)
    {
      GtkTreeIter sibling;

      gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (list->store),
                                                            &sibling, NULL, n);
      gtk_list_store_insert_after (list->store, &iter, &sibling);
    }
  else
    {
      gtk_list_store_prepend (list->store, &iter);
    }
  gtk_list_store_set (list->store, &iter,
        MM_LIST_ID3TAG,            id3tag,
        MM_LIST_FILE,              id3tag->file,
        MM_LIST_TITLE,             id3tag->title,
        MM_LIST_ARTIST,            id3tag->artist,
        MM_LIST_ALBUM,             id3tag->album,
        MM_LIST_YEAR,              id3tag->year,
        MM_LIST_COMMENT,           id3tag->comment,
        MM_LIST_TRACK,             0 <= id3tag->track && id3tag->track <= 255
                                    ? g_strdup_printf ("%03d", id3tag->track)
                                    : NULL,
        MM_LIST_GENRE,             id3tag->genre,
        MM_LIST_CHARSET,           id3tag->charset,
        MM_LIST_STATUS,            NULL,
        MM_LIST_EDITED,            FALSE,
        MM_LIST_EDITABLE,          list->editable,
        MM_LIST_EDITABLE_FILE,     list->editable_file,
        MM_LIST_ADD_VISIBLE,       FALSE,
        MM_LIST_ADD_SENSITIVE,     TRUE,
        MM_LIST_ADD_TOGGLE,        FALSE,
        MM_LIST_REFRESH_VISIBLE,   FALSE,
        MM_LIST_REFRESH_SENSITIVE, TRUE,
        MM_LIST_REFRESH_TOGGLE,    FALSE,
        MM_LIST_DELETE_VISIBLE,    FALSE,
        MM_LIST_DELETE_SENSITIVE,  TRUE,
        MM_LIST_DELETE_TOGGLE,     FALSE,
        -1);
  for (i = gtk_tree_model_iter_n_children
                        (GTK_TREE_MODEL (list->store), NULL) - 1; i >= 0 ; i--)
    {
      ID3Tag *id3tag0;

      gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (list->store),
                                                            &iter, NULL, i);
      gtk_tree_model_get (GTK_TREE_MODEL (list->store), &iter,
                                                MM_LIST_ID3TAG, &id3tag0, -1);
      if (id3tag == id3tag0)
        break;
    }
  if (i >= 0 && list->signal)
    g_signal_emit_by_name (G_OBJECT (list), "inserted", i);
  return i;
}


/*  ja:項目を探す
    list,ウィジェット
     ...,コラム(gint)と比較内容,MM_LIST_INVALIDで終了
     RET,項目,-1:項目なし                                                   */
gint
mm_list_search (MmList *list,
                ...)
{
  gint i, count;

  count = gtk_tree_model_iter_n_children (GTK_TREE_MODEL (list->store), NULL);
  for (i = 0; i < count; i++)
    {
      gchar *text;
      gint c;
      va_list ap;
      GtkTreeIter iter;
      ID3Tag *id3tag;

      gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (list->store),
                                                            &iter, NULL, i);
      gtk_tree_model_get (GTK_TREE_MODEL (list->store), &iter,
                                                MM_LIST_ID3TAG, &id3tag, -1);
      va_start (ap, list);
      while ((c = va_arg (ap, gint)) >= 0)
        switch (c)
          {
            case MM_LIST_FILE:
              text = va_arg (ap, gchar *);
              if (g_strfilecmp (id3tag->file, text))
                goto loop;
              break;
            case MM_LIST_TITLE:
              text = va_arg (ap, gchar *);
              if (g_strcmp (id3tag->title, text))
                goto loop;
              break;
            case MM_LIST_ARTIST:
              text = va_arg (ap, gchar *);
              if (g_strcmp (id3tag->artist, text))
                goto loop;
              break;
            case MM_LIST_ALBUM:
              text = va_arg (ap, gchar *);
              if (g_strcmp (id3tag->album, text))
                goto loop;
              break;
            case MM_LIST_YEAR:
              text = va_arg (ap, gchar *);
              if (g_strcmp (id3tag->year, text))
                goto loop;
              break;
            case MM_LIST_COMMENT:
              text = va_arg (ap, gchar *);
              if (g_strcmp (id3tag->comment, text))
                goto loop;
              break;
            case MM_LIST_TRACK:
              {
                gint track;

                track = va_arg (ap, gint);
                if (id3tag->track != track)
                  goto loop;
              }
              break;
            case MM_LIST_GENRE:
              text = va_arg (ap, gchar *);
              if (g_strcmp (id3tag->genre, text))
                goto loop;
              break;
            case MM_LIST_CHARSET:
              text = va_arg (ap, gchar *);
              if (g_strcasecmp (id3tag->charset, text))
                goto loop;
          }
      loop:
      va_end (ap);
      if (c < 0)
        return i;
    }
  return -1;
}


/*  ja:項目を変更する
    list,ウィジェット
       n,項目(この項目の直後に挿入,-1:先頭)
     ...,コラム(gint)と変更内容,MM_LIST_INVALIDで終了                       */
void
mm_list_change (MmList     *list,
                const gint  n,
                ...)
{
  gint c;
  GtkTreeIter iter;
  va_list ap;
  ID3Tag *id3tag;

  gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (list->store), &iter, NULL, n);
  gtk_tree_model_get (GTK_TREE_MODEL (list->store), &iter,
                                                MM_LIST_ID3TAG, &id3tag, -1);
  va_start (ap, n);
  while ((c = va_arg (ap, gint)) >= 0)
    {
      switch (c)
        {
          case MM_LIST_ID3TAG:
            {
              gchar *track;
              ID3Tag *id3tag0;

              id3tag0 = va_arg (ap, ID3Tag *);
              track = 0 <= id3tag0->track && id3tag0->track <= 255
                                   ? g_strdup_printf ("%03d", id3tag0->track)
                                   : NULL;
              gtk_list_store_set (list->store, &iter,
                  MM_LIST_ID3TAG,  id3tag0,
                  MM_LIST_FILE,    id3tag0->file,
                  MM_LIST_TITLE,   id3tag0->title,
                  MM_LIST_ARTIST,  id3tag0->artist,
                  MM_LIST_ALBUM,   id3tag0->album,
                  MM_LIST_YEAR,    id3tag0->year,
                  MM_LIST_COMMENT, id3tag0->comment,
                  MM_LIST_TRACK,   track,
                  MM_LIST_GENRE,   id3tag0->genre,
                  MM_LIST_CHARSET, id3tag0->charset,
                  -1);
              g_free (track);
              g_free (id3tag->file);
              g_free (id3tag->title);
              g_free (id3tag->artist);
              g_free (id3tag->album);
              g_free (id3tag->year);
              g_free (id3tag->comment);
              g_free (id3tag->genre);
              g_free (id3tag->charset);
              g_free (id3tag);
              id3tag = id3tag0;
            }
            break;
          case MM_LIST_FILE:
            g_free (id3tag->file);
            id3tag->file = va_arg (ap, gchar *);
            gtk_list_store_set (list->store, &iter,
                                        MM_LIST_FILE, id3tag->file, -1);
            break;
          case MM_LIST_TITLE:
            g_free (id3tag->title);
            id3tag->title = va_arg (ap, gchar *);
            gtk_list_store_set (list->store, &iter,
                                        MM_LIST_TITLE, id3tag->title, -1);
            break;
          case MM_LIST_ARTIST:
            g_free (id3tag->artist);
            id3tag->artist = va_arg (ap, gchar *);
            gtk_list_store_set (list->store, &iter,
                                        MM_LIST_ARTIST, id3tag->artist, -1);
            break;
          case MM_LIST_ALBUM:
            g_free (id3tag->album);
            id3tag->album = va_arg (ap, gchar *);
            gtk_list_store_set (list->store, &iter,
                                        MM_LIST_ALBUM, id3tag->album, -1);
            break;
          case MM_LIST_YEAR:
            g_free (id3tag->year);
            id3tag->year = va_arg (ap, gchar *);
            gtk_list_store_set (list->store, &iter,
                                        MM_LIST_YEAR, id3tag->year, -1);
            break;
          case MM_LIST_COMMENT:
            g_free (id3tag->comment);
            id3tag->comment = va_arg (ap, gchar *);
            gtk_list_store_set (list->store, &iter,
                                        MM_LIST_COMMENT, id3tag->comment, -1);
            break;
          case MM_LIST_TRACK:
            {
              gchar *track;

              id3tag->track = va_arg (ap, gint);
              track = 0 <= id3tag->track && id3tag->track <= 255
                            ? g_strdup_printf ("%03d", id3tag->track) : NULL;
              gtk_list_store_set (list->store, &iter,
                                        MM_LIST_TRACK, track, -1);
              g_free (track);
            }
            break;
          case MM_LIST_GENRE:
            g_free (id3tag->genre);
            id3tag->genre = va_arg (ap, gchar *);
            gtk_list_store_set (list->store, &iter,
                                        MM_LIST_GENRE, id3tag->genre, -1);
            break;
          case MM_LIST_CHARSET:
            g_free (id3tag->charset);
            id3tag->charset = va_arg (ap, gchar *);
            gtk_list_store_set (list->store, &iter,
                                        MM_LIST_CHARSET, id3tag->charset, -1);
        }
      if (MM_LIST_ID3TAG <= c && c <= MM_LIST_CHARSET && list->signal)
        g_signal_emit_by_name (G_OBJECT (list), "edited", n, c);
    }
  va_end (ap);
}


/*  ja:項目を入れ換える
    list,ウィジェット
      n0,項目0
      n1,項目1                                                              */
void
mm_list_swap (MmList     *list,
              const gint  n0,
              const gint  n1)
{
  gboolean selected0 ,selected1;
  gboolean edited0, editable0, editable_file0;
  gboolean add_visible0, add_sensitive0, add_toggle0;
  gboolean refresh_visible0, refresh_sensitive0, refresh_toggle0;
  gboolean delete_visible0, delete_sensitive0, delete_toggle0;
  gboolean edited1, editable1, editable_file1;
  gboolean add_visible1, add_sensitive1, add_toggle1;
  gboolean refresh_visible1, refresh_sensitive1, refresh_toggle1;
  gboolean delete_visible1, delete_sensitive1, delete_toggle1;
  gchar *track0, *track1, *status0, *status1;
  GtkTreeIter iter0, iter1;
  GtkTreeSelection *select;
  ID3Tag *id3tag0, *id3tag1;

  gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (list->store),
                                                            &iter0, NULL, n0);
  gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (list->store),
                                                            &iter1, NULL, n1);
  select = gtk_tree_view_get_selection (GTK_TREE_VIEW (list));
  selected0 = gtk_tree_selection_iter_is_selected (select, &iter0);
  selected1 = gtk_tree_selection_iter_is_selected (select, &iter1);
  if (selected0 ^ selected1)
    {
      if (selected0)
        {
          gtk_tree_selection_unselect_iter (select, &iter0);
          gtk_tree_selection_select_iter (select, &iter1);
        }
      else
        {
          gtk_tree_selection_select_iter (select, &iter0);
          gtk_tree_selection_unselect_iter (select, &iter1);
        }
    }
  gtk_tree_model_get (GTK_TREE_MODEL (list->store), &iter0,
                                MM_LIST_ID3TAG,            &id3tag0,
                                MM_LIST_TRACK,             &track0,
                                MM_LIST_STATUS,            &status0,
                                MM_LIST_EDITED,            &edited0,
                                MM_LIST_EDITABLE,          &editable0,
                                MM_LIST_EDITABLE_FILE,     &editable_file0,
                                MM_LIST_ADD_VISIBLE,       &add_visible0,
                                MM_LIST_ADD_SENSITIVE,     &add_sensitive0,
                                MM_LIST_ADD_TOGGLE,        &add_toggle0,
                                MM_LIST_REFRESH_VISIBLE,   &refresh_visible0,
                                MM_LIST_REFRESH_SENSITIVE, &refresh_sensitive0,
                                MM_LIST_REFRESH_TOGGLE,    &refresh_toggle0,
                                MM_LIST_DELETE_VISIBLE,    &delete_visible0,
                                MM_LIST_DELETE_SENSITIVE,  &delete_sensitive0,
                                MM_LIST_DELETE_TOGGLE,     &delete_toggle0,
                                -1);
  gtk_tree_model_get (GTK_TREE_MODEL (list->store), &iter1,
                                MM_LIST_ID3TAG,            &id3tag1,
                                MM_LIST_TRACK,             &track1,
                                MM_LIST_STATUS,            &status1,
                                MM_LIST_EDITED,            &edited1,
                                MM_LIST_EDITABLE,          &editable1,
                                MM_LIST_EDITABLE_FILE,     &editable_file1,
                                MM_LIST_ADD_VISIBLE,       &add_visible1,
                                MM_LIST_ADD_SENSITIVE,     &add_sensitive1,
                                MM_LIST_ADD_TOGGLE,        &add_toggle1,
                                MM_LIST_REFRESH_VISIBLE,   &refresh_visible1,
                                MM_LIST_REFRESH_SENSITIVE, &refresh_sensitive1,
                                MM_LIST_REFRESH_TOGGLE,    &refresh_toggle1,
                                MM_LIST_DELETE_VISIBLE,    &delete_visible1,
                                MM_LIST_DELETE_SENSITIVE,  &delete_sensitive1,
                                MM_LIST_DELETE_TOGGLE,     &delete_toggle1,
                                -1);
  gtk_list_store_set (list->store, &iter0,
                                MM_LIST_ID3TAG,            id3tag1,
                                MM_LIST_FILE,              id3tag1->file,
                                MM_LIST_TITLE,             id3tag1->title,
                                MM_LIST_ARTIST,            id3tag1->artist,
                                MM_LIST_ALBUM,             id3tag1->album,
                                MM_LIST_YEAR,              id3tag1->year,
                                MM_LIST_COMMENT,           id3tag1->comment,
                                MM_LIST_TRACK,             track1,
                                MM_LIST_GENRE,             id3tag1->genre,
                                MM_LIST_CHARSET,           id3tag1->charset,
                                MM_LIST_STATUS,            status1,
                                MM_LIST_EDITED,            edited1,
                                MM_LIST_EDITABLE,          editable1,
                                MM_LIST_EDITABLE_FILE,     editable_file1,
                                MM_LIST_ADD_VISIBLE,       add_visible1,
                                MM_LIST_ADD_SENSITIVE,     add_sensitive1,
                                MM_LIST_ADD_TOGGLE,        add_toggle1,
                                MM_LIST_REFRESH_VISIBLE,   refresh_visible1,
                                MM_LIST_REFRESH_SENSITIVE, refresh_sensitive1,
                                MM_LIST_REFRESH_TOGGLE,    refresh_toggle1,
                                MM_LIST_DELETE_VISIBLE,    delete_visible1,
                                MM_LIST_DELETE_SENSITIVE,  delete_sensitive1,
                                MM_LIST_DELETE_TOGGLE,     delete_toggle1,
                                -1);
  gtk_list_store_set (list->store, &iter1,
                                MM_LIST_ID3TAG,            id3tag0,
                                MM_LIST_FILE,              id3tag0->file,
                                MM_LIST_TITLE,             id3tag0->title,
                                MM_LIST_ARTIST,            id3tag0->artist,
                                MM_LIST_ALBUM,             id3tag0->album,
                                MM_LIST_YEAR,              id3tag0->year,
                                MM_LIST_COMMENT,           id3tag0->comment,
                                MM_LIST_TRACK,             track0,
                                MM_LIST_GENRE,             id3tag0->genre,
                                MM_LIST_CHARSET,           id3tag0->charset,
                                MM_LIST_STATUS,            status0,
                                MM_LIST_EDITED,            edited0,
                                MM_LIST_EDITABLE,          editable0,
                                MM_LIST_EDITABLE_FILE,     editable_file0,
                                MM_LIST_ADD_VISIBLE,       add_visible0,
                                MM_LIST_ADD_SENSITIVE,     add_sensitive0,
                                MM_LIST_ADD_TOGGLE,        add_toggle0,
                                MM_LIST_REFRESH_VISIBLE,   refresh_visible0,
                                MM_LIST_REFRESH_SENSITIVE, refresh_sensitive0,
                                MM_LIST_REFRESH_TOGGLE,    refresh_toggle0,
                                MM_LIST_DELETE_VISIBLE,    delete_visible0,
                                MM_LIST_DELETE_SENSITIVE,  delete_sensitive0,
                                MM_LIST_DELETE_TOGGLE,     delete_toggle0,
                                -1);
  g_free (track0);
  g_free (track1);
  g_free (status0);
  g_free (status1);
  if (list->signal)
    g_signal_emit_by_name (G_OBJECT (list), "moved", n0, n1);
}


/*  ja:項目を削除する
    list,ウィジェット
       n,項目                                                               */
void
mm_list_delete (MmList     *list,
                const gint  n)
{
  GtkTreeIter iter;
  ID3Tag *id3tag;

  gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (list->store), &iter, NULL, n);
  gtk_tree_model_get (GTK_TREE_MODEL (list->store), &iter,
                                                MM_LIST_ID3TAG, &id3tag, -1);
  gtk_list_store_remove (list->store, &iter);
  if (id3tag)
    {
      g_free (id3tag->file);
      g_free (id3tag->title);
      g_free (id3tag->artist);
      g_free (id3tag->album);
      g_free (id3tag->year);
      g_free (id3tag->comment);
      g_free (id3tag->genre);
      g_free (id3tag->charset);
      g_free (id3tag);
    }
  if (list->signal)
    g_signal_emit_by_name (G_OBJECT (list), "deleted");
}


/*  ja:項目をすべて削除する
    list,ウィジェット                                                       */
void
mm_list_delete_all (MmList *list)
{
  gboolean signal;
  gint i;

  signal = list->signal;
  list->signal = FALSE;
  for (i = gtk_tree_model_iter_n_children
                        (GTK_TREE_MODEL (list->store), NULL) - 1; i >= 0; i--)
    mm_list_delete (list, i);
  list->signal = signal;
  if (list->signal)
    g_signal_emit_by_name (G_OBJECT (list), "deleted");
}


/******************************************************************************
*                                                                             *
******************************************************************************/
/*  ja:項目の数を取得する
    list,ウィジェット
     RET,数                                                                 */
gint
mm_list_length (MmList *list)
{
  return gtk_tree_model_iter_n_children (GTK_TREE_MODEL (list->store), NULL);
}


/*  ja:項目を取得する
    list,ウィジェット
       n,項目
     RET,タグ,NULL                                                          */
const ID3Tag *
mm_list_get_nth (MmList     *list,
                 const gint  n)
{
  GtkTreeIter iter;
  ID3Tag *id3tag;

  gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (list->store), &iter, NULL, n);
  gtk_tree_model_get (GTK_TREE_MODEL (list->store), &iter,
                                                MM_LIST_ID3TAG, &id3tag, -1);
  return id3tag;
}


/*  ja:カーソル位置を取得する
    list,ウィジェット
       n,カーソル位置,-1:カーソル位置なし
       c,コラム位置,MM_LIST_INVALID:コラム位置なし                          */
void
mm_list_get_cursor (MmList *list,
                    gint   *n,
                    gint   *c)
{
  GtkTreePath *path;
  GtkTreeViewColumn *column;

  gtk_tree_view_get_cursor (GTK_TREE_VIEW (list), &path, &column);
  if (n)
    {
      if (path)
        {
          GtkTreeIter iter;
          ID3Tag *id3tag;

          gtk_tree_model_get_iter (GTK_TREE_MODEL (list->store), &iter, path);
          gtk_tree_model_get (GTK_TREE_MODEL (list->store), &iter,
                                                MM_LIST_ID3TAG, &id3tag, -1);
          for (*n = gtk_tree_model_iter_n_children
                    (GTK_TREE_MODEL (list->store), NULL) - 1; *n >= 0; (*n)--)
            {
              ID3Tag *id3tag0;

              gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (list->store),
                                                            &iter, NULL, *n);
              gtk_tree_model_get (GTK_TREE_MODEL (list->store), &iter,
                                                MM_LIST_ID3TAG, &id3tag0, -1);
              if (id3tag == id3tag0)
                break;
            }
        }
      else
        {
          *n = -1;
        }
    }
  if (path)
    gtk_tree_path_free (path);
  if (c)
    *c = column
        ? GPOINTER_TO_INT (g_object_get_data (G_OBJECT (column), "user_data"))
        : MM_LIST_INVALID;
}


/*  ja:カーソル位置を設定する
    list,ウィジェット
       n,カーソル位置,-1:カーソル位置なし
       c,コラム位置,MM_LIST_INVALID:コラム位置なし                          */
void
mm_list_set_cursor (MmList     *list,
                    const gint  n,
                    const gint  c)
{
  GtkTreePath *path;
  GtkTreeViewColumn *column = NULL;

  if (c != MM_LIST_INVALID)
    {
      gint i;
      GList *glist;

      glist = gtk_tree_view_get_columns (GTK_TREE_VIEW (list));
      for (i = g_list_length (glist) - 1; i >= 0; i--)
        {

          column = g_list_nth_data (glist, i);
          if (GPOINTER_TO_INT (g_object_get_data (G_OBJECT (column),
                                                            "user_data")) == c)
            break;
        }
    }
  if (n >= 0)
    {
      GtkTreeIter iter;

      gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (list->store),
                                                            &iter, NULL, n);
      path = gtk_tree_model_get_path (GTK_TREE_MODEL (list->store), &iter);
    }
  else
    {
      gtk_tree_view_get_cursor (GTK_TREE_VIEW (list), &path, NULL);
    }
  gtk_tree_view_set_cursor (GTK_TREE_VIEW (list), path, column, FALSE);
  gtk_tree_path_free (path);
}


/*  ja:座標から位置を取得する
    list,ウィジェット
       x,X座標
       y,Y座標
       n,カーソル位置,-1:カーソル位置なし
       c,コラム位置,-1:コラム位置なし                                       */
void
mm_list_get_cursor_from_pos (MmList     *list,
                             const gint  x,
                             const gint  y,
                             gint       *n,
                             gint       *c)
{
  GtkTreePath *path;
  GtkTreeViewColumn *column;

  if (gtk_tree_view_get_path_at_pos (GTK_TREE_VIEW (list), x, y,
                                                &path, &column, NULL, NULL))
    {
      if (n)
        {
          if (path)
            {
              GtkTreeIter iter;
              ID3Tag *id3tag;

              gtk_tree_model_get_iter (GTK_TREE_MODEL (list->store), &iter,
                                                                        path);
              gtk_tree_model_get (GTK_TREE_MODEL (list->store), &iter,
                                                MM_LIST_ID3TAG, &id3tag, -1);
              for (*n = gtk_tree_model_iter_n_children
                    (GTK_TREE_MODEL (list->store), NULL) - 1; *n >= 0; (*n)--)
                {
                  ID3Tag *id3tag0;

                  gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (list->store),
                                                            &iter, NULL, *n);
                  gtk_tree_model_get (GTK_TREE_MODEL (list->store), &iter,
                                                MM_LIST_ID3TAG, &id3tag0, -1);
                  if (id3tag == id3tag0)
                    break;
                }
            }
          else
            {
              *n = -1;
            }
        }
      if (path)
        gtk_tree_path_free (path);
      if (c)
        *c = column
        ? GPOINTER_TO_INT (g_object_get_data (G_OBJECT (column), "user_data"))
        : MM_LIST_INVALID;
    }
  else
    {
      if (n)
        *n = -1;
      if (c)
        *c = MM_LIST_INVALID;
    }
}


/******************************************************************************
*                                                                             *
******************************************************************************/
/*  ja:項目の選択を取得
    list,ウィジェット
       n,項目
     RET,TRUE:選択されている,FALSE:選択されていない                         */
gboolean
mm_list_is_selected (MmList     *list,
                     const gint  n)
{
  GtkTreeIter iter;
  GtkTreeSelection *select;

  select = gtk_tree_view_get_selection (GTK_TREE_VIEW (list));
  gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (list->store), &iter, NULL, n);
  return gtk_tree_selection_iter_is_selected (select, &iter);
}


/*  ja:項目を選択する
    list,ウィジェット
       n,項目                                                               */
void
mm_list_select (MmList     *list,
                const gint  n)
{
  GtkTreeIter iter;
  GtkTreeSelection *select;

  select = gtk_tree_view_get_selection (GTK_TREE_VIEW (list));
  gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (list->store), &iter, NULL, n);
  gtk_tree_selection_select_iter (select, &iter);
}


/*  ja:項目を選択しない
    list,ウィジェット
       n,項目                                                               */
void
mm_list_unselect (MmList     *list,
                  const gint  n)
{
  GtkTreeIter iter;
  GtkTreeSelection *select;

  select = gtk_tree_view_get_selection (GTK_TREE_VIEW (list));
  gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (list->store), &iter, NULL, n);
  gtk_tree_selection_unselect_iter (select, &iter);
}


/*  ja:項目をすべて選択する
    list,ウィジェット                                                       */
void
mm_list_select_all (MmList *list)
{
  gtk_tree_selection_select_all
                        (gtk_tree_view_get_selection (GTK_TREE_VIEW (list)));
}


/*  ja:項目をすべて選択しない
    list,ウィジェット                                                       */
void
mm_list_unselect_all (MmList *list)
{
  gtk_tree_selection_unselect_all
                        (gtk_tree_view_get_selection (GTK_TREE_VIEW (list)));
}


/******************************************************************************
*                                                                             *
******************************************************************************/
/*  ja:追加のチェックを表示する
    list,ウィジェット
       n,項目                                                               */
void
mm_list_show_add (MmList     *list,
                  const gint  n)
{
  GtkTreeIter iter;

  gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (list->store), &iter, NULL, n);
  gtk_list_store_set (list->store, &iter, MM_LIST_ADD_VISIBLE, TRUE, -1);
  if (!list->column_add)
    {
      GtkCellRenderer *renderer;

      renderer = gtk_cell_renderer_toggle_new ();
      g_signal_connect (G_OBJECT (renderer), "toggled",
                                        G_CALLBACK (mm_list_toggled), list);
      g_object_set_data (G_OBJECT (renderer), "user_data",
                                        GINT_TO_POINTER (MM_LIST_ADD_TOGGLE));
      list->column_add = gtk_tree_view_column_new_with_attributes
                             (_("Add"), renderer,
                            "active",      MM_LIST_ADD_TOGGLE,
                            "visible",     MM_LIST_ADD_VISIBLE,
                            "activatable", MM_LIST_ADD_SENSITIVE, NULL);
      gtk_tree_view_column_set_clickable (list->column_add, TRUE);
      g_signal_connect (G_OBJECT (list->column_add), "clicked",
                                        G_CALLBACK (mm_list_clicked), list);
      g_object_set_data (G_OBJECT (list->column_add), "user_data",
                                        GINT_TO_POINTER (MM_LIST_ADD_TOGGLE));
      gtk_tree_view_insert_column (GTK_TREE_VIEW (list), list->column_add, 0);
    }
}


/*  ja:更新のチェックを表示する
    list,ウィジェット
       n,項目                                                               */
void
mm_list_show_refresh (MmList     *list,
                      const gint  n)
{
  GtkTreeIter iter;

  gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (list->store), &iter, NULL, n);
  gtk_list_store_set (list->store, &iter, MM_LIST_REFRESH_VISIBLE, TRUE, -1);
  if (!list->column_refresh)
    {
      GtkCellRenderer *renderer;

      renderer = gtk_cell_renderer_toggle_new ();
      g_signal_connect (G_OBJECT (renderer), "toggled",
                                        G_CALLBACK (mm_list_toggled), list);
      g_object_set_data (G_OBJECT (renderer), "user_data",
                                    GINT_TO_POINTER (MM_LIST_REFRESH_TOGGLE));
      list->column_refresh = gtk_tree_view_column_new_with_attributes
                            (_("Refresh"), renderer,
                            "active",      MM_LIST_REFRESH_TOGGLE,
                            "visible",     MM_LIST_REFRESH_VISIBLE,
                            "activatable", MM_LIST_REFRESH_SENSITIVE, NULL);
      gtk_tree_view_column_set_clickable (list->column_refresh, TRUE);
      g_signal_connect (G_OBJECT (list->column_refresh), "clicked",
                                        G_CALLBACK (mm_list_clicked), list);
      g_object_set_data (G_OBJECT (list->column_refresh), "user_data",
                                    GINT_TO_POINTER (MM_LIST_REFRESH_TOGGLE));
      gtk_tree_view_insert_column (GTK_TREE_VIEW (list), list->column_refresh,
                                                    list->column_add ? 1 : 0);
    }
}


/*  ja:削除のチェックを表示する
    list,ウィジェット
       n,項目                                                               */
void
mm_list_show_delete (MmList     *list,
                     const gint  n)
{
  GtkTreeIter iter;

  gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (list->store), &iter, NULL, n);
  gtk_list_store_set (list->store, &iter, MM_LIST_DELETE_VISIBLE, TRUE, -1);
  if (!list->column_delete)
    {
      GtkCellRenderer *renderer;

      renderer = gtk_cell_renderer_toggle_new ();
      g_signal_connect (G_OBJECT (renderer), "toggled",
                                        G_CALLBACK (mm_list_toggled), list);
      g_object_set_data (G_OBJECT (renderer), "user_data",
                                    GINT_TO_POINTER (MM_LIST_DELETE_TOGGLE));
      list->column_delete = gtk_tree_view_column_new_with_attributes
                            (_("Delete"), renderer,
                            "active",      MM_LIST_DELETE_TOGGLE,
                            "visible",     MM_LIST_DELETE_VISIBLE,
                            "activatable", MM_LIST_DELETE_SENSITIVE, NULL);
      gtk_tree_view_column_set_clickable (list->column_delete, TRUE);
      g_signal_connect (G_OBJECT (list->column_delete), "clicked",
                                        G_CALLBACK (mm_list_clicked), list);
      g_object_set_data (G_OBJECT (list->column_delete), "user_data",
                                    GINT_TO_POINTER (MM_LIST_DELETE_TOGGLE));
      gtk_tree_view_insert_column (GTK_TREE_VIEW (list), list->column_delete,
                (list->column_add ? 1 : 0) + (list->column_refresh ? 1 : 0));
    }
}


/*  ja:追加のチェックを隠す
    list,ウィジェット
       n,項目                                                               */
void
mm_list_hide_add (MmList     *list,
                  const gint  n)
{
  GtkTreeIter iter;

  gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (list->store), &iter, NULL, n);
  gtk_list_store_set (list->store, &iter, MM_LIST_ADD_VISIBLE, FALSE, -1);
  if (list->column_add)
    {
      gboolean visible;
      gint i;

      for (i = gtk_tree_model_iter_n_children
                        (GTK_TREE_MODEL (list->store), NULL) - 1; i >= 0; i--)
        {
          gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (list->store),
                                                            &iter, NULL, n);
          gtk_tree_model_get (GTK_TREE_MODEL (list->store),
                                &iter, MM_LIST_ADD_VISIBLE, &visible, -1);
          if (visible)
            break;
        }
      if (i < 0)
        {
          gtk_tree_view_remove_column (GTK_TREE_VIEW (list), list->column_add);
          list->column_add = NULL;
        }
    }
}


/*  ja:更新のチェックを隠す
    list,ウィジェット
       n,項目                                                               */
void
mm_list_hide_refresh (MmList     *list,
                      const gint  n)
{
  GtkTreeIter iter;

  gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (list->store), &iter, NULL, n);
  gtk_list_store_set (list->store, &iter, MM_LIST_REFRESH_VISIBLE, FALSE, -1);
  if (list->column_refresh)
    {
      gboolean visible;
      gint i;

      for (i = gtk_tree_model_iter_n_children
                        (GTK_TREE_MODEL (list->store), NULL) - 1; i >= 0; i--)
        {
          gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (list->store),
                                                            &iter, NULL, n);
          gtk_tree_model_get (GTK_TREE_MODEL (list->store),
                                &iter, MM_LIST_REFRESH_VISIBLE, &visible, -1);
          if (visible)
            break;
        }
      if (i < 0)
        {
          gtk_tree_view_remove_column (GTK_TREE_VIEW (list),
                                                        list->column_refresh);
          list->column_refresh = NULL;
        }
    }
}


/*  ja:削除のチェックを隠す
    list,ウィジェット
       n,項目                                                               */
void
mm_list_hide_delete (MmList     *list,
                     const gint  n)
{
  GtkTreeIter iter;

  gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (list->store), &iter, NULL, n);
  gtk_list_store_set (list->store, &iter, MM_LIST_DELETE_VISIBLE, FALSE, -1);
  if (list->column_delete)
    {
      gboolean visible;
      gint i;

      for (i = gtk_tree_model_iter_n_children
                        (GTK_TREE_MODEL (list->store), NULL) - 1; i >= 0; i--)
        {
          gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (list->store),
                                                            &iter, NULL, n);
          gtk_tree_model_get (GTK_TREE_MODEL (list->store),
                                &iter, MM_LIST_DELETE_VISIBLE, &visible, -1);
          if (visible)
            break;
        }
      if (i < 0)
        {
          gtk_tree_view_remove_column (GTK_TREE_VIEW (list),
                                                        list->column_delete);
          list->column_delete = NULL;
        }
    }
}


/*  ja:追加のチェックを設定する
         list,ウィジェット
            n,項目
    sensitive,TRUE:有効,FALSE:無効                                          */
void
mm_list_set_sensitive_add (MmList         *list,
                           const gint      n,
                           const gboolean  sensitive)
{
  GtkTreeIter iter;

  gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (list->store), &iter, NULL, n);
  gtk_list_store_set (list->store, &iter,
                                    MM_LIST_ADD_SENSITIVE, sensitive, -1);
}


/*  ja:更新のチェックを設定する
         list,ウィジェット
            n,項目
    sensitive,TRUE:有効,FALSE:無効                                          */
void
mm_list_set_sensitive_refresh (MmList         *list,
                               const gint      n,
                               const gboolean  sensitive)
{
  GtkTreeIter iter;

  gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (list->store), &iter, NULL, n);
  gtk_list_store_set (list->store, &iter,
                                    MM_LIST_REFRESH_SENSITIVE, sensitive, -1);
}


/*  ja:削除のチェックを設定する
         list,ウィジェット
            n,項目
    sensitive,TRUE:有効,FALSE:無効                                          */
void
mm_list_set_sensitive_delete (MmList         *list,
                              const gint      n,
                              const gboolean  sensitive)
{
  GtkTreeIter iter;

  gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (list->store), &iter, NULL, n);
  gtk_list_store_set (list->store, &iter,
                                    MM_LIST_DELETE_SENSITIVE, sensitive, -1);
}


/*  ja:追加のチェック項目を取得する
    list,ウィジェット
       n,項目
     RET,TRUE:チェックあり,FALSE:チェックなし                               */
gboolean
mm_list_get_active_add (MmList     *list,
                        const gint  n)
{
  gboolean active;
  GtkTreeIter iter;

  gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (list->store), &iter, NULL, n);
  gtk_tree_model_get (GTK_TREE_MODEL (list->store),
                                &iter, MM_LIST_ADD_TOGGLE, &active, -1);
  return active;
}


/*  ja:更新のチェック項目を取得する
         list,ウィジェット
            n,項目
     RET,TRUE:チェックあり,FALSE:チェックなし                               */
gboolean
mm_list_get_active_refresh (MmList     *list,
                            const gint  n)
{
  gboolean active;
  GtkTreeIter iter;

  gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (list->store), &iter, NULL, n);
  gtk_tree_model_get (GTK_TREE_MODEL (list->store),
                                &iter, MM_LIST_REFRESH_TOGGLE, &active, -1);
  return active;
}


/*  ja:削除のチェック項目を取得する
         list,ウィジェット
            n,項目
     RET,TRUE:チェックあり,FALSE:チェックなし                               */
gboolean
mm_list_get_active_delete (MmList     *list,
                           const gint  n)
{
  gboolean active;
  GtkTreeIter iter;

  gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (list->store), &iter, NULL, n);
  gtk_tree_model_get (GTK_TREE_MODEL (list->store),
                                &iter, MM_LIST_DELETE_TOGGLE, &active, -1);
  return active;
}


/*  ja:追加のチェック項目を設定する
         list,ウィジェット
            n,項目
    is_active,TRUE:チェックあり,FALSE:チェックなし                          */
void
mm_list_set_active_add (MmList         *list,
                        const gint      n,
                        const gboolean  active)
{
  GtkTreeIter iter;

  gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (list->store), &iter, NULL, n);
  gtk_list_store_set (list->store, &iter, MM_LIST_ADD_TOGGLE, active, -1);
}


/*  ja:更新のチェック項目を設定する
         list,ウィジェット
            n,項目
    is_active,TRUE:チェックあり,FALSE:チェックなし                          */
void
mm_list_set_active_refresh (MmList         *list,
                            const gint      n,
                            const gboolean  active)
{
  GtkTreeIter iter;

  gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (list->store), &iter, NULL, n);
  gtk_list_store_set (list->store, &iter, MM_LIST_REFRESH_TOGGLE, active, -1);
}


/*  ja:削除のチェック項目を設定する
         list,ウィジェット
            n,項目
    is_active,TRUE:チェックあり,FALSE:チェックなし                          */
void
mm_list_set_active_delete (MmList         *list,
                           const gint      n,
                           const gboolean  active)
{
  GtkTreeIter iter;

  gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (list->store), &iter, NULL, n);
  gtk_list_store_set (list->store, &iter, MM_LIST_DELETE_TOGGLE, active, -1);
}


/******************************************************************************
*                                                                             *
******************************************************************************/
/*  ja:編集可能か設定する
        list,ウィジェット
    editable,TRUE:編集可能,FALSE:編集不可                                   */
void
mm_list_set_editable (MmList         *list,
                      const gboolean  editable)
{
  gint i;

  list->editable = editable;
  for (i = gtk_tree_model_iter_n_children
                        (GTK_TREE_MODEL (list->store), NULL) - 1; i >= 0; i--)
    {
      GtkTreeIter iter;

      gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (list->store),
                                                            &iter, NULL, i);
      gtk_list_store_set (list->store, &iter, MM_LIST_EDITABLE, editable, -1);
    }
}


/*  ja:ファイルを編集可能か設定する
        list,ウィジェット
    editable,TRUE:編集可能,FALSE:編集不可                                   */
void
mm_list_set_editable_file (MmList         *list,
                           const gboolean  editable)
{
  gint i;

  list->editable_file = editable;
  for (i = gtk_tree_model_iter_n_children
                        (GTK_TREE_MODEL (list->store), NULL) - 1; i >= 0; i--)
    {
      GtkTreeIter iter;

      gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (list->store),
                                                            &iter, NULL, i);
      gtk_list_store_set (list->store, &iter,
                                        MM_LIST_EDITABLE_FILE, editable, -1);
    }
}


/*  ja:追加/編集される項目のファイル名を判別する
    list,ウィジェット
       n,編集する項目,-1:追加
    file,ファイル名(UTF-8)
     RET,TRUE:追加/編集可能,FALSE:不可能                                    */
gboolean
mm_list_is_valid_file (MmList      *list,
                       const gint   n,
                       const gchar *file)
{
  gint i;

  if (!file || file[0] == '\0'
                || file[g_strlen (file) - 1] == G_DIR_SEPARATOR
                || !g_path_is_absolute (file)
                || g_strstr (file, G_DIR_SEPARATOR_S "." G_DIR_SEPARATOR_S)
                || g_strstr (file, G_DIR_SEPARATOR_S ".." G_DIR_SEPARATOR_S))
    return FALSE;
  for (i = gtk_tree_model_iter_n_children
                        (GTK_TREE_MODEL (list->store), NULL) - 1; i >= 0; i--)
    if (n != i)
      {
        gboolean result;
        gchar *str;
        GtkTreeIter iter;

        gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (list->store),
                                                            &iter, NULL, i);
        gtk_tree_model_get (GTK_TREE_MODEL (list->store), &iter,
                                                    MM_LIST_FILE, &str, -1);
        result = !g_strfilecmp (file, str);
        g_free (str);
        if (result)
          return FALSE;
      }
  return TRUE;
}


/*  ja:キャラクターセットを更新する
        list,ウィジェット
    charlist,キャラクターセットリスト                                       */
void
mm_list_charset_renewal (MmList      *list,
                         const gchar *charlist)
{
  gint i;

  for (i = gtk_tree_model_iter_n_children
                        (GTK_TREE_MODEL (list->store), NULL) - 1; i >= 0; i--)
    {
      GtkTreeIter iter;
      ID3Tag *id3tag;

      gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (list->store),
                                                            &iter, NULL, i);
      gtk_tree_model_get (GTK_TREE_MODEL (list->store), &iter,
                                                MM_LIST_ID3TAG, &id3tag, -1);
      if (id3tag->charset != charlist)
        {
          gchar *tmp;

          tmp = id3tag->charset;
          id3tag->charset = charuty_rename (id3tag->charset, charlist);
          gtk_list_store_set (list->store, &iter,
                                        MM_LIST_CHARSET, id3tag->charset, -1);
          g_free (tmp);
        }
    }
}


/*  ja:状態を設定する
    list,ウィジェット
       n,項目
    text,文字列,NULL                                                        */
void
mm_list_set_status (MmList      *list,
                    const gint   n,
                    const gchar *text)
{
  GtkTreeIter iter;

  gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (list->store), &iter, NULL, n);
  gtk_list_store_set (list->store, &iter, MM_LIST_STATUS, text, -1);
  if (!list->column_status && text)
    {
      GtkCellRenderer *renderer;

      renderer = gtk_cell_renderer_text_new ();
      g_object_set_data (G_OBJECT (renderer), "user_data",
                                            GINT_TO_POINTER (MM_LIST_STATUS));
      list->column_status = gtk_tree_view_column_new_with_attributes
                                                (_("Status"), renderer,
                                                "text", MM_LIST_STATUS, NULL);
      gtk_tree_view_column_set_resizable (list->column_status, TRUE);
      gtk_tree_view_column_set_sort_column_id (list->column_status,
                                                            MM_LIST_STATUS);
      gtk_tree_view_append_column (GTK_TREE_VIEW (list), list->column_status);
      g_signal_connect (G_OBJECT (list->column_status), "clicked",
                                        G_CALLBACK (mm_list_clicked), list);
      g_object_set_data (G_OBJECT (list->column_status), "user_data",
                                            GINT_TO_POINTER (MM_LIST_STATUS));
    }
  else if (list->column_status && !text)
    {
      gchar *str;
      gint i;

      for (i = gtk_tree_model_iter_n_children
                        (GTK_TREE_MODEL (list->store), NULL) - 1; i >= 0; i--)
        {
          gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (list->store),
                                                            &iter, NULL, n);
          gtk_tree_model_get (GTK_TREE_MODEL (list->store),
                                            &iter, MM_LIST_STATUS, &str, -1);
          g_free (str);
          if (str)
            break;
        }
      if (i < 0)
        {
          gtk_tree_view_remove_column (GTK_TREE_VIEW (list),
                                                        list->column_status);
          list->column_status = NULL;
        }
    }
}
