/**
 * @fileOverview LPCXpresso1769のメモリアクセスクラスを定義する。
 */

(function(){
var DEV=LPCXpresso1769;
var BCF=DEV._BCF;
var EE=DEV._EE;
var isUndef=MiMicLib.isUndef;
/**
 * LPCXPresso1769.Memory (Memory)クラスのコンストラクタ。
 * Memoryクラスは、MCUのメモリ空間へアクセスする手段を提供する。
 * このクラスは、メモリアクセス対してなんら保護機能を持たない。MCUのメモリマップに十分に注意する必要がある。
 * @name LPCXpresso1769.Memory
 * @constructor
 * @param {object as LPCXpresso1769.Mcu} i_mcu
 * インスタンスを結びつけるMcuオブジェクト。
 * @param {int} i_base
 * メモリにアクセスするときのベースアドレス。省略可能である。省略時は0x0とみなす。4バイト境界でなければならない。クラスの提供する関数でアドレスを指定した時には、全てこの値が加算される。
 * @example
 * //create a controlable MCU via network.
 * var mcu=new LPCXpresso1769.Mcu(“192.168.0.39”);
 * var memory=new LPCXpresso1769.Memory(mcu); 
 */
DEV.Memory=function Memory(i_mcu,i_base)
{
	if(!isUndef(i_base)){
		this._base=i_base;
	}
	this._mcu=i_mcu;
}
DEV.Memory.prototype=
{
	_base:0x00000000,
	_mcu:null,
	/**
	 * 指定したアドレスの32ビット値を取得する。
	 * @name LPCXpresso1769.Memory#read32^2
	 * @function
	 * @param {int} i_offset
	 * コンストラクタで指定したアドレスからのオフセット位置を指定する。4バイト境界でなければならない。
	 * @return {int}
	 * 1個のint値。
	 * @example
	 * var　mcu=new LPCXpresso1769.Mcu("192.168.0.39");
	 * var mem=new LPCXpresso1769.Memory(mcu);　//create instance
	 * alert(mem.read32(0x00000000)); //show 1 value 
	 */
	/**
	 * 指定したオフセットから、i_sizeバイトのメモリに格納した値を、それぞれ32ビット単位で値を取得する。i_offsetの位置から、32bit単位でi_size/4個の値を取得することになる。シーケンシャルアクセスに使用する。
	 * @name LPCXpresso1769.Memory#read32^3
	 * @function
	 * @param {int} i_offset
	 * コンストラクタで指定したアドレスからのオフセット位置を指定する。4バイト境界でなければならない。
	 * @param {int} i_size
	 * 取得するバイト数を指定する。4バイト単位でなければならない。	 
	 * @return {int or Array[int]}
	 * 取得する値の個数は、i_size/4個である。個数により、戻り値の型が異なるので注意すること。
	 * @example
	 * var　mcu=new LPCXpresso1769.Mcu("192.168.0.39");
	 * var mem=new LPCXpresso1769.Memory(mcu);　//create instance
	 * mem.read32(0x0,8); //return 2 values.
	 */
	/**
	 * オフセットの配列要素に対応した値を、それぞれ32ビット単位で取得する。
	 * ランダムアクセスに使用する。
	 * @name LPCXpresso1769.Memory#read32^4
	 * @function
	 * @param {array[int]} i_offsets
	 * オフセットアドレスの配列。それぞれ4バイト境界でなければならない。
	 * @return {int or Array[int]}
	 * 返却される値の個数は、i_offsetの長さと同じになる。個数により、戻り値の型が異なるので注意すること。
	 * @example
	 * var　mcu=new LPCXpresso1769.Mcu("192.168.0.39");
	 * var mem=new LPCXpresso1769.Memory(mcu);　//create instance
	 * mem.read32([0x0,0x14,0x08]); //return 3 values by specified order.
	 */
	/**
	 * メモリから値を読み出して、値セットの配列、又は値を返す。
	 * 関数はバイト単位のアライメントでメモリにアクセスする。メモリアドレス、取得サイズは4バイト境界に一致させなければならない。
	 * 引数の違いにより、数種類の呼び出し方がある。
	 * @name LPCXpresso1769.Memory#read32^1
	 * @function
	 * @param ...
	 * 詳細は、read32^nを参照。
	 * @return {int or Array[int]}
	 * 返却値の数により、型が異なる。1個の場合はint値、複数の場合はint型の配列である。
	 */
	read32:function read32(/*arguments*/)
	{
		try{
			function checkAlign(v){
				if(v%4!=0){
					throw new MiMicException(EE.INVALID_ARG,"An alignment is not 32bit unit.");
				}
			}
			var bc="";
			var ar=new Array();
			var offset=arguments[0];
			switch(arguments.length){
			case 1:
				if(!isNaN(offset)){
					//read32(i_offset:int)
					checkAlign(offset);
					bc=BCF.READMEM;
					ar.push(this._base+offset);
				}else{
					//read32(i_offsets:array)
					for(var i=0;i<offset.length;i++){
						checkAlign(offset[i]);
						bc+=BCF.getMem(this._base+offset[i],ar);
					}
				}
				break;
			case 2:
				//read32(i_offset:int,i_size:int)
				checkAlign(offset);
				checkAlign(arguments[1]);
				var l=arguments[1]/4;
				for(var i=0;i<l;i++){
					bc+=BCF.READMEM;
					ar.push(this._base+offset+i*4);
				}
				break;
			default:
				break;
			}
			var ret=this._mcu.callMiMicWithCheck(bc+BCF.END,ar).stream;
			return ret.length==1?ret[0]:ret;
		}catch(e){
			throw new MiMicException(e);
		}
	}
}

}());