/*
 * ProcessManager.m
 *
 * Copyright (C) 2008 MikuInstaller Project. All rights reserved.
 * http://mikuinstaller.sourceforge.jp/
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in
 *     the documentation and/or other materials provided with the
 *     distribution.
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
 * IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#import "ProcessManager.h"
#import "ProcessWindowController.h"
#import "ApplicationInfo.h"

static NSString * const ProcessTaskKey = @"Process task";
static NSString * const ProcessFileHandleKey = @"Process fileHandle";
NSString * const ProcessCommandLineKey = @"Process commandLine";
NSString * const ProcessIdentifierKey = @"Process pid";
NSString * const ProcessDateKey = @"Process date";
NSString * const LogMessageDateKey = @"LogMessage date";
NSString * const LogMessageProcessIdentifierKey = @"LogMessage pid";
NSString * const LogMessageContentKey = @"LogMessage content";

NSString * const ProcessDidTerminateNotification =
	@"ProcessManager ProcessDidTerminateNotification";
NSString * const ProcessDidCloseOutputNotification =
	@"ProcessManager ProcessDidCloseOutputNotification";
NSString * const ProcessManagerDidChangeProcessesNotification =
	@"ProcessManager ProcessManagerDidChangeProcessesNotification";

static NSString *
localize(NSString *key)
{
	NSString *s = NSLocalizedStringFromTable(key, @"ProcessManager", nil);
	return s;
}

static NSString *
dataToString(NSData *data)
{
	static const NSStringEncoding encodings[] = {
		NSUTF8StringEncoding,
		NSShiftJISStringEncoding,
		NSWindowsCP1251StringEncoding,
		NSWindowsCP1252StringEncoding,
		NSWindowsCP1253StringEncoding,
		NSWindowsCP1254StringEncoding,
		NSWindowsCP1250StringEncoding,
		NSJapaneseEUCStringEncoding,
		NSNonLossyASCIIStringEncoding,
	};
	NSString *s = nil;
	int i;

	for (i = 0; !s && i < sizeof(encodings)/sizeof(encodings[0]); i++) {
		s = [[[NSString alloc]
		      initWithData:data encoding:encodings[i]]
		     autorelease];
	}
	if (!s)
		s = @"*** Failed to convert to NSString ***";
	return s;
}

static NSString *
taskCommandLine(NSTask *task)
{
	NSString *ret, *arg;
	NSMutableString *buf;
	NSEnumerator *e;
	NSArray *args;

	ret = [task launchPath];
	args = [task arguments];
	if (args) {
		e = [args objectEnumerator];
		while ((arg = [e nextObject])) {
			buf = [NSMutableString stringWithCapacity:[arg length]];
			[buf setString:arg];
			[buf
			 replaceOccurrencesOfString:@"'"
			 withString:@"'\\''"
			 options:NSLiteralSearch | NSBackwardsSearch
			 range:NSMakeRange(0, [buf length])];
			ret = [ret stringByAppendingFormat:@" '%@'", buf];
		}
	}
	return ret;
}

@interface ProcessDateStringGenerator : NSObject {
	NSString *lastDateString;
	NSTimeInterval lastDateInterval;
}
- (NSString *)currentTimeString;
@end

@implementation ProcessDateStringGenerator
- (void)dealloc
{
	[lastDateString release];
	[super dealloc];
}

- (NSString *)currentTimeString
{
	NSDate *now = [NSDate date];
	NSTimeInterval interval = [now timeIntervalSinceReferenceDate];

	if (!lastDateString || interval != lastDateInterval) {
		[lastDateString release];
		lastDateString =
			[[now descriptionWithCalendarFormat:@"%Y-%m-%d %H:%M:%S"
						   timeZone:nil
						     locale:nil]
			 retain];
		lastDateInterval = interval;
	}
	return lastDateString;
}
@end

/* friend instance methods */
@interface ProcessManager (ProcessManagerInternal)
- (void)addLog:(id)content withProcessID:(NSString *)processId;
- (void)notifyLogMessagesDidChange;
@end

@interface Process (ProcessInternal)
- (id)initWithTask:(NSTask *)task
	 startedAt:(NSString *)startDate
    processManager:(ProcessManager *)manager;
- (void)dataDidRead:(NSData *)data flush:(BOOL)flush;
@end

@implementation Process

- (id)initWithTask:(NSTask *)task
	 startedAt:(NSString *)startDate
    processManager:(ProcessManager *)manager
{
	self = [super init];
	if (!self)
		return self;

	processManager = manager;  /* avoid reference loop */
	noCrashAlert = NO;

	NSString *pid = [NSString stringWithFormat:@"%d",
				  [task processIdentifier]];
	NSFileHandle *fileHandle = [[task standardOutput] fileHandleForReading];

	properties =
		[[NSDictionary alloc]
		 initWithObjectsAndKeys:
		 startDate, ProcessDateKey,
		 pid, ProcessIdentifierKey,
		 taskCommandLine(task), ProcessCommandLineKey,
		 task, ProcessTaskKey,
		 fileHandle, ProcessFileHandleKey,
		 nil];

	buffer = [[NSMutableData alloc] initWithCapacity:1024];
	return self;
}

- (void)dealloc
{
	[properties release];
	[buffer release];
	[super dealloc];
}

- (BOOL)noCrashAlert
{
	return noCrashAlert;
}

- (NSDictionary *)properties
{
	return properties;
}

- (NSTask *)task
{
	return [properties objectForKey:ProcessTaskKey];
}

- (id)objectForKey:(NSString *)key
{
	id obj = [properties objectForKey:key];
	if (obj)
		return obj;
	else
		return [[[self task] environment] objectForKey:key];
}

- (void)terminateProcessBy:(NSString *)sender quiet:(BOOL)quiet
{
	/* send SIGTERM to the task */
	[[self task] terminate];
	/* send SIGPIPE to all descendant processes of this task */
	[[properties objectForKey:ProcessFileHandleKey] closeFile];

	[processManager log:@"pid %d: terminated by %@",
			[[self task] processIdentifier], sender];
	if (quiet)
		noCrashAlert = YES;
}

- (void)dataDidRead:(NSData *)data flush:(BOOL)flush
{
	NSString *pid = [properties objectForKey:ProcessIdentifierKey];
	char *bytes, *p;
	size_t len, slen;

	[buffer appendData:data];
	bytes = [buffer mutableBytes];
	len = [buffer length];

	while (len > 0) {
		/* assume that linebreaks are performed in UNIX manner. */
		p = memchr(bytes, '\n', len);
		if (p)
			slen = p - bytes, p++;
		else if (flush)
			slen = len, p = bytes + len;
		else
			break;

		data = [NSData dataWithBytesNoCopy:bytes
					    length:slen
				      freeWhenDone:NO];
		NSString *str = dataToString(data);
		[processManager addLog:str withProcessID:pid];
		len -= p - bytes;
		bytes = p;
	}

	memmove([buffer mutableBytes], bytes, len);
	[buffer setLength:len];

	[processManager notifyLogMessagesDidChange];
}

@end

@implementation ProcessManager

- (id)init
{
	if ((self = [super init])) {
		processWindow = nil;
		currentProcesses = [[NSMutableArray array] retain];
		logMessages = [[NSMutableArray array] retain];
		dateGenerator = [[ProcessDateStringGenerator alloc] init];
	}
	return self;
}

- (void)dealloc
{
	if (processWindow)
		[processWindow release];
	[currentProcesses release];
	[logMessages release];
	[dateGenerator release];
	[[NSNotificationCenter defaultCenter] removeObserver:self];
	[super dealloc];
}

- (NSArray *)currentProcesses
{
	return currentProcesses;
}

- (NSArray *)logMessages
{
	return logMessages;
}

- (NSWindowController *)processWindow
{
	if (!processWindow) {
		processWindow = [[ProcessWindowController alloc]
				 initWithProcessManager:self];
	}
	return processWindow;
}

- (void)showStatusWindow:(id)sender
{
	[[self processWindow] showWindow:sender];
}

- (void)notifyCurrentProcessesDidChange
{
	if (processWindow)
		[processWindow currentProcessesDidChange];

	[[NSNotificationCenter defaultCenter]
	 postNotificationName:ProcessManagerDidChangeProcessesNotification
	 object:self];
}

- (void)notifyLogMessagesDidChange
{
	if (processWindow)
		[processWindow logMessagesDidChange];
}

- (void)addLog:(id)content withProcessID:(NSString *)pid
{
	NSString *date = [dateGenerator currentTimeString];
	NSDictionary *log;

	NSParameterAssert(content && pid);

	log = [NSDictionary dictionaryWithObjectsAndKeys:
			    date, LogMessageDateKey,
			    pid, LogMessageProcessIdentifierKey,
			    content, LogMessageContentKey,
			    nil];

	if ([content isKindOfClass:[NSAttributedString class]])
		content = [content string];
	//NSLog(@"%@: %@", pid, content);

	[logMessages addObject:log];
}

- (void)log:(NSString *)format, ...
{
	va_list args;
	va_start(args, format);

	NSString *s = [[[NSString alloc]
			initWithFormat:format arguments:args]
		       autorelease];

	NSDictionary *attributes =
		[NSDictionary
		 dictionaryWithObject:[NSColor redColor]
		 forKey:NSForegroundColorAttributeName];

	NSAttributedString *as = [[[NSAttributedString alloc]
				   initWithString:s
				   attributes:attributes]
				  autorelease];

	[self addLog:as withProcessID:@""];
	[self notifyLogMessagesDidChange];
	va_end(args);
}

- (Process *)startProcess:(NSString *)command
		arguments:(NSArray *)args
	      environment:(NSDictionary *)env
{
	Process *proc;

	NSPipe *pipe = [NSPipe pipe];
	NSTask *task = [[[NSTask alloc] init] autorelease];
	[task setStandardInput:[NSFileHandle fileHandleWithNullDevice]];
	[task setStandardOutput:pipe];
	[task setStandardError:pipe];

	[task setLaunchPath:command];
	if (args)
		[task setArguments:args];
	if (env) {
		NSMutableDictionary *newEnv =
			[NSMutableDictionary dictionaryWithCapacity:10];
		[newEnv setDictionary:
			[[NSProcessInfo processInfo] environment]];
		[newEnv addEntriesFromDictionary:env];
		[task setEnvironment:newEnv];
	}

	@try {
		[task launch];
	} @catch (NSException *e) {
		if (![[e name] isEqual:NSInvalidArgumentException])
			@throw;
		[self log:@"task %p: Error at launch: %@", task, [e reason]];
		return nil;
	}

	[[pipe fileHandleForReading]
	 readInBackgroundAndNotifyForModes:
	 [NSArray arrayWithObjects:
		  NSDefaultRunLoopMode,
		  NSModalPanelRunLoopMode,
		  NSEventTrackingRunLoopMode,
		  nil]];

	[[NSNotificationCenter defaultCenter]
	 addObserver:self
	 selector:@selector(readData:)
	 name:NSFileHandleReadCompletionNotification
	 object:[pipe fileHandleForReading]];

	proc = [[Process alloc] initWithTask:task
				   startedAt:[dateGenerator currentTimeString]
			      processManager:self];

	[self log:@"pid %d: start with command %@",
	      [[proc task] processIdentifier],
	      [proc objectForKey:ProcessCommandLineKey]];

	[currentProcesses addObject:proc];

	[self notifyCurrentProcessesDidChange];
	return proc;
}

- (void)terminateAll
{
	NSEnumerator *e = [currentProcesses objectEnumerator];
	Process *proc;

	while ((proc = [e nextObject]))
		[proc terminateProcessBy:@"terminateAll" quiet:YES];
}

- (void)removeProcess:(Process *)proc
{
	NSTask *task = [proc task];
	int status;

	[proc dataDidRead:[NSData data] flush:YES];
	[currentProcesses removeObjectIdenticalTo:proc];
	status = [task terminationStatus];

	[self log:@"pid %d: finished at status %d",
	      [task processIdentifier], status];
	NSLog(@"%d tasks remain", [currentProcesses count]);

	if (status != 0 && ![proc noCrashAlert]) {
		NSInteger result =
			NSRunCriticalAlertPanel(
				[NSString stringWithFormat:
					  localize(@"The process %1$@ quit at"
						   " unexpected status."
						   " (pid:%2$d, status:%3$d)"),
					  [[task launchPath] lastPathComponent],
					  [task processIdentifier],
					  status],
				localize(@"Click Detail to see more details"
					 " or make a report."),
				localize(@"Detail..."),
				localize(@"Ignore"),
				nil);

		if (result == NSAlertDefaultReturn)
			[self showStatusWindow:self];
	}

	[[NSNotificationCenter defaultCenter]
	 postNotificationName:ProcessDidTerminateNotification
	 object:proc];

	[self notifyCurrentProcessesDidChange];
}

- (Process *)getProcessByObject:(id)obj forKey:(NSString *)key
{
	NSEnumerator *e = [currentProcesses objectEnumerator];
	Process *proc;

	while ((proc = [e nextObject])) {
		if ([proc objectForKey:key] == obj)
			return proc;
	}
	return nil;
}

- (void)readData:(NSNotification *)notification
{
	NSFileHandle *handle = [notification object];
	Process *proc = [self getProcessByObject:handle
					  forKey:ProcessFileHandleKey];
	NSTask *task = [proc task];

	NSData *data = [[notification userInfo]
			objectForKey:NSFileHandleNotificationDataItem];

	if ([data length] == 0) {
		/*
		 * NOTE: On Tiger, we cannot receive NSTaskDidTerminate-
		 * Notification in modal run loop. If you want to abort
		 * modal run loop when a process is terminated, observe
		 * ProcessDidCloseOutputNotification instead of
		 * ProcessDidTerminateNotification.
		 */
		[[NSNotificationCenter defaultCenter]
		 postNotificationName:ProcessDidCloseOutputNotification
		 object:proc];

		if (![task isRunning]) {
			[self removeProcess:proc];
		} else {
			[[NSNotificationCenter defaultCenter]
			 addObserver:self
			 selector:@selector(taskTerminated:)
			 name:NSTaskDidTerminateNotification
			 object:task];
		}

		[[NSNotificationCenter defaultCenter]
		 removeObserver:self
		 name:NSFileHandleReadCompletionNotification
		 object:handle];
		return;
	}

	[proc dataDidRead:data flush:NO];
	[handle readInBackgroundAndNotifyForModes:
		[NSArray arrayWithObjects:
			 NSDefaultRunLoopMode,
			 NSModalPanelRunLoopMode,
			 NSEventTrackingRunLoopMode,
			 nil]];
}

- (void)taskTerminated:(NSNotification *)notification
{
	NSTask *task = [notification object];
	Process *proc = [self getProcessByObject:task forKey:ProcessTaskKey];

	[[NSNotificationCenter defaultCenter]
	 removeObserver:self
	 name:NSTaskDidTerminateNotification
	 object:task];

	[self removeProcess:proc];
}

- (BOOL)saveReportToPath:(NSString *)filename
		   error:(NSError **)error
{
	NSString *CFBundleNameKey = (NSString *)kCFBundleNameKey;
	NSString *CFBundleVersionKey = (NSString *)kCFBundleVersionKey;
	NSMutableString *buf;

	buf = [NSMutableString stringWithCapacity:4096];

	[buf appendFormat:@"%@ log report\n\n", ApplicationName];
	[buf appendFormat:@"Date: %@\n", [NSDate date]];

	NSProcessInfo *procInfo = [NSProcessInfo processInfo];
	[buf appendFormat:@"OS: %@\nProcessName: %@\nProcessID: %d\n\n",
	     [procInfo operatingSystemVersionString],
	     [procInfo processName],
	     [procInfo processIdentifier]];

	NSBundle *mainBundle = [NSBundle mainBundle];
	[buf appendFormat:@"BundleName: %@\nBundleVersion: %@\n",
	     [mainBundle objectForInfoDictionaryKey:CFBundleNameKey],
	     [mainBundle objectForInfoDictionaryKey:CFBundleVersionKey]];

	NSBundle *appBundle = [ApplicationInfo applicationBundle];
	[buf appendFormat:
	     @"AppBundleName: %@\nAppBundleVersion: %@\nAppBundlePath: %@\n",
	     [appBundle objectForInfoDictionaryKey:CFBundleNameKey],
	     [appBundle objectForInfoDictionaryKey:CFBundleVersionKey],
	     [appBundle bundlePath]];

	NSBundle *wineBundle = [ApplicationInfo wineBundle];
	[buf appendFormat:
	     @"WineBundleName: %@\nWineBundleVersion: %@\nWineBundlePath: %@\n",
	     [wineBundle objectForInfoDictionaryKey:CFBundleNameKey],
	     [wineBundle objectForInfoDictionaryKey:CFBundleVersionKey],
	     [wineBundle bundlePath]];

	NSString *x11path =
		[[NSWorkspace sharedWorkspace] fullPathForApplication:@"X11"];
	NSBundle *x11bundle =
		x11path ? [NSBundle bundleWithPath:x11path] : nil;
	if (x11bundle) {
		[buf appendFormat:
		     @"X11BundleName: %@\nX11BundleVersion: %@\n"
		     "X11BundlePath: %@\n",
		     [x11bundle objectForInfoDictionaryKey:CFBundleNameKey],
		     [x11bundle objectForInfoDictionaryKey:CFBundleVersionKey],
		     [x11bundle bundlePath]];
	}

	[buf appendString:@"\nLogMessages:\n"];

	NSEnumerator *e = [logMessages objectEnumerator];
	NSDictionary *log;

	while ((log = [e nextObject])) {
		id content = [log objectForKey:LogMessageContentKey];
		[buf appendFormat:@"%@\t%@\t%@\n",
		     [log objectForKey:LogMessageDateKey],
		     [log objectForKey:LogMessageProcessIdentifierKey],
		     [content isKindOfClass:[NSAttributedString class]]
		     ? [content string] : content];
	}

	[buf appendString:@"\n** report ends here **\n"];

	return [buf writeToFile:filename
		     atomically:NO
		       encoding:NSUTF8StringEncoding
			  error:error];
}

@end
